"""
originpro
A package for interacting with Origin software via Python.
Copyright (c) 2020 OriginLab Corporation
"""
from .config import po, oext
from .base import *
from .utils import *

from datetime import datetime as dt, timedelta as tdlt, date, time
try:
    import pandas as pd
except ImportError:
    pass

def _DC_from_ext(ext):
    if ext in ['.csv', '.asc', '.txt', '.dat']:
        return 'csv'
    if ext in ['.xls', '.xlsx']:
        return 'excel'

    raise ValueError('file type not supported, must be text or Excel files')

def _origin_date_offset(to_df):
    nJD = int(po.LT_get_var('@DSP'))
    if nJD == 1:#True Julian Days
        return 0
    if nJD == 0:#Origin's current default
        return 0.5 if to_df else -0.5
        
    raise ValueError('@DSP with value not yet implemented')

def _series_to_julian_date(s):
    offset = _origin_date_offset(False)
    # Must check that vv is not NaN or NaT (VC handles NaT correctly via _OPyIsNaT):
    return [vv if vv!=vv else vv.to_julian_date() + offset for vv in s]

def _series_to_time(s):
    # See https://pandas.pydata.org/pandas-docs/stable/user_guide/timedeltas.html
    # under "Frequency conversion"
    # Here need to convert it to number of days as float64:
    #listtoset = [vv/np.timedelta64(1, 'D') for vv in value]
    # See https://docs.python.org/2/library/datetime.html
    # under 8.1.2. timedelta Objects
    # In Origin Time column contains numeric floating values which represent fractions of one day.
    return [vv/tdlt(days=1) for vv in s]
    
def check_convert_py_time_to_tdlt(s):
    if not s.empty:
        if isinstance(s.iloc[0], time):
            return [dt.combine(date.min, vv) - dt.min for vv in s]
    return s
    

class WSheet(DSheet):
    """
    This class represents an Origin Worksheet, it holds an instance of a PyOrigin Worksheet.
    """
    # Adds a maximum of needecols columns beginning with c1 to
    # Origin worksheet wks:
    def _check_add_cols(self, needecols, c1 = 0):
        if self.obj.Cols < needecols + c1:
            self.obj.Cols = c1 + needecols
    @staticmethod
    def _set_col_LN(colobj, key, col):
        if isinstance(key, int) and key == col:
            return
        if type(key) == str:
            colobj.LongName = key
        else:
            colobj.LongName = str(key)
    @staticmethod
    def _getlabel(colobj, type):
        if isinstance(type, int):
            if type < 0 or type > 15:
                raise ValueError('Invalid index to user parameter')
            return colobj.GetUserDefLabel(type)
        if len(type) > 1:
            raise ValueError('Invalid label row character')
        dd = {
            'L': colobj.GetLongName,
            'C': colobj.GetComments,
            'U': colobj.GetUnits,
            }
        func = dd[type]
        return func()
    @staticmethod
    def _setlabel(colobj, type, val):
        """type is int if user parameter row, or 'L', 'C', 'U' etc, see _col_label_row()"""
        if isinstance(type, int):
            if type < 0 or type > 15:
                raise ValueError('Invalid index to user parameter')
            return colobj.SetUserDefLabel(type, val)
        if len(type) > 1:
            raise ValueError('Invalid label row character')
        dd = {
            'L': colobj.SetLongName,
            'C': colobj.SetComments,
            'U': colobj.SetUnits,
            }
        func = dd[type]
        return func(val)

    @staticmethod
    def _cname(colobj, type):
        try:
            return WSheet._getlabel(colobj, type)
        except ValueError:
            pass
        lname = colobj.LongName
        if len(lname):
            return lname;
        return colobj.GetName()

    @staticmethod
    def _cgetdata(colobj, nStart=0, nEnd=-1):
        if oext:
            return colobj.GetData(9999, nStart, nEnd)
        return colobj.GetData(nStart, nEnd)
        
    def _col_label_row(self, head=''):
        """return int if user parameter row, or 'L', 'C', 'U' etc """
        if len(head) == 0:
            return 'L'
        if not isinstance(head, str):
            raise ValueError('must specify a str')
        if len(head) == 1:
            return head
        return self._user_param_row(head, True)
        
    def _user_param_row(self, name, add=False):
        """find user defined parameter row index from name, if add, then add it if not existed"""
        ltarg= '++' if add else ''
        ltarg += name
        return self.method_int('UserParam', ltarg)-1
        
    def _col_index(self, col):
        if isinstance(col,int):
            return col
        cobj = self.obj.FindCol(col)
        if cobj is None:
            return -1
        return cobj.GetIndex()

    def _find_col(self, col):
        if isinstance(col, int):
            return self.obj[col]
        return self.obj.FindCol(col)
    
    @property
    def rows(self):
        """
        Get the last row index in worksheet with data, hidden rows has no effect on this
        This is different from shape[0] which does not care of having data or not
        """
        return self.get_int('maxrows')
        
    @property
    def cols(self):
        """
        get the last column index in worksheet, this is the same as shape[1]
        """
        return self.obj.Cols
        
    @cols.setter
    def cols(self, val):
        """
        set the number of columns in worksheet
        """
        self.obj.Cols = val
        return self.cols
        
    def get_book(self):
        """
        Returns parent book of sheet.

        Parameters:

        Returns:
            (WBook)

        Examples:
            wks2 = wks.get_book().add_sheet('Result')
        """
        return WBook(self._get_book())
    
    def lt_col_index(self, col):
        """
        convert a 0-offset index to LabTalk index which is 1-offset
        
        Parameters:
            col (int or str): If int, column index. If str, tries short name and if not exists, tries column long name
            
        Return:
            (int) 1-offset column index
            
        Examples:
            ii = wks.lt_col_index('Intensity')
            if ii < 1:
                print('no such column')
        """
        return self._col_index(col) + 1
        
    def from_df(self, df, c1=0, addindex=False, head=''):
        """
        Sets a pandas DataFrame to an Origin worksheet.

        Parameters:
            df (DataFrame): Input DataFrame object
            c1 (int or str): Starting column index
            addindex (bool): add an index column at c1 if df has text indices
            head(str): if not specified, column longname is used if df has column names, 
                head can be one of the Column Label Row character like 'L', 'C', or a user parameter by its name
                
        Returns:
            None

        Examples:
            wks=op.find_sheet()
            my_df = pd.DataFrame({'aa':[1,2,3], 'bb':[4,5,6]})
            wks.from_df(my_df)
            
            import pandas as pd
            fname = op.path('e') + "Samples\\Import and Export\donations.csv"
            df = pd.read_csv(fname)
            wks.from_df(df,'B')
        """
        c1 = self._col_index(c1)
        if c1 < 0:    # caller's mistake
            raise ValueError('c1 must not be <0')

        if df.empty:
            return
        head_row = self._col_label_row(head)
        ndfcols = len(df.columns)
        #print(str(type(df.index)))
        if addindex:
            # Find out if index is not default (0, 1, 2,...)
            addindex = list(range(len(df.index))) != list(df.index)

        colstohave = ndfcols
        colBeginData = c1;
        if addindex:
            colstohave += 1
            colBeginData += 1

        self._check_add_cols(colstohave, c1)
        if addindex:
            # c1 should receive the index
            self.obj[c1].SetData(df.index)
            
        bSetColumnAsMixedInGeneralCase = 0 < int(po.LT_get_var('@DFSM'))    # ML 06/26/2020 ORG-21995-P2 FROM_DF_DONT_SET_TEXT_AND_NUMERIC_FOR_GENERAL_CASE_BY_DEFAULT

        col = colBeginData
        for key, value in df.iteritems():
            colobj = self.obj[col]
            if str(value.dtype) == 'category':
                dfseriestypechar = 'cat'        # our own "name" for convenience
            else:
                dfseriestypechar = df.dtypes[col - colBeginData].char
            #print(dfseriestypechar)
            #print(value)
            displayFmt = -1
            if dfseriestypechar == 'O':
                v1 = check_convert_py_time_to_tdlt(value)
                if v1 is not value:
                    value = v1
                    dfseriestypechar = 'm'
                    displayFmt = 2
            if dfseriestypechar == 'd':    # if floating:
                colobj.SetDataFormat(po.DF_DOUBLE)
                listtoset = list(value)
            elif dfseriestypechar == 'q':    # if integer
                colobj.SetDataFormat(po.DF_LONG)
                listtoset = list(value)
            elif dfseriestypechar == 'D':    # if complex:
                colobj.SetDataFormat(po.DF_COMPLEX)
                listtoset = list(value)
            elif dfseriestypechar == 'M':    # if datetime64[ns]:
                colobj.SetDataFormat(po.DF_DATE)
                listtoset = _series_to_julian_date(value)
            elif dfseriestypechar == 'm':    # if 'timedelta':
                colobj.SetDataFormat(po.DF_TIME)
                listtoset = _series_to_time(value)
            elif dfseriestypechar == 'cat':    # if categorical
                colobj.SetDataFormat(po.DF_TEXT)
                listtoset = list(value)
            else:
                # ML 06/26/2020 ORG-21995-P2 FROM_DF_DONT_SET_TEXT_AND_NUMERIC_FOR_GENERAL_CASE_BY_DEFAULT
                # colobj.SetDataFormat(po.DF_TEXT_NUMERIC)
                if bSetColumnAsMixedInGeneralCase:
                    colobj.SetDataFormat(po.DF_TEXT_NUMERIC)
                # end FROM_DF_DONT_SET_TEXT_AND_NUMERIC_FOR_GENERAL_CASE_BY_DEFAULT
                listtoset = list(value)

            colobj.SetData(listtoset)
            if dfseriestypechar == 'cat':    # if categorical
                #if value.cat.ordered:
                #    colobj.SetCategMapSortCategories(po.CM_SORTING_ASCENDING, [])
                #else:
                # Per sam's request we always do custom:
                colobj.SetCategMapSortCategories(po.CM_SORTING_CUSTOM, [str(v) for v in value.cat.categories])
            elif displayFmt >= 0:
                colobj.SetDisplayFormat(displayFmt)
            if len(head) == 0:
                self._set_col_LN(colobj, key, col)
            else:
                self._setlabel(colobj, head_row, key)
            col += 1

    def    to_df(self, c1=0, numcols = -1, cindex = -1, head=''):
        """
        Creates a pandas DataFrame from an Origin worksheet.

        Parameters:
            c1 (int or str): column to start the export
            numcols (int): Total number of columns, -1 to the end
            cindex (int or str): Column to use for DataFrame index if specified
            head (str): user parameter row name, if not specified, lname(column long name), or short name if lname empty
        Returns:
            (DataFrame)

        Examples:
            df = wks.to_df(2)#from 3rd column
            
            #use col(A) as index, and B and C as data
            df1 = wks.to_df('B', 2, 'A')
            
            #to use a user defined label row as column heading
            df2 = wks.to_df(head='Tag')
        """
        c1 = self._col_index(c1)
        if c1 < 0:    # caller's mistake
            raise ValueError('c1 must not be <0')
        cindex = self._col_index(cindex)
        if self.obj.Cols <= cindex:    # invalid index column
            raise ValueError('cindex outside of number of columns')
        cname=-1
        if len(head):
            row = self._col_label_row(head)
            if isinstance(row, int) and row < 0:
                raise ValueError(f'user parameter row {head} not found')
            cname = row
        
        df = pd.DataFrame()
        if numcols < 0:
            totalcols = self.obj.Cols - c1
        else:
            totalcols = min(self.obj.Cols - c1, numcols)

        for col in range(c1, c1 + totalcols):
            colobj = self.obj[col]
            coldata = self._cgetdata(colobj)
            coldatatype = colobj.GetDataFormat()
            if coldatatype == po.DF_DATE:
                offset = _origin_date_offset(True)
                coldata = [xx + offset for xx in coldata]  
                # See https://pandas.pydata.org/pandas-docs/stable/reference/api/pandas.Timestamp.html
                listtimestamps = pd.to_datetime(coldata, unit='D', origin='julian')
                dfseries = pd.Series(listtimestamps, list(range(len(listtimestamps))))
            elif coldatatype == po.DF_TIME:
                # See https://pandas.pydata.org/pandas-docs/stable/reference/api/pandas.to_timedelta.html
                listtimedeltas = pd.to_timedelta(coldata, unit='d')
                dfseries = pd.Series(listtimedeltas, list(range(len(listtimedeltas))))
            elif colobj.CategMapType != po.CMTYPE_NONE:
                sort = colobj.CategMapSort
                if sort == po.CM_SORTING_ASCENDING:
                    # if ascending, just set the Categorical property "ordered" to True:
                    dfseries = pd.Categorical(coldata, ordered=True)
                #elif sort == po.CM_SORTING_UNSORTED:
                #    dfseries = pd.Series(coldata, list(range(len(coldata))), dtype="category")
                else:
                    # For all the other cases get the categories array from the columns and set it into df:
                    categories = colobj.CategMapCategories
                    dfseries = pd.Categorical(coldata, categories)

            else:
                dfseries = pd.Series(coldata, list(range(len(coldata))))
            loc = col - c1
            # ML 06/25/2020 ORG-21995-P1 APPEND_EMPTY_ROWS_BEFORE_LONGER_SERIES_TO_ADD
            # https://stackoverflow.com/questions/39998262/append-an-empty-row-in-dataframe-using-pandas
            # If the series being added is longer than the current size, must append with empty rows first
            lengthseries = len(dfseries)
            if 0 < len(df.index) and len(df.index) < lengthseries:
                for iappend in range(len(df.index), lengthseries):
                    df = df.append(pd.Series(), ignore_index = True)
            # end APPEND_EMPTY_ROWS_BEFORE_LONGER_SERIES_TO_ADD
            df.insert(loc, self._cname(colobj, cname), dfseries)

        if 0 <= cindex:
            colobj = self.obj[cindex]
            df.index = self._cgetdata(colobj)


        return df

    def to_list(self, col):
        """
        Creates a list object from the data in an Origin column.

        Parameters:
            col (int or str): If int, column index. If str, tries short name and if not exists, tries column long name

        Returns:
            (list)

        Examples:
            data1 = wks.to_list(1)
            data2 = wks.to_list('Intensity')
        """
        ncol = self._col_index(col)
        if ncol < 0 or ncol >= self.obj.Cols:
            return []
        return self._cgetdata(self.obj[ncol])

    def from_list(self, col, data, lname='', units='', comments='', axis=''):
        """
        Sets a list object into an Origin column, optionally specifying long name, units ,comments label row values.

        Parameters:
            col (int or str): If int, column index. If str, tries short name and if not exists, tries column long name
            data (list): data to put into column
            lname (str): Optional column long name
            units (str): Optional column units
            comments (str): Optional column comments
            axis (str): empty will not set, otherwise set column designation for plotting, 
                        can be X,Y,Z, or N(None), E(Yerr), M(Xerr), L(label)
                        see https://www.originlab.com/doc/Origin-Help/WksCol-SetDesignation
        Returns:
            None

        Examples:
            data = [1,2,3,4,5]
            wks.from_list(1, data)
            wks.from_list('Intensity', data, units='a. u.', axis='Y')
        """
        ncol = self._col_index(col)
        if ncol < 0:
            raise ValueError('Column does not exist')
        if ncol >= self.obj.Cols:
            self.obj.Cols = ncol + 1
        self.obj[ncol].SetData(data)
        if len(lname):
            self.obj[ncol].SetLongName(lname)
        if len(units):
            self.obj[ncol].SetUnits(units)
        if len(comments):
            self.obj[ncol].SetComments(comments)
        if len(axis):
            if len(axis) > 1:
                raise ValueError('axis type must be a single letter')
            if not axis in list('XYZNEML'):
                raise ValueError('Invalid axis type letter')
            self.obj.SetColDesignations(axis, 0, ncol, ncol)

    def get_label(self, col, type = 'L'):
        """
        Return a column label row text.

        Parameters:
            col (int or str): If int, column index. If str, tries short name and if not exists, tries column long name
            type (str): A column label row character (see https://www.originlab.com/doc/LabTalk/ref/Column-Label-Row-Characters) or a user defined parameter name

        Returns:
            (str)

        Examples:
            wks=op.find_sheet()
            comments = wks.get_label(1,'C')
        """
        ncol = self._col_index(col)
        if ncol < 0:
            raise ValueError('Column does not exist')
        colobj = self.obj[ncol]
        if len(type) > 1:#user parameter name
            ii = self._user_param_row(type)
            if ii < 0:
                return ''#not found
            type = ii
            return self._getlabel(colobj, type)
            
        return self._getlabel(colobj, type)
        
    def set_label(self, col, val, type = 'L'):
        """
        Set a column label row text.

        Parameters:
            col (int or str): If int, column index. If str, tries short name and if not exists, tries column long name
            val (str): the text to set
            type (str): A column label row character (see https://www.originlab.com/doc/LabTalk/ref/Column-Label-Row-Characters) or a user defined parameter name

        Examples:
            wks=op.find_sheet()
            wks.set_label('A','long name for col A')
            wks.set_label(1, 'for col B, add user parameter if not there', 'Channel')
        """

        if isinstance(type,int) or len(type)==0:
            raise ValueError('type cannot be integers or empty')
            
        ncol = self._col_index(col)
        if ncol < 0:
            raise ValueError('Column does not exist')
        colobj = self.obj[ncol]
        if len(type) > 1:#user parameter name
            type = self._user_param_row(type, True)
            
        self._setlabel(colobj, type, val)
            
    def from_dict(self, data, col=0, row=0):
        """
        Set a dictionary into a Worksheet. Keys are not used for any purpose.

        Parameters:
            data (dict): The data to set
            col (int or str): If int, column index. If str, tries short name and if not exists, tries column long name
            row (int): Row index to start setting the data

        Returns:
            None

        Examples:
            data ={'aa': [1,2,3], 'bb':[4,5,6]}
            wks.from_dict(data)
            wks.from_dict(data,'C')
        """
        col = self._col_index(col)
        self.obj.SetData(data, row, col)

    def header_rows(self, spec=''):
        """
        Controls which worksheet label rows to show, same as LabTalk wks.labels string.

        Parameters:
            spec (str): A combination of letters. See https://www.originlab.com/doc/LabTalk/ref/Column-Label-Row-Characters

        Returns:
            None

        Examples:
            wks.header_rows('lu')# Show only long-name and unit.
            wks.header_rows()# Remove all label rows, keep only heading.
        """
        if len(spec)==0:
            Specs = '0'
        else:
            Specs = spec.upper()
        getattr(self.obj, 'Labels' if oext else 'ShowLabels')(Specs)

    def cols_axis(self, spec='', c1 = 0, c2 = -1, repeat=True):
        """
        Set column plotting designations with a string pattern.

        Parameters:
            spec (str): A combination of 'x', 'y', 'z', etc letters 
            c1 (int or str): Starting column to set
            c2 (int or str): Last column to set. c2 < 0 sets to last column
            repeat (bool): Repeat the last designation letter or not

        Returns:
            None

        Examples:
            wks.cols_axis('nxy') # 1st col none, 2nd=x, others=y.
            wks.cols_axis() # Clear designations from all columns.
        """
        c1 = self._col_index(c1)
        if isinstance(c1,str):
            c2 = self._col_index(c1)
        if len(spec)==0:
            Specs = 'N'
        else:
            Specs = spec.upper()
        self.obj.SetColDesignations(Specs, repeat, c1, c2)

    def from_file(self, fname, keep_DC=True):
        """
        Imports data from a file using a Data Connector.

        Parameters:
            fname (str): File path and name to import.
            keep_DC (bool): Keep the Data Connector in the book after import

        Returns:
            None

        Examples:
            wks=op.find_sheet()
            fn=op.path('e') + 'Samples\\Import and Export\donations.csv'
            wks.from_file(fn, False)#remove connector after import to allow further edit of data
            wks2=op.new_sheet()
            wks2.from_file(op.path()+'test.xlsx')#assuming you have this in UFF(user file folder)
        """
        ext = get_file_ext(fname)
        DC = _DC_from_ext(ext)
        currentDC = self.has_DC()
        if len(currentDC) and currentDC != DC:
            self.remove_DC()
            currentDC=''
        if len(currentDC) == 0:
            self.obj.DoMethod('DC.Allow', '2')
            self.obj.LT_execute(f'wbook.dc.add({DC})')
        self.obj.SetStrProp('DC.Source', fname)
        ret = self.method_int('dc.import')
        #print(f'dc import return {ret}')
        if not keep_DC:
            self.remove_DC()
            
    def clear(self, c1 = 0, ncols = 0, c2 = -1):
        """
        Clear data in worksheet

        Parameters:
            c1 (int or str): starting col, if int, column index. If str, column name, short name first, then long name.
            ncols(int): number of columns to clear if > 0, cannot specify togeher with c2
            c2 (int or str): ending col, if int, column index. If str, column name, short name first, then long name.

        Returns:
            (int): 0 for success, otherwise an internal error code

        Examples:
            wks.clear()#clear all
            wks.clear(1)#clear from 2nd column to the end
            wks.clear(1,2)#clear from 2nd and 3rd
        """
        n1 = self.lt_col_index(c1)
        n2 = self.lt_col_index(c2)
        if ncols > 0:
            if n2 > 0:
                raise ValueError('should not specify both ncols and c2')
            n2 = n1 + ncols-1
            
        return self.method_int('clear', f'{n1},{n2}')
    
    def sort(self, col, dec = False):
        """
        Sort worksheet data by a specified column

        Parameters:
            col (int or str): If int, column index. If str, column name, short name first, then long name.
            dec (bool): Accending or (dec)Decending

        Returns:
            (int): 0 for success, otherwise an internal error code

        Examples:
            wks.sort('A')#sort using 1st col, accending
            wks.sort(0, True)#sort using 1st col, decending
        """
        #we need LT col index for this
        ncol = self.lt_col_index(col)
        ndec = 1 if dec else 0
        return self.method_int('sort', f'{ncol},{ndec}')
        
    def to_list2(self, r1=0, r2=-1, c1=0, c2=-1):
        """
        get a block of cells as a list of lists
        
        Parameters:
            r1 (int): beginning row index
            r2 (int): ending row index (inclusive), -1 to the end
            c1 (int): beginning column index
            c2 (int): ending column index (inclusive), -1 to the end
            
        Returns:
            (list) list of lists of results
            
        Examples:
            #copy 1st row and append to last row
            ll = wks.to_list2(0,0)
            wks.from_list2(ll,wks.rows)
        """
        nrow, ncol = self.shape

        if c2 < 0: c2 = ncol - 1
        if r2 < 0: r2 = nrow - 1    
            
        if c1 > c2:
            raise ValueError('c1 should be less than c2')

        if r1 > r2:
            raise ValueError('r1 should be less than r2')

        if c1 < 0 or c2 >= ncol or r1 < 0 or r2 >= nrow:
            raise ValueError('coloumn and row index must be within worksheet')
        
        return self.obj.GetData(r1, c1, r2, c2)

    def from_list2(self, data, row=0, col=0):
        """
        set a block of cells
        
        Parameters:
            data (list): a list of lists of values
            row (int): starting row index
            col (int or str): starting col, by index of by name
        
        Returns:
            (None)
        
        Examples:
            wks=op.find_sheet()
            #append a row to worksheet
            data = [ [10], [40] ]
            wks.from_list2(data, wks.rows)
        See Also:
            to_list2
        """
        ncol = self._col_index(col)
        self.obj.SetData(data, row, ncol)

    def _as_datetime(self, dt, customfmt, col, fmt):
        colobj = self._find_col(col)
        colobj.SetDataFormat(dt)
        if isinstance(fmt, int):
            colobj.SetDisplayFormat(fmt)
        else:
            colobj.SetDisplayFormat(customfmt)
            colobj.SetCustomDisplay(fmt)

    def as_date(self, col, fmt):
        """
        set col as date format
        
        Parameters:
            col (int or str): If int, column index. If str, column name, short name first, then long name.
            fmt (int or str): can be custom format string, or date display option as int
        
        Returns:
            (None)
        
        Examples:
            wks=op.find_sheet()
            data = [1,2,3,4,5,6]
            wks.from_list(0, data)
            wks.as_date(0, "yyyy'Q'q")
        See Also:
        """
        self._as_datetime(po.DF_DATE, 21, col, fmt)

    def as_time(self, col, fmt):
        """
        set col as time format
        
        Parameters:
            col (int or str): If int, column index. If str, column name, short name first, then long name.
            fmt (int or str): can be custom format string, or time display option as int
        
        Returns:
            (None)
        
        Examples:
            wks=op.find_sheet()
            data = [1.1,2.2,3.3,4.4,5.5,6.6]
            wks.from_list(0, data)
            wks.as_time(0, "D:hh':'mm':'ss TT")
        See Also:
        """
        self._as_datetime(po.DF_TIME, 17, col, fmt)

    def move_cols(self, n, c1, ncols=1):
        """
        moves a contiguous set of columns
        
        Parameters:
            n (int): by how many positions to move; if negative, columns are moved left,
                                otherwise right.
            c1 (int): the index of the first columns in the contiguous set.
            ncols (int): the total number of columns in the contiguous set.
        
        Returns:
            (bool)    returns True for success.
        
        Examples:
            wks=op.find_sheet()
            wks.move_cols(2, 1, 3)    # it moves three columns, beginning with the second column,
                                    # by two positions right.
        See Also:
        """
        return self.obj.MoveColumns(c1, ncols, n)

        
class WBook(DBook):
    """
    This class represents an Origin Workbook, it holds an instance of a PyOrigin WorksheetPage.
    """
    def _sheet(self, obj):
        return WSheet(obj)
        
    def add_sheet(self, name='', active=True):
        """
        add a new worksheet to the workbook
        
        Parameters:
            name (str): the name of the new sheet. If not specified, default names will be used
            active (bool): Activate the newly added sheet or not
        """
        return WSheet(self._add_sheet(name, active))

def from_series(*args):
    data = []
    for s in args:
        if isinstance(s, pd.Series):
            dtype = str(s.dtype)
            if dtype == 'object':
                s1 = check_convert_py_time_to_tdlt(s)
                if s1 is not s:
                    s = s1
                    dtype = 'timedelta'
            if dtype == 'datetime64[ns]':
                d = _series_to_julian_date(s)
            elif dtype == 'timedelta':
                d = _series_to_time(s)
            else:
                d = s
        else:
            d = s
        data.append(d)
    return data
