# Copyright (C) 2025 Shoal Software LLC. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import enum

from pants.option.option_types import (
    BoolOption,
    DictOption,
    EnumOption,
    FloatOption,
    IntOption,
    StrOption,
)
from pants.option.subsystem import Subsystem
from pants.util.strutil import softwrap


class TracingExporterId(enum.Enum):
    HTTP = "http"
    GRPC = "grpc"
    JSON_FILE = "json-file"


class OtelCompression(enum.Enum):
    GZIP = "gzip"
    DEFLATE = "deflate"
    NONE = "none"


class TelemetrySubsystem(Subsystem):
    options_scope = "shoalsoft-opentelemetry"
    help = "Pants OpenTelemetry plugin from Shoal Software LLC"

    enabled = BoolOption(default=False, help="Whether to enable emitting OpenTelemetry spans.")

    exporter = EnumOption(
        default=TracingExporterId.JSON_FILE,
        help=softwrap(
            f"""
            Set the exporter to use when exporting workunits to external tracing systems. Choices are
            `{TracingExporterId.HTTP.value}` (OpenTelemetry OTLP HTTP),
            `{TracingExporterId.GRPC.value}` (OpenTelemetry OTLP GRPC), and
            `{TracingExporterId.JSON_FILE.value}` (OpenTelemetry debug output to a file).
            Default is `{TracingExporterId.JSON_FILE.value}`.
            """
        ),
    )

    finish_timeout = FloatOption(
        default=2.0,
        help=softwrap(
            """
            The number of seconds to wait at the end of a session for export of all tracing spans to OpenTelemetry
            to complete.
            """
        ),
    )

    parse_traceparent = BoolOption(
        default=True,
        help=softwrap(
            """
            If `True`, then parse the `TRACEPARENT` environment variable if it is present in the environment.
            `TRACEPARENT` contains the parent trace ID and parent span ID to which to link any trace generated by
            this plugin. This is useful for linking Pants-related traces together with the trace for the CI job.

            The format of the `TRACEPARENT` environment variable is defined in the W3C Trace Context specification:
            https://www.w3.org/TR/trace-context/#traceparent-header
            """
        ),
    )

    async_completion = BoolOption(
        default=True,
        help=softwrap(
            """
            If `True`, allows the plugin to finish asynchronously when Pants is shutting down. This can result in
            faster Pants exit times but may lead to some spans not being exported if the export process is slow.
            If `False`, forces synchronous completion, ensuring all spans are exported before Pants exits but
            potentially slowing down the shutdown process.
            """
        ),
    )

    json_file = StrOption(
        default="dist/otel-json-trace.jsonl",
        help=softwrap(
            f"""
            If set, Pants will write OpenTelemetry tracing spans to a local file for easier debugging. Each line
            will be a tracing span in OpenTelemetry's JSON format. The filename is relative to the build root. Export
            will only occur if the `--shoalsoft-opentelemetry-exporter` is set to `{TracingExporterId.JSON_FILE.value}`.
            """
        ),
    )

    exporter_endpoint = StrOption(
        default=None,
        help=softwrap(
            f"""
            The target to which the span exporter is going to send spans. The endpoint MUST be a valid URL host,
            and MAY contain a scheme (http or https), port and path. A scheme of https indicates a secure
            connection and takes precedence over this configuration setting. Endpoint which will receive exported
            tracing spans.

            This option is consumed by both the `{TracingExporterId.HTTP.value}` and `{TracingExporterId.GRPC.value}`
            exporters.

            Corresponds to the `OTEL_EXPORTER_OTLP_TRACES_ENDPOINT` and `OTEL_EXPORTER_OTLP_ENDPOINT`
            environment variables.
            """
        ),
    )

    exporter_certificate_file = StrOption(
        default=None,
        advanced=True,
        help=softwrap(
            """
            The path to the certificate file for TLS credentials of gRPC client for traces.
            Should only be used for a secure connection for tracing.

            Corresponds to the `OTEL_EXPORTER_OTLP_TRACES_CLIENT_KEY` and `OTEL_EXPORTER_OTLP_CERTIFICATE`
            environment variables.
            """
        ),
    )

    exporter_client_key_file = StrOption(
        default=None,
        advanced=True,
        help=softwrap(
            """
            The path to the client private key to use in mTLS communication in PEM format for traces.

            Corresponds to the `OTEL_EXPORTER_OTLP_TRACES_CLIENT_KEY` and `OTEL_EXPORTER_OTLP_CLIENT_KEY`
            environment variables.
            """
        ),
    )

    exporter_client_certificate_file = StrOption(
        default=None,
        advanced=True,
        help=softwrap(
            """
            The path to the client certificate/chain trust for clients private key to use in mTLS
            communication in PEM format for traces.

            Corresponds to the `OTEL_EXPORTER_OTLP_TRACES_CLIENT_CERTIFICATE` and
            `OTEL_EXPORTER_OTLP_CLIENT_CERTIFICATE` environment variables.
            """
        ),
    )

    exporter_headers = DictOption[str](
        advanced=True,
        help=softwrap(
            """
            The key-value pairs to be used as headers for spans associated with gRPC or HTTP requests.

            This option is consumed by both the `{TracingExporterId.HTTP.value}` and `{TracingExporterId.GRPC.value}`
            exporters.

            Corresponds to the `OTEL_EXPORTER_OTLP_TRACES_HEADERS` and `OTEL_EXPORTER_OTLP_HEADERS`
            environment variables.
            """
        ),
    )

    exporter_timeout = IntOption(
        default=None,
        advanced=True,
        help=softwrap(
            f"""
            The maximum time the OTLP exporter will wait for each batch export for spans.

            This option is consumed by both the `{TracingExporterId.HTTP.value}` and `{TracingExporterId.GRPC.value}`
            exporters.

            Corresponds to the `OTEL_EXPORTER_OTLP_TRACES_TIMEOUT` and `OTEL_EXPORTER_OTLP_TIMEOUT`
            environment variables.
            """
        ),
    )

    exporter_compression = EnumOption(
        default=OtelCompression.NONE,
        advanced=True,
        help=softwrap(
            f"""
            Specifies a gRPC compression method to be used in the OTLP exporters. Possible values are `gzip`,
            `deflate`, and `none`.

            This option is consumed by both the `{TracingExporterId.HTTP.value}` and `{TracingExporterId.GRPC.value}`
            exporters.

            Corresponds to the `OTEL_EXPORTER_OTLP_TRACES_COMPRESSION` and `OTEL_EXPORTER_OTLP_COMPRESSION`
            environment variables.
            """
        ),
    )

    exporter_insecure = BoolOption(
        default=None,
        advanced=True,
        help=softwrap(
            f"""
            Represents whether to enable client transport security for gRPC requests. An endpoint scheme of https
            will error if this option is in effect since `https` implies secure mode.

            This option is consumed by the `{TracingExporterId.GRPC.value}` exporter.

            Corresponds to the `OTEL_EXPORTER_OTLP_TRACES_INSECURE` and `OTEL_EXPORTER_OTLP_INSECURE`
            environment variables.
            """
        ),
    )
