import { IPrincipal, ServicePrincipalOpts } from './principals';
/**
 * Represents a statement in an IAM policy document.
 *
 * @stability stable
 */
export declare class PolicyStatement {
    /**
     * Creates a new PolicyStatement based on the object provided.
     *
     * This will accept an object created from the `.toJSON()` call
     *
     * @param obj the PolicyStatement in object form.
     * @stability stable
     */
    static fromJson(obj: any): PolicyStatement;
    /**
     * Statement ID for this statement.
     *
     * @stability stable
     */
    sid?: string;
    /**
     * Whether to allow or deny the actions in this statement.
     *
     * @stability stable
     */
    effect: Effect;
    private readonly action;
    private readonly notAction;
    private readonly principal;
    private readonly notPrincipal;
    private readonly resource;
    private readonly notResource;
    private readonly condition;
    private principalConditionsJson?;
    /**
     * @stability stable
     */
    constructor(props?: PolicyStatementProps);
    /**
     * Specify allowed actions into the "Action" section of the policy statement.
     *
     * @param actions actions that will be allowed.
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_action.html
     * @stability stable
     */
    addActions(...actions: string[]): void;
    /**
     * Explicitly allow all actions except the specified list of actions into the "NotAction" section of the policy document.
     *
     * @param notActions actions that will be denied.
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html
     * @stability stable
     */
    addNotActions(...notActions: string[]): void;
    /**
     * Indicates if this permission has a "Principal" section.
     *
     * @stability stable
     */
    get hasPrincipal(): boolean;
    /**
     * Adds principals to the "Principal" section of a policy statement.
     *
     * @param principals IAM principals that will be added.
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html
     * @stability stable
     */
    addPrincipals(...principals: IPrincipal[]): void;
    /**
     * Specify principals that is not allowed or denied access to the "NotPrincipal" section of a policy statement.
     *
     * @param notPrincipals IAM principals that will be denied access.
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notprincipal.html
     * @stability stable
     */
    addNotPrincipals(...notPrincipals: IPrincipal[]): void;
    private validatePolicyPrincipal;
    /**
     * Specify AWS account ID as the principal entity to the "Principal" section of a policy statement.
     *
     * @stability stable
     */
    addAwsAccountPrincipal(accountId: string): void;
    /**
     * Specify a principal using the ARN  identifier of the principal.
     *
     * You cannot specify IAM groups and instance profiles as principals.
     *
     * @param arn ARN identifier of AWS account, IAM user, or IAM role (i.e. arn:aws:iam::123456789012:user/user-name).
     * @stability stable
     */
    addArnPrincipal(arn: string): void;
    /**
     * Adds a service principal to this policy statement.
     *
     * @param service the service name for which a service principal is requested (e.g: `s3.amazonaws.com`).
     * @param opts options for adding the service principal (such as specifying a principal in a different region).
     * @stability stable
     */
    addServicePrincipal(service: string, opts?: ServicePrincipalOpts): void;
    /**
     * Adds a federated identity provider such as Amazon Cognito to this policy statement.
     *
     * @param federated federated identity provider (i.e. 'cognito-identity.amazonaws.com').
     * @param conditions The conditions under which the policy is in effect.
     * @stability stable
     */
    addFederatedPrincipal(federated: any, conditions: Conditions): void;
    /**
     * Adds an AWS account root user principal to this policy statement.
     *
     * @stability stable
     */
    addAccountRootPrincipal(): void;
    /**
     * Adds a canonical user ID principal to this policy document.
     *
     * @param canonicalUserId unique identifier assigned by AWS for every account.
     * @stability stable
     */
    addCanonicalUserPrincipal(canonicalUserId: string): void;
    /**
     * Adds all identities in all accounts ("*") to this policy statement.
     *
     * @stability stable
     */
    addAnyPrincipal(): void;
    /**
     * Specify resources that this policy statement applies into the "Resource" section of this policy statement.
     *
     * @param arns Amazon Resource Names (ARNs) of the resources that this policy statement applies to.
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_resource.html
     * @stability stable
     */
    addResources(...arns: string[]): void;
    /**
     * Specify resources that this policy statement will not apply to in the "NotResource" section of this policy statement.
     *
     * All resources except the specified list will be matched.
     *
     * @param arns Amazon Resource Names (ARNs) of the resources that this policy statement does not apply to.
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notresource.html
     * @stability stable
     */
    addNotResources(...arns: string[]): void;
    /**
     * Adds a ``"*"`` resource to this statement.
     *
     * @stability stable
     */
    addAllResources(): void;
    /**
     * Indicates if this permission has at least one resource associated with it.
     *
     * @stability stable
     */
    get hasResource(): boolean;
    /**
     * Add a condition to the Policy.
     *
     * @stability stable
     */
    addCondition(key: string, value: Condition): void;
    /**
     * Add multiple conditions to the Policy.
     *
     * @stability stable
     */
    addConditions(conditions: Conditions): void;
    /**
     * Add a condition that limits to a given account.
     *
     * @stability stable
     */
    addAccountCondition(accountId: string): void;
    /**
     * JSON-ify the policy statement.
     *
     * Used when JSON.stringify() is called
     *
     * @stability stable
     */
    toStatementJson(): any;
    /**
     * String representation of this policy statement.
     *
     * @stability stable
     */
    toString(): string;
    /**
     * JSON-ify the statement.
     *
     * Used when JSON.stringify() is called
     *
     * @stability stable
     */
    toJSON(): any;
    /**
     * Add a principal's conditions
     *
     * For convenience, principals have been modeled as both a principal
     * and a set of conditions. This makes it possible to have a single
     * object represent e.g. an "SNS Topic" (SNS service principal + aws:SourcArn
     * condition) or an Organization member (* + aws:OrgId condition).
     *
     * However, when using multiple principals in the same policy statement,
     * they must all have the same conditions or the OR samentics
     * implied by a list of principals cannot be guaranteed (user needs to
     * add multiple statements in that case).
     */
    private addPrincipalConditions;
    /**
     * Validate that the policy statement satisfies base requirements for a policy.
     *
     * @stability stable
     */
    validateForAnyPolicy(): string[];
    /**
     * Validate that the policy statement satisfies all requirements for a resource-based policy.
     *
     * @stability stable
     */
    validateForResourcePolicy(): string[];
    /**
     * Validate that the policy statement satisfies all requirements for an identity-based policy.
     *
     * @stability stable
     */
    validateForIdentityPolicy(): string[];
}
/**
 * The Effect element of an IAM policy.
 *
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_effect.html
 * @stability stable
 */
export declare enum Effect {
    /**
     * Allows access to a resource in an IAM policy statement.
     *
     * By default, access to resources are denied.
     *
     * @stability stable
     */
    ALLOW = "Allow",
    /**
     * Explicitly deny access to a resource.
     *
     * By default, all requests are denied implicitly.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_evaluation-logic.html
     * @stability stable
     */
    DENY = "Deny"
}
/**
 * Condition for when an IAM policy is in effect. Maps from the keys in a request's context to
 * a string value or array of string values. See the Conditions interface for more details.
 */
export declare type Condition = any;
/**
 * Conditions for when an IAM Policy is in effect, specified in the following structure:
 *
 * `{ "Operator": { "keyInRequestContext": "value" } }`
 *
 * The value can be either a single string value or an array of string values.
 *
 * For more information, including which operators are supported, see [the IAM
 * documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
 */
export declare type Conditions = Record<string, Condition>;
/**
 * Interface for creating a policy statement.
 *
 * @stability stable
 */
export interface PolicyStatementProps {
    /**
     * The Sid (statement ID) is an optional identifier that you provide for the policy statement.
     *
     * You can assign a Sid value to each statement in a
     * statement array. In services that let you specify an ID element, such as
     * SQS and SNS, the Sid value is just a sub-ID of the policy document's ID. In
     * IAM, the Sid value must be unique within a JSON policy.
     *
     * @default - no sid
     * @stability stable
     */
    readonly sid?: string;
    /**
     * List of actions to add to the statement.
     *
     * @default - no actions
     * @stability stable
     */
    readonly actions?: string[];
    /**
     * List of not actions to add to the statement.
     *
     * @default - no not-actions
     * @stability stable
     */
    readonly notActions?: string[];
    /**
     * List of principals to add to the statement.
     *
     * @default - no principals
     * @stability stable
     */
    readonly principals?: IPrincipal[];
    /**
     * List of not principals to add to the statement.
     *
     * @default - no not principals
     * @stability stable
     */
    readonly notPrincipals?: IPrincipal[];
    /**
     * Resource ARNs to add to the statement.
     *
     * @default - no resources
     * @stability stable
     */
    readonly resources?: string[];
    /**
     * NotResource ARNs to add to the statement.
     *
     * @default - no not-resources
     * @stability stable
     */
    readonly notResources?: string[];
    /**
     * Conditions to add to the statement.
     *
     * @default - no condition
     * @stability stable
     */
    readonly conditions?: {
        [key: string]: any;
    };
    /**
     * Whether to allow or deny the actions in this statement.
     *
     * @default Effect.ALLOW
     * @stability stable
     */
    readonly effect?: Effect;
}
