"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CompositePrincipal = exports.Anyone = exports.AnyPrincipal = exports.AccountRootPrincipal = exports.OpenIdConnectPrincipal = exports.WebIdentityPrincipal = exports.FederatedPrincipal = exports.CanonicalUserPrincipal = exports.OrganizationPrincipal = exports.ServicePrincipal = exports.AccountPrincipal = exports.ArnPrincipal = exports.PrincipalPolicyFragment = exports.PrincipalWithConditions = exports.PrincipalBase = void 0;
const cdk = require("@aws-cdk/core");
const region_info_1 = require("@aws-cdk/region-info");
const util_1 = require("./util");
/**
 * Base class for policy principals
 */
class PrincipalBase {
    constructor() {
        this.grantPrincipal = this;
        this.principalAccount = undefined;
        /**
         * When this Principal is used in an AssumeRole policy, the action to use.
         */
        this.assumeRoleAction = 'sts:AssumeRole';
    }
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
    addToPrincipalPolicy(_statement) {
        // This base class is used for non-identity principals. None of them
        // have a PolicyDocument to add to.
        return { statementAdded: false };
    }
    toString() {
        // This is a first pass to make the object readable. Descendant principals
        // should return something nicer.
        return JSON.stringify(this.policyFragment.principalJson);
    }
    /**
     * JSON-ify the principal
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        // Have to implement toJSON() because the default will lead to infinite recursion.
        return this.policyFragment.principalJson;
    }
    /**
     * Returns a new PrincipalWithConditions using this principal as the base, with the
     * passed conditions added.
     *
     * When there is a value for the same operator and key in both the principal and the
     * conditions parameter, the value from the conditions parameter will be used.
     *
     * @returns a new PrincipalWithConditions object.
     */
    withConditions(conditions) {
        return new PrincipalWithConditions(this, conditions);
    }
}
exports.PrincipalBase = PrincipalBase;
/**
 * An IAM principal with additional conditions specifying when the policy is in effect.
 *
 * For more information about conditions, see:
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html
 */
class PrincipalWithConditions {
    constructor(principal, conditions) {
        this.principal = principal;
        this.grantPrincipal = this;
        this.assumeRoleAction = this.principal.assumeRoleAction;
        this.additionalConditions = conditions;
    }
    /**
     * Add a condition to the principal
     */
    addCondition(key, value) {
        const existingValue = this.conditions[key];
        this.conditions[key] = existingValue ? { ...existingValue, ...value } : value;
    }
    /**
     * Adds multiple conditions to the principal
     *
     * Values from the conditions parameter will overwrite existing values with the same operator
     * and key.
     */
    addConditions(conditions) {
        Object.entries(conditions).forEach(([key, value]) => {
            this.addCondition(key, value);
        });
    }
    /**
     * The conditions under which the policy is in effect.
     * See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    get conditions() {
        return this.mergeConditions(this.principal.policyFragment.conditions, this.additionalConditions);
    }
    get policyFragment() {
        return new PrincipalPolicyFragment(this.principal.policyFragment.principalJson, this.conditions);
    }
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
    addToPrincipalPolicy(statement) {
        return this.principal.addToPrincipalPolicy(statement);
    }
    toString() {
        return this.principal.toString();
    }
    /**
     * JSON-ify the principal
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        // Have to implement toJSON() because the default will lead to infinite recursion.
        return this.policyFragment.principalJson;
    }
    mergeConditions(principalConditions, additionalConditions) {
        const mergedConditions = {};
        Object.entries(principalConditions).forEach(([operator, condition]) => {
            mergedConditions[operator] = condition;
        });
        Object.entries(additionalConditions).forEach(([operator, condition]) => {
            // merge the conditions if one of the additional conditions uses an
            // operator that's already used by the principal's conditions merge the
            // inner structure.
            const existing = mergedConditions[operator];
            if (!existing) {
                mergedConditions[operator] = condition;
                return; // continue
            }
            // if either the existing condition or the new one contain unresolved
            // tokens, fail the merge. this is as far as we go at this point.
            if (cdk.Token.isUnresolved(condition) || cdk.Token.isUnresolved(existing)) {
                throw new Error(`multiple "${operator}" conditions cannot be merged if one of them contains an unresolved token`);
            }
            mergedConditions[operator] = { ...existing, ...condition };
        });
        return mergedConditions;
    }
}
exports.PrincipalWithConditions = PrincipalWithConditions;
/**
 * A collection of the fields in a PolicyStatement that can be used to identify a principal.
 *
 * This consists of the JSON used in the "Principal" field, and optionally a
 * set of "Condition"s that need to be applied to the policy.
 */
class PrincipalPolicyFragment {
    /**
     *
     * @param principalJson JSON of the "Principal" section in a policy statement
     * @param conditions conditions that need to be applied to this policy
     */
    constructor(principalJson, 
    /**
     * The conditions under which the policy is in effect.
     * See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    conditions = {}) {
        this.principalJson = principalJson;
        this.conditions = conditions;
    }
}
exports.PrincipalPolicyFragment = PrincipalPolicyFragment;
/**
 * Specify a principal by the Amazon Resource Name (ARN).
 * You can specify AWS accounts, IAM users, Federated SAML users, IAM roles, and specific assumed-role sessions.
 * You cannot specify IAM groups or instance profiles as principals
 *
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html
 */
class ArnPrincipal extends PrincipalBase {
    /**
     *
     * @param arn Amazon Resource Name (ARN) of the principal entity (i.e. arn:aws:iam::123456789012:user/user-name)
     */
    constructor(arn) {
        super();
        this.arn = arn;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ AWS: [this.arn] });
    }
    toString() {
        return `ArnPrincipal(${this.arn})`;
    }
}
exports.ArnPrincipal = ArnPrincipal;
/**
 * Specify AWS account ID as the principal entity in a policy to delegate authority to the account.
 */
class AccountPrincipal extends ArnPrincipal {
    /**
     *
     * @param accountId AWS account ID (i.e. 123456789012)
     */
    constructor(accountId) {
        super(new StackDependentToken(stack => `arn:${stack.partition}:iam::${accountId}:root`).toString());
        this.accountId = accountId;
    }
    toString() {
        return `AccountPrincipal(${this.accountId})`;
    }
}
exports.AccountPrincipal = AccountPrincipal;
/**
 * An IAM principal that represents an AWS service (i.e. sqs.amazonaws.com).
 */
class ServicePrincipal extends PrincipalBase {
    /**
     *
     * @param service AWS service (i.e. sqs.amazonaws.com)
     */
    constructor(service, opts = {}) {
        super();
        this.service = service;
        this.opts = opts;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({
            Service: [
                new ServicePrincipalToken(this.service, this.opts).toString(),
            ],
        }, this.opts.conditions);
    }
    toString() {
        return `ServicePrincipal(${this.service})`;
    }
}
exports.ServicePrincipal = ServicePrincipal;
/**
 * A principal that represents an AWS Organization
 */
class OrganizationPrincipal extends PrincipalBase {
    /**
     *
     * @param organizationId The unique identifier (ID) of an organization (i.e. o-12345abcde)
     */
    constructor(organizationId) {
        super();
        this.organizationId = organizationId;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ AWS: ['*'] }, { StringEquals: { 'aws:PrincipalOrgID': this.organizationId } });
    }
    toString() {
        return `OrganizationPrincipal(${this.organizationId})`;
    }
}
exports.OrganizationPrincipal = OrganizationPrincipal;
/**
 * A policy principal for canonicalUserIds - useful for S3 bucket policies that use
 * Origin Access identities.
 *
 * See https://docs.aws.amazon.com/general/latest/gr/acct-identifiers.html
 *
 * and
 *
 * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html
 *
 * for more details.
 *
 */
class CanonicalUserPrincipal extends PrincipalBase {
    /**
     *
     * @param canonicalUserId unique identifier assigned by AWS for every account.
     *   root user and IAM users for an account all see the same ID.
     *   (i.e. 79a59df900b949e55d96a1e698fbacedfd6e09d98eacf8f8d5218e7cd47ef2be)
     */
    constructor(canonicalUserId) {
        super();
        this.canonicalUserId = canonicalUserId;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ CanonicalUser: [this.canonicalUserId] });
    }
    toString() {
        return `CanonicalUserPrincipal(${this.canonicalUserId})`;
    }
}
exports.CanonicalUserPrincipal = CanonicalUserPrincipal;
/**
 * Principal entity that represents a federated identity provider such as Amazon Cognito,
 * that can be used to provide temporary security credentials to users who have been authenticated.
 * Additional condition keys are available when the temporary security credentials are used to make a request.
 * You can use these keys to write policies that limit the access of federated users.
 *
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#condition-keys-wif
 */
class FederatedPrincipal extends PrincipalBase {
    /**
     *
     * @param federated federated identity provider (i.e. 'cognito-identity.amazonaws.com' for users authenticated through Cognito)
     * @param conditions The conditions under which the policy is in effect.
     *   See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    constructor(federated, conditions, assumeRoleAction = 'sts:AssumeRole') {
        super();
        this.federated = federated;
        this.conditions = conditions;
        this.assumeRoleAction = assumeRoleAction;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ Federated: [this.federated] }, this.conditions);
    }
    toString() {
        return `FederatedPrincipal(${this.federated})`;
    }
}
exports.FederatedPrincipal = FederatedPrincipal;
/**
 * A principal that represents a federated identity provider as Web Identity such as Cognito, Amazon,
 * Facebook, Google, etc.
 */
class WebIdentityPrincipal extends FederatedPrincipal {
    /**
     *
     * @param identityProvider identity provider (i.e. 'cognito-identity.amazonaws.com' for users authenticated through Cognito)
     * @param conditions The conditions under which the policy is in effect.
     *   See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    constructor(identityProvider, conditions = {}) {
        super(identityProvider, conditions !== null && conditions !== void 0 ? conditions : {}, 'sts:AssumeRoleWithWebIdentity');
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ Federated: [this.federated] }, this.conditions);
    }
    toString() {
        return `WebIdentityPrincipal(${this.federated})`;
    }
}
exports.WebIdentityPrincipal = WebIdentityPrincipal;
/**
 * A principal that represents a federated identity provider as from a OpenID Connect provider.
 */
class OpenIdConnectPrincipal extends WebIdentityPrincipal {
    /**
     *
     * @param openIdConnectProvider OpenID Connect provider
     * @param conditions The conditions under which the policy is in effect.
     *   See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    constructor(openIdConnectProvider, conditions = {}) {
        super(openIdConnectProvider.openIdConnectProviderArn, conditions !== null && conditions !== void 0 ? conditions : {});
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ Federated: [this.federated] }, this.conditions);
    }
    toString() {
        return `OpenIdConnectPrincipal(${this.federated})`;
    }
}
exports.OpenIdConnectPrincipal = OpenIdConnectPrincipal;
/**
 * Use the AWS account into which a stack is deployed as the principal entity in a policy
 */
class AccountRootPrincipal extends AccountPrincipal {
    constructor() {
        super(new StackDependentToken(stack => stack.account).toString());
    }
    toString() {
        return 'AccountRootPrincipal()';
    }
}
exports.AccountRootPrincipal = AccountRootPrincipal;
/**
 * A principal representing all identities in all accounts
 */
class AnyPrincipal extends ArnPrincipal {
    constructor() {
        super('*');
    }
    toString() {
        return 'AnyPrincipal()';
    }
}
exports.AnyPrincipal = AnyPrincipal;
/**
 * A principal representing all identities in all accounts
 * @deprecated use `AnyPrincipal`
 */
class Anyone extends AnyPrincipal {
}
exports.Anyone = Anyone;
/**
 * Represents a principal that has multiple types of principals. A composite principal cannot
 * have conditions. i.e. multiple ServicePrincipals that form a composite principal
 */
class CompositePrincipal extends PrincipalBase {
    constructor(...principals) {
        super();
        this.principals = new Array();
        if (principals.length === 0) {
            throw new Error('CompositePrincipals must be constructed with at least 1 Principal but none were passed.');
        }
        this.assumeRoleAction = principals[0].assumeRoleAction;
        this.addPrincipals(...principals);
    }
    /**
     * Adds IAM principals to the composite principal. Composite principals cannot have
     * conditions.
     *
     * @param principals IAM principals that will be added to the composite principal
     */
    addPrincipals(...principals) {
        for (const p of principals) {
            if (p.assumeRoleAction !== this.assumeRoleAction) {
                throw new Error('Cannot add multiple principals with different "assumeRoleAction". ' +
                    `Expecting "${this.assumeRoleAction}", got "${p.assumeRoleAction}"`);
            }
            const fragment = p.policyFragment;
            if (fragment.conditions && Object.keys(fragment.conditions).length > 0) {
                throw new Error('Components of a CompositePrincipal must not have conditions. ' +
                    `Tried to add the following fragment: ${JSON.stringify(fragment)}`);
            }
            this.principals.push(p);
        }
        return this;
    }
    get policyFragment() {
        const principalJson = {};
        for (const p of this.principals) {
            util_1.mergePrincipal(principalJson, p.policyFragment.principalJson);
        }
        return new PrincipalPolicyFragment(principalJson);
    }
    toString() {
        return `CompositePrincipal(${this.principals})`;
    }
}
exports.CompositePrincipal = CompositePrincipal;
/**
 * A lazy token that requires an instance of Stack to evaluate
 */
class StackDependentToken {
    constructor(fn) {
        this.fn = fn;
        this.creationStack = cdk.captureStackTrace();
    }
    resolve(context) {
        return this.fn(cdk.Stack.of(context.scope));
    }
    toString() {
        return cdk.Token.asString(this);
    }
    /**
     * JSON-ify the token
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        return '<unresolved-token>';
    }
}
class ServicePrincipalToken {
    constructor(service, opts) {
        this.service = service;
        this.opts = opts;
        this.creationStack = cdk.captureStackTrace();
    }
    resolve(ctx) {
        const region = this.opts.region || cdk.Stack.of(ctx.scope).region;
        const fact = region_info_1.RegionInfo.get(region).servicePrincipal(this.service);
        return fact || region_info_1.Default.servicePrincipal(this.service, region, cdk.Aws.URL_SUFFIX);
    }
    toString() {
        return cdk.Token.asString(this, {
            displayHint: this.service,
        });
    }
    /**
     * JSON-ify the token
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        return `<${this.service}>`;
    }
}
//# sourceMappingURL=data:application/json;base64,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