"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UnknownPrincipal = void 0;
const core_1 = require("@aws-cdk/core");
/**
 * A principal for use in resources that need to have a role but it's unknown
 *
 * Some resources have roles associated with them which they assume, such as
 * Lambda Functions, CodeBuild projects, StepFunctions machines, etc.
 *
 * When those resources are imported, their actual roles are not always
 * imported with them. When that happens, we use an instance of this class
 * instead, which will add user warnings when statements are attempted to be
 * added to it.
 */
class UnknownPrincipal {
    constructor(props) {
        this.assumeRoleAction = 'sts:AssumeRole';
        this.resource = props.resource;
        this.grantPrincipal = this;
    }
    get policyFragment() {
        throw new Error(`Cannot get policy fragment of ${this.resource.node.path}, resource imported without a role`);
    }
    addToPrincipalPolicy(statement) {
        const stack = core_1.Stack.of(this.resource);
        const repr = JSON.stringify(stack.resolve(statement));
        core_1.Annotations.of(this.resource).addWarning(`Add statement to this resource's role: ${repr}`);
        // Pretend we did the work. The human will do it for us, eventually.
        return { statementAdded: true, policyDependable: new core_1.ConcreteDependable() };
    }
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
}
exports.UnknownPrincipal = UnknownPrincipal;
//# sourceMappingURL=data:application/json;base64,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