"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.mergePrincipal = exports.AttachedPolicies = exports.generatePolicyName = exports.undefinedIfEmpty = void 0;
const core_1 = require("@aws-cdk/core");
const MAX_POLICY_NAME_LEN = 128;
function undefinedIfEmpty(f) {
    return core_1.Lazy.listValue({
        produce: () => {
            const array = f();
            return (array && array.length > 0) ? array : undefined;
        },
    });
}
exports.undefinedIfEmpty = undefinedIfEmpty;
/**
 * Used to generate a unique policy name based on the policy resource construct.
 * The logical ID of the resource is a great candidate as long as it doesn't exceed
 * 128 characters, so we take the last 128 characters (in order to make sure the hash
 * is there).
 */
function generatePolicyName(scope, logicalId) {
    // as logicalId is itself a Token, resolve it first
    const resolvedLogicalId = core_1.Tokenization.resolve(logicalId, {
        scope,
        resolver: new core_1.DefaultTokenResolver(new core_1.StringConcat()),
    });
    return lastNCharacters(resolvedLogicalId, MAX_POLICY_NAME_LEN);
}
exports.generatePolicyName = generatePolicyName;
/**
 * Returns a string composed of the last n characters of str.
 * If str is shorter than n, returns str.
 *
 * @param str the string to return the last n characters of
 * @param n how many characters to return
 */
function lastNCharacters(str, n) {
    const startIndex = Math.max(str.length - n, 0);
    return str.substring(startIndex, str.length);
}
/**
 * Helper class that maintains the set of attached policies for a principal.
 */
class AttachedPolicies {
    constructor() {
        this.policies = new Array();
    }
    /**
     * Adds a policy to the list of attached policies.
     *
     * If this policy is already, attached, returns false.
     * If there is another policy attached with the same name, throws an exception.
     */
    attach(policy) {
        if (this.policies.find(p => p === policy)) {
            return; // already attached
        }
        if (this.policies.find(p => p.policyName === policy.policyName)) {
            throw new Error(`A policy named "${policy.policyName}" is already attached`);
        }
        this.policies.push(policy);
    }
}
exports.AttachedPolicies = AttachedPolicies;
/**
 * Merge two dictionaries that represent IAM principals
 */
function mergePrincipal(target, source) {
    for (const key of Object.keys(source)) {
        target[key] = target[key] || [];
        let value = source[key];
        if (!Array.isArray(value)) {
            value = [value];
        }
        target[key].push(...value);
    }
    return target;
}
exports.mergePrincipal = mergePrincipal;
//# sourceMappingURL=data:application/json;base64,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