"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const iam = require("../lib");
test('use of cross-stack role reference does not lead to URLSuffix being exported', () => {
    // GIVEN
    const app = new core_1.App();
    const first = new core_1.Stack(app, 'First');
    const second = new core_1.Stack(app, 'Second');
    // WHEN
    const role = new iam.Role(first, 'Role', {
        assumedBy: new iam.ServicePrincipal('s3.amazonaws.com'),
    });
    new core_1.CfnOutput(second, 'Output', {
        value: role.roleArn,
    });
    // THEN
    app.synth();
    expect(first).toMatchTemplate({
        Resources: {
            Role1ABCC5F0: {
                Type: 'AWS::IAM::Role',
                Properties: {
                    AssumeRolePolicyDocument: {
                        Statement: [
                            {
                                Action: 'sts:AssumeRole',
                                Effect: 'Allow',
                                Principal: { Service: 's3.amazonaws.com' },
                            },
                        ],
                        Version: '2012-10-17',
                    },
                },
            },
        },
        Outputs: {
            ExportsOutputFnGetAttRole1ABCC5F0ArnB4C0B73E: {
                Value: { 'Fn::GetAtt': ['Role1ABCC5F0', 'Arn'] },
                Export: {
                    Name: 'First:ExportsOutputFnGetAttRole1ABCC5F0ArnB4C0B73E',
                },
            },
        },
    });
});
test('cannot have multiple principals with different conditions in the same statement', () => {
    const stack = new core_1.Stack(undefined, 'First');
    const user = new iam.User(stack, 'User');
    expect(() => {
        user.addToPolicy(new iam.PolicyStatement({
            principals: [
                new iam.ServicePrincipal('myService.amazon.com', {
                    conditions: {
                        StringEquals: {
                            hairColor: 'blond',
                        },
                    },
                }),
                new iam.ServicePrincipal('yourservice.amazon.com', {
                    conditions: {
                        StringEquals: {
                            hairColor: 'black',
                        },
                    },
                }),
            ],
        }));
    }).toThrow(/All principals in a PolicyStatement must have the same Conditions/);
});
test('can have multiple principals the same conditions in the same statement', () => {
    const stack = new core_1.Stack(undefined, 'First');
    const user = new iam.User(stack, 'User');
    user.addToPolicy(new iam.PolicyStatement({
        principals: [
            new iam.ServicePrincipal('myService.amazon.com'),
            new iam.ServicePrincipal('yourservice.amazon.com'),
        ],
    }));
    user.addToPolicy(new iam.PolicyStatement({
        principals: [
            new iam.ServicePrincipal('myService.amazon.com', {
                conditions: {
                    StringEquals: { hairColor: 'blond' },
                },
            }),
            new iam.ServicePrincipal('yourservice.amazon.com', {
                conditions: {
                    StringEquals: { hairColor: 'blond' },
                },
            }),
        ],
    }));
});
test('use Web Identity principal', () => {
    // GIVEN
    const stack = new core_1.Stack();
    // WHEN
    const principal = new iam.WebIdentityPrincipal('cognito-identity.amazonaws.com');
    // THEN
    expect(stack.resolve(principal.federated)).toStrictEqual('cognito-identity.amazonaws.com');
    expect(stack.resolve(principal.assumeRoleAction)).toStrictEqual('sts:AssumeRoleWithWebIdentity');
});
test('use OpenID Connect principal from provider', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const provider = new iam.OpenIdConnectProvider(stack, 'MyProvider', {
        url: 'https://openid-endpoint',
    });
    // WHEN
    const principal = new iam.OpenIdConnectPrincipal(provider);
    // THEN
    expect(stack.resolve(principal.federated)).toStrictEqual({ Ref: 'MyProvider730BA1C8' });
});
//# sourceMappingURL=data:application/json;base64,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