# Pyrogram - Telegram MTProto API Client Library for Python
# Copyright (C) 2017-2019 Dan Tès <https://github.com/delivrance>
#
# This file is part of Pyrogram.
#
# Pyrogram is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Pyrogram is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with Pyrogram.  If not, see <http://www.gnu.org/licenses/>.

from io import BytesIO

from pyrogram.api.core import *


class SecureSecretSettings(TLObject):
    """Attributes:
        LAYER: ``105``

    Attributes:
        ID: ``0x1527bcac``

    Parameters:
        secure_algo: Either :obj:`SecurePasswordKdfAlgoUnknown <pyrogram.api.types.SecurePasswordKdfAlgoUnknown>`, :obj:`SecurePasswordKdfAlgoPBKDF2HMACSHA512iter100000 <pyrogram.api.types.SecurePasswordKdfAlgoPBKDF2HMACSHA512iter100000>` or :obj:`SecurePasswordKdfAlgoSHA512 <pyrogram.api.types.SecurePasswordKdfAlgoSHA512>`
        secure_secret: ``bytes``
        secure_secret_id: ``int`` ``64-bit``
    """

    __slots__ = ["secure_algo", "secure_secret", "secure_secret_id"]

    ID = 0x1527bcac
    QUALNAME = "types.SecureSecretSettings"

    def __init__(self, *, secure_algo, secure_secret: bytes, secure_secret_id: int):
        self.secure_algo = secure_algo  # SecurePasswordKdfAlgo
        self.secure_secret = secure_secret  # bytes
        self.secure_secret_id = secure_secret_id  # long

    @staticmethod
    def read(b: BytesIO, *args) -> "SecureSecretSettings":
        # No flags
        
        secure_algo = TLObject.read(b)
        
        secure_secret = Bytes.read(b)
        
        secure_secret_id = Long.read(b)
        
        return SecureSecretSettings(secure_algo=secure_algo, secure_secret=secure_secret, secure_secret_id=secure_secret_id)

    def write(self) -> bytes:
        b = BytesIO()
        b.write(Int(self.ID, False))

        # No flags
        
        b.write(self.secure_algo.write())
        
        b.write(Bytes(self.secure_secret))
        
        b.write(Long(self.secure_secret_id))
        
        return b.getvalue()
