"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodejsFunction = void 0;
const fs = require("fs");
const path = require("path");
const lambda = require("@aws-cdk/aws-lambda");
const bundling_1 = require("./bundling");
const util_1 = require("./util");
/**
 * (experimental) A Node.js Lambda function bundled using esbuild.
 *
 * @experimental
 */
class NodejsFunction extends lambda.Function {
    /**
     * @experimental
     */
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e;
        if (props.runtime && props.runtime.family !== lambda.RuntimeFamily.NODEJS) {
            throw new Error('Only `NODEJS` runtimes are supported.');
        }
        // Find lock file
        let depsLockFilePath;
        if (props.depsLockFilePath) {
            if (!fs.existsSync(props.depsLockFilePath)) {
                throw new Error(`Lock file at ${props.depsLockFilePath} doesn't exist`);
            }
            depsLockFilePath = props.depsLockFilePath;
        }
        else {
            const lockFile = (_a = util_1.findUp(util_1.LockFile.YARN)) !== null && _a !== void 0 ? _a : util_1.findUp(util_1.LockFile.NPM);
            if (!lockFile) {
                throw new Error('Cannot find a package lock file (`yarn.lock` or `package-lock.json`). Please specify it with `depsFileLockPath`.');
            }
            depsLockFilePath = lockFile;
        }
        // Entry and defaults
        const entry = path.resolve(findEntry(id, props.entry));
        const handler = (_b = props.handler) !== null && _b !== void 0 ? _b : 'handler';
        const defaultRunTime = util_1.nodeMajorVersion() >= 12
            ? lambda.Runtime.NODEJS_12_X
            : lambda.Runtime.NODEJS_10_X;
        const runtime = (_c = props.runtime) !== null && _c !== void 0 ? _c : defaultRunTime;
        super(scope, id, {
            ...props,
            runtime,
            code: bundling_1.Bundling.bundle({
                ...(_d = props.bundling) !== null && _d !== void 0 ? _d : {},
                entry,
                runtime,
                depsLockFilePath,
            }),
            handler: `index.${handler}`,
        });
        // Enable connection reuse for aws-sdk
        if ((_e = props.awsSdkConnectionReuse) !== null && _e !== void 0 ? _e : true) {
            this.addEnvironment('AWS_NODEJS_CONNECTION_REUSE_ENABLED', '1', { removeInEdge: true });
        }
    }
}
exports.NodejsFunction = NodejsFunction;
/**
 * Searches for an entry file. Preference order is the following:
 * 1. Given entry file
 * 2. A .ts file named as the defining file with id as suffix (defining-file.id.ts)
 * 3. A .js file name as the defining file with id as suffix (defining-file.id.js)
 */
function findEntry(id, entry) {
    if (entry) {
        if (!/\.(jsx?|tsx?)$/.test(entry)) {
            throw new Error('Only JavaScript or TypeScript entry files are supported.');
        }
        if (!fs.existsSync(entry)) {
            throw new Error(`Cannot find entry file at ${entry}`);
        }
        return entry;
    }
    const definingFile = findDefiningFile();
    const extname = path.extname(definingFile);
    const tsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.ts`);
    if (fs.existsSync(tsHandlerFile)) {
        return tsHandlerFile;
    }
    const jsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.js`);
    if (fs.existsSync(jsHandlerFile)) {
        return jsHandlerFile;
    }
    throw new Error('Cannot find entry file.');
}
/**
 * Finds the name of the file where the `NodejsFunction` is defined
 */
function findDefiningFile() {
    let definingIndex;
    const sites = util_1.callsites();
    for (const [index, site] of sites.entries()) {
        if (site.getFunctionName() === 'NodejsFunction') {
            // The next site is the site where the NodejsFunction was created
            definingIndex = index + 1;
            break;
        }
    }
    if (!definingIndex || !sites[definingIndex]) {
        throw new Error('Cannot find defining file.');
    }
    return sites[definingIndex].getFileName();
}
//# sourceMappingURL=data:application/json;base64,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