import abc
from ...types import CustomSecretStr, OAuthProvider
from ..exceptions import RADKitAccessError
from ..oauth import OAuthConnectResponse
from .cert import EndpointCertificateBundlePaths, RADKitEndpointCertificate
from .client import RADKitAccessClient
from .refresh_loop import AuthTokenRefreshLoop
from _typeshed import Incomplete
from abc import ABC, abstractmethod
from collections.abc import AsyncGenerator
from contextlib import asynccontextmanager
from dataclasses import dataclass
from datetime import datetime
from enum import Enum
from radkit_common.identities import Email, EndpointID
from typing_extensions import Self

__all__ = ['AsyncAuthFlow', 'AsyncAccessTokenAuthFlow', 'AsyncOIDCAuthFlow', 'AsyncBasicAuthFlow', 'AsyncCertificateAuthFlow', 'AuthFlowType']

class InvalidAuthenticationResultError(RADKitAccessError):
    message: str
    status_code: int

class CertificateWasNotRenewedError(RADKitAccessError):
    message: str
    status_code: int

@dataclass(frozen=True)
class AuthenticationResult:
    success: bool
    error_details: str | None
    def __post_init__(self) -> None: ...

class AsyncAuthFlow(ABC, metaclass=abc.ABCMeta):
    @abstractmethod
    def cloud_request_resubmit_enabled(self) -> bool: ...
    @abstractmethod
    def access_client(self) -> RADKitAccessClient: ...
    @abstractmethod
    def identity(self) -> EndpointID: ...
    @abstractmethod
    def type(self) -> AuthFlowType: ...
    @abstractmethod
    def admin_level(self) -> int: ...
    @abstractmethod
    def flow_error_str(self) -> str | None: ...
    @abstractmethod
    def is_ready(self) -> bool: ...
    @abstractmethod
    async def wait_until_ready(self) -> bool: ...
    @abstractmethod
    async def start(self) -> AuthenticationResult: ...
    @abstractmethod
    async def restart(self) -> bool: ...

class AsyncAccessTokenAuthFlow(AsyncAuthFlow):
    access_token: Incomplete
    provider: Incomplete
    def __init__(self, access_client: RADKitAccessClient, admin_level: int, access_token: str, refresh_loop: AuthTokenRefreshLoop, provider: OAuthProvider | None = None, enable_cloud_request_resubmit: bool = False) -> None: ...
    def cloud_request_resubmit_enabled(self) -> bool: ...
    def access_client(self) -> RADKitAccessClient: ...
    def identity(self) -> EndpointID: ...
    def type(self) -> AuthFlowType: ...
    def admin_level(self) -> int: ...
    def flow_error_str(self) -> str | None: ...
    def issuer(self) -> str: ...
    @classmethod
    @asynccontextmanager
    async def create(cls, access_client: RADKitAccessClient, admin_level: int, access_token: str, provider: OAuthProvider | None = None, enable_cloud_request_resubmit: bool = False) -> AsyncGenerator[Self, None]: ...
    async def start(self) -> AuthenticationResult: ...
    async def restart(self) -> bool: ...
    async def reauthenticate(self, access_token: str | None = None, *, allow_user_interaction: bool = True) -> AuthenticationResult: ...
    def is_ready(self) -> bool: ...
    async def wait_until_ready(self) -> bool: ...

class AsyncOIDCAuthFlow(AsyncAuthFlow):
    open_browser: Incomplete
    def __init__(self, access_client: RADKitAccessClient, email: Email, admin_level: int, open_browser: bool, refresh_loop: AuthTokenRefreshLoop, provider: OAuthProvider | None, enable_cloud_request_resubmit: bool = False, oauth_connect_response: OAuthConnectResponse | None = None, silent: bool = False) -> None: ...
    def cloud_request_resubmit_enabled(self) -> bool: ...
    def access_client(self) -> RADKitAccessClient: ...
    def identity(self) -> EndpointID: ...
    def oauth_provider(self) -> OAuthProvider | None: ...
    def admin_level(self) -> int: ...
    def type(self) -> AuthFlowType: ...
    def flow_error_str(self) -> str | None: ...
    @classmethod
    @asynccontextmanager
    async def create(cls, access_client: RADKitAccessClient, admin_level: int, email: Email, open_browser: bool, provider: OAuthProvider | None = None, enable_cloud_request_resubmit: bool = False, oauth_connect_response: OAuthConnectResponse | None = None, silent: bool = False) -> AsyncGenerator[Self, None]: ...
    async def start(self) -> AuthenticationResult: ...
    async def restart(self) -> bool: ...
    async def reauthenticate(self, oauth_connect_response: OAuthConnectResponse | None = None, *, allow_user_interaction: bool = True) -> AuthenticationResult: ...
    def is_ready(self) -> bool: ...
    async def wait_until_ready(self) -> bool: ...

class AsyncBasicAuthFlow(AsyncAuthFlow):
    password: Incomplete
    def __init__(self, access_client: RADKitAccessClient, identity: EndpointID, admin_level: int, password: CustomSecretStr, refresh_loop: AuthTokenRefreshLoop, enable_cloud_request_resubmit: bool = False) -> None: ...
    def cloud_request_resubmit_enabled(self) -> bool: ...
    def access_client(self) -> RADKitAccessClient: ...
    def identity(self) -> EndpointID: ...
    def admin_level(self) -> int: ...
    def type(self) -> AuthFlowType: ...
    def flow_error_str(self) -> str | None: ...
    @classmethod
    @asynccontextmanager
    async def create(cls, access_client: RADKitAccessClient, admin_level: int, identity: EndpointID, password: CustomSecretStr, enable_cloud_request_resubmit: bool = False) -> AsyncGenerator[Self, None]: ...
    async def start(self) -> AuthenticationResult: ...
    async def restart(self) -> bool: ...
    async def reauthenticate(self, password: str | None = None, *, allow_user_interaction: bool = True) -> AuthenticationResult: ...
    def is_ready(self) -> bool: ...
    async def wait_until_ready(self) -> bool: ...

class AsyncCertificateAuthFlow(AsyncAuthFlow):
    certificate: Incomplete
    private_key_password: Incomplete
    def __init__(self, access_client: RADKitAccessClient, identity: EndpointID, admin_level: int, certificate: RADKitEndpointCertificate, private_key_password: CustomSecretStr, refresh_loop: AuthTokenRefreshLoop, enable_cloud_request_resubmit: bool = False) -> None: ...
    def cloud_request_resubmit_enabled(self) -> bool: ...
    def access_client(self) -> RADKitAccessClient: ...
    def identity(self) -> EndpointID: ...
    def type(self) -> AuthFlowType: ...
    def admin_level(self) -> int: ...
    def flow_error_str(self) -> str | None: ...
    def certificate_renewed(self) -> bool: ...
    def certificate_renewed_at(self) -> datetime: ...
    def new_certificate_path(self) -> EndpointCertificateBundlePaths: ...
    def clear_certificate_renewal(self) -> None: ...
    @classmethod
    @asynccontextmanager
    async def create(cls, access_client: RADKitAccessClient, admin_level: int, identity: EndpointID, certificate: RADKitEndpointCertificate, private_key_password: CustomSecretStr, ask_for_user_interaction: bool = True, enable_cloud_request_resubmit: bool = False) -> AsyncGenerator[Self, None]: ...
    async def start(self) -> AuthenticationResult: ...
    async def restart(self) -> bool: ...
    async def reauthenticate(self) -> AuthenticationResult: ...
    def is_ready(self) -> bool: ...
    async def wait_until_ready(self) -> bool: ...

class CloudClientAuthenticationError(Exception): ...

class AuthFlowType(Enum):
    ACCESS_TOKEN = 'ACCESS_TOKEN'
    OIDC = 'OIDC'
    BASIC = 'BASIC'
    CERTIFICATE = 'CERTIFICATE'
