# Copyright 2022 Mathias Lechner and Ramin Hasani
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     https://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
import keras


# LeCun improved tanh activation
# http://yann.lecun.com/exdb/publis/pdf/lecun-98b.pdf
@keras.utils.register_keras_serializable(package="", name="lecun_tanh")
def lecun_tanh(x):
    return 1.7159 * keras.activations.tanh(0.666 * x)


# Register the custom activation function
from keras.src.activations import ALL_OBJECTS_DICT
ALL_OBJECTS_DICT["lecun_tanh"] = lecun_tanh


@keras.utils.register_keras_serializable(package="ncps", name="CfCCell")
class CfCCell(keras.layers.Layer):
    def __init__(
        self,
        units,
        mode="default",
        activation="lecun_tanh",
        backbone_units=128,
        backbone_layers=1,
        backbone_dropout=0.1,
        sparsity_mask=None,
        **kwargs,
    ):
        """A `Closed-form Continuous-time <https://arxiv.org/abs/2106.13898>`_ cell.

        .. Note::
            This is an RNNCell that process single time-steps.
            To get a full RNN that can process sequences,
            see `ncps.keras.CfC` or wrap the cell with a `keras.layers.RNN <https://www.tensorflow.org/api_docs/python/tf/keras/layers/RNN>`_.


        :param units: Number of hidden units
        :param input_sparsity:
        :param recurrent_sparsity:
        :param mode: Either "default", "pure" (direct solution approximation), or "no_gate" (without second gate).
        :param activation: Activation function used in the backbone layers
        :param backbone_units: Number of hidden units in the backbone layer (default 128)
        :param backbone_layers: Number of backbone layers (default 1)
        :param backbone_dropout: Dropout rate in the backbone layers (default 0)
        :param kwargs:
        """
        super().__init__(**kwargs)
        self.units = units
        self.sparsity_mask = sparsity_mask
        if sparsity_mask is not None:
            # No backbone is allowed
            if backbone_units > 0:
                raise ValueError("If sparsity of a CfC cell is set, then no backbone is allowed")

        allowed_modes = ["default", "pure", "no_gate"]
        if mode not in allowed_modes:
            raise ValueError(f"Unknown mode '{mode}', valid options are {str(allowed_modes)}")
        self.mode = mode
        self.backbone_fn = None
        self._activation = keras.activations.get(activation)
        self._backbone_units = backbone_units
        self._backbone_layers = backbone_layers
        self._backbone_dropout = backbone_dropout
        self._cfc_layers = []

    @property
    def state_size(self):
        return self.units

    def build(self, input_shape):
        if isinstance(input_shape[0], tuple) or isinstance(input_shape[0], keras.KerasTensor):
            # Nested tuple -> First item represent feature dimension
            input_dim = input_shape[0][-1]
        else:
            input_dim = input_shape[-1]

        if self._backbone_layers > 0:
            backbone_layers = []
            for i in range(self._backbone_layers):
                backbone_layers.append(keras.layers.Dense(self._backbone_units, self._activation, name=f"backbone{i}"))
                backbone_layers.append(keras.layers.Dropout(self._backbone_dropout))

            self.backbone_fn = keras.models.Sequential(backbone_layers)
            self.backbone_fn.build((None, self.state_size + input_dim))
            cat_shape = int(self._backbone_units)
        else:
            cat_shape = int(self.state_size + input_dim)

        self.ff1_kernel = self.add_weight(
            shape=(cat_shape, self.state_size),
            initializer="glorot_uniform",
            name="ff1_weight",
        )
        self.ff1_bias = self.add_weight(
            shape=(self.state_size,),
            initializer="zeros",
            name="ff1_bias",
        )

        if self.mode == "pure":
            self.w_tau = self.add_weight(
                shape=(1, self.state_size),
                initializer=keras.initializers.Zeros(),
                name="w_tau",
            )
            self.A = self.add_weight(
                shape=(1, self.state_size),
                initializer=keras.initializers.Ones(),
                name="A",
            )
        else:
            self.ff2_kernel = self.add_weight(
                shape=(cat_shape, self.state_size),
                initializer="glorot_uniform",
                name="ff2_weight",
            )
            self.ff2_bias = self.add_weight(
                shape=(self.state_size,),
                initializer="zeros",
                name="ff2_bias",
            )

            self.time_a = keras.layers.Dense(self.state_size, name="time_a")
            self.time_b = keras.layers.Dense(self.state_size, name="time_b")
            input_shape = (None, self.state_size + input_dim)
            if self._backbone_layers > 0:
                input_shape = self.backbone_fn.output_shape
            self.time_a.build(input_shape)
            self.time_b.build(input_shape)
        self.built = True

    def call(self, inputs, states, **kwargs):
        if isinstance(inputs, (tuple, list)):
            # Irregularly sampled mode
            inputs, t = inputs
            t = keras.ops.reshape(t, [-1, 1])
        else:
            # Regularly sampled mode (elapsed time = 1 second)
            t = kwargs.get("time") or 1.0
        x = keras.layers.Concatenate()([inputs, states[0]])
        if self._backbone_layers > 0:
            x = self.backbone_fn(x)
        if self.sparsity_mask is not None:
            ff1_kernel = self.ff1_kernel * self.sparsity_mask
            ff1 = keras.ops.matmul(x, ff1_kernel) + self.ff1_bias
        else:
            ff1 = keras.ops.matmul(x, self.ff1_kernel) + self.ff1_bias
        if self.mode == "pure":
            # Solution
            new_hidden = (
                -self.A
                * keras.ops.exp(-t * (keras.ops.abs(self.w_tau) + keras.ops.abs(ff1)))
                * ff1
                + self.A
            )
        else:
            # Cfc
            if self.sparsity_mask is not None:
                ff2_kernel = self.ff2_kernel * self.sparsity_mask
                ff2 = keras.ops.matmul(x, ff2_kernel) + self.ff2_bias
            else:
                ff2 = keras.ops.matmul(x, self.ff2_kernel) + self.ff2_bias
            t_a = self.time_a(x)
            t_b = self.time_b(x)
            t_interp = keras.activations.sigmoid(-t_a * t + t_b)
            if self.mode == "no_gate":
                new_hidden = ff1 + t_interp * ff2
            else:
                new_hidden = ff1 * (1.0 - t_interp) + t_interp * ff2

        return new_hidden, [new_hidden]

    def get_config(self):
        config = {
            "units": self.units,
            "mode": self.mode,
            "activation": self._activation,
            "backbone_units": self._backbone_units,
            "backbone_layers": self._backbone_layers,
            "backbone_dropout": self._backbone_dropout,
            "sparsity_mask": self.sparsity_mask,
        }
        base_config = super().get_config()
        return {**base_config, **config}

    @classmethod
    def from_config(cls, config, custom_objects=None):
        return cls(**config)
