"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SimpleNAT = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/* eslint @typescript-eslint/no-require-imports: "off" */
const fs = require("fs");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const Mustache = require("mustache");
const fetch = require('sync-fetch');
/**
 * Simple NAT instaces construct.
 */
class SimpleNAT extends aws_cdk_lib_1.Resource {
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f;
        super(scope, id);
        this.gateways = new PrefSet();
        this._routeTablesLimit = new Map();
        this._defaultRoutesPerTable = 50;
        this._ipV6Regex = new RegExp(SimpleNAT.Ipv6Regex);
        var subnets;
        try {
            subnets = props.vpc.selectSubnets((_b = props.natSubnetsSelection) !== null && _b !== void 0 ? _b : {
                subnetType: aws_ec2_1.SubnetType.PUBLIC,
                onePerAz: true,
            });
        }
        catch (e) {
            throw new Error('NAT instances must reside in public subnets.');
        }
        if (!subnets.hasPublic) {
            throw new Error('The custom NAT subnet selection MUST select PUBLIC subnets.');
        }
        const machineImage = (_c = props.machineImage) !== null && _c !== void 0 ? _c : aws_ec2_1.MachineImage.latestAmazonLinux({
            generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            storage: aws_ec2_1.AmazonLinuxStorage.GENERAL_PURPOSE,
            cpuType: aws_ec2_1.AmazonLinuxCpuType.X86_64,
        });
        if (machineImage.getImage(this).osType != aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('The OS of custom AMI must be Linux.');
        }
        this._securityGroup = new aws_ec2_1.SecurityGroup(scope, 'NatSecurityGroup', {
            vpc: props.vpc,
            description: 'Security Group for NAT instances',
            allowAllOutbound: true,
        });
        this._securityGroup.addIngressRule(aws_ec2_1.Peer.ipv4(props.vpc.vpcCidrBlock), aws_ec2_1.Port.allTraffic());
        const role = (_d = props.role) !== null && _d !== void 0 ? _d : new aws_iam_1.Role(scope, 'NatRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
        });
        role.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        role.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'ec2:AttachNetworkInterface',
                'ec2:DetachNetworkInterface',
                'ec2:DescribeNetworkInterfaces',
            ],
            resources: ['*'],
        }));
        for (const sub of subnets.subnets) {
            const eni = new aws_ec2_1.CfnNetworkInterface(sub, 'ENI1', {
                subnetId: sub.subnetId,
                sourceDestCheck: false,
                description: 'ENI for binding EIP',
                groupSet: [this._securityGroup.securityGroupId],
            });
            const eip = new aws_ec2_1.CfnEIP(sub, 'NatInstanceEIP', {
                tags: [
                    {
                        key: 'Name',
                        value: `EIP for NAT instance in subnet '${sub.subnetId}'.`,
                    },
                ],
            });
            eip.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.RETAIN);
            new aws_ec2_1.CfnEIPAssociation(sub, 'EIPAssocation', {
                allocationId: eip.attrAllocationId,
                networkInterfaceId: eni.ref,
            });
            const configs = [
                aws_ec2_1.InitFile.fromFileInline('/opt/nat/snat.sh', path.join(__dirname, 'snat.sh'), {
                    mode: '000755',
                }),
                aws_ec2_1.InitFile.fromFileInline('/etc/systemd/system/snat.service', path.join(__dirname, 'snat.service')),
                aws_ec2_1.InitFile.fromString('/opt/nat/runonce.sh', Mustache.render(fs.readFileSync(path.join(__dirname, 'runonce.sh'), 'utf-8'), {
                    eniId: eni.ref,
                }), {
                    mode: '000755',
                }),
                aws_ec2_1.InitCommand.shellCommand('/opt/nat/runonce.sh'),
            ];
            if (props.customScripts) {
                configs.push(aws_ec2_1.InitFile.fromString('/opt/nat/custom.sh', props.customScripts, {
                    mode: '000755',
                }));
                configs.push(aws_ec2_1.InitCommand.shellCommand('/opt/nat/custom.sh'));
            }
            const natInstance = new aws_ec2_1.Instance(sub, 'NatInstance', {
                instanceType: (_e = props.instanceType) !== null && _e !== void 0 ? _e : aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.MICRO),
                machineImage,
                vpc: props.vpc,
                vpcSubnets: { subnets: [sub] },
                securityGroup: this._securityGroup,
                role,
                keyName: props.keyName,
                init: aws_ec2_1.CloudFormationInit.fromConfigSets({
                    configSets: {
                        default: ['yumPreinstall', 'config'],
                    },
                    configs: {
                        yumPreinstall: new aws_ec2_1.InitConfig([
                            // Install an Amazon Linux package using yum
                            aws_ec2_1.InitPackage.yum('jq'),
                        ]),
                        config: new aws_ec2_1.InitConfig(configs),
                    },
                }),
                initOptions: {
                    // Optional, which configsets to activate (['default'] by default)
                    configSets: ['default'],
                    timeout: aws_cdk_lib_1.Duration.minutes(5 + (props.customScripts ? 10 : 0)),
                },
            });
            // NAT instance routes all traffic, both ways
            this.gateways.add(sub.availabilityZone, {
                instance: natInstance,
                eni,
            });
        }
        this._routeMappingSubnets = props.vpc.selectSubnets((_f = props.privateSubnetsSelection) !== null && _f !== void 0 ? _f : {
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
        }).subnets.reduce((routeMapping, sub) => {
            if (routeMapping.has(sub.routeTable.routeTableId)) {
                routeMapping.get(sub.routeTable.routeTableId).push(sub);
            }
            else {
                routeMapping.set(sub.routeTable.routeTableId, [sub]);
            }
            return routeMapping;
        }, new Map());
        aws_cdk_lib_1.Tags.of(this).add('construct', 'simple-nat');
    }
    addV4Route(v4CIDR) {
        // Add routes to them in the private subnets
        for (const [routeId, subnets] of this._routeMappingSubnets) {
            this._configureSubnet(routeId, subnets, v4CIDR);
        }
        return this;
    }
    addV6Route(v6CIDR) {
        // Add routes to them in the private subnets
        for (const [routeId, subnets] of this._routeMappingSubnets) {
            this._configureSubnet(routeId, subnets, undefined, v6CIDR);
        }
        return this;
    }
    /**
     * Add Github IPs to route table
     */
    withGithubRoute(props) {
        var _b;
        const githubMeta = fetch('https://api.github.com/meta').json();
        for (const cidr of githubMeta.git) {
            for (const [routeId, subnets] of this._routeMappingSubnets) {
                if (this._ipV6Regex.test(cidr)) {
                    const excludeIPv6 = (_b = props === null || props === void 0 ? void 0 : props.excludeIPv6) !== null && _b !== void 0 ? _b : false;
                    if (!excludeIPv6) {
                        this._configureSubnet(routeId, subnets, undefined, cidr);
                    }
                }
                else {
                    this._configureSubnet(routeId, subnets, cidr);
                }
            }
        }
        return this;
    }
    /**
     * Add Google IPs to route table
     */
    withGoogleRoute(props) {
        var _b;
        const googleMeta = fetch('https://www.gstatic.com/ipranges/goog.json').json();
        const excludeIPv6 = (_b = props === null || props === void 0 ? void 0 : props.excludeIPv6) !== null && _b !== void 0 ? _b : false;
        for (const cidr of googleMeta.prefixes) {
            for (const [routeId, subnets] of this._routeMappingSubnets) {
                if (cidr.ipv4Prefix) {
                    this._configureSubnet(routeId, subnets, cidr.ipv4Prefix);
                }
                if (cidr.ipv6Prefix && !excludeIPv6) {
                    this._configureSubnet(routeId, subnets, undefined, cidr.ipv6Prefix);
                }
            }
        }
        return this;
    }
    /**
     * Add Cloudflare IPs to route table
     *
     * See https://www.cloudflare.com/ips/ for details
     */
    withCloudflareRoute(props) {
        var _b;
        const ipV4 = fetch('https://www.cloudflare.com/ips-v4').text().split(/\r?\n/);
        for (const cidr of ipV4) {
            for (const [routeId, subnets] of this._routeMappingSubnets) {
                this._configureSubnet(routeId, subnets, cidr);
            }
        }
        const excludeIPv6 = (_b = props === null || props === void 0 ? void 0 : props.excludeIPv6) !== null && _b !== void 0 ? _b : false;
        if (!excludeIPv6) {
            const ipV6 = fetch('https://www.cloudflare.com/ips-v6').text().split(/\r?\n/);
            for (const cidr of ipV6) {
                for (const [routeId, subnets] of this._routeMappingSubnets) {
                    this._configureSubnet(routeId, subnets, undefined, cidr);
                }
            }
        }
        return this;
    }
    _configureSubnet(_routeId, subnets, v4CIDR, v6CIDR) {
        const az = subnets[0].availabilityZone;
        const natInstance = this.gateways.pick(az);
        this._addRoute(`Route-${v4CIDR ? 'v4-' + (v4CIDR === null || v4CIDR === void 0 ? void 0 : v4CIDR.replace(/[\./]/gi, '-')) : 'v6-' + (v6CIDR === null || v6CIDR === void 0 ? void 0 : v6CIDR.replace(/[:/]/gi, '-'))}`, subnets[0], {
            destinationCidrBlock: v4CIDR,
            destinationIpv6CidrBlock: v6CIDR,
            routerType: aws_ec2_1.RouterType.NETWORK_INTERFACE,
            routerId: natInstance.eni.ref,
            enablesInternetConnectivity: true,
        });
        return this;
    }
    _addRoute(id, subnet, options) {
        if (options.destinationCidrBlock && options.destinationIpv6CidrBlock) {
            throw new Error('Cannot specify both \'destinationCidrBlock\' and \'destinationIpv6CidrBlock\'');
        }
        new aws_ec2_1.CfnRoute(subnet, id, {
            routeTableId: subnet.routeTable.routeTableId,
            destinationCidrBlock: options.destinationCidrBlock || (options.destinationIpv6CidrBlock === undefined ? '0.0.0.0/0' : undefined),
            destinationIpv6CidrBlock: options.destinationIpv6CidrBlock,
            [routerTypeToPropName(options.routerType)]: options.routerId,
        });
        const isIpv4 = (options.destinationCidrBlock != undefined);
        if (this._routeTablesLimit.has(subnet.routeTable.routeTableId)) {
            const stats = this._routeTablesLimit.get(subnet.routeTable.routeTableId);
            if (isIpv4) {
                stats.ipv4 += 1;
            }
            else {
                stats.ipv6 += 1;
            }
            this._routeTablesLimit.set(subnet.routeTable.routeTableId, stats);
            const totalRoutes = (isIpv4 ? stats.ipv4 : stats.ipv6);
            if (totalRoutes > this._defaultRoutesPerTable) {
                aws_cdk_lib_1.Annotations.of(this).addWarning(`The current routes in route table '${subnet.routeTable.routeTableId}' is ${totalRoutes} which exceeds the default limit ${this._defaultRoutesPerTable}. You can open ticket to increase it.`);
            }
        }
        else {
            this._routeTablesLimit.set(subnet.routeTable.routeTableId, {
                ipv4: isIpv4 ? 1 : 0,
                ipv6: isIpv4 ? 0 : 1,
            });
        }
    }
}
exports.SimpleNAT = SimpleNAT;
_a = JSII_RTTI_SYMBOL_1;
SimpleNAT[_a] = { fqn: "cdk-construct-simple-nat.SimpleNAT", version: "0.2.110" };
SimpleNAT.Ipv6Regex = '^s*((([0-9A-Fa-f]{1,4}:){7}([0-9A-Fa-f]{1,4}|:))|(([0-9A-Fa-f]{1,4}:){6}(:[0-9A-Fa-f]{1,4}|((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3})|:))|(([0-9A-Fa-f]{1,4}:){5}(((:[0-9A-Fa-f]{1,4}){1,2})|:((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3})|:))|(([0-9A-Fa-f]{1,4}:){4}(((:[0-9A-Fa-f]{1,4}){1,3})|((:[0-9A-Fa-f]{1,4})?:((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3}))|:))|(([0-9A-Fa-f]{1,4}:){3}(((:[0-9A-Fa-f]{1,4}){1,4})|((:[0-9A-Fa-f]{1,4}){0,2}:((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3}))|:))|(([0-9A-Fa-f]{1,4}:){2}(((:[0-9A-Fa-f]{1,4}){1,5})|((:[0-9A-Fa-f]{1,4}){0,3}:((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3}))|:))|(([0-9A-Fa-f]{1,4}:){1}(((:[0-9A-Fa-f]{1,4}){1,6})|((:[0-9A-Fa-f]{1,4}){0,4}:((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3}))|:))|(:(((:[0-9A-Fa-f]{1,4}){1,7})|((:[0-9A-Fa-f]{1,4}){0,5}:((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3}))|:)))(%.+)?s*(\/([0-9]|[1-9][0-9]|1[0-1][0-9]|12[0-8]))?$';
function routerTypeToPropName(routerType) {
    return ({
        [aws_ec2_1.RouterType.EGRESS_ONLY_INTERNET_GATEWAY]: 'egressOnlyInternetGatewayId',
        [aws_ec2_1.RouterType.GATEWAY]: 'gatewayId',
        [aws_ec2_1.RouterType.INSTANCE]: 'instanceId',
        [aws_ec2_1.RouterType.NAT_GATEWAY]: 'natGatewayId',
        [aws_ec2_1.RouterType.NETWORK_INTERFACE]: 'networkInterfaceId',
        [aws_ec2_1.RouterType.VPC_PEERING_CONNECTION]: 'vpcPeeringConnectionId',
    })[routerType];
}
/**
 * Preferential set
 *
 * Picks the value with the given key if available, otherwise distributes
 * evenly among the available options.
 */
class PrefSet {
    constructor() {
        this.map = {};
        this.vals = new Array();
        this.next = 0;
    }
    add(pref, value) {
        this.map[pref] = value;
        this.vals.push([pref, value]);
    }
    pick(pref) {
        if (this.vals.length === 0) {
            throw new Error('Cannot pick, set is empty');
        }
        if (pref in this.map) {
            return this.map[pref];
        }
        return this.vals[this.next++ % this.vals.length][1];
    }
    values() {
        return this.vals;
    }
}
//# sourceMappingURL=data:application/json;base64,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