"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.IdentitySource = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_verifiedpermissions_1 = require("aws-cdk-lib/aws-verifiedpermissions");
const core_1 = require("aws-cdk-lib/core");
var ConfigurationMode;
(function (ConfigurationMode) {
    ConfigurationMode["COGNITO"] = "COGNITO";
    ConfigurationMode["OIDC_ACCESS_TOKEN"] = "OIDC_ACCESS_TOKEN";
    ConfigurationMode["OIDC_ID_TOKEN"] = "OIDC_ID_TOKEN";
})(ConfigurationMode || (ConfigurationMode = {}));
class IdentitySourceBase extends core_1.Resource {
}
class IdentitySource extends IdentitySourceBase {
    /**
     * Creates Identity Source from its attributes
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs An `IdentitySourceAttributes` object.
     */
    static fromIdentitySourceAttributes(scope, id, attrs) {
        class Import extends IdentitySourceBase {
            constructor(identitySourceId) {
                super(scope, id);
                this.identitySourceId = identitySourceId;
            }
        }
        let identitySourceId;
        identitySourceId = attrs.identitySourceId;
        return new Import(identitySourceId);
    }
    /**
     * Create an Identity Source from its identifier
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param identitySourceId The Identity Source identifier.
     */
    static fromIdentitySourceId(scope, id, identitySourceId) {
        return IdentitySource.fromIdentitySourceAttributes(scope, id, {
            identitySourceId,
        });
    }
    constructor(scope, id, props) {
        super(scope, id);
        if (props.configuration.cognitoUserPoolConfiguration && props.configuration.openIdConnectConfiguration) {
            throw new Error('Only one between cognitoUserPoolConfiguration or openIdConnectConfiguration must be defined');
        }
        let cfnConfiguration;
        let issuer;
        if (props.configuration.cognitoUserPoolConfiguration) {
            this.clientIds = props.configuration.cognitoUserPoolConfiguration.clientIds ?? [];
            this.audiencesOIDC = [];
            const cognitoGroupConfiguration = props.configuration.cognitoUserPoolConfiguration.groupConfiguration?.groupEntityType
                ? {
                    groupEntityType: props.configuration.cognitoUserPoolConfiguration.groupConfiguration.groupEntityType,
                }
                : undefined;
            cfnConfiguration = {
                cognitoUserPoolConfiguration: {
                    clientIds: core_1.Lazy.list({ produce: () => this.clientIds }),
                    userPoolArn: props.configuration.cognitoUserPoolConfiguration.userPool.userPoolArn,
                    groupConfiguration: cognitoGroupConfiguration,
                },
            };
            this.cognitoGroupEntityType = cognitoGroupConfiguration?.groupEntityType;
            issuer = 'COGNITO';
            this.configurationMode = ConfigurationMode.COGNITO;
        }
        else if (props.configuration.openIdConnectConfiguration) {
            if (props.configuration.openIdConnectConfiguration.accessTokenOnly &&
                props.configuration.openIdConnectConfiguration.identityTokenOnly) {
                throw new Error('Exactly one token selection method between accessTokenOnly and identityTokenOnly must be defined');
            }
            let tokenSelection;
            if (props.configuration.openIdConnectConfiguration.accessTokenOnly) {
                if (!props.configuration.openIdConnectConfiguration.accessTokenOnly.audiences ||
                    props.configuration.openIdConnectConfiguration.accessTokenOnly.audiences.length == 0) {
                    throw new Error('At least one audience is expected in OIDC Access token selection mode');
                }
                this.clientIds = [];
                this.audiencesOIDC = props.configuration.openIdConnectConfiguration.accessTokenOnly.audiences;
                tokenSelection = {
                    accessTokenOnly: {
                        audiences: core_1.Lazy.list({ produce: () => this.audiencesOIDC }),
                        principalIdClaim: props.configuration.openIdConnectConfiguration.accessTokenOnly.principalIdClaim,
                    },
                };
                this.principalIdClaimOIDC = props.configuration.openIdConnectConfiguration.accessTokenOnly.principalIdClaim;
                this.configurationMode = ConfigurationMode.OIDC_ACCESS_TOKEN;
            }
            else if (props.configuration.openIdConnectConfiguration.identityTokenOnly) {
                this.clientIds = props.configuration.openIdConnectConfiguration.identityTokenOnly.clientIds ?? [];
                this.audiencesOIDC = [];
                tokenSelection = {
                    identityTokenOnly: {
                        clientIds: core_1.Lazy.list({ produce: () => this.clientIds }),
                        principalIdClaim: props.configuration.openIdConnectConfiguration.identityTokenOnly.principalIdClaim,
                    },
                };
                this.principalIdClaimOIDC = props.configuration.openIdConnectConfiguration.identityTokenOnly.principalIdClaim;
                this.configurationMode = ConfigurationMode.OIDC_ID_TOKEN;
            }
            else {
                throw new Error('One token selection method between accessTokenOnly and identityTokenOnly must be defined');
            }
            cfnConfiguration = {
                openIdConnectConfiguration: {
                    issuer: props.configuration.openIdConnectConfiguration.issuer,
                    entityIdPrefix: props.configuration.openIdConnectConfiguration.entityIdPrefix,
                    groupConfiguration: props.configuration.openIdConnectConfiguration.groupConfiguration ? {
                        groupClaim: props.configuration.openIdConnectConfiguration.groupConfiguration.groupClaim,
                        groupEntityType: props.configuration.openIdConnectConfiguration.groupConfiguration.groupEntityType,
                    } : undefined,
                    tokenSelection: tokenSelection,
                },
            };
            this.groupConfigGroupClaimOIDC = props.configuration.openIdConnectConfiguration.groupConfiguration?.groupClaim;
            this.groupConfigGroupEntityTypeOIDC = props.configuration.openIdConnectConfiguration.groupConfiguration?.groupEntityType;
            issuer = props.configuration.openIdConnectConfiguration.issuer;
        }
        else {
            throw new Error('One Identity provider configuration between cognitoUserPoolConfiguration and openIdConnectConfiguration must be defined');
        }
        this.identitySource = new aws_verifiedpermissions_1.CfnIdentitySource(this, id, {
            configuration: cfnConfiguration,
            policyStoreId: props.policyStore.policyStoreId,
            principalEntityType: props.principalEntityType,
        });
        this.userPoolArn = props.configuration.cognitoUserPoolConfiguration?.userPool.userPoolArn || undefined;
        this.identitySourceId = this.identitySource.attrIdentitySourceId;
        this.issuer = issuer;
        this.policyStore = props.policyStore;
    }
    /**
     * Add a User Pool Client
     * The method can be called only when the Identity Source is configured with Cognito auth provider
     *
     * @param userPoolClient The User Pool Client Construct.
     */
    addUserPoolClient(userPoolClient) {
        if (this.configurationMode != ConfigurationMode.COGNITO) {
            throw new Error('Cannot add User Pool Client when IdentitySource auth provider is not Cognito');
        }
        this.addClientId(userPoolClient.userPoolClientId);
    }
    /**
     * Add a clientId to the list
     * The method can be called only when the Identity Source is configured with one of these configs:
     *  - Cognito auth provider
     *  - OIDC auth provider and ID Token Selection mode
     *
     * @param clientId The clientId to be added
     */
    addClientId(clientId) {
        if (this.configurationMode != ConfigurationMode.COGNITO && this.configurationMode != ConfigurationMode.OIDC_ID_TOKEN) {
            throw new Error('Adding a Client ID is only supported for the auth providers Cognito or OIDC with configured with ID Token');
        }
        this.clientIds.push(clientId);
    }
    /**
     * Add an audience to the list.
     * The method can be called only when the Identity Source is configured with OIDC auth provider and Access Token Selection mode
     *
     * @param audience the audience to be added
     */
    addAudience(audience) {
        if (this.configurationMode != ConfigurationMode.OIDC_ACCESS_TOKEN) {
            throw new Error('Cannot add audience when IdentitySource auth provider is not OIDC with Access Token');
        }
        this.audiencesOIDC.push(audience);
    }
}
exports.IdentitySource = IdentitySource;
_a = JSII_RTTI_SYMBOL_1;
IdentitySource[_a] = { fqn: "@cdklabs/cdk-verified-permissions.IdentitySource", version: "0.0.6" };
//# sourceMappingURL=data:application/json;base64,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