"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyType = exports.Policy = exports.POLICY_DESC_SUFFIX_FROM_FILE = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const aws_verifiedpermissions_1 = require("aws-cdk-lib/aws-verifiedpermissions");
const core_1 = require("aws-cdk-lib/core");
const cedar_helpers_1 = require("./cedar-helpers");
exports.POLICY_DESC_SUFFIX_FROM_FILE = ' - Created by CDK';
class PolicyBase extends core_1.Resource {
}
class Policy extends PolicyBase {
    /**
     *  Import a policy into the CDK using its id.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct id.
     * @param policyId The policy id.
     */
    static fromPolicyId(scope, id, policyId) {
        return Policy.fromPolicyAttributes(scope, id, { policyId });
    }
    /**
     * Import a Policy construct from attributes.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct id.
     * @param attrs A `PolicyAttributes` object.
     * @returns
     */
    static fromPolicyAttributes(scope, id, attrs) {
        class Import extends PolicyBase {
            constructor(policyType, policyId) {
                super(scope, id);
                this.policyType = policyType;
                this.policyId = policyId;
            }
        }
        const policyType = attrs.policyType ?? PolicyType.STATIC;
        return new Import(policyType, attrs.policyId);
    }
    /**
     * Create a policy based on a file containing a cedar policy. Best practice would be
     * for the file name to end in `.cedar` but this is not required. Policy is parsed for valid
     * syntax but not validated against schema. In order to validate against schema, use
     * `PolicyStore.addPoliciesFromPath()`
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct id.
     * @param props A `StaticPolicyFromFileProps` object.
     */
    static fromFile(scope, id, props) {
        const policyFileContents = fs.readFileSync(props.path).toString();
        (0, cedar_helpers_1.checkParsePolicy)(policyFileContents);
        let relativePath = path.basename(props.path);
        return new Policy(scope, id, {
            definition: {
                static: {
                    statement: policyFileContents,
                    description: props.description || `${relativePath}${exports.POLICY_DESC_SUFFIX_FROM_FILE}`,
                },
            },
            policyStore: props.policyStore,
        });
    }
    constructor(scope, id, props) {
        super(scope, id);
        // validations
        if (props.definition.static && props.definition.templateLinked) {
            throw new Error('Policy can either be static or templateLinked');
        }
        let definition;
        if (props.definition.static) {
            (0, cedar_helpers_1.checkParsePolicy)(props.definition.static.statement);
            definition = {
                static: {
                    ...props.definition.static,
                    statement: props.definition.static.statement,
                },
            };
        }
        else if (props.definition.templateLinked) {
            definition = {
                templateLinked: {
                    policyTemplateId: props.definition.templateLinked.policyTemplate.policyTemplateId,
                    principal: props.definition.templateLinked.principal,
                    resource: props.definition.templateLinked.resource,
                },
            };
        }
        else {
            throw new Error('Policy must either be static or templateLinked');
        }
        // resource
        this.policy = new aws_verifiedpermissions_1.CfnPolicy(this, id, {
            definition: definition,
            policyStoreId: props.policyStore.policyStoreId,
        });
        // assign construct props
        this.policyId = this.policy.attrPolicyId;
        this.policyType = props.definition.static
            ? PolicyType.STATIC
            : PolicyType.TEMPLATELINKED;
        this.definition = props.definition;
        this.policyStoreId = props.policyStore.policyStoreId;
    }
}
exports.Policy = Policy;
_a = JSII_RTTI_SYMBOL_1;
Policy[_a] = { fqn: "@cdklabs/cdk-verified-permissions.Policy", version: "0.0.6" };
/**
 * PolicyType options
 */
var PolicyType;
(function (PolicyType) {
    PolicyType["STATIC"] = "Static";
    PolicyType["TEMPLATELINKED"] = "TemplateLinked";
})(PolicyType || (exports.PolicyType = PolicyType = {}));
//# sourceMappingURL=data:application/json;base64,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