import torch
import torch.nn as nn
import torch.nn.functional as F
from torch import optim
from common.text_utils import MAX_LENGTH


device = torch.device('cuda' if torch.cuda.is_available() else 'cpu')


class EncoderRNN(nn.Module):

    def __init__(self, input_size, hidden_size):
        """
        input_size - vocabulary size
        hidden_size - embedding dim
        """
        super(EncoderRNN, self).__init__()
        self.hidden_size = hidden_size

        self.embedding = nn.Embedding(input_size, hidden_size)
        self.gru = nn.GRU(hidden_size, hidden_size)
    

    def forward(self, input, hidden):
        """
        input - shape of (1, hidden_size)
        """
        # self.embedding(input) returns a 2 dim matrix(tensor)
        # needs to be transformed to 3 dim to match the requirement of PyTorch
        embedded = self.embedding(input).view(1, 1, -1)
        output = embedded
        output, hidden = self.gru(output, hidden)
        
        return output, hidden
    

    def initHidden(self):
        return torch.zeros(1, 1, self.hidden_size, device=device)


class AttnDecoderRNN(nn.Module):

    def __init__(self, hidden_size, output_size, dropout_p=0.1, max_length=MAX_LENGTH):
        super(AttnDecoderRNN, self).__init__()
        self.hidden_size = hidden_size
        self.output_size = output_size
        self.dropout_p = dropout_p
        self.max_length = max_length

        self.embedding = nn.Embedding(output_size, hidden_size)
        self.dropout = nn.Dropout(dropout_p)
        self.attn = nn.Linear(hidden_size * 2, max_length)
        self.attn_combine = nn.Linear(hidden_size * 2, hidden_size)
        self.gru = nn.GRU(hidden_size, hidden_size)
        self.out = nn.Linear(hidden_size, output_size)
    

    def forward(self, input, hidden, encoder_outputs):
        """
        encoder_outputs - shape of (MAX_LENGTH, hidden_size)
        
        output - shape of (1, output_size)
        """
        embeded = self.embedding(input).view(1, 1, -1)
        embeded = self.dropout(embeded)

        attn_weights = F.softmax(
            self.attn(torch.cat((embeded[0], hidden[0]), 1)),
            dim=1
        )
        attn_applied = torch.bmm(attn_weights.unsqueeze(0), encoder_outputs.unsqueeze(0))
        
        output = torch.cat((embeded[0], attn_applied[0]), 1)
        output = self.attn_combine(output).unsqueeze(0)
        
        output = F.relu(output)
        output, hidden = self.gru(output, hidden)
        
        output = self.out(output[0])
        output = F.log_softmax(output, dim=1)
        
        return output, hidden, attn_weights
    

    def initHidden(self):
        return torch.zeros(1, 1, self.hidden_size, device=device)
