r'''
# Amazon Bedrock Construct Library

<!--BEGIN STABILITY BANNER-->---


![cdk-constructs: Experimental](https://img.shields.io/badge/cdk--constructs-experimental-important.svg?style=for-the-badge)

> The APIs of higher level constructs in this module are experimental and under active development.
> They are subject to non-backward compatible changes or removal in any future version. These are
> not subject to the [Semantic Versioning](https://semver.org/) model and breaking changes will be
> announced in the release notes. This means that while you may use them, you may need to update
> your source code when upgrading to a newer version of this package.

---
<!--END STABILITY BANNER-->

| **Language**                                                                                   | **Package**                             |
| :--------------------------------------------------------------------------------------------- | --------------------------------------- |
| ![Typescript Logo](https://docs.aws.amazon.com/cdk/api/latest/img/typescript32.png) TypeScript | `@aws-cdk/aws-bedrock-alpha` |

[Amazon Bedrock](https://aws.amazon.com/bedrock/) is a fully managed service that offers a choice of high-performing foundation models (FMs) from leading AI companies and Amazon through a single API, along with a broad set of capabilities you need to build generative AI applications with security, privacy, and responsible AI.

This construct library facilitates the deployment of Bedrock Agents, enabling you to create sophisticated AI applications that can interact with your systems and data sources.

## Table of contents

* [Agents](#agents)

  * [Create an Agent](#create-an-agent)
  * [Action groups](#action-groups)
  * [Prepare the Agent](#prepare-the-agent)
  * [Prompt Override Configuration](#prompt-override-configuration)
  * [Memory Configuration](#memory-configuration)
  * [Agent Collaboration](#agent-collaboration)
  * [Custom Orchestration](#custom-orchestration)
  * [Agent Alias](#agent-alias)

## Agents

Amazon Bedrock Agents allow generative AI applications to automate complex, multistep tasks by seamlessly integrating with your company's systems, APIs, and data sources. It uses the reasoning of foundation models (FMs), APIs, and data to break down user requests, gather relevant information, and efficiently complete tasks.

### Create an Agent

Building an agent is straightforward and fast.
The following example creates an Agent with a simple instruction and default prompts:

```python
agent = bedrock.Agent(self, "Agent",
    foundation_model=bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_HAIKU_V1_0,
    instruction="You are a helpful and friendly agent that answers questions about literature."
)
```

### Agent Properties

The Bedrock Agent class supports the following properties.

| Name | Type | Required | Description |
|---|---|---|---|
| name | string | No | The name of the agent. Defaults to a name generated by CDK |
| instruction | string | Yes | The instruction used by the agent that determines how it will perform its task. Must have a minimum of 40 characters |
| foundationModel | IBedrockInvokable | Yes | The foundation model used for orchestration by the agent |
| existingRole | iam.IRole | No | The existing IAM Role for the agent to use. Must have a trust policy allowing Bedrock service to assume the role. Defaults to a new created role |
| shouldPrepareAgent | boolean | No | Specifies whether to automatically update the `DRAFT` version of the agent after making changes. Defaults to false |
| idleSessionTTL | Duration | No | How long sessions should be kept open for the agent. Session expires if no conversation occurs during this time. Defaults to 1 hour |
| kmsKey | kms.IKey | No | The KMS key of the agent if custom encryption is configured. Defaults to AWS managed key |
| description | string | No | A description of the agent. Defaults to no description |
| actionGroups | AgentActionGroup[] | No | The Action Groups associated with the agent |
| promptOverrideConfiguration | PromptOverrideConfiguration | No | Overrides some prompt templates in different parts of an agent sequence configuration |
| userInputEnabled | boolean | No | Select whether the agent can prompt additional information from the user when it lacks enough information. Defaults to false |
| codeInterpreterEnabled | boolean | No | Select whether the agent can generate, run, and troubleshoot code when trying to complete a task. Defaults to false |
| forceDelete | boolean | No | Whether to delete the resource even if it's in use. Defaults to true |
| agentCollaboration | AgentCollaboration | No | Configuration for agent collaboration settings, including type and collaborators. This property allows you to define how the agent collaborates with other agents and what collaborators it can work with. Defaults to no agent collaboration configuration |
| customOrchestrationExecutor | CustomOrchestrationExecutor | No | The Lambda function to use for custom orchestration. If provided, orchestrationType is set to CUSTOM_ORCHESTRATION. If not provided, orchestrationType defaults to DEFAULT. Defaults to default orchestration |

### Action Groups

An action group defines functions your agent can call. The functions are Lambda functions. The action group uses an OpenAPI schema to tell the agent what your functions do and how to call them.

#### Action Group Properties

The AgentActionGroup class supports the following properties.

| Name | Type | Required | Description |
|---|---|---|---|
| name | string | No | The name of the action group. Defaults to a name generated in the format 'action_group_quick_start_UUID' |
| description | string | No | A description of the action group |
| apiSchema | ApiSchema | No | The OpenAPI schema that defines the functions in the action group |
| executor | ActionGroupExecutor | No | The Lambda function that executes the actions in the group |
| enabled | boolean | No | Whether the action group is enabled. Defaults to true |
| forceDelete | boolean | No | Whether to delete the resource even if it's in use. Defaults to false |
| functionSchema | FunctionSchema | No | Defines functions that each define parameters that the agent needs to invoke from the user |
| parentActionGroupSignature | ParentActionGroupSignature | No | The AWS Defined signature for enabling certain capabilities in your agent |

There are three ways to provide an API schema for your action group:

From a local asset file (requires binding to scope):

```python
action_group_function = lambda_.Function(self, "ActionGroupFunction",
    runtime=lambda_.Runtime.PYTHON_3_12,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "../lambda/action-group"))
)

# When using ApiSchema.fromLocalAsset, you must bind the schema to a scope
schema = bedrock.ApiSchema.from_local_asset(path.join(__dirname, "action-group.yaml"))
schema.bind(self)

action_group = bedrock.AgentActionGroup(
    name="query-library",
    description="Use these functions to get information about the books in the library.",
    executor=bedrock.ActionGroupExecutor.from_lambda(action_group_function),
    enabled=True,
    api_schema=schema
)

agent = bedrock.Agent(self, "Agent",
    foundation_model=bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_HAIKU_V1_0,
    instruction="You are a helpful and friendly agent that answers questions about literature."
)

agent.add_action_group(action_group)
```

From an inline OpenAPI schema:

```python
inline_schema = bedrock.ApiSchema.from_inline("""
    openapi: 3.0.3
    info:
      title: Library API
      version: 1.0.0
    paths:
      /search:
        get:
          summary: Search for books
          operationId: searchBooks
          parameters:
            - name: query
              in: query
              required: true
              schema:
                type: string
    """)

action_group_function = lambda_.Function(self, "ActionGroupFunction",
    runtime=lambda_.Runtime.PYTHON_3_12,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "../lambda/action-group"))
)

action_group = bedrock.AgentActionGroup(
    name="query-library",
    description="Use these functions to get information about the books in the library.",
    executor=bedrock.ActionGroupExecutor.from_lambda(action_group_function),
    enabled=True,
    api_schema=inline_schema
)

agent = bedrock.Agent(self, "Agent",
    foundation_model=bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_HAIKU_V1_0,
    instruction="You are a helpful and friendly agent that answers questions about literature."
)

agent.add_action_group(action_group)
```

From an existing S3 file:

```python
bucket = s3.Bucket.from_bucket_name(self, "ExistingBucket", "my-schema-bucket")
s3_schema = bedrock.ApiSchema.from_s3_file(bucket, "schemas/action-group.yaml")

action_group_function = lambda_.Function(self, "ActionGroupFunction",
    runtime=lambda_.Runtime.PYTHON_3_12,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "../lambda/action-group"))
)

action_group = bedrock.AgentActionGroup(
    name="query-library",
    description="Use these functions to get information about the books in the library.",
    executor=bedrock.ActionGroupExecutor.from_lambda(action_group_function),
    enabled=True,
    api_schema=s3_schema
)

agent = bedrock.Agent(self, "Agent",
    foundation_model=bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_HAIKU_V1_0,
    instruction="You are a helpful and friendly agent that answers questions about literature."
)

agent.add_action_group(action_group)
```

### Using FunctionSchema with Action Groups

As an alternative to using OpenAPI schemas, you can define functions directly using the `FunctionSchema` class. This approach provides a more structured way to define the functions that your agent can call.

```python
action_group_function = lambda_.Function(self, "ActionGroupFunction",
    runtime=lambda_.Runtime.PYTHON_3_12,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "../lambda/action-group"))
)

# Define a function schema with parameters
function_schema = bedrock.FunctionSchema(
    functions=[bedrock.FunctionProps(
        name="searchBooks",
        description="Search for books in the library catalog",
        parameters={
            "query": bedrock.FunctionParameterProps(
                type=bedrock.ParameterType.STRING,
                required=True,
                description="The search query string"
            ),
            "maxResults": bedrock.FunctionParameterProps(
                type=bedrock.ParameterType.INTEGER,
                required=False,
                description="Maximum number of results to return"
            ),
            "includeOutOfPrint": bedrock.FunctionParameterProps(
                type=bedrock.ParameterType.BOOLEAN,
                required=False,
                description="Whether to include out-of-print books"
            )
        },
        require_confirmation=bedrock.RequireConfirmation.DISABLED
    ), bedrock.FunctionProps(
        name="getBookDetails",
        description="Get detailed information about a specific book",
        parameters={
            "bookId": bedrock.FunctionParameterProps(
                type=bedrock.ParameterType.STRING,
                required=True,
                description="The unique identifier of the book"
            )
        },
        require_confirmation=bedrock.RequireConfirmation.ENABLED
    )
    ]
)

# Create an action group using the function schema
action_group = bedrock.AgentActionGroup(
    name="library-functions",
    description="Functions for interacting with the library catalog",
    executor=bedrock.ActionGroupExecutor.from_lambda(action_group_function),
    function_schema=function_schema,
    enabled=True
)

agent = bedrock.Agent(self, "Agent",
    foundation_model=bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_HAIKU_V1_0,
    instruction="You are a helpful and friendly agent that answers questions about literature.",
    action_groups=[action_group]
)
```

The `FunctionSchema` approach offers several advantages:

* Type-safe definition of functions and parameters
* Built-in validation of parameter names, descriptions, and other properties
* Clear structure that maps directly to the AWS Bedrock API
* Support for parameter types including string, number, integer, boolean, array, and object
* Option to require user confirmation before executing specific functions

If you chose to load your schema file from S3, the construct will provide the necessary permissions to your agent's execution role to access the schema file from the specific bucket. Similar to performing the operation through the console, the agent execution role will get a permission like:

```json
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Sid": "AmazonBedrockAgentS3PolicyProd",
            "Effect": "Allow",
            "Action": [
                "s3:GetObject"
            ],
            "Resource": [
                "arn:aws:s3:::<BUCKET_NAME>/<OBJECT_KEY>"
            ],
            "Condition": {
                "StringEquals": {
                    "aws:ResourceAccount": "ACCOUNT_NUMBER"
                }
            }
        }
    ]
}
```

```python
# create a bucket containing the input schema
schema_bucket = s3.Bucket(self, "SchemaBucket",
    enforce_sSL=True,
    versioned=True,
    public_read_access=False,
    block_public_access=s3.BlockPublicAccess.BLOCK_ALL,
    encryption=s3.BucketEncryption.S3_MANAGED,
    removal_policy=RemovalPolicy.DESTROY,
    auto_delete_objects=True
)

# deploy the local schema file to S3
deployement = aws_s3_deployment.BucketDeployment(self, "DeployWebsite",
    sources=[aws_s3_deployment.Source.asset(path.join(__dirname, "../inputschema"))],
    destination_bucket=schema_bucket,
    destination_key_prefix="inputschema"
)

# create the agent
agent = bedrock.Agent(self, "Agent",
    foundation_model=bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_3_5_SONNET_V1_0,
    instruction="You are a helpful and friendly agent that answers questions about literature.",
    user_input_enabled=True,
    should_prepare_agent=True
)

# create a lambda function
action_group_function = lambda_.Function(self, "ActionGroupFunction",
    runtime=lambda_.Runtime.PYTHON_3_12,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "../lambda/action-group"))
)

# create an action group and read the schema file from S3
action_group = bedrock.AgentActionGroup(
    name="query-library",
    description="Use these functions to get information about the books in the library.",
    executor=bedrock.ActionGroupExecutor.from_lambda(action_group_function),
    enabled=True,
    api_schema=bedrock.ApiSchema.from_s3_file(schema_bucket, "inputschema/action-group.yaml")
)

# add the action group to the agent
agent.add_action_group(action_group)

# add dependency for the agent on the s3 deployment
agent.node.add_dependency(deployement)
```

### Prepare the Agent

The `Agent` constructs take an optional parameter `shouldPrepareAgent` to indicate that the Agent should be prepared after any updates to an agent or action group. This may increase the time to create and update those resources. By default, this value is false.

#### Prepare Agent Properties

| Name | Type | Required | Description |
|---|---|---|---|
| shouldPrepareAgent | boolean | No | Whether to automatically update the DRAFT version of the agent after making changes. Defaults to false |

Creating an agent alias will not prepare the agent, so if you create an alias using the `AgentAlias` resource then you should set `shouldPrepareAgent` to ***true***.

### Prompt Override Configuration

Bedrock Agents allows you to customize the prompts and LLM configuration for different steps in the agent sequence. The implementation provides type-safe configurations for each step type, ensuring correct usage at compile time.

#### Prompt Override Configuration Properties

| Name | Type | Required | Description |
|---|---|---|---|
| steps | PromptStepConfiguration[] | Yes | Array of step configurations for different parts of the agent sequence |
| parser | lambda.IFunction | No | Lambda function for custom parsing of agent responses |

#### Prompt Step Configuration Properties

Each step in the `steps` array supports the following properties:

| Name | Type | Required | Description |
|---|---|---|---|
| stepType | AgentStepType | Yes | The type of step being configured (PRE_PROCESSING, ORCHESTRATION, POST_PROCESSING, ROUTING_CLASSIFIER, MEMORY_SUMMARIZATION, KNOWLEDGE_BASE_RESPONSE_GENERATION) |
| stepEnabled | boolean | No | Whether this step is enabled. Defaults to true |
| customPromptTemplate | string | No | Custom prompt template to use for this step |
| inferenceConfig | InferenceConfiguration | No | Configuration for model inference parameters |
| foundationModel | BedrockFoundationModel | No | Alternative foundation model to use for this step (only valid for ROUTING_CLASSIFIER step) |
| useCustomParser | boolean | No | Whether to use a custom parser for this step. Requires parser to be provided in PromptOverrideConfiguration |

#### Inference Configuration Properties

When providing `inferenceConfig`, the following properties are supported:

| Name | Type | Required | Description |
|---|---|---|---|
| temperature | number | No | Controls randomness in the model's output (0.0-1.0) |
| topP | number | No | Controls diversity via nucleus sampling (0.0-1.0) |
| topK | number | No | Controls diversity by limiting the cumulative probability |
| maximumLength | number | No | Maximum length of generated text |
| stopSequences | string[] | No | Sequences where the model should stop generating |

The following steps can be configured:

* PRE_PROCESSING: Prepares the user input for orchestration
* ORCHESTRATION: Main step that determines the agent's actions
* POST_PROCESSING: Refines the agent's response
* ROUTING_CLASSIFIER: Classifies and routes requests to appropriate collaborators
* MEMORY_SUMMARIZATION: Summarizes conversation history for memory retention
* KNOWLEDGE_BASE_RESPONSE_GENERATION: Generates responses using knowledge base content

Example with pre-processing configuration:

```python
agent = bedrock.Agent(self, "Agent",
    foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
    instruction="You are a helpful assistant.",
    prompt_override_configuration=bedrock.PromptOverrideConfiguration.from_steps([
        step_type=bedrock.AgentStepType.PRE_PROCESSING,
        step_enabled=True,
        custom_prompt_template="Your custom prompt template here",
        inference_config=bedrock.InferenceConfiguration(
            temperature=0,
            top_p=1,
            top_k=250,
            maximum_length=1,
            stop_sequences=["\n\nHuman:"]
        )

    ])
)
```

Example with routing classifier and foundation model:

```python
agent = bedrock.Agent(self, "Agent",
    foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
    instruction="You are a helpful assistant.",
    prompt_override_configuration=bedrock.PromptOverrideConfiguration.from_steps([
        step_type=bedrock.AgentStepType.ROUTING_CLASSIFIER,
        step_enabled=True,
        custom_prompt_template="Your routing template here",
        foundation_model=bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_V2

    ])
)
```

Using a custom Lambda parser:

```python
parser_function = lambda_.Function(self, "ParserFunction",
    runtime=lambda_.Runtime.PYTHON_3_10,
    handler="index.handler",
    code=lambda_.Code.from_asset("lambda")
)

agent = bedrock.Agent(self, "Agent",
    foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
    instruction="You are a helpful assistant.",
    prompt_override_configuration=bedrock.PromptOverrideConfiguration.with_custom_parser(
        parser=parser_function,
        pre_processing_step=bedrock.PromptPreProcessingConfigCustomParser(
            step_type=bedrock.AgentStepType.PRE_PROCESSING,
            use_custom_parser=True
        )
    )
)
```

Foundation models can only be specified for the ROUTING_CLASSIFIER step.

### Memory Configuration

Agents can maintain context across multiple sessions and recall past interactions using memory. This feature is useful for creating a more coherent conversational experience.

#### Memory Configuration Properties

| Name | Type | Required | Description |
|---|---|---|---|
| maxRecentSessions | number | No | Maximum number of recent session summaries to retain |
| memoryDuration | Duration | No | How long to retain session summaries |

Example:

```python
agent = bedrock.Agent(self, "MyAgent",
    agent_name="MyAgent",
    instruction="Your instruction here",
    foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
    memory=Memory.session_summary(
        max_recent_sessions=10,  # Keep the last 10 session summaries
        memory_duration=Duration.days(20)
    )
)
```

### Agent Collaboration

Agent Collaboration enables multiple Bedrock Agents to work together on complex tasks. This feature allows agents to specialize in different areas and collaborate to provide more comprehensive responses to user queries.

#### Agent Collaboration Properties

| Name | Type | Required | Description |
|---|---|---|---|
| type | AgentCollaboratorType | Yes | Type of collaboration (SUPERVISOR or PEER) |
| collaborators | AgentCollaborator[] | Yes | List of agent collaborators |

#### Agent Collaborator Properties

| Name | Type | Required | Description |
|---|---|---|---|
| agentAlias | AgentAlias | Yes | The agent alias to collaborate with |
| collaborationInstruction | string | Yes | Instructions for how to collaborate with this agent |
| collaboratorName | string | Yes | Name of the collaborator |
| relayConversationHistory | boolean | No | Whether to relay conversation history to the collaborator. Defaults to false |

Example:

```python
# Create a specialized agent
customer_support_agent = bedrock.Agent(self, "CustomerSupportAgent",
    instruction="You specialize in answering customer support questions.",
    foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1
)

# Create an agent alias
customer_support_alias = bedrock.AgentAlias(self, "CustomerSupportAlias",
    agent=customer_support_agent,
    agent_alias_name="production"
)

# Create a main agent that collaborates with the specialized agent
main_agent = bedrock.Agent(self, "MainAgent",
    instruction="You route specialized questions to other agents.",
    foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
    agent_collaboration={
        "type": bedrock.AgentCollaboratorType.SUPERVISOR,
        "collaborators": [
            bedrock.AgentCollaborator(
                agent_alias=customer_support_alias,
                collaboration_instruction="Route customer support questions to this agent.",
                collaborator_name="CustomerSupport",
                relay_conversation_history=True
            )
        ]
    }
)
```

### Custom Orchestration

Custom Orchestration allows you to override the default agent orchestration flow with your own Lambda function. This enables more control over how the agent processes user inputs and invokes action groups.

When you provide a customOrchestrationExecutor, the agent's orchestrationType is automatically set to CUSTOM_ORCHESTRATION. If no customOrchestrationExecutor is provided, the orchestrationType defaults to DEFAULT, using Amazon Bedrock's built-in orchestration.

#### Custom Orchestration Properties

| Name | Type | Required | Description |
|---|---|---|---|
| function | lambda.IFunction | Yes | The Lambda function that implements the custom orchestration logic |

Example:

```python
orchestration_function = lambda_.Function(self, "OrchestrationFunction",
    runtime=lambda_.Runtime.PYTHON_3_10,
    handler="index.handler",
    code=lambda_.Code.from_asset("lambda/orchestration")
)

agent = bedrock.Agent(self, "CustomOrchestrationAgent",
    instruction="You are a helpful assistant with custom orchestration logic.",
    foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
    custom_orchestration_executor=bedrock.CustomOrchestrationExecutor.from_lambda(orchestration_function)
)
```

### Agent Alias

After you have sufficiently iterated on your working draft and are satisfied with the behavior of your agent, you can set it up for deployment and integration into your application by creating aliases.

To deploy your agent, you need to create an alias. During alias creation, Amazon Bedrock automatically creates a version of your agent. The alias points to this newly created version. You can point the alias to a previously created version if necessary. You then configure your application to make API calls to that alias.

By default, the Agent resource creates a test alias named 'AgentTestAlias' that points to the 'DRAFT' version. This test alias is accessible via the `testAlias` property of the agent. You can also create additional aliases for different environments using the AgentAlias construct.

#### Agent Alias Properties

| Name | Type | Required | Description |
|---|---|---|---|
| agent | Agent | Yes | The agent to create an alias for |
| agentAliasName | string | No | The name of the agent alias. Defaults to a name generated by CDK |
| description | string | No | A description of the agent alias. Defaults to no description |
| routingConfiguration | AgentAliasRoutingConfiguration | No | Configuration for routing traffic between agent versions |
| agentVersion | string | No | The version of the agent to use. If not specified, a new version is created |

When redeploying an agent with changes, you must ensure the agent version is updated to avoid deployment failures with "agent already exists" errors. The recommended way to handle this is to include the `lastUpdated` property in the agent's description, which automatically updates whenever the agent is modified. This ensures a new version is created on each deployment.

Example:

```python
agent = bedrock.Agent(self, "Agent",
    foundation_model=bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_HAIKU_V1_0,
    instruction="You are a helpful and friendly agent that answers questions about literature."
)

agent_alias = bedrock.AgentAlias(self, "myAlias",
    agent_alias_name="production",
    agent=agent,
    description=f"Production version of my agent. Created at {agent.lastUpdated}"
)
```
'''
from pkgutil import extend_path
__path__ = extend_path(__path__, __name__)

import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

import typeguard
from importlib.metadata import version as _metadata_package_version
TYPEGUARD_MAJOR_VERSION = int(_metadata_package_version('typeguard').split('.')[0])

def check_type(argname: str, value: object, expected_type: typing.Any) -> typing.Any:
    if TYPEGUARD_MAJOR_VERSION <= 2:
        return typeguard.check_type(argname=argname, value=value, expected_type=expected_type) # type:ignore
    else:
        if isinstance(value, jsii._reference_map.InterfaceDynamicProxy): # pyright: ignore [reportAttributeAccessIssue]
           pass
        else:
            if TYPEGUARD_MAJOR_VERSION == 3:
                typeguard.config.collection_check_strategy = typeguard.CollectionCheckStrategy.ALL_ITEMS # type:ignore
                typeguard.check_type(value=value, expected_type=expected_type) # type:ignore
            else:
                typeguard.check_type(value=value, expected_type=expected_type, collection_check_strategy=typeguard.CollectionCheckStrategy.ALL_ITEMS) # type:ignore

from ._jsii import *

import aws_cdk as _aws_cdk_ceddda9d
import aws_cdk.aws_bedrock as _aws_cdk_aws_bedrock_ceddda9d
import aws_cdk.aws_cloudwatch as _aws_cdk_aws_cloudwatch_ceddda9d
import aws_cdk.aws_events as _aws_cdk_aws_events_ceddda9d
import aws_cdk.aws_iam as _aws_cdk_aws_iam_ceddda9d
import aws_cdk.aws_kms as _aws_cdk_aws_kms_ceddda9d
import aws_cdk.aws_lambda as _aws_cdk_aws_lambda_ceddda9d
import aws_cdk.aws_s3 as _aws_cdk_aws_s3_ceddda9d
import aws_cdk.aws_s3_assets as _aws_cdk_aws_s3_assets_ceddda9d
import constructs as _constructs_77d1e7e8


class ActionGroupExecutor(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-bedrock-alpha.ActionGroupExecutor",
):
    '''(experimental) Defines how fulfillment of the action group is handled after the necessary information has been elicited from the user.

    Valid executors are:

    - Lambda function
    - Return Control

    :see: https://docs.aws.amazon.com/bedrock/latest/userguide/action-handle.html
    :stability: experimental
    :exampleMetadata: fixture=default infused

    Example::

        bucket = s3.Bucket.from_bucket_name(self, "ExistingBucket", "my-schema-bucket")
        s3_schema = bedrock.ApiSchema.from_s3_file(bucket, "schemas/action-group.yaml")
        
        action_group_function = lambda_.Function(self, "ActionGroupFunction",
            runtime=lambda_.Runtime.PYTHON_3_12,
            handler="index.handler",
            code=lambda_.Code.from_asset(path.join(__dirname, "../lambda/action-group"))
        )
        
        action_group = bedrock.AgentActionGroup(
            name="query-library",
            description="Use these functions to get information about the books in the library.",
            executor=bedrock.ActionGroupExecutor.from_lambda(action_group_function),
            enabled=True,
            api_schema=s3_schema
        )
        
        agent = bedrock.Agent(self, "Agent",
            foundation_model=bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_HAIKU_V1_0,
            instruction="You are a helpful and friendly agent that answers questions about literature."
        )
        
        agent.add_action_group(action_group)
    '''

    @jsii.member(jsii_name="fromLambda")
    @builtins.classmethod
    def from_lambda(
        cls,
        lambda_function: _aws_cdk_aws_lambda_ceddda9d.IFunction,
    ) -> "ActionGroupExecutor":
        '''(experimental) Defines an action group with a Lambda function containing the business logic.

        :param lambda_function: - Lambda function to be called by the action group.

        :see: https://docs.aws.amazon.com/bedrock/latest/userguide/agents-lambda.html
        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b3f8715f9bf580f1336c3ef0001b564cd5377a641554f928ee8a9337f101d4ea)
            check_type(argname="argument lambda_function", value=lambda_function, expected_type=type_hints["lambda_function"])
        return typing.cast("ActionGroupExecutor", jsii.sinvoke(cls, "fromLambda", [lambda_function]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="RETURN_CONTROL")
    def RETURN_CONTROL(cls) -> "ActionGroupExecutor":
        '''(experimental) Returns the action group invocation results directly in the InvokeAgent response.

        The information and parameters can be sent to your own systems to yield results.

        :see: https://docs.aws.amazon.com/bedrock/latest/userguide/agents-returncontrol.html
        :stability: experimental
        '''
        return typing.cast("ActionGroupExecutor", jsii.sget(cls, "RETURN_CONTROL"))

    @builtins.property
    @jsii.member(jsii_name="customControl")
    def custom_control(self) -> typing.Optional["CustomControl"]:
        '''(experimental) The custom control type for the action group executor.

        Currently only supports 'RETURN_CONTROL' which returns results directly in the InvokeAgent response.

        :stability: experimental
        '''
        return typing.cast(typing.Optional["CustomControl"], jsii.get(self, "customControl"))

    @builtins.property
    @jsii.member(jsii_name="lambdaFunction")
    def lambda_function(
        self,
    ) -> typing.Optional[_aws_cdk_aws_lambda_ceddda9d.IFunction]:
        '''(experimental) The Lambda function that will be called by the action group.

        Contains the business logic for handling the action group's invocation.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[_aws_cdk_aws_lambda_ceddda9d.IFunction], jsii.get(self, "lambdaFunction"))


class AgentActionGroup(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-bedrock-alpha.AgentActionGroup",
):
    '''(experimental) ****************************************************************************                        DEF - Action Group Class ***************************************************************************.

    :stability: experimental
    :exampleMetadata: fixture=default infused

    Example::

        bucket = s3.Bucket.from_bucket_name(self, "ExistingBucket", "my-schema-bucket")
        s3_schema = bedrock.ApiSchema.from_s3_file(bucket, "schemas/action-group.yaml")
        
        action_group_function = lambda_.Function(self, "ActionGroupFunction",
            runtime=lambda_.Runtime.PYTHON_3_12,
            handler="index.handler",
            code=lambda_.Code.from_asset(path.join(__dirname, "../lambda/action-group"))
        )
        
        action_group = bedrock.AgentActionGroup(
            name="query-library",
            description="Use these functions to get information about the books in the library.",
            executor=bedrock.ActionGroupExecutor.from_lambda(action_group_function),
            enabled=True,
            api_schema=s3_schema
        )
        
        agent = bedrock.Agent(self, "Agent",
            foundation_model=bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_HAIKU_V1_0,
            instruction="You are a helpful and friendly agent that answers questions about literature."
        )
        
        agent.add_action_group(action_group)
    '''

    def __init__(
        self,
        *,
        api_schema: typing.Optional["ApiSchema"] = None,
        description: typing.Optional[builtins.str] = None,
        enabled: typing.Optional[builtins.bool] = None,
        executor: typing.Optional[ActionGroupExecutor] = None,
        force_delete: typing.Optional[builtins.bool] = None,
        function_schema: typing.Optional["FunctionSchema"] = None,
        name: typing.Optional[builtins.str] = None,
        parent_action_group_signature: typing.Optional["ParentActionGroupSignature"] = None,
    ) -> None:
        '''
        :param api_schema: (experimental) The API Schema defining the functions available to the agent. Default: undefined - No API Schema is provided
        :param description: (experimental) A description of the action group. Default: undefined - No description is provided
        :param enabled: (experimental) Specifies whether the action group is available for the agent to invoke or not when sending an InvokeAgent request. Default: true - The action group is enabled
        :param executor: (experimental) The action group executor that implements the API functions. Default: undefined - No executor is provided
        :param force_delete: (experimental) Specifies whether to delete the resource even if it's in use. Default: false - The resource will not be deleted if it's in use
        :param function_schema: (experimental) Defines functions that each define parameters that the agent needs to invoke from the user. NO L2 yet as this doesn't make much sense IMHO. Default: undefined - No function schema is provided
        :param name: (experimental) The name of the action group. Default: - A unique name is generated in the format 'action_group_quick_start_UUID'
        :param parent_action_group_signature: (experimental) The AWS Defined signature for enabling certain capabilities in your agent. When this property is specified, you must leave the description, apiSchema, and actionGroupExecutor fields blank for this action group. Default: undefined - No parent action group signature is provided

        :stability: experimental
        '''
        props = AgentActionGroupProps(
            api_schema=api_schema,
            description=description,
            enabled=enabled,
            executor=executor,
            force_delete=force_delete,
            function_schema=function_schema,
            name=name,
            parent_action_group_signature=parent_action_group_signature,
        )

        jsii.create(self.__class__, self, [props])

    @jsii.member(jsii_name="codeInterpreter")
    @builtins.classmethod
    def code_interpreter(cls, enabled: builtins.bool) -> "AgentActionGroup":
        '''(experimental) Defines an action group that allows your agent to request the user for additional information when trying to complete a task.

        :param enabled: Specifies whether the action group is available for the agent.

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f216a3b9945ded433d1397b46354d778b44544b12c7e1444f18f26697b23c949)
            check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
        return typing.cast("AgentActionGroup", jsii.sinvoke(cls, "codeInterpreter", [enabled]))

    @jsii.member(jsii_name="userInput")
    @builtins.classmethod
    def user_input(cls, enabled: builtins.bool) -> "AgentActionGroup":
        '''(experimental) Defines an action group that allows your agent to request the user for additional information when trying to complete a task.

        :param enabled: Specifies whether the action group is available for the agent.

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c7e35cbb8b23de7b78171edfc3dfea96900c1650cd60b84a3b3daafe40b03e1e)
            check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
        return typing.cast("AgentActionGroup", jsii.sinvoke(cls, "userInput", [enabled]))

    @builtins.property
    @jsii.member(jsii_name="enabled")
    def enabled(self) -> builtins.bool:
        '''(experimental) Whether this action group is available for the agent to invoke or not.

        :stability: experimental
        '''
        return typing.cast(builtins.bool, jsii.get(self, "enabled"))

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        '''(experimental) The name of the action group.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "name"))

    @builtins.property
    @jsii.member(jsii_name="apiSchema")
    def api_schema(self) -> typing.Optional["ApiSchema"]:
        '''(experimental) The api schema for this action group (if defined).

        :stability: experimental
        '''
        return typing.cast(typing.Optional["ApiSchema"], jsii.get(self, "apiSchema"))

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''(experimental) A description of the action group.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @builtins.property
    @jsii.member(jsii_name="executor")
    def executor(self) -> typing.Optional[ActionGroupExecutor]:
        '''(experimental) The action group executor for this action group (if defined).

        :stability: experimental
        '''
        return typing.cast(typing.Optional[ActionGroupExecutor], jsii.get(self, "executor"))

    @builtins.property
    @jsii.member(jsii_name="forceDelete")
    def force_delete(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to delete the resource even if it's in use.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.bool], jsii.get(self, "forceDelete"))

    @builtins.property
    @jsii.member(jsii_name="functionSchema")
    def function_schema(self) -> typing.Optional["FunctionSchema"]:
        '''(experimental) The function schema for this action group (if defined).

        :stability: experimental
        '''
        return typing.cast(typing.Optional["FunctionSchema"], jsii.get(self, "functionSchema"))

    @builtins.property
    @jsii.member(jsii_name="parentActionGroupSignature")
    def parent_action_group_signature(
        self,
    ) -> typing.Optional["ParentActionGroupSignature"]:
        '''(experimental) The AWS Defined signature (if defined).

        :stability: experimental
        '''
        return typing.cast(typing.Optional["ParentActionGroupSignature"], jsii.get(self, "parentActionGroupSignature"))


@jsii.data_type(
    jsii_type="@aws-cdk/aws-bedrock-alpha.AgentActionGroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "api_schema": "apiSchema",
        "description": "description",
        "enabled": "enabled",
        "executor": "executor",
        "force_delete": "forceDelete",
        "function_schema": "functionSchema",
        "name": "name",
        "parent_action_group_signature": "parentActionGroupSignature",
    },
)
class AgentActionGroupProps:
    def __init__(
        self,
        *,
        api_schema: typing.Optional["ApiSchema"] = None,
        description: typing.Optional[builtins.str] = None,
        enabled: typing.Optional[builtins.bool] = None,
        executor: typing.Optional[ActionGroupExecutor] = None,
        force_delete: typing.Optional[builtins.bool] = None,
        function_schema: typing.Optional["FunctionSchema"] = None,
        name: typing.Optional[builtins.str] = None,
        parent_action_group_signature: typing.Optional["ParentActionGroupSignature"] = None,
    ) -> None:
        '''(experimental) ****************************************************************************                        PROPS - Action Group Class ***************************************************************************.

        :param api_schema: (experimental) The API Schema defining the functions available to the agent. Default: undefined - No API Schema is provided
        :param description: (experimental) A description of the action group. Default: undefined - No description is provided
        :param enabled: (experimental) Specifies whether the action group is available for the agent to invoke or not when sending an InvokeAgent request. Default: true - The action group is enabled
        :param executor: (experimental) The action group executor that implements the API functions. Default: undefined - No executor is provided
        :param force_delete: (experimental) Specifies whether to delete the resource even if it's in use. Default: false - The resource will not be deleted if it's in use
        :param function_schema: (experimental) Defines functions that each define parameters that the agent needs to invoke from the user. NO L2 yet as this doesn't make much sense IMHO. Default: undefined - No function schema is provided
        :param name: (experimental) The name of the action group. Default: - A unique name is generated in the format 'action_group_quick_start_UUID'
        :param parent_action_group_signature: (experimental) The AWS Defined signature for enabling certain capabilities in your agent. When this property is specified, you must leave the description, apiSchema, and actionGroupExecutor fields blank for this action group. Default: undefined - No parent action group signature is provided

        :stability: experimental
        :exampleMetadata: fixture=default infused

        Example::

            bucket = s3.Bucket.from_bucket_name(self, "ExistingBucket", "my-schema-bucket")
            s3_schema = bedrock.ApiSchema.from_s3_file(bucket, "schemas/action-group.yaml")
            
            action_group_function = lambda_.Function(self, "ActionGroupFunction",
                runtime=lambda_.Runtime.PYTHON_3_12,
                handler="index.handler",
                code=lambda_.Code.from_asset(path.join(__dirname, "../lambda/action-group"))
            )
            
            action_group = bedrock.AgentActionGroup(
                name="query-library",
                description="Use these functions to get information about the books in the library.",
                executor=bedrock.ActionGroupExecutor.from_lambda(action_group_function),
                enabled=True,
                api_schema=s3_schema
            )
            
            agent = bedrock.Agent(self, "Agent",
                foundation_model=bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_HAIKU_V1_0,
                instruction="You are a helpful and friendly agent that answers questions about literature."
            )
            
            agent.add_action_group(action_group)
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__96565235b78a53a99b9a5b0de1f6e1cb436217f7023225486fb3b6b0402c16b1)
            check_type(argname="argument api_schema", value=api_schema, expected_type=type_hints["api_schema"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
            check_type(argname="argument executor", value=executor, expected_type=type_hints["executor"])
            check_type(argname="argument force_delete", value=force_delete, expected_type=type_hints["force_delete"])
            check_type(argname="argument function_schema", value=function_schema, expected_type=type_hints["function_schema"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument parent_action_group_signature", value=parent_action_group_signature, expected_type=type_hints["parent_action_group_signature"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if api_schema is not None:
            self._values["api_schema"] = api_schema
        if description is not None:
            self._values["description"] = description
        if enabled is not None:
            self._values["enabled"] = enabled
        if executor is not None:
            self._values["executor"] = executor
        if force_delete is not None:
            self._values["force_delete"] = force_delete
        if function_schema is not None:
            self._values["function_schema"] = function_schema
        if name is not None:
            self._values["name"] = name
        if parent_action_group_signature is not None:
            self._values["parent_action_group_signature"] = parent_action_group_signature

    @builtins.property
    def api_schema(self) -> typing.Optional["ApiSchema"]:
        '''(experimental) The API Schema defining the functions available to the agent.

        :default: undefined - No API Schema is provided

        :stability: experimental
        '''
        result = self._values.get("api_schema")
        return typing.cast(typing.Optional["ApiSchema"], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(experimental) A description of the action group.

        :default: undefined - No description is provided

        :stability: experimental
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def enabled(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether the action group is available for the agent to invoke or not when sending an InvokeAgent request.

        :default: true - The action group is enabled

        :stability: experimental
        '''
        result = self._values.get("enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def executor(self) -> typing.Optional[ActionGroupExecutor]:
        '''(experimental) The action group executor that implements the API functions.

        :default: undefined - No executor is provided

        :stability: experimental
        '''
        result = self._values.get("executor")
        return typing.cast(typing.Optional[ActionGroupExecutor], result)

    @builtins.property
    def force_delete(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether to delete the resource even if it's in use.

        :default: false - The resource will not be deleted if it's in use

        :stability: experimental
        '''
        result = self._values.get("force_delete")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def function_schema(self) -> typing.Optional["FunctionSchema"]:
        '''(experimental) Defines functions that each define parameters that the agent needs to invoke from the user.

        NO L2 yet as this doesn't make much sense IMHO.

        :default: undefined - No function schema is provided

        :stability: experimental
        '''
        result = self._values.get("function_schema")
        return typing.cast(typing.Optional["FunctionSchema"], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the action group.

        :default: - A unique name is generated in the format 'action_group_quick_start_UUID'

        :stability: experimental
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def parent_action_group_signature(
        self,
    ) -> typing.Optional["ParentActionGroupSignature"]:
        '''(experimental) The AWS Defined signature for enabling certain capabilities in your agent.

        When this property is specified, you must leave the description, apiSchema,
        and actionGroupExecutor fields blank for this action group.

        :default: undefined - No parent action group signature is provided

        :stability: experimental
        '''
        result = self._values.get("parent_action_group_signature")
        return typing.cast(typing.Optional["ParentActionGroupSignature"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AgentActionGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-bedrock-alpha.AgentAliasAttributes",
    jsii_struct_bases=[],
    name_mapping={
        "agent": "agent",
        "agent_version": "agentVersion",
        "alias_id": "aliasId",
        "alias_name": "aliasName",
    },
)
class AgentAliasAttributes:
    def __init__(
        self,
        *,
        agent: "IAgent",
        agent_version: builtins.str,
        alias_id: builtins.str,
        alias_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) Attributes needed to create an import.

        :param agent: (experimental) The underlying agent for this alias.
        :param agent_version: (experimental) The agent version for this alias.
        :param alias_id: (experimental) The Id of the agent alias.
        :param alias_name: (experimental) The name of the agent alias. Default: undefined - No alias name is provided

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_bedrock_alpha as bedrock_alpha
            
            # agent: bedrock_alpha.Agent
            
            agent_alias_attributes = bedrock_alpha.AgentAliasAttributes(
                agent=agent,
                agent_version="agentVersion",
                alias_id="aliasId",
            
                # the properties below are optional
                alias_name="aliasName"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2cf3ee27a5cffdeb97ea680716346f7407718c3709c2070bc5cfe5846adf5e53)
            check_type(argname="argument agent", value=agent, expected_type=type_hints["agent"])
            check_type(argname="argument agent_version", value=agent_version, expected_type=type_hints["agent_version"])
            check_type(argname="argument alias_id", value=alias_id, expected_type=type_hints["alias_id"])
            check_type(argname="argument alias_name", value=alias_name, expected_type=type_hints["alias_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "agent": agent,
            "agent_version": agent_version,
            "alias_id": alias_id,
        }
        if alias_name is not None:
            self._values["alias_name"] = alias_name

    @builtins.property
    def agent(self) -> "IAgent":
        '''(experimental) The underlying agent for this alias.

        :stability: experimental
        '''
        result = self._values.get("agent")
        assert result is not None, "Required property 'agent' is missing"
        return typing.cast("IAgent", result)

    @builtins.property
    def agent_version(self) -> builtins.str:
        '''(experimental) The agent version for this alias.

        :stability: experimental
        '''
        result = self._values.get("agent_version")
        assert result is not None, "Required property 'agent_version' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def alias_id(self) -> builtins.str:
        '''(experimental) The Id of the agent alias.

        :stability: experimental
        '''
        result = self._values.get("alias_id")
        assert result is not None, "Required property 'alias_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def alias_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the agent alias.

        :default: undefined - No alias name is provided

        :stability: experimental
        '''
        result = self._values.get("alias_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AgentAliasAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-bedrock-alpha.AgentAliasProps",
    jsii_struct_bases=[],
    name_mapping={
        "agent": "agent",
        "agent_alias_name": "agentAliasName",
        "agent_version": "agentVersion",
        "description": "description",
    },
)
class AgentAliasProps:
    def __init__(
        self,
        *,
        agent: "IAgent",
        agent_alias_name: typing.Optional[builtins.str] = None,
        agent_version: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) Properties for creating a CDK-Managed Agent Alias.

        :param agent: (experimental) The agent associated to this alias.
        :param agent_alias_name: (experimental) The name for the agent alias. This will be used as the physical name of the agent alias. Default: - "latest"
        :param agent_version: (experimental) The version of the agent to associate with the agent alias. Default: - Creates a new version of the agent.
        :param description: (experimental) Description for the agent alias. Default: undefined - No description is provided

        :stability: experimental
        :exampleMetadata: fixture=default infused

        Example::

            # Create a specialized agent
            customer_support_agent = bedrock.Agent(self, "CustomerSupportAgent",
                instruction="You specialize in answering customer support questions.",
                foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1
            )
            
            # Create an agent alias
            customer_support_alias = bedrock.AgentAlias(self, "CustomerSupportAlias",
                agent=customer_support_agent,
                agent_alias_name="production"
            )
            
            # Create a main agent that collaborates with the specialized agent
            main_agent = bedrock.Agent(self, "MainAgent",
                instruction="You route specialized questions to other agents.",
                foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
                agent_collaboration={
                    "type": bedrock.AgentCollaboratorType.SUPERVISOR,
                    "collaborators": [
                        bedrock.AgentCollaborator(
                            agent_alias=customer_support_alias,
                            collaboration_instruction="Route customer support questions to this agent.",
                            collaborator_name="CustomerSupport",
                            relay_conversation_history=True
                        )
                    ]
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__223e58e52a6c762628fbd5894a3746df0edc021abe283a3fa5fae709b84c6e2b)
            check_type(argname="argument agent", value=agent, expected_type=type_hints["agent"])
            check_type(argname="argument agent_alias_name", value=agent_alias_name, expected_type=type_hints["agent_alias_name"])
            check_type(argname="argument agent_version", value=agent_version, expected_type=type_hints["agent_version"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "agent": agent,
        }
        if agent_alias_name is not None:
            self._values["agent_alias_name"] = agent_alias_name
        if agent_version is not None:
            self._values["agent_version"] = agent_version
        if description is not None:
            self._values["description"] = description

    @builtins.property
    def agent(self) -> "IAgent":
        '''(experimental) The agent associated to this alias.

        :stability: experimental
        '''
        result = self._values.get("agent")
        assert result is not None, "Required property 'agent' is missing"
        return typing.cast("IAgent", result)

    @builtins.property
    def agent_alias_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name for the agent alias.

        This will be used as the physical name of the agent alias.

        :default: - "latest"

        :stability: experimental
        '''
        result = self._values.get("agent_alias_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def agent_version(self) -> typing.Optional[builtins.str]:
        '''(experimental) The version of the agent to associate with the agent alias.

        :default: - Creates a new version of the agent.

        :stability: experimental
        '''
        result = self._values.get("agent_version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(experimental) Description for the agent alias.

        :default: undefined - No description is provided

        :stability: experimental
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AgentAliasProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-bedrock-alpha.AgentAttributes",
    jsii_struct_bases=[],
    name_mapping={
        "agent_arn": "agentArn",
        "role_arn": "roleArn",
        "agent_version": "agentVersion",
        "kms_key_arn": "kmsKeyArn",
        "last_updated": "lastUpdated",
    },
)
class AgentAttributes:
    def __init__(
        self,
        *,
        agent_arn: builtins.str,
        role_arn: builtins.str,
        agent_version: typing.Optional[builtins.str] = None,
        kms_key_arn: typing.Optional[builtins.str] = None,
        last_updated: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) Attributes for specifying an imported Bedrock Agent.

        :param agent_arn: (experimental) The ARN of the agent.
        :param role_arn: (experimental) The ARN of the IAM role associated to the agent.
        :param agent_version: (experimental) The agent version. If no explicit versions have been created, leave this empty to use the DRAFT version. Otherwise, use the version number (e.g. 1). Default: 'DRAFT'
        :param kms_key_arn: (experimental) Optional KMS encryption key associated with this agent. Default: undefined - An AWS managed key is used
        :param last_updated: (experimental) When this agent was last updated. Default: undefined - No last updated timestamp is provided

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_bedrock_alpha as bedrock_alpha
            
            agent_attributes = bedrock_alpha.AgentAttributes(
                agent_arn="agentArn",
                role_arn="roleArn",
            
                # the properties below are optional
                agent_version="agentVersion",
                kms_key_arn="kmsKeyArn",
                last_updated="lastUpdated"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ec4361e61c0d3b4b62f635521602d37a1a496b4cd051caa617df76fefcfb8daa)
            check_type(argname="argument agent_arn", value=agent_arn, expected_type=type_hints["agent_arn"])
            check_type(argname="argument role_arn", value=role_arn, expected_type=type_hints["role_arn"])
            check_type(argname="argument agent_version", value=agent_version, expected_type=type_hints["agent_version"])
            check_type(argname="argument kms_key_arn", value=kms_key_arn, expected_type=type_hints["kms_key_arn"])
            check_type(argname="argument last_updated", value=last_updated, expected_type=type_hints["last_updated"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "agent_arn": agent_arn,
            "role_arn": role_arn,
        }
        if agent_version is not None:
            self._values["agent_version"] = agent_version
        if kms_key_arn is not None:
            self._values["kms_key_arn"] = kms_key_arn
        if last_updated is not None:
            self._values["last_updated"] = last_updated

    @builtins.property
    def agent_arn(self) -> builtins.str:
        '''(experimental) The ARN of the agent.

        :stability: experimental
        :attribute: true
        '''
        result = self._values.get("agent_arn")
        assert result is not None, "Required property 'agent_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def role_arn(self) -> builtins.str:
        '''(experimental) The ARN of the IAM role associated to the agent.

        :stability: experimental
        :attribute: true
        '''
        result = self._values.get("role_arn")
        assert result is not None, "Required property 'role_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def agent_version(self) -> typing.Optional[builtins.str]:
        '''(experimental) The agent version.

        If no explicit versions have been created,
        leave this empty to use the DRAFT version. Otherwise, use the
        version number (e.g. 1).

        :default: 'DRAFT'

        :stability: experimental
        '''
        result = self._values.get("agent_version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def kms_key_arn(self) -> typing.Optional[builtins.str]:
        '''(experimental) Optional KMS encryption key associated with this agent.

        :default: undefined - An AWS managed key is used

        :stability: experimental
        '''
        result = self._values.get("kms_key_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def last_updated(self) -> typing.Optional[builtins.str]:
        '''(experimental) When this agent was last updated.

        :default: undefined - No last updated timestamp is provided

        :stability: experimental
        '''
        result = self._values.get("last_updated")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AgentAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AgentCollaboration(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-bedrock-alpha.AgentCollaboration",
):
    '''(experimental) Class to manage agent collaboration configuration.

    :stability: experimental
    :exampleMetadata: fixture=default infused

    Example::

        # Create a specialized agent
        customer_support_agent = bedrock.Agent(self, "CustomerSupportAgent",
            instruction="You specialize in answering customer support questions.",
            foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1
        )
        
        # Create an agent alias
        customer_support_alias = bedrock.AgentAlias(self, "CustomerSupportAlias",
            agent=customer_support_agent,
            agent_alias_name="production"
        )
        
        # Create a main agent that collaborates with the specialized agent
        main_agent = bedrock.Agent(self, "MainAgent",
            instruction="You route specialized questions to other agents.",
            foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
            agent_collaboration={
                "type": bedrock.AgentCollaboratorType.SUPERVISOR,
                "collaborators": [
                    bedrock.AgentCollaborator(
                        agent_alias=customer_support_alias,
                        collaboration_instruction="Route customer support questions to this agent.",
                        collaborator_name="CustomerSupport",
                        relay_conversation_history=True
                    )
                ]
            }
        )
    '''

    def __init__(
        self,
        *,
        collaborators: typing.Sequence["AgentCollaborator"],
        type: "AgentCollaboratorType",
    ) -> None:
        '''
        :param collaborators: (experimental) Collaborators that this agent will work with.
        :param type: (experimental) The collaboration type for the agent.

        :stability: experimental
        '''
        config = AgentCollaborationConfig(collaborators=collaborators, type=type)

        jsii.create(self.__class__, self, [config])

    @builtins.property
    @jsii.member(jsii_name="collaborators")
    def collaborators(self) -> typing.List["AgentCollaborator"]:
        '''(experimental) Collaborators that this agent will work with.

        :stability: experimental
        '''
        return typing.cast(typing.List["AgentCollaborator"], jsii.get(self, "collaborators"))

    @builtins.property
    @jsii.member(jsii_name="type")
    def type(self) -> "AgentCollaboratorType":
        '''(experimental) The collaboration type for the agent.

        :stability: experimental
        '''
        return typing.cast("AgentCollaboratorType", jsii.get(self, "type"))


@jsii.data_type(
    jsii_type="@aws-cdk/aws-bedrock-alpha.AgentCollaborationConfig",
    jsii_struct_bases=[],
    name_mapping={"collaborators": "collaborators", "type": "type"},
)
class AgentCollaborationConfig:
    def __init__(
        self,
        *,
        collaborators: typing.Sequence["AgentCollaborator"],
        type: "AgentCollaboratorType",
    ) -> None:
        '''(experimental) Configuration for agent collaboration settings.

        :param collaborators: (experimental) Collaborators that this agent will work with.
        :param type: (experimental) The collaboration type for the agent.

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_bedrock_alpha as bedrock_alpha
            
            # agent_collaborator: bedrock_alpha.AgentCollaborator
            
            agent_collaboration_config = bedrock_alpha.AgentCollaborationConfig(
                collaborators=[agent_collaborator],
                type=bedrock_alpha.AgentCollaboratorType.SUPERVISOR
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8a9907acc35e0edfd5ff2b73fc849757eb706524befb93fa2d2644edefa77814)
            check_type(argname="argument collaborators", value=collaborators, expected_type=type_hints["collaborators"])
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "collaborators": collaborators,
            "type": type,
        }

    @builtins.property
    def collaborators(self) -> typing.List["AgentCollaborator"]:
        '''(experimental) Collaborators that this agent will work with.

        :stability: experimental
        '''
        result = self._values.get("collaborators")
        assert result is not None, "Required property 'collaborators' is missing"
        return typing.cast(typing.List["AgentCollaborator"], result)

    @builtins.property
    def type(self) -> "AgentCollaboratorType":
        '''(experimental) The collaboration type for the agent.

        :stability: experimental
        '''
        result = self._values.get("type")
        assert result is not None, "Required property 'type' is missing"
        return typing.cast("AgentCollaboratorType", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AgentCollaborationConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AgentCollaborator(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-bedrock-alpha.AgentCollaborator",
):
    '''(experimental) ****************************************************************************                        Agent Collaborator Class ***************************************************************************.

    :stability: experimental
    :exampleMetadata: fixture=default infused

    Example::

        # Create a specialized agent
        customer_support_agent = bedrock.Agent(self, "CustomerSupportAgent",
            instruction="You specialize in answering customer support questions.",
            foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1
        )
        
        # Create an agent alias
        customer_support_alias = bedrock.AgentAlias(self, "CustomerSupportAlias",
            agent=customer_support_agent,
            agent_alias_name="production"
        )
        
        # Create a main agent that collaborates with the specialized agent
        main_agent = bedrock.Agent(self, "MainAgent",
            instruction="You route specialized questions to other agents.",
            foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
            agent_collaboration={
                "type": bedrock.AgentCollaboratorType.SUPERVISOR,
                "collaborators": [
                    bedrock.AgentCollaborator(
                        agent_alias=customer_support_alias,
                        collaboration_instruction="Route customer support questions to this agent.",
                        collaborator_name="CustomerSupport",
                        relay_conversation_history=True
                    )
                ]
            }
        )
    '''

    def __init__(
        self,
        *,
        agent_alias: "IAgentAlias",
        collaboration_instruction: builtins.str,
        collaborator_name: builtins.str,
        relay_conversation_history: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''
        :param agent_alias: (experimental) Descriptor for the collaborating agent. This cannot be the TSTALIASID (``agent.testAlias``).
        :param collaboration_instruction: (experimental) Instructions on how this agent should collaborate with the main agent.
        :param collaborator_name: (experimental) A friendly name for the collaborator.
        :param relay_conversation_history: (experimental) Whether to relay conversation history to this collaborator. Default: - undefined (uses service default)

        :stability: experimental
        '''
        props = AgentCollaboratorProps(
            agent_alias=agent_alias,
            collaboration_instruction=collaboration_instruction,
            collaborator_name=collaborator_name,
            relay_conversation_history=relay_conversation_history,
        )

        jsii.create(self.__class__, self, [props])

    @jsii.member(jsii_name="grant")
    def grant(
        self,
        grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
    ) -> _aws_cdk_aws_iam_ceddda9d.Grant:
        '''(experimental) Grants the given identity permissions to collaborate with the agent.

        :param grantee: The principal to grant permissions to.

        :return: The Grant object

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e5dcc21bf3943fccd3a13e28a26304d667dc4ab7c6a95e6ba7b947029e97b6f3)
            check_type(argname="argument grantee", value=grantee, expected_type=type_hints["grantee"])
        return typing.cast(_aws_cdk_aws_iam_ceddda9d.Grant, jsii.invoke(self, "grant", [grantee]))

    @builtins.property
    @jsii.member(jsii_name="agentAlias")
    def agent_alias(self) -> "IAgentAlias":
        '''(experimental) The agent alias that this collaborator represents.

        This is the agent that will be called upon for collaboration.

        :stability: experimental
        '''
        return typing.cast("IAgentAlias", jsii.get(self, "agentAlias"))

    @builtins.property
    @jsii.member(jsii_name="collaborationInstruction")
    def collaboration_instruction(self) -> builtins.str:
        '''(experimental) Instructions on how this agent should collaborate with the main agent.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "collaborationInstruction"))

    @builtins.property
    @jsii.member(jsii_name="collaboratorName")
    def collaborator_name(self) -> builtins.str:
        '''(experimental) A friendly name for the collaborator.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "collaboratorName"))

    @builtins.property
    @jsii.member(jsii_name="relayConversationHistory")
    def relay_conversation_history(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to relay conversation history to this collaborator.

        :default: - undefined (uses service default)

        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.bool], jsii.get(self, "relayConversationHistory"))


@jsii.data_type(
    jsii_type="@aws-cdk/aws-bedrock-alpha.AgentCollaboratorProps",
    jsii_struct_bases=[],
    name_mapping={
        "agent_alias": "agentAlias",
        "collaboration_instruction": "collaborationInstruction",
        "collaborator_name": "collaboratorName",
        "relay_conversation_history": "relayConversationHistory",
    },
)
class AgentCollaboratorProps:
    def __init__(
        self,
        *,
        agent_alias: "IAgentAlias",
        collaboration_instruction: builtins.str,
        collaborator_name: builtins.str,
        relay_conversation_history: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''(experimental) ****************************************************************************                   PROPS - Agent Collaborator Class ***************************************************************************.

        :param agent_alias: (experimental) Descriptor for the collaborating agent. This cannot be the TSTALIASID (``agent.testAlias``).
        :param collaboration_instruction: (experimental) Instructions on how this agent should collaborate with the main agent.
        :param collaborator_name: (experimental) A friendly name for the collaborator.
        :param relay_conversation_history: (experimental) Whether to relay conversation history to this collaborator. Default: - undefined (uses service default)

        :stability: experimental
        :exampleMetadata: fixture=default infused

        Example::

            # Create a specialized agent
            customer_support_agent = bedrock.Agent(self, "CustomerSupportAgent",
                instruction="You specialize in answering customer support questions.",
                foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1
            )
            
            # Create an agent alias
            customer_support_alias = bedrock.AgentAlias(self, "CustomerSupportAlias",
                agent=customer_support_agent,
                agent_alias_name="production"
            )
            
            # Create a main agent that collaborates with the specialized agent
            main_agent = bedrock.Agent(self, "MainAgent",
                instruction="You route specialized questions to other agents.",
                foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
                agent_collaboration={
                    "type": bedrock.AgentCollaboratorType.SUPERVISOR,
                    "collaborators": [
                        bedrock.AgentCollaborator(
                            agent_alias=customer_support_alias,
                            collaboration_instruction="Route customer support questions to this agent.",
                            collaborator_name="CustomerSupport",
                            relay_conversation_history=True
                        )
                    ]
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__38cb6ea0e190b8eef95b7964639213d5273c82e36cad360e5f54a538ea1015a1)
            check_type(argname="argument agent_alias", value=agent_alias, expected_type=type_hints["agent_alias"])
            check_type(argname="argument collaboration_instruction", value=collaboration_instruction, expected_type=type_hints["collaboration_instruction"])
            check_type(argname="argument collaborator_name", value=collaborator_name, expected_type=type_hints["collaborator_name"])
            check_type(argname="argument relay_conversation_history", value=relay_conversation_history, expected_type=type_hints["relay_conversation_history"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "agent_alias": agent_alias,
            "collaboration_instruction": collaboration_instruction,
            "collaborator_name": collaborator_name,
        }
        if relay_conversation_history is not None:
            self._values["relay_conversation_history"] = relay_conversation_history

    @builtins.property
    def agent_alias(self) -> "IAgentAlias":
        '''(experimental) Descriptor for the collaborating agent.

        This cannot be the TSTALIASID (``agent.testAlias``).

        :stability: experimental
        '''
        result = self._values.get("agent_alias")
        assert result is not None, "Required property 'agent_alias' is missing"
        return typing.cast("IAgentAlias", result)

    @builtins.property
    def collaboration_instruction(self) -> builtins.str:
        '''(experimental) Instructions on how this agent should collaborate with the main agent.

        :stability: experimental
        '''
        result = self._values.get("collaboration_instruction")
        assert result is not None, "Required property 'collaboration_instruction' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def collaborator_name(self) -> builtins.str:
        '''(experimental) A friendly name for the collaborator.

        :stability: experimental
        '''
        result = self._values.get("collaborator_name")
        assert result is not None, "Required property 'collaborator_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def relay_conversation_history(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to relay conversation history to this collaborator.

        :default: - undefined (uses service default)

        :stability: experimental
        '''
        result = self._values.get("relay_conversation_history")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AgentCollaboratorProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@aws-cdk/aws-bedrock-alpha.AgentCollaboratorType")
class AgentCollaboratorType(enum.Enum):
    '''(experimental) Enum for collaborator's relay conversation history types.

    :stability: experimental
    :exampleMetadata: fixture=default infused

    Example::

        # Create a specialized agent
        customer_support_agent = bedrock.Agent(self, "CustomerSupportAgent",
            instruction="You specialize in answering customer support questions.",
            foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1
        )
        
        # Create an agent alias
        customer_support_alias = bedrock.AgentAlias(self, "CustomerSupportAlias",
            agent=customer_support_agent,
            agent_alias_name="production"
        )
        
        # Create a main agent that collaborates with the specialized agent
        main_agent = bedrock.Agent(self, "MainAgent",
            instruction="You route specialized questions to other agents.",
            foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
            agent_collaboration={
                "type": bedrock.AgentCollaboratorType.SUPERVISOR,
                "collaborators": [
                    bedrock.AgentCollaborator(
                        agent_alias=customer_support_alias,
                        collaboration_instruction="Route customer support questions to this agent.",
                        collaborator_name="CustomerSupport",
                        relay_conversation_history=True
                    )
                ]
            }
        )
    '''

    SUPERVISOR = "SUPERVISOR"
    '''(experimental) Supervisor agent.

    :stability: experimental
    '''
    DISABLED = "DISABLED"
    '''(experimental) Disabling collaboration.

    :stability: experimental
    '''
    SUPERVISOR_ROUTER = "SUPERVISOR_ROUTER"
    '''(experimental) Supervisor router.

    :stability: experimental
    '''


@jsii.data_type(
    jsii_type="@aws-cdk/aws-bedrock-alpha.AgentProps",
    jsii_struct_bases=[],
    name_mapping={
        "foundation_model": "foundationModel",
        "instruction": "instruction",
        "action_groups": "actionGroups",
        "agent_collaboration": "agentCollaboration",
        "agent_name": "agentName",
        "code_interpreter_enabled": "codeInterpreterEnabled",
        "custom_orchestration_executor": "customOrchestrationExecutor",
        "description": "description",
        "existing_role": "existingRole",
        "force_delete": "forceDelete",
        "idle_session_ttl": "idleSessionTTL",
        "kms_key": "kmsKey",
        "memory": "memory",
        "prompt_override_configuration": "promptOverrideConfiguration",
        "should_prepare_agent": "shouldPrepareAgent",
        "user_input_enabled": "userInputEnabled",
    },
)
class AgentProps:
    def __init__(
        self,
        *,
        foundation_model: "IBedrockInvokable",
        instruction: builtins.str,
        action_groups: typing.Optional[typing.Sequence[AgentActionGroup]] = None,
        agent_collaboration: typing.Optional[AgentCollaboration] = None,
        agent_name: typing.Optional[builtins.str] = None,
        code_interpreter_enabled: typing.Optional[builtins.bool] = None,
        custom_orchestration_executor: typing.Optional["CustomOrchestrationExecutor"] = None,
        description: typing.Optional[builtins.str] = None,
        existing_role: typing.Optional[_aws_cdk_aws_iam_ceddda9d.IRole] = None,
        force_delete: typing.Optional[builtins.bool] = None,
        idle_session_ttl: typing.Optional[_aws_cdk_ceddda9d.Duration] = None,
        kms_key: typing.Optional[_aws_cdk_aws_kms_ceddda9d.IKey] = None,
        memory: typing.Optional["Memory"] = None,
        prompt_override_configuration: typing.Optional["PromptOverrideConfiguration"] = None,
        should_prepare_agent: typing.Optional[builtins.bool] = None,
        user_input_enabled: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''(experimental) Properties for creating a CDK managed Bedrock Agent.

        TODO: Knowledge bases configuration will be added in a future update
        TODO: Guardrails configuration will be added in a future update
        TODO: Inference profile configuration will be added in a future update

        :param foundation_model: (experimental) The foundation model used for orchestration by the agent.
        :param instruction: (experimental) The instruction used by the agent. This determines how the agent will perform his task. This instruction must have a minimum of 40 characters.
        :param action_groups: (experimental) The Action Groups associated with the agent. Default: - Only default action groups (UserInput and CodeInterpreter) are added
        :param agent_collaboration: (experimental) Configuration for agent collaboration settings, including AgentCollaboratorType and AgentCollaborators. This property allows you to define how the agent collaborates with other agents and what collaborators it can work with. Default: - No agent collaboration configuration.
        :param agent_name: (experimental) The name of the agent. This will be used as the physical name of the agent. Default: - A name is generated by CDK. Supported pattern : ^([0-9a-zA-Z][_-]?){1,100}$
        :param code_interpreter_enabled: (experimental) Select whether the agent can generate, run, and troubleshoot code when trying to complete a task. Default: - false
        :param custom_orchestration_executor: (experimental) The Lambda function to use for custom orchestration. If provided, custom orchestration will be used. If not provided, default orchestration will be used. Default: - Default orchestration
        :param description: (experimental) A description of the agent. Default: - No description is provided.
        :param existing_role: (experimental) An existing IAM Role to associate with this agent. Use this property when you want to reuse an existing IAM role rather than create a new one. The role must have a trust policy that allows the Bedrock service to assume it. Default: - A new role is created for you.
        :param force_delete: (experimental) Whether to delete the resource even if it's in use. Default: - false
        :param idle_session_ttl: (experimental) How long sessions should be kept open for the agent. If no conversation occurs during this time, the session expires and Amazon Bedrock deletes any data provided before the timeout. Default: - 10 minutes
        :param kms_key: (experimental) The KMS key of the agent if custom encryption is configured. Default: - An AWS managed key is used.
        :param memory: (experimental) The type and configuration of the memory to maintain context across multiple sessions and recall past interactions. This can be useful for maintaining continuity in multi-turn conversations and recalling user preferences or past interactions. Default: - No memory will be used. Agents will retain context from the current session only.
        :param prompt_override_configuration: (experimental) Overrides some prompt templates in different parts of an agent sequence configuration. Default: - No overrides are provided.
        :param should_prepare_agent: (experimental) Specifies whether to automatically update the ``DRAFT`` version of the agent after making changes to the agent. The ``DRAFT`` version can be continually iterated upon during internal development. Default: - false
        :param user_input_enabled: (experimental) Select whether the agent can prompt additional information from the user when it does not have enough information to respond to an utterance. Default: - false

        :stability: experimental
        :exampleMetadata: fixture=default infused

        Example::

            parser_function = lambda_.Function(self, "ParserFunction",
                runtime=lambda_.Runtime.PYTHON_3_10,
                handler="index.handler",
                code=lambda_.Code.from_asset("lambda")
            )
            
            agent = bedrock.Agent(self, "Agent",
                foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
                instruction="You are a helpful assistant.",
                prompt_override_configuration=bedrock.PromptOverrideConfiguration.with_custom_parser(
                    parser=parser_function,
                    pre_processing_step=bedrock.PromptPreProcessingConfigCustomParser(
                        step_type=bedrock.AgentStepType.PRE_PROCESSING,
                        use_custom_parser=True
                    )
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1e71f5eada4b0acb202729bade487b2adb4e2e2b8fd4546f4cb3bcbc8f3d8cc1)
            check_type(argname="argument foundation_model", value=foundation_model, expected_type=type_hints["foundation_model"])
            check_type(argname="argument instruction", value=instruction, expected_type=type_hints["instruction"])
            check_type(argname="argument action_groups", value=action_groups, expected_type=type_hints["action_groups"])
            check_type(argname="argument agent_collaboration", value=agent_collaboration, expected_type=type_hints["agent_collaboration"])
            check_type(argname="argument agent_name", value=agent_name, expected_type=type_hints["agent_name"])
            check_type(argname="argument code_interpreter_enabled", value=code_interpreter_enabled, expected_type=type_hints["code_interpreter_enabled"])
            check_type(argname="argument custom_orchestration_executor", value=custom_orchestration_executor, expected_type=type_hints["custom_orchestration_executor"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument existing_role", value=existing_role, expected_type=type_hints["existing_role"])
            check_type(argname="argument force_delete", value=force_delete, expected_type=type_hints["force_delete"])
            check_type(argname="argument idle_session_ttl", value=idle_session_ttl, expected_type=type_hints["idle_session_ttl"])
            check_type(argname="argument kms_key", value=kms_key, expected_type=type_hints["kms_key"])
            check_type(argname="argument memory", value=memory, expected_type=type_hints["memory"])
            check_type(argname="argument prompt_override_configuration", value=prompt_override_configuration, expected_type=type_hints["prompt_override_configuration"])
            check_type(argname="argument should_prepare_agent", value=should_prepare_agent, expected_type=type_hints["should_prepare_agent"])
            check_type(argname="argument user_input_enabled", value=user_input_enabled, expected_type=type_hints["user_input_enabled"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "foundation_model": foundation_model,
            "instruction": instruction,
        }
        if action_groups is not None:
            self._values["action_groups"] = action_groups
        if agent_collaboration is not None:
            self._values["agent_collaboration"] = agent_collaboration
        if agent_name is not None:
            self._values["agent_name"] = agent_name
        if code_interpreter_enabled is not None:
            self._values["code_interpreter_enabled"] = code_interpreter_enabled
        if custom_orchestration_executor is not None:
            self._values["custom_orchestration_executor"] = custom_orchestration_executor
        if description is not None:
            self._values["description"] = description
        if existing_role is not None:
            self._values["existing_role"] = existing_role
        if force_delete is not None:
            self._values["force_delete"] = force_delete
        if idle_session_ttl is not None:
            self._values["idle_session_ttl"] = idle_session_ttl
        if kms_key is not None:
            self._values["kms_key"] = kms_key
        if memory is not None:
            self._values["memory"] = memory
        if prompt_override_configuration is not None:
            self._values["prompt_override_configuration"] = prompt_override_configuration
        if should_prepare_agent is not None:
            self._values["should_prepare_agent"] = should_prepare_agent
        if user_input_enabled is not None:
            self._values["user_input_enabled"] = user_input_enabled

    @builtins.property
    def foundation_model(self) -> "IBedrockInvokable":
        '''(experimental) The foundation model used for orchestration by the agent.

        :stability: experimental
        '''
        result = self._values.get("foundation_model")
        assert result is not None, "Required property 'foundation_model' is missing"
        return typing.cast("IBedrockInvokable", result)

    @builtins.property
    def instruction(self) -> builtins.str:
        '''(experimental) The instruction used by the agent.

        This determines how the agent will perform his task.
        This instruction must have a minimum of 40 characters.

        :stability: experimental
        '''
        result = self._values.get("instruction")
        assert result is not None, "Required property 'instruction' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def action_groups(self) -> typing.Optional[typing.List[AgentActionGroup]]:
        '''(experimental) The Action Groups associated with the agent.

        :default: - Only default action groups (UserInput and CodeInterpreter) are added

        :stability: experimental
        '''
        result = self._values.get("action_groups")
        return typing.cast(typing.Optional[typing.List[AgentActionGroup]], result)

    @builtins.property
    def agent_collaboration(self) -> typing.Optional[AgentCollaboration]:
        '''(experimental) Configuration for agent collaboration settings, including AgentCollaboratorType and AgentCollaborators.

        This property allows you to define how the agent collaborates with other agents
        and what collaborators it can work with.

        :default: - No agent collaboration configuration.

        :stability: experimental
        '''
        result = self._values.get("agent_collaboration")
        return typing.cast(typing.Optional[AgentCollaboration], result)

    @builtins.property
    def agent_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the agent.

        This will be used as the physical name of the agent.

        :default:

        - A name is generated by CDK.
        Supported pattern : ^([0-9a-zA-Z][_-]?){1,100}$

        :stability: experimental
        '''
        result = self._values.get("agent_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def code_interpreter_enabled(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Select whether the agent can generate, run, and troubleshoot code when trying to complete a task.

        :default: - false

        :stability: experimental
        '''
        result = self._values.get("code_interpreter_enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def custom_orchestration_executor(
        self,
    ) -> typing.Optional["CustomOrchestrationExecutor"]:
        '''(experimental) The Lambda function to use for custom orchestration.

        If provided, custom orchestration will be used.
        If not provided, default orchestration will be used.

        :default: - Default orchestration

        :stability: experimental
        '''
        result = self._values.get("custom_orchestration_executor")
        return typing.cast(typing.Optional["CustomOrchestrationExecutor"], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(experimental) A description of the agent.

        :default: - No description is provided.

        :stability: experimental
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def existing_role(self) -> typing.Optional[_aws_cdk_aws_iam_ceddda9d.IRole]:
        '''(experimental) An existing IAM Role to associate with this agent.

        Use this property when you want to reuse an existing IAM role rather than create a new one.
        The role must have a trust policy that allows the Bedrock service to assume it.

        :default: - A new role is created for you.

        :stability: experimental
        '''
        result = self._values.get("existing_role")
        return typing.cast(typing.Optional[_aws_cdk_aws_iam_ceddda9d.IRole], result)

    @builtins.property
    def force_delete(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to delete the resource even if it's in use.

        :default: - false

        :stability: experimental
        '''
        result = self._values.get("force_delete")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def idle_session_ttl(self) -> typing.Optional[_aws_cdk_ceddda9d.Duration]:
        '''(experimental) How long sessions should be kept open for the agent.

        If no conversation occurs
        during this time, the session expires and Amazon Bedrock deletes any data
        provided before the timeout.

        :default: - 10 minutes

        :stability: experimental
        '''
        result = self._values.get("idle_session_ttl")
        return typing.cast(typing.Optional[_aws_cdk_ceddda9d.Duration], result)

    @builtins.property
    def kms_key(self) -> typing.Optional[_aws_cdk_aws_kms_ceddda9d.IKey]:
        '''(experimental) The KMS key of the agent if custom encryption is configured.

        :default: - An AWS managed key is used.

        :stability: experimental
        '''
        result = self._values.get("kms_key")
        return typing.cast(typing.Optional[_aws_cdk_aws_kms_ceddda9d.IKey], result)

    @builtins.property
    def memory(self) -> typing.Optional["Memory"]:
        '''(experimental) The type and configuration of the memory to maintain context across multiple sessions and recall past interactions.

        This can be useful for maintaining continuity in multi-turn conversations and recalling user preferences
        or past interactions.

        :default: - No memory will be used. Agents will retain context from the current session only.

        :see: https://docs.aws.amazon.com/bedrock/latest/userguide/agents-memory.html
        :stability: experimental
        '''
        result = self._values.get("memory")
        return typing.cast(typing.Optional["Memory"], result)

    @builtins.property
    def prompt_override_configuration(
        self,
    ) -> typing.Optional["PromptOverrideConfiguration"]:
        '''(experimental) Overrides some prompt templates in different parts of an agent sequence configuration.

        :default: - No overrides are provided.

        :stability: experimental
        '''
        result = self._values.get("prompt_override_configuration")
        return typing.cast(typing.Optional["PromptOverrideConfiguration"], result)

    @builtins.property
    def should_prepare_agent(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether to automatically update the ``DRAFT`` version of the agent after making changes to the agent.

        The ``DRAFT`` version can be continually iterated
        upon during internal development.

        :default: - false

        :stability: experimental
        '''
        result = self._values.get("should_prepare_agent")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def user_input_enabled(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Select whether the agent can prompt additional information from the user when it does not have enough information to respond to an utterance.

        :default: - false

        :stability: experimental
        '''
        result = self._values.get("user_input_enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AgentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@aws-cdk/aws-bedrock-alpha.AgentStepType")
class AgentStepType(enum.Enum):
    '''(experimental) The step in the agent sequence that this prompt configuration applies to.

    :stability: experimental
    :exampleMetadata: fixture=default infused

    Example::

        agent = bedrock.Agent(self, "Agent",
            foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
            instruction="You are a helpful assistant.",
            prompt_override_configuration=bedrock.PromptOverrideConfiguration.from_steps([
                step_type=bedrock.AgentStepType.ROUTING_CLASSIFIER,
                step_enabled=True,
                custom_prompt_template="Your routing template here",
                foundation_model=bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_V2
        
            ])
        )
    '''

    PRE_PROCESSING = "PRE_PROCESSING"
    '''(experimental) Pre-processing step that prepares the user input for orchestration.

    :stability: experimental
    '''
    ORCHESTRATION = "ORCHESTRATION"
    '''(experimental) Main orchestration step that determines the agent's actions.

    :stability: experimental
    '''
    POST_PROCESSING = "POST_PROCESSING"
    '''(experimental) Post-processing step that refines the agent's response.

    :stability: experimental
    '''
    ROUTING_CLASSIFIER = "ROUTING_CLASSIFIER"
    '''(experimental) Step that classifies and routes requests to appropriate collaborators.

    :stability: experimental
    '''
    MEMORY_SUMMARIZATION = "MEMORY_SUMMARIZATION"
    '''(experimental) Step that summarizes conversation history for memory retention.

    :stability: experimental
    '''
    KNOWLEDGE_BASE_RESPONSE_GENERATION = "KNOWLEDGE_BASE_RESPONSE_GENERATION"
    '''(experimental) Step that generates responses using knowledge base content.

    :stability: experimental
    '''


class ApiSchema(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@aws-cdk/aws-bedrock-alpha.ApiSchema",
):
    '''(experimental) Represents the concept of an API Schema for a Bedrock Agent Action Group.

    :stability: experimental
    :exampleMetadata: fixture=default infused

    Example::

        bucket = s3.Bucket.from_bucket_name(self, "ExistingBucket", "my-schema-bucket")
        s3_schema = bedrock.ApiSchema.from_s3_file(bucket, "schemas/action-group.yaml")
        
        action_group_function = lambda_.Function(self, "ActionGroupFunction",
            runtime=lambda_.Runtime.PYTHON_3_12,
            handler="index.handler",
            code=lambda_.Code.from_asset(path.join(__dirname, "../lambda/action-group"))
        )
        
        action_group = bedrock.AgentActionGroup(
            name="query-library",
            description="Use these functions to get information about the books in the library.",
            executor=bedrock.ActionGroupExecutor.from_lambda(action_group_function),
            enabled=True,
            api_schema=s3_schema
        )
        
        agent = bedrock.Agent(self, "Agent",
            foundation_model=bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_HAIKU_V1_0,
            instruction="You are a helpful and friendly agent that answers questions about literature."
        )
        
        agent.add_action_group(action_group)
    '''

    def __init__(
        self,
        s3_file: typing.Optional[typing.Union[_aws_cdk_aws_s3_ceddda9d.Location, typing.Dict[builtins.str, typing.Any]]] = None,
        inline_schema: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param s3_file: -
        :param inline_schema: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fef98095a98b1c3218daff037e79d43c41de797ec47d848bf1caa44f27c10abf)
            check_type(argname="argument s3_file", value=s3_file, expected_type=type_hints["s3_file"])
            check_type(argname="argument inline_schema", value=inline_schema, expected_type=type_hints["inline_schema"])
        jsii.create(self.__class__, self, [s3_file, inline_schema])

    @jsii.member(jsii_name="fromInline")
    @builtins.classmethod
    def from_inline(cls, schema: builtins.str) -> "InlineApiSchema":
        '''(experimental) Creates an API Schema from an inline string.

        :param schema: - the JSON or YAML payload defining the OpenAPI schema for the action group.

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ea650d724bddc459b33660757cb186cf804ba8ff34e5149c2ba116012bac3d13)
            check_type(argname="argument schema", value=schema, expected_type=type_hints["schema"])
        return typing.cast("InlineApiSchema", jsii.sinvoke(cls, "fromInline", [schema]))

    @jsii.member(jsii_name="fromLocalAsset")
    @builtins.classmethod
    def from_local_asset(cls, path: builtins.str) -> "AssetApiSchema":
        '''(experimental) Creates an API Schema from a local file.

        :param path: - the path to the local file containing the OpenAPI schema for the action group.

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1ef4f4f3b1ccc0ba1ca114b89046ebec72d8562e37d54d5ce5a48fb13bdb98e5)
            check_type(argname="argument path", value=path, expected_type=type_hints["path"])
        return typing.cast("AssetApiSchema", jsii.sinvoke(cls, "fromLocalAsset", [path]))

    @jsii.member(jsii_name="fromS3File")
    @builtins.classmethod
    def from_s3_file(
        cls,
        bucket: _aws_cdk_aws_s3_ceddda9d.IBucket,
        object_key: builtins.str,
    ) -> "S3ApiSchema":
        '''(experimental) Creates an API Schema from an S3 File.

        :param bucket: - the bucket containing the local file containing the OpenAPI schema for the action group.
        :param object_key: - object key in the bucket.

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__882387202413f33bb2e0f658215337e89d82d8bab90973056398f7bc7f939f52)
            check_type(argname="argument bucket", value=bucket, expected_type=type_hints["bucket"])
            check_type(argname="argument object_key", value=object_key, expected_type=type_hints["object_key"])
        return typing.cast("S3ApiSchema", jsii.sinvoke(cls, "fromS3File", [bucket, object_key]))

    @builtins.property
    @jsii.member(jsii_name="inlineSchema")
    def inline_schema(self) -> typing.Optional[builtins.str]:
        '''(experimental) The inline OpenAPI schema definition as a string, if using an inline schema.

        Can be in JSON or YAML format.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "inlineSchema"))

    @builtins.property
    @jsii.member(jsii_name="s3File")
    def s3_file(self) -> typing.Optional[_aws_cdk_aws_s3_ceddda9d.Location]:
        '''(experimental) The S3 location of the API schema file, if using an S3-based schema.

        Contains the bucket name and object key information.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[_aws_cdk_aws_s3_ceddda9d.Location], jsii.get(self, "s3File"))


class _ApiSchemaProxy(ApiSchema):
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, ApiSchema).__jsii_proxy_class__ = lambda : _ApiSchemaProxy


class AssetApiSchema(
    ApiSchema,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-bedrock-alpha.AssetApiSchema",
):
    '''(experimental) API Schema from a local asset.

    The asset is uploaded to an S3 staging bucket, then moved to its final location
    by CloudFormation during deployment.

    :stability: experimental
    :exampleMetadata: fixture=default infused

    Example::

        action_group_function = lambda_.Function(self, "ActionGroupFunction",
            runtime=lambda_.Runtime.PYTHON_3_12,
            handler="index.handler",
            code=lambda_.Code.from_asset(path.join(__dirname, "../lambda/action-group"))
        )
        
        # When using ApiSchema.fromLocalAsset, you must bind the schema to a scope
        schema = bedrock.ApiSchema.from_local_asset(path.join(__dirname, "action-group.yaml"))
        schema.bind(self)
        
        action_group = bedrock.AgentActionGroup(
            name="query-library",
            description="Use these functions to get information about the books in the library.",
            executor=bedrock.ActionGroupExecutor.from_lambda(action_group_function),
            enabled=True,
            api_schema=schema
        )
        
        agent = bedrock.Agent(self, "Agent",
            foundation_model=bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_HAIKU_V1_0,
            instruction="You are a helpful and friendly agent that answers questions about literature."
        )
        
        agent.add_action_group(action_group)
    '''

    def __init__(
        self,
        path: builtins.str,
        *,
        deploy_time: typing.Optional[builtins.bool] = None,
        display_name: typing.Optional[builtins.str] = None,
        readers: typing.Optional[typing.Sequence[_aws_cdk_aws_iam_ceddda9d.IGrantable]] = None,
        source_kms_key: typing.Optional[_aws_cdk_aws_kms_ceddda9d.IKey] = None,
        asset_hash: typing.Optional[builtins.str] = None,
        asset_hash_type: typing.Optional[_aws_cdk_ceddda9d.AssetHashType] = None,
        bundling: typing.Optional[typing.Union[_aws_cdk_ceddda9d.BundlingOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow_symlinks: typing.Optional[_aws_cdk_ceddda9d.SymlinkFollowMode] = None,
        ignore_mode: typing.Optional[_aws_cdk_ceddda9d.IgnoreMode] = None,
    ) -> None:
        '''
        :param path: -
        :param deploy_time: Whether or not the asset needs to exist beyond deployment time; i.e. are copied over to a different location and not needed afterwards. Setting this property to true has an impact on the lifecycle of the asset, because we will assume that it is safe to delete after the CloudFormation deployment succeeds. For example, Lambda Function assets are copied over to Lambda during deployment. Therefore, it is not necessary to store the asset in S3, so we consider those deployTime assets. Default: false
        :param display_name: A display name for this asset. If supplied, the display name will be used in locations where the asset identifier is printed, like in the CLI progress information. If the same asset is added multiple times, the display name of the first occurrence is used. The default is the construct path of the Asset construct, with respect to the enclosing stack. If the asset is produced by a construct helper function (such as ``lambda.Code.fromAsset()``), this will look like ``MyFunction/Code``. We use the stack-relative construct path so that in the common case where you have multiple stacks with the same asset, we won't show something like ``/MyBetaStack/MyFunction/Code`` when you are actually deploying to production. Default: - Stack-relative construct path
        :param readers: A list of principals that should be able to read this asset from S3. You can use ``asset.grantRead(principal)`` to grant read permissions later. Default: - No principals that can read file asset.
        :param source_kms_key: The ARN of the KMS key used to encrypt the handler code. Default: - the default server-side encryption with Amazon S3 managed keys(SSE-S3) key will be used.
        :param asset_hash: Specify a custom hash for this asset. If ``assetHashType`` is set it must be set to ``AssetHashType.CUSTOM``. For consistency, this custom hash will be SHA256 hashed and encoded as hex. The resulting hash will be the asset hash. NOTE: the hash is used in order to identify a specific revision of the asset, and used for optimizing and caching deployment activities related to this asset such as packaging, uploading to Amazon S3, etc. If you chose to customize the hash, you will need to make sure it is updated every time the asset changes, or otherwise it is possible that some deployments will not be invalidated. Default: - based on ``assetHashType``
        :param asset_hash_type: Specifies the type of hash to calculate for this asset. If ``assetHash`` is configured, this option must be ``undefined`` or ``AssetHashType.CUSTOM``. Default: - the default is ``AssetHashType.SOURCE``, but if ``assetHash`` is explicitly specified this value defaults to ``AssetHashType.CUSTOM``.
        :param bundling: Bundle the asset by executing a command in a Docker container or a custom bundling provider. The asset path will be mounted at ``/asset-input``. The Docker container is responsible for putting content at ``/asset-output``. The content at ``/asset-output`` will be zipped and used as the final asset. Default: - uploaded as-is to S3 if the asset is a regular file or a .zip file, archived into a .zip file and uploaded to S3 otherwise
        :param exclude: File paths matching the patterns will be excluded. See ``ignoreMode`` to set the matching behavior. Has no effect on Assets bundled using the ``bundling`` property. Default: - nothing is excluded
        :param follow_symlinks: A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER
        :param ignore_mode: The ignore behavior to use for ``exclude`` patterns. Default: IgnoreMode.GLOB

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__aee3c625b88d3cf9715cbd94ca9cf4c640ffe4007031b066fcea7749d1fd3b20)
            check_type(argname="argument path", value=path, expected_type=type_hints["path"])
        options = _aws_cdk_aws_s3_assets_ceddda9d.AssetOptions(
            deploy_time=deploy_time,
            display_name=display_name,
            readers=readers,
            source_kms_key=source_kms_key,
            asset_hash=asset_hash,
            asset_hash_type=asset_hash_type,
            bundling=bundling,
            exclude=exclude,
            follow_symlinks=follow_symlinks,
            ignore_mode=ignore_mode,
        )

        jsii.create(self.__class__, self, [path, options])

    @jsii.member(jsii_name="bind")
    def bind(self, scope: _constructs_77d1e7e8.Construct) -> None:
        '''(experimental) Binds this API schema to a construct scope.

        This method initializes the S3 asset if it hasn't been initialized yet.
        Must be called before rendering the schema as CFN properties.

        :param scope: - The construct scope to bind to.

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e42433d3c2e0a249ee91a1f011a13449ed0bdad079ab434f1317b403ad25b8e7)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        return typing.cast(None, jsii.invoke(self, "bind", [scope]))


@jsii.data_type(
    jsii_type="@aws-cdk/aws-bedrock-alpha.BedrockFoundationModelProps",
    jsii_struct_bases=[],
    name_mapping={
        "legacy": "legacy",
        "optimized_for_agents": "optimizedForAgents",
        "supported_vector_type": "supportedVectorType",
        "supports_agents": "supportsAgents",
        "supports_cross_region": "supportsCrossRegion",
        "supports_knowledge_base": "supportsKnowledgeBase",
        "vector_dimensions": "vectorDimensions",
    },
)
class BedrockFoundationModelProps:
    def __init__(
        self,
        *,
        legacy: typing.Optional[builtins.bool] = None,
        optimized_for_agents: typing.Optional[builtins.bool] = None,
        supported_vector_type: typing.Optional[typing.Sequence["VectorType"]] = None,
        supports_agents: typing.Optional[builtins.bool] = None,
        supports_cross_region: typing.Optional[builtins.bool] = None,
        supports_knowledge_base: typing.Optional[builtins.bool] = None,
        vector_dimensions: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''(experimental) Properties for configuring a Bedrock Foundation Model.

        These properties define the model's capabilities and supported features.

        :param legacy: (experimental) https://docs.aws.amazon.com/bedrock/latest/userguide/model-lifecycle.html A version is marked Legacy when there is a more recent version which provides superior performance. Amazon Bedrock sets an EOL date for Legacy versions. Default: - false
        :param optimized_for_agents: (experimental) Currently, some of the offered models are optimized with prompts/parsers fine-tuned for integrating with the agents architecture. When true, the model has been specifically optimized for agent interactions. Default: - false
        :param supported_vector_type: (experimental) Embeddings models have different supported vector types. Defines whether the model supports floating-point or binary vectors. Default: - undefined
        :param supports_agents: (experimental) Bedrock Agents can use this model. When true, the model can be used with Bedrock Agents for automated task execution. Default: - false
        :param supports_cross_region: (experimental) Can be used with a Cross-Region Inference Profile. When true, the model supports inference across different AWS regions. Default: - false
        :param supports_knowledge_base: (experimental) Bedrock Knowledge Base can use this model. When true, the model can be used for knowledge base operations. Default: - false
        :param vector_dimensions: (experimental) Embedding models have different vector dimensions. Only applicable for embedding models. Defines the dimensionality of the vector embeddings. Default: - undefined

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_bedrock_alpha as bedrock_alpha
            
            bedrock_foundation_model_props = bedrock_alpha.BedrockFoundationModelProps(
                legacy=False,
                optimized_for_agents=False,
                supported_vector_type=[bedrock_alpha.VectorType.FLOATING_POINT],
                supports_agents=False,
                supports_cross_region=False,
                supports_knowledge_base=False,
                vector_dimensions=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9871e1b6209dedd6005dcce72b65ee8c40c6a25156e22de4c732060c640acf67)
            check_type(argname="argument legacy", value=legacy, expected_type=type_hints["legacy"])
            check_type(argname="argument optimized_for_agents", value=optimized_for_agents, expected_type=type_hints["optimized_for_agents"])
            check_type(argname="argument supported_vector_type", value=supported_vector_type, expected_type=type_hints["supported_vector_type"])
            check_type(argname="argument supports_agents", value=supports_agents, expected_type=type_hints["supports_agents"])
            check_type(argname="argument supports_cross_region", value=supports_cross_region, expected_type=type_hints["supports_cross_region"])
            check_type(argname="argument supports_knowledge_base", value=supports_knowledge_base, expected_type=type_hints["supports_knowledge_base"])
            check_type(argname="argument vector_dimensions", value=vector_dimensions, expected_type=type_hints["vector_dimensions"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if legacy is not None:
            self._values["legacy"] = legacy
        if optimized_for_agents is not None:
            self._values["optimized_for_agents"] = optimized_for_agents
        if supported_vector_type is not None:
            self._values["supported_vector_type"] = supported_vector_type
        if supports_agents is not None:
            self._values["supports_agents"] = supports_agents
        if supports_cross_region is not None:
            self._values["supports_cross_region"] = supports_cross_region
        if supports_knowledge_base is not None:
            self._values["supports_knowledge_base"] = supports_knowledge_base
        if vector_dimensions is not None:
            self._values["vector_dimensions"] = vector_dimensions

    @builtins.property
    def legacy(self) -> typing.Optional[builtins.bool]:
        '''(experimental) https://docs.aws.amazon.com/bedrock/latest/userguide/model-lifecycle.html A version is marked Legacy when there is a more recent version which provides superior performance. Amazon Bedrock sets an EOL date for Legacy versions.

        :default: - false

        :stability: experimental
        '''
        result = self._values.get("legacy")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def optimized_for_agents(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Currently, some of the offered models are optimized with prompts/parsers fine-tuned for integrating with the agents architecture.

        When true, the model has been specifically optimized for agent interactions.

        :default: - false

        :stability: experimental
        '''
        result = self._values.get("optimized_for_agents")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def supported_vector_type(self) -> typing.Optional[typing.List["VectorType"]]:
        '''(experimental) Embeddings models have different supported vector types.

        Defines whether the model supports floating-point or binary vectors.

        :default: - undefined

        :stability: experimental
        '''
        result = self._values.get("supported_vector_type")
        return typing.cast(typing.Optional[typing.List["VectorType"]], result)

    @builtins.property
    def supports_agents(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Bedrock Agents can use this model.

        When true, the model can be used with Bedrock Agents for automated task execution.

        :default: - false

        :stability: experimental
        '''
        result = self._values.get("supports_agents")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def supports_cross_region(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Can be used with a Cross-Region Inference Profile.

        When true, the model supports inference across different AWS regions.

        :default: - false

        :stability: experimental
        '''
        result = self._values.get("supports_cross_region")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def supports_knowledge_base(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Bedrock Knowledge Base can use this model.

        When true, the model can be used for knowledge base operations.

        :default: - false

        :stability: experimental
        '''
        result = self._values.get("supports_knowledge_base")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def vector_dimensions(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Embedding models have different vector dimensions.

        Only applicable for embedding models. Defines the dimensionality of the vector embeddings.

        :default: - undefined

        :stability: experimental
        '''
        result = self._values.get("vector_dimensions")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BedrockFoundationModelProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@aws-cdk/aws-bedrock-alpha.CustomControl")
class CustomControl(enum.Enum):
    '''(experimental) The type of custom control for the action group executor.

    :stability: experimental
    '''

    RETURN_CONTROL = "RETURN_CONTROL"
    '''(experimental) Returns the action group invocation results directly in the InvokeAgent response.

    :stability: experimental
    '''


class CustomOrchestrationExecutor(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-bedrock-alpha.CustomOrchestrationExecutor",
):
    '''(experimental) Contains details about the Lambda function containing the orchestration logic carried out upon invoking the custom orchestration.

    :stability: experimental
    :exampleMetadata: fixture=default infused

    Example::

        orchestration_function = lambda_.Function(self, "OrchestrationFunction",
            runtime=lambda_.Runtime.PYTHON_3_10,
            handler="index.handler",
            code=lambda_.Code.from_asset("lambda/orchestration")
        )
        
        agent = bedrock.Agent(self, "CustomOrchestrationAgent",
            instruction="You are a helpful assistant with custom orchestration logic.",
            foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
            custom_orchestration_executor=bedrock.CustomOrchestrationExecutor.from_lambda(orchestration_function)
        )
    '''

    @jsii.member(jsii_name="fromLambda")
    @builtins.classmethod
    def from_lambda(
        cls,
        lambda_function: _aws_cdk_aws_lambda_ceddda9d.IFunction,
    ) -> "CustomOrchestrationExecutor":
        '''(experimental) Defines an orchestration executor with a Lambda function containing the business logic.

        :param lambda_function: - Lambda function to be called by the orchestration.

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9581a8fe03e677bcf39692a4a671ac83b906583168b1efeb85160cd87ace6e24)
            check_type(argname="argument lambda_function", value=lambda_function, expected_type=type_hints["lambda_function"])
        return typing.cast("CustomOrchestrationExecutor", jsii.sinvoke(cls, "fromLambda", [lambda_function]))

    @builtins.property
    @jsii.member(jsii_name="lambdaFunction")
    def lambda_function(self) -> _aws_cdk_aws_lambda_ceddda9d.IFunction:
        '''(experimental) The Lambda function that contains the custom orchestration logic.

        This function is called when the agent needs to make decisions about action execution.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_lambda_ceddda9d.IFunction, jsii.get(self, "lambdaFunction"))

    @builtins.property
    @jsii.member(jsii_name="type")
    def type(self) -> "OrchestrationType":
        '''(experimental) The type of orchestration this executor performs.

        :stability: experimental
        '''
        return typing.cast("OrchestrationType", jsii.get(self, "type"))


@jsii.data_type(
    jsii_type="@aws-cdk/aws-bedrock-alpha.CustomParserProps",
    jsii_struct_bases=[],
    name_mapping={
        "knowledge_base_response_generation_step": "knowledgeBaseResponseGenerationStep",
        "memory_summarization_step": "memorySummarizationStep",
        "orchestration_step": "orchestrationStep",
        "parser": "parser",
        "post_processing_step": "postProcessingStep",
        "pre_processing_step": "preProcessingStep",
        "routing_classifier_step": "routingClassifierStep",
    },
)
class CustomParserProps:
    def __init__(
        self,
        *,
        knowledge_base_response_generation_step: typing.Optional[typing.Union["PromptKnowledgeBaseResponseGenerationConfigCustomParser", typing.Dict[builtins.str, typing.Any]]] = None,
        memory_summarization_step: typing.Optional[typing.Union["PromptMemorySummarizationConfigCustomParser", typing.Dict[builtins.str, typing.Any]]] = None,
        orchestration_step: typing.Optional[typing.Union["PromptOrchestrationConfigCustomParser", typing.Dict[builtins.str, typing.Any]]] = None,
        parser: typing.Optional[_aws_cdk_aws_lambda_ceddda9d.IFunction] = None,
        post_processing_step: typing.Optional[typing.Union["PromptPostProcessingConfigCustomParser", typing.Dict[builtins.str, typing.Any]]] = None,
        pre_processing_step: typing.Optional[typing.Union["PromptPreProcessingConfigCustomParser", typing.Dict[builtins.str, typing.Any]]] = None,
        routing_classifier_step: typing.Optional[typing.Union["PromptRoutingClassifierConfigCustomParser", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''(experimental) Properties for configuring a custom Lambda parser for prompt overrides.

        :param knowledge_base_response_generation_step: (experimental) Configuration for the knowledge base response generation step. Default: undefined - No knowledge base response generation configuration
        :param memory_summarization_step: (experimental) Configuration for the memory summarization step. Default: undefined - No memory summarization configuration
        :param orchestration_step: (experimental) Configuration for the orchestration step. Default: undefined - No orchestration configuration
        :param parser: (experimental) Lambda function to use as custom parser. Default: undefined - No custom parser is used
        :param post_processing_step: (experimental) Configuration for the post-processing step. Default: undefined - No post-processing configuration
        :param pre_processing_step: (experimental) Configuration for the pre-processing step. Default: undefined - No pre-processing configuration
        :param routing_classifier_step: (experimental) Configuration for the routing classifier step. Default: undefined - No routing classifier configuration

        :stability: experimental
        :exampleMetadata: fixture=default infused

        Example::

            parser_function = lambda_.Function(self, "ParserFunction",
                runtime=lambda_.Runtime.PYTHON_3_10,
                handler="index.handler",
                code=lambda_.Code.from_asset("lambda")
            )
            
            agent = bedrock.Agent(self, "Agent",
                foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
                instruction="You are a helpful assistant.",
                prompt_override_configuration=bedrock.PromptOverrideConfiguration.with_custom_parser(
                    parser=parser_function,
                    pre_processing_step=bedrock.PromptPreProcessingConfigCustomParser(
                        step_type=bedrock.AgentStepType.PRE_PROCESSING,
                        use_custom_parser=True
                    )
                )
            )
        '''
        if isinstance(knowledge_base_response_generation_step, dict):
            knowledge_base_response_generation_step = PromptKnowledgeBaseResponseGenerationConfigCustomParser(**knowledge_base_response_generation_step)
        if isinstance(memory_summarization_step, dict):
            memory_summarization_step = PromptMemorySummarizationConfigCustomParser(**memory_summarization_step)
        if isinstance(orchestration_step, dict):
            orchestration_step = PromptOrchestrationConfigCustomParser(**orchestration_step)
        if isinstance(post_processing_step, dict):
            post_processing_step = PromptPostProcessingConfigCustomParser(**post_processing_step)
        if isinstance(pre_processing_step, dict):
            pre_processing_step = PromptPreProcessingConfigCustomParser(**pre_processing_step)
        if isinstance(routing_classifier_step, dict):
            routing_classifier_step = PromptRoutingClassifierConfigCustomParser(**routing_classifier_step)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5e9247125ea57043b0c8014036fd32415bac33f25230256cde57ebee738d550b)
            check_type(argname="argument knowledge_base_response_generation_step", value=knowledge_base_response_generation_step, expected_type=type_hints["knowledge_base_response_generation_step"])
            check_type(argname="argument memory_summarization_step", value=memory_summarization_step, expected_type=type_hints["memory_summarization_step"])
            check_type(argname="argument orchestration_step", value=orchestration_step, expected_type=type_hints["orchestration_step"])
            check_type(argname="argument parser", value=parser, expected_type=type_hints["parser"])
            check_type(argname="argument post_processing_step", value=post_processing_step, expected_type=type_hints["post_processing_step"])
            check_type(argname="argument pre_processing_step", value=pre_processing_step, expected_type=type_hints["pre_processing_step"])
            check_type(argname="argument routing_classifier_step", value=routing_classifier_step, expected_type=type_hints["routing_classifier_step"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if knowledge_base_response_generation_step is not None:
            self._values["knowledge_base_response_generation_step"] = knowledge_base_response_generation_step
        if memory_summarization_step is not None:
            self._values["memory_summarization_step"] = memory_summarization_step
        if orchestration_step is not None:
            self._values["orchestration_step"] = orchestration_step
        if parser is not None:
            self._values["parser"] = parser
        if post_processing_step is not None:
            self._values["post_processing_step"] = post_processing_step
        if pre_processing_step is not None:
            self._values["pre_processing_step"] = pre_processing_step
        if routing_classifier_step is not None:
            self._values["routing_classifier_step"] = routing_classifier_step

    @builtins.property
    def knowledge_base_response_generation_step(
        self,
    ) -> typing.Optional["PromptKnowledgeBaseResponseGenerationConfigCustomParser"]:
        '''(experimental) Configuration for the knowledge base response generation step.

        :default: undefined - No knowledge base response generation configuration

        :stability: experimental
        '''
        result = self._values.get("knowledge_base_response_generation_step")
        return typing.cast(typing.Optional["PromptKnowledgeBaseResponseGenerationConfigCustomParser"], result)

    @builtins.property
    def memory_summarization_step(
        self,
    ) -> typing.Optional["PromptMemorySummarizationConfigCustomParser"]:
        '''(experimental) Configuration for the memory summarization step.

        :default: undefined - No memory summarization configuration

        :stability: experimental
        '''
        result = self._values.get("memory_summarization_step")
        return typing.cast(typing.Optional["PromptMemorySummarizationConfigCustomParser"], result)

    @builtins.property
    def orchestration_step(
        self,
    ) -> typing.Optional["PromptOrchestrationConfigCustomParser"]:
        '''(experimental) Configuration for the orchestration step.

        :default: undefined - No orchestration configuration

        :stability: experimental
        '''
        result = self._values.get("orchestration_step")
        return typing.cast(typing.Optional["PromptOrchestrationConfigCustomParser"], result)

    @builtins.property
    def parser(self) -> typing.Optional[_aws_cdk_aws_lambda_ceddda9d.IFunction]:
        '''(experimental) Lambda function to use as custom parser.

        :default: undefined - No custom parser is used

        :stability: experimental
        '''
        result = self._values.get("parser")
        return typing.cast(typing.Optional[_aws_cdk_aws_lambda_ceddda9d.IFunction], result)

    @builtins.property
    def post_processing_step(
        self,
    ) -> typing.Optional["PromptPostProcessingConfigCustomParser"]:
        '''(experimental) Configuration for the post-processing step.

        :default: undefined - No post-processing configuration

        :stability: experimental
        '''
        result = self._values.get("post_processing_step")
        return typing.cast(typing.Optional["PromptPostProcessingConfigCustomParser"], result)

    @builtins.property
    def pre_processing_step(
        self,
    ) -> typing.Optional["PromptPreProcessingConfigCustomParser"]:
        '''(experimental) Configuration for the pre-processing step.

        :default: undefined - No pre-processing configuration

        :stability: experimental
        '''
        result = self._values.get("pre_processing_step")
        return typing.cast(typing.Optional["PromptPreProcessingConfigCustomParser"], result)

    @builtins.property
    def routing_classifier_step(
        self,
    ) -> typing.Optional["PromptRoutingClassifierConfigCustomParser"]:
        '''(experimental) Configuration for the routing classifier step.

        :default: undefined - No routing classifier configuration

        :stability: experimental
        '''
        result = self._values.get("routing_classifier_step")
        return typing.cast(typing.Optional["PromptRoutingClassifierConfigCustomParser"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CustomParserProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class Function(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-bedrock-alpha.Function",
):
    '''(experimental) Represents a function in a function schema.

    :stability: experimental
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_bedrock_alpha as bedrock_alpha
        
        function_ = bedrock_alpha.Function(
            description="description",
            name="name",
        
            # the properties below are optional
            parameters={
                "parameters_key": bedrock_alpha.FunctionParameterProps(
                    type=bedrock_alpha.ParameterType.STRING,
        
                    # the properties below are optional
                    description="description",
                    required=False
                )
            },
            require_confirmation=bedrock_alpha.RequireConfirmation.ENABLED
        )
    '''

    def __init__(
        self,
        *,
        description: builtins.str,
        name: builtins.str,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Union["FunctionParameterProps", typing.Dict[builtins.str, typing.Any]]]] = None,
        require_confirmation: typing.Optional["RequireConfirmation"] = None,
    ) -> None:
        '''
        :param description: (experimental) Description of the function.
        :param name: (experimental) The name of the function.
        :param parameters: (experimental) Parameters for the function as a record of parameter name to parameter properties. Default: {}
        :param require_confirmation: (experimental) Whether to require confirmation before executing the function. Default: RequireConfirmation.DISABLED

        :stability: experimental
        '''
        props = FunctionProps(
            description=description,
            name=name,
            parameters=parameters,
            require_confirmation=require_confirmation,
        )

        jsii.create(self.__class__, self, [props])

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> builtins.str:
        '''(experimental) Description of the function.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "description"))

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        '''(experimental) The name of the function.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "name"))

    @builtins.property
    @jsii.member(jsii_name="parameters")
    def parameters(self) -> typing.Mapping[builtins.str, "FunctionParameter"]:
        '''(experimental) Parameters for the function.

        :stability: experimental
        '''
        return typing.cast(typing.Mapping[builtins.str, "FunctionParameter"], jsii.get(self, "parameters"))

    @builtins.property
    @jsii.member(jsii_name="requireConfirmation")
    def require_confirmation(self) -> "RequireConfirmation":
        '''(experimental) Whether to require confirmation before executing the function.

        :stability: experimental
        '''
        return typing.cast("RequireConfirmation", jsii.get(self, "requireConfirmation"))


class FunctionParameter(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-bedrock-alpha.FunctionParameter",
):
    '''(experimental) Represents a function parameter in a function schema.

    :stability: experimental
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_bedrock_alpha as bedrock_alpha
        
        function_parameter = bedrock_alpha.FunctionParameter(
            type=bedrock_alpha.ParameterType.STRING,
        
            # the properties below are optional
            description="description",
            required=False
        )
    '''

    def __init__(
        self,
        *,
        type: "ParameterType",
        description: typing.Optional[builtins.str] = None,
        required: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''
        :param type: (experimental) The type of the parameter.
        :param description: (experimental) Description of the parameter. Default: undefined no description will be present
        :param required: (experimental) Whether the parameter is required. Default: true

        :stability: experimental
        '''
        props = FunctionParameterProps(
            type=type, description=description, required=required
        )

        jsii.create(self.__class__, self, [props])

    @builtins.property
    @jsii.member(jsii_name="required")
    def required(self) -> builtins.bool:
        '''(experimental) Whether the parameter is required.

        :stability: experimental
        '''
        return typing.cast(builtins.bool, jsii.get(self, "required"))

    @builtins.property
    @jsii.member(jsii_name="type")
    def type(self) -> "ParameterType":
        '''(experimental) The type of the parameter.

        :stability: experimental
        '''
        return typing.cast("ParameterType", jsii.get(self, "type"))

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''(experimental) Description of the parameter.

        :default: undefined no description will be present

        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))


@jsii.data_type(
    jsii_type="@aws-cdk/aws-bedrock-alpha.FunctionParameterProps",
    jsii_struct_bases=[],
    name_mapping={
        "type": "type",
        "description": "description",
        "required": "required",
    },
)
class FunctionParameterProps:
    def __init__(
        self,
        *,
        type: "ParameterType",
        description: typing.Optional[builtins.str] = None,
        required: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''(experimental) Properties for a function parameter.

        :param type: (experimental) The type of the parameter.
        :param description: (experimental) Description of the parameter. Default: undefined no description will be present
        :param required: (experimental) Whether the parameter is required. Default: true

        :stability: experimental
        :exampleMetadata: fixture=default infused

        Example::

            action_group_function = lambda_.Function(self, "ActionGroupFunction",
                runtime=lambda_.Runtime.PYTHON_3_12,
                handler="index.handler",
                code=lambda_.Code.from_asset(path.join(__dirname, "../lambda/action-group"))
            )
            
            # Define a function schema with parameters
            function_schema = bedrock.FunctionSchema(
                functions=[bedrock.FunctionProps(
                    name="searchBooks",
                    description="Search for books in the library catalog",
                    parameters={
                        "query": bedrock.FunctionParameterProps(
                            type=bedrock.ParameterType.STRING,
                            required=True,
                            description="The search query string"
                        ),
                        "maxResults": bedrock.FunctionParameterProps(
                            type=bedrock.ParameterType.INTEGER,
                            required=False,
                            description="Maximum number of results to return"
                        ),
                        "includeOutOfPrint": bedrock.FunctionParameterProps(
                            type=bedrock.ParameterType.BOOLEAN,
                            required=False,
                            description="Whether to include out-of-print books"
                        )
                    },
                    require_confirmation=bedrock.RequireConfirmation.DISABLED
                ), bedrock.FunctionProps(
                    name="getBookDetails",
                    description="Get detailed information about a specific book",
                    parameters={
                        "bookId": bedrock.FunctionParameterProps(
                            type=bedrock.ParameterType.STRING,
                            required=True,
                            description="The unique identifier of the book"
                        )
                    },
                    require_confirmation=bedrock.RequireConfirmation.ENABLED
                )
                ]
            )
            
            # Create an action group using the function schema
            action_group = bedrock.AgentActionGroup(
                name="library-functions",
                description="Functions for interacting with the library catalog",
                executor=bedrock.ActionGroupExecutor.from_lambda(action_group_function),
                function_schema=function_schema,
                enabled=True
            )
            
            agent = bedrock.Agent(self, "Agent",
                foundation_model=bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_HAIKU_V1_0,
                instruction="You are a helpful and friendly agent that answers questions about literature.",
                action_groups=[action_group]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__afff183f47ab9cba063848fa5cc915f6deaad94e1aaeca1688be3a79be4bf746)
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument required", value=required, expected_type=type_hints["required"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "type": type,
        }
        if description is not None:
            self._values["description"] = description
        if required is not None:
            self._values["required"] = required

    @builtins.property
    def type(self) -> "ParameterType":
        '''(experimental) The type of the parameter.

        :stability: experimental
        '''
        result = self._values.get("type")
        assert result is not None, "Required property 'type' is missing"
        return typing.cast("ParameterType", result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(experimental) Description of the parameter.

        :default: undefined no description will be present

        :stability: experimental
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def required(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether the parameter is required.

        :default: true

        :stability: experimental
        '''
        result = self._values.get("required")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FunctionParameterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-bedrock-alpha.FunctionProps",
    jsii_struct_bases=[],
    name_mapping={
        "description": "description",
        "name": "name",
        "parameters": "parameters",
        "require_confirmation": "requireConfirmation",
    },
)
class FunctionProps:
    def __init__(
        self,
        *,
        description: builtins.str,
        name: builtins.str,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Union[FunctionParameterProps, typing.Dict[builtins.str, typing.Any]]]] = None,
        require_confirmation: typing.Optional["RequireConfirmation"] = None,
    ) -> None:
        '''(experimental) Properties for a function in a function schema.

        :param description: (experimental) Description of the function.
        :param name: (experimental) The name of the function.
        :param parameters: (experimental) Parameters for the function as a record of parameter name to parameter properties. Default: {}
        :param require_confirmation: (experimental) Whether to require confirmation before executing the function. Default: RequireConfirmation.DISABLED

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_bedrock_alpha as bedrock_alpha
            
            function_props = bedrock_alpha.FunctionProps(
                description="description",
                name="name",
            
                # the properties below are optional
                parameters={
                    "parameters_key": bedrock_alpha.FunctionParameterProps(
                        type=bedrock_alpha.ParameterType.STRING,
            
                        # the properties below are optional
                        description="description",
                        required=False
                    )
                },
                require_confirmation=bedrock_alpha.RequireConfirmation.ENABLED
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d6a1e6361e2edbbe58148c14c19a168946ef6213e0d123499089a4f3e48f6a0d)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument parameters", value=parameters, expected_type=type_hints["parameters"])
            check_type(argname="argument require_confirmation", value=require_confirmation, expected_type=type_hints["require_confirmation"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "description": description,
            "name": name,
        }
        if parameters is not None:
            self._values["parameters"] = parameters
        if require_confirmation is not None:
            self._values["require_confirmation"] = require_confirmation

    @builtins.property
    def description(self) -> builtins.str:
        '''(experimental) Description of the function.

        :stability: experimental
        '''
        result = self._values.get("description")
        assert result is not None, "Required property 'description' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def name(self) -> builtins.str:
        '''(experimental) The name of the function.

        :stability: experimental
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def parameters(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, FunctionParameterProps]]:
        '''(experimental) Parameters for the function as a record of parameter name to parameter properties.

        :default: {}

        :stability: experimental
        '''
        result = self._values.get("parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, FunctionParameterProps]], result)

    @builtins.property
    def require_confirmation(self) -> typing.Optional["RequireConfirmation"]:
        '''(experimental) Whether to require confirmation before executing the function.

        :default: RequireConfirmation.DISABLED

        :stability: experimental
        '''
        result = self._values.get("require_confirmation")
        return typing.cast(typing.Optional["RequireConfirmation"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FunctionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class FunctionSchema(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-bedrock-alpha.FunctionSchema",
):
    '''(experimental) Represents a function schema for a Bedrock Agent Action Group.

    :stability: experimental
    :exampleMetadata: fixture=default infused

    Example::

        action_group_function = lambda_.Function(self, "ActionGroupFunction",
            runtime=lambda_.Runtime.PYTHON_3_12,
            handler="index.handler",
            code=lambda_.Code.from_asset(path.join(__dirname, "../lambda/action-group"))
        )
        
        # Define a function schema with parameters
        function_schema = bedrock.FunctionSchema(
            functions=[bedrock.FunctionProps(
                name="searchBooks",
                description="Search for books in the library catalog",
                parameters={
                    "query": bedrock.FunctionParameterProps(
                        type=bedrock.ParameterType.STRING,
                        required=True,
                        description="The search query string"
                    ),
                    "maxResults": bedrock.FunctionParameterProps(
                        type=bedrock.ParameterType.INTEGER,
                        required=False,
                        description="Maximum number of results to return"
                    ),
                    "includeOutOfPrint": bedrock.FunctionParameterProps(
                        type=bedrock.ParameterType.BOOLEAN,
                        required=False,
                        description="Whether to include out-of-print books"
                    )
                },
                require_confirmation=bedrock.RequireConfirmation.DISABLED
            ), bedrock.FunctionProps(
                name="getBookDetails",
                description="Get detailed information about a specific book",
                parameters={
                    "bookId": bedrock.FunctionParameterProps(
                        type=bedrock.ParameterType.STRING,
                        required=True,
                        description="The unique identifier of the book"
                    )
                },
                require_confirmation=bedrock.RequireConfirmation.ENABLED
            )
            ]
        )
        
        # Create an action group using the function schema
        action_group = bedrock.AgentActionGroup(
            name="library-functions",
            description="Functions for interacting with the library catalog",
            executor=bedrock.ActionGroupExecutor.from_lambda(action_group_function),
            function_schema=function_schema,
            enabled=True
        )
        
        agent = bedrock.Agent(self, "Agent",
            foundation_model=bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_HAIKU_V1_0,
            instruction="You are a helpful and friendly agent that answers questions about literature.",
            action_groups=[action_group]
        )
    '''

    def __init__(
        self,
        *,
        functions: typing.Sequence[typing.Union[FunctionProps, typing.Dict[builtins.str, typing.Any]]],
    ) -> None:
        '''
        :param functions: (experimental) Functions defined in the schema.

        :stability: experimental
        '''
        props = FunctionSchemaProps(functions=functions)

        jsii.create(self.__class__, self, [props])

    @builtins.property
    @jsii.member(jsii_name="functions")
    def functions(self) -> typing.List[Function]:
        '''(experimental) The functions defined in the schema.

        :stability: experimental
        '''
        return typing.cast(typing.List[Function], jsii.get(self, "functions"))


@jsii.data_type(
    jsii_type="@aws-cdk/aws-bedrock-alpha.FunctionSchemaProps",
    jsii_struct_bases=[],
    name_mapping={"functions": "functions"},
)
class FunctionSchemaProps:
    def __init__(
        self,
        *,
        functions: typing.Sequence[typing.Union[FunctionProps, typing.Dict[builtins.str, typing.Any]]],
    ) -> None:
        '''(experimental) Properties for a function schema.

        :param functions: (experimental) Functions defined in the schema.

        :stability: experimental
        :exampleMetadata: fixture=default infused

        Example::

            action_group_function = lambda_.Function(self, "ActionGroupFunction",
                runtime=lambda_.Runtime.PYTHON_3_12,
                handler="index.handler",
                code=lambda_.Code.from_asset(path.join(__dirname, "../lambda/action-group"))
            )
            
            # Define a function schema with parameters
            function_schema = bedrock.FunctionSchema(
                functions=[bedrock.FunctionProps(
                    name="searchBooks",
                    description="Search for books in the library catalog",
                    parameters={
                        "query": bedrock.FunctionParameterProps(
                            type=bedrock.ParameterType.STRING,
                            required=True,
                            description="The search query string"
                        ),
                        "maxResults": bedrock.FunctionParameterProps(
                            type=bedrock.ParameterType.INTEGER,
                            required=False,
                            description="Maximum number of results to return"
                        ),
                        "includeOutOfPrint": bedrock.FunctionParameterProps(
                            type=bedrock.ParameterType.BOOLEAN,
                            required=False,
                            description="Whether to include out-of-print books"
                        )
                    },
                    require_confirmation=bedrock.RequireConfirmation.DISABLED
                ), bedrock.FunctionProps(
                    name="getBookDetails",
                    description="Get detailed information about a specific book",
                    parameters={
                        "bookId": bedrock.FunctionParameterProps(
                            type=bedrock.ParameterType.STRING,
                            required=True,
                            description="The unique identifier of the book"
                        )
                    },
                    require_confirmation=bedrock.RequireConfirmation.ENABLED
                )
                ]
            )
            
            # Create an action group using the function schema
            action_group = bedrock.AgentActionGroup(
                name="library-functions",
                description="Functions for interacting with the library catalog",
                executor=bedrock.ActionGroupExecutor.from_lambda(action_group_function),
                function_schema=function_schema,
                enabled=True
            )
            
            agent = bedrock.Agent(self, "Agent",
                foundation_model=bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_HAIKU_V1_0,
                instruction="You are a helpful and friendly agent that answers questions about literature.",
                action_groups=[action_group]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__64dfdaad570b5bc4904b9d9f356eb416095866cc4cca0546c2bb37dbb844e916)
            check_type(argname="argument functions", value=functions, expected_type=type_hints["functions"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "functions": functions,
        }

    @builtins.property
    def functions(self) -> typing.List[FunctionProps]:
        '''(experimental) Functions defined in the schema.

        :stability: experimental
        '''
        result = self._values.get("functions")
        assert result is not None, "Required property 'functions' is missing"
        return typing.cast(typing.List[FunctionProps], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FunctionSchemaProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.interface(jsii_type="@aws-cdk/aws-bedrock-alpha.IAgent")
class IAgent(
    _aws_cdk_ceddda9d.IResource,
    _aws_cdk_aws_iam_ceddda9d.IGrantable,
    typing_extensions.Protocol,
):
    '''(experimental) Represents an Agent, either created with CDK or imported.

    :stability: experimental
    '''

    @builtins.property
    @jsii.member(jsii_name="agentArn")
    def agent_arn(self) -> builtins.str:
        '''(experimental) The ARN of the agent.

        :stability: experimental
        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="agentId")
    def agent_id(self) -> builtins.str:
        '''(experimental) The ID of the Agent.

        :stability: experimental
        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="role")
    def role(self) -> _aws_cdk_aws_iam_ceddda9d.IRole:
        '''(experimental) The IAM role associated to the agent.

        :stability: experimental
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="kmsKey")
    def kms_key(self) -> typing.Optional[_aws_cdk_aws_kms_ceddda9d.IKey]:
        '''(experimental) Optional KMS encryption key associated with this agent.

        :stability: experimental
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="lastUpdated")
    def last_updated(self) -> typing.Optional[builtins.str]:
        '''(experimental) When this agent was last updated.

        :stability: experimental
        :attribute: true
        '''
        ...

    @jsii.member(jsii_name="grantInvoke")
    def grant_invoke(
        self,
        grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
    ) -> _aws_cdk_aws_iam_ceddda9d.Grant:
        '''(experimental) Grant invoke permissions on this agent to an IAM principal.

        Note: This grant will only work when the grantee is in the same AWS account
        where the agent is defined. Cross-account invocation is not supported.

        :param grantee: -

        :stability: experimental
        '''
        ...

    @jsii.member(jsii_name="metricCount")
    def metric_count(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions_map: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        id: typing.Optional[builtins.str] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[_aws_cdk_ceddda9d.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        stack_account: typing.Optional[builtins.str] = None,
        stack_region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[_aws_cdk_aws_cloudwatch_ceddda9d.Unit] = None,
        visible: typing.Optional[builtins.bool] = None,
    ) -> _aws_cdk_aws_cloudwatch_ceddda9d.Metric:
        '''(experimental) Return the CloudWatch metric for agent count.

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions_map: Dimensions of the metric. Default: - No dimensions.
        :param id: Unique identifier for this metric when used in dashboard widgets. The id can be used as a variable to represent this metric in math expressions. Valid characters are letters, numbers, and underscore. The first character must be a lowercase letter. Default: - No ID
        :param label: Label for this metric when added to a Graph in a Dashboard. You can use `dynamic labels <https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/graph-dynamic-labels.html>`_ to show summary information about the entire displayed time series in the legend. For example, if you use:: [max: ${MAX}] MyMetric As the metric label, the maximum value in the visible range will be shown next to the time series name in the graph's legend. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param stack_account: Account of the stack this metric is attached to. Default: - Deployment account.
        :param stack_region: Region of the stack this metric is attached to. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Use the ``aws_cloudwatch.Stats`` helper class to construct valid input strings. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" - "tmNN.NN" | "tm(NN.NN%:NN.NN%)" - "iqm" - "wmNN.NN" | "wm(NN.NN%:NN.NN%)" - "tcNN.NN" | "tc(NN.NN%:NN.NN%)" - "tsNN.NN" | "ts(NN.NN%:NN.NN%)" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        :param visible: Whether this metric should be visible in dashboard graphs. Setting this to false is useful when you want to hide raw metrics that are used in math expressions, and show only the expression results. Default: true

        :stability: experimental
        '''
        ...

    @jsii.member(jsii_name="onEvent")
    def on_event(
        self,
        id: builtins.str,
        *,
        target: typing.Optional[_aws_cdk_aws_events_ceddda9d.IRuleTarget] = None,
        cross_stack_scope: typing.Optional[_constructs_77d1e7e8.Construct] = None,
        description: typing.Optional[builtins.str] = None,
        event_pattern: typing.Optional[typing.Union[_aws_cdk_aws_events_ceddda9d.EventPattern, typing.Dict[builtins.str, typing.Any]]] = None,
        rule_name: typing.Optional[builtins.str] = None,
    ) -> _aws_cdk_aws_events_ceddda9d.Rule:
        '''(experimental) Defines a CloudWatch event rule triggered by agent events.

        :param id: -
        :param target: The target to register for the event. Default: - No target is added to the rule. Use ``addTarget()`` to add a target.
        :param cross_stack_scope: The scope to use if the source of the rule and its target are in different Stacks (but in the same account & region). This helps dealing with cycles that often arise in these situations. Default: - none (the main scope will be used, even for cross-stack Events)
        :param description: A description of the rule's purpose. Default: - No description
        :param event_pattern: Additional restrictions for the event to route to the specified target. The method that generates the rule probably imposes some type of event filtering. The filtering implied by what you pass here is added on top of that filtering. Default: - No additional filtering based on an event pattern.
        :param rule_name: A name for the rule. Default: AWS CloudFormation generates a unique physical ID.

        :stability: experimental
        '''
        ...


class _IAgentProxy(
    jsii.proxy_for(_aws_cdk_ceddda9d.IResource), # type: ignore[misc]
    jsii.proxy_for(_aws_cdk_aws_iam_ceddda9d.IGrantable), # type: ignore[misc]
):
    '''(experimental) Represents an Agent, either created with CDK or imported.

    :stability: experimental
    '''

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-bedrock-alpha.IAgent"

    @builtins.property
    @jsii.member(jsii_name="agentArn")
    def agent_arn(self) -> builtins.str:
        '''(experimental) The ARN of the agent.

        :stability: experimental
        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "agentArn"))

    @builtins.property
    @jsii.member(jsii_name="agentId")
    def agent_id(self) -> builtins.str:
        '''(experimental) The ID of the Agent.

        :stability: experimental
        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "agentId"))

    @builtins.property
    @jsii.member(jsii_name="role")
    def role(self) -> _aws_cdk_aws_iam_ceddda9d.IRole:
        '''(experimental) The IAM role associated to the agent.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_iam_ceddda9d.IRole, jsii.get(self, "role"))

    @builtins.property
    @jsii.member(jsii_name="kmsKey")
    def kms_key(self) -> typing.Optional[_aws_cdk_aws_kms_ceddda9d.IKey]:
        '''(experimental) Optional KMS encryption key associated with this agent.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[_aws_cdk_aws_kms_ceddda9d.IKey], jsii.get(self, "kmsKey"))

    @builtins.property
    @jsii.member(jsii_name="lastUpdated")
    def last_updated(self) -> typing.Optional[builtins.str]:
        '''(experimental) When this agent was last updated.

        :stability: experimental
        :attribute: true
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "lastUpdated"))

    @jsii.member(jsii_name="grantInvoke")
    def grant_invoke(
        self,
        grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
    ) -> _aws_cdk_aws_iam_ceddda9d.Grant:
        '''(experimental) Grant invoke permissions on this agent to an IAM principal.

        Note: This grant will only work when the grantee is in the same AWS account
        where the agent is defined. Cross-account invocation is not supported.

        :param grantee: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f618bd28d205f12fe18f577c6cf06cfa5419cd5eba307a62054fe15003721240)
            check_type(argname="argument grantee", value=grantee, expected_type=type_hints["grantee"])
        return typing.cast(_aws_cdk_aws_iam_ceddda9d.Grant, jsii.invoke(self, "grantInvoke", [grantee]))

    @jsii.member(jsii_name="metricCount")
    def metric_count(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions_map: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        id: typing.Optional[builtins.str] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[_aws_cdk_ceddda9d.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        stack_account: typing.Optional[builtins.str] = None,
        stack_region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[_aws_cdk_aws_cloudwatch_ceddda9d.Unit] = None,
        visible: typing.Optional[builtins.bool] = None,
    ) -> _aws_cdk_aws_cloudwatch_ceddda9d.Metric:
        '''(experimental) Return the CloudWatch metric for agent count.

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions_map: Dimensions of the metric. Default: - No dimensions.
        :param id: Unique identifier for this metric when used in dashboard widgets. The id can be used as a variable to represent this metric in math expressions. Valid characters are letters, numbers, and underscore. The first character must be a lowercase letter. Default: - No ID
        :param label: Label for this metric when added to a Graph in a Dashboard. You can use `dynamic labels <https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/graph-dynamic-labels.html>`_ to show summary information about the entire displayed time series in the legend. For example, if you use:: [max: ${MAX}] MyMetric As the metric label, the maximum value in the visible range will be shown next to the time series name in the graph's legend. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param stack_account: Account of the stack this metric is attached to. Default: - Deployment account.
        :param stack_region: Region of the stack this metric is attached to. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Use the ``aws_cloudwatch.Stats`` helper class to construct valid input strings. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" - "tmNN.NN" | "tm(NN.NN%:NN.NN%)" - "iqm" - "wmNN.NN" | "wm(NN.NN%:NN.NN%)" - "tcNN.NN" | "tc(NN.NN%:NN.NN%)" - "tsNN.NN" | "ts(NN.NN%:NN.NN%)" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        :param visible: Whether this metric should be visible in dashboard graphs. Setting this to false is useful when you want to hide raw metrics that are used in math expressions, and show only the expression results. Default: true

        :stability: experimental
        '''
        props = _aws_cdk_aws_cloudwatch_ceddda9d.MetricOptions(
            account=account,
            color=color,
            dimensions_map=dimensions_map,
            id=id,
            label=label,
            period=period,
            region=region,
            stack_account=stack_account,
            stack_region=stack_region,
            statistic=statistic,
            unit=unit,
            visible=visible,
        )

        return typing.cast(_aws_cdk_aws_cloudwatch_ceddda9d.Metric, jsii.invoke(self, "metricCount", [props]))

    @jsii.member(jsii_name="onEvent")
    def on_event(
        self,
        id: builtins.str,
        *,
        target: typing.Optional[_aws_cdk_aws_events_ceddda9d.IRuleTarget] = None,
        cross_stack_scope: typing.Optional[_constructs_77d1e7e8.Construct] = None,
        description: typing.Optional[builtins.str] = None,
        event_pattern: typing.Optional[typing.Union[_aws_cdk_aws_events_ceddda9d.EventPattern, typing.Dict[builtins.str, typing.Any]]] = None,
        rule_name: typing.Optional[builtins.str] = None,
    ) -> _aws_cdk_aws_events_ceddda9d.Rule:
        '''(experimental) Defines a CloudWatch event rule triggered by agent events.

        :param id: -
        :param target: The target to register for the event. Default: - No target is added to the rule. Use ``addTarget()`` to add a target.
        :param cross_stack_scope: The scope to use if the source of the rule and its target are in different Stacks (but in the same account & region). This helps dealing with cycles that often arise in these situations. Default: - none (the main scope will be used, even for cross-stack Events)
        :param description: A description of the rule's purpose. Default: - No description
        :param event_pattern: Additional restrictions for the event to route to the specified target. The method that generates the rule probably imposes some type of event filtering. The filtering implied by what you pass here is added on top of that filtering. Default: - No additional filtering based on an event pattern.
        :param rule_name: A name for the rule. Default: AWS CloudFormation generates a unique physical ID.

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__36b2c150112fc98f8b2c8eebfbc62fdd6f55ded880e47bf4dffbba187d0bf1c4)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        options = _aws_cdk_aws_events_ceddda9d.OnEventOptions(
            target=target,
            cross_stack_scope=cross_stack_scope,
            description=description,
            event_pattern=event_pattern,
            rule_name=rule_name,
        )

        return typing.cast(_aws_cdk_aws_events_ceddda9d.Rule, jsii.invoke(self, "onEvent", [id, options]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IAgent).__jsii_proxy_class__ = lambda : _IAgentProxy


@jsii.interface(jsii_type="@aws-cdk/aws-bedrock-alpha.IAgentAlias")
class IAgentAlias(_aws_cdk_ceddda9d.IResource, typing_extensions.Protocol):
    '''(experimental) Represents an Agent Alias, either created with CDK or imported.

    :stability: experimental
    '''

    @builtins.property
    @jsii.member(jsii_name="agent")
    def agent(self) -> IAgent:
        '''(experimental) The underlying agent for this alias.

        :stability: experimental
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="aliasArn")
    def alias_arn(self) -> builtins.str:
        '''(experimental) The ARN of the agent alias.

        :stability: experimental
        :attributes: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="aliasId")
    def alias_id(self) -> builtins.str:
        '''(experimental) The unique identifier of the agent alias.

        :stability: experimental
        :attributes: true
        '''
        ...

    @jsii.member(jsii_name="grant")
    def grant(
        self,
        grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
        *actions: builtins.str,
    ) -> _aws_cdk_aws_iam_ceddda9d.Grant:
        '''(experimental) Grant the given principal identity permissions to perform actions on this agent alias.

        :param grantee: -
        :param actions: -

        :stability: experimental
        '''
        ...

    @jsii.member(jsii_name="grantGet")
    def grant_get(
        self,
        grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
    ) -> _aws_cdk_aws_iam_ceddda9d.Grant:
        '''(experimental) Grant the given identity permissions to get the agent alias.

        :param grantee: -

        :stability: experimental
        '''
        ...

    @jsii.member(jsii_name="grantInvoke")
    def grant_invoke(
        self,
        grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
    ) -> _aws_cdk_aws_iam_ceddda9d.Grant:
        '''(experimental) Grant the given identity permissions to invoke the agent alias.

        :param grantee: -

        :stability: experimental
        '''
        ...

    @jsii.member(jsii_name="onCloudTrailEvent")
    def on_cloud_trail_event(
        self,
        id: builtins.str,
        *,
        target: typing.Optional[_aws_cdk_aws_events_ceddda9d.IRuleTarget] = None,
        cross_stack_scope: typing.Optional[_constructs_77d1e7e8.Construct] = None,
        description: typing.Optional[builtins.str] = None,
        event_pattern: typing.Optional[typing.Union[_aws_cdk_aws_events_ceddda9d.EventPattern, typing.Dict[builtins.str, typing.Any]]] = None,
        rule_name: typing.Optional[builtins.str] = None,
    ) -> _aws_cdk_aws_events_ceddda9d.Rule:
        '''(experimental) Define an EventBridge rule that triggers when something happens to this agent alias.

        Requires that there exists at least one CloudTrail Trail in your account
        that captures the event. This method will not create the Trail.

        :param id: The id of the rule.
        :param target: The target to register for the event. Default: - No target is added to the rule. Use ``addTarget()`` to add a target.
        :param cross_stack_scope: The scope to use if the source of the rule and its target are in different Stacks (but in the same account & region). This helps dealing with cycles that often arise in these situations. Default: - none (the main scope will be used, even for cross-stack Events)
        :param description: A description of the rule's purpose. Default: - No description
        :param event_pattern: Additional restrictions for the event to route to the specified target. The method that generates the rule probably imposes some type of event filtering. The filtering implied by what you pass here is added on top of that filtering. Default: - No additional filtering based on an event pattern.
        :param rule_name: A name for the rule. Default: AWS CloudFormation generates a unique physical ID.

        :stability: experimental
        '''
        ...


class _IAgentAliasProxy(
    jsii.proxy_for(_aws_cdk_ceddda9d.IResource), # type: ignore[misc]
):
    '''(experimental) Represents an Agent Alias, either created with CDK or imported.

    :stability: experimental
    '''

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-bedrock-alpha.IAgentAlias"

    @builtins.property
    @jsii.member(jsii_name="agent")
    def agent(self) -> IAgent:
        '''(experimental) The underlying agent for this alias.

        :stability: experimental
        '''
        return typing.cast(IAgent, jsii.get(self, "agent"))

    @builtins.property
    @jsii.member(jsii_name="aliasArn")
    def alias_arn(self) -> builtins.str:
        '''(experimental) The ARN of the agent alias.

        :stability: experimental
        :attributes: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "aliasArn"))

    @builtins.property
    @jsii.member(jsii_name="aliasId")
    def alias_id(self) -> builtins.str:
        '''(experimental) The unique identifier of the agent alias.

        :stability: experimental
        :attributes: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "aliasId"))

    @jsii.member(jsii_name="grant")
    def grant(
        self,
        grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
        *actions: builtins.str,
    ) -> _aws_cdk_aws_iam_ceddda9d.Grant:
        '''(experimental) Grant the given principal identity permissions to perform actions on this agent alias.

        :param grantee: -
        :param actions: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f4cca4a424332ea1f29775bfb3a1ac6db0f5368e9dfae8d1adb7fe67e042a6af)
            check_type(argname="argument grantee", value=grantee, expected_type=type_hints["grantee"])
            check_type(argname="argument actions", value=actions, expected_type=typing.Tuple[type_hints["actions"], ...]) # pyright: ignore [reportGeneralTypeIssues]
        return typing.cast(_aws_cdk_aws_iam_ceddda9d.Grant, jsii.invoke(self, "grant", [grantee, *actions]))

    @jsii.member(jsii_name="grantGet")
    def grant_get(
        self,
        grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
    ) -> _aws_cdk_aws_iam_ceddda9d.Grant:
        '''(experimental) Grant the given identity permissions to get the agent alias.

        :param grantee: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4f3e1ad5ab734c71be4df5c8ad8732b4f05f16f371b0379a039c7c2d3b7dec8a)
            check_type(argname="argument grantee", value=grantee, expected_type=type_hints["grantee"])
        return typing.cast(_aws_cdk_aws_iam_ceddda9d.Grant, jsii.invoke(self, "grantGet", [grantee]))

    @jsii.member(jsii_name="grantInvoke")
    def grant_invoke(
        self,
        grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
    ) -> _aws_cdk_aws_iam_ceddda9d.Grant:
        '''(experimental) Grant the given identity permissions to invoke the agent alias.

        :param grantee: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c205eff6a91ceb0127ca10998bf86f4e152274ba9eceea415fa6e71f802904bb)
            check_type(argname="argument grantee", value=grantee, expected_type=type_hints["grantee"])
        return typing.cast(_aws_cdk_aws_iam_ceddda9d.Grant, jsii.invoke(self, "grantInvoke", [grantee]))

    @jsii.member(jsii_name="onCloudTrailEvent")
    def on_cloud_trail_event(
        self,
        id: builtins.str,
        *,
        target: typing.Optional[_aws_cdk_aws_events_ceddda9d.IRuleTarget] = None,
        cross_stack_scope: typing.Optional[_constructs_77d1e7e8.Construct] = None,
        description: typing.Optional[builtins.str] = None,
        event_pattern: typing.Optional[typing.Union[_aws_cdk_aws_events_ceddda9d.EventPattern, typing.Dict[builtins.str, typing.Any]]] = None,
        rule_name: typing.Optional[builtins.str] = None,
    ) -> _aws_cdk_aws_events_ceddda9d.Rule:
        '''(experimental) Define an EventBridge rule that triggers when something happens to this agent alias.

        Requires that there exists at least one CloudTrail Trail in your account
        that captures the event. This method will not create the Trail.

        :param id: The id of the rule.
        :param target: The target to register for the event. Default: - No target is added to the rule. Use ``addTarget()`` to add a target.
        :param cross_stack_scope: The scope to use if the source of the rule and its target are in different Stacks (but in the same account & region). This helps dealing with cycles that often arise in these situations. Default: - none (the main scope will be used, even for cross-stack Events)
        :param description: A description of the rule's purpose. Default: - No description
        :param event_pattern: Additional restrictions for the event to route to the specified target. The method that generates the rule probably imposes some type of event filtering. The filtering implied by what you pass here is added on top of that filtering. Default: - No additional filtering based on an event pattern.
        :param rule_name: A name for the rule. Default: AWS CloudFormation generates a unique physical ID.

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ad4bacfc955253cfea89b4d09ebbbf2ccae8ba53fb98679bb4f2f96718f4477c)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        options = _aws_cdk_aws_events_ceddda9d.OnEventOptions(
            target=target,
            cross_stack_scope=cross_stack_scope,
            description=description,
            event_pattern=event_pattern,
            rule_name=rule_name,
        )

        return typing.cast(_aws_cdk_aws_events_ceddda9d.Rule, jsii.invoke(self, "onCloudTrailEvent", [id, options]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IAgentAlias).__jsii_proxy_class__ = lambda : _IAgentAliasProxy


@jsii.interface(jsii_type="@aws-cdk/aws-bedrock-alpha.IBedrockInvokable")
class IBedrockInvokable(typing_extensions.Protocol):
    '''(experimental) Represents an Amazon Bedrock abstraction on which you can run the ``Invoke`` API.

    This can be a Foundational Model,
    a Custom Model, or an Inference Profile.

    :stability: experimental
    '''

    @builtins.property
    @jsii.member(jsii_name="invokableArn")
    def invokable_arn(self) -> builtins.str:
        '''(experimental) The ARN of the Bedrock invokable abstraction.

        :stability: experimental
        '''
        ...

    @jsii.member(jsii_name="grantInvoke")
    def grant_invoke(
        self,
        grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
    ) -> _aws_cdk_aws_iam_ceddda9d.Grant:
        '''(experimental) Gives the appropriate policies to invoke and use the invokable abstraction.

        :param grantee: -

        :stability: experimental
        '''
        ...


class _IBedrockInvokableProxy:
    '''(experimental) Represents an Amazon Bedrock abstraction on which you can run the ``Invoke`` API.

    This can be a Foundational Model,
    a Custom Model, or an Inference Profile.

    :stability: experimental
    '''

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-bedrock-alpha.IBedrockInvokable"

    @builtins.property
    @jsii.member(jsii_name="invokableArn")
    def invokable_arn(self) -> builtins.str:
        '''(experimental) The ARN of the Bedrock invokable abstraction.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "invokableArn"))

    @jsii.member(jsii_name="grantInvoke")
    def grant_invoke(
        self,
        grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
    ) -> _aws_cdk_aws_iam_ceddda9d.Grant:
        '''(experimental) Gives the appropriate policies to invoke and use the invokable abstraction.

        :param grantee: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__aeff941da61743d78967d00b93f8c3e1baf285b4d10bc9d78579d8de4c9e8a77)
            check_type(argname="argument grantee", value=grantee, expected_type=type_hints["grantee"])
        return typing.cast(_aws_cdk_aws_iam_ceddda9d.Grant, jsii.invoke(self, "grantInvoke", [grantee]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IBedrockInvokable).__jsii_proxy_class__ = lambda : _IBedrockInvokableProxy


@jsii.data_type(
    jsii_type="@aws-cdk/aws-bedrock-alpha.InferenceConfiguration",
    jsii_struct_bases=[],
    name_mapping={
        "maximum_length": "maximumLength",
        "stop_sequences": "stopSequences",
        "temperature": "temperature",
        "top_k": "topK",
        "top_p": "topP",
    },
)
class InferenceConfiguration:
    def __init__(
        self,
        *,
        maximum_length: jsii.Number,
        stop_sequences: typing.Sequence[builtins.str],
        temperature: jsii.Number,
        top_k: jsii.Number,
        top_p: jsii.Number,
    ) -> None:
        '''(experimental) LLM inference configuration.

        :param maximum_length: (experimental) The maximum number of tokens to generate in the response. Integer min 0 max 4096
        :param stop_sequences: (experimental) A list of stop sequences. A stop sequence is a sequence of characters that causes the model to stop generating the response. length 0-4
        :param temperature: (experimental) The likelihood of the model selecting higher-probability options while generating a response. A lower value makes the model more likely to choose higher-probability options, while a higher value makes the model more likely to choose lower-probability options. Floating point min 0 max 1
        :param top_k: (experimental) While generating a response, the model determines the probability of the following token at each point of generation. The value that you set for topK is the number of most-likely candidates from which the model chooses the next token in the sequence. For example, if you set topK to 50, the model selects the next token from among the top 50 most likely choices. Integer min 0 max 500
        :param top_p: (experimental) While generating a response, the model determines the probability of the following token at each point of generation. The value that you set for Top P determines the number of most-likely candidates from which the model chooses the next token in the sequence. For example, if you set topP to 80, the model only selects the next token from the top 80% of the probability distribution of next tokens. Floating point min 0 max 1

        :stability: experimental
        :exampleMetadata: fixture=default infused

        Example::

            agent = bedrock.Agent(self, "Agent",
                foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
                instruction="You are a helpful assistant.",
                prompt_override_configuration=bedrock.PromptOverrideConfiguration.from_steps([
                    step_type=bedrock.AgentStepType.PRE_PROCESSING,
                    step_enabled=True,
                    custom_prompt_template="Your custom prompt template here",
                    inference_config=bedrock.InferenceConfiguration(
                        temperature=0,
                        top_p=1,
                        top_k=250,
                        maximum_length=1,
                        stop_sequences=["\n\nHuman:"]
                    )
            
                ])
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__715d624dff386da6c193fd1c39e8c824ded7833eed6a175f93d3637254cabb5e)
            check_type(argname="argument maximum_length", value=maximum_length, expected_type=type_hints["maximum_length"])
            check_type(argname="argument stop_sequences", value=stop_sequences, expected_type=type_hints["stop_sequences"])
            check_type(argname="argument temperature", value=temperature, expected_type=type_hints["temperature"])
            check_type(argname="argument top_k", value=top_k, expected_type=type_hints["top_k"])
            check_type(argname="argument top_p", value=top_p, expected_type=type_hints["top_p"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "maximum_length": maximum_length,
            "stop_sequences": stop_sequences,
            "temperature": temperature,
            "top_k": top_k,
            "top_p": top_p,
        }

    @builtins.property
    def maximum_length(self) -> jsii.Number:
        '''(experimental) The maximum number of tokens to generate in the response.

        Integer

        min 0
        max 4096

        :stability: experimental
        '''
        result = self._values.get("maximum_length")
        assert result is not None, "Required property 'maximum_length' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def stop_sequences(self) -> typing.List[builtins.str]:
        '''(experimental) A list of stop sequences.

        A stop sequence is a sequence of characters that
        causes the model to stop generating the response.

        length 0-4

        :stability: experimental
        '''
        result = self._values.get("stop_sequences")
        assert result is not None, "Required property 'stop_sequences' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def temperature(self) -> jsii.Number:
        '''(experimental) The likelihood of the model selecting higher-probability options while generating a response.

        A lower value makes the model more likely to choose
        higher-probability options, while a higher value makes the model more
        likely to choose lower-probability options.

        Floating point

        min 0
        max 1

        :stability: experimental
        '''
        result = self._values.get("temperature")
        assert result is not None, "Required property 'temperature' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def top_k(self) -> jsii.Number:
        '''(experimental) While generating a response, the model determines the probability of the following token at each point of generation.

        The value that you set for
        topK is the number of most-likely candidates from which the model chooses
        the next token in the sequence. For example, if you set topK to 50, the
        model selects the next token from among the top 50 most likely choices.

        Integer

        min 0
        max 500

        :stability: experimental
        '''
        result = self._values.get("top_k")
        assert result is not None, "Required property 'top_k' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def top_p(self) -> jsii.Number:
        '''(experimental) While generating a response, the model determines the probability of the following token at each point of generation.

        The value that you set for
        Top P determines the number of most-likely candidates from which the model
        chooses the next token in the sequence. For example, if you set topP to
        80, the model only selects the next token from the top 80% of the
        probability distribution of next tokens.

        Floating point

        min 0
        max 1

        :stability: experimental
        '''
        result = self._values.get("top_p")
        assert result is not None, "Required property 'top_p' is missing"
        return typing.cast(jsii.Number, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "InferenceConfiguration(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class InlineApiSchema(
    ApiSchema,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-bedrock-alpha.InlineApiSchema",
):
    '''(experimental) Class to define an API Schema from an inline string.

    The schema can be provided directly as a string in either JSON or YAML format.

    :stability: experimental
    :exampleMetadata: fixture=default infused

    Example::

        inline_schema = bedrock.ApiSchema.from_inline("""
            openapi: 3.0.3
            info:
              title: Library API
              version: 1.0.0
            paths:
              /search:
                get:
                  summary: Search for books
                  operationId: searchBooks
                  parameters:
                    - name: query
                      in: query
                      required: true
                      schema:
                        type: string
            """)
        
        action_group_function = lambda_.Function(self, "ActionGroupFunction",
            runtime=lambda_.Runtime.PYTHON_3_12,
            handler="index.handler",
            code=lambda_.Code.from_asset(path.join(__dirname, "../lambda/action-group"))
        )
        
        action_group = bedrock.AgentActionGroup(
            name="query-library",
            description="Use these functions to get information about the books in the library.",
            executor=bedrock.ActionGroupExecutor.from_lambda(action_group_function),
            enabled=True,
            api_schema=inline_schema
        )
        
        agent = bedrock.Agent(self, "Agent",
            foundation_model=bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_HAIKU_V1_0,
            instruction="You are a helpful and friendly agent that answers questions about literature."
        )
        
        agent.add_action_group(action_group)
    '''

    def __init__(self, schema: builtins.str) -> None:
        '''
        :param schema: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0c276a821c9a889fcfc1d87230ab9f92627a99c7dc83fc06a8d00bb8d1534657)
            check_type(argname="argument schema", value=schema, expected_type=type_hints["schema"])
        jsii.create(self.__class__, self, [schema])


class Memory(metaclass=jsii.JSIIMeta, jsii_type="@aws-cdk/aws-bedrock-alpha.Memory"):
    '''(experimental) Memory class for managing Bedrock Agent memory configurations.

    Enables conversational context retention
    across multiple sessions through session identifiers. Memory context is stored with unique
    memory IDs per user, allowing access to conversation history and summaries. Supports viewing
    stored sessions and clearing memory.

    :see: https://docs.aws.amazon.com/bedrock/latest/userguide/agents-memory.html
    :stability: experimental
    :exampleMetadata: fixture=default infused

    Example::

        agent = bedrock.Agent(self, "MyAgent",
            agent_name="MyAgent",
            instruction="Your instruction here",
            foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
            memory=Memory.session_summary(
                max_recent_sessions=10,  # Keep the last 10 session summaries
                memory_duration=Duration.days(20)
            )
        )
    '''

    def __init__(
        self,
        *,
        max_recent_sessions: typing.Optional[jsii.Number] = None,
        memory_duration: typing.Optional[_aws_cdk_ceddda9d.Duration] = None,
    ) -> None:
        '''
        :param max_recent_sessions: (experimental) Maximum number of recent session summaries to include (min 1). Default: 20
        :param memory_duration: (experimental) Duration for which session summaries are retained (between 1 and 365 days). Default: Duration.days(30)

        :stability: experimental
        '''
        props = SessionSummaryMemoryProps(
            max_recent_sessions=max_recent_sessions, memory_duration=memory_duration
        )

        jsii.create(self.__class__, self, [props])

    @jsii.member(jsii_name="sessionSummary")
    @builtins.classmethod
    def session_summary(
        cls,
        *,
        max_recent_sessions: typing.Optional[jsii.Number] = None,
        memory_duration: typing.Optional[_aws_cdk_ceddda9d.Duration] = None,
    ) -> "Memory":
        '''(experimental) Creates a session summary memory with custom configuration.

        :param max_recent_sessions: (experimental) Maximum number of recent session summaries to include (min 1). Default: 20
        :param memory_duration: (experimental) Duration for which session summaries are retained (between 1 and 365 days). Default: Duration.days(30)

        :return: Memory instance

        :stability: experimental
        '''
        props = SessionSummaryMemoryProps(
            max_recent_sessions=max_recent_sessions, memory_duration=memory_duration
        )

        return typing.cast("Memory", jsii.sinvoke(cls, "sessionSummary", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SESSION_SUMMARY")
    def SESSION_SUMMARY(cls) -> "Memory":
        '''(experimental) Returns session summary memory with default configuration.

        :default: memoryDuration=Duration.days(30), maxRecentSessions=20

        :stability: experimental
        '''
        return typing.cast("Memory", jsii.sget(cls, "SESSION_SUMMARY"))


@jsii.enum(jsii_type="@aws-cdk/aws-bedrock-alpha.OrchestrationType")
class OrchestrationType(enum.Enum):
    '''(experimental) Enum for orchestration types available for agents.

    :stability: experimental
    '''

    DEFAULT = "DEFAULT"
    '''(experimental) Default orchestration by the agent.

    :stability: experimental
    '''
    CUSTOM_ORCHESTRATION = "CUSTOM_ORCHESTRATION"
    '''(experimental) Custom orchestration using Lambda.

    :stability: experimental
    '''


@jsii.enum(jsii_type="@aws-cdk/aws-bedrock-alpha.ParameterType")
class ParameterType(enum.Enum):
    '''(experimental) Enum for parameter types in function schemas.

    :stability: experimental
    :exampleMetadata: fixture=default infused

    Example::

        action_group_function = lambda_.Function(self, "ActionGroupFunction",
            runtime=lambda_.Runtime.PYTHON_3_12,
            handler="index.handler",
            code=lambda_.Code.from_asset(path.join(__dirname, "../lambda/action-group"))
        )
        
        # Define a function schema with parameters
        function_schema = bedrock.FunctionSchema(
            functions=[bedrock.FunctionProps(
                name="searchBooks",
                description="Search for books in the library catalog",
                parameters={
                    "query": bedrock.FunctionParameterProps(
                        type=bedrock.ParameterType.STRING,
                        required=True,
                        description="The search query string"
                    ),
                    "maxResults": bedrock.FunctionParameterProps(
                        type=bedrock.ParameterType.INTEGER,
                        required=False,
                        description="Maximum number of results to return"
                    ),
                    "includeOutOfPrint": bedrock.FunctionParameterProps(
                        type=bedrock.ParameterType.BOOLEAN,
                        required=False,
                        description="Whether to include out-of-print books"
                    )
                },
                require_confirmation=bedrock.RequireConfirmation.DISABLED
            ), bedrock.FunctionProps(
                name="getBookDetails",
                description="Get detailed information about a specific book",
                parameters={
                    "bookId": bedrock.FunctionParameterProps(
                        type=bedrock.ParameterType.STRING,
                        required=True,
                        description="The unique identifier of the book"
                    )
                },
                require_confirmation=bedrock.RequireConfirmation.ENABLED
            )
            ]
        )
        
        # Create an action group using the function schema
        action_group = bedrock.AgentActionGroup(
            name="library-functions",
            description="Functions for interacting with the library catalog",
            executor=bedrock.ActionGroupExecutor.from_lambda(action_group_function),
            function_schema=function_schema,
            enabled=True
        )
        
        agent = bedrock.Agent(self, "Agent",
            foundation_model=bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_HAIKU_V1_0,
            instruction="You are a helpful and friendly agent that answers questions about literature.",
            action_groups=[action_group]
        )
    '''

    STRING = "STRING"
    '''(experimental) String parameter type.

    :stability: experimental
    '''
    NUMBER = "NUMBER"
    '''(experimental) Number parameter type.

    :stability: experimental
    '''
    INTEGER = "INTEGER"
    '''(experimental) Integer parameter type.

    :stability: experimental
    '''
    BOOLEAN = "BOOLEAN"
    '''(experimental) Boolean parameter type.

    :stability: experimental
    '''
    ARRAY = "ARRAY"
    '''(experimental) Array parameter type.

    :stability: experimental
    '''
    OBJECT = "OBJECT"
    '''(experimental) Object parameter type.

    :stability: experimental
    '''


class ParentActionGroupSignature(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-bedrock-alpha.ParentActionGroupSignature",
):
    '''(experimental) AWS Defined signatures for enabling certain capabilities in your agent.

    :stability: experimental
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_bedrock_alpha as bedrock_alpha
        
        parent_action_group_signature = bedrock_alpha.ParentActionGroupSignature("value")
    '''

    def __init__(self, value: builtins.str) -> None:
        '''(experimental) Constructor should be used as a temporary solution when a new signature is supported but its implementation in CDK hasn't been added yet.

        :param value: The AWS-defined signature value for this action group capability.

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ebccabd52fd67d660adf169e439525008c110204b48e0ed669d4ffeedce9e65e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.create(self.__class__, self, [value])

    @jsii.member(jsii_name="toString")
    def to_string(self) -> builtins.str:
        '''(experimental) Returns the string representation of the signature value.

        Used when configuring the action group in CloudFormation.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "toString", []))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CODE_INTERPRETER")
    def CODE_INTERPRETER(cls) -> "ParentActionGroupSignature":
        '''(experimental) Signature that allows your agent to generate, run, and troubleshoot code when trying to complete a task.

        :stability: experimental
        '''
        return typing.cast("ParentActionGroupSignature", jsii.sget(cls, "CODE_INTERPRETER"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="USER_INPUT")
    def USER_INPUT(cls) -> "ParentActionGroupSignature":
        '''(experimental) Signature that allows your agent to request the user for additional information when trying to complete a task.

        :stability: experimental
        '''
        return typing.cast("ParentActionGroupSignature", jsii.sget(cls, "USER_INPUT"))

    @builtins.property
    @jsii.member(jsii_name="value")
    def value(self) -> builtins.str:
        '''(experimental) The AWS-defined signature value for this action group capability.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "value"))


class PromptOverrideConfiguration(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-bedrock-alpha.PromptOverrideConfiguration",
):
    '''(experimental) Configuration for overriding prompt templates and behaviors in different parts of an agent's sequence.

    This allows customizing how the agent processes inputs,
    makes decisions, and generates responses.

    :stability: experimental
    :exampleMetadata: fixture=default infused

    Example::

        agent = bedrock.Agent(self, "Agent",
            foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
            instruction="You are a helpful assistant.",
            prompt_override_configuration=bedrock.PromptOverrideConfiguration.from_steps([
                step_type=bedrock.AgentStepType.ROUTING_CLASSIFIER,
                step_enabled=True,
                custom_prompt_template="Your routing template here",
                foundation_model=bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_V2
        
            ])
        )
    '''

    @jsii.member(jsii_name="fromSteps")
    @builtins.classmethod
    def from_steps(
        cls,
        steps: typing.Sequence[typing.Union["PromptStepConfigBase", typing.Dict[builtins.str, typing.Any]]],
    ) -> "PromptOverrideConfiguration":
        '''(experimental) Creates a PromptOverrideConfiguration from individual step configurations.

        Use this method when you want to override prompts without using a custom parser.

        :param steps: The step configurations to use.

        :return: A new PromptOverrideConfiguration instance

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1d39057223b726590938606b5e3ac9d394a1418fec4a81576b5273321f3b86d8)
            check_type(argname="argument steps", value=steps, expected_type=type_hints["steps"])
        return typing.cast("PromptOverrideConfiguration", jsii.sinvoke(cls, "fromSteps", [steps]))

    @jsii.member(jsii_name="withCustomParser")
    @builtins.classmethod
    def with_custom_parser(
        cls,
        *,
        knowledge_base_response_generation_step: typing.Optional[typing.Union["PromptKnowledgeBaseResponseGenerationConfigCustomParser", typing.Dict[builtins.str, typing.Any]]] = None,
        memory_summarization_step: typing.Optional[typing.Union["PromptMemorySummarizationConfigCustomParser", typing.Dict[builtins.str, typing.Any]]] = None,
        orchestration_step: typing.Optional[typing.Union["PromptOrchestrationConfigCustomParser", typing.Dict[builtins.str, typing.Any]]] = None,
        parser: typing.Optional[_aws_cdk_aws_lambda_ceddda9d.IFunction] = None,
        post_processing_step: typing.Optional[typing.Union["PromptPostProcessingConfigCustomParser", typing.Dict[builtins.str, typing.Any]]] = None,
        pre_processing_step: typing.Optional[typing.Union["PromptPreProcessingConfigCustomParser", typing.Dict[builtins.str, typing.Any]]] = None,
        routing_classifier_step: typing.Optional[typing.Union["PromptRoutingClassifierConfigCustomParser", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> "PromptOverrideConfiguration":
        '''(experimental) Creates a PromptOverrideConfiguration with a custom Lambda parser function.

        :param knowledge_base_response_generation_step: (experimental) Configuration for the knowledge base response generation step. Default: undefined - No knowledge base response generation configuration
        :param memory_summarization_step: (experimental) Configuration for the memory summarization step. Default: undefined - No memory summarization configuration
        :param orchestration_step: (experimental) Configuration for the orchestration step. Default: undefined - No orchestration configuration
        :param parser: (experimental) Lambda function to use as custom parser. Default: undefined - No custom parser is used
        :param post_processing_step: (experimental) Configuration for the post-processing step. Default: undefined - No post-processing configuration
        :param pre_processing_step: (experimental) Configuration for the pre-processing step. Default: undefined - No pre-processing configuration
        :param routing_classifier_step: (experimental) Configuration for the routing classifier step. Default: undefined - No routing classifier configuration

        :stability: experimental
        '''
        props = CustomParserProps(
            knowledge_base_response_generation_step=knowledge_base_response_generation_step,
            memory_summarization_step=memory_summarization_step,
            orchestration_step=orchestration_step,
            parser=parser,
            post_processing_step=post_processing_step,
            pre_processing_step=pre_processing_step,
            routing_classifier_step=routing_classifier_step,
        )

        return typing.cast("PromptOverrideConfiguration", jsii.sinvoke(cls, "withCustomParser", [props]))

    @builtins.property
    @jsii.member(jsii_name="knowledgeBaseResponseGenerationStep")
    def knowledge_base_response_generation_step(
        self,
    ) -> typing.Optional["PromptKnowledgeBaseResponseGenerationConfigCustomParser"]:
        '''(experimental) Configuration for the knowledge base response generation step.

        :stability: experimental
        '''
        return typing.cast(typing.Optional["PromptKnowledgeBaseResponseGenerationConfigCustomParser"], jsii.get(self, "knowledgeBaseResponseGenerationStep"))

    @builtins.property
    @jsii.member(jsii_name="memorySummarizationStep")
    def memory_summarization_step(
        self,
    ) -> typing.Optional["PromptMemorySummarizationConfigCustomParser"]:
        '''(experimental) Configuration for the memory summarization step.

        :stability: experimental
        '''
        return typing.cast(typing.Optional["PromptMemorySummarizationConfigCustomParser"], jsii.get(self, "memorySummarizationStep"))

    @builtins.property
    @jsii.member(jsii_name="orchestrationStep")
    def orchestration_step(
        self,
    ) -> typing.Optional["PromptOrchestrationConfigCustomParser"]:
        '''(experimental) Configuration for the orchestration step.

        :stability: experimental
        '''
        return typing.cast(typing.Optional["PromptOrchestrationConfigCustomParser"], jsii.get(self, "orchestrationStep"))

    @builtins.property
    @jsii.member(jsii_name="parser")
    def parser(self) -> typing.Optional[_aws_cdk_aws_lambda_ceddda9d.IFunction]:
        '''(experimental) The custom Lambda parser function to use.

        The Lambda parser processes and interprets the raw foundation model output.
        It receives an input event with:

        - messageVersion: Version of message format (1.0)
        - agent: Info about the agent (name, id, alias, version)
        - invokeModelRawResponse: Raw model output to parse
        - promptType: Type of prompt being parsed
        - overrideType: Type of override (OUTPUT_PARSER)

        The Lambda must return a response that the agent uses for next actions.

        :see: https://docs.aws.amazon.com/bedrock/latest/userguide/lambda-parser.html
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_aws_cdk_aws_lambda_ceddda9d.IFunction], jsii.get(self, "parser"))

    @builtins.property
    @jsii.member(jsii_name="postProcessingStep")
    def post_processing_step(
        self,
    ) -> typing.Optional["PromptPostProcessingConfigCustomParser"]:
        '''(experimental) Configuration for the post-processing step.

        :stability: experimental
        '''
        return typing.cast(typing.Optional["PromptPostProcessingConfigCustomParser"], jsii.get(self, "postProcessingStep"))

    @builtins.property
    @jsii.member(jsii_name="preProcessingStep")
    def pre_processing_step(
        self,
    ) -> typing.Optional["PromptPreProcessingConfigCustomParser"]:
        '''(experimental) Configuration for the pre-processing step.

        :stability: experimental
        '''
        return typing.cast(typing.Optional["PromptPreProcessingConfigCustomParser"], jsii.get(self, "preProcessingStep"))

    @builtins.property
    @jsii.member(jsii_name="routingClassifierStep")
    def routing_classifier_step(
        self,
    ) -> typing.Optional["PromptRoutingClassifierConfigCustomParser"]:
        '''(experimental) Configuration for the routing classifier step.

        :stability: experimental
        '''
        return typing.cast(typing.Optional["PromptRoutingClassifierConfigCustomParser"], jsii.get(self, "routingClassifierStep"))


@jsii.data_type(
    jsii_type="@aws-cdk/aws-bedrock-alpha.PromptStepConfigBase",
    jsii_struct_bases=[],
    name_mapping={
        "step_type": "stepType",
        "custom_prompt_template": "customPromptTemplate",
        "inference_config": "inferenceConfig",
        "step_enabled": "stepEnabled",
        "use_custom_parser": "useCustomParser",
    },
)
class PromptStepConfigBase:
    def __init__(
        self,
        *,
        step_type: AgentStepType,
        custom_prompt_template: typing.Optional[builtins.str] = None,
        inference_config: typing.Optional[typing.Union[InferenceConfiguration, typing.Dict[builtins.str, typing.Any]]] = None,
        step_enabled: typing.Optional[builtins.bool] = None,
        use_custom_parser: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''(experimental) Base configuration interface for all prompt step types.

        :param step_type: (experimental) The type of step this configuration applies to.
        :param custom_prompt_template: (experimental) The custom prompt template to be used. Default: - The default prompt template will be used.
        :param inference_config: (experimental) The inference configuration parameters to use. Default: undefined - Default inference configuration will be used
        :param step_enabled: (experimental) Whether to enable or skip this step in the agent sequence. Default: - The default state for each step type is as follows. PRE_PROCESSING – ENABLED ORCHESTRATION – ENABLED KNOWLEDGE_BASE_RESPONSE_GENERATION – ENABLED POST_PROCESSING – DISABLED
        :param use_custom_parser: (experimental) Whether to use the custom Lambda parser defined for the sequence. Default: - false

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_bedrock_alpha as bedrock_alpha
            
            prompt_step_config_base = bedrock_alpha.PromptStepConfigBase(
                step_type=bedrock_alpha.AgentStepType.PRE_PROCESSING,
            
                # the properties below are optional
                custom_prompt_template="customPromptTemplate",
                inference_config=bedrock_alpha.InferenceConfiguration(
                    maximum_length=123,
                    stop_sequences=["stopSequences"],
                    temperature=123,
                    top_k=123,
                    top_p=123
                ),
                step_enabled=False,
                use_custom_parser=False
            )
        '''
        if isinstance(inference_config, dict):
            inference_config = InferenceConfiguration(**inference_config)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c80dbca03d170ec1393aad8bb4aee5d9109f13b498913a973f3a939b8dfeb044)
            check_type(argname="argument step_type", value=step_type, expected_type=type_hints["step_type"])
            check_type(argname="argument custom_prompt_template", value=custom_prompt_template, expected_type=type_hints["custom_prompt_template"])
            check_type(argname="argument inference_config", value=inference_config, expected_type=type_hints["inference_config"])
            check_type(argname="argument step_enabled", value=step_enabled, expected_type=type_hints["step_enabled"])
            check_type(argname="argument use_custom_parser", value=use_custom_parser, expected_type=type_hints["use_custom_parser"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "step_type": step_type,
        }
        if custom_prompt_template is not None:
            self._values["custom_prompt_template"] = custom_prompt_template
        if inference_config is not None:
            self._values["inference_config"] = inference_config
        if step_enabled is not None:
            self._values["step_enabled"] = step_enabled
        if use_custom_parser is not None:
            self._values["use_custom_parser"] = use_custom_parser

    @builtins.property
    def step_type(self) -> AgentStepType:
        '''(experimental) The type of step this configuration applies to.

        :stability: experimental
        '''
        result = self._values.get("step_type")
        assert result is not None, "Required property 'step_type' is missing"
        return typing.cast(AgentStepType, result)

    @builtins.property
    def custom_prompt_template(self) -> typing.Optional[builtins.str]:
        '''(experimental) The custom prompt template to be used.

        :default: - The default prompt template will be used.

        :see: https://docs.aws.amazon.com/bedrock/latest/userguide/prompt-placeholders.html
        :stability: experimental
        '''
        result = self._values.get("custom_prompt_template")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def inference_config(self) -> typing.Optional[InferenceConfiguration]:
        '''(experimental) The inference configuration parameters to use.

        :default: undefined - Default inference configuration will be used

        :stability: experimental
        '''
        result = self._values.get("inference_config")
        return typing.cast(typing.Optional[InferenceConfiguration], result)

    @builtins.property
    def step_enabled(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to enable or skip this step in the agent sequence.

        :default:

        - The default state for each step type is as follows.

        PRE_PROCESSING – ENABLED
        ORCHESTRATION – ENABLED
        KNOWLEDGE_BASE_RESPONSE_GENERATION – ENABLED
        POST_PROCESSING – DISABLED

        :stability: experimental
        '''
        result = self._values.get("step_enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def use_custom_parser(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to use the custom Lambda parser defined for the sequence.

        :default: - false

        :stability: experimental
        '''
        result = self._values.get("use_custom_parser")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "PromptStepConfigBase(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@aws-cdk/aws-bedrock-alpha.RequireConfirmation")
class RequireConfirmation(enum.Enum):
    '''(experimental) Enum for require confirmation state in function schemas.

    :stability: experimental
    :exampleMetadata: fixture=default infused

    Example::

        action_group_function = lambda_.Function(self, "ActionGroupFunction",
            runtime=lambda_.Runtime.PYTHON_3_12,
            handler="index.handler",
            code=lambda_.Code.from_asset(path.join(__dirname, "../lambda/action-group"))
        )
        
        # Define a function schema with parameters
        function_schema = bedrock.FunctionSchema(
            functions=[bedrock.FunctionProps(
                name="searchBooks",
                description="Search for books in the library catalog",
                parameters={
                    "query": bedrock.FunctionParameterProps(
                        type=bedrock.ParameterType.STRING,
                        required=True,
                        description="The search query string"
                    ),
                    "maxResults": bedrock.FunctionParameterProps(
                        type=bedrock.ParameterType.INTEGER,
                        required=False,
                        description="Maximum number of results to return"
                    ),
                    "includeOutOfPrint": bedrock.FunctionParameterProps(
                        type=bedrock.ParameterType.BOOLEAN,
                        required=False,
                        description="Whether to include out-of-print books"
                    )
                },
                require_confirmation=bedrock.RequireConfirmation.DISABLED
            ), bedrock.FunctionProps(
                name="getBookDetails",
                description="Get detailed information about a specific book",
                parameters={
                    "bookId": bedrock.FunctionParameterProps(
                        type=bedrock.ParameterType.STRING,
                        required=True,
                        description="The unique identifier of the book"
                    )
                },
                require_confirmation=bedrock.RequireConfirmation.ENABLED
            )
            ]
        )
        
        # Create an action group using the function schema
        action_group = bedrock.AgentActionGroup(
            name="library-functions",
            description="Functions for interacting with the library catalog",
            executor=bedrock.ActionGroupExecutor.from_lambda(action_group_function),
            function_schema=function_schema,
            enabled=True
        )
        
        agent = bedrock.Agent(self, "Agent",
            foundation_model=bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_HAIKU_V1_0,
            instruction="You are a helpful and friendly agent that answers questions about literature.",
            action_groups=[action_group]
        )
    '''

    ENABLED = "ENABLED"
    '''(experimental) Confirmation is enabled.

    :stability: experimental
    '''
    DISABLED = "DISABLED"
    '''(experimental) Confirmation is disabled.

    :stability: experimental
    '''


class S3ApiSchema(
    ApiSchema,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-bedrock-alpha.S3ApiSchema",
):
    '''(experimental) Class to define an API Schema from an S3 object.

    :stability: experimental
    :exampleMetadata: fixture=default infused

    Example::

        bucket = s3.Bucket.from_bucket_name(self, "ExistingBucket", "my-schema-bucket")
        s3_schema = bedrock.ApiSchema.from_s3_file(bucket, "schemas/action-group.yaml")
        
        action_group_function = lambda_.Function(self, "ActionGroupFunction",
            runtime=lambda_.Runtime.PYTHON_3_12,
            handler="index.handler",
            code=lambda_.Code.from_asset(path.join(__dirname, "../lambda/action-group"))
        )
        
        action_group = bedrock.AgentActionGroup(
            name="query-library",
            description="Use these functions to get information about the books in the library.",
            executor=bedrock.ActionGroupExecutor.from_lambda(action_group_function),
            enabled=True,
            api_schema=s3_schema
        )
        
        agent = bedrock.Agent(self, "Agent",
            foundation_model=bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_HAIKU_V1_0,
            instruction="You are a helpful and friendly agent that answers questions about literature."
        )
        
        agent.add_action_group(action_group)
    '''

    def __init__(
        self,
        *,
        bucket_name: builtins.str,
        object_key: builtins.str,
        object_version: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param bucket_name: The name of the S3 Bucket the object is in.
        :param object_key: The path inside the Bucket where the object is located at.
        :param object_version: The S3 object version.

        :stability: experimental
        '''
        location = _aws_cdk_aws_s3_ceddda9d.Location(
            bucket_name=bucket_name,
            object_key=object_key,
            object_version=object_version,
        )

        jsii.create(self.__class__, self, [location])


@jsii.data_type(
    jsii_type="@aws-cdk/aws-bedrock-alpha.SessionSummaryMemoryProps",
    jsii_struct_bases=[],
    name_mapping={
        "max_recent_sessions": "maxRecentSessions",
        "memory_duration": "memoryDuration",
    },
)
class SessionSummaryMemoryProps:
    def __init__(
        self,
        *,
        max_recent_sessions: typing.Optional[jsii.Number] = None,
        memory_duration: typing.Optional[_aws_cdk_ceddda9d.Duration] = None,
    ) -> None:
        '''(experimental) Properties for SessionSummaryConfiguration.

        :param max_recent_sessions: (experimental) Maximum number of recent session summaries to include (min 1). Default: 20
        :param memory_duration: (experimental) Duration for which session summaries are retained (between 1 and 365 days). Default: Duration.days(30)

        :stability: experimental
        :exampleMetadata: fixture=default infused

        Example::

            agent = bedrock.Agent(self, "MyAgent",
                agent_name="MyAgent",
                instruction="Your instruction here",
                foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
                memory=Memory.session_summary(
                    max_recent_sessions=10,  # Keep the last 10 session summaries
                    memory_duration=Duration.days(20)
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__66ca7c708f8916f74c35e9ad0069f982f7710c5742e63b6e61b9889bdcea86ee)
            check_type(argname="argument max_recent_sessions", value=max_recent_sessions, expected_type=type_hints["max_recent_sessions"])
            check_type(argname="argument memory_duration", value=memory_duration, expected_type=type_hints["memory_duration"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if max_recent_sessions is not None:
            self._values["max_recent_sessions"] = max_recent_sessions
        if memory_duration is not None:
            self._values["memory_duration"] = memory_duration

    @builtins.property
    def max_recent_sessions(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Maximum number of recent session summaries to include (min 1).

        :default: 20

        :stability: experimental
        '''
        result = self._values.get("max_recent_sessions")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def memory_duration(self) -> typing.Optional[_aws_cdk_ceddda9d.Duration]:
        '''(experimental) Duration for which session summaries are retained (between 1 and 365 days).

        :default: Duration.days(30)

        :stability: experimental
        '''
        result = self._values.get("memory_duration")
        return typing.cast(typing.Optional[_aws_cdk_ceddda9d.Duration], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SessionSummaryMemoryProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@aws-cdk/aws-bedrock-alpha.VectorType")
class VectorType(enum.Enum):
    '''(experimental) The data type for the vectors when using a model to convert text into vector embeddings.

    The model must support the specified data type for vector embeddings. Floating-point (float32)
    is the default data type, and is supported by most models for vector embeddings. See Supported
    embeddings models for information on the available models and their vector data types.

    :stability: experimental
    '''

    FLOATING_POINT = "FLOATING_POINT"
    '''(experimental) ``FLOATING_POINT`` convert the data to floating-point (float32) vector embeddings (more precise, but more costly).

    :stability: experimental
    '''
    BINARY = "BINARY"
    '''(experimental) ``BINARY`` convert the data to binary vector embeddings (less precise, but less costly).

    :stability: experimental
    '''


@jsii.implements(IAgentAlias)
class AgentAliasBase(
    _aws_cdk_ceddda9d.Resource,
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@aws-cdk/aws-bedrock-alpha.AgentAliasBase",
):
    '''(experimental) Abstract base class for an Agent.

    Contains methods and attributes valid for Agents either created with CDK or imported.

    :stability: experimental
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        account: typing.Optional[builtins.str] = None,
        environment_from_arn: typing.Optional[builtins.str] = None,
        physical_name: typing.Optional[builtins.str] = None,
        region: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param account: The AWS account ID this resource belongs to. Default: - the resource is in the same account as the stack it belongs to
        :param environment_from_arn: ARN to deduce region and account from. The ARN is parsed and the account and region are taken from the ARN. This should be used for imported resources. Cannot be supplied together with either ``account`` or ``region``. Default: - take environment from ``account``, ``region`` parameters, or use Stack environment.
        :param physical_name: The value passed in by users to the physical name prop of the resource. - ``undefined`` implies that a physical name will be allocated by CloudFormation during deployment. - a concrete value implies a specific physical name - ``PhysicalName.GENERATE_IF_NEEDED`` is a marker that indicates that a physical will only be generated by the CDK if it is needed for cross-environment references. Otherwise, it will be allocated by CloudFormation. Default: - The physical name will be allocated by CloudFormation at deployment time
        :param region: The AWS region this resource belongs to. Default: - the resource is in the same region as the stack it belongs to
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c76c254ff658709c91b9f92836c40baa806c28af85c028c89448b6d932a0915c)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = _aws_cdk_ceddda9d.ResourceProps(
            account=account,
            environment_from_arn=environment_from_arn,
            physical_name=physical_name,
            region=region,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="grant")
    def grant(
        self,
        grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
        *actions: builtins.str,
    ) -> _aws_cdk_aws_iam_ceddda9d.Grant:
        '''(experimental) Grant the given principal identity permissions to perform actions on this agent alias.

        Note: This grant will only work when the grantee is in the same AWS account
        where the agent alias is defined. Cross-account grant is not supported.

        :param grantee: -
        :param actions: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f605b4cd4baa1eb130f51bcadc5c4a81822ae2cab31ec49a4e9b5b32f8fc9c45)
            check_type(argname="argument grantee", value=grantee, expected_type=type_hints["grantee"])
            check_type(argname="argument actions", value=actions, expected_type=typing.Tuple[type_hints["actions"], ...]) # pyright: ignore [reportGeneralTypeIssues]
        return typing.cast(_aws_cdk_aws_iam_ceddda9d.Grant, jsii.invoke(self, "grant", [grantee, *actions]))

    @jsii.member(jsii_name="grantGet")
    def grant_get(
        self,
        grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
    ) -> _aws_cdk_aws_iam_ceddda9d.Grant:
        '''(experimental) Grant the given identity permissions to get the agent alias.

        Note: This grant will only work when the grantee is in the same AWS account
        where the agent alias is defined. Cross-account agent read is not supported.

        :param grantee: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dbf39e88263f8cef13dfe1f4d67a47c3bf2490d27c8ce9a70ade0cad70e3b7d6)
            check_type(argname="argument grantee", value=grantee, expected_type=type_hints["grantee"])
        return typing.cast(_aws_cdk_aws_iam_ceddda9d.Grant, jsii.invoke(self, "grantGet", [grantee]))

    @jsii.member(jsii_name="grantInvoke")
    def grant_invoke(
        self,
        grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
    ) -> _aws_cdk_aws_iam_ceddda9d.Grant:
        '''(experimental) Grant the given identity permissions to invoke the agent alias.

        Note: This grant will only work when the grantee is in the same AWS account
        where the agent alias is defined. Cross-account invocation is not supported.

        :param grantee: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d6a50391f64a87b84a4584ba8e48f37b926fb74fc6eceb7e6109f491509b2092)
            check_type(argname="argument grantee", value=grantee, expected_type=type_hints["grantee"])
        return typing.cast(_aws_cdk_aws_iam_ceddda9d.Grant, jsii.invoke(self, "grantInvoke", [grantee]))

    @jsii.member(jsii_name="onCloudTrailEvent")
    def on_cloud_trail_event(
        self,
        id: builtins.str,
        *,
        target: typing.Optional[_aws_cdk_aws_events_ceddda9d.IRuleTarget] = None,
        cross_stack_scope: typing.Optional[_constructs_77d1e7e8.Construct] = None,
        description: typing.Optional[builtins.str] = None,
        event_pattern: typing.Optional[typing.Union[_aws_cdk_aws_events_ceddda9d.EventPattern, typing.Dict[builtins.str, typing.Any]]] = None,
        rule_name: typing.Optional[builtins.str] = None,
    ) -> _aws_cdk_aws_events_ceddda9d.Rule:
        '''(experimental) Define an EventBridge rule that triggers when something happens to this agent alias.

        Requires that there exists at least one CloudTrail Trail in your account
        that captures the event. This method will not create the Trail.

        :param id: The id of the rule.
        :param target: The target to register for the event. Default: - No target is added to the rule. Use ``addTarget()`` to add a target.
        :param cross_stack_scope: The scope to use if the source of the rule and its target are in different Stacks (but in the same account & region). This helps dealing with cycles that often arise in these situations. Default: - none (the main scope will be used, even for cross-stack Events)
        :param description: A description of the rule's purpose. Default: - No description
        :param event_pattern: Additional restrictions for the event to route to the specified target. The method that generates the rule probably imposes some type of event filtering. The filtering implied by what you pass here is added on top of that filtering. Default: - No additional filtering based on an event pattern.
        :param rule_name: A name for the rule. Default: AWS CloudFormation generates a unique physical ID.

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__095d243c58d0aea9f21e7e1d1b7f1cd5ad28281a11a261aa1fbae414e5a7456f)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        options = _aws_cdk_aws_events_ceddda9d.OnEventOptions(
            target=target,
            cross_stack_scope=cross_stack_scope,
            description=description,
            event_pattern=event_pattern,
            rule_name=rule_name,
        )

        return typing.cast(_aws_cdk_aws_events_ceddda9d.Rule, jsii.invoke(self, "onCloudTrailEvent", [id, options]))

    @builtins.property
    @jsii.member(jsii_name="agent")
    @abc.abstractmethod
    def agent(self) -> IAgent:
        '''(experimental) The underlying agent for this alias.

        :stability: experimental
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="aliasArn")
    @abc.abstractmethod
    def alias_arn(self) -> builtins.str:
        '''(experimental) The ARN of the agent alias.

        :stability: experimental
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="aliasId")
    @abc.abstractmethod
    def alias_id(self) -> builtins.str:
        '''(experimental) The unique identifier of the agent alias.

        :stability: experimental
        '''
        ...


class _AgentAliasBaseProxy(
    AgentAliasBase,
    jsii.proxy_for(_aws_cdk_ceddda9d.Resource), # type: ignore[misc]
):
    @builtins.property
    @jsii.member(jsii_name="agent")
    def agent(self) -> IAgent:
        '''(experimental) The underlying agent for this alias.

        :stability: experimental
        '''
        return typing.cast(IAgent, jsii.get(self, "agent"))

    @builtins.property
    @jsii.member(jsii_name="aliasArn")
    def alias_arn(self) -> builtins.str:
        '''(experimental) The ARN of the agent alias.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "aliasArn"))

    @builtins.property
    @jsii.member(jsii_name="aliasId")
    def alias_id(self) -> builtins.str:
        '''(experimental) The unique identifier of the agent alias.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "aliasId"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, AgentAliasBase).__jsii_proxy_class__ = lambda : _AgentAliasBaseProxy


@jsii.implements(IAgent)
class AgentBase(
    _aws_cdk_ceddda9d.Resource,
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@aws-cdk/aws-bedrock-alpha.AgentBase",
):
    '''(experimental) Abstract base class for an Agent.

    Contains methods and attributes valid for Agents either created with CDK or imported.

    :stability: experimental
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        account: typing.Optional[builtins.str] = None,
        environment_from_arn: typing.Optional[builtins.str] = None,
        physical_name: typing.Optional[builtins.str] = None,
        region: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param account: The AWS account ID this resource belongs to. Default: - the resource is in the same account as the stack it belongs to
        :param environment_from_arn: ARN to deduce region and account from. The ARN is parsed and the account and region are taken from the ARN. This should be used for imported resources. Cannot be supplied together with either ``account`` or ``region``. Default: - take environment from ``account``, ``region`` parameters, or use Stack environment.
        :param physical_name: The value passed in by users to the physical name prop of the resource. - ``undefined`` implies that a physical name will be allocated by CloudFormation during deployment. - a concrete value implies a specific physical name - ``PhysicalName.GENERATE_IF_NEEDED`` is a marker that indicates that a physical will only be generated by the CDK if it is needed for cross-environment references. Otherwise, it will be allocated by CloudFormation. Default: - The physical name will be allocated by CloudFormation at deployment time
        :param region: The AWS region this resource belongs to. Default: - the resource is in the same region as the stack it belongs to
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d35aa54a631bc9ca37bc8477f55b0cac6286ce864c5768fdc4f1d2e480ec1799)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = _aws_cdk_ceddda9d.ResourceProps(
            account=account,
            environment_from_arn=environment_from_arn,
            physical_name=physical_name,
            region=region,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="grantInvoke")
    def grant_invoke(
        self,
        grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
    ) -> _aws_cdk_aws_iam_ceddda9d.Grant:
        '''(experimental) Grant invoke permissions on this agent to an IAM principal.

        :param grantee: - The IAM principal to grant invoke permissions to.

        :default:

        - Default grant configuration:
        - actions: ['bedrock:InvokeAgent']
        - resourceArns: [this.agentArn]

        :return: An IAM Grant object representing the granted permissions

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__87c56642c7d7642cf9968ade38695bd0f8a20c4c562d1611dce60c2bfb519373)
            check_type(argname="argument grantee", value=grantee, expected_type=type_hints["grantee"])
        return typing.cast(_aws_cdk_aws_iam_ceddda9d.Grant, jsii.invoke(self, "grantInvoke", [grantee]))

    @jsii.member(jsii_name="metricCount")
    def metric_count(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions_map: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        id: typing.Optional[builtins.str] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[_aws_cdk_ceddda9d.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        stack_account: typing.Optional[builtins.str] = None,
        stack_region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[_aws_cdk_aws_cloudwatch_ceddda9d.Unit] = None,
        visible: typing.Optional[builtins.bool] = None,
    ) -> _aws_cdk_aws_cloudwatch_ceddda9d.Metric:
        '''(experimental) Creates a CloudWatch metric for tracking agent invocations.

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions_map: Dimensions of the metric. Default: - No dimensions.
        :param id: Unique identifier for this metric when used in dashboard widgets. The id can be used as a variable to represent this metric in math expressions. Valid characters are letters, numbers, and underscore. The first character must be a lowercase letter. Default: - No ID
        :param label: Label for this metric when added to a Graph in a Dashboard. You can use `dynamic labels <https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/graph-dynamic-labels.html>`_ to show summary information about the entire displayed time series in the legend. For example, if you use:: [max: ${MAX}] MyMetric As the metric label, the maximum value in the visible range will be shown next to the time series name in the graph's legend. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param stack_account: Account of the stack this metric is attached to. Default: - Deployment account.
        :param stack_region: Region of the stack this metric is attached to. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Use the ``aws_cloudwatch.Stats`` helper class to construct valid input strings. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" - "tmNN.NN" | "tm(NN.NN%:NN.NN%)" - "iqm" - "wmNN.NN" | "wm(NN.NN%:NN.NN%)" - "tcNN.NN" | "tc(NN.NN%:NN.NN%)" - "tsNN.NN" | "ts(NN.NN%:NN.NN%)" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        :param visible: Whether this metric should be visible in dashboard graphs. Setting this to false is useful when you want to hide raw metrics that are used in math expressions, and show only the expression results. Default: true

        :default:

        - Default metric configuration:
        - namespace: 'AWS/Bedrock'
        - metricName: 'Invocations'
        - dimensionsMap: { AgentId: this.agentId }

        :return: A CloudWatch Metric configured for agent invocation counts

        :stability: experimental
        '''
        props = _aws_cdk_aws_cloudwatch_ceddda9d.MetricOptions(
            account=account,
            color=color,
            dimensions_map=dimensions_map,
            id=id,
            label=label,
            period=period,
            region=region,
            stack_account=stack_account,
            stack_region=stack_region,
            statistic=statistic,
            unit=unit,
            visible=visible,
        )

        return typing.cast(_aws_cdk_aws_cloudwatch_ceddda9d.Metric, jsii.invoke(self, "metricCount", [props]))

    @jsii.member(jsii_name="onEvent")
    def on_event(
        self,
        id: builtins.str,
        *,
        target: typing.Optional[_aws_cdk_aws_events_ceddda9d.IRuleTarget] = None,
        cross_stack_scope: typing.Optional[_constructs_77d1e7e8.Construct] = None,
        description: typing.Optional[builtins.str] = None,
        event_pattern: typing.Optional[typing.Union[_aws_cdk_aws_events_ceddda9d.EventPattern, typing.Dict[builtins.str, typing.Any]]] = None,
        rule_name: typing.Optional[builtins.str] = None,
    ) -> _aws_cdk_aws_events_ceddda9d.Rule:
        '''(experimental) Creates an EventBridge rule for agent events.

        :param id: - Unique identifier for the rule.
        :param target: The target to register for the event. Default: - No target is added to the rule. Use ``addTarget()`` to add a target.
        :param cross_stack_scope: The scope to use if the source of the rule and its target are in different Stacks (but in the same account & region). This helps dealing with cycles that often arise in these situations. Default: - none (the main scope will be used, even for cross-stack Events)
        :param description: A description of the rule's purpose. Default: - No description
        :param event_pattern: Additional restrictions for the event to route to the specified target. The method that generates the rule probably imposes some type of event filtering. The filtering implied by what you pass here is added on top of that filtering. Default: - No additional filtering based on an event pattern.
        :param rule_name: A name for the rule. Default: AWS CloudFormation generates a unique physical ID.

        :default:

        - Default event pattern:
        - source: ['aws.bedrock']
        - detail: { 'agent-id': [this.agentId] }

        :return: An EventBridge Rule configured for agent events

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2e2bf859b46e6177afc37c92a35764e2e847cf84005e6f9511ec77cff25aff6a)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        options = _aws_cdk_aws_events_ceddda9d.OnEventOptions(
            target=target,
            cross_stack_scope=cross_stack_scope,
            description=description,
            event_pattern=event_pattern,
            rule_name=rule_name,
        )

        return typing.cast(_aws_cdk_aws_events_ceddda9d.Rule, jsii.invoke(self, "onEvent", [id, options]))

    @builtins.property
    @jsii.member(jsii_name="agentArn")
    @abc.abstractmethod
    def agent_arn(self) -> builtins.str:
        '''(experimental) The ARN of the agent.

        :stability: experimental
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="agentId")
    @abc.abstractmethod
    def agent_id(self) -> builtins.str:
        '''(experimental) The ID of the Agent.

        :stability: experimental
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="agentVersion")
    @abc.abstractmethod
    def agent_version(self) -> builtins.str:
        '''(experimental) The version of the agent.

        :stability: experimental
        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="grantPrincipal")
    @abc.abstractmethod
    def grant_principal(self) -> _aws_cdk_aws_iam_ceddda9d.IPrincipal:
        '''(experimental) The principal to grant permissions to.

        :stability: experimental
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="role")
    @abc.abstractmethod
    def role(self) -> _aws_cdk_aws_iam_ceddda9d.IRole:
        '''(experimental) The IAM role associated to the agent.

        :stability: experimental
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="kmsKey")
    @abc.abstractmethod
    def kms_key(self) -> typing.Optional[_aws_cdk_aws_kms_ceddda9d.IKey]:
        '''(experimental) Optional KMS encryption key associated with this agent.

        :stability: experimental
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="lastUpdated")
    @abc.abstractmethod
    def last_updated(self) -> typing.Optional[builtins.str]:
        '''(experimental) When this agent was last updated.

        :stability: experimental
        '''
        ...


class _AgentBaseProxy(
    AgentBase,
    jsii.proxy_for(_aws_cdk_ceddda9d.Resource), # type: ignore[misc]
):
    @builtins.property
    @jsii.member(jsii_name="agentArn")
    def agent_arn(self) -> builtins.str:
        '''(experimental) The ARN of the agent.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "agentArn"))

    @builtins.property
    @jsii.member(jsii_name="agentId")
    def agent_id(self) -> builtins.str:
        '''(experimental) The ID of the Agent.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "agentId"))

    @builtins.property
    @jsii.member(jsii_name="agentVersion")
    def agent_version(self) -> builtins.str:
        '''(experimental) The version of the agent.

        :stability: experimental
        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "agentVersion"))

    @builtins.property
    @jsii.member(jsii_name="grantPrincipal")
    def grant_principal(self) -> _aws_cdk_aws_iam_ceddda9d.IPrincipal:
        '''(experimental) The principal to grant permissions to.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_iam_ceddda9d.IPrincipal, jsii.get(self, "grantPrincipal"))

    @builtins.property
    @jsii.member(jsii_name="role")
    def role(self) -> _aws_cdk_aws_iam_ceddda9d.IRole:
        '''(experimental) The IAM role associated to the agent.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_iam_ceddda9d.IRole, jsii.get(self, "role"))

    @builtins.property
    @jsii.member(jsii_name="kmsKey")
    def kms_key(self) -> typing.Optional[_aws_cdk_aws_kms_ceddda9d.IKey]:
        '''(experimental) Optional KMS encryption key associated with this agent.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[_aws_cdk_aws_kms_ceddda9d.IKey], jsii.get(self, "kmsKey"))

    @builtins.property
    @jsii.member(jsii_name="lastUpdated")
    def last_updated(self) -> typing.Optional[builtins.str]:
        '''(experimental) When this agent was last updated.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "lastUpdated"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, AgentBase).__jsii_proxy_class__ = lambda : _AgentBaseProxy


@jsii.implements(IBedrockInvokable)
class BedrockFoundationModel(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-bedrock-alpha.BedrockFoundationModel",
):
    '''(experimental) Bedrock models.

    If you need to use a model name that doesn't exist as a static member, you
    can instantiate a ``BedrockFoundationModel`` object, e.g: ``new BedrockFoundationModel('my-model')``.

    :stability: experimental
    :exampleMetadata: fixture=default infused

    Example::

        parser_function = lambda_.Function(self, "ParserFunction",
            runtime=lambda_.Runtime.PYTHON_3_10,
            handler="index.handler",
            code=lambda_.Code.from_asset("lambda")
        )
        
        agent = bedrock.Agent(self, "Agent",
            foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
            instruction="You are a helpful assistant.",
            prompt_override_configuration=bedrock.PromptOverrideConfiguration.with_custom_parser(
                parser=parser_function,
                pre_processing_step=bedrock.PromptPreProcessingConfigCustomParser(
                    step_type=bedrock.AgentStepType.PRE_PROCESSING,
                    use_custom_parser=True
                )
            )
        )
    '''

    def __init__(
        self,
        value: builtins.str,
        *,
        legacy: typing.Optional[builtins.bool] = None,
        optimized_for_agents: typing.Optional[builtins.bool] = None,
        supported_vector_type: typing.Optional[typing.Sequence[VectorType]] = None,
        supports_agents: typing.Optional[builtins.bool] = None,
        supports_cross_region: typing.Optional[builtins.bool] = None,
        supports_knowledge_base: typing.Optional[builtins.bool] = None,
        vector_dimensions: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''
        :param value: -
        :param legacy: (experimental) https://docs.aws.amazon.com/bedrock/latest/userguide/model-lifecycle.html A version is marked Legacy when there is a more recent version which provides superior performance. Amazon Bedrock sets an EOL date for Legacy versions. Default: - false
        :param optimized_for_agents: (experimental) Currently, some of the offered models are optimized with prompts/parsers fine-tuned for integrating with the agents architecture. When true, the model has been specifically optimized for agent interactions. Default: - false
        :param supported_vector_type: (experimental) Embeddings models have different supported vector types. Defines whether the model supports floating-point or binary vectors. Default: - undefined
        :param supports_agents: (experimental) Bedrock Agents can use this model. When true, the model can be used with Bedrock Agents for automated task execution. Default: - false
        :param supports_cross_region: (experimental) Can be used with a Cross-Region Inference Profile. When true, the model supports inference across different AWS regions. Default: - false
        :param supports_knowledge_base: (experimental) Bedrock Knowledge Base can use this model. When true, the model can be used for knowledge base operations. Default: - false
        :param vector_dimensions: (experimental) Embedding models have different vector dimensions. Only applicable for embedding models. Defines the dimensionality of the vector embeddings. Default: - undefined

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c72ea354d94de945db88ea8f90368338db6b71c53c1de96d6b6ff8e5ca8bd139)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        props = BedrockFoundationModelProps(
            legacy=legacy,
            optimized_for_agents=optimized_for_agents,
            supported_vector_type=supported_vector_type,
            supports_agents=supports_agents,
            supports_cross_region=supports_cross_region,
            supports_knowledge_base=supports_knowledge_base,
            vector_dimensions=vector_dimensions,
        )

        jsii.create(self.__class__, self, [value, props])

    @jsii.member(jsii_name="fromCdkFoundationModel")
    @builtins.classmethod
    def from_cdk_foundation_model(
        cls,
        model_id: _aws_cdk_aws_bedrock_ceddda9d.FoundationModel,
        *,
        legacy: typing.Optional[builtins.bool] = None,
        optimized_for_agents: typing.Optional[builtins.bool] = None,
        supported_vector_type: typing.Optional[typing.Sequence[VectorType]] = None,
        supports_agents: typing.Optional[builtins.bool] = None,
        supports_cross_region: typing.Optional[builtins.bool] = None,
        supports_knowledge_base: typing.Optional[builtins.bool] = None,
        vector_dimensions: typing.Optional[jsii.Number] = None,
    ) -> "BedrockFoundationModel":
        '''(experimental) Creates a BedrockFoundationModel from a FoundationModel.

        Use this method when you have a foundation model from the CDK.

        :param model_id: - The foundation model.
        :param legacy: (experimental) https://docs.aws.amazon.com/bedrock/latest/userguide/model-lifecycle.html A version is marked Legacy when there is a more recent version which provides superior performance. Amazon Bedrock sets an EOL date for Legacy versions. Default: - false
        :param optimized_for_agents: (experimental) Currently, some of the offered models are optimized with prompts/parsers fine-tuned for integrating with the agents architecture. When true, the model has been specifically optimized for agent interactions. Default: - false
        :param supported_vector_type: (experimental) Embeddings models have different supported vector types. Defines whether the model supports floating-point or binary vectors. Default: - undefined
        :param supports_agents: (experimental) Bedrock Agents can use this model. When true, the model can be used with Bedrock Agents for automated task execution. Default: - false
        :param supports_cross_region: (experimental) Can be used with a Cross-Region Inference Profile. When true, the model supports inference across different AWS regions. Default: - false
        :param supports_knowledge_base: (experimental) Bedrock Knowledge Base can use this model. When true, the model can be used for knowledge base operations. Default: - false
        :param vector_dimensions: (experimental) Embedding models have different vector dimensions. Only applicable for embedding models. Defines the dimensionality of the vector embeddings. Default: - undefined

        :return: A new BedrockFoundationModel instance

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__21746783fde44094590eeb2e0bf9318cd7c94a7d9fdbbc276e7c07e9882807b9)
            check_type(argname="argument model_id", value=model_id, expected_type=type_hints["model_id"])
        props = BedrockFoundationModelProps(
            legacy=legacy,
            optimized_for_agents=optimized_for_agents,
            supported_vector_type=supported_vector_type,
            supports_agents=supports_agents,
            supports_cross_region=supports_cross_region,
            supports_knowledge_base=supports_knowledge_base,
            vector_dimensions=vector_dimensions,
        )

        return typing.cast("BedrockFoundationModel", jsii.sinvoke(cls, "fromCdkFoundationModel", [model_id, props]))

    @jsii.member(jsii_name="fromCdkFoundationModelId")
    @builtins.classmethod
    def from_cdk_foundation_model_id(
        cls,
        model_id: _aws_cdk_aws_bedrock_ceddda9d.FoundationModelIdentifier,
        *,
        legacy: typing.Optional[builtins.bool] = None,
        optimized_for_agents: typing.Optional[builtins.bool] = None,
        supported_vector_type: typing.Optional[typing.Sequence[VectorType]] = None,
        supports_agents: typing.Optional[builtins.bool] = None,
        supports_cross_region: typing.Optional[builtins.bool] = None,
        supports_knowledge_base: typing.Optional[builtins.bool] = None,
        vector_dimensions: typing.Optional[jsii.Number] = None,
    ) -> "BedrockFoundationModel":
        '''(experimental) Creates a BedrockFoundationModel from a FoundationModelIdentifier.

        Use this method when you have a model identifier from the CDK.

        :param model_id: - The foundation model identifier.
        :param legacy: (experimental) https://docs.aws.amazon.com/bedrock/latest/userguide/model-lifecycle.html A version is marked Legacy when there is a more recent version which provides superior performance. Amazon Bedrock sets an EOL date for Legacy versions. Default: - false
        :param optimized_for_agents: (experimental) Currently, some of the offered models are optimized with prompts/parsers fine-tuned for integrating with the agents architecture. When true, the model has been specifically optimized for agent interactions. Default: - false
        :param supported_vector_type: (experimental) Embeddings models have different supported vector types. Defines whether the model supports floating-point or binary vectors. Default: - undefined
        :param supports_agents: (experimental) Bedrock Agents can use this model. When true, the model can be used with Bedrock Agents for automated task execution. Default: - false
        :param supports_cross_region: (experimental) Can be used with a Cross-Region Inference Profile. When true, the model supports inference across different AWS regions. Default: - false
        :param supports_knowledge_base: (experimental) Bedrock Knowledge Base can use this model. When true, the model can be used for knowledge base operations. Default: - false
        :param vector_dimensions: (experimental) Embedding models have different vector dimensions. Only applicable for embedding models. Defines the dimensionality of the vector embeddings. Default: - undefined

        :return: A new BedrockFoundationModel instance

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__99e6623f3ac6adc2fae17fdb40358d156dfbc20814dd317a4e724bf44ede364e)
            check_type(argname="argument model_id", value=model_id, expected_type=type_hints["model_id"])
        props = BedrockFoundationModelProps(
            legacy=legacy,
            optimized_for_agents=optimized_for_agents,
            supported_vector_type=supported_vector_type,
            supports_agents=supports_agents,
            supports_cross_region=supports_cross_region,
            supports_knowledge_base=supports_knowledge_base,
            vector_dimensions=vector_dimensions,
        )

        return typing.cast("BedrockFoundationModel", jsii.sinvoke(cls, "fromCdkFoundationModelId", [model_id, props]))

    @jsii.member(jsii_name="asArn")
    def as_arn(self) -> builtins.str:
        '''(experimental) Returns the ARN of the foundation model in the following format: ``arn:${Partition}:bedrock:${Region}::foundation-model/${ResourceId}``.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "asArn", []))

    @jsii.member(jsii_name="asIModel")
    def as_i_model(self) -> _aws_cdk_aws_bedrock_ceddda9d.IModel:
        '''(experimental) Returns the IModel.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_bedrock_ceddda9d.IModel, jsii.invoke(self, "asIModel", []))

    @jsii.member(jsii_name="grantInvoke")
    def grant_invoke(
        self,
        grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
    ) -> _aws_cdk_aws_iam_ceddda9d.Grant:
        '''(experimental) Gives the appropriate policies to invoke and use the Foundation Model in the stack region.

        :param grantee: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ffad53dee884df6b2476ed3cd5b2db73771ff68f85c38de0570b5b16841bcd21)
            check_type(argname="argument grantee", value=grantee, expected_type=type_hints["grantee"])
        return typing.cast(_aws_cdk_aws_iam_ceddda9d.Grant, jsii.invoke(self, "grantInvoke", [grantee]))

    @jsii.member(jsii_name="grantInvokeAllRegions")
    def grant_invoke_all_regions(
        self,
        grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
    ) -> _aws_cdk_aws_iam_ceddda9d.Grant:
        '''(experimental) Gives the appropriate policies to invoke and use the Foundation Model in all regions.

        :param grantee: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c036ac01b7085698fb4634261fe75aea9d21d24dd6c4187d1454cda55c36912a)
            check_type(argname="argument grantee", value=grantee, expected_type=type_hints["grantee"])
        return typing.cast(_aws_cdk_aws_iam_ceddda9d.Grant, jsii.invoke(self, "grantInvokeAllRegions", [grantee]))

    @jsii.member(jsii_name="toString")
    def to_string(self) -> builtins.str:
        '''(experimental) Returns a string representation of an object.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "toString", []))

    @jsii.python.classproperty
    @jsii.member(jsii_name="AI21_JAMBA_1_5_LARGE_V1")
    def AI21_JAMBA_1_5_LARGE_V1(cls) -> "BedrockFoundationModel":
        '''(experimental) AI21's Jamba 1.5 Large model optimized for text generation tasks. Suitable for complex language understanding and generation tasks.

        Features:

        - Supports Bedrock Agents integration
        - Optimized for natural language processing
        - Best for: Content generation, summarization, and complex text analysis

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "AI21_JAMBA_1_5_LARGE_V1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="AI21_JAMBA_1_5_MINI_V1")
    def AI21_JAMBA_1_5_MINI_V1(cls) -> "BedrockFoundationModel":
        '''(experimental) AI21's Jamba 1.5 Mini model, a lighter version optimized for faster processing. Balances performance with efficiency for general text tasks.

        Features:

        - Supports Bedrock Agents integration
        - Faster response times compared to larger models
        - Best for: Quick text processing, basic content generation

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "AI21_JAMBA_1_5_MINI_V1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="AI21_JAMBA_INSTRUCT_V1")
    def AI21_JAMBA_INSTRUCT_V1(cls) -> "BedrockFoundationModel":
        '''(experimental) AI21's Jamba Instruct model, specifically designed for instruction-following tasks. Optimized for understanding and executing specific instructions.

        Features:

        - Supports Bedrock Agents integration
        - Enhanced instruction understanding
        - Best for: Task-specific instructions, command processing

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "AI21_JAMBA_INSTRUCT_V1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="AMAZON_NOVA_LITE_V1")
    def AMAZON_NOVA_LITE_V1(cls) -> "BedrockFoundationModel":
        '''(experimental) Amazon's Nova Lite model, balancing performance with efficiency.

        Features:

        - Supports Bedrock Agents integration
        - Cross-region support
        - Optimized for agents
        - Best for: General-purpose language tasks, moderate complexity

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "AMAZON_NOVA_LITE_V1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="AMAZON_NOVA_MICRO_V1")
    def AMAZON_NOVA_MICRO_V1(cls) -> "BedrockFoundationModel":
        '''(experimental) Amazon's Nova Micro model, a lightweight model optimized for efficiency.

        Features:

        - Supports Bedrock Agents integration
        - Cross-region support
        - Optimized for agents
        - Best for: Quick processing tasks, basic language understanding

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "AMAZON_NOVA_MICRO_V1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="AMAZON_NOVA_PREMIER_V1")
    def AMAZON_NOVA_PREMIER_V1(cls) -> "BedrockFoundationModel":
        '''(experimental) Amazon's Nova Premier model, the most advanced in the Nova series.

        Features:

        - Supports Bedrock Agents integration
        - Cross-region support
        - Optimized for agents
        - Best for: High-end applications, complex analysis, premium performance

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "AMAZON_NOVA_PREMIER_V1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="AMAZON_NOVA_PRO_V1")
    def AMAZON_NOVA_PRO_V1(cls) -> "BedrockFoundationModel":
        '''(experimental) Amazon's Nova Pro model, offering advanced capabilities for complex tasks.

        Features:

        - Supports Bedrock Agents integration
        - Cross-region support
        - Optimized for agents
        - Best for: Complex language tasks, professional applications

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "AMAZON_NOVA_PRO_V1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="AMAZON_TITAN_PREMIER_V1_0")
    def AMAZON_TITAN_PREMIER_V1_0(cls) -> "BedrockFoundationModel":
        '''(experimental) Amazon's Titan Text Premier model, designed for high-quality text generation. Offers enhanced capabilities for complex language tasks.

        Features:

        - Supports Bedrock Agents integration
        - Advanced language understanding
        - Best for: Complex content generation, detailed analysis

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "AMAZON_TITAN_PREMIER_V1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="AMAZON_TITAN_TEXT_EXPRESS_V1")
    def AMAZON_TITAN_TEXT_EXPRESS_V1(cls) -> "BedrockFoundationModel":
        '''(experimental) Amazon's Titan Text Express model, optimized for fast text generation. Provides quick responses while maintaining good quality output.

        Features:

        - Supports Bedrock Agents integration
        - Fast response times
        - Best for: Real-time applications, chatbots, quick content generation

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "AMAZON_TITAN_TEXT_EXPRESS_V1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ANTHROPIC_CLAUDE_3_5_HAIKU_V1_0")
    def ANTHROPIC_CLAUDE_3_5_HAIKU_V1_0(cls) -> "BedrockFoundationModel":
        '''(experimental) Anthropic's Claude 3.5 Haiku model, optimized for quick responses. Lightweight model focused on speed and efficiency.

        Features:

        - Supports Bedrock Agents integration
        - Cross-region support
        - Optimized for agents
        - Best for: Fast responses, lightweight processing

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "ANTHROPIC_CLAUDE_3_5_HAIKU_V1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ANTHROPIC_CLAUDE_3_5_SONNET_V1_0")
    def ANTHROPIC_CLAUDE_3_5_SONNET_V1_0(cls) -> "BedrockFoundationModel":
        '''(experimental) Anthropic's Claude 3.5 Sonnet V1 model, balanced performance model. Offers good balance between performance and efficiency.

        Features:

        - Supports Bedrock Agents integration
        - Cross-region support
        - Optimized for agents
        - Best for: General language tasks, balanced performance

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "ANTHROPIC_CLAUDE_3_5_SONNET_V1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ANTHROPIC_CLAUDE_3_5_SONNET_V2_0")
    def ANTHROPIC_CLAUDE_3_5_SONNET_V2_0(cls) -> "BedrockFoundationModel":
        '''(experimental) Anthropic's Claude 3.5 Sonnet V2 model, optimized for agent interactions. Enhanced version with improved performance and capabilities.

        Features:

        - Supports Bedrock Agents integration
        - Cross-region support
        - Optimized for agents
        - Best for: Agent-based applications, complex dialogue

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "ANTHROPIC_CLAUDE_3_5_SONNET_V2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ANTHROPIC_CLAUDE_3_7_SONNET_V1_0")
    def ANTHROPIC_CLAUDE_3_7_SONNET_V1_0(cls) -> "BedrockFoundationModel":
        '''(experimental) Anthropic's Claude 3.7 Sonnet model, latest in the Claude 3 series. Advanced language model with enhanced capabilities.

        Features:

        - Supports Bedrock Agents integration
        - Cross-region support
        - Best for: Complex reasoning, analysis, and content generation

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "ANTHROPIC_CLAUDE_3_7_SONNET_V1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ANTHROPIC_CLAUDE_HAIKU_V1_0")
    def ANTHROPIC_CLAUDE_HAIKU_V1_0(cls) -> "BedrockFoundationModel":
        '''(experimental) Anthropic's Claude Haiku model, optimized for efficiency. Fast and efficient model for lightweight tasks.

        Features:

        - Supports Bedrock Agents integration
        - Cross-region support
        - Optimized for agents
        - Best for: Quick responses, simple tasks

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "ANTHROPIC_CLAUDE_HAIKU_V1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ANTHROPIC_CLAUDE_INSTANT_V1_2")
    def ANTHROPIC_CLAUDE_INSTANT_V1_2(cls) -> "BedrockFoundationModel":
        '''(experimental) Anthropic's Claude Instant V1.2 model, legacy version. Fast and efficient model optimized for quick responses.

        Features:

        - Supports Bedrock Agents integration
        - Legacy model with EOL date
        - Optimized for agents
        - Best for: Quick responses, simple tasks, legacy applications

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "ANTHROPIC_CLAUDE_INSTANT_V1_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ANTHROPIC_CLAUDE_OPUS_V1_0")
    def ANTHROPIC_CLAUDE_OPUS_V1_0(cls) -> "BedrockFoundationModel":
        '''(experimental) Anthropic's Claude Opus model, designed for advanced tasks. High-performance model with extensive capabilities.

        Features:

        - Supports Bedrock Agents integration
        - Optimized for agents
        - Best for: Complex reasoning, research, and analysis

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "ANTHROPIC_CLAUDE_OPUS_V1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ANTHROPIC_CLAUDE_SONNET_V1_0")
    def ANTHROPIC_CLAUDE_SONNET_V1_0(cls) -> "BedrockFoundationModel":
        '''(experimental) Anthropic's Claude Sonnet model, legacy version. Balanced model for general-purpose tasks.

        Features:

        - Supports Bedrock Agents integration
        - Cross-region support
        - Legacy model with EOL date
        - Best for: General language tasks, standard applications

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "ANTHROPIC_CLAUDE_SONNET_V1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ANTHROPIC_CLAUDE_V2")
    def ANTHROPIC_CLAUDE_V2(cls) -> "BedrockFoundationModel":
        '''(experimental) Anthropic's Claude V2 model, legacy version. Original Claude V2 model with broad capabilities.

        Features:

        - Supports Bedrock Agents integration
        - Legacy model with EOL date
        - Optimized for agents
        - Best for: General language tasks, legacy applications

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "ANTHROPIC_CLAUDE_V2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ANTHROPIC_CLAUDE_V2_1")
    def ANTHROPIC_CLAUDE_V2_1(cls) -> "BedrockFoundationModel":
        '''(experimental) Anthropic's Claude V2.1 model, legacy version. Improved version of Claude V2 with enhanced capabilities.

        Features:

        - Supports Bedrock Agents integration
        - Legacy model with EOL date
        - Optimized for agents
        - Best for: General language tasks, legacy applications

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "ANTHROPIC_CLAUDE_V2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="COHERE_EMBED_ENGLISH_V3")
    def COHERE_EMBED_ENGLISH_V3(cls) -> "BedrockFoundationModel":
        '''(experimental) Cohere's English embedding model, optimized for English text embeddings. Specialized for semantic understanding of English content.

        Features:

        - Supports Knowledge Base integration
        - 1024-dimensional vectors
        - Supports both floating-point and binary vectors
        - Best for: English text embeddings, semantic search, content similarity

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "COHERE_EMBED_ENGLISH_V3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="COHERE_EMBED_MULTILINGUAL_V3")
    def COHERE_EMBED_MULTILINGUAL_V3(cls) -> "BedrockFoundationModel":
        '''(experimental) Cohere's Multilingual embedding model, supporting multiple languages. Enables semantic understanding across different languages.

        Features:

        - Supports Knowledge Base integration
        - 1024-dimensional vectors
        - Supports both floating-point and binary vectors
        - Best for: Cross-lingual embeddings, multilingual semantic search

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "COHERE_EMBED_MULTILINGUAL_V3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DEEPSEEK_R1_V1")
    def DEEPSEEK_R1_V1(cls) -> "BedrockFoundationModel":
        '''(experimental) Deepseek's R1 model, designed for general language understanding. Balanced model for various language tasks.

        Features:

        - Supports Bedrock Agents integration
        - Cross-region support
        - Best for: General language tasks, content generation

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "DEEPSEEK_R1_V1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_LLAMA_3_1_70B_INSTRUCT_V1")
    def META_LLAMA_3_1_70_B_INSTRUCT_V1(cls) -> "BedrockFoundationModel":
        '''(experimental) Meta's Llama 3 70B Instruct model, large-scale instruction model. High-capacity model for complex language understanding.

        Features:

        - Supports Bedrock Agents integration
        - Cross-region support
        - Best for: Complex instructions, advanced language tasks

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "META_LLAMA_3_1_70B_INSTRUCT_V1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_LLAMA_3_1_8B_INSTRUCT_V1")
    def META_LLAMA_3_1_8_B_INSTRUCT_V1(cls) -> "BedrockFoundationModel":
        '''(experimental) Meta's Llama 3 1.8B Instruct model, compact instruction-following model. Efficient model optimized for instruction-based tasks.

        Features:

        - Supports Bedrock Agents integration
        - Cross-region support
        - Best for: Lightweight instruction processing, quick responses

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "META_LLAMA_3_1_8B_INSTRUCT_V1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_LLAMA_3_2_11B_INSTRUCT_V1")
    def META_LLAMA_3_2_11_B_INSTRUCT_V1(cls) -> "BedrockFoundationModel":
        '''(experimental) Meta's Llama 3.2 11B Instruct model, mid-sized instruction model. Balanced model for general instruction processing.

        Features:

        - Supports Bedrock Agents integration
        - Cross-region support
        - Best for: General instruction tasks, balanced performance

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "META_LLAMA_3_2_11B_INSTRUCT_V1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_LLAMA_3_2_1B_INSTRUCT_V1")
    def META_LLAMA_3_2_1_B_INSTRUCT_V1(cls) -> "BedrockFoundationModel":
        '''(experimental) Meta's Llama 3.2 1B Instruct model, ultra-lightweight model. Most compact model in the Llama 3.2 series.

        Features:

        - Supports Bedrock Agents integration
        - Cross-region support
        - Best for: Simple instructions, fastest response times

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "META_LLAMA_3_2_1B_INSTRUCT_V1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_LLAMA_3_2_3B_INSTRUCT_V1")
    def META_LLAMA_3_2_3_B_INSTRUCT_V1(cls) -> "BedrockFoundationModel":
        '''(experimental) Meta's Llama 3.2 3B Instruct model, compact efficient model. Lightweight model for basic instruction processing.

        Features:

        - Supports Bedrock Agents integration
        - Cross-region support
        - Best for: Basic instructions, efficient processing

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "META_LLAMA_3_2_3B_INSTRUCT_V1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_LLAMA_3_3_70B_INSTRUCT_V1")
    def META_LLAMA_3_3_70_B_INSTRUCT_V1(cls) -> "BedrockFoundationModel":
        '''(experimental) Meta's Llama 3.3 70B Instruct model, latest large-scale model. Advanced model with enhanced capabilities.

        Features:

        - Supports Bedrock Agents integration
        - Cross-region support
        - Best for: Complex reasoning, advanced language tasks

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "META_LLAMA_3_3_70B_INSTRUCT_V1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_LLAMA_4_MAVERICK_17B_INSTRUCT_V1")
    def META_LLAMA_4_MAVERICK_17_B_INSTRUCT_V1(cls) -> "BedrockFoundationModel":
        '''(experimental) Meta's Llama 4 Maverick 17B Instruct model, innovative mid-sized model. Specialized for creative and dynamic responses.

        Features:

        - Supports Bedrock Agents integration
        - Cross-region support
        - Best for: Creative tasks, innovative solutions

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "META_LLAMA_4_MAVERICK_17B_INSTRUCT_V1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_LLAMA_4_SCOUT_17B_INSTRUCT_V1")
    def META_LLAMA_4_SCOUT_17_B_INSTRUCT_V1(cls) -> "BedrockFoundationModel":
        '''(experimental) Meta's Llama 4 Scout 17B Instruct model, analytical mid-sized model. Focused on precise and analytical responses.

        Features:

        - Supports Bedrock Agents integration
        - Cross-region support
        - Best for: Analytical tasks, precise responses

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "META_LLAMA_4_SCOUT_17B_INSTRUCT_V1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MISTRAL_7B_INSTRUCT_V0")
    def MISTRAL_7_B_INSTRUCT_V0(cls) -> "BedrockFoundationModel":
        '''(experimental) Mistral's 7B Instruct model, efficient instruction-following model. Balanced performance for instruction processing.

        Features:

        - Supports Bedrock Agents integration
        - Optimized for instruction tasks
        - Best for: General instruction processing, balanced performance

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "MISTRAL_7B_INSTRUCT_V0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MISTRAL_LARGE_2402_V1")
    def MISTRAL_LARGE_2402_V1(cls) -> "BedrockFoundationModel":
        '''(experimental) Mistral's Large 2402 model, high-capacity language model. Advanced model for complex language understanding.

        Features:

        - Supports Bedrock Agents integration
        - Enhanced language capabilities
        - Best for: Complex reasoning, detailed analysis

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "MISTRAL_LARGE_2402_V1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MISTRAL_LARGE_2407_V1")
    def MISTRAL_LARGE_2407_V1(cls) -> "BedrockFoundationModel":
        '''(experimental) Mistral's Large 2407 model, updated large-scale model. Enhanced version with improved capabilities.

        Features:

        - Supports Bedrock Agents integration
        - Advanced language processing
        - Best for: Sophisticated language tasks, complex analysis

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "MISTRAL_LARGE_2407_V1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MISTRAL_MIXTRAL_8X7B_INSTRUCT_V0")
    def MISTRAL_MIXTRAL_8_X7_B_INSTRUCT_V0(cls) -> "BedrockFoundationModel":
        '''(experimental) Mistral's Mixtral 8x7B Instruct model, mixture-of-experts architecture. Advanced model combining multiple expert networks.

        Features:

        - Supports Bedrock Agents integration
        - Specialized expert networks
        - Best for: Complex tasks, diverse language understanding

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "MISTRAL_MIXTRAL_8X7B_INSTRUCT_V0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MISTRAL_PIXTRAL_LARGE_2502_V1")
    def MISTRAL_PIXTRAL_LARGE_2502_V1(cls) -> "BedrockFoundationModel":
        '''(experimental) Mistral's Pixtral Large 2502 model, specialized large model. Advanced model with cross-region support.

        Features:

        - Supports Bedrock Agents integration
        - Cross-region support
        - Best for: Advanced language tasks, distributed applications

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "MISTRAL_PIXTRAL_LARGE_2502_V1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MISTRAL_SMALL_2402_V1")
    def MISTRAL_SMALL_2402_V1(cls) -> "BedrockFoundationModel":
        '''(experimental) Mistral's Small 2402 model, compact efficient model. Optimized for quick responses and efficiency.

        Features:

        - Supports Bedrock Agents integration
        - Efficient processing
        - Best for: Quick responses, basic language tasks

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "MISTRAL_SMALL_2402_V1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="TITAN_EMBED_TEXT_V1")
    def TITAN_EMBED_TEXT_V1(cls) -> "BedrockFoundationModel":
        '''(experimental) Amazon's Titan Embed Text V1 model for text embeddings.

        Features:

        - Supports Knowledge Base integration
        - 1536-dimensional vectors
        - Floating-point vector type
        - Best for: Text embeddings, semantic search, document similarity

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "TITAN_EMBED_TEXT_V1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="TITAN_EMBED_TEXT_V2_1024")
    def TITAN_EMBED_TEXT_V2_1024(cls) -> "BedrockFoundationModel":
        '''(experimental) Amazon's Titan Embed Text V2 model with 1024-dimensional vectors.

        Features:

        - Supports Knowledge Base integration
        - 1024-dimensional vectors
        - Supports both floating-point and binary vectors
        - Best for: High-dimensional embeddings, advanced semantic search

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "TITAN_EMBED_TEXT_V2_1024"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="TITAN_EMBED_TEXT_V2_256")
    def TITAN_EMBED_TEXT_V2_256(cls) -> "BedrockFoundationModel":
        '''(experimental) Amazon's Titan Embed Text V2 model with 256-dimensional vectors.

        Features:

        - Supports Knowledge Base integration
        - 256-dimensional vectors
        - Supports both floating-point and binary vectors
        - Best for: Efficient embeddings with lower dimensionality

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "TITAN_EMBED_TEXT_V2_256"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="TITAN_EMBED_TEXT_V2_512")
    def TITAN_EMBED_TEXT_V2_512(cls) -> "BedrockFoundationModel":
        '''(experimental) Amazon's Titan Embed Text V2 model with 512-dimensional vectors.

        Features:

        - Supports Knowledge Base integration
        - 512-dimensional vectors
        - Supports both floating-point and binary vectors
        - Best for: Balanced performance and dimensionality

        :stability: experimental
        '''
        return typing.cast("BedrockFoundationModel", jsii.sget(cls, "TITAN_EMBED_TEXT_V2_512"))

    @builtins.property
    @jsii.member(jsii_name="invokableArn")
    def invokable_arn(self) -> builtins.str:
        '''(experimental) The ARN used for invoking the model.

        This is the same as modelArn for foundation models.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "invokableArn"))

    @builtins.property
    @jsii.member(jsii_name="modelArn")
    def model_arn(self) -> builtins.str:
        '''(experimental) The ARN of the foundation model.

        Format: arn:${Partition}:bedrock:${Region}::foundation-model/${ResourceId}

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "modelArn"))

    @builtins.property
    @jsii.member(jsii_name="modelId")
    def model_id(self) -> builtins.str:
        '''(experimental) The unique identifier of the foundation model.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "modelId"))

    @builtins.property
    @jsii.member(jsii_name="supportsAgents")
    def supports_agents(self) -> builtins.bool:
        '''(experimental) Whether this model supports integration with Bedrock Agents.

        When true, the model can be used with Bedrock Agents for automated task execution.

        :stability: experimental
        '''
        return typing.cast(builtins.bool, jsii.get(self, "supportsAgents"))

    @builtins.property
    @jsii.member(jsii_name="supportsCrossRegion")
    def supports_cross_region(self) -> builtins.bool:
        '''(experimental) Whether this model supports cross-region inference.

        When true, the model can be used with Cross-Region Inference Profiles.

        :stability: experimental
        '''
        return typing.cast(builtins.bool, jsii.get(self, "supportsCrossRegion"))

    @builtins.property
    @jsii.member(jsii_name="supportsKnowledgeBase")
    def supports_knowledge_base(self) -> builtins.bool:
        '''(experimental) Whether this model supports integration with Bedrock Knowledge Base.

        When true, the model can be used for knowledge base operations.

        :stability: experimental
        '''
        return typing.cast(builtins.bool, jsii.get(self, "supportsKnowledgeBase"))

    @builtins.property
    @jsii.member(jsii_name="supportedVectorType")
    def supported_vector_type(self) -> typing.Optional[typing.List[VectorType]]:
        '''(experimental) The vector types supported by this model for embeddings.

        Defines whether the model supports floating-point or binary vectors.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[VectorType]], jsii.get(self, "supportedVectorType"))

    @builtins.property
    @jsii.member(jsii_name="vectorDimensions")
    def vector_dimensions(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The dimensionality of the vector embeddings produced by this model.

        Only applicable for embedding models.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "vectorDimensions"))


@jsii.data_type(
    jsii_type="@aws-cdk/aws-bedrock-alpha.PromptKnowledgeBaseResponseGenerationConfigCustomParser",
    jsii_struct_bases=[PromptStepConfigBase],
    name_mapping={
        "step_type": "stepType",
        "custom_prompt_template": "customPromptTemplate",
        "inference_config": "inferenceConfig",
        "step_enabled": "stepEnabled",
        "use_custom_parser": "useCustomParser",
    },
)
class PromptKnowledgeBaseResponseGenerationConfigCustomParser(PromptStepConfigBase):
    def __init__(
        self,
        *,
        step_type: AgentStepType,
        custom_prompt_template: typing.Optional[builtins.str] = None,
        inference_config: typing.Optional[typing.Union[InferenceConfiguration, typing.Dict[builtins.str, typing.Any]]] = None,
        step_enabled: typing.Optional[builtins.bool] = None,
        use_custom_parser: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''(experimental) Configuration for the knowledge base response generation step.

        :param step_type: (experimental) The type of step this configuration applies to.
        :param custom_prompt_template: (experimental) The custom prompt template to be used. Default: - The default prompt template will be used.
        :param inference_config: (experimental) The inference configuration parameters to use. Default: undefined - Default inference configuration will be used
        :param step_enabled: (experimental) Whether to enable or skip this step in the agent sequence. Default: - The default state for each step type is as follows. PRE_PROCESSING – ENABLED ORCHESTRATION – ENABLED KNOWLEDGE_BASE_RESPONSE_GENERATION – ENABLED POST_PROCESSING – DISABLED
        :param use_custom_parser: (experimental) Whether to use the custom Lambda parser defined for the sequence. Default: - false

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_bedrock_alpha as bedrock_alpha
            
            prompt_knowledge_base_response_generation_config_custom_parser = bedrock_alpha.PromptKnowledgeBaseResponseGenerationConfigCustomParser(
                step_type=bedrock_alpha.AgentStepType.PRE_PROCESSING,
            
                # the properties below are optional
                custom_prompt_template="customPromptTemplate",
                inference_config=bedrock_alpha.InferenceConfiguration(
                    maximum_length=123,
                    stop_sequences=["stopSequences"],
                    temperature=123,
                    top_k=123,
                    top_p=123
                ),
                step_enabled=False,
                use_custom_parser=False
            )
        '''
        if isinstance(inference_config, dict):
            inference_config = InferenceConfiguration(**inference_config)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ff59ff4e8f8241de7bd05809f66c99d45b60f152560b50867f223721c27d2bc3)
            check_type(argname="argument step_type", value=step_type, expected_type=type_hints["step_type"])
            check_type(argname="argument custom_prompt_template", value=custom_prompt_template, expected_type=type_hints["custom_prompt_template"])
            check_type(argname="argument inference_config", value=inference_config, expected_type=type_hints["inference_config"])
            check_type(argname="argument step_enabled", value=step_enabled, expected_type=type_hints["step_enabled"])
            check_type(argname="argument use_custom_parser", value=use_custom_parser, expected_type=type_hints["use_custom_parser"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "step_type": step_type,
        }
        if custom_prompt_template is not None:
            self._values["custom_prompt_template"] = custom_prompt_template
        if inference_config is not None:
            self._values["inference_config"] = inference_config
        if step_enabled is not None:
            self._values["step_enabled"] = step_enabled
        if use_custom_parser is not None:
            self._values["use_custom_parser"] = use_custom_parser

    @builtins.property
    def step_type(self) -> AgentStepType:
        '''(experimental) The type of step this configuration applies to.

        :stability: experimental
        '''
        result = self._values.get("step_type")
        assert result is not None, "Required property 'step_type' is missing"
        return typing.cast(AgentStepType, result)

    @builtins.property
    def custom_prompt_template(self) -> typing.Optional[builtins.str]:
        '''(experimental) The custom prompt template to be used.

        :default: - The default prompt template will be used.

        :see: https://docs.aws.amazon.com/bedrock/latest/userguide/prompt-placeholders.html
        :stability: experimental
        '''
        result = self._values.get("custom_prompt_template")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def inference_config(self) -> typing.Optional[InferenceConfiguration]:
        '''(experimental) The inference configuration parameters to use.

        :default: undefined - Default inference configuration will be used

        :stability: experimental
        '''
        result = self._values.get("inference_config")
        return typing.cast(typing.Optional[InferenceConfiguration], result)

    @builtins.property
    def step_enabled(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to enable or skip this step in the agent sequence.

        :default:

        - The default state for each step type is as follows.

        PRE_PROCESSING – ENABLED
        ORCHESTRATION – ENABLED
        KNOWLEDGE_BASE_RESPONSE_GENERATION – ENABLED
        POST_PROCESSING – DISABLED

        :stability: experimental
        '''
        result = self._values.get("step_enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def use_custom_parser(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to use the custom Lambda parser defined for the sequence.

        :default: - false

        :stability: experimental
        '''
        result = self._values.get("use_custom_parser")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "PromptKnowledgeBaseResponseGenerationConfigCustomParser(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-bedrock-alpha.PromptMemorySummarizationConfigCustomParser",
    jsii_struct_bases=[PromptStepConfigBase],
    name_mapping={
        "step_type": "stepType",
        "custom_prompt_template": "customPromptTemplate",
        "inference_config": "inferenceConfig",
        "step_enabled": "stepEnabled",
        "use_custom_parser": "useCustomParser",
    },
)
class PromptMemorySummarizationConfigCustomParser(PromptStepConfigBase):
    def __init__(
        self,
        *,
        step_type: AgentStepType,
        custom_prompt_template: typing.Optional[builtins.str] = None,
        inference_config: typing.Optional[typing.Union[InferenceConfiguration, typing.Dict[builtins.str, typing.Any]]] = None,
        step_enabled: typing.Optional[builtins.bool] = None,
        use_custom_parser: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''(experimental) Configuration for the memory summarization step.

        :param step_type: (experimental) The type of step this configuration applies to.
        :param custom_prompt_template: (experimental) The custom prompt template to be used. Default: - The default prompt template will be used.
        :param inference_config: (experimental) The inference configuration parameters to use. Default: undefined - Default inference configuration will be used
        :param step_enabled: (experimental) Whether to enable or skip this step in the agent sequence. Default: - The default state for each step type is as follows. PRE_PROCESSING – ENABLED ORCHESTRATION – ENABLED KNOWLEDGE_BASE_RESPONSE_GENERATION – ENABLED POST_PROCESSING – DISABLED
        :param use_custom_parser: (experimental) Whether to use the custom Lambda parser defined for the sequence. Default: - false

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_bedrock_alpha as bedrock_alpha
            
            prompt_memory_summarization_config_custom_parser = bedrock_alpha.PromptMemorySummarizationConfigCustomParser(
                step_type=bedrock_alpha.AgentStepType.PRE_PROCESSING,
            
                # the properties below are optional
                custom_prompt_template="customPromptTemplate",
                inference_config=bedrock_alpha.InferenceConfiguration(
                    maximum_length=123,
                    stop_sequences=["stopSequences"],
                    temperature=123,
                    top_k=123,
                    top_p=123
                ),
                step_enabled=False,
                use_custom_parser=False
            )
        '''
        if isinstance(inference_config, dict):
            inference_config = InferenceConfiguration(**inference_config)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2c3254c964ab870d157cd9dac8226191b56dea63507327a11b9d17e91dc9d8ae)
            check_type(argname="argument step_type", value=step_type, expected_type=type_hints["step_type"])
            check_type(argname="argument custom_prompt_template", value=custom_prompt_template, expected_type=type_hints["custom_prompt_template"])
            check_type(argname="argument inference_config", value=inference_config, expected_type=type_hints["inference_config"])
            check_type(argname="argument step_enabled", value=step_enabled, expected_type=type_hints["step_enabled"])
            check_type(argname="argument use_custom_parser", value=use_custom_parser, expected_type=type_hints["use_custom_parser"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "step_type": step_type,
        }
        if custom_prompt_template is not None:
            self._values["custom_prompt_template"] = custom_prompt_template
        if inference_config is not None:
            self._values["inference_config"] = inference_config
        if step_enabled is not None:
            self._values["step_enabled"] = step_enabled
        if use_custom_parser is not None:
            self._values["use_custom_parser"] = use_custom_parser

    @builtins.property
    def step_type(self) -> AgentStepType:
        '''(experimental) The type of step this configuration applies to.

        :stability: experimental
        '''
        result = self._values.get("step_type")
        assert result is not None, "Required property 'step_type' is missing"
        return typing.cast(AgentStepType, result)

    @builtins.property
    def custom_prompt_template(self) -> typing.Optional[builtins.str]:
        '''(experimental) The custom prompt template to be used.

        :default: - The default prompt template will be used.

        :see: https://docs.aws.amazon.com/bedrock/latest/userguide/prompt-placeholders.html
        :stability: experimental
        '''
        result = self._values.get("custom_prompt_template")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def inference_config(self) -> typing.Optional[InferenceConfiguration]:
        '''(experimental) The inference configuration parameters to use.

        :default: undefined - Default inference configuration will be used

        :stability: experimental
        '''
        result = self._values.get("inference_config")
        return typing.cast(typing.Optional[InferenceConfiguration], result)

    @builtins.property
    def step_enabled(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to enable or skip this step in the agent sequence.

        :default:

        - The default state for each step type is as follows.

        PRE_PROCESSING – ENABLED
        ORCHESTRATION – ENABLED
        KNOWLEDGE_BASE_RESPONSE_GENERATION – ENABLED
        POST_PROCESSING – DISABLED

        :stability: experimental
        '''
        result = self._values.get("step_enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def use_custom_parser(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to use the custom Lambda parser defined for the sequence.

        :default: - false

        :stability: experimental
        '''
        result = self._values.get("use_custom_parser")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "PromptMemorySummarizationConfigCustomParser(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-bedrock-alpha.PromptOrchestrationConfigCustomParser",
    jsii_struct_bases=[PromptStepConfigBase],
    name_mapping={
        "step_type": "stepType",
        "custom_prompt_template": "customPromptTemplate",
        "inference_config": "inferenceConfig",
        "step_enabled": "stepEnabled",
        "use_custom_parser": "useCustomParser",
    },
)
class PromptOrchestrationConfigCustomParser(PromptStepConfigBase):
    def __init__(
        self,
        *,
        step_type: AgentStepType,
        custom_prompt_template: typing.Optional[builtins.str] = None,
        inference_config: typing.Optional[typing.Union[InferenceConfiguration, typing.Dict[builtins.str, typing.Any]]] = None,
        step_enabled: typing.Optional[builtins.bool] = None,
        use_custom_parser: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''(experimental) Configuration for the orchestration step.

        :param step_type: (experimental) The type of step this configuration applies to.
        :param custom_prompt_template: (experimental) The custom prompt template to be used. Default: - The default prompt template will be used.
        :param inference_config: (experimental) The inference configuration parameters to use. Default: undefined - Default inference configuration will be used
        :param step_enabled: (experimental) Whether to enable or skip this step in the agent sequence. Default: - The default state for each step type is as follows. PRE_PROCESSING – ENABLED ORCHESTRATION – ENABLED KNOWLEDGE_BASE_RESPONSE_GENERATION – ENABLED POST_PROCESSING – DISABLED
        :param use_custom_parser: (experimental) Whether to use the custom Lambda parser defined for the sequence. Default: - false

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_bedrock_alpha as bedrock_alpha
            
            prompt_orchestration_config_custom_parser = bedrock_alpha.PromptOrchestrationConfigCustomParser(
                step_type=bedrock_alpha.AgentStepType.PRE_PROCESSING,
            
                # the properties below are optional
                custom_prompt_template="customPromptTemplate",
                inference_config=bedrock_alpha.InferenceConfiguration(
                    maximum_length=123,
                    stop_sequences=["stopSequences"],
                    temperature=123,
                    top_k=123,
                    top_p=123
                ),
                step_enabled=False,
                use_custom_parser=False
            )
        '''
        if isinstance(inference_config, dict):
            inference_config = InferenceConfiguration(**inference_config)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1bd90bd69c0117ad5c57c3ca7c500809ffe9f26fd492886d93c33613b05951ac)
            check_type(argname="argument step_type", value=step_type, expected_type=type_hints["step_type"])
            check_type(argname="argument custom_prompt_template", value=custom_prompt_template, expected_type=type_hints["custom_prompt_template"])
            check_type(argname="argument inference_config", value=inference_config, expected_type=type_hints["inference_config"])
            check_type(argname="argument step_enabled", value=step_enabled, expected_type=type_hints["step_enabled"])
            check_type(argname="argument use_custom_parser", value=use_custom_parser, expected_type=type_hints["use_custom_parser"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "step_type": step_type,
        }
        if custom_prompt_template is not None:
            self._values["custom_prompt_template"] = custom_prompt_template
        if inference_config is not None:
            self._values["inference_config"] = inference_config
        if step_enabled is not None:
            self._values["step_enabled"] = step_enabled
        if use_custom_parser is not None:
            self._values["use_custom_parser"] = use_custom_parser

    @builtins.property
    def step_type(self) -> AgentStepType:
        '''(experimental) The type of step this configuration applies to.

        :stability: experimental
        '''
        result = self._values.get("step_type")
        assert result is not None, "Required property 'step_type' is missing"
        return typing.cast(AgentStepType, result)

    @builtins.property
    def custom_prompt_template(self) -> typing.Optional[builtins.str]:
        '''(experimental) The custom prompt template to be used.

        :default: - The default prompt template will be used.

        :see: https://docs.aws.amazon.com/bedrock/latest/userguide/prompt-placeholders.html
        :stability: experimental
        '''
        result = self._values.get("custom_prompt_template")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def inference_config(self) -> typing.Optional[InferenceConfiguration]:
        '''(experimental) The inference configuration parameters to use.

        :default: undefined - Default inference configuration will be used

        :stability: experimental
        '''
        result = self._values.get("inference_config")
        return typing.cast(typing.Optional[InferenceConfiguration], result)

    @builtins.property
    def step_enabled(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to enable or skip this step in the agent sequence.

        :default:

        - The default state for each step type is as follows.

        PRE_PROCESSING – ENABLED
        ORCHESTRATION – ENABLED
        KNOWLEDGE_BASE_RESPONSE_GENERATION – ENABLED
        POST_PROCESSING – DISABLED

        :stability: experimental
        '''
        result = self._values.get("step_enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def use_custom_parser(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to use the custom Lambda parser defined for the sequence.

        :default: - false

        :stability: experimental
        '''
        result = self._values.get("use_custom_parser")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "PromptOrchestrationConfigCustomParser(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-bedrock-alpha.PromptPostProcessingConfigCustomParser",
    jsii_struct_bases=[PromptStepConfigBase],
    name_mapping={
        "step_type": "stepType",
        "custom_prompt_template": "customPromptTemplate",
        "inference_config": "inferenceConfig",
        "step_enabled": "stepEnabled",
        "use_custom_parser": "useCustomParser",
    },
)
class PromptPostProcessingConfigCustomParser(PromptStepConfigBase):
    def __init__(
        self,
        *,
        step_type: AgentStepType,
        custom_prompt_template: typing.Optional[builtins.str] = None,
        inference_config: typing.Optional[typing.Union[InferenceConfiguration, typing.Dict[builtins.str, typing.Any]]] = None,
        step_enabled: typing.Optional[builtins.bool] = None,
        use_custom_parser: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''(experimental) Configuration for the post-processing step.

        :param step_type: (experimental) The type of step this configuration applies to.
        :param custom_prompt_template: (experimental) The custom prompt template to be used. Default: - The default prompt template will be used.
        :param inference_config: (experimental) The inference configuration parameters to use. Default: undefined - Default inference configuration will be used
        :param step_enabled: (experimental) Whether to enable or skip this step in the agent sequence. Default: - The default state for each step type is as follows. PRE_PROCESSING – ENABLED ORCHESTRATION – ENABLED KNOWLEDGE_BASE_RESPONSE_GENERATION – ENABLED POST_PROCESSING – DISABLED
        :param use_custom_parser: (experimental) Whether to use the custom Lambda parser defined for the sequence. Default: - false

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_bedrock_alpha as bedrock_alpha
            
            prompt_post_processing_config_custom_parser = bedrock_alpha.PromptPostProcessingConfigCustomParser(
                step_type=bedrock_alpha.AgentStepType.PRE_PROCESSING,
            
                # the properties below are optional
                custom_prompt_template="customPromptTemplate",
                inference_config=bedrock_alpha.InferenceConfiguration(
                    maximum_length=123,
                    stop_sequences=["stopSequences"],
                    temperature=123,
                    top_k=123,
                    top_p=123
                ),
                step_enabled=False,
                use_custom_parser=False
            )
        '''
        if isinstance(inference_config, dict):
            inference_config = InferenceConfiguration(**inference_config)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__59f64d023eb42889bb015de46d2debde642498318f963061062b0299ec1bbe58)
            check_type(argname="argument step_type", value=step_type, expected_type=type_hints["step_type"])
            check_type(argname="argument custom_prompt_template", value=custom_prompt_template, expected_type=type_hints["custom_prompt_template"])
            check_type(argname="argument inference_config", value=inference_config, expected_type=type_hints["inference_config"])
            check_type(argname="argument step_enabled", value=step_enabled, expected_type=type_hints["step_enabled"])
            check_type(argname="argument use_custom_parser", value=use_custom_parser, expected_type=type_hints["use_custom_parser"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "step_type": step_type,
        }
        if custom_prompt_template is not None:
            self._values["custom_prompt_template"] = custom_prompt_template
        if inference_config is not None:
            self._values["inference_config"] = inference_config
        if step_enabled is not None:
            self._values["step_enabled"] = step_enabled
        if use_custom_parser is not None:
            self._values["use_custom_parser"] = use_custom_parser

    @builtins.property
    def step_type(self) -> AgentStepType:
        '''(experimental) The type of step this configuration applies to.

        :stability: experimental
        '''
        result = self._values.get("step_type")
        assert result is not None, "Required property 'step_type' is missing"
        return typing.cast(AgentStepType, result)

    @builtins.property
    def custom_prompt_template(self) -> typing.Optional[builtins.str]:
        '''(experimental) The custom prompt template to be used.

        :default: - The default prompt template will be used.

        :see: https://docs.aws.amazon.com/bedrock/latest/userguide/prompt-placeholders.html
        :stability: experimental
        '''
        result = self._values.get("custom_prompt_template")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def inference_config(self) -> typing.Optional[InferenceConfiguration]:
        '''(experimental) The inference configuration parameters to use.

        :default: undefined - Default inference configuration will be used

        :stability: experimental
        '''
        result = self._values.get("inference_config")
        return typing.cast(typing.Optional[InferenceConfiguration], result)

    @builtins.property
    def step_enabled(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to enable or skip this step in the agent sequence.

        :default:

        - The default state for each step type is as follows.

        PRE_PROCESSING – ENABLED
        ORCHESTRATION – ENABLED
        KNOWLEDGE_BASE_RESPONSE_GENERATION – ENABLED
        POST_PROCESSING – DISABLED

        :stability: experimental
        '''
        result = self._values.get("step_enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def use_custom_parser(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to use the custom Lambda parser defined for the sequence.

        :default: - false

        :stability: experimental
        '''
        result = self._values.get("use_custom_parser")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "PromptPostProcessingConfigCustomParser(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-bedrock-alpha.PromptPreProcessingConfigCustomParser",
    jsii_struct_bases=[PromptStepConfigBase],
    name_mapping={
        "step_type": "stepType",
        "custom_prompt_template": "customPromptTemplate",
        "inference_config": "inferenceConfig",
        "step_enabled": "stepEnabled",
        "use_custom_parser": "useCustomParser",
    },
)
class PromptPreProcessingConfigCustomParser(PromptStepConfigBase):
    def __init__(
        self,
        *,
        step_type: AgentStepType,
        custom_prompt_template: typing.Optional[builtins.str] = None,
        inference_config: typing.Optional[typing.Union[InferenceConfiguration, typing.Dict[builtins.str, typing.Any]]] = None,
        step_enabled: typing.Optional[builtins.bool] = None,
        use_custom_parser: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''(experimental) Configuration for the pre-processing step.

        :param step_type: (experimental) The type of step this configuration applies to.
        :param custom_prompt_template: (experimental) The custom prompt template to be used. Default: - The default prompt template will be used.
        :param inference_config: (experimental) The inference configuration parameters to use. Default: undefined - Default inference configuration will be used
        :param step_enabled: (experimental) Whether to enable or skip this step in the agent sequence. Default: - The default state for each step type is as follows. PRE_PROCESSING – ENABLED ORCHESTRATION – ENABLED KNOWLEDGE_BASE_RESPONSE_GENERATION – ENABLED POST_PROCESSING – DISABLED
        :param use_custom_parser: (experimental) Whether to use the custom Lambda parser defined for the sequence. Default: - false

        :stability: experimental
        :exampleMetadata: fixture=default infused

        Example::

            parser_function = lambda_.Function(self, "ParserFunction",
                runtime=lambda_.Runtime.PYTHON_3_10,
                handler="index.handler",
                code=lambda_.Code.from_asset("lambda")
            )
            
            agent = bedrock.Agent(self, "Agent",
                foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
                instruction="You are a helpful assistant.",
                prompt_override_configuration=bedrock.PromptOverrideConfiguration.with_custom_parser(
                    parser=parser_function,
                    pre_processing_step=bedrock.PromptPreProcessingConfigCustomParser(
                        step_type=bedrock.AgentStepType.PRE_PROCESSING,
                        use_custom_parser=True
                    )
                )
            )
        '''
        if isinstance(inference_config, dict):
            inference_config = InferenceConfiguration(**inference_config)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6d80ef278090d225b8a023bd94ab2dadffa101778d0344756a8f633a90aa20e9)
            check_type(argname="argument step_type", value=step_type, expected_type=type_hints["step_type"])
            check_type(argname="argument custom_prompt_template", value=custom_prompt_template, expected_type=type_hints["custom_prompt_template"])
            check_type(argname="argument inference_config", value=inference_config, expected_type=type_hints["inference_config"])
            check_type(argname="argument step_enabled", value=step_enabled, expected_type=type_hints["step_enabled"])
            check_type(argname="argument use_custom_parser", value=use_custom_parser, expected_type=type_hints["use_custom_parser"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "step_type": step_type,
        }
        if custom_prompt_template is not None:
            self._values["custom_prompt_template"] = custom_prompt_template
        if inference_config is not None:
            self._values["inference_config"] = inference_config
        if step_enabled is not None:
            self._values["step_enabled"] = step_enabled
        if use_custom_parser is not None:
            self._values["use_custom_parser"] = use_custom_parser

    @builtins.property
    def step_type(self) -> AgentStepType:
        '''(experimental) The type of step this configuration applies to.

        :stability: experimental
        '''
        result = self._values.get("step_type")
        assert result is not None, "Required property 'step_type' is missing"
        return typing.cast(AgentStepType, result)

    @builtins.property
    def custom_prompt_template(self) -> typing.Optional[builtins.str]:
        '''(experimental) The custom prompt template to be used.

        :default: - The default prompt template will be used.

        :see: https://docs.aws.amazon.com/bedrock/latest/userguide/prompt-placeholders.html
        :stability: experimental
        '''
        result = self._values.get("custom_prompt_template")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def inference_config(self) -> typing.Optional[InferenceConfiguration]:
        '''(experimental) The inference configuration parameters to use.

        :default: undefined - Default inference configuration will be used

        :stability: experimental
        '''
        result = self._values.get("inference_config")
        return typing.cast(typing.Optional[InferenceConfiguration], result)

    @builtins.property
    def step_enabled(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to enable or skip this step in the agent sequence.

        :default:

        - The default state for each step type is as follows.

        PRE_PROCESSING – ENABLED
        ORCHESTRATION – ENABLED
        KNOWLEDGE_BASE_RESPONSE_GENERATION – ENABLED
        POST_PROCESSING – DISABLED

        :stability: experimental
        '''
        result = self._values.get("step_enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def use_custom_parser(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to use the custom Lambda parser defined for the sequence.

        :default: - false

        :stability: experimental
        '''
        result = self._values.get("use_custom_parser")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "PromptPreProcessingConfigCustomParser(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-bedrock-alpha.PromptRoutingClassifierConfigCustomParser",
    jsii_struct_bases=[PromptStepConfigBase],
    name_mapping={
        "step_type": "stepType",
        "custom_prompt_template": "customPromptTemplate",
        "inference_config": "inferenceConfig",
        "step_enabled": "stepEnabled",
        "use_custom_parser": "useCustomParser",
        "foundation_model": "foundationModel",
    },
)
class PromptRoutingClassifierConfigCustomParser(PromptStepConfigBase):
    def __init__(
        self,
        *,
        step_type: AgentStepType,
        custom_prompt_template: typing.Optional[builtins.str] = None,
        inference_config: typing.Optional[typing.Union[InferenceConfiguration, typing.Dict[builtins.str, typing.Any]]] = None,
        step_enabled: typing.Optional[builtins.bool] = None,
        use_custom_parser: typing.Optional[builtins.bool] = None,
        foundation_model: IBedrockInvokable,
    ) -> None:
        '''(experimental) Configuration for the routing classifier step.

        :param step_type: (experimental) The type of step this configuration applies to.
        :param custom_prompt_template: (experimental) The custom prompt template to be used. Default: - The default prompt template will be used.
        :param inference_config: (experimental) The inference configuration parameters to use. Default: undefined - Default inference configuration will be used
        :param step_enabled: (experimental) Whether to enable or skip this step in the agent sequence. Default: - The default state for each step type is as follows. PRE_PROCESSING – ENABLED ORCHESTRATION – ENABLED KNOWLEDGE_BASE_RESPONSE_GENERATION – ENABLED POST_PROCESSING – DISABLED
        :param use_custom_parser: (experimental) Whether to use the custom Lambda parser defined for the sequence. Default: - false
        :param foundation_model: (experimental) The foundation model to use for the routing classifier step. This is required for the routing classifier step.

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_bedrock_alpha as bedrock_alpha
            
            # bedrock_invokable: bedrock_alpha.IBedrockInvokable
            
            prompt_routing_classifier_config_custom_parser = bedrock_alpha.PromptRoutingClassifierConfigCustomParser(
                foundation_model=bedrock_invokable,
                step_type=bedrock_alpha.AgentStepType.PRE_PROCESSING,
            
                # the properties below are optional
                custom_prompt_template="customPromptTemplate",
                inference_config=bedrock_alpha.InferenceConfiguration(
                    maximum_length=123,
                    stop_sequences=["stopSequences"],
                    temperature=123,
                    top_k=123,
                    top_p=123
                ),
                step_enabled=False,
                use_custom_parser=False
            )
        '''
        if isinstance(inference_config, dict):
            inference_config = InferenceConfiguration(**inference_config)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fc21a218a77483a368cb274b36d03170fc21d293e79e34fd252a38e1277e3a90)
            check_type(argname="argument step_type", value=step_type, expected_type=type_hints["step_type"])
            check_type(argname="argument custom_prompt_template", value=custom_prompt_template, expected_type=type_hints["custom_prompt_template"])
            check_type(argname="argument inference_config", value=inference_config, expected_type=type_hints["inference_config"])
            check_type(argname="argument step_enabled", value=step_enabled, expected_type=type_hints["step_enabled"])
            check_type(argname="argument use_custom_parser", value=use_custom_parser, expected_type=type_hints["use_custom_parser"])
            check_type(argname="argument foundation_model", value=foundation_model, expected_type=type_hints["foundation_model"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "step_type": step_type,
            "foundation_model": foundation_model,
        }
        if custom_prompt_template is not None:
            self._values["custom_prompt_template"] = custom_prompt_template
        if inference_config is not None:
            self._values["inference_config"] = inference_config
        if step_enabled is not None:
            self._values["step_enabled"] = step_enabled
        if use_custom_parser is not None:
            self._values["use_custom_parser"] = use_custom_parser

    @builtins.property
    def step_type(self) -> AgentStepType:
        '''(experimental) The type of step this configuration applies to.

        :stability: experimental
        '''
        result = self._values.get("step_type")
        assert result is not None, "Required property 'step_type' is missing"
        return typing.cast(AgentStepType, result)

    @builtins.property
    def custom_prompt_template(self) -> typing.Optional[builtins.str]:
        '''(experimental) The custom prompt template to be used.

        :default: - The default prompt template will be used.

        :see: https://docs.aws.amazon.com/bedrock/latest/userguide/prompt-placeholders.html
        :stability: experimental
        '''
        result = self._values.get("custom_prompt_template")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def inference_config(self) -> typing.Optional[InferenceConfiguration]:
        '''(experimental) The inference configuration parameters to use.

        :default: undefined - Default inference configuration will be used

        :stability: experimental
        '''
        result = self._values.get("inference_config")
        return typing.cast(typing.Optional[InferenceConfiguration], result)

    @builtins.property
    def step_enabled(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to enable or skip this step in the agent sequence.

        :default:

        - The default state for each step type is as follows.

        PRE_PROCESSING – ENABLED
        ORCHESTRATION – ENABLED
        KNOWLEDGE_BASE_RESPONSE_GENERATION – ENABLED
        POST_PROCESSING – DISABLED

        :stability: experimental
        '''
        result = self._values.get("step_enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def use_custom_parser(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to use the custom Lambda parser defined for the sequence.

        :default: - false

        :stability: experimental
        '''
        result = self._values.get("use_custom_parser")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def foundation_model(self) -> IBedrockInvokable:
        '''(experimental) The foundation model to use for the routing classifier step.

        This is required for the routing classifier step.

        :stability: experimental
        '''
        result = self._values.get("foundation_model")
        assert result is not None, "Required property 'foundation_model' is missing"
        return typing.cast(IBedrockInvokable, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "PromptRoutingClassifierConfigCustomParser(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IAgent)
class Agent(
    AgentBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-bedrock-alpha.Agent",
):
    '''(experimental) Class to create (or import) an Agent with CDK.

    :stability: experimental
    :cloudformationResource: AWS::Bedrock::Agent
    :exampleMetadata: fixture=default infused

    Example::

        parser_function = lambda_.Function(self, "ParserFunction",
            runtime=lambda_.Runtime.PYTHON_3_10,
            handler="index.handler",
            code=lambda_.Code.from_asset("lambda")
        )
        
        agent = bedrock.Agent(self, "Agent",
            foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
            instruction="You are a helpful assistant.",
            prompt_override_configuration=bedrock.PromptOverrideConfiguration.with_custom_parser(
                parser=parser_function,
                pre_processing_step=bedrock.PromptPreProcessingConfigCustomParser(
                    step_type=bedrock.AgentStepType.PRE_PROCESSING,
                    use_custom_parser=True
                )
            )
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        foundation_model: IBedrockInvokable,
        instruction: builtins.str,
        action_groups: typing.Optional[typing.Sequence[AgentActionGroup]] = None,
        agent_collaboration: typing.Optional[AgentCollaboration] = None,
        agent_name: typing.Optional[builtins.str] = None,
        code_interpreter_enabled: typing.Optional[builtins.bool] = None,
        custom_orchestration_executor: typing.Optional[CustomOrchestrationExecutor] = None,
        description: typing.Optional[builtins.str] = None,
        existing_role: typing.Optional[_aws_cdk_aws_iam_ceddda9d.IRole] = None,
        force_delete: typing.Optional[builtins.bool] = None,
        idle_session_ttl: typing.Optional[_aws_cdk_ceddda9d.Duration] = None,
        kms_key: typing.Optional[_aws_cdk_aws_kms_ceddda9d.IKey] = None,
        memory: typing.Optional[Memory] = None,
        prompt_override_configuration: typing.Optional[PromptOverrideConfiguration] = None,
        should_prepare_agent: typing.Optional[builtins.bool] = None,
        user_input_enabled: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param foundation_model: (experimental) The foundation model used for orchestration by the agent.
        :param instruction: (experimental) The instruction used by the agent. This determines how the agent will perform his task. This instruction must have a minimum of 40 characters.
        :param action_groups: (experimental) The Action Groups associated with the agent. Default: - Only default action groups (UserInput and CodeInterpreter) are added
        :param agent_collaboration: (experimental) Configuration for agent collaboration settings, including AgentCollaboratorType and AgentCollaborators. This property allows you to define how the agent collaborates with other agents and what collaborators it can work with. Default: - No agent collaboration configuration.
        :param agent_name: (experimental) The name of the agent. This will be used as the physical name of the agent. Default: - A name is generated by CDK. Supported pattern : ^([0-9a-zA-Z][_-]?){1,100}$
        :param code_interpreter_enabled: (experimental) Select whether the agent can generate, run, and troubleshoot code when trying to complete a task. Default: - false
        :param custom_orchestration_executor: (experimental) The Lambda function to use for custom orchestration. If provided, custom orchestration will be used. If not provided, default orchestration will be used. Default: - Default orchestration
        :param description: (experimental) A description of the agent. Default: - No description is provided.
        :param existing_role: (experimental) An existing IAM Role to associate with this agent. Use this property when you want to reuse an existing IAM role rather than create a new one. The role must have a trust policy that allows the Bedrock service to assume it. Default: - A new role is created for you.
        :param force_delete: (experimental) Whether to delete the resource even if it's in use. Default: - false
        :param idle_session_ttl: (experimental) How long sessions should be kept open for the agent. If no conversation occurs during this time, the session expires and Amazon Bedrock deletes any data provided before the timeout. Default: - 10 minutes
        :param kms_key: (experimental) The KMS key of the agent if custom encryption is configured. Default: - An AWS managed key is used.
        :param memory: (experimental) The type and configuration of the memory to maintain context across multiple sessions and recall past interactions. This can be useful for maintaining continuity in multi-turn conversations and recalling user preferences or past interactions. Default: - No memory will be used. Agents will retain context from the current session only.
        :param prompt_override_configuration: (experimental) Overrides some prompt templates in different parts of an agent sequence configuration. Default: - No overrides are provided.
        :param should_prepare_agent: (experimental) Specifies whether to automatically update the ``DRAFT`` version of the agent after making changes to the agent. The ``DRAFT`` version can be continually iterated upon during internal development. Default: - false
        :param user_input_enabled: (experimental) Select whether the agent can prompt additional information from the user when it does not have enough information to respond to an utterance. Default: - false

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5c1da6dbc073c2e3dee222bd452cbcf2d4ea3cf0842dc28465a37f8887438de3)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AgentProps(
            foundation_model=foundation_model,
            instruction=instruction,
            action_groups=action_groups,
            agent_collaboration=agent_collaboration,
            agent_name=agent_name,
            code_interpreter_enabled=code_interpreter_enabled,
            custom_orchestration_executor=custom_orchestration_executor,
            description=description,
            existing_role=existing_role,
            force_delete=force_delete,
            idle_session_ttl=idle_session_ttl,
            kms_key=kms_key,
            memory=memory,
            prompt_override_configuration=prompt_override_configuration,
            should_prepare_agent=should_prepare_agent,
            user_input_enabled=user_input_enabled,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="fromAgentAttributes")
    @builtins.classmethod
    def from_agent_attributes(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        agent_arn: builtins.str,
        role_arn: builtins.str,
        agent_version: typing.Optional[builtins.str] = None,
        kms_key_arn: typing.Optional[builtins.str] = None,
        last_updated: typing.Optional[builtins.str] = None,
    ) -> IAgent:
        '''(experimental) Creates an Agent reference from an existing agent's attributes.

        :param scope: - The construct scope.
        :param id: - Identifier of the construct.
        :param agent_arn: (experimental) The ARN of the agent.
        :param role_arn: (experimental) The ARN of the IAM role associated to the agent.
        :param agent_version: (experimental) The agent version. If no explicit versions have been created, leave this empty to use the DRAFT version. Otherwise, use the version number (e.g. 1). Default: 'DRAFT'
        :param kms_key_arn: (experimental) Optional KMS encryption key associated with this agent. Default: undefined - An AWS managed key is used
        :param last_updated: (experimental) When this agent was last updated. Default: undefined - No last updated timestamp is provided

        :default: - For attrs.agentVersion: 'DRAFT' if no explicit version is provided

        :return: An IAgent reference to the existing agent

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c540786cbad361f254315ba321394ff0856d947cee9a259f1d02470add47fe69)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        attrs = AgentAttributes(
            agent_arn=agent_arn,
            role_arn=role_arn,
            agent_version=agent_version,
            kms_key_arn=kms_key_arn,
            last_updated=last_updated,
        )

        return typing.cast(IAgent, jsii.sinvoke(cls, "fromAgentAttributes", [scope, id, attrs]))

    @jsii.member(jsii_name="addActionGroup")
    def add_action_group(self, action_group: AgentActionGroup) -> None:
        '''(experimental) Adds an action group to the agent and configures necessary permissions.

        :param action_group: - The action group to add.

        :default:

        - Default permissions:
        - Lambda function invoke permissions if executor is present
        - S3 GetObject permissions if apiSchema.s3File is present

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e41ea74e490315a61ad841dea482cc082c005209a203d74fa4a65c176a9732e5)
            check_type(argname="argument action_group", value=action_group, expected_type=type_hints["action_group"])
        return typing.cast(None, jsii.invoke(self, "addActionGroup", [action_group]))

    @jsii.member(jsii_name="addActionGroups")
    def add_action_groups(self, *action_groups: AgentActionGroup) -> None:
        '''(experimental) Configuration for agent collaboration.

        :param action_groups: -

        :default: - No collaboration configuration.

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b558923d37923289e77a58811a37fe7a7a02b5cd20db20812845e8dd62b96b19)
            check_type(argname="argument action_groups", value=action_groups, expected_type=typing.Tuple[type_hints["action_groups"], ...]) # pyright: ignore [reportGeneralTypeIssues]
        return typing.cast(None, jsii.invoke(self, "addActionGroups", [*action_groups]))

    @jsii.member(jsii_name="generatePhysicalName")
    def _generate_physical_name(self) -> builtins.str:
        '''(experimental) Generates a physical name for the agent.

        :default:

        - Generated name format: 'agent-{hash}-{uniqueName}' with:
        - maxLength: MAXLENGTH_FOR_ROLE_NAME - '-bedrockagent'.length
        - lower: true
        - separator: '-'

        :return: A unique name for the agent with appropriate length constraints

        :stability: experimental
        :protected: true
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "generatePhysicalName", []))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PROPERTY_INJECTION_ID")
    def PROPERTY_INJECTION_ID(cls) -> builtins.str:
        '''(experimental) Uniquely identifies this class.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sget(cls, "PROPERTY_INJECTION_ID"))

    @builtins.property
    @jsii.member(jsii_name="actionGroups")
    def action_groups(self) -> typing.List[AgentActionGroup]:
        '''(experimental) action groups associated with the ageny.

        :stability: experimental
        '''
        return typing.cast(typing.List[AgentActionGroup], jsii.get(self, "actionGroups"))

    @builtins.property
    @jsii.member(jsii_name="agentArn")
    def agent_arn(self) -> builtins.str:
        '''(experimental) The ARN of the agent.

        :stability: experimental
        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "agentArn"))

    @builtins.property
    @jsii.member(jsii_name="agentId")
    def agent_id(self) -> builtins.str:
        '''(experimental) The unique identifier for the agent.

        :stability: experimental
        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "agentId"))

    @builtins.property
    @jsii.member(jsii_name="agentVersion")
    def agent_version(self) -> builtins.str:
        '''(experimental) The version of the agent.

        :stability: experimental
        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "agentVersion"))

    @builtins.property
    @jsii.member(jsii_name="grantPrincipal")
    def grant_principal(self) -> _aws_cdk_aws_iam_ceddda9d.IPrincipal:
        '''(experimental) The principal to grant permissions to.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_iam_ceddda9d.IPrincipal, jsii.get(self, "grantPrincipal"))

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        '''(experimental) The name of the agent.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "name"))

    @builtins.property
    @jsii.member(jsii_name="role")
    def role(self) -> _aws_cdk_aws_iam_ceddda9d.IRole:
        '''(experimental) The IAM role associated to the agent.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_iam_ceddda9d.IRole, jsii.get(self, "role"))

    @builtins.property
    @jsii.member(jsii_name="testAlias")
    def test_alias(self) -> IAgentAlias:
        '''(experimental) Default alias of the agent.

        :stability: experimental
        '''
        return typing.cast(IAgentAlias, jsii.get(self, "testAlias"))

    @builtins.property
    @jsii.member(jsii_name="kmsKey")
    def kms_key(self) -> typing.Optional[_aws_cdk_aws_kms_ceddda9d.IKey]:
        '''(experimental) Optional KMS encryption key associated with this agent.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[_aws_cdk_aws_kms_ceddda9d.IKey], jsii.get(self, "kmsKey"))

    @builtins.property
    @jsii.member(jsii_name="lastUpdated")
    def last_updated(self) -> typing.Optional[builtins.str]:
        '''(experimental) When this agent was last updated.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "lastUpdated"))


class AgentAlias(
    AgentAliasBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-bedrock-alpha.AgentAlias",
):
    '''(experimental) Class to create an Agent Alias with CDK.

    :stability: experimental
    :cloudformationResource: AWS::Bedrock::AgentAlias
    :exampleMetadata: fixture=default infused

    Example::

        # Create a specialized agent
        customer_support_agent = bedrock.Agent(self, "CustomerSupportAgent",
            instruction="You specialize in answering customer support questions.",
            foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1
        )
        
        # Create an agent alias
        customer_support_alias = bedrock.AgentAlias(self, "CustomerSupportAlias",
            agent=customer_support_agent,
            agent_alias_name="production"
        )
        
        # Create a main agent that collaborates with the specialized agent
        main_agent = bedrock.Agent(self, "MainAgent",
            instruction="You route specialized questions to other agents.",
            foundation_model=bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
            agent_collaboration={
                "type": bedrock.AgentCollaboratorType.SUPERVISOR,
                "collaborators": [
                    bedrock.AgentCollaborator(
                        agent_alias=customer_support_alias,
                        collaboration_instruction="Route customer support questions to this agent.",
                        collaborator_name="CustomerSupport",
                        relay_conversation_history=True
                    )
                ]
            }
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        agent: IAgent,
        agent_alias_name: typing.Optional[builtins.str] = None,
        agent_version: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param agent: (experimental) The agent associated to this alias.
        :param agent_alias_name: (experimental) The name for the agent alias. This will be used as the physical name of the agent alias. Default: - "latest"
        :param agent_version: (experimental) The version of the agent to associate with the agent alias. Default: - Creates a new version of the agent.
        :param description: (experimental) Description for the agent alias. Default: undefined - No description is provided

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__42756e488adc0d798aade78ff819dc550151bd380ede90fe77d6aa5c01bebb44)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AgentAliasProps(
            agent=agent,
            agent_alias_name=agent_alias_name,
            agent_version=agent_version,
            description=description,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="fromAttributes")
    @builtins.classmethod
    def from_attributes(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        agent: IAgent,
        agent_version: builtins.str,
        alias_id: builtins.str,
        alias_name: typing.Optional[builtins.str] = None,
    ) -> IAgentAlias:
        '''(experimental) Brings an Agent Alias from an existing one created outside of CDK.

        :param scope: -
        :param id: -
        :param agent: (experimental) The underlying agent for this alias.
        :param agent_version: (experimental) The agent version for this alias.
        :param alias_id: (experimental) The Id of the agent alias.
        :param alias_name: (experimental) The name of the agent alias. Default: undefined - No alias name is provided

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__61e68f17caa900cc2bc2da7f093e54644fc9ad79356dad14956cfac36fadae9d)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        attrs = AgentAliasAttributes(
            agent=agent,
            agent_version=agent_version,
            alias_id=alias_id,
            alias_name=alias_name,
        )

        return typing.cast(IAgentAlias, jsii.sinvoke(cls, "fromAttributes", [scope, id, attrs]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PROPERTY_INJECTION_ID")
    def PROPERTY_INJECTION_ID(cls) -> builtins.str:
        '''(experimental) Uniquely identifies this class.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sget(cls, "PROPERTY_INJECTION_ID"))

    @builtins.property
    @jsii.member(jsii_name="agent")
    def agent(self) -> IAgent:
        '''(experimental) The underlying agent for this alias.

        :stability: experimental
        '''
        return typing.cast(IAgent, jsii.get(self, "agent"))

    @builtins.property
    @jsii.member(jsii_name="aliasArn")
    def alias_arn(self) -> builtins.str:
        '''(experimental) The ARN of the agent alias.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "aliasArn"))

    @builtins.property
    @jsii.member(jsii_name="aliasId")
    def alias_id(self) -> builtins.str:
        '''(experimental) The unique identifier of the agent alias.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "aliasId"))

    @builtins.property
    @jsii.member(jsii_name="aliasName")
    def alias_name(self) -> builtins.str:
        '''(experimental) The name of the agent alias.

        This is either provided by the user or generated from a hash.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "aliasName"))


__all__ = [
    "ActionGroupExecutor",
    "Agent",
    "AgentActionGroup",
    "AgentActionGroupProps",
    "AgentAlias",
    "AgentAliasAttributes",
    "AgentAliasBase",
    "AgentAliasProps",
    "AgentAttributes",
    "AgentBase",
    "AgentCollaboration",
    "AgentCollaborationConfig",
    "AgentCollaborator",
    "AgentCollaboratorProps",
    "AgentCollaboratorType",
    "AgentProps",
    "AgentStepType",
    "ApiSchema",
    "AssetApiSchema",
    "BedrockFoundationModel",
    "BedrockFoundationModelProps",
    "CustomControl",
    "CustomOrchestrationExecutor",
    "CustomParserProps",
    "Function",
    "FunctionParameter",
    "FunctionParameterProps",
    "FunctionProps",
    "FunctionSchema",
    "FunctionSchemaProps",
    "IAgent",
    "IAgentAlias",
    "IBedrockInvokable",
    "InferenceConfiguration",
    "InlineApiSchema",
    "Memory",
    "OrchestrationType",
    "ParameterType",
    "ParentActionGroupSignature",
    "PromptKnowledgeBaseResponseGenerationConfigCustomParser",
    "PromptMemorySummarizationConfigCustomParser",
    "PromptOrchestrationConfigCustomParser",
    "PromptOverrideConfiguration",
    "PromptPostProcessingConfigCustomParser",
    "PromptPreProcessingConfigCustomParser",
    "PromptRoutingClassifierConfigCustomParser",
    "PromptStepConfigBase",
    "RequireConfirmation",
    "S3ApiSchema",
    "SessionSummaryMemoryProps",
    "VectorType",
]

publication.publish()

def _typecheckingstub__b3f8715f9bf580f1336c3ef0001b564cd5377a641554f928ee8a9337f101d4ea(
    lambda_function: _aws_cdk_aws_lambda_ceddda9d.IFunction,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__f216a3b9945ded433d1397b46354d778b44544b12c7e1444f18f26697b23c949(
    enabled: builtins.bool,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__c7e35cbb8b23de7b78171edfc3dfea96900c1650cd60b84a3b3daafe40b03e1e(
    enabled: builtins.bool,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__96565235b78a53a99b9a5b0de1f6e1cb436217f7023225486fb3b6b0402c16b1(
    *,
    api_schema: typing.Optional[ApiSchema] = None,
    description: typing.Optional[builtins.str] = None,
    enabled: typing.Optional[builtins.bool] = None,
    executor: typing.Optional[ActionGroupExecutor] = None,
    force_delete: typing.Optional[builtins.bool] = None,
    function_schema: typing.Optional[FunctionSchema] = None,
    name: typing.Optional[builtins.str] = None,
    parent_action_group_signature: typing.Optional[ParentActionGroupSignature] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__2cf3ee27a5cffdeb97ea680716346f7407718c3709c2070bc5cfe5846adf5e53(
    *,
    agent: IAgent,
    agent_version: builtins.str,
    alias_id: builtins.str,
    alias_name: typing.Optional[builtins.str] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__223e58e52a6c762628fbd5894a3746df0edc021abe283a3fa5fae709b84c6e2b(
    *,
    agent: IAgent,
    agent_alias_name: typing.Optional[builtins.str] = None,
    agent_version: typing.Optional[builtins.str] = None,
    description: typing.Optional[builtins.str] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__ec4361e61c0d3b4b62f635521602d37a1a496b4cd051caa617df76fefcfb8daa(
    *,
    agent_arn: builtins.str,
    role_arn: builtins.str,
    agent_version: typing.Optional[builtins.str] = None,
    kms_key_arn: typing.Optional[builtins.str] = None,
    last_updated: typing.Optional[builtins.str] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__8a9907acc35e0edfd5ff2b73fc849757eb706524befb93fa2d2644edefa77814(
    *,
    collaborators: typing.Sequence[AgentCollaborator],
    type: AgentCollaboratorType,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__e5dcc21bf3943fccd3a13e28a26304d667dc4ab7c6a95e6ba7b947029e97b6f3(
    grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__38cb6ea0e190b8eef95b7964639213d5273c82e36cad360e5f54a538ea1015a1(
    *,
    agent_alias: IAgentAlias,
    collaboration_instruction: builtins.str,
    collaborator_name: builtins.str,
    relay_conversation_history: typing.Optional[builtins.bool] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__1e71f5eada4b0acb202729bade487b2adb4e2e2b8fd4546f4cb3bcbc8f3d8cc1(
    *,
    foundation_model: IBedrockInvokable,
    instruction: builtins.str,
    action_groups: typing.Optional[typing.Sequence[AgentActionGroup]] = None,
    agent_collaboration: typing.Optional[AgentCollaboration] = None,
    agent_name: typing.Optional[builtins.str] = None,
    code_interpreter_enabled: typing.Optional[builtins.bool] = None,
    custom_orchestration_executor: typing.Optional[CustomOrchestrationExecutor] = None,
    description: typing.Optional[builtins.str] = None,
    existing_role: typing.Optional[_aws_cdk_aws_iam_ceddda9d.IRole] = None,
    force_delete: typing.Optional[builtins.bool] = None,
    idle_session_ttl: typing.Optional[_aws_cdk_ceddda9d.Duration] = None,
    kms_key: typing.Optional[_aws_cdk_aws_kms_ceddda9d.IKey] = None,
    memory: typing.Optional[Memory] = None,
    prompt_override_configuration: typing.Optional[PromptOverrideConfiguration] = None,
    should_prepare_agent: typing.Optional[builtins.bool] = None,
    user_input_enabled: typing.Optional[builtins.bool] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__fef98095a98b1c3218daff037e79d43c41de797ec47d848bf1caa44f27c10abf(
    s3_file: typing.Optional[typing.Union[_aws_cdk_aws_s3_ceddda9d.Location, typing.Dict[builtins.str, typing.Any]]] = None,
    inline_schema: typing.Optional[builtins.str] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__ea650d724bddc459b33660757cb186cf804ba8ff34e5149c2ba116012bac3d13(
    schema: builtins.str,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__1ef4f4f3b1ccc0ba1ca114b89046ebec72d8562e37d54d5ce5a48fb13bdb98e5(
    path: builtins.str,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__882387202413f33bb2e0f658215337e89d82d8bab90973056398f7bc7f939f52(
    bucket: _aws_cdk_aws_s3_ceddda9d.IBucket,
    object_key: builtins.str,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__aee3c625b88d3cf9715cbd94ca9cf4c640ffe4007031b066fcea7749d1fd3b20(
    path: builtins.str,
    *,
    deploy_time: typing.Optional[builtins.bool] = None,
    display_name: typing.Optional[builtins.str] = None,
    readers: typing.Optional[typing.Sequence[_aws_cdk_aws_iam_ceddda9d.IGrantable]] = None,
    source_kms_key: typing.Optional[_aws_cdk_aws_kms_ceddda9d.IKey] = None,
    asset_hash: typing.Optional[builtins.str] = None,
    asset_hash_type: typing.Optional[_aws_cdk_ceddda9d.AssetHashType] = None,
    bundling: typing.Optional[typing.Union[_aws_cdk_ceddda9d.BundlingOptions, typing.Dict[builtins.str, typing.Any]]] = None,
    exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
    follow_symlinks: typing.Optional[_aws_cdk_ceddda9d.SymlinkFollowMode] = None,
    ignore_mode: typing.Optional[_aws_cdk_ceddda9d.IgnoreMode] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__e42433d3c2e0a249ee91a1f011a13449ed0bdad079ab434f1317b403ad25b8e7(
    scope: _constructs_77d1e7e8.Construct,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__9871e1b6209dedd6005dcce72b65ee8c40c6a25156e22de4c732060c640acf67(
    *,
    legacy: typing.Optional[builtins.bool] = None,
    optimized_for_agents: typing.Optional[builtins.bool] = None,
    supported_vector_type: typing.Optional[typing.Sequence[VectorType]] = None,
    supports_agents: typing.Optional[builtins.bool] = None,
    supports_cross_region: typing.Optional[builtins.bool] = None,
    supports_knowledge_base: typing.Optional[builtins.bool] = None,
    vector_dimensions: typing.Optional[jsii.Number] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__9581a8fe03e677bcf39692a4a671ac83b906583168b1efeb85160cd87ace6e24(
    lambda_function: _aws_cdk_aws_lambda_ceddda9d.IFunction,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__5e9247125ea57043b0c8014036fd32415bac33f25230256cde57ebee738d550b(
    *,
    knowledge_base_response_generation_step: typing.Optional[typing.Union[PromptKnowledgeBaseResponseGenerationConfigCustomParser, typing.Dict[builtins.str, typing.Any]]] = None,
    memory_summarization_step: typing.Optional[typing.Union[PromptMemorySummarizationConfigCustomParser, typing.Dict[builtins.str, typing.Any]]] = None,
    orchestration_step: typing.Optional[typing.Union[PromptOrchestrationConfigCustomParser, typing.Dict[builtins.str, typing.Any]]] = None,
    parser: typing.Optional[_aws_cdk_aws_lambda_ceddda9d.IFunction] = None,
    post_processing_step: typing.Optional[typing.Union[PromptPostProcessingConfigCustomParser, typing.Dict[builtins.str, typing.Any]]] = None,
    pre_processing_step: typing.Optional[typing.Union[PromptPreProcessingConfigCustomParser, typing.Dict[builtins.str, typing.Any]]] = None,
    routing_classifier_step: typing.Optional[typing.Union[PromptRoutingClassifierConfigCustomParser, typing.Dict[builtins.str, typing.Any]]] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__afff183f47ab9cba063848fa5cc915f6deaad94e1aaeca1688be3a79be4bf746(
    *,
    type: ParameterType,
    description: typing.Optional[builtins.str] = None,
    required: typing.Optional[builtins.bool] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__d6a1e6361e2edbbe58148c14c19a168946ef6213e0d123499089a4f3e48f6a0d(
    *,
    description: builtins.str,
    name: builtins.str,
    parameters: typing.Optional[typing.Mapping[builtins.str, typing.Union[FunctionParameterProps, typing.Dict[builtins.str, typing.Any]]]] = None,
    require_confirmation: typing.Optional[RequireConfirmation] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__64dfdaad570b5bc4904b9d9f356eb416095866cc4cca0546c2bb37dbb844e916(
    *,
    functions: typing.Sequence[typing.Union[FunctionProps, typing.Dict[builtins.str, typing.Any]]],
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__f618bd28d205f12fe18f577c6cf06cfa5419cd5eba307a62054fe15003721240(
    grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__36b2c150112fc98f8b2c8eebfbc62fdd6f55ded880e47bf4dffbba187d0bf1c4(
    id: builtins.str,
    *,
    target: typing.Optional[_aws_cdk_aws_events_ceddda9d.IRuleTarget] = None,
    cross_stack_scope: typing.Optional[_constructs_77d1e7e8.Construct] = None,
    description: typing.Optional[builtins.str] = None,
    event_pattern: typing.Optional[typing.Union[_aws_cdk_aws_events_ceddda9d.EventPattern, typing.Dict[builtins.str, typing.Any]]] = None,
    rule_name: typing.Optional[builtins.str] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__f4cca4a424332ea1f29775bfb3a1ac6db0f5368e9dfae8d1adb7fe67e042a6af(
    grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
    *actions: builtins.str,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__4f3e1ad5ab734c71be4df5c8ad8732b4f05f16f371b0379a039c7c2d3b7dec8a(
    grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__c205eff6a91ceb0127ca10998bf86f4e152274ba9eceea415fa6e71f802904bb(
    grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__ad4bacfc955253cfea89b4d09ebbbf2ccae8ba53fb98679bb4f2f96718f4477c(
    id: builtins.str,
    *,
    target: typing.Optional[_aws_cdk_aws_events_ceddda9d.IRuleTarget] = None,
    cross_stack_scope: typing.Optional[_constructs_77d1e7e8.Construct] = None,
    description: typing.Optional[builtins.str] = None,
    event_pattern: typing.Optional[typing.Union[_aws_cdk_aws_events_ceddda9d.EventPattern, typing.Dict[builtins.str, typing.Any]]] = None,
    rule_name: typing.Optional[builtins.str] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__aeff941da61743d78967d00b93f8c3e1baf285b4d10bc9d78579d8de4c9e8a77(
    grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__715d624dff386da6c193fd1c39e8c824ded7833eed6a175f93d3637254cabb5e(
    *,
    maximum_length: jsii.Number,
    stop_sequences: typing.Sequence[builtins.str],
    temperature: jsii.Number,
    top_k: jsii.Number,
    top_p: jsii.Number,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__0c276a821c9a889fcfc1d87230ab9f92627a99c7dc83fc06a8d00bb8d1534657(
    schema: builtins.str,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__ebccabd52fd67d660adf169e439525008c110204b48e0ed669d4ffeedce9e65e(
    value: builtins.str,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__1d39057223b726590938606b5e3ac9d394a1418fec4a81576b5273321f3b86d8(
    steps: typing.Sequence[typing.Union[PromptStepConfigBase, typing.Dict[builtins.str, typing.Any]]],
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__c80dbca03d170ec1393aad8bb4aee5d9109f13b498913a973f3a939b8dfeb044(
    *,
    step_type: AgentStepType,
    custom_prompt_template: typing.Optional[builtins.str] = None,
    inference_config: typing.Optional[typing.Union[InferenceConfiguration, typing.Dict[builtins.str, typing.Any]]] = None,
    step_enabled: typing.Optional[builtins.bool] = None,
    use_custom_parser: typing.Optional[builtins.bool] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__66ca7c708f8916f74c35e9ad0069f982f7710c5742e63b6e61b9889bdcea86ee(
    *,
    max_recent_sessions: typing.Optional[jsii.Number] = None,
    memory_duration: typing.Optional[_aws_cdk_ceddda9d.Duration] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__c76c254ff658709c91b9f92836c40baa806c28af85c028c89448b6d932a0915c(
    scope: _constructs_77d1e7e8.Construct,
    id: builtins.str,
    *,
    account: typing.Optional[builtins.str] = None,
    environment_from_arn: typing.Optional[builtins.str] = None,
    physical_name: typing.Optional[builtins.str] = None,
    region: typing.Optional[builtins.str] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__f605b4cd4baa1eb130f51bcadc5c4a81822ae2cab31ec49a4e9b5b32f8fc9c45(
    grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
    *actions: builtins.str,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__dbf39e88263f8cef13dfe1f4d67a47c3bf2490d27c8ce9a70ade0cad70e3b7d6(
    grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__d6a50391f64a87b84a4584ba8e48f37b926fb74fc6eceb7e6109f491509b2092(
    grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__095d243c58d0aea9f21e7e1d1b7f1cd5ad28281a11a261aa1fbae414e5a7456f(
    id: builtins.str,
    *,
    target: typing.Optional[_aws_cdk_aws_events_ceddda9d.IRuleTarget] = None,
    cross_stack_scope: typing.Optional[_constructs_77d1e7e8.Construct] = None,
    description: typing.Optional[builtins.str] = None,
    event_pattern: typing.Optional[typing.Union[_aws_cdk_aws_events_ceddda9d.EventPattern, typing.Dict[builtins.str, typing.Any]]] = None,
    rule_name: typing.Optional[builtins.str] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__d35aa54a631bc9ca37bc8477f55b0cac6286ce864c5768fdc4f1d2e480ec1799(
    scope: _constructs_77d1e7e8.Construct,
    id: builtins.str,
    *,
    account: typing.Optional[builtins.str] = None,
    environment_from_arn: typing.Optional[builtins.str] = None,
    physical_name: typing.Optional[builtins.str] = None,
    region: typing.Optional[builtins.str] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__87c56642c7d7642cf9968ade38695bd0f8a20c4c562d1611dce60c2bfb519373(
    grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__2e2bf859b46e6177afc37c92a35764e2e847cf84005e6f9511ec77cff25aff6a(
    id: builtins.str,
    *,
    target: typing.Optional[_aws_cdk_aws_events_ceddda9d.IRuleTarget] = None,
    cross_stack_scope: typing.Optional[_constructs_77d1e7e8.Construct] = None,
    description: typing.Optional[builtins.str] = None,
    event_pattern: typing.Optional[typing.Union[_aws_cdk_aws_events_ceddda9d.EventPattern, typing.Dict[builtins.str, typing.Any]]] = None,
    rule_name: typing.Optional[builtins.str] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__c72ea354d94de945db88ea8f90368338db6b71c53c1de96d6b6ff8e5ca8bd139(
    value: builtins.str,
    *,
    legacy: typing.Optional[builtins.bool] = None,
    optimized_for_agents: typing.Optional[builtins.bool] = None,
    supported_vector_type: typing.Optional[typing.Sequence[VectorType]] = None,
    supports_agents: typing.Optional[builtins.bool] = None,
    supports_cross_region: typing.Optional[builtins.bool] = None,
    supports_knowledge_base: typing.Optional[builtins.bool] = None,
    vector_dimensions: typing.Optional[jsii.Number] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__21746783fde44094590eeb2e0bf9318cd7c94a7d9fdbbc276e7c07e9882807b9(
    model_id: _aws_cdk_aws_bedrock_ceddda9d.FoundationModel,
    *,
    legacy: typing.Optional[builtins.bool] = None,
    optimized_for_agents: typing.Optional[builtins.bool] = None,
    supported_vector_type: typing.Optional[typing.Sequence[VectorType]] = None,
    supports_agents: typing.Optional[builtins.bool] = None,
    supports_cross_region: typing.Optional[builtins.bool] = None,
    supports_knowledge_base: typing.Optional[builtins.bool] = None,
    vector_dimensions: typing.Optional[jsii.Number] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__99e6623f3ac6adc2fae17fdb40358d156dfbc20814dd317a4e724bf44ede364e(
    model_id: _aws_cdk_aws_bedrock_ceddda9d.FoundationModelIdentifier,
    *,
    legacy: typing.Optional[builtins.bool] = None,
    optimized_for_agents: typing.Optional[builtins.bool] = None,
    supported_vector_type: typing.Optional[typing.Sequence[VectorType]] = None,
    supports_agents: typing.Optional[builtins.bool] = None,
    supports_cross_region: typing.Optional[builtins.bool] = None,
    supports_knowledge_base: typing.Optional[builtins.bool] = None,
    vector_dimensions: typing.Optional[jsii.Number] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__ffad53dee884df6b2476ed3cd5b2db73771ff68f85c38de0570b5b16841bcd21(
    grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__c036ac01b7085698fb4634261fe75aea9d21d24dd6c4187d1454cda55c36912a(
    grantee: _aws_cdk_aws_iam_ceddda9d.IGrantable,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__ff59ff4e8f8241de7bd05809f66c99d45b60f152560b50867f223721c27d2bc3(
    *,
    step_type: AgentStepType,
    custom_prompt_template: typing.Optional[builtins.str] = None,
    inference_config: typing.Optional[typing.Union[InferenceConfiguration, typing.Dict[builtins.str, typing.Any]]] = None,
    step_enabled: typing.Optional[builtins.bool] = None,
    use_custom_parser: typing.Optional[builtins.bool] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__2c3254c964ab870d157cd9dac8226191b56dea63507327a11b9d17e91dc9d8ae(
    *,
    step_type: AgentStepType,
    custom_prompt_template: typing.Optional[builtins.str] = None,
    inference_config: typing.Optional[typing.Union[InferenceConfiguration, typing.Dict[builtins.str, typing.Any]]] = None,
    step_enabled: typing.Optional[builtins.bool] = None,
    use_custom_parser: typing.Optional[builtins.bool] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__1bd90bd69c0117ad5c57c3ca7c500809ffe9f26fd492886d93c33613b05951ac(
    *,
    step_type: AgentStepType,
    custom_prompt_template: typing.Optional[builtins.str] = None,
    inference_config: typing.Optional[typing.Union[InferenceConfiguration, typing.Dict[builtins.str, typing.Any]]] = None,
    step_enabled: typing.Optional[builtins.bool] = None,
    use_custom_parser: typing.Optional[builtins.bool] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__59f64d023eb42889bb015de46d2debde642498318f963061062b0299ec1bbe58(
    *,
    step_type: AgentStepType,
    custom_prompt_template: typing.Optional[builtins.str] = None,
    inference_config: typing.Optional[typing.Union[InferenceConfiguration, typing.Dict[builtins.str, typing.Any]]] = None,
    step_enabled: typing.Optional[builtins.bool] = None,
    use_custom_parser: typing.Optional[builtins.bool] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__6d80ef278090d225b8a023bd94ab2dadffa101778d0344756a8f633a90aa20e9(
    *,
    step_type: AgentStepType,
    custom_prompt_template: typing.Optional[builtins.str] = None,
    inference_config: typing.Optional[typing.Union[InferenceConfiguration, typing.Dict[builtins.str, typing.Any]]] = None,
    step_enabled: typing.Optional[builtins.bool] = None,
    use_custom_parser: typing.Optional[builtins.bool] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__fc21a218a77483a368cb274b36d03170fc21d293e79e34fd252a38e1277e3a90(
    *,
    step_type: AgentStepType,
    custom_prompt_template: typing.Optional[builtins.str] = None,
    inference_config: typing.Optional[typing.Union[InferenceConfiguration, typing.Dict[builtins.str, typing.Any]]] = None,
    step_enabled: typing.Optional[builtins.bool] = None,
    use_custom_parser: typing.Optional[builtins.bool] = None,
    foundation_model: IBedrockInvokable,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__5c1da6dbc073c2e3dee222bd452cbcf2d4ea3cf0842dc28465a37f8887438de3(
    scope: _constructs_77d1e7e8.Construct,
    id: builtins.str,
    *,
    foundation_model: IBedrockInvokable,
    instruction: builtins.str,
    action_groups: typing.Optional[typing.Sequence[AgentActionGroup]] = None,
    agent_collaboration: typing.Optional[AgentCollaboration] = None,
    agent_name: typing.Optional[builtins.str] = None,
    code_interpreter_enabled: typing.Optional[builtins.bool] = None,
    custom_orchestration_executor: typing.Optional[CustomOrchestrationExecutor] = None,
    description: typing.Optional[builtins.str] = None,
    existing_role: typing.Optional[_aws_cdk_aws_iam_ceddda9d.IRole] = None,
    force_delete: typing.Optional[builtins.bool] = None,
    idle_session_ttl: typing.Optional[_aws_cdk_ceddda9d.Duration] = None,
    kms_key: typing.Optional[_aws_cdk_aws_kms_ceddda9d.IKey] = None,
    memory: typing.Optional[Memory] = None,
    prompt_override_configuration: typing.Optional[PromptOverrideConfiguration] = None,
    should_prepare_agent: typing.Optional[builtins.bool] = None,
    user_input_enabled: typing.Optional[builtins.bool] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__c540786cbad361f254315ba321394ff0856d947cee9a259f1d02470add47fe69(
    scope: _constructs_77d1e7e8.Construct,
    id: builtins.str,
    *,
    agent_arn: builtins.str,
    role_arn: builtins.str,
    agent_version: typing.Optional[builtins.str] = None,
    kms_key_arn: typing.Optional[builtins.str] = None,
    last_updated: typing.Optional[builtins.str] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__e41ea74e490315a61ad841dea482cc082c005209a203d74fa4a65c176a9732e5(
    action_group: AgentActionGroup,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__b558923d37923289e77a58811a37fe7a7a02b5cd20db20812845e8dd62b96b19(
    *action_groups: AgentActionGroup,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__42756e488adc0d798aade78ff819dc550151bd380ede90fe77d6aa5c01bebb44(
    scope: _constructs_77d1e7e8.Construct,
    id: builtins.str,
    *,
    agent: IAgent,
    agent_alias_name: typing.Optional[builtins.str] = None,
    agent_version: typing.Optional[builtins.str] = None,
    description: typing.Optional[builtins.str] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__61e68f17caa900cc2bc2da7f093e54644fc9ad79356dad14956cfac36fadae9d(
    scope: _constructs_77d1e7e8.Construct,
    id: builtins.str,
    *,
    agent: IAgent,
    agent_version: builtins.str,
    alias_id: builtins.str,
    alias_name: typing.Optional[builtins.str] = None,
) -> None:
    """Type checking stubs"""
    pass
