import { CfnAgent } from 'aws-cdk-lib/aws-bedrock';
import { ActionGroupSchema } from './schema-base';
/**
 * Enum for parameter types in function schemas
 */
export declare enum ParameterType {
    /**
     * String parameter type
     */
    STRING = "string",
    /**
     * Number parameter type
     */
    NUMBER = "number",
    /**
     * Integer parameter type
     */
    INTEGER = "integer",
    /**
     * Boolean parameter type
     */
    BOOLEAN = "boolean",
    /**
     * Array parameter type
     */
    ARRAY = "array",
    /**
     * Object parameter type
     */
    OBJECT = "object"
}
/**
 * Enum for require confirmation state in function schemas
 */
export declare enum RequireConfirmation {
    /**
     * Confirmation is enabled
     */
    ENABLED = "ENABLED",
    /**
     * Confirmation is disabled
     */
    DISABLED = "DISABLED"
}
/**
 * Properties for a function parameter
 */
export interface FunctionParameterProps {
    /**
     * The type of the parameter
     */
    readonly type: ParameterType;
    /**
     * Whether the parameter is required
     * @default true
     */
    readonly required?: boolean;
    /**
     * Description of the parameter
     * @default undefined no description will be present
     */
    readonly description?: string;
}
/**
 * Properties for a function in a function schema
 */
export interface FunctionProps {
    /**
     * The name of the function
     */
    readonly name: string;
    /**
     * Description of the function
     */
    readonly description: string;
    /**
     * Parameters for the function as a record of parameter name to parameter properties
     * @default {}
     */
    readonly parameters?: Record<string, FunctionParameterProps>;
    /**
     * Whether to require confirmation before executing the function
     * @default RequireConfirmation.DISABLED
     */
    readonly requireConfirmation?: RequireConfirmation;
}
/**
 * Properties for a function schema
 */
export interface FunctionSchemaProps {
    /**
     * Functions defined in the schema
     */
    readonly functions: FunctionProps[];
}
/**
 * Represents a function parameter in a function schema
 */
export declare class FunctionParameter {
    /**
     * The type of the parameter
     */
    readonly type: ParameterType;
    /**
     * Whether the parameter is required
     */
    readonly required: boolean;
    /**
     * Description of the parameter
     * @default undefined no description will be present
     */
    readonly description?: string;
    constructor(props: FunctionParameterProps);
    /**
     * Render the parameter as a CloudFormation property
     * @internal
     */
    _render(): any;
}
/**
 * Represents a function in a function schema
 */
export declare class Function {
    /**
     * The name of the function
     */
    readonly name: string;
    /**
     * Description of the function
     */
    readonly description: string;
    /**
     * Parameters for the function
     */
    readonly parameters: Record<string, FunctionParameter>;
    /**
     * Whether to require confirmation before executing the function
     */
    readonly requireConfirmation: RequireConfirmation;
    constructor(props: FunctionProps);
    /**
     * Render the function as a CloudFormation property
     * @internal
     */
    _render(): any;
}
/**
 * Represents a function schema for a Bedrock Agent Action Group
 */
export declare class FunctionSchema extends ActionGroupSchema {
    /**
     * The functions defined in the schema
     */
    readonly functions: Function[];
    constructor(props: FunctionSchemaProps);
    /**
     * Render the function schema as a CloudFormation property
     * @internal
     */
    _render(): CfnAgent.FunctionSchemaProperty;
}
