"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ValidationError = void 0;
exports.validateStringFieldLength = validateStringFieldLength;
exports.validateFieldPattern = validateFieldPattern;
exports.throwIfInvalid = throwIfInvalid;
const aws_cdk_lib_1 = require("aws-cdk-lib");
/**
 * Error thrown when validation fails
 */
class ValidationError extends Error {
    constructor(message) {
        super(message);
        this.name = 'ValidationError';
    }
}
exports.ValidationError = ValidationError;
/**
 * Validates the length of a string field against minimum and maximum constraints.
 * @param value - The string value to validate
 * @param fieldName - Name of the field being validated (for error messages)
 * @param minLength - Minimum allowed length (defaults to 0)
 * @param maxLength - Maximum allowed length
 * @returns true if validation passes
 * @throws Error if validation fails with current length information
 */
function validateStringFieldLength(params) {
    const currentLength = params.value.length;
    const errors = [];
    // Evaluate only if it is not an unresolved Token
    if (!aws_cdk_lib_1.Token.isUnresolved(params.fieldName)) {
        if (params.value.length > params.maxLength) {
            errors.push(`The field ${params.fieldName} is ${currentLength} characters long but must be less than or equal to ${params.maxLength} characters`);
        }
        if (params.value.length < params.minLength) {
            errors.push(`The field ${params.fieldName} is ${currentLength} characters long but must be at least ${params.minLength} characters`);
        }
    }
    return errors;
}
/**
 * Validates a string field against a regex pattern.
 * @param value - The string value to validate
 * @param fieldName - Name of the field being validated (for error messages)
 * @param pattern - Regular expression pattern to test against
 * @param customMessage - Optional custom error message
 * @returns true if validation passes
 * @throws Error if validation fails with detailed message
 */
function validateFieldPattern(value, fieldName, pattern, customMessage) {
    const errors = [];
    // Evaluate only if it is not an unresolved Token
    if (!aws_cdk_lib_1.Token.isUnresolved(value)) {
        // Verify type
        if (typeof value !== 'string') {
            errors.push(`Expected string for ${fieldName}, got ${typeof value}`);
        }
        // Validate specified regex
        if (!(pattern instanceof RegExp)) {
            errors.push('Pattern must be a valid regular expression');
        }
        // Pattern validation
        if (!pattern.test(value)) {
            const defaultMessage = `The field ${fieldName} with value "${value}" does not match the required pattern ${pattern}`;
            errors.push(customMessage || defaultMessage);
        }
    }
    return errors;
}
function throwIfInvalid(validationFn, param) {
    const errors = validationFn(param);
    if (errors.length > 0) {
        throw new ValidationError(errors.join('\n'));
    }
    return param;
}
//# sourceMappingURL=data:application/json;base64,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