import { Duration, IResource, Resource } from 'aws-cdk-lib/core';
import * as cloudwatch from 'aws-cdk-lib/aws-cloudwatch';
import * as events from 'aws-cdk-lib/aws-events';
import * as iam from 'aws-cdk-lib/aws-iam';
import * as kms from 'aws-cdk-lib/aws-kms';
import { Construct } from 'constructs';
import { AgentActionGroup } from './action-group';
import { IAgentAlias } from './agent-alias';
import { AgentCollaboration } from './agent-collaboration';
import { PromptOverrideConfiguration } from './prompt-override';
import { IBedrockInvokable } from '.././models';
import { Memory } from './memory';
import { CustomOrchestrationExecutor } from './orchestration-executor';
/******************************************************************************
 *                              COMMON
 *****************************************************************************/
/**
 * Represents an Agent, either created with CDK or imported.
 */
export interface IAgent extends IResource, iam.IGrantable {
    /**
     * The ARN of the agent.
     * @attribute
     */
    readonly agentArn: string;
    /**
     * The ID of the Agent.
     * @attribute
     */
    readonly agentId: string;
    /**
     * The IAM role associated to the agent.
     */
    readonly role: iam.IRole;
    /**
     * Optional KMS encryption key associated with this agent
     */
    readonly kmsKey?: kms.IKey;
    /**
     * When this agent was last updated.
     * @attribute
     */
    readonly lastUpdated?: string;
    /**
     * Grant invoke permissions on this agent to an IAM principal.
     * Note: This grant will only work when the grantee is in the same AWS account
     * where the agent is defined. Cross-account invocation is not supported.
     */
    grantInvoke(grantee: iam.IGrantable): iam.Grant;
    /**
     * Defines a CloudWatch event rule triggered by agent events.
     */
    onEvent(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Return the CloudWatch metric for agent count.
     */
    metricCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
}
/******************************************************************************
 *                        ABSTRACT BASE CLASS
 *****************************************************************************/
/**
 * Abstract base class for an Agent.
 * Contains methods and attributes valid for Agents either created with CDK or imported.
 */
export declare abstract class AgentBase extends Resource implements IAgent {
    abstract readonly agentArn: string;
    abstract readonly agentId: string;
    abstract readonly role: iam.IRole;
    abstract readonly kmsKey?: kms.IKey;
    abstract readonly lastUpdated?: string;
    /**
     * The version of the agent.
     * @attribute
     */
    abstract readonly agentVersion: string;
    abstract readonly grantPrincipal: iam.IPrincipal;
    /**
     * Grant invoke permissions on this agent to an IAM principal.
     *
     * @param grantee - The IAM principal to grant invoke permissions to
     * @default - Default grant configuration:
     * - actions: ['bedrock:InvokeAgent']
     * - resourceArns: [this.agentArn]
     * @returns An IAM Grant object representing the granted permissions
     */
    grantInvoke(grantee: iam.IGrantable): iam.Grant;
    /**
     * Creates an EventBridge rule for agent events.
     *
     * @param id - Unique identifier for the rule
     * @param options - Configuration options for the event rule
     * @default - Default event pattern:
     * - source: ['aws.bedrock']
     * - detail: { 'agent-id': [this.agentId] }
     * @returns An EventBridge Rule configured for agent events
     */
    onEvent(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Creates a CloudWatch metric for tracking agent invocations.
     *
     * @param props - Configuration options for the metric
     * @default - Default metric configuration:
     * - namespace: 'AWS/Bedrock'
     * - metricName: 'Invocations'
     * - dimensionsMap: { AgentId: this.agentId }
     * @returns A CloudWatch Metric configured for agent invocation counts
     */
    metricCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
}
/******************************************************************************
 *                        PROPS FOR NEW CONSTRUCT
 *****************************************************************************/
/**
 * Properties for creating a CDK managed Bedrock Agent.
 * TODO: Knowledge bases configuration will be added in a future update
 * TODO: Guardrails configuration will be added in a future update
 * TODO: Inference profile configuration will be added in a future update
 *
 */
export interface AgentProps {
    /**
     * The name of the agent.
     * This will be used as the physical name of the agent.
     *
     * @default - A name is generated by CDK.
     * Supported pattern : ^([0-9a-zA-Z][_-]?){1,100}$
     */
    readonly agentName?: string;
    /**
     * The instruction used by the agent. This determines how the agent will perform his task.
     * This instruction must have a minimum of 40 characters.
     */
    readonly instruction: string;
    /**
     * The foundation model used for orchestration by the agent.
     */
    readonly foundationModel: IBedrockInvokable;
    /**
     * An existing IAM Role to associate with this agent.
     * Use this property when you want to reuse an existing IAM role rather than create a new one.
     * The role must have a trust policy that allows the Bedrock service to assume it.
     * @default - A new role is created for you.
     */
    readonly existingRole?: iam.IRole;
    /**
     * Specifies whether to automatically update the `DRAFT` version of the agent after
     * making changes to the agent. The `DRAFT` version can be continually iterated
     * upon during internal development.
     *
     * @default - false
     */
    readonly shouldPrepareAgent?: boolean;
    /**
     * How long sessions should be kept open for the agent. If no conversation occurs
     * during this time, the session expires and Amazon Bedrock deletes any data
     * provided before the timeout.
     *
     * @default - 10 minutes
     */
    readonly idleSessionTTL?: Duration;
    /**
     * The KMS key of the agent if custom encryption is configured.
     *
     * @default - An AWS managed key is used.
     */
    readonly kmsKey?: kms.IKey;
    /**
     * A description of the agent.
     *
     * @default - No description is provided.
     */
    readonly description?: string;
    /**
     * The Action Groups associated with the agent.
     * @default - Only default action groups (UserInput and CodeInterpreter) are added
     */
    readonly actionGroups?: AgentActionGroup[];
    /**
     * Overrides some prompt templates in different parts of an agent sequence configuration.
     *
     * @default - No overrides are provided.
     */
    readonly promptOverrideConfiguration?: PromptOverrideConfiguration;
    /**
     * Select whether the agent can prompt additional information from the user when it does not have
     * enough information to respond to an utterance
     *
     * @default - false
     */
    readonly userInputEnabled?: boolean;
    /**
     * Select whether the agent can generate, run, and troubleshoot code when trying to complete a task
     *
     * @default - false
     */
    readonly codeInterpreterEnabled?: boolean;
    /**
     * Whether to delete the resource even if it's in use.
     *
     * @default - false
     */
    readonly forceDelete?: boolean;
    /**
     * The type and configuration of the memory to maintain context across multiple sessions and recall past interactions.
     * This can be useful for maintaining continuity in multi-turn conversations and recalling user preferences
     * or past interactions.
     *
     * @see https://docs.aws.amazon.com/bedrock/latest/userguide/agents-memory.html
     * @default - No memory will be used. Agents will retain context from the current session only.
     */
    readonly memory?: Memory;
    /**
     * Configuration for agent collaboration settings, including AgentCollaboratorType and AgentCollaborators.
     * This property allows you to define how the agent collaborates with other agents
     * and what collaborators it can work with.
     *
     * @default - No agent collaboration configuration.
     */
    readonly agentCollaboration?: AgentCollaboration;
    /**
     * The Lambda function to use for custom orchestration.
     * If provided, custom orchestration will be used.
     * If not provided, default orchestration will be used.
     *
     * @default - Default orchestration
     */
    readonly customOrchestrationExecutor?: CustomOrchestrationExecutor;
}
/******************************************************************************
 *                      ATTRS FOR IMPORTED CONSTRUCT
 *****************************************************************************/
/**
 * Attributes for specifying an imported Bedrock Agent.
 */
export interface AgentAttributes {
    /**
     * The ARN of the agent.
     * @attribute
     */
    readonly agentArn: string;
    /**
     * The ARN of the IAM role associated to the agent.
     * @attribute
     */
    readonly roleArn: string;
    /**
     * Optional KMS encryption key associated with this agent
     * @default undefined - An AWS managed key is used
     */
    readonly kmsKeyArn?: string;
    /**
     * When this agent was last updated.
     * @default undefined - No last updated timestamp is provided
     */
    readonly lastUpdated?: string;
    /**
     * The agent version. If no explicit versions have been created,
     * leave this empty to use the DRAFT version. Otherwise, use the
     * version number (e.g. 1).
     * @default 'DRAFT'
     */
    readonly agentVersion?: string;
}
/******************************************************************************
 *                        NEW CONSTRUCT DEFINITION
 *****************************************************************************/
/**
 * Class to create (or import) an Agent with CDK.
 * @cloudformationResource AWS::Bedrock::Agent
 */
export declare class Agent extends AgentBase implements IAgent {
    /** Uniquely identifies this class. */
    static readonly PROPERTY_INJECTION_ID: string;
    /**
     * Static Method for importing an existing Bedrock Agent.
     */
    /**
     * Creates an Agent reference from an existing agent's attributes.
     *
     * @param scope - The construct scope
     * @param id - Identifier of the construct
     * @param attrs - Attributes of the existing agent
     * @default - For attrs.agentVersion: 'DRAFT' if no explicit version is provided
     * @returns An IAgent reference to the existing agent
     */
    static fromAgentAttributes(scope: Construct, id: string, attrs: AgentAttributes): IAgent;
    /**
     * The unique identifier for the agent
     * @attribute
     */
    readonly agentId: string;
    /**
     * The ARN of the agent.
     * @attribute
     */
    readonly agentArn: string;
    /**
     * The version of the agent.
     * @attribute
     */
    readonly agentVersion: string;
    /**
     * The IAM role associated to the agent.
     */
    readonly role: iam.IRole;
    /**
     * Optional KMS encryption key associated with this agent
     */
    readonly kmsKey?: kms.IKey;
    /**
     * When this agent was last updated.
     */
    readonly lastUpdated?: string;
    /**
     * The principal to grant permissions to
     */
    readonly grantPrincipal: iam.IPrincipal;
    /**
     * Default alias of the agent
     */
    readonly testAlias: IAgentAlias;
    /**
     * action groups associated with the ageny
     */
    readonly actionGroups: AgentActionGroup[];
    /**
     * The name of the agent.
     */
    readonly name: string;
    private readonly ROLE_NAME_SUFFIX;
    private readonly MAXLENGTH_FOR_ROLE_NAME;
    private readonly idleSessionTTL;
    private readonly foundationModel;
    private readonly userInputEnabled;
    private readonly codeInterpreterEnabled;
    private readonly agentCollaboration?;
    private readonly customOrchestrationExecutor?;
    private readonly promptOverrideConfiguration?;
    private readonly __resource;
    constructor(scope: Construct, id: string, props: AgentProps);
    /**
     * Adds an action group to the agent and configures necessary permissions.
     *
     * @param actionGroup - The action group to add
     * @default - Default permissions:
     * - Lambda function invoke permissions if executor is present
     * - S3 GetObject permissions if apiSchema.s3File is present
     */
    addActionGroup(actionGroup: AgentActionGroup): void;
    /**
     * Adds a collaborator to the agent and grants necessary permissions.
     * @param agentCollaborator - The collaborator to add
     * @internal This method is used internally by the constructor and should not be called directly.
     */
    private addAgentCollaborator;
    /**
     * Configuration for agent collaboration.
     *
     * @default - No collaboration configuration.
     */
    addActionGroups(...actionGroups: AgentActionGroup[]): void;
    /**
     * Render the action groups
     *
     * @returns Array of AgentActionGroupProperty objects in CloudFormation format
     * @default - Empty array if no action groups are defined
     * @internal This is an internal core function and should not be called directly.
     */
    private renderActionGroups;
    /**
     * Render the agent collaborators.
     *
     * @returns Array of AgentCollaboratorProperty objects in CloudFormation format, or undefined if no collaborators
     * @default - undefined if no collaborators are defined or array is empty
     * @internal This is an internal core function and should not be called directly.
     */
    private renderAgentCollaborators;
    /**
     * Render the custom orchestration.
     *
     * @returns CustomOrchestrationProperty object in CloudFormation format, or undefined if no custom orchestration
     * @default - undefined if no custom orchestration is defined
     * @internal This is an internal core function and should not be called directly.
     */
    private renderCustomOrchestration;
    /**
     * Check if the action group is valid
     *
     * @param actionGroup - The action group to validate
     * @returns Array of validation error messages, empty if valid
     */
    private validateActionGroup;
    /**
     * Generates a unique, deterministic name for AWS resources that includes a hash component.
     * This method ensures consistent naming while avoiding conflicts and adhering to AWS naming constraints.
     * @param scope - The construct scope used for generating unique names
     * @param prefix - The prefix to prepend to the generated name
     * @param options - Configuration options for name generation
     * @param options.maxLength - Maximum length of the generated name
     * @default - maxLength: 256
     * @param options.lower - Convert the generated name to lowercase
     * @default - lower: false
     * @param options.separator - Character(s) to use between name components
     * @default - separator: ''
     * @param options.allowedSpecialCharacters - String of allowed special characters
     * @default - undefined
     * @param options.destroyCreate - Object to include in hash generation for destroy/create operations
     * @default - undefined
     * @returns A string containing the generated name with format: prefix + hash + separator + uniqueName
     * @throws ValidationError if the generated name would exceed maxLength or if prefix is too long
     * @internal
     */
    private generatePhysicalNameHash;
    /**
     * Generates a physical name for the agent.
     * @returns A unique name for the agent with appropriate length constraints
     * @default - Generated name format: 'agent-{hash}-{uniqueName}' with:
     * - maxLength: MAXLENGTH_FOR_ROLE_NAME - '-bedrockagent'.length
     * - lower: true
     * - separator: '-'
     * @protected
     */
    protected generatePhysicalName(): string;
}
