"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Agent = exports.AgentBase = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const core_1 = require("aws-cdk-lib/core");
const bedrock = require("aws-cdk-lib/aws-bedrock");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const events = require("aws-cdk-lib/aws-events");
const iam = require("aws-cdk-lib/aws-iam");
const kms = require("aws-cdk-lib/aws-kms");
const s3 = require("aws-cdk-lib/aws-s3");
const metadata_resource_1 = require("aws-cdk-lib/core/lib/metadata-resource");
const prop_injectable_1 = require("aws-cdk-lib/core/lib/prop-injectable");
// Internal Libs
const action_group_1 = require("./action-group");
const agent_alias_1 = require("./agent-alias");
const api_schema_1 = require("./api-schema");
const validation = require("./validation-helpers");
const orchestration_executor_1 = require("./orchestration-executor");
/******************************************************************************
 *                              CONSTANTS
 *****************************************************************************/
/**
 * The minimum number of characters required for an agent instruction.
 * @internal
 */
const MIN_INSTRUCTION_LENGTH = 40;
/**
 * The maximum length for the node address in permission policy names.
 * @internal
 */
const MAX_POLICY_NAME_NODE_LENGTH = 16;
/******************************************************************************
 *                        ABSTRACT BASE CLASS
 *****************************************************************************/
/**
 * Abstract base class for an Agent.
 * Contains methods and attributes valid for Agents either created with CDK or imported.
 */
class AgentBase extends core_1.Resource {
    /**
     * Grant invoke permissions on this agent to an IAM principal.
     *
     * @param grantee - The IAM principal to grant invoke permissions to
     * @default - Default grant configuration:
     * - actions: ['bedrock:InvokeAgent']
     * - resourceArns: [this.agentArn]
     * @returns An IAM Grant object representing the granted permissions
     */
    grantInvoke(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['bedrock:InvokeAgent'],
            resourceArns: [this.agentArn],
        });
    }
    /**
     * Creates an EventBridge rule for agent events.
     *
     * @param id - Unique identifier for the rule
     * @param options - Configuration options for the event rule
     * @default - Default event pattern:
     * - source: ['aws.bedrock']
     * - detail: { 'agent-id': [this.agentId] }
     * @returns An EventBridge Rule configured for agent events
     */
    onEvent(id, options = {}) {
        // Create rule with minimal props and event pattern
        const rule = new events.Rule(this, id, {
            description: options.description,
            eventPattern: {
                source: ['aws.bedrock'],
                detail: {
                    'agent-id': [this.agentId],
                },
            },
        });
        // Add target if provided
        if (options.target) {
            rule.addTarget(options.target);
        }
        return rule;
    }
    /**
     * Creates a CloudWatch metric for tracking agent invocations.
     *
     * @param props - Configuration options for the metric
     * @default - Default metric configuration:
     * - namespace: 'AWS/Bedrock'
     * - metricName: 'Invocations'
     * - dimensionsMap: { AgentId: this.agentId }
     * @returns A CloudWatch Metric configured for agent invocation counts
     */
    metricCount(props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/Bedrock',
            metricName: 'Invocations',
            dimensionsMap: {
                AgentId: this.agentId,
            },
            ...props,
        }).attachTo(this);
    }
}
exports.AgentBase = AgentBase;
_a = JSII_RTTI_SYMBOL_1;
AgentBase[_a] = { fqn: "@aws-cdk/aws-bedrock-alpha.AgentBase", version: "2.206.0-alpha.0" };
/******************************************************************************
 *                        NEW CONSTRUCT DEFINITION
 *****************************************************************************/
/**
 * Class to create (or import) an Agent with CDK.
 * @cloudformationResource AWS::Bedrock::Agent
 */
let Agent = class Agent extends AgentBase {
    /**
     * Static Method for importing an existing Bedrock Agent.
     */
    /**
     * Creates an Agent reference from an existing agent's attributes.
     *
     * @param scope - The construct scope
     * @param id - Identifier of the construct
     * @param attrs - Attributes of the existing agent
     * @default - For attrs.agentVersion: 'DRAFT' if no explicit version is provided
     * @returns An IAgent reference to the existing agent
     */
    static fromAgentAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_bedrock_alpha_AgentAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromAgentAttributes);
            }
            throw error;
        }
        class Import extends AgentBase {
            constructor() {
                super(...arguments);
                this.agentArn = attrs.agentArn;
                this.agentId = core_1.Arn.split(attrs.agentArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
                this.role = iam.Role.fromRoleArn(scope, `${id}Role`, attrs.roleArn);
                this.kmsKey = attrs.kmsKeyArn ? kms.Key.fromKeyArn(scope, `${id}Key`, attrs.kmsKeyArn) : undefined;
                this.lastUpdated = attrs.lastUpdated;
                this.agentVersion = attrs.agentVersion ?? 'DRAFT';
                this.grantPrincipal = this.role;
            }
        }
        // Return new Agent
        return new Import(scope, id);
    }
    // ------------------------------------------------------
    // CONSTRUCTOR
    // ------------------------------------------------------
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * action groups associated with the ageny
         */
        this.actionGroups = [];
        // ------------------------------------------------------
        // Internal Only
        // ------------------------------------------------------
        this.ROLE_NAME_SUFFIX = '-bedrockagent';
        this.MAXLENGTH_FOR_ROLE_NAME = 64;
        // ------------------------------------------------------
        // Validators
        // ------------------------------------------------------
        /**
         * Check if the action group is valid
         *
         * @param actionGroup - The action group to validate
         * @returns Array of validation error messages, empty if valid
         */
        this.validateActionGroup = (actionGroup) => {
            let errors = [];
            // Find if there is a conflicting action group name
            if (this.actionGroups?.find(ag => ag.name === actionGroup.name)) {
                errors.push('Action group already exists');
            }
            return errors;
        };
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_bedrock_alpha_AgentProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Agent);
            }
            throw error;
        }
        // Enhanced CDK Analytics Telemetry
        (0, metadata_resource_1.addConstructMetadata)(this, props);
        // ------------------------------------------------------
        // Validate props
        // ------------------------------------------------------
        if (props.instruction !== undefined &&
            !core_1.Token.isUnresolved(props.instruction) &&
            props.instruction.length < MIN_INSTRUCTION_LENGTH) {
            throw new core_1.ValidationError(`instruction must be at least ${MIN_INSTRUCTION_LENGTH} characters`, this);
        }
        // Validate idleSessionTTL
        if (props.idleSessionTTL !== undefined &&
            !core_1.Token.isUnresolved(props.idleSessionTTL) &&
            (props.idleSessionTTL.toMinutes() < 1 || props.idleSessionTTL.toMinutes() > 60)) {
            throw new core_1.ValidationError('idleSessionTTL must be between 1 and 60 minutes', this);
        }
        // ------------------------------------------------------
        // Set properties and defaults
        // ------------------------------------------------------
        this.name =
            props.agentName ?? this.generatePhysicalName() + this.ROLE_NAME_SUFFIX;
        this.idleSessionTTL = props.idleSessionTTL ?? core_1.Duration.minutes(10);
        this.userInputEnabled = props.userInputEnabled ?? false;
        this.codeInterpreterEnabled = props.codeInterpreterEnabled ?? false;
        this.foundationModel = props.foundationModel;
        // Optional
        this.promptOverrideConfiguration = props.promptOverrideConfiguration;
        this.kmsKey = props.kmsKey;
        this.customOrchestrationExecutor = props.customOrchestrationExecutor;
        // ------------------------------------------------------
        // Role
        // ------------------------------------------------------
        // If existing role is provided, use it.
        if (props.existingRole) {
            this.role = props.existingRole;
            this.grantPrincipal = this.role;
            // Otherwise, create a new one
        }
        else {
            this.role = new iam.Role(this, 'Role', {
                // generate a role name
                roleName: this.generatePhysicalName() + this.ROLE_NAME_SUFFIX,
                // ensure the role has a trust policy that allows the Bedrock service to assume the role
                assumedBy: new iam.ServicePrincipal('bedrock.amazonaws.com').withConditions({
                    StringEquals: {
                        'aws:SourceAccount': { Ref: 'AWS::AccountId' },
                    },
                    ArnLike: {
                        'aws:SourceArn': core_1.Stack.of(this).formatArn({
                            service: 'bedrock',
                            resource: 'agent',
                            resourceName: '*',
                            arnFormat: core_1.ArnFormat.SLASH_RESOURCE_NAME,
                        }),
                    },
                }),
            });
            this.grantPrincipal = this.role;
        }
        // ------------------------------------------------------
        // Set Lazy Props initial values
        // ------------------------------------------------------
        // Add Default Action Groups
        this.addActionGroup(action_group_1.AgentActionGroup.userInput(this.userInputEnabled));
        this.addActionGroup(action_group_1.AgentActionGroup.codeInterpreter(this.codeInterpreterEnabled));
        // Add specified elems through methods to handle permissions
        // this needs to happen after role creation / assignment
        props.actionGroups?.forEach(ag => {
            this.addActionGroup(ag);
        });
        // Set agent collaboration configuration
        this.agentCollaboration = props.agentCollaboration;
        if (props.agentCollaboration) {
            props.agentCollaboration.collaborators.forEach(ac => {
                this.addAgentCollaborator(ac);
            });
        }
        // Grant permissions for custom orchestration if provided
        if (this.customOrchestrationExecutor?.lambdaFunction) {
            this.customOrchestrationExecutor.lambdaFunction.grantInvoke(this.role);
            this.customOrchestrationExecutor.lambdaFunction.addPermission(`OrchestrationLambdaInvocationPolicy-${this.node.addr.slice(0, MAX_POLICY_NAME_NODE_LENGTH)}`, {
                principal: new iam.ServicePrincipal('bedrock.amazonaws.com'),
                sourceArn: core_1.Lazy.string({ produce: () => this.agentArn }),
                sourceAccount: { Ref: 'AWS::AccountId' },
            });
        }
        // ------------------------------------------------------
        // CFN Props - With Lazy support
        // ------------------------------------------------------
        const cfnProps = {
            actionGroups: core_1.Lazy.any({ produce: () => this.renderActionGroups() }, { omitEmptyArray: true }),
            agentName: this.name,
            agentResourceRoleArn: this.role.roleArn,
            autoPrepare: props.shouldPrepareAgent ?? false,
            customerEncryptionKeyArn: props.kmsKey?.keyArn,
            description: props.description,
            foundationModel: this.foundationModel.invokableArn,
            idleSessionTtlInSeconds: this.idleSessionTTL.toSeconds(),
            instruction: props.instruction,
            memoryConfiguration: props.memory?._render(),
            promptOverrideConfiguration: this.promptOverrideConfiguration?._render(),
            skipResourceInUseCheckOnDelete: props.forceDelete ?? false,
            agentCollaboration: props.agentCollaboration?.type,
            agentCollaborators: core_1.Lazy.any({ produce: () => this.renderAgentCollaborators() }, { omitEmptyArray: true }),
            customOrchestration: this.renderCustomOrchestration(),
            orchestrationType: this.customOrchestrationExecutor ? orchestration_executor_1.OrchestrationType.CUSTOM_ORCHESTRATION : orchestration_executor_1.OrchestrationType.DEFAULT,
        };
        // ------------------------------------------------------
        // L1 Instantiation
        // ------------------------------------------------------
        this.__resource = new bedrock.CfnAgent(this, 'Resource', cfnProps);
        this.agentId = this.__resource.attrAgentId;
        this.agentArn = this.__resource.attrAgentArn;
        this.agentVersion = this.__resource.attrAgentVersion;
        this.lastUpdated = this.__resource.attrUpdatedAt;
        // Add explicit dependency between the agent resource and the agent's role default policy
        // See https://github.com/awslabs/generative-ai-cdk-constructs/issues/899
        if (!props.existingRole) {
            // add the appropriate permissions to use the FM
            const grant = this.foundationModel.grantInvoke(this.role);
            grant.applyBefore(this.__resource);
        }
        this.testAlias = agent_alias_1.AgentAlias.fromAttributes(this, 'DefaultAlias', {
            aliasId: 'TSTALIASID',
            aliasName: 'AgentTestAlias',
            agentVersion: 'DRAFT',
            agent: this,
        });
    }
    // ------------------------------------------------------
    // HELPER METHODS - addX()
    // ------------------------------------------------------
    /**
     * Adds an action group to the agent and configures necessary permissions.
     *
     * @param actionGroup - The action group to add
     * @default - Default permissions:
     * - Lambda function invoke permissions if executor is present
     * - S3 GetObject permissions if apiSchema.s3File is present
     */
    addActionGroup(actionGroup) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_bedrock_alpha_AgentActionGroup(actionGroup);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addActionGroup);
            }
            throw error;
        }
        validation.throwIfInvalid(this.validateActionGroup, actionGroup);
        this.actionGroups.push(actionGroup);
        // Handle permissions to invoke the lambda function
        actionGroup.executor?.lambdaFunction?.grantInvoke(this.role);
        actionGroup.executor?.lambdaFunction?.addPermission(`LambdaInvocationPolicy-${this.node.addr.slice(0, MAX_POLICY_NAME_NODE_LENGTH)}`, {
            principal: new iam.ServicePrincipal('bedrock.amazonaws.com'),
            sourceArn: this.agentArn,
            sourceAccount: { Ref: 'AWS::AccountId' },
        });
        // Handle permissions to access the schema file from S3
        if (actionGroup.apiSchema instanceof api_schema_1.AssetApiSchema) {
            const rendered = actionGroup.apiSchema._render();
            if (!('s3' in rendered) || !rendered.s3) {
                throw new core_1.ValidationError('S3 configuration is missing in AssetApiSchema', this);
            }
            const s3Config = rendered.s3;
            if (!('s3BucketName' in s3Config) || !('s3ObjectKey' in s3Config)) {
                throw new core_1.ValidationError('S3 bucket name and object key are required in AssetApiSchema', this);
            }
            const bucketName = s3Config.s3BucketName;
            const objectKey = s3Config.s3ObjectKey;
            if (!bucketName || bucketName.trim() === '') {
                throw new core_1.ValidationError('S3 bucket name cannot be empty in AssetApiSchema', this);
            }
            if (!objectKey || objectKey.trim() === '') {
                throw new core_1.ValidationError('S3 object key cannot be empty in AssetApiSchema', this);
            }
            const bucket = s3.Bucket.fromBucketName(this, `${actionGroup.name}SchemaBucket`, bucketName);
            bucket.grantRead(this.role, objectKey);
        }
        else if (actionGroup.apiSchema instanceof api_schema_1.S3ApiSchema) {
            const s3File = actionGroup.apiSchema.s3File;
            if (!s3File) {
                throw new core_1.ValidationError('S3 file configuration is missing in S3ApiSchema', this);
            }
            if (!s3File.bucketName || s3File.bucketName.trim() === '') {
                throw new core_1.ValidationError('S3 bucket name cannot be empty in S3ApiSchema', this);
            }
            if (!s3File.objectKey || s3File.objectKey.trim() === '') {
                throw new core_1.ValidationError('S3 object key cannot be empty in S3ApiSchema', this);
            }
            const bucket = s3.Bucket.fromBucketName(this, `${actionGroup.name}SchemaBucket`, s3File.bucketName);
            bucket.grantRead(this.role, s3File.objectKey);
        }
    }
    /**
     * Adds a collaborator to the agent and grants necessary permissions.
     * @param agentCollaborator - The collaborator to add
     * @internal This method is used internally by the constructor and should not be called directly.
     */
    addAgentCollaborator(agentCollaborator) {
        agentCollaborator.grant(this.role);
    }
    /**
     * Configuration for agent collaboration.
     *
     * @default - No collaboration configuration.
     */
    addActionGroups(...actionGroups) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_bedrock_alpha_AgentActionGroup(actionGroups);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addActionGroups);
            }
            throw error;
        }
        actionGroups.forEach(ag => this.addActionGroup(ag));
    }
    // ------------------------------------------------------
    // Lazy Renderers
    // ------------------------------------------------------
    /**
     * Render the action groups
     *
     * @returns Array of AgentActionGroupProperty objects in CloudFormation format
     * @default - Empty array if no action groups are defined
     * @internal This is an internal core function and should not be called directly.
     */
    renderActionGroups() {
        const actionGroupsCfn = [];
        // Build the associations in the CFN format
        this.actionGroups.forEach(ag => {
            actionGroupsCfn.push(ag._render());
        });
        return actionGroupsCfn;
    }
    /**
     * Render the agent collaborators.
     *
     * @returns Array of AgentCollaboratorProperty objects in CloudFormation format, or undefined if no collaborators
     * @default - undefined if no collaborators are defined or array is empty
     * @internal This is an internal core function and should not be called directly.
     */
    renderAgentCollaborators() {
        if (!this.agentCollaboration) {
            return undefined;
        }
        return this.agentCollaboration.collaborators.map(ac => ac._render());
    }
    /**
     * Render the custom orchestration.
     *
     * @returns CustomOrchestrationProperty object in CloudFormation format, or undefined if no custom orchestration
     * @default - undefined if no custom orchestration is defined
     * @internal This is an internal core function and should not be called directly.
     */
    renderCustomOrchestration() {
        if (!this.customOrchestrationExecutor) {
            return undefined;
        }
        return {
            executor: {
                lambda: this.customOrchestrationExecutor.lambdaFunction.functionArn,
            },
        };
    }
    /**
     * Generates a unique, deterministic name for AWS resources that includes a hash component.
     * This method ensures consistent naming while avoiding conflicts and adhering to AWS naming constraints.
     * @param scope - The construct scope used for generating unique names
     * @param prefix - The prefix to prepend to the generated name
     * @param options - Configuration options for name generation
     * @param options.maxLength - Maximum length of the generated name
     * @default - maxLength: 256
     * @param options.lower - Convert the generated name to lowercase
     * @default - lower: false
     * @param options.separator - Character(s) to use between name components
     * @default - separator: ''
     * @param options.allowedSpecialCharacters - String of allowed special characters
     * @default - undefined
     * @param options.destroyCreate - Object to include in hash generation for destroy/create operations
     * @default - undefined
     * @returns A string containing the generated name with format: prefix + hash + separator + uniqueName
     * @throws ValidationError if the generated name would exceed maxLength or if prefix is too long
     * @internal
     */
    generatePhysicalNameHash(scope, prefix, options) {
        const objectToHash = (obj) => {
            if (obj === undefined) {
                return '';
            }
            const jsonString = JSON.stringify(obj);
            const hash = crypto.createHash('sha256');
            return hash.update(jsonString).digest('hex').slice(0, 7);
        };
        const { maxLength = 256, lower = false, separator = '', allowedSpecialCharacters = undefined, destroyCreate = undefined, } = options ?? {};
        const hash = objectToHash(destroyCreate);
        if (maxLength < (prefix + hash + separator).length) {
            throw new core_1.ValidationError('The prefix is longer than the maximum length.', this);
        }
        const uniqueName = core_1.Names.uniqueResourceName(scope, { maxLength: maxLength - (prefix + hash + separator).length, separator, allowedSpecialCharacters });
        const name = `${prefix}${hash}${separator}${uniqueName}`;
        if (name.length > maxLength) {
            throw new core_1.ValidationError(`The generated name is longer than the maximum length of ${maxLength}`, this);
        }
        return lower ? name.toLowerCase() : name;
    }
    /**
     * Generates a physical name for the agent.
     * @returns A unique name for the agent with appropriate length constraints
     * @default - Generated name format: 'agent-{hash}-{uniqueName}' with:
     * - maxLength: MAXLENGTH_FOR_ROLE_NAME - '-bedrockagent'.length
     * - lower: true
     * - separator: '-'
     * @protected
     */
    generatePhysicalName() {
        const maxLength = this.MAXLENGTH_FOR_ROLE_NAME - this.ROLE_NAME_SUFFIX.length;
        return this.generatePhysicalNameHash(this, 'agent', {
            maxLength,
            lower: true,
            separator: '-',
        });
    }
};
exports.Agent = Agent;
_b = JSII_RTTI_SYMBOL_1;
Agent[_b] = { fqn: "@aws-cdk/aws-bedrock-alpha.Agent", version: "2.206.0-alpha.0" };
/** Uniquely identifies this class. */
Agent.PROPERTY_INJECTION_ID = '@aws-cdk.aws-bedrock-alpha.Agent';
__decorate([
    (0, metadata_resource_1.MethodMetadata)()
], Agent.prototype, "addActionGroup", null);
__decorate([
    (0, metadata_resource_1.MethodMetadata)()
], Agent.prototype, "addActionGroups", null);
exports.Agent = Agent = __decorate([
    prop_injectable_1.propertyInjectable
], Agent);
//# sourceMappingURL=data:application/json;base64,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