import { Construct } from 'constructs';
import * as s3_assets from 'aws-cdk-lib/aws-s3-assets';
import { CfnAgent } from 'aws-cdk-lib/aws-bedrock';
import { IBucket, Location } from 'aws-cdk-lib/aws-s3';
import { ActionGroupSchema } from './schema-base';
/******************************************************************************
 *                       API SCHEMA CLASS
 *****************************************************************************/
/**
 * Represents the concept of an API Schema for a Bedrock Agent Action Group.
 */
export declare abstract class ApiSchema extends ActionGroupSchema {
    /**
     * Creates an API Schema from a local file.
     * @param path - the path to the local file containing the OpenAPI schema for the action group
     */
    static fromLocalAsset(path: string): AssetApiSchema;
    /**
     * Creates an API Schema from an inline string.
     * @param schema - the JSON or YAML payload defining the OpenAPI schema for the action group
     */
    static fromInline(schema: string): InlineApiSchema;
    /**
     * Creates an API Schema from an S3 File
     * @param bucket - the bucket containing the local file containing the OpenAPI schema for the action group
     * @param objectKey - object key in the bucket
     */
    static fromS3File(bucket: IBucket, objectKey: string): S3ApiSchema;
    /**
     * The S3 location of the API schema file, if using an S3-based schema.
     * Contains the bucket name and object key information.
     */
    readonly s3File?: Location;
    /**
     * The inline OpenAPI schema definition as a string, if using an inline schema.
     * Can be in JSON or YAML format.
     */
    readonly inlineSchema?: string;
    protected constructor(s3File?: Location, inlineSchema?: string);
    /**
     * Format as CFN properties
     *
     * @internal This is an internal core function and should not be called directly.
     */
    abstract _render(): CfnAgent.APISchemaProperty;
}
/**
 * API Schema from a local asset.
 *
 * The asset is uploaded to an S3 staging bucket, then moved to its final location
 * by CloudFormation during deployment.
 */
export declare class AssetApiSchema extends ApiSchema {
    private readonly path;
    private readonly options;
    private asset?;
    constructor(path: string, options?: s3_assets.AssetOptions);
    /**
     * Binds this API schema to a construct scope.
     * This method initializes the S3 asset if it hasn't been initialized yet.
     * Must be called before rendering the schema as CFN properties.
     *
     * @param scope - The construct scope to bind to
     */
    bind(scope: Construct): void;
    /**
     * Format as CFN properties
     * @internal This is an internal core function and should not be called directly.
     */
    _render(): CfnAgent.APISchemaProperty;
}
/**
 * Class to define an API Schema from an inline string.
 * The schema can be provided directly as a string in either JSON or YAML format.
 */
export declare class InlineApiSchema extends ApiSchema {
    private readonly schema;
    constructor(schema: string);
    /**
     * @internal This is an internal core function and should not be called directly.
     */
    _render(): CfnAgent.APISchemaProperty;
}
/**
 * Class to define an API Schema from an S3 object.
 */
export declare class S3ApiSchema extends ApiSchema {
    private readonly location;
    constructor(location: Location);
    /**
     * @internal This is an internal core function and should not be called directly.
     */
    _render(): CfnAgent.APISchemaProperty;
}
