"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3ApiSchema = exports.InlineApiSchema = exports.AssetApiSchema = exports.ApiSchema = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const s3_assets = require("aws-cdk-lib/aws-s3-assets");
const schema_base_1 = require("./schema-base");
/**
 * Error thrown when an ApiSchema is not properly initialized.
 */
class ApiSchemaError extends Error {
    constructor(message, cause) {
        super(message);
        this.cause = cause;
        this.name = 'ApiSchemaError';
    }
}
/******************************************************************************
 *                       API SCHEMA CLASS
 *****************************************************************************/
/**
 * Represents the concept of an API Schema for a Bedrock Agent Action Group.
 */
class ApiSchema extends schema_base_1.ActionGroupSchema {
    /**
     * Creates an API Schema from a local file.
     * @param path - the path to the local file containing the OpenAPI schema for the action group
     */
    static fromLocalAsset(path) {
        return new AssetApiSchema(path);
    }
    /**
     * Creates an API Schema from an inline string.
     * @param schema - the JSON or YAML payload defining the OpenAPI schema for the action group
     */
    static fromInline(schema) {
        return new InlineApiSchema(schema);
    }
    /**
     * Creates an API Schema from an S3 File
     * @param bucket - the bucket containing the local file containing the OpenAPI schema for the action group
     * @param objectKey - object key in the bucket
     */
    static fromS3File(bucket, objectKey) {
        return new S3ApiSchema({
            bucketName: bucket.bucketName,
            objectKey: objectKey,
        });
    }
    constructor(s3File, inlineSchema) {
        super();
        this.s3File = s3File;
        this.inlineSchema = inlineSchema;
    }
}
exports.ApiSchema = ApiSchema;
_a = JSII_RTTI_SYMBOL_1;
ApiSchema[_a] = { fqn: "@aws-cdk/aws-bedrock-alpha.ApiSchema", version: "2.206.0-alpha.0" };
/**
 * API Schema from a local asset.
 *
 * The asset is uploaded to an S3 staging bucket, then moved to its final location
 * by CloudFormation during deployment.
 */
class AssetApiSchema extends ApiSchema {
    constructor(path, options = {}) {
        super();
        this.path = path;
        this.options = options;
    }
    /**
     * Binds this API schema to a construct scope.
     * This method initializes the S3 asset if it hasn't been initialized yet.
     * Must be called before rendering the schema as CFN properties.
     *
     * @param scope - The construct scope to bind to
     */
    bind(scope) {
        // If the same AssetApiSchema is used multiple times, retain only the first instantiation
        if (!this.asset) {
            this.asset = new s3_assets.Asset(scope, 'Schema', {
                path: this.path,
                ...this.options,
            });
            // Note: Permissions will be granted by the Agent construct when adding the action group
        }
    }
    /**
     * Format as CFN properties
     * @internal This is an internal core function and should not be called directly.
     */
    _render() {
        if (!this.asset) {
            throw new ApiSchemaError('ApiSchema must be bound to a scope before rendering. Call bind() first.', 'Asset not initialized');
        }
        return {
            s3: {
                s3BucketName: this.asset.s3BucketName,
                s3ObjectKey: this.asset.s3ObjectKey,
            },
        };
    }
}
exports.AssetApiSchema = AssetApiSchema;
_b = JSII_RTTI_SYMBOL_1;
AssetApiSchema[_b] = { fqn: "@aws-cdk/aws-bedrock-alpha.AssetApiSchema", version: "2.206.0-alpha.0" };
// ------------------------------------------------------
/**
 * Class to define an API Schema from an inline string.
 * The schema can be provided directly as a string in either JSON or YAML format.
 */
class InlineApiSchema extends ApiSchema {
    constructor(schema) {
        super(undefined, schema);
        this.schema = schema;
    }
    /**
     * @internal This is an internal core function and should not be called directly.
     */
    _render() {
        return {
            payload: this.schema,
        };
    }
}
exports.InlineApiSchema = InlineApiSchema;
_c = JSII_RTTI_SYMBOL_1;
InlineApiSchema[_c] = { fqn: "@aws-cdk/aws-bedrock-alpha.InlineApiSchema", version: "2.206.0-alpha.0" };
// ------------------------------------------------------
// S3 File
// ------------------------------------------------------
/**
 * Class to define an API Schema from an S3 object.
 */
class S3ApiSchema extends ApiSchema {
    constructor(location) {
        super(location, undefined);
        this.location = location;
    }
    /**
     * @internal This is an internal core function and should not be called directly.
     */
    _render() {
        return {
            s3: {
                s3BucketName: this.location.bucketName,
                s3ObjectKey: this.location.objectKey,
            },
        };
    }
}
exports.S3ApiSchema = S3ApiSchema;
_d = JSII_RTTI_SYMBOL_1;
S3ApiSchema[_d] = { fqn: "@aws-cdk/aws-bedrock-alpha.S3ApiSchema", version: "2.206.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBpLXNjaGVtYS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImFwaS1zY2hlbWEudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFDQSx1REFBdUQ7QUFHdkQsK0NBQWtEO0FBRWxEOztHQUVHO0FBQ0gsTUFBTSxjQUFlLFNBQVEsS0FBSztJQUNoQyxZQUFZLE9BQWUsRUFBa0IsS0FBYztRQUN6RCxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUM7UUFENEIsVUFBSyxHQUFMLEtBQUssQ0FBUztRQUV6RCxJQUFJLENBQUMsSUFBSSxHQUFHLGdCQUFnQixDQUFDO0tBQzlCO0NBQ0Y7QUFFRDs7K0VBRStFO0FBQy9FOztHQUVHO0FBQ0gsTUFBc0IsU0FBVSxTQUFRLCtCQUFpQjtJQUN2RDs7O09BR0c7SUFDSSxNQUFNLENBQUMsY0FBYyxDQUFDLElBQVk7UUFDdkMsT0FBTyxJQUFJLGNBQWMsQ0FBQyxJQUFJLENBQUMsQ0FBQztLQUNqQztJQUVEOzs7T0FHRztJQUNJLE1BQU0sQ0FBQyxVQUFVLENBQUMsTUFBYztRQUNyQyxPQUFPLElBQUksZUFBZSxDQUFDLE1BQU0sQ0FBQyxDQUFDO0tBQ3BDO0lBRUQ7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxVQUFVLENBQUMsTUFBZSxFQUFFLFNBQWlCO1FBQ3pELE9BQU8sSUFBSSxXQUFXLENBQUM7WUFDckIsVUFBVSxFQUFFLE1BQU0sQ0FBQyxVQUFVO1lBQzdCLFNBQVMsRUFBRSxTQUFTO1NBQ3JCLENBQUMsQ0FBQztLQUNKO0lBY0QsWUFBc0IsTUFBaUIsRUFBRSxZQUFxQjtRQUM1RCxLQUFLLEVBQUUsQ0FBQztRQUNSLElBQUksQ0FBQyxNQUFNLEdBQUcsTUFBTSxDQUFDO1FBQ3JCLElBQUksQ0FBQyxZQUFZLEdBQUcsWUFBWSxDQUFDO0tBQ2xDOztBQTdDSCw4QkFxREM7OztBQUVEOzs7OztHQUtHO0FBQ0gsTUFBYSxjQUFlLFNBQVEsU0FBUztJQUczQyxZQUE2QixJQUFZLEVBQW1CLFVBQWtDLEVBQUU7UUFDOUYsS0FBSyxFQUFFLENBQUM7UUFEbUIsU0FBSSxHQUFKLElBQUksQ0FBUTtRQUFtQixZQUFPLEdBQVAsT0FBTyxDQUE2QjtLQUUvRjtJQUVEOzs7Ozs7T0FNRztJQUNJLElBQUksQ0FBQyxLQUFnQjtRQUMxQix5RkFBeUY7UUFDekYsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUNoQixJQUFJLENBQUMsS0FBSyxHQUFHLElBQUksU0FBUyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO2dCQUNoRCxJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUk7Z0JBQ2YsR0FBRyxJQUFJLENBQUMsT0FBTzthQUNoQixDQUFDLENBQUM7WUFDSCx3RkFBd0Y7UUFDMUYsQ0FBQztLQUNGO0lBRUQ7OztPQUdHO0lBQ0ksT0FBTztRQUNaLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDaEIsTUFBTSxJQUFJLGNBQWMsQ0FBQyx5RUFBeUUsRUFBRSx1QkFBdUIsQ0FBQyxDQUFDO1FBQy9ILENBQUM7UUFFRCxPQUFPO1lBQ0wsRUFBRSxFQUFFO2dCQUNGLFlBQVksRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVk7Z0JBQ3JDLFdBQVcsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVc7YUFDcEM7U0FDRixDQUFDO0tBQ0g7O0FBeENILHdDQXlDQzs7O0FBRUQseURBQXlEO0FBQ3pEOzs7R0FHRztBQUNILE1BQWEsZUFBZ0IsU0FBUSxTQUFTO0lBQzVDLFlBQTZCLE1BQWM7UUFDekMsS0FBSyxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQztRQURFLFdBQU0sR0FBTixNQUFNLENBQVE7S0FFMUM7SUFFRDs7T0FFRztJQUNJLE9BQU87UUFDWixPQUFPO1lBQ0wsT0FBTyxFQUFFLElBQUksQ0FBQyxNQUFNO1NBQ3JCLENBQUM7S0FDSDs7QUFaSCwwQ0FhQzs7O0FBRUQseURBQXlEO0FBQ3pELFVBQVU7QUFDVix5REFBeUQ7QUFDekQ7O0dBRUc7QUFDSCxNQUFhLFdBQVksU0FBUSxTQUFTO0lBQ3hDLFlBQTZCLFFBQWtCO1FBQzdDLEtBQUssQ0FBQyxRQUFRLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFEQSxhQUFRLEdBQVIsUUFBUSxDQUFVO0tBRTlDO0lBQ0Q7O09BRUc7SUFDSSxPQUFPO1FBQ1osT0FBTztZQUNMLEVBQUUsRUFBRTtnQkFDRixZQUFZLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxVQUFVO2dCQUN0QyxXQUFXLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxTQUFTO2FBQ3JDO1NBQ0YsQ0FBQztLQUNIOztBQWRILGtDQWVDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgKiBhcyBzM19hc3NldHMgZnJvbSAnYXdzLWNkay1saWIvYXdzLXMzLWFzc2V0cyc7XG5pbXBvcnQgeyBDZm5BZ2VudCB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1iZWRyb2NrJztcbmltcG9ydCB7IElCdWNrZXQsIExvY2F0aW9uIH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLXMzJztcbmltcG9ydCB7IEFjdGlvbkdyb3VwU2NoZW1hIH0gZnJvbSAnLi9zY2hlbWEtYmFzZSc7XG5cbi8qKlxuICogRXJyb3IgdGhyb3duIHdoZW4gYW4gQXBpU2NoZW1hIGlzIG5vdCBwcm9wZXJseSBpbml0aWFsaXplZC5cbiAqL1xuY2xhc3MgQXBpU2NoZW1hRXJyb3IgZXh0ZW5kcyBFcnJvciB7XG4gIGNvbnN0cnVjdG9yKG1lc3NhZ2U6IHN0cmluZywgcHVibGljIHJlYWRvbmx5IGNhdXNlPzogc3RyaW5nKSB7XG4gICAgc3VwZXIobWVzc2FnZSk7XG4gICAgdGhpcy5uYW1lID0gJ0FwaVNjaGVtYUVycm9yJztcbiAgfVxufVxuXG4vKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqXG4gKiAgICAgICAgICAgICAgICAgICAgICAgQVBJIFNDSEVNQSBDTEFTU1xuICoqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqL1xuLyoqXG4gKiBSZXByZXNlbnRzIHRoZSBjb25jZXB0IG9mIGFuIEFQSSBTY2hlbWEgZm9yIGEgQmVkcm9jayBBZ2VudCBBY3Rpb24gR3JvdXAuXG4gKi9cbmV4cG9ydCBhYnN0cmFjdCBjbGFzcyBBcGlTY2hlbWEgZXh0ZW5kcyBBY3Rpb25Hcm91cFNjaGVtYSB7XG4gIC8qKlxuICAgKiBDcmVhdGVzIGFuIEFQSSBTY2hlbWEgZnJvbSBhIGxvY2FsIGZpbGUuXG4gICAqIEBwYXJhbSBwYXRoIC0gdGhlIHBhdGggdG8gdGhlIGxvY2FsIGZpbGUgY29udGFpbmluZyB0aGUgT3BlbkFQSSBzY2hlbWEgZm9yIHRoZSBhY3Rpb24gZ3JvdXBcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUxvY2FsQXNzZXQocGF0aDogc3RyaW5nKTogQXNzZXRBcGlTY2hlbWEge1xuICAgIHJldHVybiBuZXcgQXNzZXRBcGlTY2hlbWEocGF0aCk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhbiBBUEkgU2NoZW1hIGZyb20gYW4gaW5saW5lIHN0cmluZy5cbiAgICogQHBhcmFtIHNjaGVtYSAtIHRoZSBKU09OIG9yIFlBTUwgcGF5bG9hZCBkZWZpbmluZyB0aGUgT3BlbkFQSSBzY2hlbWEgZm9yIHRoZSBhY3Rpb24gZ3JvdXBcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUlubGluZShzY2hlbWE6IHN0cmluZyk6IElubGluZUFwaVNjaGVtYSB7XG4gICAgcmV0dXJuIG5ldyBJbmxpbmVBcGlTY2hlbWEoc2NoZW1hKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGFuIEFQSSBTY2hlbWEgZnJvbSBhbiBTMyBGaWxlXG4gICAqIEBwYXJhbSBidWNrZXQgLSB0aGUgYnVja2V0IGNvbnRhaW5pbmcgdGhlIGxvY2FsIGZpbGUgY29udGFpbmluZyB0aGUgT3BlbkFQSSBzY2hlbWEgZm9yIHRoZSBhY3Rpb24gZ3JvdXBcbiAgICogQHBhcmFtIG9iamVjdEtleSAtIG9iamVjdCBrZXkgaW4gdGhlIGJ1Y2tldFxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tUzNGaWxlKGJ1Y2tldDogSUJ1Y2tldCwgb2JqZWN0S2V5OiBzdHJpbmcpOiBTM0FwaVNjaGVtYSB7XG4gICAgcmV0dXJuIG5ldyBTM0FwaVNjaGVtYSh7XG4gICAgICBidWNrZXROYW1lOiBidWNrZXQuYnVja2V0TmFtZSxcbiAgICAgIG9iamVjdEtleTogb2JqZWN0S2V5LFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBTMyBsb2NhdGlvbiBvZiB0aGUgQVBJIHNjaGVtYSBmaWxlLCBpZiB1c2luZyBhbiBTMy1iYXNlZCBzY2hlbWEuXG4gICAqIENvbnRhaW5zIHRoZSBidWNrZXQgbmFtZSBhbmQgb2JqZWN0IGtleSBpbmZvcm1hdGlvbi5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBzM0ZpbGU/OiBMb2NhdGlvbjtcblxuICAvKipcbiAgICogVGhlIGlubGluZSBPcGVuQVBJIHNjaGVtYSBkZWZpbml0aW9uIGFzIGEgc3RyaW5nLCBpZiB1c2luZyBhbiBpbmxpbmUgc2NoZW1hLlxuICAgKiBDYW4gYmUgaW4gSlNPTiBvciBZQU1MIGZvcm1hdC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBpbmxpbmVTY2hlbWE/OiBzdHJpbmc7XG5cbiAgcHJvdGVjdGVkIGNvbnN0cnVjdG9yKHMzRmlsZT86IExvY2F0aW9uLCBpbmxpbmVTY2hlbWE/OiBzdHJpbmcpIHtcbiAgICBzdXBlcigpO1xuICAgIHRoaXMuczNGaWxlID0gczNGaWxlO1xuICAgIHRoaXMuaW5saW5lU2NoZW1hID0gaW5saW5lU2NoZW1hO1xuICB9XG5cbiAgLyoqXG4gICAqIEZvcm1hdCBhcyBDRk4gcHJvcGVydGllc1xuICAgKlxuICAgKiBAaW50ZXJuYWwgVGhpcyBpcyBhbiBpbnRlcm5hbCBjb3JlIGZ1bmN0aW9uIGFuZCBzaG91bGQgbm90IGJlIGNhbGxlZCBkaXJlY3RseS5cbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCBfcmVuZGVyKCk6IENmbkFnZW50LkFQSVNjaGVtYVByb3BlcnR5O1xufVxuXG4vKipcbiAqIEFQSSBTY2hlbWEgZnJvbSBhIGxvY2FsIGFzc2V0LlxuICpcbiAqIFRoZSBhc3NldCBpcyB1cGxvYWRlZCB0byBhbiBTMyBzdGFnaW5nIGJ1Y2tldCwgdGhlbiBtb3ZlZCB0byBpdHMgZmluYWwgbG9jYXRpb25cbiAqIGJ5IENsb3VkRm9ybWF0aW9uIGR1cmluZyBkZXBsb3ltZW50LlxuICovXG5leHBvcnQgY2xhc3MgQXNzZXRBcGlTY2hlbWEgZXh0ZW5kcyBBcGlTY2hlbWEge1xuICBwcml2YXRlIGFzc2V0PzogczNfYXNzZXRzLkFzc2V0O1xuXG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgcGF0aDogc3RyaW5nLCBwcml2YXRlIHJlYWRvbmx5IG9wdGlvbnM6IHMzX2Fzc2V0cy5Bc3NldE9wdGlvbnMgPSB7fSkge1xuICAgIHN1cGVyKCk7XG4gIH1cblxuICAvKipcbiAgICogQmluZHMgdGhpcyBBUEkgc2NoZW1hIHRvIGEgY29uc3RydWN0IHNjb3BlLlxuICAgKiBUaGlzIG1ldGhvZCBpbml0aWFsaXplcyB0aGUgUzMgYXNzZXQgaWYgaXQgaGFzbid0IGJlZW4gaW5pdGlhbGl6ZWQgeWV0LlxuICAgKiBNdXN0IGJlIGNhbGxlZCBiZWZvcmUgcmVuZGVyaW5nIHRoZSBzY2hlbWEgYXMgQ0ZOIHByb3BlcnRpZXMuXG4gICAqXG4gICAqIEBwYXJhbSBzY29wZSAtIFRoZSBjb25zdHJ1Y3Qgc2NvcGUgdG8gYmluZCB0b1xuICAgKi9cbiAgcHVibGljIGJpbmQoc2NvcGU6IENvbnN0cnVjdCk6IHZvaWQge1xuICAgIC8vIElmIHRoZSBzYW1lIEFzc2V0QXBpU2NoZW1hIGlzIHVzZWQgbXVsdGlwbGUgdGltZXMsIHJldGFpbiBvbmx5IHRoZSBmaXJzdCBpbnN0YW50aWF0aW9uXG4gICAgaWYgKCF0aGlzLmFzc2V0KSB7XG4gICAgICB0aGlzLmFzc2V0ID0gbmV3IHMzX2Fzc2V0cy5Bc3NldChzY29wZSwgJ1NjaGVtYScsIHtcbiAgICAgICAgcGF0aDogdGhpcy5wYXRoLFxuICAgICAgICAuLi50aGlzLm9wdGlvbnMsXG4gICAgICB9KTtcbiAgICAgIC8vIE5vdGU6IFBlcm1pc3Npb25zIHdpbGwgYmUgZ3JhbnRlZCBieSB0aGUgQWdlbnQgY29uc3RydWN0IHdoZW4gYWRkaW5nIHRoZSBhY3Rpb24gZ3JvdXBcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogRm9ybWF0IGFzIENGTiBwcm9wZXJ0aWVzXG4gICAqIEBpbnRlcm5hbCBUaGlzIGlzIGFuIGludGVybmFsIGNvcmUgZnVuY3Rpb24gYW5kIHNob3VsZCBub3QgYmUgY2FsbGVkIGRpcmVjdGx5LlxuICAgKi9cbiAgcHVibGljIF9yZW5kZXIoKTogQ2ZuQWdlbnQuQVBJU2NoZW1hUHJvcGVydHkge1xuICAgIGlmICghdGhpcy5hc3NldCkge1xuICAgICAgdGhyb3cgbmV3IEFwaVNjaGVtYUVycm9yKCdBcGlTY2hlbWEgbXVzdCBiZSBib3VuZCB0byBhIHNjb3BlIGJlZm9yZSByZW5kZXJpbmcuIENhbGwgYmluZCgpIGZpcnN0LicsICdBc3NldCBub3QgaW5pdGlhbGl6ZWQnKTtcbiAgICB9XG5cbiAgICByZXR1cm4ge1xuICAgICAgczM6IHtcbiAgICAgICAgczNCdWNrZXROYW1lOiB0aGlzLmFzc2V0LnMzQnVja2V0TmFtZSxcbiAgICAgICAgczNPYmplY3RLZXk6IHRoaXMuYXNzZXQuczNPYmplY3RLZXksXG4gICAgICB9LFxuICAgIH07XG4gIH1cbn1cblxuLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4vKipcbiAqIENsYXNzIHRvIGRlZmluZSBhbiBBUEkgU2NoZW1hIGZyb20gYW4gaW5saW5lIHN0cmluZy5cbiAqIFRoZSBzY2hlbWEgY2FuIGJlIHByb3ZpZGVkIGRpcmVjdGx5IGFzIGEgc3RyaW5nIGluIGVpdGhlciBKU09OIG9yIFlBTUwgZm9ybWF0LlxuICovXG5leHBvcnQgY2xhc3MgSW5saW5lQXBpU2NoZW1hIGV4dGVuZHMgQXBpU2NoZW1hIHtcbiAgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBzY2hlbWE6IHN0cmluZykge1xuICAgIHN1cGVyKHVuZGVmaW5lZCwgc2NoZW1hKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAaW50ZXJuYWwgVGhpcyBpcyBhbiBpbnRlcm5hbCBjb3JlIGZ1bmN0aW9uIGFuZCBzaG91bGQgbm90IGJlIGNhbGxlZCBkaXJlY3RseS5cbiAgICovXG4gIHB1YmxpYyBfcmVuZGVyKCk6IENmbkFnZW50LkFQSVNjaGVtYVByb3BlcnR5IHtcbiAgICByZXR1cm4ge1xuICAgICAgcGF5bG9hZDogdGhpcy5zY2hlbWEsXG4gICAgfTtcbiAgfVxufVxuXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbi8vIFMzIEZpbGVcbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuLyoqXG4gKiBDbGFzcyB0byBkZWZpbmUgYW4gQVBJIFNjaGVtYSBmcm9tIGFuIFMzIG9iamVjdC5cbiAqL1xuZXhwb3J0IGNsYXNzIFMzQXBpU2NoZW1hIGV4dGVuZHMgQXBpU2NoZW1hIHtcbiAgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBsb2NhdGlvbjogTG9jYXRpb24pIHtcbiAgICBzdXBlcihsb2NhdGlvbiwgdW5kZWZpbmVkKTtcbiAgfVxuICAvKipcbiAgICogQGludGVybmFsIFRoaXMgaXMgYW4gaW50ZXJuYWwgY29yZSBmdW5jdGlvbiBhbmQgc2hvdWxkIG5vdCBiZSBjYWxsZWQgZGlyZWN0bHkuXG4gICAqL1xuICBwdWJsaWMgX3JlbmRlcigpOiBDZm5BZ2VudC5BUElTY2hlbWFQcm9wZXJ0eSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIHMzOiB7XG4gICAgICAgIHMzQnVja2V0TmFtZTogdGhpcy5sb2NhdGlvbi5idWNrZXROYW1lLFxuICAgICAgICBzM09iamVjdEtleTogdGhpcy5sb2NhdGlvbi5vYmplY3RLZXksXG4gICAgICB9LFxuICAgIH07XG4gIH1cbn1cbiJdfQ==