"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PromptOverrideConfiguration = exports.AgentStepType = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const validation = require("./validation-helpers");
/**
 * The step in the agent sequence that this prompt configuration applies to.
 */
var AgentStepType;
(function (AgentStepType) {
    /**
     * Pre-processing step that prepares the user input for orchestration.
     */
    AgentStepType["PRE_PROCESSING"] = "PRE_PROCESSING";
    /**
     * Main orchestration step that determines the agent's actions.
     */
    AgentStepType["ORCHESTRATION"] = "ORCHESTRATION";
    /**
     * Post-processing step that refines the agent's response.
     */
    AgentStepType["POST_PROCESSING"] = "POST_PROCESSING";
    /**
     * Step that classifies and routes requests to appropriate collaborators.
     */
    AgentStepType["ROUTING_CLASSIFIER"] = "ROUTING_CLASSIFIER";
    /**
     * Step that summarizes conversation history for memory retention.
     */
    AgentStepType["MEMORY_SUMMARIZATION"] = "MEMORY_SUMMARIZATION";
    /**
     * Step that generates responses using knowledge base content.
     */
    AgentStepType["KNOWLEDGE_BASE_RESPONSE_GENERATION"] = "KNOWLEDGE_BASE_RESPONSE_GENERATION";
})(AgentStepType || (exports.AgentStepType = AgentStepType = {}));
/**
 * Configuration for overriding prompt templates and behaviors in different parts
 * of an agent's sequence. This allows customizing how the agent processes inputs,
 * makes decisions, and generates responses.
 */
class PromptOverrideConfiguration {
    /**
     * Creates a PromptOverrideConfiguration from individual step configurations.
     * Use this method when you want to override prompts without using a custom parser.
     * @param steps The step configurations to use
     * @returns A new PromptOverrideConfiguration instance
     */
    static fromSteps(steps) {
        if (!steps || steps.length === 0) {
            throw new validation.ValidationError('Steps array cannot be empty');
        }
        // Convert steps array to props format
        const stepMap = steps.reduce((acc, step) => {
            switch (step.stepType) {
                case AgentStepType.PRE_PROCESSING:
                    return { ...acc, preProcessingStep: step };
                case AgentStepType.ORCHESTRATION:
                    return { ...acc, orchestrationStep: step };
                case AgentStepType.POST_PROCESSING:
                    return { ...acc, postProcessingStep: step };
                case AgentStepType.ROUTING_CLASSIFIER:
                    return { ...acc, routingClassifierStep: step };
                case AgentStepType.MEMORY_SUMMARIZATION:
                    return { ...acc, memorySummarizationStep: step };
                case AgentStepType.KNOWLEDGE_BASE_RESPONSE_GENERATION:
                    return { ...acc, knowledgeBaseResponseGenerationStep: step };
                default:
                    return acc;
            }
        }, {});
        return new PromptOverrideConfiguration(stepMap);
    }
    /**
     * Creates a PromptOverrideConfiguration with a custom Lambda parser function.
     * @param props Configuration including:
     *   - `parser`: Lambda function to use as custom parser
     *   - Individual step configurations. At least one of the steps must make use of the custom parser.
     */
    static withCustomParser(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_bedrock_alpha_CustomParserProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.withCustomParser);
            }
            throw error;
        }
        return new PromptOverrideConfiguration(props);
    }
    /**
     * Create a new PromptOverrideConfiguration.
     *
     * @internal - This is marked as private so end users leverage it only through static methods
     */
    constructor(props) {
        this.validateInferenceConfig = (config) => {
            const errors = [];
            if (config) {
                if (config.temperature < 0 || config.temperature > 1) {
                    errors.push('Temperature must be between 0 and 1');
                }
                if (config.topP < 0 || config.topP > 1) {
                    errors.push('TopP must be between 0 and 1');
                }
                if (config.topK < 0 || config.topK > 500) {
                    errors.push('TopK must be between 0 and 500');
                }
                if (config.stopSequences.length > 4) {
                    errors.push('Maximum 4 stop sequences allowed');
                }
                if (config.maximumLength < 0 || config.maximumLength > 4096) {
                    errors.push('MaximumLength must be between 0 and 4096');
                }
            }
            return errors;
        };
        this.validateSteps = (props) => {
            const errors = [];
            // Check if any steps are defined
            const hasSteps = [
                props.preProcessingStep,
                props.orchestrationStep,
                props.postProcessingStep,
                props.routingClassifierStep,
                props.memorySummarizationStep,
                props.knowledgeBaseResponseGenerationStep,
            ].some(step => step !== undefined);
            if (!hasSteps) {
                errors.push('Steps array cannot be empty');
            }
            // Helper function to validate a step's inference config
            const validateStep = (step, stepName) => {
                if (step) {
                    // Check for foundation model in non-ROUTING_CLASSIFIER steps
                    if ('foundationModel' in step && step.stepType !== AgentStepType.ROUTING_CLASSIFIER) {
                        errors.push('Foundation model can only be specified for ROUTING_CLASSIFIER step type');
                    }
                    const inferenceErrors = this.validateInferenceConfig(step.inferenceConfig);
                    if (inferenceErrors.length > 0) {
                        errors.push(`${stepName} step: ${inferenceErrors.join(', ')}`);
                    }
                }
            };
            // Validate each step's inference config
            validateStep(props.preProcessingStep, 'Pre-processing');
            validateStep(props.orchestrationStep, 'Orchestration');
            validateStep(props.postProcessingStep, 'Post-processing');
            validateStep(props.routingClassifierStep, 'Routing classifier');
            validateStep(props.memorySummarizationStep, 'Memory summarization');
            validateStep(props.knowledgeBaseResponseGenerationStep, 'Knowledge base response generation');
            // Validate routing classifier's foundation model if provided
            if (props.routingClassifierStep?.foundationModel) {
                if (!props.routingClassifierStep.foundationModel.invokableArn) {
                    errors.push('Foundation model must be a valid IBedrockInvokable with an invokableArn');
                }
            }
            return errors;
        };
        this.validateCustomParser = (props) => {
            const errors = [];
            // Check if at least one step uses custom parser
            const hasCustomParser = [
                props.preProcessingStep?.useCustomParser,
                props.orchestrationStep?.useCustomParser,
                props.postProcessingStep?.useCustomParser,
                props.routingClassifierStep?.useCustomParser,
                props.memorySummarizationStep?.useCustomParser,
                props.knowledgeBaseResponseGenerationStep?.useCustomParser,
            ].some(useCustomParser => useCustomParser === true);
            if (!hasCustomParser) {
                errors.push('At least one step must use custom parser');
            }
            return errors;
        };
        // Validate props
        validation.throwIfInvalid(this.validateSteps, props);
        if (props.parser) {
            validation.throwIfInvalid(this.validateCustomParser, props);
        }
        this.parser = props.parser;
        this.preProcessingStep = props.preProcessingStep;
        this.orchestrationStep = props.orchestrationStep;
        this.postProcessingStep = props.postProcessingStep;
        this.routingClassifierStep = props.routingClassifierStep;
        this.memorySummarizationStep = props.memorySummarizationStep;
        this.knowledgeBaseResponseGenerationStep = props.knowledgeBaseResponseGenerationStep;
    }
    /**
     * Format as CfnAgent.PromptOverrideConfigurationProperty
     *
     * @internal This is an internal core function and should not be called directly.
     */
    _render() {
        const configurations = [];
        // Helper function to add configuration if step exists
        const addConfiguration = (step, type) => {
            if (step) {
                configurations.push({
                    promptType: type,
                    promptState: step.stepEnabled === undefined ? undefined : step.stepEnabled ? 'ENABLED' : 'DISABLED',
                    parserMode: step.useCustomParser === undefined ? undefined : step.useCustomParser ? 'OVERRIDDEN' : 'DEFAULT',
                    promptCreationMode: step.customPromptTemplate === undefined ? undefined : step.customPromptTemplate ? 'OVERRIDDEN' : 'DEFAULT',
                    basePromptTemplate: step.customPromptTemplate,
                    inferenceConfiguration: step.inferenceConfig,
                    // Include foundation model if it's a routing classifier step
                    foundationModel: type === AgentStepType.ROUTING_CLASSIFIER
                        ? step.foundationModel?.invokableArn
                        : undefined,
                });
            }
        };
        // Add configurations for each step type if defined
        addConfiguration(this.preProcessingStep, AgentStepType.PRE_PROCESSING);
        addConfiguration(this.orchestrationStep, AgentStepType.ORCHESTRATION);
        addConfiguration(this.postProcessingStep, AgentStepType.POST_PROCESSING);
        addConfiguration(this.routingClassifierStep, AgentStepType.ROUTING_CLASSIFIER);
        addConfiguration(this.memorySummarizationStep, AgentStepType.MEMORY_SUMMARIZATION);
        addConfiguration(this.knowledgeBaseResponseGenerationStep, AgentStepType.KNOWLEDGE_BASE_RESPONSE_GENERATION);
        return {
            overrideLambda: this.parser?.functionArn,
            promptConfigurations: configurations,
        };
    }
}
exports.PromptOverrideConfiguration = PromptOverrideConfiguration;
_a = JSII_RTTI_SYMBOL_1;
PromptOverrideConfiguration[_a] = { fqn: "@aws-cdk/aws-bedrock-alpha.PromptOverrideConfiguration", version: "2.206.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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