"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BedrockFoundationModel = exports.VectorType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
/**
 * The data type for the vectors when using a model to convert text into vector embeddings.
 * The model must support the specified data type for vector embeddings. Floating-point (float32)
 * is the default data type, and is supported by most models for vector embeddings. See Supported
 * embeddings models for information on the available models and their vector data types.
 */
var VectorType;
(function (VectorType) {
    /**
     * `FLOATING_POINT` convert the data to floating-point (float32) vector embeddings (more precise, but more costly).
     */
    VectorType["FLOATING_POINT"] = "FLOAT32";
    /**
     * `BINARY` convert the data to binary vector embeddings (less precise, but less costly).
     */
    VectorType["BINARY"] = "BINARY";
})(VectorType || (exports.VectorType = VectorType = {}));
/**
 * Bedrock models.
 *
 * If you need to use a model name that doesn't exist as a static member, you
 * can instantiate a `BedrockFoundationModel` object, e.g: `new BedrockFoundationModel('my-model')`.
 */
class BedrockFoundationModel {
    /**
     * Creates a BedrockFoundationModel from a FoundationModelIdentifier.
     * Use this method when you have a model identifier from the CDK.
     * @param modelId - The foundation model identifier
     * @param props - Optional properties for the model
     * @returns A new BedrockFoundationModel instance
     */
    static fromCdkFoundationModelId(modelId, props = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_bedrock_alpha_BedrockFoundationModelProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromCdkFoundationModelId);
            }
            throw error;
        }
        return new BedrockFoundationModel(modelId.modelId, props);
    }
    /**
     * Creates a BedrockFoundationModel from a FoundationModel.
     * Use this method when you have a foundation model from the CDK.
     * @param modelId - The foundation model
     * @param props - Optional properties for the model
     * @returns A new BedrockFoundationModel instance
     */
    static fromCdkFoundationModel(modelId, props = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_bedrock_alpha_BedrockFoundationModelProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromCdkFoundationModel);
            }
            throw error;
        }
        return new BedrockFoundationModel(modelId.modelId, props);
    }
    constructor(value, props = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_bedrock_alpha_BedrockFoundationModelProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, BedrockFoundationModel);
            }
            throw error;
        }
        this.modelId = value;
        this.modelArn = aws_cdk_lib_1.Arn.format({
            partition: aws_cdk_lib_1.Aws.PARTITION,
            service: 'bedrock',
            region: aws_cdk_lib_1.Aws.REGION,
            account: '',
            resource: 'foundation-model',
            resourceName: this.modelId,
            arnFormat: aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME,
        });
        this.invokableArn = this.modelArn;
        this.supportsCrossRegion = props.supportsCrossRegion ?? false;
        this.supportsAgents = props.supportsAgents ?? false;
        this.vectorDimensions = props.vectorDimensions;
        this.supportsKnowledgeBase = props.supportsKnowledgeBase ?? false;
        this.supportedVectorType = props.supportedVectorType;
    }
    toString() {
        return this.modelId;
    }
    /**
     * Returns the ARN of the foundation model in the following format:
     * `arn:${Partition}:bedrock:${Region}::foundation-model/${ResourceId}`
     */
    asArn() {
        return this.modelArn;
    }
    /**
     * Returns the IModel
     */
    asIModel() {
        return this;
    }
    /**
     * Gives the appropriate policies to invoke and use the Foundation Model in the stack region.
     */
    grantInvoke(grantee) {
        const grant = aws_iam_1.Grant.addToPrincipal({
            grantee: grantee,
            actions: ['bedrock:InvokeModel*', 'bedrock:GetFoundationModel'],
            resourceArns: [this.invokableArn],
        });
        return grant;
    }
    /**
     * Gives the appropriate policies to invoke and use the Foundation Model in all regions.
     */
    grantInvokeAllRegions(grantee) {
        const invokableArn = aws_cdk_lib_1.Arn.format({
            partition: aws_cdk_lib_1.Aws.PARTITION,
            service: 'bedrock',
            region: '*',
            account: '',
            resource: 'foundation-model',
            resourceName: this.modelId,
            arnFormat: aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME,
        });
        return aws_iam_1.Grant.addToPrincipal({
            grantee: grantee,
            actions: ['bedrock:InvokeModel*', 'bedrock:GetFoundationModel'],
            resourceArns: [invokableArn],
        });
    }
}
exports.BedrockFoundationModel = BedrockFoundationModel;
_a = JSII_RTTI_SYMBOL_1;
BedrockFoundationModel[_a] = { fqn: "@aws-cdk/aws-bedrock-alpha.BedrockFoundationModel", version: "2.206.0-alpha.0" };
/****************************************************************************
 *                            AI21
 ***************************************************************************/
/**
 * AI21's Jamba 1.5 Large model optimized for text generation tasks.
 * Suitable for complex language understanding and generation tasks.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Optimized for natural language processing
 * - Best for: Content generation, summarization, and complex text analysis
 */
BedrockFoundationModel.AI21_JAMBA_1_5_LARGE_V1 = new BedrockFoundationModel('ai21.jamba-1-5-large-v1:0', {
    supportsAgents: true,
});
/**
 * AI21's Jamba 1.5 Mini model, a lighter version optimized for faster processing.
 * Balances performance with efficiency for general text tasks.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Faster response times compared to larger models
 * - Best for: Quick text processing, basic content generation
 */
BedrockFoundationModel.AI21_JAMBA_1_5_MINI_V1 = new BedrockFoundationModel('ai21.jamba-1-5-mini-v1:0', {
    supportsAgents: true,
});
/**
 * AI21's Jamba Instruct model, specifically designed for instruction-following tasks.
 * Optimized for understanding and executing specific instructions.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Enhanced instruction understanding
 * - Best for: Task-specific instructions, command processing
 */
BedrockFoundationModel.AI21_JAMBA_INSTRUCT_V1 = new BedrockFoundationModel('ai21.jamba-instruct-v1:0', {
    supportsAgents: true,
});
/****************************************************************************
 *                            AMAZON
 ***************************************************************************/
/**
 * Amazon's Titan Text Express model, optimized for fast text generation.
 * Provides quick responses while maintaining good quality output.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Fast response times
 * - Best for: Real-time applications, chatbots, quick content generation
 */
BedrockFoundationModel.AMAZON_TITAN_TEXT_EXPRESS_V1 = new BedrockFoundationModel('amazon.titan-text-express-v1', {
    supportsAgents: true,
});
/**
 * Amazon's Titan Text Premier model, designed for high-quality text generation.
 * Offers enhanced capabilities for complex language tasks.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Advanced language understanding
 * - Best for: Complex content generation, detailed analysis
 */
BedrockFoundationModel.AMAZON_TITAN_PREMIER_V1_0 = new BedrockFoundationModel('amazon.titan-text-premier-v1:0', {
    supportsAgents: true,
});
/**
 * Amazon's Nova Micro model, a lightweight model optimized for efficiency.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Cross-region support
 * - Optimized for agents
 * - Best for: Quick processing tasks, basic language understanding
 */
BedrockFoundationModel.AMAZON_NOVA_MICRO_V1 = new BedrockFoundationModel('amazon.nova-micro-v1:0', {
    supportsAgents: true,
    supportsCrossRegion: true,
    optimizedForAgents: true,
});
/**
 * Amazon's Nova Lite model, balancing performance with efficiency.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Cross-region support
 * - Optimized for agents
 * - Best for: General-purpose language tasks, moderate complexity
 */
BedrockFoundationModel.AMAZON_NOVA_LITE_V1 = new BedrockFoundationModel('amazon.nova-lite-v1:0', {
    supportsAgents: true,
    supportsCrossRegion: true,
    optimizedForAgents: true,
});
/**
 * Amazon's Nova Pro model, offering advanced capabilities for complex tasks.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Cross-region support
 * - Optimized for agents
 * - Best for: Complex language tasks, professional applications
 */
BedrockFoundationModel.AMAZON_NOVA_PRO_V1 = new BedrockFoundationModel('amazon.nova-pro-v1:0', {
    supportsAgents: true,
    supportsCrossRegion: true,
    optimizedForAgents: true,
});
/**
 * Amazon's Nova Premier model, the most advanced in the Nova series.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Cross-region support
 * - Optimized for agents
 * - Best for: High-end applications, complex analysis, premium performance
 */
BedrockFoundationModel.AMAZON_NOVA_PREMIER_V1 = new BedrockFoundationModel('amazon.nova-premier-v1:0', {
    supportsAgents: true,
    supportsCrossRegion: true,
    optimizedForAgents: true,
});
/**
 * Amazon's Titan Embed Text V1 model for text embeddings.
 *
 * Features:
 * - Supports Knowledge Base integration
 * - 1536-dimensional vectors
 * - Floating-point vector type
 * - Best for: Text embeddings, semantic search, document similarity
 */
BedrockFoundationModel.TITAN_EMBED_TEXT_V1 = new BedrockFoundationModel('amazon.titan-embed-text-v1', {
    supportsKnowledgeBase: true,
    vectorDimensions: 1536,
    supportedVectorType: [VectorType.FLOATING_POINT],
});
/**
 * Amazon's Titan Embed Text V2 model with 1024-dimensional vectors.
 *
 * Features:
 * - Supports Knowledge Base integration
 * - 1024-dimensional vectors
 * - Supports both floating-point and binary vectors
 * - Best for: High-dimensional embeddings, advanced semantic search
 */
BedrockFoundationModel.TITAN_EMBED_TEXT_V2_1024 = new BedrockFoundationModel('amazon.titan-embed-text-v2:0', {
    supportsKnowledgeBase: true,
    vectorDimensions: 1024,
    supportedVectorType: [VectorType.FLOATING_POINT, VectorType.BINARY],
});
/**
 * Amazon's Titan Embed Text V2 model with 512-dimensional vectors.
 *
 * Features:
 * - Supports Knowledge Base integration
 * - 512-dimensional vectors
 * - Supports both floating-point and binary vectors
 * - Best for: Balanced performance and dimensionality
 */
BedrockFoundationModel.TITAN_EMBED_TEXT_V2_512 = new BedrockFoundationModel('amazon.titan-embed-text-v2:0', {
    supportsKnowledgeBase: true,
    vectorDimensions: 512,
    supportedVectorType: [VectorType.FLOATING_POINT, VectorType.BINARY],
});
/**
 * Amazon's Titan Embed Text V2 model with 256-dimensional vectors.
 *
 * Features:
 * - Supports Knowledge Base integration
 * - 256-dimensional vectors
 * - Supports both floating-point and binary vectors
 * - Best for: Efficient embeddings with lower dimensionality
 */
BedrockFoundationModel.TITAN_EMBED_TEXT_V2_256 = new BedrockFoundationModel('amazon.titan-embed-text-v2:0', {
    supportsKnowledgeBase: true,
    vectorDimensions: 256,
    supportedVectorType: [VectorType.FLOATING_POINT, VectorType.BINARY],
});
/****************************************************************************
 *                            ANTHROPIC
 ***************************************************************************/
/**
 * Anthropic's Claude 3.7 Sonnet model, latest in the Claude 3 series.
 * Advanced language model with enhanced capabilities.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Cross-region support
 * - Best for: Complex reasoning, analysis, and content generation
 */
BedrockFoundationModel.ANTHROPIC_CLAUDE_3_7_SONNET_V1_0 = new BedrockFoundationModel('anthropic.claude-3-7-sonnet-20250219-v1:0', { supportsAgents: true, supportsCrossRegion: true, optimizedForAgents: false });
/**
 * Anthropic's Claude 3.5 Sonnet V2 model, optimized for agent interactions.
 * Enhanced version with improved performance and capabilities.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Cross-region support
 * - Optimized for agents
 * - Best for: Agent-based applications, complex dialogue
 */
BedrockFoundationModel.ANTHROPIC_CLAUDE_3_5_SONNET_V2_0 = new BedrockFoundationModel('anthropic.claude-3-5-sonnet-20241022-v2:0', { supportsAgents: true, supportsCrossRegion: true, optimizedForAgents: true });
/**
 * Anthropic's Claude 3.5 Sonnet V1 model, balanced performance model.
 * Offers good balance between performance and efficiency.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Cross-region support
 * - Optimized for agents
 * - Best for: General language tasks, balanced performance
 */
BedrockFoundationModel.ANTHROPIC_CLAUDE_3_5_SONNET_V1_0 = new BedrockFoundationModel('anthropic.claude-3-5-sonnet-20240620-v1:0', { supportsAgents: true, supportsCrossRegion: true, optimizedForAgents: true });
/**
 * Anthropic's Claude 3.5 Haiku model, optimized for quick responses.
 * Lightweight model focused on speed and efficiency.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Cross-region support
 * - Optimized for agents
 * - Best for: Fast responses, lightweight processing
 */
BedrockFoundationModel.ANTHROPIC_CLAUDE_3_5_HAIKU_V1_0 = new BedrockFoundationModel('anthropic.claude-3-5-haiku-20241022-v1:0', { supportsAgents: true, supportsCrossRegion: true, optimizedForAgents: true });
/**
 * Anthropic's Claude Opus model, designed for advanced tasks.
 * High-performance model with extensive capabilities.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Optimized for agents
 * - Best for: Complex reasoning, research, and analysis
 */
BedrockFoundationModel.ANTHROPIC_CLAUDE_OPUS_V1_0 = new BedrockFoundationModel('anthropic.claude-3-opus-20240229-v1:0', { supportsAgents: true, optimizedForAgents: true });
/**
 * Anthropic's Claude Sonnet model, legacy version.
 * Balanced model for general-purpose tasks.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Cross-region support
 * - Legacy model with EOL date
 * - Best for: General language tasks, standard applications
 */
BedrockFoundationModel.ANTHROPIC_CLAUDE_SONNET_V1_0 = new BedrockFoundationModel('anthropic.claude-3-sonnet-20240229-v1:0', { supportsAgents: true, supportsCrossRegion: true, legacy: true, optimizedForAgents: true });
/**
 * Anthropic's Claude Haiku model, optimized for efficiency.
 * Fast and efficient model for lightweight tasks.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Cross-region support
 * - Optimized for agents
 * - Best for: Quick responses, simple tasks
 */
BedrockFoundationModel.ANTHROPIC_CLAUDE_HAIKU_V1_0 = new BedrockFoundationModel('anthropic.claude-3-haiku-20240307-v1:0', { supportsAgents: true, supportsCrossRegion: true, optimizedForAgents: true });
/**
 * Anthropic's Claude V2.1 model, legacy version.
 * Improved version of Claude V2 with enhanced capabilities.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Legacy model with EOL date
 * - Optimized for agents
 * - Best for: General language tasks, legacy applications
 */
BedrockFoundationModel.ANTHROPIC_CLAUDE_V2_1 = new BedrockFoundationModel('anthropic.claude-v2:1', {
    supportsAgents: true,
    legacy: true,
    optimizedForAgents: true,
});
/**
 * Anthropic's Claude V2 model, legacy version.
 * Original Claude V2 model with broad capabilities.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Legacy model with EOL date
 * - Optimized for agents
 * - Best for: General language tasks, legacy applications
 */
BedrockFoundationModel.ANTHROPIC_CLAUDE_V2 = new BedrockFoundationModel('anthropic.claude-v2', {
    supportsAgents: true,
    legacy: true,
    optimizedForAgents: true,
});
/**
 * Anthropic's Claude Instant V1.2 model, legacy version.
 * Fast and efficient model optimized for quick responses.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Legacy model with EOL date
 * - Optimized for agents
 * - Best for: Quick responses, simple tasks, legacy applications
 */
BedrockFoundationModel.ANTHROPIC_CLAUDE_INSTANT_V1_2 = new BedrockFoundationModel('anthropic.claude-instant-v1', {
    supportsAgents: true,
    legacy: true,
    optimizedForAgents: true,
});
/****************************************************************************
 *                            COHERE
 ***************************************************************************/
/**
 * Cohere's English embedding model, optimized for English text embeddings.
 * Specialized for semantic understanding of English content.
 *
 * Features:
 * - Supports Knowledge Base integration
 * - 1024-dimensional vectors
 * - Supports both floating-point and binary vectors
 * - Best for: English text embeddings, semantic search, content similarity
 */
BedrockFoundationModel.COHERE_EMBED_ENGLISH_V3 = new BedrockFoundationModel('cohere.embed-english-v3', {
    supportsKnowledgeBase: true,
    vectorDimensions: 1024,
    supportedVectorType: [VectorType.FLOATING_POINT, VectorType.BINARY],
});
/**
 * Cohere's Multilingual embedding model, supporting multiple languages.
 * Enables semantic understanding across different languages.
 *
 * Features:
 * - Supports Knowledge Base integration
 * - 1024-dimensional vectors
 * - Supports both floating-point and binary vectors
 * - Best for: Cross-lingual embeddings, multilingual semantic search
 */
BedrockFoundationModel.COHERE_EMBED_MULTILINGUAL_V3 = new BedrockFoundationModel('cohere.embed-multilingual-v3', {
    supportsKnowledgeBase: true,
    vectorDimensions: 1024,
    supportedVectorType: [VectorType.FLOATING_POINT, VectorType.BINARY],
});
/****************************************************************************
 *                            DEEPSEEK
 ***************************************************************************/
/**
 * Deepseek's R1 model, designed for general language understanding.
 * Balanced model for various language tasks.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Cross-region support
 * - Best for: General language tasks, content generation
 */
BedrockFoundationModel.DEEPSEEK_R1_V1 = new BedrockFoundationModel('deepseek.r1-v1:0', {
    supportsAgents: true,
    supportsCrossRegion: true,
});
/****************************************************************************
 *                            META
 ***************************************************************************/
/**
 * Meta's Llama 3 1.8B Instruct model, compact instruction-following model.
 * Efficient model optimized for instruction-based tasks.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Cross-region support
 * - Best for: Lightweight instruction processing, quick responses
 */
BedrockFoundationModel.META_LLAMA_3_1_8B_INSTRUCT_V1 = new BedrockFoundationModel('meta.llama3-1-8b-instruct-v1:0', {
    supportsAgents: true,
    supportsCrossRegion: true,
});
/**
 * Meta's Llama 3 70B Instruct model, large-scale instruction model.
 * High-capacity model for complex language understanding.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Cross-region support
 * - Best for: Complex instructions, advanced language tasks
 */
BedrockFoundationModel.META_LLAMA_3_1_70B_INSTRUCT_V1 = new BedrockFoundationModel('meta.llama3-1-70b-instruct-v1:0', {
    supportsAgents: true,
    supportsCrossRegion: true,
});
/**
 * Meta's Llama 3.2 11B Instruct model, mid-sized instruction model.
 * Balanced model for general instruction processing.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Cross-region support
 * - Best for: General instruction tasks, balanced performance
 */
BedrockFoundationModel.META_LLAMA_3_2_11B_INSTRUCT_V1 = new BedrockFoundationModel('meta.llama3-2-11b-instruct-v1:0', {
    supportsAgents: true,
    supportsCrossRegion: true,
});
/**
 * Meta's Llama 3.2 3B Instruct model, compact efficient model.
 * Lightweight model for basic instruction processing.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Cross-region support
 * - Best for: Basic instructions, efficient processing
 */
BedrockFoundationModel.META_LLAMA_3_2_3B_INSTRUCT_V1 = new BedrockFoundationModel('meta.llama3-2-3b-instruct-v1:0', {
    supportsAgents: true,
    supportsCrossRegion: true,
});
/**
 * Meta's Llama 3.2 1B Instruct model, ultra-lightweight model.
 * Most compact model in the Llama 3.2 series.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Cross-region support
 * - Best for: Simple instructions, fastest response times
 */
BedrockFoundationModel.META_LLAMA_3_2_1B_INSTRUCT_V1 = new BedrockFoundationModel('meta.llama3-2-1b-instruct-v1:0', {
    supportsAgents: true,
    supportsCrossRegion: true,
});
/**
 * Meta's Llama 3.3 70B Instruct model, latest large-scale model.
 * Advanced model with enhanced capabilities.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Cross-region support
 * - Best for: Complex reasoning, advanced language tasks
 */
BedrockFoundationModel.META_LLAMA_3_3_70B_INSTRUCT_V1 = new BedrockFoundationModel('meta.llama3-3-70b-instruct-v1:0', {
    supportsAgents: true,
    supportsCrossRegion: true,
});
/**
 * Meta's Llama 4 Maverick 17B Instruct model, innovative mid-sized model.
 * Specialized for creative and dynamic responses.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Cross-region support
 * - Best for: Creative tasks, innovative solutions
 */
BedrockFoundationModel.META_LLAMA_4_MAVERICK_17B_INSTRUCT_V1 = new BedrockFoundationModel('meta.llama4-maverick-17b-instruct-v1:0', {
    supportsAgents: true,
    supportsCrossRegion: true,
});
/**
 * Meta's Llama 4 Scout 17B Instruct model, analytical mid-sized model.
 * Focused on precise and analytical responses.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Cross-region support
 * - Best for: Analytical tasks, precise responses
 */
BedrockFoundationModel.META_LLAMA_4_SCOUT_17B_INSTRUCT_V1 = new BedrockFoundationModel('meta.llama4-scout-17b-instruct-v1:0', {
    supportsAgents: true,
    supportsCrossRegion: true,
});
/****************************************************************************
 *                            MISTRAL AI
 ***************************************************************************/
/**
 * Mistral's 7B Instruct model, efficient instruction-following model.
 * Balanced performance for instruction processing.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Optimized for instruction tasks
 * - Best for: General instruction processing, balanced performance
 */
BedrockFoundationModel.MISTRAL_7B_INSTRUCT_V0 = new BedrockFoundationModel('mistral.mistral-7b-instruct-v0:2', {
    supportsAgents: true,
    optimizedForAgents: false,
    supportsCrossRegion: false,
});
/**
 * Mistral's Mixtral 8x7B Instruct model, mixture-of-experts architecture.
 * Advanced model combining multiple expert networks.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Specialized expert networks
 * - Best for: Complex tasks, diverse language understanding
 */
BedrockFoundationModel.MISTRAL_MIXTRAL_8X7B_INSTRUCT_V0 = new BedrockFoundationModel('mistral.mixtral-8x7b-instruct-v0:1', {
    supportsAgents: true,
    optimizedForAgents: false,
    supportsCrossRegion: false,
});
/**
 * Mistral's Small 2402 model, compact efficient model.
 * Optimized for quick responses and efficiency.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Efficient processing
 * - Best for: Quick responses, basic language tasks
 */
BedrockFoundationModel.MISTRAL_SMALL_2402_V1 = new BedrockFoundationModel('mistral.mistral-small-2402-v1:0', {
    supportsAgents: true,
    optimizedForAgents: false,
    supportsCrossRegion: false,
});
/**
 * Mistral's Large 2402 model, high-capacity language model.
 * Advanced model for complex language understanding.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Enhanced language capabilities
 * - Best for: Complex reasoning, detailed analysis
 */
BedrockFoundationModel.MISTRAL_LARGE_2402_V1 = new BedrockFoundationModel('mistral.mistral-large-2402-v1:0', {
    supportsAgents: true,
    optimizedForAgents: false,
    supportsCrossRegion: false,
});
/**
 * Mistral's Large 2407 model, updated large-scale model.
 * Enhanced version with improved capabilities.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Advanced language processing
 * - Best for: Sophisticated language tasks, complex analysis
 */
BedrockFoundationModel.MISTRAL_LARGE_2407_V1 = new BedrockFoundationModel('mistral.mistral-large-2407-v1:0', {
    supportsAgents: true,
    optimizedForAgents: false,
    supportsCrossRegion: false,
});
/**
 * Mistral's Pixtral Large 2502 model, specialized large model.
 * Advanced model with cross-region support.
 *
 * Features:
 * - Supports Bedrock Agents integration
 * - Cross-region support
 * - Best for: Advanced language tasks, distributed applications
 */
BedrockFoundationModel.MISTRAL_PIXTRAL_LARGE_2502_V1 = new BedrockFoundationModel('mistral.pixtral-large-2502-v1:0', {
    supportsAgents: true,
    optimizedForAgents: false,
    supportsCrossRegion: true,
});
//# sourceMappingURL=data:application/json;base64,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