import os

import typer
from rich.console import Console
from typing_extensions import Annotated, Optional

from mkrp.modules import get_gitignore
from mkrp.modules.license import MIT
from mkrp.modules.settings import SETTINGS_PYTHON
from mkrp.modules.workflows import CI_PYTHON

__version__ = '0.1.0'
app = typer.Typer(help='A CLI to generate utils files for new projects.')
console = Console()


def get_version(value: bool):
    """Returns the version of mkrp and exits the program.

    Args:
        value (bool): A flag to show the version of mkrp.
    """
    if value:
        console.print(
            f'[bold blue]mkrp[/bold blue] version: [green]{__version__}[/green]',
        )
        console.print('Developed by [bold]Henrique Sebastião[/bold]')
        raise typer.Exit()


@app.command(help='Set up a Python project.')
def python(
    license: Annotated[
        bool,
        typer.Option('--license', '-l', help='Generate a LICENSE file.'),
    ] = False,
):
    """Settings and files for a python project.

    Args:
        license(str): Generate a LICENSE file.
    """
    gitignore = get_gitignore(['python'])
    with open('.gitignore', 'w') as file:
        file.write('# Generated by mkrp\n')
        file.write('# https://github.com/henriquesebastiao/mkrp\n')
        file.write('\n')
        file.write(gitignore)
        file.write('\n')
        file.write('poetry.lock\n')
        file.write('.idea/\n')
        file.write('.vscode/\n')
    console.print('Generated [bold green].gitignore[/bold green] file.')

    if license:
        with open('LICENSE', 'w') as file:
            file.write(MIT)
        console.print('Generated [bold green]LICENSE.md[/bold green] file.')

    os.makedirs('.github/workflows', exist_ok=True)
    with open('.github/workflows/ci.yml', 'w') as file:
        file.write(CI_PYTHON)
    console.print(
        'Generated [bold green].github/workflows/ci.yml[/bold green] file.'
    )
    with open('pyproject.toml', 'a') as file:
        file.write(SETTINGS_PYTHON)
    console.print('Generated [bold green]pyproject.toml[/bold green] file.')
    console.print()
    if license:
        console.print(
            'Remember to [bold yellow]change your name in LICENSE.md[/bold yellow] file.'
        )
    console.print(
        'Remember to [bold yellow]configure your package name in pyproject.toml[/bold yellow].'
    )
    typer.Exit()


@app.callback(invoke_without_command=True)
def main(
    ctx: typer.Context,
    version: Annotated[  # noqa: F841
        Optional[bool],
        typer.Option(
            '--version',
            '-v',
            callback=get_version,
            help='Returns the version of mkrp.',
        ),
    ] = None,
):
    """A CLI to generate utils files for new projects."""
    message = """USAGE: mkrp [OPTIONS] COMMAND [OPTIONS]

There are 1 commands available:

- [bold]python[/bold]: Set up a Python project.

[bold]Examples:[/bold]
[italic yellow]skyport apod[/italic yellow] (search for the image of the day)

[bold]For more information:[/bold] [italic cyan]mkrp --help[/italic cyan]
[bold]For more detailed information:[/bold] [cyan][link=https://github.com/henriquesebastiao/mkrp]repository[/cyan]
"""
    if ctx.invoked_subcommand:
        return
    console.print(message)
