"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Lazy = void 0;
const debug_1 = require("./debug");
const stack_trace_1 = require("./stack-trace");
const token_1 = require("./token");
/**
 * Lazily produce a value.
 *
 * Can be used to return a string, list or numeric value whose actual value
 * will only be calculated later, during synthesis.
 */
class Lazy {
    /**
     *
     */
    static stringValue(producer, options = {}) {
        return token_1.Token.asString(new LazyString(producer), options);
    }
    /**
     *
     */
    static numberValue(producer) {
        return token_1.Token.asNumber(new LazyNumber(producer));
    }
    /**
     *
     */
    static listValue(producer, options = {}) {
        return token_1.Token.asList(new LazyList(producer, options), options);
    }
    /**
     *
     */
    static anyValue(producer, options = {}) {
        return new LazyAny(producer, options);
    }
    constructor() {
    }
}
exports.Lazy = Lazy;
class LazyBase {
    constructor() {
        // Stack trace capture is conditionned to `debugModeEnabled()`, because
        // lazies can be created in a fairly thrashy way, and the stack traces are
        // large and slow to obtain; but are mostly useful only when debugging a
        // resolution issue.
        this.creationStack = debug_1.debugModeEnabled()
            ? stack_trace_1.captureStackTrace(this.constructor)
            : [`Execute again with ${debug_1.CDK_DEBUG}=true to capture stack traces`];
    }
    toString() {
        return token_1.Token.asString(this);
    }
    /**
     * Turn this Token into JSON
     *
     * Called automatically when JSON.stringify() is called on a Token.
     */
    toJSON() {
        return '<unresolved-lazy>';
    }
}
class LazyString extends LazyBase {
    constructor(producer) {
        super();
        this.producer = producer;
    }
    resolve(context) {
        return this.producer.produce(context);
    }
}
class LazyNumber extends LazyBase {
    constructor(producer) {
        super();
        this.producer = producer;
    }
    resolve(context) {
        return this.producer.produce(context);
    }
}
class LazyList extends LazyBase {
    constructor(producer, options = {}) {
        super();
        this.producer = producer;
        this.options = options;
    }
    resolve(context) {
        const ret = this.producer.produce(context);
        if (ret !== undefined && ret.length === 0 && this.options.omitEmpty) {
            return undefined;
        }
        return ret;
    }
}
class LazyAny extends LazyBase {
    constructor(producer, options = {}) {
        super();
        this.producer = producer;
        this.options = options;
    }
    resolve(context) {
        const ret = this.producer.produce(context);
        if (Array.isArray(ret) && ret.length === 0 && this.options.omitEmptyArray) {
            return undefined;
        }
        return ret;
    }
}
//# sourceMappingURL=data:application/json;base64,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