"""BasicFetch - Cross-platform System Information Display Tool

A powerful, feature-rich system information tool similar to neofetch/fastfetch.
Displays comprehensive system information with beautiful ASCII logos and rich color support.

Copyright (C) 2025 Yakup Kaya (@y4kupkaya)
Website: https://yakupkaya.me
Telegram: @YakupKaya
GitHub: @y4kupkaya

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program. If not, see <https://www.gnu.org/licenses/>.

"""

import argparse
import json
import os
import platform
import re
import subprocess
import time
from typing import Any, Dict, List, Optional, Tuple

import psutil
from rich.console import Console

from .distro_logo import get_logo, data

__version__ = "1.0.0"

console = Console()


def strip_ansi_codes(text: str) -> int:
    """Remove ANSI escape sequences from text and return character count."""
    ansi_pattern = re.compile(r"\x1b\[[0-9;]*m|\033\[[0-9;]*m")
    return len(ansi_pattern.sub("", text))


parser = argparse.ArgumentParser(description="System Information Display Tool")

# Display mode arguments (mutually exclusive group)
display_group = parser.add_mutually_exclusive_group()
display_group.add_argument(
    "-s", "--simple", action="store_true", help="Show simple system information"
)
display_group.add_argument(
    "-f",
    "--full",
    action="store_true",
    help="Show full system information with color palette",
)
display_group.add_argument(
    "-j", "--json", action="store_true", help="Show system information in JSON format"
)
display_group.add_argument(
    "-p", "--palette", action="store_true", help="Show only color palette"
)
display_group.add_argument(
    "-l",
    "--logo",
    metavar="DISTRO",
    help="Show only ASCII distro logo for specified distro",
)

# Logo and display options
parser.add_argument(
    "-d",
    "--distro",
    metavar="DISTRO",
    help="Use specific distro logo instead of system logo",
)
parser.add_argument("-n", "--no-logo", action="store_true", help="Show without logo")
parser.add_argument("--no-color", action="store_true", help="Disable colors")
parser.add_argument("--no-clear", action="store_true", help="Don't clear console")

args = parser.parse_args()


class ColorSystem:
    """Handles color conversion and ANSI escape sequences."""

    @staticmethod
    def hex_to_rgb(hex_color: str) -> Tuple[int, int, int]:
        """Convert hex color to RGB tuple."""
        hex_color = hex_color.lstrip("#")
        return tuple(int(hex_color[i : i + 2], 16) for i in (0, 2, 4))

    @staticmethod
    def rgb_to_hex(rgb_color: Tuple[int, int, int]) -> str:
        """Convert RGB tuple to hex color."""
        return "#%02x%02x%02x" % rgb_color

    @staticmethod
    def convert_color_to_ansi(color: Any) -> str:
        """Convert various color formats to ANSI escape sequence."""
        if args.no_color:
            return ""

        if color is None:
            return ""

        if isinstance(color, tuple):
            red, green, blue = color
        elif isinstance(color, str):
            color = color.lower()
            if color.startswith("#"):
                red, green, blue = ColorSystem.hex_to_rgb(color)
            elif "," in color:
                red, green, blue = map(int, color.split(","))
            else:
                raise ValueError("Invalid color value")
        else:
            raise ValueError("Invalid color type")

        return f"\033[38;2;{red};{green};{blue}m"


class SystemLogo:
    """Handles system-specific logo generation."""

    @staticmethod
    def _get_system_name() -> str:
        """Get standardized system name for logo selection."""
        distros = data

        def get_os_name():
            system = platform.system().lower()

            # Quick return for Windows and macOS
            if system == "windows":
                if platform.release() == "10":
                    return "Windows 10"
                if platform.release() == "8":
                    return "Windows"
                return "Windows"

            if system == "darwin":
                return "Darwin"

            if system == "mac":
                return "Mac"

            if system == "mac_small":
                return "Mac"

            # Linux-based systems
            if system == "linux":
                os_release_file = "/etc/os-release"
                if os.path.exists(os_release_file):
                    with open(os_release_file, "r", encoding="utf-8") as f:
                        data = f.read()
                    match = re.search(r'^PRETTY_NAME="?(.*?)"?$', data, re.MULTILINE)
                    if match:
                        return match.group(1)
                    match = re.search(r'^ID="?(.*?)"?$', data, re.MULTILINE)
                    if match:
                        return match.group(1)

                # Fallback to other release files
                for path in ["/etc/issue", "/etc/redhat-release", "/etc/lsb-release"]:
                    if os.path.exists(path):
                        with open(path, "r", encoding="utf-8", errors="ignore") as f:
                            return f.readline().strip()

                return "Linux"

            # BSD derivatives
            if "bsd" in system:
                return "BSD"

            return platform.system()

        name = get_os_name()
        name_lower = name.lower()

        # Match against distros list
        for d in distros:
            if name_lower in [alias.strip().lower() for alias in d.split("|")]:
                return d.split("|")[0].strip()

        return name

    @staticmethod
    def get_system_logo(system_name: str, total_lines: int) -> List[str]:
        """Get system-specific logo with proper height adjustment."""
        logo = get_logo(system_name, nocolor=args.no_color)

        if not logo:
            return []

        if total_lines > len(logo):
            logo_lines = logo
            logo_height = len(logo_lines)
            padding_top = max(0, (total_lines - logo_height) // 2)
            padding_bottom = max(0, total_lines - logo_height - padding_top)

            return [""] * padding_top + logo_lines + [""] * padding_bottom
        else:
            return logo


class SystemInfoCollector:
    """Collects comprehensive system information across platforms."""

    def __init__(self):
        self.system_name = platform.system().lower()
        self.distros = data

    def get_basic_info(self) -> Dict[str, Any]:
        """Get basic system information."""
        return {
            "hostname": platform.node(),
            "os": f"{platform.system()} {platform.release()}",
            "architecture": platform.architecture()[0],
            "machine": platform.machine(),
            "host": platform.platform(),
            "kernel": platform.version(),
        }

    def get_cpu_info(self) -> Dict[str, Any]:
        """Get detailed CPU information."""
        try:
            cpu_info = platform.processor()
            cpu_cores = psutil.cpu_count(logical=False)
            cpu_threads = psutil.cpu_count(logical=True)
            cpu_freq = psutil.cpu_freq()
            cpu_usage = psutil.cpu_percent(interval=1)

            return {
                "cpu": cpu_info,
                "cpu_cores": cpu_cores,
                "cpu_threads": cpu_threads,
                "cpu_freq": f"{cpu_freq.current:.0f} MHz" if cpu_freq else "Unknown",
                "cpu_usage": f"{cpu_usage}%",
            }
        except Exception:
            return {
                "cpu": "Unknown",
                "cpu_cores": "Unknown",
                "cpu_threads": "Unknown",
                "cpu_freq": "Unknown",
                "cpu_usage": "Unknown",
            }

    def get_memory_info(self) -> Dict[str, Any]:
        """Get memory information."""
        try:
            memory = psutil.virtual_memory()
            swap = psutil.swap_memory()

            return {
                "memory_total": round(memory.total / (1024**3), 1),
                "memory_used": round(memory.used / (1024**3), 1),
                "memory_available": round(memory.available / (1024**3), 1),
                "memory_percent": memory.percent,
                "swap_total": round(swap.total / (1024**3), 1) if swap.total > 0 else 0,
                "swap_used": round(swap.used / (1024**3), 1) if swap.used > 0 else 0,
            }
        except Exception:
            return {
                "memory_total": 0,
                "memory_used": 0,
                "memory_available": 0,
                "memory_percent": 0,
                "swap_total": 0,
                "swap_used": 0,
            }

    def get_disk_info(self) -> Dict[str, Any]:
        """Get disk information."""
        try:
            all_disks = psutil.disk_partitions()
            disk_info = {}

            for disk in all_disks:
                try:
                    usage = psutil.disk_usage(disk.mountpoint)
                    disk_info[disk.device] = {
                        "mountpoint": disk.mountpoint,
                        "fstype": disk.fstype,
                        "total": round(usage.total / (1024**3), 1),
                        "used": round(usage.used / (1024**3), 1),
                        "free": round(usage.free / (1024**3), 1),
                        "percent": round(usage.percent, 1),
                    }
                except PermissionError:
                    continue

            return {"all_disks": disk_info}
        except Exception:
            return {"all_disks": {}}

    def get_network_info(self) -> Dict[str, Any]:
        """Get network information."""
        try:
            net_io = psutil.net_io_counters()
            return {
                "network": {
                    "bytes_sent": round(net_io.bytes_sent / (1024**3), 2),
                    "bytes_recv": round(net_io.bytes_recv / (1024**3), 2),
                }
            }
        except Exception:
            return {"network": {"bytes_sent": 0, "bytes_recv": 0}}

    def get_platform_specific_info(self) -> Dict[str, Any]:
        """Get platform-specific information."""
        info = {
            "motherboard": "Unknown",
            "gpu_list": [],
            "shell": "Unknown",
            "packages": "0 packages",
            "resolution": "Unknown",
            "terminal": "Unknown",
        }

        if self.system_name == "windows":
            info.update(self._get_windows_info())
        elif self.system_name == "linux":
            info.update(self._get_linux_info())
        elif self.system_name == "darwin":
            info.update(self._get_macos_info())

        return info

    def _get_windows_info(self) -> Dict[str, Any]:
        """Get Windows-specific information."""
        info = {"shell": "PowerShell (Unknown version)"}

        try:
            import wmi

            c = wmi.WMI()

            for board in c.Win32_BaseBoard():
                info["motherboard"] = f"{board.Manufacturer} {board.Product}"
                break

            gpu_list = []
            for gpu in c.Win32_VideoController():
                if gpu.Name:
                    gpu_list.append(gpu.Name)
            info["gpu_list"] = gpu_list

        except ImportError:
            pass

        try:
            ps_version = subprocess.run(
                ["powershell", "-Command", "$PSVersionTable.PSVersion.Major"],
                capture_output=True,
                text=True,
                timeout=5,
            )
            if ps_version.returncode == 0:
                info["shell"] = f"PowerShell v{ps_version.stdout.strip()}"
        except Exception:
            pass

        try:
            pip_list = subprocess.run(
                ["pip", "list", "--format=freeze"],
                capture_output=True,
                text=True,
                timeout=10,
            )
            if pip_list.returncode == 0:
                package_count = len(
                    [line for line in pip_list.stdout.strip().split("\n") if line]
                )
                info["packages"] = f"{package_count} packages"
        except Exception:
            pass

        try:
            import tkinter as tk

            root = tk.Tk()
            info["resolution"] = (
                f"{root.winfo_screenwidth()}x{root.winfo_screenheight()}"
            )
            root.destroy()
        except Exception:
            pass

        return info

    def _get_linux_info(self) -> Dict[str, Any]:
        """Get Linux-specific information."""
        info = {"shell": os.environ.get("SHELL", "Unknown")}

        try:
            with open("/etc/os-release", "r") as f:
                os_release = f.read()
                for line in os_release.split("\n"):
                    if line.startswith("PRETTY_NAME="):
                        info["os"] = line.split("=")[1].strip('"')
                        break
        except Exception:
            pass

        try:
            lspci_output = subprocess.run(
                ["lspci", "-nn"], capture_output=True, text=True, timeout=5
            )
            gpu_list = []
            for line in lspci_output.stdout.split("\n"):
                if "VGA" in line or "Display" in line:
                    gpu_list.append(line.split(": ")[-1])
            info["gpu_list"] = gpu_list
        except Exception:
            pass

        package_managers = [
            (["dpkg", "-l"], "dpkg"),
            (["rpm", "-qa"], "rpm"),
            (["pacman", "-Q"], "pacman"),
            (["pip", "list", "--format=freeze"], "pip"),
        ]

        package_count = 0
        for cmd, manager in package_managers:
            try:
                result = subprocess.run(cmd, capture_output=True, text=True, timeout=10)
                if result.returncode == 0:
                    count = len(
                        [line for line in result.stdout.strip().split("\n") if line]
                    )
                    package_count += count
            except Exception:
                continue

        info["packages"] = f"{package_count} packages"

        try:
            xrandr_output = subprocess.run(
                ["xrandr"], capture_output=True, text=True, timeout=5
            )
            for line in xrandr_output.stdout.split("\n"):
                if "*" in line:
                    info["resolution"] = line.split()[0]
                    break
        except Exception:
            pass

        return info

    def _get_macos_info(self) -> Dict[str, Any]:
        """Get macOS-specific information."""
        info = {"shell": os.environ.get("SHELL", "Unknown")}

        try:
            system_profiler = subprocess.run(
                ["system_profiler", "SPDisplaysDataType"],
                capture_output=True,
                text=True,
                timeout=10,
            )
            gpu_list = []
            lines = system_profiler.stdout.split("\n")
            for line in lines:
                if "Chipset Model:" in line:
                    gpu_list.append(line.split(":")[-1].strip())
            info["gpu_list"] = gpu_list
        except Exception:
            pass

        try:
            brew_list = subprocess.run(
                ["brew", "list"], capture_output=True, text=True, timeout=10
            )
            if brew_list.returncode == 0:
                package_count = len(brew_list.stdout.strip().split("\n"))
                info["packages"] = f"{package_count} packages"
        except Exception:
            try:
                pip_list = subprocess.run(
                    ["pip", "list", "--format=freeze"],
                    capture_output=True,
                    text=True,
                    timeout=10,
                )
                if pip_list.returncode == 0:
                    package_count = len(
                        [line for line in pip_list.stdout.strip().split("\n") if line]
                    )
                    info["packages"] = f"{package_count} packages"
            except Exception:
                pass

        try:
            system_profiler = subprocess.run(
                ["system_profiler", "SPDisplaysDataType"],
                capture_output=True,
                text=True,
                timeout=5,
            )
            for line in system_profiler.stdout.split("\n"):
                if "Resolution:" in line:
                    info["resolution"] = line.split(":")[-1].strip()
                    break
        except Exception:
            pass

        return info

    def get_uptime_info(self) -> Dict[str, Any]:
        """Get system uptime information."""
        try:
            uptime_seconds = time.time() - psutil.boot_time()
            uptime_hours = int(uptime_seconds // 3600)
            uptime_minutes = int((uptime_seconds % 3600) // 60)
            return {"uptime": f"{uptime_hours}h {uptime_minutes}m"}
        except Exception:
            return {"uptime": "Unknown"}

    def get_gpu_temp(self) -> Dict[str, Any]:
        """Get GPU temperature if available."""
        try:
            nvidia_smi = subprocess.run(
                [
                    "nvidia-smi",
                    "--query-gpu=temperature.gpu",
                    "--format=csv,noheader,nounits",
                ],
                capture_output=True,
                text=True,
                timeout=5,
            )
            if nvidia_smi.returncode == 0:
                return {"gpu_temp": f"{nvidia_smi.stdout.strip()}°C"}
        except Exception:
            pass
        return {"gpu_temp": "N/A"}

    def get_terminal_info(self) -> Dict[str, Any]:
        """Get terminal information."""
        terminal = os.environ.get(
            "TERM_PROGRAM",
            os.environ.get("TERMINAL_EMULATOR", os.environ.get("COLORTERM", "Unknown")),
        )
        return {"terminal": terminal}

    def collect_all_info(self) -> Dict[str, Any]:
        """Collect all system information."""
        try:
            info = {}
            info.update(self.get_basic_info())
            info.update(self.get_cpu_info())
            info.update(self.get_memory_info())
            info.update(self.get_disk_info())
            info.update(self.get_network_info())
            info.update(self.get_platform_specific_info())
            info.update(self.get_uptime_info())
            info.update(self.get_gpu_temp())
            info.update(self.get_terminal_info())
            info["system_name"] = self.system_name

            return info
        except Exception as e:
            console.print(f"[red]Error collecting system info: {e}[/red]")
            return {}


class SystemInfoPrinter:
    """Handles printing system information in various formats."""

    def __init__(self):
        self.collector = SystemInfoCollector()
        self.colors = (
            {
                "blue": "\033[94m",
                "yellow": "\033[93m",
                "green": "\033[92m",
                "red": "\033[91m",
                "cyan": "\033[96m",
                "magenta": "\033[95m",
                "bold": "\033[1m",
                "reset": "\033[0m",
            }
            if not args.no_color
            else {
                "blue": "",
                "yellow": "",
                "green": "",
                "red": "",
                "cyan": "",
                "magenta": "",
                "bold": "",
                "reset": "",
            }
        )

    def calculate_total_lines(self, info: Dict[str, Any]) -> int:
        """Calculate total lines needed for logo alignment."""
        if args.full:
            base_lines = 18 + 4 + 7

            if info.get("all_disks"):
                base_lines += min(len(info["all_disks"]), 3)

            if info.get("network") and (
                info["network"].get("bytes_sent", 0) > 0
                or info["network"].get("bytes_recv", 0) > 0
            ):
                base_lines += 4

            if info.get("gpu_list"):
                base_lines += min(len(info["gpu_list"]), 2)

            return base_lines

        if args.simple:
            return 11

        return 0

    def format_system_info(
        self,
        info: Dict[str, Any],
        mode: str = "detailed",
    ) -> List[str]:
        """Format system information for display.

        Args:
            info: System information dictionary
            mode: Display mode ('detailed', 'simple', 'json', 'none')

        Returns:
            List of formatted strings for display
        """
        if not info:
            return [
                f"{self.colors['red']}System information could not be retrieved!{self.colors['reset']}"
            ]

        username = info.get("hostname", "Unknown")
        if mode == "detailed":
            info_lines = [
                f"{self.colors['bold']}{self.colors['yellow']}System Info @{username}{self.colors['reset']}",
                f"{self.colors['blue']}{'─' * (len(username) * 2 + 1)}{self.colors['reset']}",
                f"{self.colors['bold']}{self.colors['magenta']}● System Info{self.colors['reset']}",
                f"{self.colors['cyan']}OS{self.colors['reset']}: {info.get('os', 'Unknown')} {info.get('architecture', '')}",
                f"{self.colors['cyan']}Host{self.colors['reset']}: {info.get('machine', 'Unknown')}",
                f"{self.colors['cyan']}Kernel{self.colors['reset']}: {info.get('kernel', 'Unknown')[:50]}{'...' if len(info.get('kernel', '')) > 50 else ''}",
                f"{self.colors['cyan']}Motherboard{self.colors['reset']}: {info.get('motherboard', 'Unknown')}",
                f"{self.colors['cyan']}Uptime{self.colors['reset']}: {info.get('uptime', 'Unknown')}",
                f"{self.colors['cyan']}Packages{self.colors['reset']}: {info.get('packages', 'Unknown')}",
                f"{self.colors['cyan']}Shell{self.colors['reset']}: {info.get('shell', 'Unknown')}",
                f"{self.colors['cyan']}Resolution{self.colors['reset']}: {info.get('resolution', 'Unknown')}",
                f"{self.colors['cyan']}Terminal{self.colors['reset']}: {info.get('terminal', 'Unknown')}",
                "",
                f"{self.colors['bold']}{self.colors['magenta']}● Hardware Info{self.colors['reset']}",
            ]

            cpu_name = info.get("cpu", "Unknown")
            if len(cpu_name) > 45:
                cpu_name = cpu_name[:42] + "..."

            info_lines.extend(
                [
                    f"{self.colors['cyan']}CPU{self.colors['reset']}: {cpu_name}",
                    f"{self.colors['cyan']}CPU Cores{self.colors['reset']}: {info.get('cpu_cores', 'Unknown')} physical, {info.get('cpu_threads', 'Unknown')} logical",
                    f"{self.colors['cyan']}CPU Frequency{self.colors['reset']}: {info.get('cpu_freq', 'Unknown')}",
                    f"{self.colors['cyan']}CPU Usage{self.colors['reset']}: {info.get('cpu_usage', 'Unknown')}",
                ]
            )

            gpu_list = info.get("gpu_list", ["Unknown"])
            for i, gpu in enumerate(gpu_list[:2]):
                gpu_name = gpu if len(gpu) <= 45 else gpu[:42] + "..."
                if i == 0:
                    info_lines.append(
                        f"{self.colors['cyan']}GPU{self.colors['reset']}: {gpu_name}"
                    )
                else:
                    info_lines.append(
                        f"{self.colors['cyan']}GPU{i+1}{self.colors['reset']}: {gpu_name}"
                    )

            if info.get("gpu_temp", "N/A") != "N/A":
                info_lines.append(
                    f"{self.colors['cyan']}GPU Temp{self.colors['reset']}: {info.get('gpu_temp', 'N/A')}"
                )

            info_lines.extend(
                [
                    "",
                    f"{self.colors['bold']}{self.colors['magenta']}● Memory Info{self.colors['reset']}",
                    f"{self.colors['cyan']}RAM{self.colors['reset']}: {info.get('memory_used', 0)} GB / {info.get('memory_total', 0)} GB ({info.get('memory_percent', 0):.1f}%)",
                    f"{self.colors['cyan']}Available{self.colors['reset']}: {info.get('memory_available', 0)} GB",
                ]
            )

            if info.get("swap_total", 0) > 0:
                info_lines.append(
                    f"{self.colors['cyan']}Swap{self.colors['reset']}: {info.get('swap_used', 0)} GB / {info.get('swap_total', 0)} GB"
                )

            info_lines.append("")
            info_lines.append(
                f"{self.colors['bold']}{self.colors['magenta']}● Storage Info{self.colors['reset']}"
            )

            disk_count = 0
            for disk, details in info.get("all_disks", {}).items():
                if disk_count >= 3:
                    break
                disk_name = disk.replace("\\", "") if "\\" in disk else disk
                if len(disk_name) > 10:
                    disk_name = disk_name[:8] + ".."
                info_lines.append(
                    f"{self.colors['cyan']}Disk ({disk_name}){self.colors['reset']}: {details['used']} GB / {details['total']} GB "
                    f"({details['percent']:.1f}%) [{details['fstype']}]"
                )
                disk_count += 1

            network = info.get("network", {})
            if network and (
                network.get("bytes_sent", 0) > 0 or network.get("bytes_recv", 0) > 0
            ):
                info_lines.extend(
                    [
                        "",
                        f"{self.colors['bold']}{self.colors['magenta']}● Network Info{self.colors['reset']}",
                        f"{self.colors['cyan']}Sent{self.colors['reset']}: {network.get('bytes_sent', 0)} GB",
                        f"{self.colors['cyan']}Received{self.colors['reset']}: {network.get('bytes_recv', 0)} GB",
                    ]
                )

        if mode == "json":
            return [json.dumps(info, indent=4, ensure_ascii=False)]

        if mode == "none":
            return []

        if mode == "simple":
            return [
                f"{self.colors['bold']}{self.colors['yellow']}System Info @{username}{self.colors['reset']}",
                f"{self.colors['blue']}{'─' * (len(username) * 2 + 1)}{self.colors['reset']}",
                f"{self.colors['bold']}{self.colors['magenta']}● System Info{self.colors['reset']}",
                f"{self.colors['cyan']}OS{self.colors['reset']}: {info.get('os', 'Unknown')} {info.get('architecture', '')}",
                f"{self.colors['cyan']}Host{self.colors['reset']}: {info.get('machine', 'Unknown')}",
                f"{self.colors['cyan']}Kernel{self.colors['reset']}: {info.get('kernel', 'Unknown')[:50]}{'...' if len(info.get('kernel', '')) > 50 else ''}",
                f"{self.colors['cyan']}Uptime{self.colors['reset']}: {info.get('uptime', 'Unknown')}",
                f"{self.colors['cyan']}CPU{self.colors['reset']}: {info.get('cpu', 'Unknown')}",
                f"{self.colors['cyan']}RAM{self.colors['reset']}: {info.get('memory_used', 0)} GB / {info.get('memory_total', 0)} GB ({info.get('memory_percent', 0):.1f}%)",
            ]

        return info_lines

    def print_color_palette(self):
        """Print color palette in horizontal layout."""

        colors = [
            ("Red", (255, 0, 0)),
            ("Green", (0, 255, 0)),
            ("Blue", (0, 0, 255)),
            ("Yellow", (255, 255, 0)),
            ("Cyan", (0, 255, 255)),
            ("Magenta", (255, 0, 255)),
            ("White", (255, 255, 255)),
            ("Black", (0, 0, 0)),
            ("Orange", (255, 165, 0)),
            ("Purple", (128, 0, 128)),
            ("Pink", (255, 192, 203)),
            ("Brown", (165, 42, 42)),
            ("Gray", (128, 128, 128)),
        ]

        # First row - bright colors
        bright_row = ""
        for name, rgb in colors:
            ansi_color = ColorSystem.convert_color_to_ansi(rgb)
            bright_row += f"{ansi_color}███{self.colors['reset']} "

        # Second row - dark colors
        dark_row = ""
        for name, rgb in colors:
            dark_rgb = tuple(max(0, c - 100) for c in rgb)
            ansi_dark = ColorSystem.convert_color_to_ansi(dark_rgb)
            dark_row += f"{ansi_dark}███{self.colors['reset']} "

        print(bright_row)
        print(dark_row)

    def print_system_info(
        self,
        mode: str = "detailed",
        print_logo: bool = True,
        color_palette: bool = False,
        spesific_logo: Optional[str] = None,
    ):
        """Print system information with optional logo and color palette.

        Args:
            mode: Display mode ('detailed', 'simple', 'json', 'none')
            print_logo: Whether to display system logo
            color_palette: Whether to display color palette
            spesific_logo: Specific logo name to use instead of system logo
        """
        info = self.collector.collect_all_info()

        if mode == "json":
            console.print_json(str(json.dumps(info, indent=4, ensure_ascii=False)))
            return

        if not info:
            console.print(
                f"{self.colors['red']}System information could not be retrieved!{self.colors['reset']}"
            )
            return

        total_lines = self.calculate_total_lines(info)
        logo = (
            SystemLogo.get_system_logo(info.get("os", "Ubuntu"), total_lines)
            if spesific_logo is None
            else SystemLogo.get_system_logo(spesific_logo, total_lines)
        )
        info_lines = self.format_system_info(info, mode)

        if print_logo:
            logo = [line.rstrip() for line in logo]
            max_line_logo = max(strip_ansi_codes(line) for line in logo) if logo else 0

            if total_lines > len(logo):
                for line in logo:
                    display_length = strip_ansi_codes(line)
                    if display_length < max_line_logo:
                        line += " " * (max_line_logo - display_length)
                    print(line, end="")
                    if info_lines:
                        print(" " + info_lines.pop(0))
                    else:
                        pass
            else:
                for i in range(len(logo)):
                    line = logo[i]
                    display_length = strip_ansi_codes(line)
                    if display_length < max_line_logo:
                        line += " " * (max_line_logo - display_length)
                    print(line, end="")
                    if i < len(info_lines):
                        print(" " + info_lines[i])
                    else:
                        print()
        else:
            for line in info_lines:
                print(line)

        if color_palette:
            self.print_color_palette()


def clear_console():
    """Clear the console screen."""
    os.system("cls" if os.name == "nt" else "clear")


def main():
    """Main function to handle command line arguments and execute."""
    # Clear console unless disabled
    if not args.no_clear:
        clear_console()

    printer = SystemInfoPrinter()

    # Determine display mode and options
    mode = "detailed"  # default
    show_logo = True
    show_color_palette = False
    specific_logo = args.distro

    # Handle mutually exclusive display modes
    if args.simple:
        mode = "simple"
    elif args.full:
        mode = "detailed"
        show_color_palette = False
    elif args.json:
        mode = "json"
        show_logo = False
    elif args.palette:
        mode = "none"
        show_logo = False
        show_color_palette = True
    elif args.logo:
        # Show only logo
        logo = get_logo(args.logo, nocolor=args.no_color)
        if logo:
            for line in logo:
                print(line)
        else:
            print(f"No logo found for distro: {args.logo}")
        return

    # Apply logo override options
    if args.no_logo:
        show_logo = False

    # Print system information
    printer.print_system_info(
        mode=mode,
        print_logo=show_logo,
        color_palette=show_color_palette,
        spesific_logo=specific_logo,
    )

    print()


if __name__ == "__main__":
    main()
