"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KafkaCluster = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [kafka-cluster](https://docs.aws.amazon.com/service-authorization/latest/reference/list_apachekafkaapisforamazonmskclusters.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class KafkaCluster extends shared_1.PolicyStatement {
    /**
     * Grants permission to alter various aspects of the cluster, equivalent to Apache Kafka's ALTER CLUSTER ACL
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kafka-cluster:Connect
     * - kafka-cluster:DescribeCluster
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/iam-access-control.html#actions
     */
    toAlterCluster() {
        return this.to('AlterCluster');
    }
    /**
     * Grants permission to alter the dynamic configuration of a cluster, equivalent to Apache Kafka's ALTER_CONFIGS CLUSTER ACL
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kafka-cluster:Connect
     * - kafka-cluster:DescribeClusterDynamicConfiguration
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/iam-access-control.html#actions
     */
    toAlterClusterDynamicConfiguration() {
        return this.to('AlterClusterDynamicConfiguration');
    }
    /**
     * Grants permission to join groups on a cluster, equivalent to Apache Kafka's READ GROUP ACL
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kafka-cluster:Connect
     * - kafka-cluster:DescribeGroup
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/iam-access-control.html#actions
     */
    toAlterGroup() {
        return this.to('AlterGroup');
    }
    /**
     * Grants permission to alter topics on a cluster, equivalent to Apache Kafka's ALTER TOPIC ACL
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kafka-cluster:Connect
     * - kafka-cluster:DescribeTopic
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/iam-access-control.html#actions
     */
    toAlterTopic() {
        return this.to('AlterTopic');
    }
    /**
     * Grants permission to alter the dynamic configuration of topics on a cluster, equivalent to Apache Kafka's ALTER_CONFIGS TOPIC ACL
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kafka-cluster:Connect
     * - kafka-cluster:DescribeTopicDynamicConfiguration
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/iam-access-control.html#actions
     */
    toAlterTopicDynamicConfiguration() {
        return this.to('AlterTopicDynamicConfiguration');
    }
    /**
     * Grants permission to alter transactional IDs on a cluster, equivalent to Apache Kafka's WRITE TRANSACTIONAL_ID ACL
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kafka-cluster:Connect
     * - kafka-cluster:DescribeTransactionalId
     * - kafka-cluster:WriteData
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/iam-access-control.html#actions
     */
    toAlterTransactionalId() {
        return this.to('AlterTransactionalId');
    }
    /**
     * Grants permission to connect and authenticate to the cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/iam-access-control.html#actions
     */
    toConnect() {
        return this.to('Connect');
    }
    /**
     * Grants permission to create topics on a cluster, equivalent to Apache Kafka's CREATE CLUSTER/TOPIC ACL
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kafka-cluster:Connect
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/iam-access-control.html#actions
     */
    toCreateTopic() {
        return this.to('CreateTopic');
    }
    /**
     * Grants permission to delete groups on a cluster, equivalent to Apache Kafka's DELETE GROUP ACL
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kafka-cluster:Connect
     * - kafka-cluster:DescribeGroup
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/iam-access-control.html#actions
     */
    toDeleteGroup() {
        return this.to('DeleteGroup');
    }
    /**
     * Grants permission to delete topics on a cluster, equivalent to Apache Kafka's DELETE TOPIC ACL
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kafka-cluster:Connect
     * - kafka-cluster:DescribeTopic
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/iam-access-control.html#actions
     */
    toDeleteTopic() {
        return this.to('DeleteTopic');
    }
    /**
     * Grants permission to describe various aspects of the cluster, equivalent to Apache Kafka's DESCRIBE CLUSTER ACL
     *
     * Access Level: List
     *
     * Dependent actions:
     * - kafka-cluster:Connect
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/iam-access-control.html#actions
     */
    toDescribeCluster() {
        return this.to('DescribeCluster');
    }
    /**
     * Grants permission to describe the dynamic configuration of a cluster, equivalent to Apache Kafka's DESCRIBE_CONFIGS CLUSTER ACL
     *
     * Access Level: List
     *
     * Dependent actions:
     * - kafka-cluster:Connect
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/iam-access-control.html#actions
     */
    toDescribeClusterDynamicConfiguration() {
        return this.to('DescribeClusterDynamicConfiguration');
    }
    /**
     * Grants permission to describe groups on a cluster, equivalent to Apache Kafka's DESCRIBE GROUP ACL
     *
     * Access Level: List
     *
     * Dependent actions:
     * - kafka-cluster:Connect
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/iam-access-control.html#actions
     */
    toDescribeGroup() {
        return this.to('DescribeGroup');
    }
    /**
     * Grants permission to describe topics on a cluster, equivalent to Apache Kafka's DESCRIBE TOPIC ACL
     *
     * Access Level: List
     *
     * Dependent actions:
     * - kafka-cluster:Connect
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/iam-access-control.html#actions
     */
    toDescribeTopic() {
        return this.to('DescribeTopic');
    }
    /**
     * Grants permission to describe the dynamic configuration of topics on a cluster, equivalent to Apache Kafka's DESCRIBE_CONFIGS TOPIC ACL
     *
     * Access Level: List
     *
     * Dependent actions:
     * - kafka-cluster:Connect
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/iam-access-control.html#actions
     */
    toDescribeTopicDynamicConfiguration() {
        return this.to('DescribeTopicDynamicConfiguration');
    }
    /**
     * Grants permission to describe transactional IDs on a cluster, equivalent to Apache Kafka's DESCRIBE TRANSACTIONAL_ID ACL
     *
     * Access Level: List
     *
     * Dependent actions:
     * - kafka-cluster:Connect
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/iam-access-control.html#actions
     */
    toDescribeTransactionalId() {
        return this.to('DescribeTransactionalId');
    }
    /**
     * Grants permission to read data from topics on a cluster, equivalent to Apache Kafka's READ TOPIC ACL
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kafka-cluster:AlterGroup
     * - kafka-cluster:Connect
     * - kafka-cluster:DescribeTopic
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/iam-access-control.html#actions
     */
    toReadData() {
        return this.to('ReadData');
    }
    /**
     * Grants permission to write data to topics on a cluster, equivalent to Apache Kafka's WRITE TOPIC ACL
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kafka-cluster:Connect
     * - kafka-cluster:DescribeTopic
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/iam-access-control.html#actions
     */
    toWriteData() {
        return this.to('WriteData');
    }
    /**
     * Grants permission to write data idempotently on a cluster, equivalent to Apache Kafka's IDEMPOTENT_WRITE CLUSTER ACL
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kafka-cluster:Connect
     * - kafka-cluster:WriteData
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/iam-access-control.html#actions
     */
    toWriteDataIdempotently() {
        return this.to('WriteDataIdempotently');
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/iam-access-control.html#msk-iam-resources
     *
     * @param clusterName - Identifier for the clusterName.
     * @param clusterUuid - Identifier for the clusterUuid.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCluster(clusterName, clusterUuid, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kafka:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:cluster/${clusterName}/${clusterUuid}`);
    }
    /**
     * Adds a resource of type topic to the statement
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/iam-access-control.html#msk-iam-resources
     *
     * @param clusterName - Identifier for the clusterName.
     * @param clusterUuid - Identifier for the clusterUuid.
     * @param topicName - Identifier for the topicName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onTopic(clusterName, clusterUuid, topicName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kafka:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:topic/${clusterName}/${clusterUuid}/${topicName}`);
    }
    /**
     * Adds a resource of type group to the statement
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/iam-access-control.html#msk-iam-resources
     *
     * @param clusterName - Identifier for the clusterName.
     * @param clusterUuid - Identifier for the clusterUuid.
     * @param groupName - Identifier for the groupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onGroup(clusterName, clusterUuid, groupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kafka:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:group/${clusterName}/${clusterUuid}/${groupName}`);
    }
    /**
     * Adds a resource of type transactional-id to the statement
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/iam-access-control.html#msk-iam-resources
     *
     * @param clusterName - Identifier for the clusterName.
     * @param clusterUuid - Identifier for the clusterUuid.
     * @param transactionalId - Identifier for the transactionalId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onTransactionalId(clusterName, clusterUuid, transactionalId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kafka:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:transactional-id/${clusterName}/${clusterUuid}/${transactionalId}`);
    }
    /**
     * Filters actions based on tag key-value pairs attached to the resource. The resource tag context key will only apply to the cluster resource, not topics, groups and transactional IDs
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - cluster
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [kafka-cluster](https://docs.aws.amazon.com/service-authorization/latest/reference/list_apachekafkaapisforamazonmskclusters.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'kafka-cluster';
        this.accessLevelList = {
            Write: [
                'AlterCluster',
                'AlterClusterDynamicConfiguration',
                'AlterGroup',
                'AlterTopic',
                'AlterTopicDynamicConfiguration',
                'AlterTransactionalId',
                'Connect',
                'CreateTopic',
                'DeleteGroup',
                'DeleteTopic',
                'WriteData',
                'WriteDataIdempotently'
            ],
            List: [
                'DescribeCluster',
                'DescribeClusterDynamicConfiguration',
                'DescribeGroup',
                'DescribeTopic',
                'DescribeTopicDynamicConfiguration',
                'DescribeTransactionalId'
            ],
            Read: [
                'ReadData'
            ]
        };
    }
}
exports.KafkaCluster = KafkaCluster;
//# sourceMappingURL=data:application/json;base64,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