"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Apigateway = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [apigateway](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonapigatewaymanagement.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Apigateway extends shared_1.PolicyStatement {
    /**
     * Grants permission to add certificates for mutual TLS authentication to a domain name. This is an additional authorization control for managing the DomainName resource due to the sensitive nature of mTLS
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Operations.html
     */
    toAddCertificateToDomain() {
        return this.to('AddCertificateToDomain');
    }
    /**
     * Grants permission to create an access association from an access association source to a custom domain name for private APIs
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Operations.html
     */
    toCreateAccessAssociation() {
        return this.to('CreateAccessAssociation');
    }
    /**
     * Grants permission to delete a particular resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Operations.html
     */
    toDELETE() {
        return this.to('DELETE');
    }
    /**
     * Grants permission to read a particular resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Operations.html
     */
    toGET() {
        return this.to('GET');
    }
    /**
     * Grants permission to update a particular resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Operations.html
     */
    toPATCH() {
        return this.to('PATCH');
    }
    /**
     * Grants permission to create a particular resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Operations.html
     */
    toPOST() {
        return this.to('POST');
    }
    /**
     * Grants permission to update a particular resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Operations.html
     */
    toPUT() {
        return this.to('PUT');
    }
    /**
     * Grants permission to reject an existing access association owned by another account to a custom domain name for private APIs
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Operations.html
     */
    toRejectAccessAssociation() {
        return this.to('RejectAccessAssociation');
    }
    /**
     * Grants permission to remove certificates for mutual TLS authentication from a domain name. This is an additional authorization control for managing the DomainName resource due to the sensitive nature of mTLS
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Operations.html
     */
    toRemoveCertificateFromDomain() {
        return this.to('RemoveCertificateFromDomain');
    }
    /**
     * Grants permission to set a WAF access control list (ACL). This is an additional authorization control for managing the Stage resource due to the sensitive nature of WebAcl's
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Operations.html
     */
    toSetWebACL() {
        return this.to('SetWebACL');
    }
    /**
     * Grants permission to update the management policy of a custom domain name for private APIs
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Operations.html
     */
    toUpdateDomainNameManagementPolicy() {
        return this.to('UpdateDomainNameManagementPolicy');
    }
    /**
     * Grants permission to update the invoke policy of a custom domain name for private APIs
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Operations.html
     */
    toUpdateDomainNamePolicy() {
        return this.to('UpdateDomainNamePolicy');
    }
    /**
     * Grants permission to manage the IAM resource policy for an API. This is an additional authorization control for managing an API due to the sensitive nature of the resource policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Operations.html
     */
    toUpdateRestApiPolicy() {
        return this.to('UpdateRestApiPolicy');
    }
    /**
     * Adds a resource of type Account to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAccount(region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/account`);
    }
    /**
     * Adds a resource of type ApiKey to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_ApiKey.html
     *
     * @param apiKeyId - Identifier for the apiKeyId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApiKey(apiKeyId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/apikeys/${apiKeyId}`);
    }
    /**
     * Adds a resource of type ApiKeys to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_ApiKey.html
     *
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApiKeys(region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/apikeys`);
    }
    /**
     * Adds a resource of type Authorizer to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Authorizer.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param authorizerId - Identifier for the authorizerId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifRequestAuthorizerType()
     * - .ifRequestAuthorizerUri()
     * - .ifRequestCognitoUserPoolProviderArn()
     * - .ifResourceAuthorizerType()
     * - .ifResourceAuthorizerUri()
     * - .ifResourceCognitoUserPoolProviderArn()
     * - .ifAwsResourceTag()
     */
    onAuthorizer(restApiId, authorizerId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/authorizers/${authorizerId}`);
    }
    /**
     * Adds a resource of type Authorizers to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Authorizer.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifRequestAuthorizerType()
     * - .ifRequestAuthorizerUri()
     * - .ifRequestCognitoUserPoolProviderArn()
     * - .ifAwsResourceTag()
     */
    onAuthorizers(restApiId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/authorizers`);
    }
    /**
     * Adds a resource of type BasePathMapping to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_BasePathMapping.html
     *
     * @param domainName - Identifier for the domainName.
     * @param basePath - Identifier for the basePath.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBasePathMapping(domainName, basePath, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/domainnames/${domainName}/basepathmappings/${basePath}`);
    }
    /**
     * Adds a resource of type BasePathMappings to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_BasePathMapping.html
     *
     * @param domainName - Identifier for the domainName.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBasePathMappings(domainName, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/domainnames/${domainName}/basepathmappings`);
    }
    /**
     * Adds a resource of type ClientCertificate to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_ClientCertificate.html
     *
     * @param clientCertificateId - Identifier for the clientCertificateId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onClientCertificate(clientCertificateId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/clientcertificates/${clientCertificateId}`);
    }
    /**
     * Adds a resource of type ClientCertificates to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_ClientCertificate.html
     *
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onClientCertificates(region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/clientcertificates`);
    }
    /**
     * Adds a resource of type Deployment to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Deployment.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param deploymentId - Identifier for the deploymentId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeployment(restApiId, deploymentId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/deployments/${deploymentId}`);
    }
    /**
     * Adds a resource of type Deployments to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Deployment.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifRequestStageName()
     * - .ifAwsResourceTag()
     */
    onDeployments(restApiId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/deployments`);
    }
    /**
     * Adds a resource of type DocumentationPart to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_DocumentationPart.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param documentationPartId - Identifier for the documentationPartId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDocumentationPart(restApiId, documentationPartId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/documentation/parts/${documentationPartId}`);
    }
    /**
     * Adds a resource of type DocumentationParts to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_DocumentationPart.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDocumentationParts(restApiId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/documentation/parts`);
    }
    /**
     * Adds a resource of type DocumentationVersion to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_DocumentationVersion.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param documentationVersionId - Identifier for the documentationVersionId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDocumentationVersion(restApiId, documentationVersionId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/documentation/versions/${documentationVersionId}`);
    }
    /**
     * Adds a resource of type DocumentationVersions to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_DocumentationVersion.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDocumentationVersions(restApiId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/documentation/versions`);
    }
    /**
     * Adds a resource of type DomainName to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_DomainName.html
     *
     * @param domainName - Identifier for the domainName.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifRequestEndpointType()
     * - .ifRequestMtlsTrustStoreUri()
     * - .ifRequestMtlsTrustStoreVersion()
     * - .ifRequestSecurityPolicy()
     * - .ifResourceEndpointType()
     * - .ifResourceMtlsTrustStoreUri()
     * - .ifResourceMtlsTrustStoreVersion()
     * - .ifResourceRoutingMode()
     * - .ifResourceSecurityPolicy()
     * - .ifAwsResourceTag()
     */
    onDomainName(domainName, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/domainnames/${domainName}`);
    }
    /**
     * Adds a resource of type DomainNames to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_DomainName.html
     *
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifRequestEndpointType()
     * - .ifRequestMtlsTrustStoreUri()
     * - .ifRequestMtlsTrustStoreVersion()
     * - .ifRequestSecurityPolicy()
     * - .ifResourceRoutingMode()
     * - .ifAwsResourceTag()
     */
    onDomainNames(region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/domainnames`);
    }
    /**
     * Adds a resource of type DomainNameAccessAssociation to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_DomainNameAccessAssociation.html
     *
     * @param domainName - Identifier for the domainName.
     * @param sourceType - Identifier for the sourceType.
     * @param sourceId - Identifier for the sourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onDomainNameAccessAssociation(domainName, sourceType, sourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/domainnameaccessassociations/domainname/${domainName}/${sourceType}/${sourceId}`);
    }
    /**
     * Adds a resource of type DomainNameAccessAssociations to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_DomainNameAccessAssociation.html
     *
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifRequestAccessAssociationSource()
     * - .ifRequestDomainNameArn()
     * - .ifAwsResourceTag()
     */
    onDomainNameAccessAssociations(account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/domainnameaccessassociations`);
    }
    /**
     * Adds a resource of type GatewayResponse to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_GatewayResponse.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param responseType - Identifier for the responseType.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGatewayResponse(restApiId, responseType, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/gatewayresponses/${responseType}`);
    }
    /**
     * Adds a resource of type GatewayResponses to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_GatewayResponse.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGatewayResponses(restApiId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/gatewayresponses`);
    }
    /**
     * Adds a resource of type Integration to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Integration.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param resourceId - Identifier for the resourceId.
     * @param httpMethodType - Identifier for the httpMethodType.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIntegration(restApiId, resourceId, httpMethodType, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/resources/${resourceId}/methods/${httpMethodType}/integration`);
    }
    /**
     * Adds a resource of type IntegrationResponse to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_IntegrationResponse.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param resourceId - Identifier for the resourceId.
     * @param httpMethodType - Identifier for the httpMethodType.
     * @param statusCode - Identifier for the statusCode.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIntegrationResponse(restApiId, resourceId, httpMethodType, statusCode, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/resources/${resourceId}/methods/${httpMethodType}/integration/responses/${statusCode}`);
    }
    /**
     * Adds a resource of type Method to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Method.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param resourceId - Identifier for the resourceId.
     * @param httpMethodType - Identifier for the httpMethodType.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifRequestApiKeyRequired()
     * - .ifRequestRouteAuthorizationType()
     * - .ifResourceApiKeyRequired()
     * - .ifResourceRouteAuthorizationType()
     * - .ifAwsResourceTag()
     */
    onMethod(restApiId, resourceId, httpMethodType, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/resources/${resourceId}/methods/${httpMethodType}`);
    }
    /**
     * Adds a resource of type MethodResponse to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_MethodResponse.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param resourceId - Identifier for the resourceId.
     * @param httpMethodType - Identifier for the httpMethodType.
     * @param statusCode - Identifier for the statusCode.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMethodResponse(restApiId, resourceId, httpMethodType, statusCode, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/resources/${resourceId}/methods/${httpMethodType}/responses/${statusCode}`);
    }
    /**
     * Adds a resource of type Model to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Model.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param modelName - Identifier for the modelName.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onModel(restApiId, modelName, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/models/${modelName}`);
    }
    /**
     * Adds a resource of type Models to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Model.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onModels(restApiId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/models`);
    }
    /**
     * Adds a resource of type PrivateBasePathMapping to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_BasePathMapping.html
     *
     * @param domainName - Identifier for the domainName.
     * @param domainIdentifier - Identifier for the domainIdentifier.
     * @param basePath - Identifier for the basePath.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPrivateBasePathMapping(domainName, domainIdentifier, basePath, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/domainnames/${domainName}+${domainIdentifier}/basepathmappings/${basePath}`);
    }
    /**
     * Adds a resource of type PrivateBasePathMappings to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_BasePathMapping.html
     *
     * @param domainName - Identifier for the domainName.
     * @param domainIdentifier - Identifier for the domainIdentifier.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPrivateBasePathMappings(domainName, domainIdentifier, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/domainnames/${domainName}+${domainIdentifier}/basepathmappings`);
    }
    /**
     * Adds a resource of type PrivateDomainName to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_DomainName.html
     *
     * @param domainName - Identifier for the domainName.
     * @param domainIdentifier - Identifier for the domainIdentifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifRequestEndpointType()
     * - .ifResourceEndpointType()
     * - .ifResourceRoutingMode()
     * - .ifAwsResourceTag()
     */
    onPrivateDomainName(domainName, domainIdentifier, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/domainnames/${domainName}+${domainIdentifier}`);
    }
    /**
     * Adds a resource of type RequestValidator to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_RequestValidator.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param requestValidatorId - Identifier for the requestValidatorId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRequestValidator(restApiId, requestValidatorId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/requestvalidators/${requestValidatorId}`);
    }
    /**
     * Adds a resource of type RequestValidators to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_RequestValidator.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRequestValidators(restApiId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/requestvalidators`);
    }
    /**
     * Adds a resource of type Resource to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Resource.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param resourceId - Identifier for the resourceId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onResource(restApiId, resourceId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/resources/${resourceId}`);
    }
    /**
     * Adds a resource of type Resources to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Resource.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onResources(restApiId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/resources`);
    }
    /**
     * Adds a resource of type RestApi to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_RestApi.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifRequestApiKeyRequired()
     * - .ifRequestApiName()
     * - .ifRequestAuthorizerType()
     * - .ifRequestAuthorizerUri()
     * - .ifRequestDisableExecuteApiEndpoint()
     * - .ifRequestEndpointType()
     * - .ifRequestRouteAuthorizationType()
     * - .ifResourceApiKeyRequired()
     * - .ifResourceApiName()
     * - .ifResourceAuthorizerType()
     * - .ifResourceAuthorizerUri()
     * - .ifResourceDisableExecuteApiEndpoint()
     * - .ifResourceEndpointType()
     * - .ifResourceRouteAuthorizationType()
     * - .ifAwsResourceTag()
     */
    onRestApi(restApiId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}`);
    }
    /**
     * Adds a resource of type RestApis to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_RestApi.html
     *
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifRequestApiKeyRequired()
     * - .ifRequestApiName()
     * - .ifRequestAuthorizerType()
     * - .ifRequestAuthorizerUri()
     * - .ifRequestDisableExecuteApiEndpoint()
     * - .ifRequestEndpointType()
     * - .ifRequestRouteAuthorizationType()
     * - .ifAwsResourceTag()
     */
    onRestApis(region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis`);
    }
    /**
     * Adds a resource of type Sdk to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param stageName - Identifier for the stageName.
     * @param sdkType - Identifier for the sdkType.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSdk(restApiId, stageName, sdkType, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/stages/${stageName}/sdks/${sdkType}`);
    }
    /**
     * Adds a resource of type Stage to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Stage.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param stageName - Identifier for the stageName.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifRequestAccessLoggingDestination()
     * - .ifRequestAccessLoggingFormat()
     * - .ifResourceAccessLoggingDestination()
     * - .ifResourceAccessLoggingFormat()
     * - .ifAwsResourceTag()
     */
    onStage(restApiId, stageName, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/stages/${stageName}`);
    }
    /**
     * Adds a resource of type Stages to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Stage.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifRequestAccessLoggingDestination()
     * - .ifRequestAccessLoggingFormat()
     * - .ifAwsResourceTag()
     */
    onStages(restApiId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/stages`);
    }
    /**
     * Adds a resource of type Template to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * @param modelName - Identifier for the modelName.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTemplate(modelName, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/models/${modelName}/template`);
    }
    /**
     * Adds a resource of type UsagePlan to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_UsagePlan.html
     *
     * @param usagePlanId - Identifier for the usagePlanId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onUsagePlan(usagePlanId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/usageplans/${usagePlanId}`);
    }
    /**
     * Adds a resource of type UsagePlans to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_UsagePlan.html
     *
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onUsagePlans(region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/usageplans`);
    }
    /**
     * Adds a resource of type UsagePlanKey to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_UsagePlanKey.html
     *
     * @param usagePlanId - Identifier for the usagePlanId.
     * @param id - Identifier for the id.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onUsagePlanKey(usagePlanId, id, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/usageplans/${usagePlanId}/keys/${id}`);
    }
    /**
     * Adds a resource of type UsagePlanKeys to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_UsagePlanKey.html
     *
     * @param usagePlanId - Identifier for the usagePlanId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onUsagePlanKeys(usagePlanId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/usageplans/${usagePlanId}/keys`);
    }
    /**
     * Adds a resource of type VpcLink to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_VpcLink.html
     *
     * @param vpcLinkId - Identifier for the vpcLinkId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVpcLink(vpcLinkId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/vpclinks/${vpcLinkId}`);
    }
    /**
     * Adds a resource of type VpcLinks to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_VpcLink.html
     *
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVpcLinks(region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/vpclinks`);
    }
    /**
     * Adds a resource of type Tags to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-tagging.html
     *
     * @param urlEncodedResourceARN - Identifier for the urlEncodedResourceARN.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onTags(urlEncodedResourceARN, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/tags/${urlEncodedResourceARN}`);
    }
    /**
     * Filters access by access association source. Available during the CreateDomainNameAccessAssociation operation
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - DomainNameAccessAssociations
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestAccessAssociationSource(value, operator) {
        return this.if(`Request/AccessAssociationSource`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by access log destination. Available during the CreateStage and UpdateStage operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Stage
     * - Stages
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestAccessLoggingDestination(value, operator) {
        return this.if(`Request/AccessLoggingDestination`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by access log format. Available during the CreateStage and UpdateStage operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Stage
     * - Stages
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestAccessLoggingFormat(value, operator) {
        return this.if(`Request/AccessLoggingFormat`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by whether an API key is required or not. Available during the CreateMethod and PutMethod operations. Also available as a collection during import and reimport
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Method
     * - RestApi
     * - RestApis
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifRequestApiKeyRequired(value) {
        return this.if(`Request/ApiKeyRequired`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by API name. Available during the CreateRestApi and UpdateRestApi operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - RestApi
     * - RestApis
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestApiName(value, operator) {
        return this.if(`Request/ApiName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by type of authorizer in the request, for example TOKEN, REQUEST, JWT. Available during CreateAuthorizer and UpdateAuthorizer. Also available during import and reimport as an ArrayOfString
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Authorizer
     * - Authorizers
     * - RestApi
     * - RestApis
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestAuthorizerType(value, operator) {
        return this.if(`Request/AuthorizerType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by URI of a Lambda authorizer function. Available during CreateAuthorizer and UpdateAuthorizer. Also available during import and reimport as an ArrayOfString
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Authorizer
     * - Authorizers
     * - RestApi
     * - RestApis
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestAuthorizerUri(value, operator) {
        return this.if(`Request/AuthorizerUri`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by Cognito user pool provider ARN. Available during CreateAuthorizer and UpdateAuthorizer
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Authorizer
     * - Authorizers
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifRequestCognitoUserPoolProviderArn(value, operator) {
        return this.if(`Request/CognitoUserPoolProviderArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by status of the default execute-api endpoint. Available during the CreateRestApi and DeleteRestApi operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - RestApi
     * - RestApis
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifRequestDisableExecuteApiEndpoint(value) {
        return this.if(`Request/DisableExecuteApiEndpoint`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by domain name ARN. Available during the CreateDomainNameAccessAssociation operation
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - DomainNameAccessAssociations
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifRequestDomainNameArn(value, operator) {
        return this.if(`Request/DomainNameArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by endpoint type. Available during the CreateDomainName, UpdateDomainName, CreateRestApi, and UpdateRestApi operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - DomainName
     * - DomainNames
     * - PrivateDomainName
     * - RestApi
     * - RestApis
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestEndpointType(value, operator) {
        return this.if(`Request/EndpointType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by URI of the truststore used for mutual TLS authentication. Available during the CreateDomainName and UpdateDomainName operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - DomainName
     * - DomainNames
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestMtlsTrustStoreUri(value, operator) {
        return this.if(`Request/MtlsTrustStoreUri`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by version of the truststore used for mutual TLS authentication. Available during the CreateDomainName and UpdateDomainName operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - DomainName
     * - DomainNames
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestMtlsTrustStoreVersion(value, operator) {
        return this.if(`Request/MtlsTrustStoreVersion`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by authorization type, for example NONE, AWS_IAM, CUSTOM, JWT, COGNITO_USER_POOLS. Available during the CreateMethod and PutMethod operations Also available as a collection during import
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Method
     * - RestApi
     * - RestApis
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestRouteAuthorizationType(value, operator) {
        return this.if(`Request/RouteAuthorizationType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by routing mode of the domain name. Available during the CreateDomainName and UpdateDomainName operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_id-based-policy-examples.html#security_iam_id-based-policy-examples-routing-mode
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestRoutingMode(value, operator) {
        return this.if(`Request/RoutingMode`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by TLS version. Available during the CreateDomain and UpdateDomain operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - DomainName
     * - DomainNames
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestSecurityPolicy(value, operator) {
        return this.if(`Request/SecurityPolicy`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by stage name of the deployment that you attempt to create. Available during the CreateDeployment operation
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Deployments
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestStageName(value, operator) {
        return this.if(`Request/StageName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by access log destination of the current Stage resource. Available during the UpdateStage and DeleteStage operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Stage
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceAccessLoggingDestination(value, operator) {
        return this.if(`Resource/AccessLoggingDestination`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by access log format of the current Stage resource. Available during the UpdateStage and DeleteStage operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Stage
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceAccessLoggingFormat(value, operator) {
        return this.if(`Resource/AccessLoggingFormat`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by whether an API key is required or not for the existing Method resource. Available during the PutMethod and DeleteMethod operations. Also available as a collection during reimport
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Method
     * - RestApi
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifResourceApiKeyRequired(value) {
        return this.if(`Resource/ApiKeyRequired`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by API name of the existing RestApi resource. Available during UpdateRestApi and DeleteRestApi operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - RestApi
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceApiName(value, operator) {
        return this.if(`Resource/ApiName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the current type of authorizer, for example TOKEN, REQUEST, JWT. Available during UpdateAuthorizer and DeleteAuthorizer operations. Also available during reimport as an ArrayOfString
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Authorizer
     * - RestApi
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceAuthorizerType(value, operator) {
        return this.if(`Resource/AuthorizerType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by URI of a Lambda authorizer function. Available during UpdateAuthorizer and DeleteAuthorizer operations. Also available during reimport as an ArrayOfString
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Authorizer
     * - RestApi
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceAuthorizerUri(value, operator) {
        return this.if(`Resource/AuthorizerUri`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by Cognito user pool provider ARN. Available during CreateAuthorizer and UpdateAuthorizer
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Authorizer
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifResourceCognitoUserPoolProviderArn(value, operator) {
        return this.if(`Resource/CognitoUserPoolProviderArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by status of the default execute-api endpoint of the current RestApi resource. Available during UpdateRestApi and DeleteRestApi operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - RestApi
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifResourceDisableExecuteApiEndpoint(value) {
        return this.if(`Resource/DisableExecuteApiEndpoint`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by endpoint type. Available during the UpdateDomainName, DeleteDomainName, UpdateRestApi, and DeleteRestApi operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - DomainName
     * - PrivateDomainName
     * - RestApi
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceEndpointType(value, operator) {
        return this.if(`Resource/EndpointType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by URI of the truststore used for mutual TLS authentication. Available during UpdateDomainName and DeleteDomainName operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - DomainName
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceMtlsTrustStoreUri(value, operator) {
        return this.if(`Resource/MtlsTrustStoreUri`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by version of the truststore used for mutual TLS authentication. Available during UpdateDomainName and DeleteDomainName operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - DomainName
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceMtlsTrustStoreVersion(value, operator) {
        return this.if(`Resource/MtlsTrustStoreVersion`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by authorization type of the existing Method resource, for example NONE, AWS_IAM, CUSTOM, JWT, COGNITO_USER_POOLS. Available during the PutMethod and DeleteMethod operations. Also available as a collection during reimport
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Method
     * - RestApi
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceRouteAuthorizationType(value, operator) {
        return this.if(`Resource/RouteAuthorizationType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by routing mode of the domain name. Available during the UpdateDomainName and DeleteDomainName operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_id-based-policy-examples.html#security_iam_id-based-policy-examples-routing-mode
     *
     * Applies to resource types:
     * - DomainName
     * - DomainNames
     * - PrivateDomainName
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceRoutingMode(value, operator) {
        return this.if(`Resource/RoutingMode`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by TLS version. Available during UpdateDomain and DeleteDomain operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - DomainName
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceSecurityPolicy(value, operator) {
        return this.if(`Resource/SecurityPolicy`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-tagging.html
     *
     * Applies to actions:
     * - .toDELETE()
     * - .toPATCH()
     * - .toPOST()
     * - .toPUT()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags attached to the resource
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-tagging.html
     *
     * Applies to resource types:
     * - ApiKey
     * - ApiKeys
     * - Authorizer
     * - Authorizers
     * - BasePathMapping
     * - BasePathMappings
     * - ClientCertificate
     * - ClientCertificates
     * - Deployment
     * - Deployments
     * - DocumentationPart
     * - DocumentationParts
     * - DocumentationVersion
     * - DocumentationVersions
     * - DomainName
     * - DomainNames
     * - DomainNameAccessAssociations
     * - GatewayResponse
     * - GatewayResponses
     * - Integration
     * - IntegrationResponse
     * - Method
     * - MethodResponse
     * - Model
     * - Models
     * - PrivateBasePathMapping
     * - PrivateBasePathMappings
     * - PrivateDomainName
     * - RequestValidator
     * - RequestValidators
     * - Resource
     * - Resources
     * - RestApi
     * - RestApis
     * - Sdk
     * - Stage
     * - Stages
     * - Template
     * - UsagePlan
     * - UsagePlans
     * - UsagePlanKey
     * - UsagePlanKeys
     * - VpcLink
     * - VpcLinks
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys in the request
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-tagging.html
     *
     * Applies to actions:
     * - .toDELETE()
     * - .toPATCH()
     * - .toPOST()
     * - .toPUT()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [apigateway](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonapigatewaymanagement.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'apigateway';
        this.accessLevelList = {
            'Permissions management': [
                'AddCertificateToDomain',
                'CreateAccessAssociation',
                'RejectAccessAssociation',
                'RemoveCertificateFromDomain',
                'SetWebACL',
                'UpdateDomainNameManagementPolicy',
                'UpdateDomainNamePolicy',
                'UpdateRestApiPolicy'
            ],
            Write: [
                'DELETE',
                'PATCH',
                'POST',
                'PUT'
            ],
            Read: [
                'GET'
            ]
        };
    }
}
exports.Apigateway = Apigateway;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBpZ2F0ZXdheW1hbmFnZW1lbnQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhcGlnYXRld2F5bWFuYWdlbWVudC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFDQSx5Q0FBeUQ7QUFHekQ7Ozs7R0FJRztBQUNILE1BQWEsVUFBVyxTQUFRLHdCQUFlO0lBRzdDOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCO1FBQzlCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksUUFBUTtRQUNiLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxRQUFRLENBQUMsQ0FBQztJQUMzQixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksS0FBSztRQUNWLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUN4QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLE9BQU87UUFDWixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBTyxDQUFDLENBQUM7SUFDMUIsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxNQUFNO1FBQ1gsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxDQUFDO0lBQ3pCLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksS0FBSztRQUNWLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUN4QixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCO1FBQzlCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw2QkFBNkI7UUFDbEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7SUFDaEQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFdBQVcsQ0FBQyxDQUFDO0lBQzlCLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQ0FBa0M7UUFDdkMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtDQUFrQyxDQUFDLENBQUM7SUFDckQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUF3QkQ7Ozs7Ozs7T0FPRztJQUNJLFNBQVMsQ0FBQyxNQUFlLEVBQUUsU0FBa0I7UUFDbEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLFlBQVksQ0FBQyxDQUFDO0lBQ3ZILENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLFFBQVEsQ0FBQyxRQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNuRSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsY0FBZSxRQUFTLEVBQUUsQ0FBQyxDQUFDO0lBQ3JJLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksU0FBUyxDQUFDLE1BQWUsRUFBRSxTQUFrQjtRQUNsRCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsWUFBWSxDQUFDLENBQUM7SUFDdkgsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FrQkc7SUFDSSxZQUFZLENBQUMsU0FBaUIsRUFBRSxZQUFvQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUM5RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsZUFBZ0IsU0FBVSxnQkFBaUIsWUFBYSxFQUFFLENBQUMsQ0FBQztJQUNySyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSxhQUFhLENBQUMsU0FBaUIsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDekUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGVBQWdCLFNBQVUsY0FBYyxDQUFDLENBQUM7SUFDbkosQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGlCQUFpQixDQUFDLFVBQWtCLEVBQUUsUUFBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDaEcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGtCQUFtQixVQUFXLHFCQUFzQixRQUFTLEVBQUUsQ0FBQyxDQUFDO0lBQzFLLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLGtCQUFrQixDQUFDLFVBQWtCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQy9FLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxrQkFBbUIsVUFBVyxtQkFBbUIsQ0FBQyxDQUFDO0lBQzVKLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLG1CQUFtQixDQUFDLG1CQUEyQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN6RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMseUJBQTBCLG1CQUFvQixFQUFFLENBQUMsQ0FBQztJQUMzSixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLG9CQUFvQixDQUFDLE1BQWUsRUFBRSxTQUFrQjtRQUM3RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsdUJBQXVCLENBQUMsQ0FBQztJQUNsSSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksWUFBWSxDQUFDLFNBQWlCLEVBQUUsWUFBb0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDOUYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGVBQWdCLFNBQVUsZ0JBQWlCLFlBQWEsRUFBRSxDQUFDLENBQUM7SUFDckssQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGFBQWEsQ0FBQyxTQUFpQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN6RSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsZUFBZ0IsU0FBVSxjQUFjLENBQUMsQ0FBQztJQUNuSixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksbUJBQW1CLENBQUMsU0FBaUIsRUFBRSxtQkFBMkIsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDNUcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGVBQWdCLFNBQVUsd0JBQXlCLG1CQUFvQixFQUFFLENBQUMsQ0FBQztJQUNwTCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxvQkFBb0IsQ0FBQyxTQUFpQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNoRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsZUFBZ0IsU0FBVSxzQkFBc0IsQ0FBQyxDQUFDO0lBQzNKLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxzQkFBc0IsQ0FBQyxTQUFpQixFQUFFLHNCQUE4QixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNsSCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsZUFBZ0IsU0FBVSwyQkFBNEIsc0JBQXVCLEVBQUUsQ0FBQyxDQUFDO0lBQzFMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLHVCQUF1QixDQUFDLFNBQWlCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ25GLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxlQUFnQixTQUFVLHlCQUF5QixDQUFDLENBQUM7SUFDOUosQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQW9CRztJQUNJLFlBQVksQ0FBQyxVQUFrQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN6RSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsa0JBQW1CLFVBQVcsRUFBRSxDQUFDLENBQUM7SUFDM0ksQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7T0FlRztJQUNJLGFBQWEsQ0FBQyxNQUFlLEVBQUUsU0FBa0I7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGdCQUFnQixDQUFDLENBQUM7SUFDM0gsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksNkJBQTZCLENBQUMsVUFBa0IsRUFBRSxVQUFrQixFQUFFLFFBQWdCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDbEosT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLDZDQUE4QyxVQUFXLElBQUssVUFBVyxJQUFLLFFBQVMsRUFBRSxDQUFDLENBQUM7SUFDeE8sQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSw4QkFBOEIsQ0FBQyxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN6RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsZ0NBQWdDLENBQUMsQ0FBQztJQUMvSyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksaUJBQWlCLENBQUMsU0FBaUIsRUFBRSxZQUFvQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNuRyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsZUFBZ0IsU0FBVSxxQkFBc0IsWUFBYSxFQUFFLENBQUMsQ0FBQztJQUMxSyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxrQkFBa0IsQ0FBQyxTQUFpQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUM5RSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsZUFBZ0IsU0FBVSxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3hKLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksYUFBYSxDQUFDLFNBQWlCLEVBQUUsVUFBa0IsRUFBRSxjQUFzQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNySCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsZUFBZ0IsU0FBVSxjQUFlLFVBQVcsWUFBYSxjQUFlLGNBQWMsQ0FBQyxDQUFDO0lBQ3pNLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7T0FjRztJQUNJLHFCQUFxQixDQUFDLFNBQWlCLEVBQUUsVUFBa0IsRUFBRSxjQUFzQixFQUFFLFVBQWtCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ2pKLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxlQUFnQixTQUFVLGNBQWUsVUFBVyxZQUFhLGNBQWUsMEJBQTJCLFVBQVcsRUFBRSxDQUFDLENBQUM7SUFDbk8sQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7OztPQWlCRztJQUNJLFFBQVEsQ0FBQyxTQUFpQixFQUFFLFVBQWtCLEVBQUUsY0FBc0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDaEgsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGVBQWdCLFNBQVUsY0FBZSxVQUFXLFlBQWEsY0FBZSxFQUFFLENBQUMsQ0FBQztJQUM3TCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSxnQkFBZ0IsQ0FBQyxTQUFpQixFQUFFLFVBQWtCLEVBQUUsY0FBc0IsRUFBRSxVQUFrQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUM1SSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsZUFBZ0IsU0FBVSxjQUFlLFVBQVcsWUFBYSxjQUFlLGNBQWUsVUFBVyxFQUFFLENBQUMsQ0FBQztJQUN2TixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksT0FBTyxDQUFDLFNBQWlCLEVBQUUsU0FBaUIsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDdEYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGVBQWdCLFNBQVUsV0FBWSxTQUFVLEVBQUUsQ0FBQyxDQUFDO0lBQzdKLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLFFBQVEsQ0FBQyxTQUFpQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNwRSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsZUFBZ0IsU0FBVSxTQUFTLENBQUMsQ0FBQztJQUM5SSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLHdCQUF3QixDQUFDLFVBQWtCLEVBQUUsZ0JBQXdCLEVBQUUsUUFBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDakksT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGtCQUFtQixVQUFXLElBQUssZ0JBQWlCLHFCQUFzQixRQUFTLEVBQUUsQ0FBQyxDQUFDO0lBQ2hNLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSx5QkFBeUIsQ0FBQyxVQUFrQixFQUFFLGdCQUF3QixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNoSCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsa0JBQW1CLFVBQVcsSUFBSyxnQkFBaUIsbUJBQW1CLENBQUMsQ0FBQztJQUNsTCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7T0FnQkc7SUFDSSxtQkFBbUIsQ0FBQyxVQUFrQixFQUFFLGdCQUF3QixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzVILE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxpQkFBa0IsVUFBVyxJQUFLLGdCQUFpQixFQUFFLENBQUMsQ0FBQztJQUNwTSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksa0JBQWtCLENBQUMsU0FBaUIsRUFBRSxrQkFBMEIsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDMUcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGVBQWdCLFNBQVUsc0JBQXVCLGtCQUFtQixFQUFFLENBQUMsQ0FBQztJQUNqTCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxtQkFBbUIsQ0FBQyxTQUFpQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUMvRSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsZUFBZ0IsU0FBVSxvQkFBb0IsQ0FBQyxDQUFDO0lBQ3pKLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxVQUFVLENBQUMsU0FBaUIsRUFBRSxVQUFrQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUMxRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsZUFBZ0IsU0FBVSxjQUFlLFVBQVcsRUFBRSxDQUFDLENBQUM7SUFDakssQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksV0FBVyxDQUFDLFNBQWlCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3ZFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxlQUFnQixTQUFVLFlBQVksQ0FBQyxDQUFDO0lBQ2pKLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQXlCRztJQUNJLFNBQVMsQ0FBQyxTQUFpQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNyRSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsZUFBZ0IsU0FBVSxFQUFFLENBQUMsQ0FBQztJQUN2SSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7O09BaUJHO0lBQ0ksVUFBVSxDQUFDLE1BQWUsRUFBRSxTQUFrQjtRQUNuRCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsYUFBYSxDQUFDLENBQUM7SUFDeEgsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxLQUFLLENBQUMsU0FBaUIsRUFBRSxTQUFpQixFQUFFLE9BQWUsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDckcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGVBQWdCLFNBQVUsV0FBWSxTQUFVLFNBQVUsT0FBUSxFQUFFLENBQUMsQ0FBQztJQUMvSyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7T0FnQkc7SUFDSSxPQUFPLENBQUMsU0FBaUIsRUFBRSxTQUFpQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN0RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsZUFBZ0IsU0FBVSxXQUFZLFNBQVUsRUFBRSxDQUFDLENBQUM7SUFDN0osQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxRQUFRLENBQUMsU0FBaUIsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDcEUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGVBQWdCLFNBQVUsU0FBUyxDQUFDLENBQUM7SUFDOUksQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksVUFBVSxDQUFDLFNBQWlCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3RFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxzQkFBdUIsU0FBVSxXQUFXLENBQUMsQ0FBQztJQUN2SixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxXQUFXLENBQUMsV0FBbUIsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDekUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGlCQUFrQixXQUFZLEVBQUUsQ0FBQyxDQUFDO0lBQzNJLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksWUFBWSxDQUFDLE1BQWUsRUFBRSxTQUFrQjtRQUNyRCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsZUFBZSxDQUFDLENBQUM7SUFDMUgsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGNBQWMsQ0FBQyxXQUFtQixFQUFFLEVBQVUsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDeEYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGlCQUFrQixXQUFZLFNBQVUsRUFBRyxFQUFFLENBQUMsQ0FBQztJQUN4SixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxlQUFlLENBQUMsV0FBbUIsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDN0UsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGlCQUFrQixXQUFZLE9BQU8sQ0FBQyxDQUFDO0lBQ2hKLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLFNBQVMsQ0FBQyxTQUFpQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNyRSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsZUFBZ0IsU0FBVSxFQUFFLENBQUMsQ0FBQztJQUN2SSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLFVBQVUsQ0FBQyxNQUFlLEVBQUUsU0FBa0I7UUFDbkQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGFBQWEsQ0FBQyxDQUFDO0lBQ3hILENBQUM7SUFFRDs7Ozs7Ozs7T0FRRztJQUNJLE1BQU0sQ0FBQyxxQkFBNkIsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDOUUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLFdBQVkscUJBQXNCLEVBQUUsQ0FBQyxDQUFDO0lBQy9JLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksZ0NBQWdDLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUM1RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUNBQWlDLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNyRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxpQ0FBaUMsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQzdGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQ0FBa0MsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ3RGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLDRCQUE0QixDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDeEYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDakYsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksdUJBQXVCLENBQUMsS0FBZTtRQUM1QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLEVBQUUsQ0FBQyxPQUFPLEtBQUssS0FBSyxXQUFXLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEVBQUUsTUFBTSxDQUFDLENBQUM7SUFDbEcsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksZ0JBQWdCLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUM1RSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNyRSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLHVCQUF1QixDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDbkYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDNUUsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxzQkFBc0IsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ2xGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQzNFLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLG1DQUFtQyxDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDL0YsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksU0FBUyxDQUFDLENBQUM7SUFDckYsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxrQ0FBa0MsQ0FBQyxLQUFlO1FBQ3ZELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQ0FBbUMsRUFBRSxDQUFDLE9BQU8sS0FBSyxLQUFLLFdBQVcsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsRUFBRSxNQUFNLENBQUMsQ0FBQztJQUM3RyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLHNCQUFzQixDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDbEYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksU0FBUyxDQUFDLENBQUM7SUFDeEUsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7OztPQWNHO0lBQ0kscUJBQXFCLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUNqRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUMxRSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSwwQkFBMEIsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3RGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywyQkFBMkIsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQy9FLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLDhCQUE4QixDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDMUYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLCtCQUErQixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDbkYsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLCtCQUErQixDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDM0YsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdDQUFnQyxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDcEYsQ0FBQztJQUVEOzs7Ozs7O09BT0c7SUFDSSxvQkFBb0IsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ2hGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ3pFLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLHVCQUF1QixDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDbkYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDNUUsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxrQkFBa0IsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQzlFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ3ZFLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksa0NBQWtDLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUM5RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUNBQW1DLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUN2RixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLDZCQUE2QixDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDekYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDhCQUE4QixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDbEYsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSx3QkFBd0IsQ0FBQyxLQUFlO1FBQzdDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsRUFBRSxDQUFDLE9BQU8sS0FBSyxLQUFLLFdBQVcsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsRUFBRSxNQUFNLENBQUMsQ0FBQztJQUNuRyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGlCQUFpQixDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDN0UsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDdEUsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksd0JBQXdCLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUNwRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUM3RSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSx1QkFBdUIsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ25GLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQzVFLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksb0NBQW9DLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUNoRyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUNBQXFDLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxTQUFTLENBQUMsQ0FBQztJQUN0RixDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksbUNBQW1DLENBQUMsS0FBZTtRQUN4RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0NBQW9DLEVBQUUsQ0FBQyxPQUFPLEtBQUssS0FBSyxXQUFXLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEVBQUUsTUFBTSxDQUFDLENBQUM7SUFDOUcsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLHNCQUFzQixDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDbEYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDM0UsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSwyQkFBMkIsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3ZGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw0QkFBNEIsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2hGLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksK0JBQStCLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUMzRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0NBQWdDLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNwRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxnQ0FBZ0MsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQzVGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQ0FBaUMsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ3JGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxxQkFBcUIsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ2pGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQzFFLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksd0JBQXdCLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUNwRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUM3RSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSxlQUFlLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDM0YsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFtQixNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2hGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09Bc0RHO0lBQ0ksZ0JBQWdCLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDNUYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFvQixNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2pGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksWUFBWSxDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDeEUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2pFLENBQUM7SUFFRDs7O09BR0c7SUFDSCxZQUFZLEtBQWdDO1FBQzFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQS9qRFIsa0JBQWEsR0FBRyxZQUFZLENBQUM7UUFpSzFCLG9CQUFlLEdBQW9CO1lBQzNDLHdCQUF3QixFQUFFO2dCQUN4Qix3QkFBd0I7Z0JBQ3hCLHlCQUF5QjtnQkFDekIseUJBQXlCO2dCQUN6Qiw2QkFBNkI7Z0JBQzdCLFdBQVc7Z0JBQ1gsa0NBQWtDO2dCQUNsQyx3QkFBd0I7Z0JBQ3hCLHFCQUFxQjthQUN0QjtZQUNELEtBQUssRUFBRTtnQkFDTCxRQUFRO2dCQUNSLE9BQU87Z0JBQ1AsTUFBTTtnQkFDTixLQUFLO2FBQ047WUFDRCxJQUFJLEVBQUU7Z0JBQ0osS0FBSzthQUNOO1NBQ0YsQ0FBQztJQTI0Q0YsQ0FBQztDQUNGO0FBbGtERCxnQ0Fra0RDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQWNjZXNzTGV2ZWxMaXN0IH0gZnJvbSAnLi4vLi4vc2hhcmVkL2FjY2Vzcy1sZXZlbCc7XG5pbXBvcnQgeyBQb2xpY3lTdGF0ZW1lbnQsIE9wZXJhdG9yIH0gZnJvbSAnLi4vLi4vc2hhcmVkJztcbmltcG9ydCB7IGF3c19pYW0gYXMgaWFtIH0gZnJvbSBcImF3cy1jZGstbGliXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFthcGlnYXRld2F5XShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hbWF6b25hcGlnYXRld2F5bWFuYWdlbWVudC5odG1sKS5cbiAqXG4gKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAqL1xuZXhwb3J0IGNsYXNzIEFwaWdhdGV3YXkgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICdhcGlnYXRld2F5JztcblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYWRkIGNlcnRpZmljYXRlcyBmb3IgbXV0dWFsIFRMUyBhdXRoZW50aWNhdGlvbiB0byBhIGRvbWFpbiBuYW1lLiBUaGlzIGlzIGFuIGFkZGl0aW9uYWwgYXV0aG9yaXphdGlvbiBjb250cm9sIGZvciBtYW5hZ2luZyB0aGUgRG9tYWluTmFtZSByZXNvdXJjZSBkdWUgdG8gdGhlIHNlbnNpdGl2ZSBuYXR1cmUgb2YgbVRMU1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfT3BlcmF0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9BZGRDZXJ0aWZpY2F0ZVRvRG9tYWluKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdBZGRDZXJ0aWZpY2F0ZVRvRG9tYWluJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGFuIGFjY2VzcyBhc3NvY2lhdGlvbiBmcm9tIGFuIGFjY2VzcyBhc3NvY2lhdGlvbiBzb3VyY2UgdG8gYSBjdXN0b20gZG9tYWluIG5hbWUgZm9yIHByaXZhdGUgQVBJc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfT3BlcmF0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVBY2Nlc3NBc3NvY2lhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlQWNjZXNzQXNzb2NpYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBwYXJ0aWN1bGFyIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfT3BlcmF0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9ERUxFVEUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RFTEVURScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlYWQgYSBwYXJ0aWN1bGFyIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9PcGVyYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dFVCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR0VUJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGEgcGFydGljdWxhciByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX09wZXJhdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUEFUQ0goKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1BBVENIJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgcGFydGljdWxhciByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX09wZXJhdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUE9TVCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnUE9TVCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhIHBhcnRpY3VsYXIgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9PcGVyYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1BVVCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnUFVUJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVqZWN0IGFuIGV4aXN0aW5nIGFjY2VzcyBhc3NvY2lhdGlvbiBvd25lZCBieSBhbm90aGVyIGFjY291bnQgdG8gYSBjdXN0b20gZG9tYWluIG5hbWUgZm9yIHByaXZhdGUgQVBJc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfT3BlcmF0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9SZWplY3RBY2Nlc3NBc3NvY2lhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnUmVqZWN0QWNjZXNzQXNzb2NpYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZW1vdmUgY2VydGlmaWNhdGVzIGZvciBtdXR1YWwgVExTIGF1dGhlbnRpY2F0aW9uIGZyb20gYSBkb21haW4gbmFtZS4gVGhpcyBpcyBhbiBhZGRpdGlvbmFsIGF1dGhvcml6YXRpb24gY29udHJvbCBmb3IgbWFuYWdpbmcgdGhlIERvbWFpbk5hbWUgcmVzb3VyY2UgZHVlIHRvIHRoZSBzZW5zaXRpdmUgbmF0dXJlIG9mIG1UTFNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX09wZXJhdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUmVtb3ZlQ2VydGlmaWNhdGVGcm9tRG9tYWluKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdSZW1vdmVDZXJ0aWZpY2F0ZUZyb21Eb21haW4nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzZXQgYSBXQUYgYWNjZXNzIGNvbnRyb2wgbGlzdCAoQUNMKS4gVGhpcyBpcyBhbiBhZGRpdGlvbmFsIGF1dGhvcml6YXRpb24gY29udHJvbCBmb3IgbWFuYWdpbmcgdGhlIFN0YWdlIHJlc291cmNlIGR1ZSB0byB0aGUgc2Vuc2l0aXZlIG5hdHVyZSBvZiBXZWJBY2wnc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfT3BlcmF0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TZXRXZWJBQ0woKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1NldFdlYkFDTCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSB0aGUgbWFuYWdlbWVudCBwb2xpY3kgb2YgYSBjdXN0b20gZG9tYWluIG5hbWUgZm9yIHByaXZhdGUgQVBJc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfT3BlcmF0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVEb21haW5OYW1lTWFuYWdlbWVudFBvbGljeSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlRG9tYWluTmFtZU1hbmFnZW1lbnRQb2xpY3knKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgdGhlIGludm9rZSBwb2xpY3kgb2YgYSBjdXN0b20gZG9tYWluIG5hbWUgZm9yIHByaXZhdGUgQVBJc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfT3BlcmF0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVEb21haW5OYW1lUG9saWN5KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVEb21haW5OYW1lUG9saWN5Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbWFuYWdlIHRoZSBJQU0gcmVzb3VyY2UgcG9saWN5IGZvciBhbiBBUEkuIFRoaXMgaXMgYW4gYWRkaXRpb25hbCBhdXRob3JpemF0aW9uIGNvbnRyb2wgZm9yIG1hbmFnaW5nIGFuIEFQSSBkdWUgdG8gdGhlIHNlbnNpdGl2ZSBuYXR1cmUgb2YgdGhlIHJlc291cmNlIHBvbGljeVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfT3BlcmF0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVSZXN0QXBpUG9saWN5KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVSZXN0QXBpUG9saWN5Jyk7XG4gIH1cblxuICBwcm90ZWN0ZWQgYWNjZXNzTGV2ZWxMaXN0OiBBY2Nlc3NMZXZlbExpc3QgPSB7XG4gICAgJ1Blcm1pc3Npb25zIG1hbmFnZW1lbnQnOiBbXG4gICAgICAnQWRkQ2VydGlmaWNhdGVUb0RvbWFpbicsXG4gICAgICAnQ3JlYXRlQWNjZXNzQXNzb2NpYXRpb24nLFxuICAgICAgJ1JlamVjdEFjY2Vzc0Fzc29jaWF0aW9uJyxcbiAgICAgICdSZW1vdmVDZXJ0aWZpY2F0ZUZyb21Eb21haW4nLFxuICAgICAgJ1NldFdlYkFDTCcsXG4gICAgICAnVXBkYXRlRG9tYWluTmFtZU1hbmFnZW1lbnRQb2xpY3knLFxuICAgICAgJ1VwZGF0ZURvbWFpbk5hbWVQb2xpY3knLFxuICAgICAgJ1VwZGF0ZVJlc3RBcGlQb2xpY3knXG4gICAgXSxcbiAgICBXcml0ZTogW1xuICAgICAgJ0RFTEVURScsXG4gICAgICAnUEFUQ0gnLFxuICAgICAgJ1BPU1QnLFxuICAgICAgJ1BVVCdcbiAgICBdLFxuICAgIFJlYWQ6IFtcbiAgICAgICdHRVQnXG4gICAgXVxuICB9O1xuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBBY2NvdW50IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKi9cbiAgcHVibGljIG9uQWNjb3VudChyZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovYWNjb3VudGApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIEFwaUtleSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX0FwaUtleS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBhcGlLZXlJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBhcGlLZXlJZC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkFwaUtleShhcGlLZXlJZDogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovYXBpa2V5cy8keyBhcGlLZXlJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgQXBpS2V5cyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX0FwaUtleS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25BcGlLZXlzKHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi9hcGlrZXlzYCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgQXV0aG9yaXplciB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX0F1dGhvcml6ZXIuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVzdEFwaUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc3RBcGlJZC5cbiAgICogQHBhcmFtIGF1dGhvcml6ZXJJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBhdXRob3JpemVySWQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZSZXF1ZXN0QXV0aG9yaXplclR5cGUoKVxuICAgKiAtIC5pZlJlcXVlc3RBdXRob3JpemVyVXJpKClcbiAgICogLSAuaWZSZXF1ZXN0Q29nbml0b1VzZXJQb29sUHJvdmlkZXJBcm4oKVxuICAgKiAtIC5pZlJlc291cmNlQXV0aG9yaXplclR5cGUoKVxuICAgKiAtIC5pZlJlc291cmNlQXV0aG9yaXplclVyaSgpXG4gICAqIC0gLmlmUmVzb3VyY2VDb2duaXRvVXNlclBvb2xQcm92aWRlckFybigpXG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uQXV0aG9yaXplcihyZXN0QXBpSWQ6IHN0cmluZywgYXV0aG9yaXplcklkOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi9yZXN0YXBpcy8keyByZXN0QXBpSWQgfS9hdXRob3JpemVycy8keyBhdXRob3JpemVySWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIEF1dGhvcml6ZXJzIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfQXV0aG9yaXplci5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZXN0QXBpSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzdEFwaUlkLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmUmVxdWVzdEF1dGhvcml6ZXJUeXBlKClcbiAgICogLSAuaWZSZXF1ZXN0QXV0aG9yaXplclVyaSgpXG4gICAqIC0gLmlmUmVxdWVzdENvZ25pdG9Vc2VyUG9vbFByb3ZpZGVyQXJuKClcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25BdXRob3JpemVycyhyZXN0QXBpSWQ6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L3Jlc3RhcGlzLyR7IHJlc3RBcGlJZCB9L2F1dGhvcml6ZXJzYCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgQmFzZVBhdGhNYXBwaW5nIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfQmFzZVBhdGhNYXBwaW5nLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGRvbWFpbk5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgZG9tYWluTmFtZS5cbiAgICogQHBhcmFtIGJhc2VQYXRoIC0gSWRlbnRpZmllciBmb3IgdGhlIGJhc2VQYXRoLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uQmFzZVBhdGhNYXBwaW5nKGRvbWFpbk5hbWU6IHN0cmluZywgYmFzZVBhdGg6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L2RvbWFpbm5hbWVzLyR7IGRvbWFpbk5hbWUgfS9iYXNlcGF0aG1hcHBpbmdzLyR7IGJhc2VQYXRoIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBCYXNlUGF0aE1hcHBpbmdzIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfQmFzZVBhdGhNYXBwaW5nLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGRvbWFpbk5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgZG9tYWluTmFtZS5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkJhc2VQYXRoTWFwcGluZ3MoZG9tYWluTmFtZTogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovZG9tYWlubmFtZXMvJHsgZG9tYWluTmFtZSB9L2Jhc2VwYXRobWFwcGluZ3NgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBDbGllbnRDZXJ0aWZpY2F0ZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX0NsaWVudENlcnRpZmljYXRlLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGNsaWVudENlcnRpZmljYXRlSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgY2xpZW50Q2VydGlmaWNhdGVJZC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkNsaWVudENlcnRpZmljYXRlKGNsaWVudENlcnRpZmljYXRlSWQ6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L2NsaWVudGNlcnRpZmljYXRlcy8keyBjbGllbnRDZXJ0aWZpY2F0ZUlkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBDbGllbnRDZXJ0aWZpY2F0ZXMgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9DbGllbnRDZXJ0aWZpY2F0ZS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25DbGllbnRDZXJ0aWZpY2F0ZXMocmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L2NsaWVudGNlcnRpZmljYXRlc2ApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIERlcGxveW1lbnQgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9EZXBsb3ltZW50Lmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlc3RBcGlJZCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXN0QXBpSWQuXG4gICAqIEBwYXJhbSBkZXBsb3ltZW50SWQgLSBJZGVudGlmaWVyIGZvciB0aGUgZGVwbG95bWVudElkLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRGVwbG95bWVudChyZXN0QXBpSWQ6IHN0cmluZywgZGVwbG95bWVudElkOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi9yZXN0YXBpcy8keyByZXN0QXBpSWQgfS9kZXBsb3ltZW50cy8keyBkZXBsb3ltZW50SWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIERlcGxveW1lbnRzIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfRGVwbG95bWVudC5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZXN0QXBpSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzdEFwaUlkLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmUmVxdWVzdFN0YWdlTmFtZSgpXG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRGVwbG95bWVudHMocmVzdEFwaUlkOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi9yZXN0YXBpcy8keyByZXN0QXBpSWQgfS9kZXBsb3ltZW50c2ApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIERvY3VtZW50YXRpb25QYXJ0IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfRG9jdW1lbnRhdGlvblBhcnQuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVzdEFwaUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc3RBcGlJZC5cbiAgICogQHBhcmFtIGRvY3VtZW50YXRpb25QYXJ0SWQgLSBJZGVudGlmaWVyIGZvciB0aGUgZG9jdW1lbnRhdGlvblBhcnRJZC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkRvY3VtZW50YXRpb25QYXJ0KHJlc3RBcGlJZDogc3RyaW5nLCBkb2N1bWVudGF0aW9uUGFydElkOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi9yZXN0YXBpcy8keyByZXN0QXBpSWQgfS9kb2N1bWVudGF0aW9uL3BhcnRzLyR7IGRvY3VtZW50YXRpb25QYXJ0SWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIERvY3VtZW50YXRpb25QYXJ0cyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX0RvY3VtZW50YXRpb25QYXJ0Lmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlc3RBcGlJZCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXN0QXBpSWQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25Eb2N1bWVudGF0aW9uUGFydHMocmVzdEFwaUlkOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi9yZXN0YXBpcy8keyByZXN0QXBpSWQgfS9kb2N1bWVudGF0aW9uL3BhcnRzYCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgRG9jdW1lbnRhdGlvblZlcnNpb24gdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9Eb2N1bWVudGF0aW9uVmVyc2lvbi5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZXN0QXBpSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzdEFwaUlkLlxuICAgKiBAcGFyYW0gZG9jdW1lbnRhdGlvblZlcnNpb25JZCAtIElkZW50aWZpZXIgZm9yIHRoZSBkb2N1bWVudGF0aW9uVmVyc2lvbklkLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRG9jdW1lbnRhdGlvblZlcnNpb24ocmVzdEFwaUlkOiBzdHJpbmcsIGRvY3VtZW50YXRpb25WZXJzaW9uSWQ6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L3Jlc3RhcGlzLyR7IHJlc3RBcGlJZCB9L2RvY3VtZW50YXRpb24vdmVyc2lvbnMvJHsgZG9jdW1lbnRhdGlvblZlcnNpb25JZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgRG9jdW1lbnRhdGlvblZlcnNpb25zIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfRG9jdW1lbnRhdGlvblZlcnNpb24uaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVzdEFwaUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc3RBcGlJZC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkRvY3VtZW50YXRpb25WZXJzaW9ucyhyZXN0QXBpSWQ6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L3Jlc3RhcGlzLyR7IHJlc3RBcGlJZCB9L2RvY3VtZW50YXRpb24vdmVyc2lvbnNgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBEb21haW5OYW1lIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfRG9tYWluTmFtZS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBkb21haW5OYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGRvbWFpbk5hbWUuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZSZXF1ZXN0RW5kcG9pbnRUeXBlKClcbiAgICogLSAuaWZSZXF1ZXN0TXRsc1RydXN0U3RvcmVVcmkoKVxuICAgKiAtIC5pZlJlcXVlc3RNdGxzVHJ1c3RTdG9yZVZlcnNpb24oKVxuICAgKiAtIC5pZlJlcXVlc3RTZWN1cml0eVBvbGljeSgpXG4gICAqIC0gLmlmUmVzb3VyY2VFbmRwb2ludFR5cGUoKVxuICAgKiAtIC5pZlJlc291cmNlTXRsc1RydXN0U3RvcmVVcmkoKVxuICAgKiAtIC5pZlJlc291cmNlTXRsc1RydXN0U3RvcmVWZXJzaW9uKClcbiAgICogLSAuaWZSZXNvdXJjZVJvdXRpbmdNb2RlKClcbiAgICogLSAuaWZSZXNvdXJjZVNlY3VyaXR5UG9saWN5KClcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25Eb21haW5OYW1lKGRvbWFpbk5hbWU6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L2RvbWFpbm5hbWVzLyR7IGRvbWFpbk5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIERvbWFpbk5hbWVzIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfRG9tYWluTmFtZS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZSZXF1ZXN0RW5kcG9pbnRUeXBlKClcbiAgICogLSAuaWZSZXF1ZXN0TXRsc1RydXN0U3RvcmVVcmkoKVxuICAgKiAtIC5pZlJlcXVlc3RNdGxzVHJ1c3RTdG9yZVZlcnNpb24oKVxuICAgKiAtIC5pZlJlcXVlc3RTZWN1cml0eVBvbGljeSgpXG4gICAqIC0gLmlmUmVzb3VyY2VSb3V0aW5nTW9kZSgpXG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRG9tYWluTmFtZXMocmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L2RvbWFpbm5hbWVzYCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgRG9tYWluTmFtZUFjY2Vzc0Fzc29jaWF0aW9uIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfRG9tYWluTmFtZUFjY2Vzc0Fzc29jaWF0aW9uLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGRvbWFpbk5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgZG9tYWluTmFtZS5cbiAgICogQHBhcmFtIHNvdXJjZVR5cGUgLSBJZGVudGlmaWVyIGZvciB0aGUgc291cmNlVHlwZS5cbiAgICogQHBhcmFtIHNvdXJjZUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHNvdXJjZUlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICovXG4gIHB1YmxpYyBvbkRvbWFpbk5hbWVBY2Nlc3NBc3NvY2lhdGlvbihkb21haW5OYW1lOiBzdHJpbmcsIHNvdXJjZVR5cGU6IHN0cmluZywgc291cmNlSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTovZG9tYWlubmFtZWFjY2Vzc2Fzc29jaWF0aW9ucy9kb21haW5uYW1lLyR7IGRvbWFpbk5hbWUgfS8keyBzb3VyY2VUeXBlIH0vJHsgc291cmNlSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIERvbWFpbk5hbWVBY2Nlc3NBc3NvY2lhdGlvbnMgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9Eb21haW5OYW1lQWNjZXNzQXNzb2NpYXRpb24uaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZSZXF1ZXN0QWNjZXNzQXNzb2NpYXRpb25Tb3VyY2UoKVxuICAgKiAtIC5pZlJlcXVlc3REb21haW5OYW1lQXJuKClcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25Eb21haW5OYW1lQWNjZXNzQXNzb2NpYXRpb25zKGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06L2RvbWFpbm5hbWVhY2Nlc3Nhc3NvY2lhdGlvbnNgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBHYXRld2F5UmVzcG9uc2UgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9HYXRld2F5UmVzcG9uc2UuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVzdEFwaUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc3RBcGlJZC5cbiAgICogQHBhcmFtIHJlc3BvbnNlVHlwZSAtIElkZW50aWZpZXIgZm9yIHRoZSByZXNwb25zZVR5cGUuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25HYXRld2F5UmVzcG9uc2UocmVzdEFwaUlkOiBzdHJpbmcsIHJlc3BvbnNlVHlwZTogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovcmVzdGFwaXMvJHsgcmVzdEFwaUlkIH0vZ2F0ZXdheXJlc3BvbnNlcy8keyByZXNwb25zZVR5cGUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIEdhdGV3YXlSZXNwb25zZXMgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9HYXRld2F5UmVzcG9uc2UuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVzdEFwaUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc3RBcGlJZC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkdhdGV3YXlSZXNwb25zZXMocmVzdEFwaUlkOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi9yZXN0YXBpcy8keyByZXN0QXBpSWQgfS9nYXRld2F5cmVzcG9uc2VzYCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgSW50ZWdyYXRpb24gdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9JbnRlZ3JhdGlvbi5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZXN0QXBpSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzdEFwaUlkLlxuICAgKiBAcGFyYW0gcmVzb3VyY2VJZCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXNvdXJjZUlkLlxuICAgKiBAcGFyYW0gaHR0cE1ldGhvZFR5cGUgLSBJZGVudGlmaWVyIGZvciB0aGUgaHR0cE1ldGhvZFR5cGUuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25JbnRlZ3JhdGlvbihyZXN0QXBpSWQ6IHN0cmluZywgcmVzb3VyY2VJZDogc3RyaW5nLCBodHRwTWV0aG9kVHlwZTogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovcmVzdGFwaXMvJHsgcmVzdEFwaUlkIH0vcmVzb3VyY2VzLyR7IHJlc291cmNlSWQgfS9tZXRob2RzLyR7IGh0dHBNZXRob2RUeXBlIH0vaW50ZWdyYXRpb25gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBJbnRlZ3JhdGlvblJlc3BvbnNlIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfSW50ZWdyYXRpb25SZXNwb25zZS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZXN0QXBpSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzdEFwaUlkLlxuICAgKiBAcGFyYW0gcmVzb3VyY2VJZCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXNvdXJjZUlkLlxuICAgKiBAcGFyYW0gaHR0cE1ldGhvZFR5cGUgLSBJZGVudGlmaWVyIGZvciB0aGUgaHR0cE1ldGhvZFR5cGUuXG4gICAqIEBwYXJhbSBzdGF0dXNDb2RlIC0gSWRlbnRpZmllciBmb3IgdGhlIHN0YXR1c0NvZGUuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25JbnRlZ3JhdGlvblJlc3BvbnNlKHJlc3RBcGlJZDogc3RyaW5nLCByZXNvdXJjZUlkOiBzdHJpbmcsIGh0dHBNZXRob2RUeXBlOiBzdHJpbmcsIHN0YXR1c0NvZGU6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L3Jlc3RhcGlzLyR7IHJlc3RBcGlJZCB9L3Jlc291cmNlcy8keyByZXNvdXJjZUlkIH0vbWV0aG9kcy8keyBodHRwTWV0aG9kVHlwZSB9L2ludGVncmF0aW9uL3Jlc3BvbnNlcy8keyBzdGF0dXNDb2RlIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBNZXRob2QgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9NZXRob2QuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVzdEFwaUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc3RBcGlJZC5cbiAgICogQHBhcmFtIHJlc291cmNlSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzb3VyY2VJZC5cbiAgICogQHBhcmFtIGh0dHBNZXRob2RUeXBlIC0gSWRlbnRpZmllciBmb3IgdGhlIGh0dHBNZXRob2RUeXBlLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmUmVxdWVzdEFwaUtleVJlcXVpcmVkKClcbiAgICogLSAuaWZSZXF1ZXN0Um91dGVBdXRob3JpemF0aW9uVHlwZSgpXG4gICAqIC0gLmlmUmVzb3VyY2VBcGlLZXlSZXF1aXJlZCgpXG4gICAqIC0gLmlmUmVzb3VyY2VSb3V0ZUF1dGhvcml6YXRpb25UeXBlKClcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25NZXRob2QocmVzdEFwaUlkOiBzdHJpbmcsIHJlc291cmNlSWQ6IHN0cmluZywgaHR0cE1ldGhvZFR5cGU6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L3Jlc3RhcGlzLyR7IHJlc3RBcGlJZCB9L3Jlc291cmNlcy8keyByZXNvdXJjZUlkIH0vbWV0aG9kcy8keyBodHRwTWV0aG9kVHlwZSB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgTWV0aG9kUmVzcG9uc2UgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9NZXRob2RSZXNwb25zZS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZXN0QXBpSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzdEFwaUlkLlxuICAgKiBAcGFyYW0gcmVzb3VyY2VJZCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXNvdXJjZUlkLlxuICAgKiBAcGFyYW0gaHR0cE1ldGhvZFR5cGUgLSBJZGVudGlmaWVyIGZvciB0aGUgaHR0cE1ldGhvZFR5cGUuXG4gICAqIEBwYXJhbSBzdGF0dXNDb2RlIC0gSWRlbnRpZmllciBmb3IgdGhlIHN0YXR1c0NvZGUuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25NZXRob2RSZXNwb25zZShyZXN0QXBpSWQ6IHN0cmluZywgcmVzb3VyY2VJZDogc3RyaW5nLCBodHRwTWV0aG9kVHlwZTogc3RyaW5nLCBzdGF0dXNDb2RlOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi9yZXN0YXBpcy8keyByZXN0QXBpSWQgfS9yZXNvdXJjZXMvJHsgcmVzb3VyY2VJZCB9L21ldGhvZHMvJHsgaHR0cE1ldGhvZFR5cGUgfS9yZXNwb25zZXMvJHsgc3RhdHVzQ29kZSB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgTW9kZWwgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9Nb2RlbC5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZXN0QXBpSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzdEFwaUlkLlxuICAgKiBAcGFyYW0gbW9kZWxOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIG1vZGVsTmFtZS5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbk1vZGVsKHJlc3RBcGlJZDogc3RyaW5nLCBtb2RlbE5hbWU6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L3Jlc3RhcGlzLyR7IHJlc3RBcGlJZCB9L21vZGVscy8keyBtb2RlbE5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIE1vZGVscyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX01vZGVsLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlc3RBcGlJZCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXN0QXBpSWQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25Nb2RlbHMocmVzdEFwaUlkOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi9yZXN0YXBpcy8keyByZXN0QXBpSWQgfS9tb2RlbHNgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBQcml2YXRlQmFzZVBhdGhNYXBwaW5nIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfQmFzZVBhdGhNYXBwaW5nLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGRvbWFpbk5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgZG9tYWluTmFtZS5cbiAgICogQHBhcmFtIGRvbWFpbklkZW50aWZpZXIgLSBJZGVudGlmaWVyIGZvciB0aGUgZG9tYWluSWRlbnRpZmllci5cbiAgICogQHBhcmFtIGJhc2VQYXRoIC0gSWRlbnRpZmllciBmb3IgdGhlIGJhc2VQYXRoLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uUHJpdmF0ZUJhc2VQYXRoTWFwcGluZyhkb21haW5OYW1lOiBzdHJpbmcsIGRvbWFpbklkZW50aWZpZXI6IHN0cmluZywgYmFzZVBhdGg6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L2RvbWFpbm5hbWVzLyR7IGRvbWFpbk5hbWUgfSskeyBkb21haW5JZGVudGlmaWVyIH0vYmFzZXBhdGhtYXBwaW5ncy8keyBiYXNlUGF0aCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgUHJpdmF0ZUJhc2VQYXRoTWFwcGluZ3MgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9CYXNlUGF0aE1hcHBpbmcuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gZG9tYWluTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBkb21haW5OYW1lLlxuICAgKiBAcGFyYW0gZG9tYWluSWRlbnRpZmllciAtIElkZW50aWZpZXIgZm9yIHRoZSBkb21haW5JZGVudGlmaWVyLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uUHJpdmF0ZUJhc2VQYXRoTWFwcGluZ3MoZG9tYWluTmFtZTogc3RyaW5nLCBkb21haW5JZGVudGlmaWVyOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi9kb21haW5uYW1lcy8keyBkb21haW5OYW1lIH0rJHsgZG9tYWluSWRlbnRpZmllciB9L2Jhc2VwYXRobWFwcGluZ3NgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBQcml2YXRlRG9tYWluTmFtZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX0RvbWFpbk5hbWUuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gZG9tYWluTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBkb21haW5OYW1lLlxuICAgKiBAcGFyYW0gZG9tYWluSWRlbnRpZmllciAtIElkZW50aWZpZXIgZm9yIHRoZSBkb21haW5JZGVudGlmaWVyLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZSZXF1ZXN0RW5kcG9pbnRUeXBlKClcbiAgICogLSAuaWZSZXNvdXJjZUVuZHBvaW50VHlwZSgpXG4gICAqIC0gLmlmUmVzb3VyY2VSb3V0aW5nTW9kZSgpXG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uUHJpdmF0ZURvbWFpbk5hbWUoZG9tYWluTmFtZTogc3RyaW5nLCBkb21haW5JZGVudGlmaWVyOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06L2RvbWFpbm5hbWVzLyR7IGRvbWFpbk5hbWUgfSskeyBkb21haW5JZGVudGlmaWVyIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBSZXF1ZXN0VmFsaWRhdG9yIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfUmVxdWVzdFZhbGlkYXRvci5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZXN0QXBpSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzdEFwaUlkLlxuICAgKiBAcGFyYW0gcmVxdWVzdFZhbGlkYXRvcklkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlcXVlc3RWYWxpZGF0b3JJZC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblJlcXVlc3RWYWxpZGF0b3IocmVzdEFwaUlkOiBzdHJpbmcsIHJlcXVlc3RWYWxpZGF0b3JJZDogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovcmVzdGFwaXMvJHsgcmVzdEFwaUlkIH0vcmVxdWVzdHZhbGlkYXRvcnMvJHsgcmVxdWVzdFZhbGlkYXRvcklkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBSZXF1ZXN0VmFsaWRhdG9ycyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX1JlcXVlc3RWYWxpZGF0b3IuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVzdEFwaUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc3RBcGlJZC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblJlcXVlc3RWYWxpZGF0b3JzKHJlc3RBcGlJZDogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovcmVzdGFwaXMvJHsgcmVzdEFwaUlkIH0vcmVxdWVzdHZhbGlkYXRvcnNgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBSZXNvdXJjZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX1Jlc291cmNlLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlc3RBcGlJZCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXN0QXBpSWQuXG4gICAqIEBwYXJhbSByZXNvdXJjZUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc291cmNlSWQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25SZXNvdXJjZShyZXN0QXBpSWQ6IHN0cmluZywgcmVzb3VyY2VJZDogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovcmVzdGFwaXMvJHsgcmVzdEFwaUlkIH0vcmVzb3VyY2VzLyR7IHJlc291cmNlSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIFJlc291cmNlcyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX1Jlc291cmNlLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlc3RBcGlJZCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXN0QXBpSWQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25SZXNvdXJjZXMocmVzdEFwaUlkOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi9yZXN0YXBpcy8keyByZXN0QXBpSWQgfS9yZXNvdXJjZXNgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBSZXN0QXBpIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfUmVzdEFwaS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZXN0QXBpSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzdEFwaUlkLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmUmVxdWVzdEFwaUtleVJlcXVpcmVkKClcbiAgICogLSAuaWZSZXF1ZXN0QXBpTmFtZSgpXG4gICAqIC0gLmlmUmVxdWVzdEF1dGhvcml6ZXJUeXBlKClcbiAgICogLSAuaWZSZXF1ZXN0QXV0aG9yaXplclVyaSgpXG4gICAqIC0gLmlmUmVxdWVzdERpc2FibGVFeGVjdXRlQXBpRW5kcG9pbnQoKVxuICAgKiAtIC5pZlJlcXVlc3RFbmRwb2ludFR5cGUoKVxuICAgKiAtIC5pZlJlcXVlc3RSb3V0ZUF1dGhvcml6YXRpb25UeXBlKClcbiAgICogLSAuaWZSZXNvdXJjZUFwaUtleVJlcXVpcmVkKClcbiAgICogLSAuaWZSZXNvdXJjZUFwaU5hbWUoKVxuICAgKiAtIC5pZlJlc291cmNlQXV0aG9yaXplclR5cGUoKVxuICAgKiAtIC5pZlJlc291cmNlQXV0aG9yaXplclVyaSgpXG4gICAqIC0gLmlmUmVzb3VyY2VEaXNhYmxlRXhlY3V0ZUFwaUVuZHBvaW50KClcbiAgICogLSAuaWZSZXNvdXJjZUVuZHBvaW50VHlwZSgpXG4gICAqIC0gLmlmUmVzb3VyY2VSb3V0ZUF1dGhvcml6YXRpb25UeXBlKClcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25SZXN0QXBpKHJlc3RBcGlJZDogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovcmVzdGFwaXMvJHsgcmVzdEFwaUlkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBSZXN0QXBpcyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX1Jlc3RBcGkuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmUmVxdWVzdEFwaUtleVJlcXVpcmVkKClcbiAgICogLSAuaWZSZXF1ZXN0QXBpTmFtZSgpXG4gICAqIC0gLmlmUmVxdWVzdEF1dGhvcml6ZXJUeXBlKClcbiAgICogLSAuaWZSZXF1ZXN0QXV0aG9yaXplclVyaSgpXG4gICAqIC0gLmlmUmVxdWVzdERpc2FibGVFeGVjdXRlQXBpRW5kcG9pbnQoKVxuICAgKiAtIC5pZlJlcXVlc3RFbmRwb2ludFR5cGUoKVxuICAgKiAtIC5pZlJlcXVlc3RSb3V0ZUF1dGhvcml6YXRpb25UeXBlKClcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25SZXN0QXBpcyhyZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovcmVzdGFwaXNgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBTZGsgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVzdEFwaUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc3RBcGlJZC5cbiAgICogQHBhcmFtIHN0YWdlTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBzdGFnZU5hbWUuXG4gICAqIEBwYXJhbSBzZGtUeXBlIC0gSWRlbnRpZmllciBmb3IgdGhlIHNka1R5cGUuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25TZGsocmVzdEFwaUlkOiBzdHJpbmcsIHN0YWdlTmFtZTogc3RyaW5nLCBzZGtUeXBlOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi9yZXN0YXBpcy8keyByZXN0QXBpSWQgfS9zdGFnZXMvJHsgc3RhZ2VOYW1lIH0vc2Rrcy8keyBzZGtUeXBlIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBTdGFnZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX1N0YWdlLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlc3RBcGlJZCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXN0QXBpSWQuXG4gICAqIEBwYXJhbSBzdGFnZU5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgc3RhZ2VOYW1lLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmUmVxdWVzdEFjY2Vzc0xvZ2dpbmdEZXN0aW5hdGlvbigpXG4gICAqIC0gLmlmUmVxdWVzdEFjY2Vzc0xvZ2dpbmdGb3JtYXQoKVxuICAgKiAtIC5pZlJlc291cmNlQWNjZXNzTG9nZ2luZ0Rlc3RpbmF0aW9uKClcbiAgICogLSAuaWZSZXNvdXJjZUFjY2Vzc0xvZ2dpbmdGb3JtYXQoKVxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblN0YWdlKHJlc3RBcGlJZDogc3RyaW5nLCBzdGFnZU5hbWU6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L3Jlc3RhcGlzLyR7IHJlc3RBcGlJZCB9L3N0YWdlcy8keyBzdGFnZU5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIFN0YWdlcyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX1N0YWdlLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlc3RBcGlJZCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXN0QXBpSWQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZSZXF1ZXN0QWNjZXNzTG9nZ2luZ0Rlc3RpbmF0aW9uKClcbiAgICogLSAuaWZSZXF1ZXN0QWNjZXNzTG9nZ2luZ0Zvcm1hdCgpXG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uU3RhZ2VzKHJlc3RBcGlJZDogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovcmVzdGFwaXMvJHsgcmVzdEFwaUlkIH0vc3RhZ2VzYCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgVGVtcGxhdGUgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gbW9kZWxOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIG1vZGVsTmFtZS5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblRlbXBsYXRlKG1vZGVsTmFtZTogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovcmVzdGFwaXMvbW9kZWxzLyR7IG1vZGVsTmFtZSB9L3RlbXBsYXRlYCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgVXNhZ2VQbGFuIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfVXNhZ2VQbGFuLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHVzYWdlUGxhbklkIC0gSWRlbnRpZmllciBmb3IgdGhlIHVzYWdlUGxhbklkLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uVXNhZ2VQbGFuKHVzYWdlUGxhbklkOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi91c2FnZXBsYW5zLyR7IHVzYWdlUGxhbklkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBVc2FnZVBsYW5zIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfVXNhZ2VQbGFuLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblVzYWdlUGxhbnMocmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L3VzYWdlcGxhbnNgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBVc2FnZVBsYW5LZXkgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9Vc2FnZVBsYW5LZXkuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gdXNhZ2VQbGFuSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgdXNhZ2VQbGFuSWQuXG4gICAqIEBwYXJhbSBpZCAtIElkZW50aWZpZXIgZm9yIHRoZSBpZC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblVzYWdlUGxhbktleSh1c2FnZVBsYW5JZDogc3RyaW5nLCBpZDogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovdXNhZ2VwbGFucy8keyB1c2FnZVBsYW5JZCB9L2tleXMvJHsgaWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIFVzYWdlUGxhbktleXMgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9Vc2FnZVBsYW5LZXkuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gdXNhZ2VQbGFuSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgdXNhZ2VQbGFuSWQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25Vc2FnZVBsYW5LZXlzKHVzYWdlUGxhbklkOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi91c2FnZXBsYW5zLyR7IHVzYWdlUGxhbklkIH0va2V5c2ApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIFZwY0xpbmsgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9WcGNMaW5rLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHZwY0xpbmtJZCAtIElkZW50aWZpZXIgZm9yIHRoZSB2cGNMaW5rSWQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25WcGNMaW5rKHZwY0xpbmtJZDogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovdnBjbGlua3MvJHsgdnBjTGlua0lkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBWcGNMaW5rcyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX1ZwY0xpbmsuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uVnBjTGlua3MocmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L3ZwY2xpbmtzYCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgVGFncyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9hcGlnYXRld2F5LXRhZ2dpbmcuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gdXJsRW5jb2RlZFJlc291cmNlQVJOIC0gSWRlbnRpZmllciBmb3IgdGhlIHVybEVuY29kZWRSZXNvdXJjZUFSTi5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKi9cbiAgcHVibGljIG9uVGFncyh1cmxFbmNvZGVkUmVzb3VyY2VBUk46IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L3RhZ3MvJHsgdXJsRW5jb2RlZFJlc291cmNlQVJOIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBhY2Nlc3MgYXNzb2NpYXRpb24gc291cmNlLiBBdmFpbGFibGUgZHVyaW5nIHRoZSBDcmVhdGVEb21haW5OYW1lQWNjZXNzQXNzb2NpYXRpb24gb3BlcmF0aW9uXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gRG9tYWluTmFtZUFjY2Vzc0Fzc29jaWF0aW9uc1xuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlcXVlc3RBY2Nlc3NBc3NvY2lhdGlvblNvdXJjZSh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgUmVxdWVzdC9BY2Nlc3NBc3NvY2lhdGlvblNvdXJjZWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IGFjY2VzcyBsb2cgZGVzdGluYXRpb24uIEF2YWlsYWJsZSBkdXJpbmcgdGhlIENyZWF0ZVN0YWdlIGFuZCBVcGRhdGVTdGFnZSBvcGVyYXRpb25zXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gU3RhZ2VcbiAgICogLSBTdGFnZXNcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZSZXF1ZXN0QWNjZXNzTG9nZ2luZ0Rlc3RpbmF0aW9uKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBSZXF1ZXN0L0FjY2Vzc0xvZ2dpbmdEZXN0aW5hdGlvbmAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IGFjY2VzcyBsb2cgZm9ybWF0LiBBdmFpbGFibGUgZHVyaW5nIHRoZSBDcmVhdGVTdGFnZSBhbmQgVXBkYXRlU3RhZ2Ugb3BlcmF0aW9uc1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIFN0YWdlXG4gICAqIC0gU3RhZ2VzXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmUmVxdWVzdEFjY2Vzc0xvZ2dpbmdGb3JtYXQodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFJlcXVlc3QvQWNjZXNzTG9nZ2luZ0Zvcm1hdGAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHdoZXRoZXIgYW4gQVBJIGtleSBpcyByZXF1aXJlZCBvciBub3QuIEF2YWlsYWJsZSBkdXJpbmcgdGhlIENyZWF0ZU1ldGhvZCBhbmQgUHV0TWV0aG9kIG9wZXJhdGlvbnMuIEFsc28gYXZhaWxhYmxlIGFzIGEgY29sbGVjdGlvbiBkdXJpbmcgaW1wb3J0IGFuZCByZWltcG9ydFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIE1ldGhvZFxuICAgKiAtIFJlc3RBcGlcbiAgICogLSBSZXN0QXBpc1xuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgYHRydWVgIG9yIGBmYWxzZWAuICoqRGVmYXVsdDoqKiBgdHJ1ZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlcXVlc3RBcGlLZXlSZXF1aXJlZCh2YWx1ZT86IGJvb2xlYW4pIHtcbiAgICByZXR1cm4gdGhpcy5pZihgUmVxdWVzdC9BcGlLZXlSZXF1aXJlZGAsICh0eXBlb2YgdmFsdWUgIT09ICd1bmRlZmluZWQnID8gdmFsdWUgOiB0cnVlKSwgJ0Jvb2wnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBBUEkgbmFtZS4gQXZhaWxhYmxlIGR1cmluZyB0aGUgQ3JlYXRlUmVzdEFwaSBhbmQgVXBkYXRlUmVzdEFwaSBvcGVyYXRpb25zXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gUmVzdEFwaVxuICAgKiAtIFJlc3RBcGlzXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmUmVxdWVzdEFwaU5hbWUodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFJlcXVlc3QvQXBpTmFtZWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHR5cGUgb2YgYXV0aG9yaXplciBpbiB0aGUgcmVxdWVzdCwgZm9yIGV4YW1wbGUgVE9LRU4sIFJFUVVFU1QsIEpXVC4gQXZhaWxhYmxlIGR1cmluZyBDcmVhdGVBdXRob3JpemVyIGFuZCBVcGRhdGVBdXRob3JpemVyLiBBbHNvIGF2YWlsYWJsZSBkdXJpbmcgaW1wb3J0IGFuZCByZWltcG9ydCBhcyBhbiBBcnJheU9mU3RyaW5nXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gQXV0aG9yaXplclxuICAgKiAtIEF1dGhvcml6ZXJzXG4gICAqIC0gUmVzdEFwaVxuICAgKiAtIFJlc3RBcGlzXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmUmVxdWVzdEF1dGhvcml6ZXJUeXBlKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBSZXF1ZXN0L0F1dGhvcml6ZXJUeXBlYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgVVJJIG9mIGEgTGFtYmRhIGF1dGhvcml6ZXIgZnVuY3Rpb24uIEF2YWlsYWJsZSBkdXJpbmcgQ3JlYXRlQXV0aG9yaXplciBhbmQgVXBkYXRlQXV0aG9yaXplci4gQWxzbyBhdmFpbGFibGUgZHVyaW5nIGltcG9ydCBhbmQgcmVpbXBvcnQgYXMgYW4gQXJyYXlPZlN0cmluZ1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIEF1dGhvcml6ZXJcbiAgICogLSBBdXRob3JpemVyc1xuICAgKiAtIFJlc3RBcGlcbiAgICogLSBSZXN0QXBpc1xuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlcXVlc3RBdXRob3JpemVyVXJpKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBSZXF1ZXN0L0F1dGhvcml6ZXJVcmlgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBDb2duaXRvIHVzZXIgcG9vbCBwcm92aWRlciBBUk4uIEF2YWlsYWJsZSBkdXJpbmcgQ3JlYXRlQXV0aG9yaXplciBhbmQgVXBkYXRlQXV0aG9yaXplclxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIEF1dGhvcml6ZXJcbiAgICogLSBBdXRob3JpemVyc1xuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFthcm4gb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX0FSTikuICoqRGVmYXVsdDoqKiBgQXJuTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlcXVlc3RDb2duaXRvVXNlclBvb2xQcm92aWRlckFybih2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgUmVxdWVzdC9Db2duaXRvVXNlclBvb2xQcm92aWRlckFybmAsIHZhbHVlLCBvcGVyYXRvciA/PyAnQXJuTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHN0YXR1cyBvZiB0aGUgZGVmYXVsdCBleGVjdXRlLWFwaSBlbmRwb2ludC4gQXZhaWxhYmxlIGR1cmluZyB0aGUgQ3JlYXRlUmVzdEFwaSBhbmQgRGVsZXRlUmVzdEFwaSBvcGVyYXRpb25zXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gUmVzdEFwaVxuICAgKiAtIFJlc3RBcGlzXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBgdHJ1ZWAgb3IgYGZhbHNlYC4gKipEZWZhdWx0OioqIGB0cnVlYFxuICAgKi9cbiAgcHVibGljIGlmUmVxdWVzdERpc2FibGVFeGVjdXRlQXBpRW5kcG9pbnQodmFsdWU/OiBib29sZWFuKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFJlcXVlc3QvRGlzYWJsZUV4ZWN1dGVBcGlFbmRwb2ludGAsICh0eXBlb2YgdmFsdWUgIT09ICd1bmRlZmluZWQnID8gdmFsdWUgOiB0cnVlKSwgJ0Jvb2wnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBkb21haW4gbmFtZSBBUk4uIEF2YWlsYWJsZSBkdXJpbmcgdGhlIENyZWF0ZURvbWFpbk5hbWVBY2Nlc3NBc3NvY2lhdGlvbiBvcGVyYXRpb25cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWxcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBEb21haW5OYW1lQWNjZXNzQXNzb2NpYXRpb25zXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW2FybiBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfQVJOKS4gKipEZWZhdWx0OioqIGBBcm5MaWtlYFxuICAgKi9cbiAgcHVibGljIGlmUmVxdWVzdERvbWFpbk5hbWVBcm4odmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFJlcXVlc3QvRG9tYWluTmFtZUFybmAsIHZhbHVlLCBvcGVyYXRvciA/PyAnQXJuTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IGVuZHBvaW50IHR5cGUuIEF2YWlsYWJsZSBkdXJpbmcgdGhlIENyZWF0ZURvbWFpbk5hbWUsIFVwZGF0ZURvbWFpbk5hbWUsIENyZWF0ZVJlc3RBcGksIGFuZCBVcGRhdGVSZXN0QXBpIG9wZXJhdGlvbnNcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWxcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBEb21haW5OYW1lXG4gICAqIC0gRG9tYWluTmFtZXNcbiAgICogLSBQcml2YXRlRG9tYWluTmFtZVxuICAgKiAtIFJlc3RBcGlcbiAgICogLSBSZXN0QXBpc1xuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlcXVlc3RFbmRwb2ludFR5cGUodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFJlcXVlc3QvRW5kcG9pbnRUeXBlYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgVVJJIG9mIHRoZSB0cnVzdHN0b3JlIHVzZWQgZm9yIG11dHVhbCBUTFMgYXV0aGVudGljYXRpb24uIEF2YWlsYWJsZSBkdXJpbmcgdGhlIENyZWF0ZURvbWFpbk5hbWUgYW5kIFVwZGF0ZURvbWFpbk5hbWUgb3BlcmF0aW9uc1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIERvbWFpbk5hbWVcbiAgICogLSBEb21haW5OYW1lc1xuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlcXVlc3RNdGxzVHJ1c3RTdG9yZVVyaSh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgUmVxdWVzdC9NdGxzVHJ1c3RTdG9yZVVyaWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHZlcnNpb24gb2YgdGhlIHRydXN0c3RvcmUgdXNlZCBmb3IgbXV0dWFsIFRMUyBhdXRoZW50aWNhdGlvbi4gQXZhaWxhYmxlIGR1cmluZyB0aGUgQ3JlYXRlRG9tYWluTmFtZSBhbmQgVXBkYXRlRG9tYWluTmFtZSBvcGVyYXRpb25zXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gRG9tYWluTmFtZVxuICAgKiAtIERvbWFpbk5hbWVzXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmUmVxdWVzdE10bHNUcnVzdFN0b3JlVmVyc2lvbih2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgUmVxdWVzdC9NdGxzVHJ1c3RTdG9yZVZlcnNpb25gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBhdXRob3JpemF0aW9uIHR5cGUsIGZvciBleGFtcGxlIE5PTkUsIEFXU19JQU0sIENVU1RPTSwgSldULCBDT0dOSVRPX1VTRVJfUE9PTFMuIEF2YWlsYWJsZSBkdXJpbmcgdGhlIENyZWF0ZU1ldGhvZCBhbmQgUHV0TWV0aG9kIG9wZXJhdGlvbnMgQWxzbyBhdmFpbGFibGUgYXMgYSBjb2xsZWN0aW9uIGR1cmluZyBpbXBvcnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWxcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBNZXRob2RcbiAgICogLSBSZXN0QXBpXG4gICAqIC0gUmVzdEFwaXNcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZSZXF1ZXN0Um91dGVBdXRob3JpemF0aW9uVHlwZSh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgUmVxdWVzdC9Sb3V0ZUF1dGhvcml6YXRpb25UeXBlYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgcm91dGluZyBtb2RlIG9mIHRoZSBkb21haW4gbmFtZS4gQXZhaWxhYmxlIGR1cmluZyB0aGUgQ3JlYXRlRG9tYWluTmFtZSBhbmQgVXBkYXRlRG9tYWluTmFtZSBvcGVyYXRpb25zXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zZWN1cml0eV9pYW1faWQtYmFzZWQtcG9saWN5LWV4YW1wbGVzLmh0bWwjc2VjdXJpdHlfaWFtX2lkLWJhc2VkLXBvbGljeS1leGFtcGxlcy1yb3V0aW5nLW1vZGVcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZSZXF1ZXN0Um91dGluZ01vZGUodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFJlcXVlc3QvUm91dGluZ01vZGVgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBUTFMgdmVyc2lvbi4gQXZhaWxhYmxlIGR1cmluZyB0aGUgQ3JlYXRlRG9tYWluIGFuZCBVcGRhdGVEb21haW4gb3BlcmF0aW9uc1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIERvbWFpbk5hbWVcbiAgICogLSBEb21haW5OYW1lc1xuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlcXVlc3RTZWN1cml0eVBvbGljeSh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgUmVxdWVzdC9TZWN1cml0eVBvbGljeWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHN0YWdlIG5hbWUgb2YgdGhlIGRlcGxveW1lbnQgdGhhdCB5b3UgYXR0ZW1wdCB0byBjcmVhdGUuIEF2YWlsYWJsZSBkdXJpbmcgdGhlIENyZWF0ZURlcGxveW1lbnQgb3BlcmF0aW9uXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gRGVwbG95bWVudHNcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZSZXF1ZXN0U3RhZ2VOYW1lKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBSZXF1ZXN0L1N0YWdlTmFtZWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IGFjY2VzcyBsb2cgZGVzdGluYXRpb24gb2YgdGhlIGN1cnJlbnQgU3RhZ2UgcmVzb3VyY2UuIEF2YWlsYWJsZSBkdXJpbmcgdGhlIFVwZGF0ZVN0YWdlIGFuZCBEZWxldGVTdGFnZSBvcGVyYXRpb25zXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gU3RhZ2VcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZSZXNvdXJjZUFjY2Vzc0xvZ2dpbmdEZXN0aW5hdGlvbih2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgUmVzb3VyY2UvQWNjZXNzTG9nZ2luZ0Rlc3RpbmF0aW9uYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgYWNjZXNzIGxvZyBmb3JtYXQgb2YgdGhlIGN1cnJlbnQgU3RhZ2UgcmVzb3VyY2UuIEF2YWlsYWJsZSBkdXJpbmcgdGhlIFVwZGF0ZVN0YWdlIGFuZCBEZWxldGVTdGFnZSBvcGVyYXRpb25zXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gU3RhZ2VcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZSZXNvdXJjZUFjY2Vzc0xvZ2dpbmdGb3JtYXQodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFJlc291cmNlL0FjY2Vzc0xvZ2dpbmdGb3JtYXRgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB3aGV0aGVyIGFuIEFQSSBrZXkgaXMgcmVxdWlyZWQgb3Igbm90IGZvciB0aGUgZXhpc3RpbmcgTWV0aG9kIHJlc291cmNlLiBBdmFpbGFibGUgZHVyaW5nIHRoZSBQdXRNZXRob2QgYW5kIERlbGV0ZU1ldGhvZCBvcGVyYXRpb25zLiBBbHNvIGF2YWlsYWJsZSBhcyBhIGNvbGxlY3Rpb24gZHVyaW5nIHJlaW1wb3J0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gTWV0aG9kXG4gICAqIC0gUmVzdEFwaVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgYHRydWVgIG9yIGBmYWxzZWAuICoqRGVmYXVsdDoqKiBgdHJ1ZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlc291cmNlQXBpS2V5UmVxdWlyZWQodmFsdWU/OiBib29sZWFuKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFJlc291cmNlL0FwaUtleVJlcXVpcmVkYCwgKHR5cGVvZiB2YWx1ZSAhPT0gJ3VuZGVmaW5lZCcgPyB2YWx1ZSA6IHRydWUpLCAnQm9vbCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IEFQSSBuYW1lIG9mIHRoZSBleGlzdGluZyBSZXN0QXBpIHJlc291cmNlLiBBdmFpbGFibGUgZHVyaW5nIFVwZGF0ZVJlc3RBcGkgYW5kIERlbGV0ZVJlc3RBcGkgb3BlcmF0aW9uc1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIFJlc3RBcGlcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZSZXNvdXJjZUFwaU5hbWUodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFJlc291cmNlL0FwaU5hbWVgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgY3VycmVudCB0eXBlIG9mIGF1dGhvcml6ZXIsIGZvciBleGFtcGxlIFRPS0VOLCBSRVFVRVNULCBKV1QuIEF2YWlsYWJsZSBkdXJpbmcgVXBkYXRlQXV0aG9yaXplciBhbmQgRGVsZXRlQXV0aG9yaXplciBvcGVyYXRpb25zLiBBbHNvIGF2YWlsYWJsZSBkdXJpbmcgcmVpbXBvcnQgYXMgYW4gQXJyYXlPZlN0cmluZ1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIEF1dGhvcml6ZXJcbiAgICogLSBSZXN0QXBpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmUmVzb3VyY2VBdXRob3JpemVyVHlwZSh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgUmVzb3VyY2UvQXV0aG9yaXplclR5cGVgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBVUkkgb2YgYSBMYW1iZGEgYXV0aG9yaXplciBmdW5jdGlvbi4gQXZhaWxhYmxlIGR1cmluZyBVcGRhdGVBdXRob3JpemVyIGFuZCBEZWxldGVBdXRob3JpemVyIG9wZXJhdGlvbnMuIEFsc28gYXZhaWxhYmxlIGR1cmluZyByZWltcG9ydCBhcyBhbiBBcnJheU9mU3RyaW5nXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gQXV0aG9yaXplclxuICAgKiAtIFJlc3RBcGlcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZSZXNvdXJjZUF1dGhvcml6ZXJVcmkodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFJlc291cmNlL0F1dGhvcml6ZXJVcmlgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBDb2duaXRvIHVzZXIgcG9vbCBwcm92aWRlciBBUk4uIEF2YWlsYWJsZSBkdXJpbmcgQ3JlYXRlQXV0aG9yaXplciBhbmQgVXBkYXRlQXV0aG9yaXplclxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIEF1dGhvcml6ZXJcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbYXJuIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19BUk4pLiAqKkRlZmF1bHQ6KiogYEFybkxpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZSZXNvdXJjZUNvZ25pdG9Vc2VyUG9vbFByb3ZpZGVyQXJuKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBSZXNvdXJjZS9Db2duaXRvVXNlclBvb2xQcm92aWRlckFybmAsIHZhbHVlLCBvcGVyYXRvciA/PyAnQXJuTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHN0YXR1cyBvZiB0aGUgZGVmYXVsdCBleGVjdXRlLWFwaSBlbmRwb2ludCBvZiB0aGUgY3VycmVudCBSZXN0QXBpIHJlc291cmNlLiBBdmFpbGFibGUgZHVyaW5nIFVwZGF0ZVJlc3RBcGkgYW5kIERlbGV0ZVJlc3RBcGkgb3BlcmF0aW9uc1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIFJlc3RBcGlcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIGB0cnVlYCBvciBgZmFsc2VgLiAqKkRlZmF1bHQ6KiogYHRydWVgXG4gICAqL1xuICBwdWJsaWMgaWZSZXNvdXJjZURpc2FibGVFeGVjdXRlQXBpRW5kcG9pbnQodmFsdWU/OiBib29sZWFuKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFJlc291cmNlL0Rpc2FibGVFeGVjdXRlQXBpRW5kcG9pbnRgLCAodHlwZW9mIHZhbHVlICE9PSAndW5kZWZpbmVkJyA/IHZhbHVlIDogdHJ1ZSksICdCb29sJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgZW5kcG9pbnQgdHlwZS4gQXZhaWxhYmxlIGR1cmluZyB0aGUgVXBkYXRlRG9tYWluTmFtZSwgRGVsZXRlRG9tYWluTmFtZSwgVXBkYXRlUmVzdEFwaSwgYW5kIERlbGV0ZVJlc3RBcGkgb3BlcmF0aW9uc1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIERvbWFpbk5hbWVcbiAgICogLSBQcml2YXRlRG9tYWluTmFtZVxuICAgKiAtIFJlc3RBcGlcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZSZXNvdXJjZUVuZHBvaW50VHlwZSh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgUmVzb3VyY2UvRW5kcG9pbnRUeXBlYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgVVJJIG9mIHRoZSB0cnVzdHN0b3JlIHVzZWQgZm9yIG11dHVhbCBUTFMgYXV0aGVudGljYXRpb24uIEF2YWlsYWJsZSBkdXJpbmcgVXBkYXRlRG9tYWluTmFtZSBhbmQgRGVsZXRlRG9tYWluTmFtZSBvcGVyYXRpb25zXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gRG9tYWluTmFtZVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlc291cmNlTXRsc1RydXN0U3RvcmVVcmkodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFJlc291cmNlL010bHNUcnVzdFN0b3JlVXJpYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdmVyc2lvbiBvZiB0aGUgdHJ1c3RzdG9yZSB1c2VkIGZvciBtdXR1YWwgVExTIGF1dGhlbnRpY2F0aW9uLiBBdmFpbGFibGUgZHVyaW5nIFVwZGF0ZURvbWFpbk5hbWUgYW5kIERlbGV0ZURvbWFpbk5hbWUgb3BlcmF0aW9uc1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIERvbWFpbk5hbWVcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZSZXNvdXJjZU10bHNUcnVzdFN0b3JlVmVyc2lvbih2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgUmVzb3VyY2UvTXRsc1RydXN0U3RvcmVWZXJzaW9uYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgYXV0aG9yaXphdGlvbiB0eXBlIG9mIHRoZSBleGlzdGluZyBNZXRob2QgcmVzb3VyY2UsIGZvciBleGFtcGxlIE5PTkUsIEFXU19JQU0sIENVU1RPTSwgSldULCBDT0dOSVRPX1VTRVJfUE9PTFMuIEF2YWlsYWJsZSBkdXJpbmcgdGhlIFB1dE1ldGhvZCBhbmQgRGVsZXRlTWV0aG9kIG9wZXJhdGlvbnMuIEFsc28gYXZhaWxhYmxlIGFzIGEgY29sbGVjdGlvbiBkdXJpbmcgcmVpbXBvcnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWxcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBNZXRob2RcbiAgICogLSBSZXN0QXBpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmUmVzb3VyY2VSb3V0ZUF1dGhvcml6YXRpb25UeXBlKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBSZXNvdXJjZS9Sb3V0ZUF1dGhvcml6YXRpb25UeXBlYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgcm91dGluZyBtb2RlIG9mIHRoZSBkb21haW4gbmFtZS4gQXZhaWxhYmxlIGR1cmluZyB0aGUgVXBkYXRlRG9tYWluTmFtZSBhbmQgRGVsZXRlRG9tYWluTmFtZSBvcGVyYXRpb25zXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zZWN1cml0eV9pYW1faWQtYmFzZWQtcG9saWN5LWV4YW1wbGVzLmh0bWwjc2VjdXJpdHlfaWFtX2lkLWJhc2VkLXBvbGljeS1leGFtcGxlcy1yb3V0aW5nLW1vZGVcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBEb21haW5OYW1lXG4gICAqIC0gRG9tYWluTmFtZXNcbiAgICogLSBQcml2YXRlRG9tYWluTmFtZVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlc291cmNlUm91dGluZ01vZGUodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFJlc291cmNlL1JvdXRpbmdNb2RlYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgVExTIHZlcnNpb24uIEF2YWlsYWJsZSBkdXJpbmcgVXBkYXRlRG9tYWluIGFuZCBEZWxldGVEb21haW4gb3BlcmF0aW9uc1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIERvbWFpbk5hbWVcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZSZXNvdXJjZVNlY3VyaXR5UG9saWN5KHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBSZXNvdXJjZS9TZWN1cml0eVBvbGljeWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSB0YWcga2V5LXZhbHVlIHBhaXJzIGluIHRoZSByZXF1ZXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9hcGlnYXRld2F5LXRhZ2dpbmcuaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvREVMRVRFKClcbiAgICogLSAudG9QQVRDSCgpXG4gICAqIC0gLnRvUE9TVCgpXG4gICAqIC0gLnRvUFVUKClcbiAgICpcbiAgICogQHBhcmFtIHRhZ0tleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1JlcXVlc3RUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6UmVxdWVzdFRhZy8keyB0YWdLZXkgfWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSB0YWdzIGF0dGFjaGVkIHRvIHRoZSByZXNvdXJjZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvYXBpZ2F0ZXdheS10YWdnaW5nLmh0bWxcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBBcGlLZXlcbiAgICogLSBBcGlLZXlzXG4gICAqIC0gQXV0aG9yaXplclxuICAgKiAtIEF1dGhvcml6ZXJzXG4gICAqIC0gQmFzZVBhdGhNYXBwaW5nXG4gICAqIC0gQmFzZVBhdGhNYXBwaW5nc1xuICAgKiAtIENsaWVudENlcnRpZmljYXRlXG4gICAqIC0gQ2xpZW50Q2VydGlmaWNhdGVzXG4gICAqIC0gRGVwbG95bWVudFxuICAgKiAtIERlcGxveW1lbnRzXG4gICAqIC0gRG9jdW1lbnRhdGlvblBhcnRcbiAgICogLSBEb2N1bWVudGF0aW9uUGFydHNcbiAgICogLSBEb2N1bWVudGF0aW9uVmVyc2lvblxuICAgKiAtIERvY3VtZW50YXRpb25WZXJzaW9uc1xuICAgKiAtIERvbWFpbk5hbWVcbiAgICogLSBEb21haW5OYW1lc1xuICAgKiAtIERvbWFpbk5hbWVBY2Nlc3NBc3NvY2lhdGlvbnNcbiAgICogLSBHYXRld2F5UmVzcG9uc2VcbiAgICogLSBHYXRld2F5UmVzcG9uc2VzXG4gICAqIC0gSW50ZWdyYXRpb25cbiAgICogLSBJbnRlZ3JhdGlvblJlc3BvbnNlXG4gICAqIC0gTWV0aG9kXG4gICAqIC0gTWV0aG9kUmVzcG9uc2VcbiAgICogLSBNb2RlbFxuICAgKiAtIE1vZGVsc1xuICAgKiAtIFByaXZhdGVCYXNlUGF0aE1hcHBpbmdcbiAgICogLSBQcml2YXRlQmFzZVBhdGhNYXBwaW5nc1xuICAgKiAtIFByaXZhdGVEb21haW5OYW1lXG4gICAqIC0gUmVxdWVzdFZhbGlkYXRvclxuICAgKiAtIFJlcXVlc3RWYWxpZGF0b3JzXG4gICAqIC0gUmVzb3VyY2VcbiAgICogLSBSZXNvdXJjZXNcbiAgICogLSBSZXN0QXBpXG4gICAqIC0gUmVzdEFwaXNcbiAgICogLSBTZGtcbiAgICogLSBTdGFnZVxuICAgKiAtIFN0YWdlc1xuICAgKiAtIFRlbXBsYXRlXG4gICAqIC0gVXNhZ2VQbGFuXG4gICAqIC0gVXNhZ2VQbGFuc1xuICAgKiAtIFVzYWdlUGxhbktleVxuICAgKiAtIFVzYWdlUGxhbktleXNcbiAgICogLSBWcGNMaW5rXG4gICAqIC0gVnBjTGlua3NcbiAgICpcbiAgICogQHBhcmFtIHRhZ0tleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1Jlc291cmNlVGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlJlc291cmNlVGFnLyR7IHRhZ0tleSB9YCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIHRhZyBrZXlzIGluIHRoZSByZXF1ZXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9hcGlnYXRld2F5LXRhZ2dpbmcuaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvREVMRVRFKClcbiAgICogLSAudG9QQVRDSCgpXG4gICAqIC0gLnRvUE9TVCgpXG4gICAqIC0gLnRvUFVUKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NUYWdLZXlzKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6VGFnS2V5c2AsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbYXBpZ2F0ZXdheV0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYW1hem9uYXBpZ2F0ZXdheW1hbmFnZW1lbnQuaHRtbCkuXG4gICAqXG4gICAqL1xuICBjb25zdHJ1Y3Rvcihwcm9wcz86IGlhbS5Qb2xpY3lTdGF0ZW1lbnRQcm9wcykge1xuICAgIHN1cGVyKHByb3BzKTtcbiAgfVxufVxuIl19