"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Appfabric = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [appfabric](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsappfabric.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Appfabric extends shared_1.PolicyStatement {
    /**
     * Grants permission to start user access tasks for multiple users
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_BatchGetUserAccessTasks.html
     */
    toBatchGetUserAccessTasks() {
        return this.to('BatchGetUserAccessTasks');
    }
    /**
     * Grants permission to connect app authorizations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_ConnectAppAuthorization.html
     */
    toConnectAppAuthorization() {
        return this.to('ConnectAppAuthorization');
    }
    /**
     * Grants permission to create app authorizations for app bundles
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_CreateAppAuthorization.html
     */
    toCreateAppAuthorization() {
        return this.to('CreateAppAuthorization');
    }
    /**
     * Grants permission to create app bundles in your account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_CreateAppBundle.html
     */
    toCreateAppBundle() {
        return this.to('CreateAppBundle');
    }
    /**
     * Grants permission to create ingestions for app bundles
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_CreateIngestion.html
     */
    toCreateIngestion() {
        return this.to('CreateIngestion');
    }
    /**
     * Grants permission to create ingestion destinations for app bundles
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_CreateIngestionDestination.html
     */
    toCreateIngestionDestination() {
        return this.to('CreateIngestionDestination');
    }
    /**
     * Grants permission to delete app authorizations within an app bundle
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_DeleteAppAuthorization.html
     */
    toDeleteAppAuthorization() {
        return this.to('DeleteAppAuthorization');
    }
    /**
     * Grants permission to delete app bundles in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_DeleteAppBundle.html
     */
    toDeleteAppBundle() {
        return this.to('DeleteAppBundle');
    }
    /**
     * Grants permission to delete ingestions within an app bundle
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_DeleteIngestion.html
     */
    toDeleteIngestion() {
        return this.to('DeleteIngestion');
    }
    /**
     * Grants permission to delete destinations within an ingestion
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_DeleteIngestionDestination.html
     */
    toDeleteIngestionDestination() {
        return this.to('DeleteIngestionDestination');
    }
    /**
     * Grants permission to view details about app authorizations
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_GetAppAuthorization.html
     */
    toGetAppAuthorization() {
        return this.to('GetAppAuthorization');
    }
    /**
     * Grants permission to view details about app bundles
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_GetAppBundle.html
     */
    toGetAppBundle() {
        return this.to('GetAppBundle');
    }
    /**
     * Grants permission to view details about ingestions
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_GetIngestion.html
     */
    toGetIngestion() {
        return this.to('GetIngestion');
    }
    /**
     * Grants permission to view details about ingestion destinations
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_GetIngestionDestination.html
     */
    toGetIngestionDestination() {
        return this.to('GetIngestionDestination');
    }
    /**
     * Grants permission to retrieve a list of app authorizations within an app bundle
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_ListAppAuthorizations.html
     */
    toListAppAuthorizations() {
        return this.to('ListAppAuthorizations');
    }
    /**
     * Grants permission to retrieve a list of app bundles in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_ListAppBundles.html
     */
    toListAppBundles() {
        return this.to('ListAppBundles');
    }
    /**
     * Grants permission to retrieve a list of destinations within an ingestion
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_ListIngestionDestinations.html
     */
    toListIngestionDestinations() {
        return this.to('ListIngestionDestinations');
    }
    /**
     * Grants permission to retrieve a list of ingestions within an app bundle
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_ListIngestions.html
     */
    toListIngestions() {
        return this.to('ListIngestions');
    }
    /**
     * Grants permission to list tags for AppFabric resources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to start ingestions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_StartIngestion.html
     */
    toStartIngestion() {
        return this.to('StartIngestion');
    }
    /**
     * Grants permission to start user access tasks
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_StartUserAccessTasks.html
     */
    toStartUserAccessTasks() {
        return this.to('StartUserAccessTasks');
    }
    /**
     * Grants permission to stop ingestions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_StopIngestion.html
     */
    toStopIngestion() {
        return this.to('StopIngestion');
    }
    /**
     * Grants permission to tag AppFabric resources
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag AppFabric resources
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update app authorizations within app bundles
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_UpdateAppAuthorization.html
     */
    toUpdateAppAuthorization() {
        return this.to('UpdateAppAuthorization');
    }
    /**
     * Grants permission to update destinations within ingestions
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_UpdateIngestionDestination.html
     */
    toUpdateIngestionDestination() {
        return this.to('UpdateIngestionDestination');
    }
    /**
     * Adds a resource of type appbundle to the statement
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_AppBundle.html
     *
     * @param appBundleIdentifier - Identifier for the appBundleIdentifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAppbundle(appBundleIdentifier, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appfabric:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:appbundle/${appBundleIdentifier}`);
    }
    /**
     * Adds a resource of type appauthorization to the statement
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_AppAuthorization.html
     *
     * @param appbundleId - Identifier for the appbundleId.
     * @param appAuthorizationIdentifier - Identifier for the appAuthorizationIdentifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAppauthorization(appbundleId, appAuthorizationIdentifier, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appfabric:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:appbundle/${appbundleId}/appauthorization/${appAuthorizationIdentifier}`);
    }
    /**
     * Adds a resource of type ingestion to the statement
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_Ingestion.html
     *
     * @param appbundleId - Identifier for the appbundleId.
     * @param ingestionIdentifier - Identifier for the ingestionIdentifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIngestion(appbundleId, ingestionIdentifier, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appfabric:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:appbundle/${appbundleId}/ingestion/${ingestionIdentifier}`);
    }
    /**
     * Adds a resource of type ingestiondestination to the statement
     *
     * https://docs.aws.amazon.com/appfabric/latest/api/API_IngestionDestination.html
     *
     * @param appbundleId - Identifier for the appbundleId.
     * @param ingestionIdentifier - Identifier for the ingestionIdentifier.
     * @param ingestionDestinationIdentifier - Identifier for the ingestionDestinationIdentifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIngestiondestination(appbundleId, ingestionIdentifier, ingestionDestinationIdentifier, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appfabric:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:appbundle/${appbundleId}/ingestion/${ingestionIdentifier}/ingestiondestination/${ingestionDestinationIdentifier}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAppAuthorization()
     * - .toCreateAppBundle()
     * - .toCreateIngestion()
     * - .toCreateIngestionDestination()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toGetAppAuthorization()
     * - .toGetAppBundle()
     * - .toGetIngestion()
     * - .toGetIngestionDestination()
     * - .toTagResource()
     * - .toUpdateAppAuthorization()
     * - .toUpdateIngestionDestination()
     *
     * Applies to resource types:
     * - appbundle
     * - appauthorization
     * - ingestion
     * - ingestiondestination
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAppAuthorization()
     * - .toCreateAppBundle()
     * - .toCreateIngestion()
     * - .toCreateIngestionDestination()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [appfabric](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsappfabric.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'appfabric';
        this.accessLevelList = {
            Write: [
                'BatchGetUserAccessTasks',
                'ConnectAppAuthorization',
                'CreateAppAuthorization',
                'CreateAppBundle',
                'CreateIngestion',
                'CreateIngestionDestination',
                'DeleteAppAuthorization',
                'DeleteAppBundle',
                'DeleteIngestion',
                'DeleteIngestionDestination',
                'StartIngestion',
                'StartUserAccessTasks',
                'StopIngestion',
                'UpdateAppAuthorization',
                'UpdateIngestionDestination'
            ],
            Read: [
                'GetAppAuthorization',
                'GetAppBundle',
                'GetIngestion',
                'GetIngestionDestination',
                'ListTagsForResource'
            ],
            List: [
                'ListAppAuthorizations',
                'ListAppBundles',
                'ListIngestionDestinations',
                'ListIngestions'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Appfabric = Appfabric;
//# sourceMappingURL=data:application/json;base64,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