"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ArcZonalShift = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [arc-zonal-shift](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonapplicationrecoverycontroller-zonalshift.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class ArcZonalShift extends shared_1.PolicyStatement {
    /**
     * Grants permission to cancel an active practice run
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceIdentifier()
     * - .ifAwsResourceTag()
     * - .ifElasticloadbalancingResourceTag()
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_CancelPracticeRun.html
     */
    toCancelPracticeRun() {
        return this.to('CancelPracticeRun');
    }
    /**
     * Grants permission to cancel an active zonal shift
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceIdentifier()
     * - .ifAwsResourceTag()
     * - .ifElasticloadbalancingResourceTag()
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_CancelZonalShift.html
     */
    toCancelZonalShift() {
        return this.to('CancelZonalShift');
    }
    /**
     * Grants permission to create a practice run configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceIdentifier()
     * - .ifAwsResourceTag()
     * - .ifElasticloadbalancingResourceTag()
     *
     * Dependent actions:
     * - cloudwatch:DescribeAlarms
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_CreatePracticeRunConfiguration.html
     */
    toCreatePracticeRunConfiguration() {
        return this.to('CreatePracticeRunConfiguration');
    }
    /**
     * Grants permission to delete a practice run configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceIdentifier()
     * - .ifAwsResourceTag()
     * - .ifElasticloadbalancingResourceTag()
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_DeletePracticeRunConfiguration.html
     */
    toDeletePracticeRunConfiguration() {
        return this.to('DeletePracticeRunConfiguration');
    }
    /**
     * Grants permission to get autoshift observer notification status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_GetAutoshiftObserverNotificationStatus.html
     */
    toGetAutoshiftObserverNotificationStatus() {
        return this.to('GetAutoshiftObserverNotificationStatus');
    }
    /**
     * Grants permission to get information about a managed resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifResourceIdentifier()
     * - .ifAwsResourceTag()
     * - .ifElasticloadbalancingResourceTag()
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_GetManagedResource.html
     */
    toGetManagedResource() {
        return this.to('GetManagedResource');
    }
    /**
     * Grants permission to list active and completed autoshifts
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_ListAutoshifts.html
     */
    toListAutoshifts() {
        return this.to('ListAutoshifts');
    }
    /**
     * Grants permission to list managed resources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_ListManagedResources.html
     */
    toListManagedResources() {
        return this.to('ListManagedResources');
    }
    /**
     * Grants permission to list zonal shifts
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_ListZonalShifts.html
     */
    toListZonalShifts() {
        return this.to('ListZonalShifts');
    }
    /**
     * Grants permission to start a practice run
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceIdentifier()
     * - .ifAwsResourceTag()
     * - .ifElasticloadbalancingResourceTag()
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_StartPracticeRun.html
     */
    toStartPracticeRun() {
        return this.to('StartPracticeRun');
    }
    /**
     * Grants permission to start a zonal shift
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceIdentifier()
     * - .ifAwsResourceTag()
     * - .ifElasticloadbalancingResourceTag()
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_StartZonalShift.html
     */
    toStartZonalShift() {
        return this.to('StartZonalShift');
    }
    /**
     * Grants permission to update autoshift observer notification status
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_UpdateAutoshiftObserverNotificationStatus.html
     */
    toUpdateAutoshiftObserverNotificationStatus() {
        return this.to('UpdateAutoshiftObserverNotificationStatus');
    }
    /**
     * Grants permission to update a practice run configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceIdentifier()
     * - .ifAwsResourceTag()
     * - .ifElasticloadbalancingResourceTag()
     *
     * Dependent actions:
     * - cloudwatch:DescribeAlarms
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_UpdatePracticeRunConfiguration.html
     */
    toUpdatePracticeRunConfiguration() {
        return this.to('UpdatePracticeRunConfiguration');
    }
    /**
     * Grants permission to update a zonal autoshift status
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceIdentifier()
     * - .ifAwsResourceTag()
     * - .ifElasticloadbalancingResourceTag()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_UpdateZonalAutoshiftConfiguration.html
     */
    toUpdateZonalAutoshiftConfiguration() {
        return this.to('UpdateZonalAutoshiftConfiguration');
    }
    /**
     * Grants permission to update an existing zonal shift
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceIdentifier()
     * - .ifAwsResourceTag()
     * - .ifElasticloadbalancingResourceTag()
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_UpdateZonalShift.html
     */
    toUpdateZonalShift() {
        return this.to('UpdateZonalShift');
    }
    /**
     * Adds a resource of type ALB to the statement
     *
     * https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.resource-types.html
     *
     * @param loadBalancerName - Identifier for the loadBalancerName.
     * @param loadBalancerId - Identifier for the loadBalancerId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifResourceIdentifier()
     * - .ifAwsResourceTag()
     * - .ifElasticloadbalancingResourceTag()
     */
    onALB(loadBalancerName, loadBalancerId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticloadbalancing:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:loadbalancer/app/${loadBalancerName}/${loadBalancerId}`);
    }
    /**
     * Adds a resource of type NLB to the statement
     *
     * https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.resource-types.html
     *
     * @param loadBalancerName - Identifier for the loadBalancerName.
     * @param loadBalancerId - Identifier for the loadBalancerId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifResourceIdentifier()
     * - .ifAwsResourceTag()
     * - .ifElasticloadbalancingResourceTag()
     */
    onNLB(loadBalancerName, loadBalancerId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticloadbalancing:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:loadbalancer/net/${loadBalancerName}/${loadBalancerId}`);
    }
    /**
     * Filters access by the resource identifier of the managed resource
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonroute53applicationrecoverycontroller-zonalshift.html#amazonroute53applicationrecoverycontroller-zonalshift-policy-keys
     *
     * Applies to actions:
     * - .toCancelPracticeRun()
     * - .toCancelZonalShift()
     * - .toCreatePracticeRunConfiguration()
     * - .toDeletePracticeRunConfiguration()
     * - .toGetManagedResource()
     * - .toStartPracticeRun()
     * - .toStartZonalShift()
     * - .toUpdatePracticeRunConfiguration()
     * - .toUpdateZonalAutoshiftConfiguration()
     * - .toUpdateZonalShift()
     *
     * Applies to resource types:
     * - ALB
     * - NLB
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceIdentifier(value, operator) {
        return this.if(`ResourceIdentifier`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the managed resource
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/userguide/load-balancer-authentication-access-control.html#elb-condition-keys
     *
     * Applies to actions:
     * - .toCancelPracticeRun()
     * - .toCancelZonalShift()
     * - .toCreatePracticeRunConfiguration()
     * - .toDeletePracticeRunConfiguration()
     * - .toGetManagedResource()
     * - .toStartPracticeRun()
     * - .toStartZonalShift()
     * - .toUpdatePracticeRunConfiguration()
     * - .toUpdateZonalAutoshiftConfiguration()
     * - .toUpdateZonalShift()
     *
     * Applies to resource types:
     * - ALB
     * - NLB
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the managed resource
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/userguide/load-balancer-authentication-access-control.html#elb-condition-keys
     *
     * Applies to actions:
     * - .toCancelPracticeRun()
     * - .toCancelZonalShift()
     * - .toCreatePracticeRunConfiguration()
     * - .toDeletePracticeRunConfiguration()
     * - .toGetManagedResource()
     * - .toStartPracticeRun()
     * - .toStartZonalShift()
     * - .toUpdatePracticeRunConfiguration()
     * - .toUpdateZonalAutoshiftConfiguration()
     * - .toUpdateZonalShift()
     *
     * Applies to resource types:
     * - ALB
     * - NLB
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifElasticloadbalancingResourceTag(tagKey, value, operator) {
        return this.if(`elasticloadbalancing:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [arc-zonal-shift](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonapplicationrecoverycontroller-zonalshift.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'arc-zonal-shift';
        this.accessLevelList = {
            Write: [
                'CancelPracticeRun',
                'CancelZonalShift',
                'CreatePracticeRunConfiguration',
                'DeletePracticeRunConfiguration',
                'StartPracticeRun',
                'StartZonalShift',
                'UpdateAutoshiftObserverNotificationStatus',
                'UpdatePracticeRunConfiguration',
                'UpdateZonalAutoshiftConfiguration',
                'UpdateZonalShift'
            ],
            Read: [
                'GetAutoshiftObserverNotificationStatus',
                'GetManagedResource'
            ],
            List: [
                'ListAutoshifts',
                'ListManagedResources',
                'ListZonalShifts'
            ]
        };
    }
}
exports.ArcZonalShift = ArcZonalShift;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBwbGljYXRpb25yZWNvdmVyeWNvbnRyb2xsZXItem9uYWxzaGlmdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImFwcGxpY2F0aW9ucmVjb3Zlcnljb250cm9sbGVyLXpvbmFsc2hpZnQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0EseUNBQXlEO0FBR3pEOzs7O0dBSUc7QUFDSCxNQUFhLGFBQWMsU0FBUSx3QkFBZTtJQUdoRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7T0FlRztJQUNJLGdDQUFnQztRQUNyQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztJQUNuRCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxnQ0FBZ0M7UUFDckMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdDQUFnQyxDQUFDLENBQUM7SUFDbkQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdDQUF3QztRQUM3QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0NBQXdDLENBQUMsQ0FBQztJQUMzRCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxvQkFBb0I7UUFDekIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLGlCQUFpQjtRQUN0QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMkNBQTJDO1FBQ2hELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywyQ0FBMkMsQ0FBQyxDQUFDO0lBQzlELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7O09BZUc7SUFDSSxnQ0FBZ0M7UUFDckMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdDQUFnQyxDQUFDLENBQUM7SUFDbkQsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7OztPQWNHO0lBQ0ksbUNBQW1DO1FBQ3hDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQ0FBbUMsQ0FBQyxDQUFDO0lBQ3RELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLGtCQUFrQjtRQUN2QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQWtCLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBMEJEOzs7Ozs7Ozs7Ozs7Ozs7T0FlRztJQUNJLEtBQUssQ0FBQyxnQkFBd0IsRUFBRSxjQUFzQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ2xILE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLHlCQUEwQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUscUJBQXNCLGdCQUFpQixJQUFLLGNBQWUsRUFBRSxDQUFDLENBQUM7SUFDdE4sQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7T0FlRztJQUNJLEtBQUssQ0FBQyxnQkFBd0IsRUFBRSxjQUFzQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ2xILE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLHlCQUEwQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUscUJBQXNCLGdCQUFpQixJQUFLLGNBQWUsRUFBRSxDQUFDLENBQUM7SUFDdE4sQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQXVCRztJQUNJLG9CQUFvQixDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDaEYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9CQUFvQixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDeEUsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0F3Qkc7SUFDSSxnQkFBZ0IsQ0FBQyxNQUFjLEVBQUUsS0FBd0IsRUFBRSxRQUE0QjtRQUM1RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW9CLE1BQU8sRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDakYsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0F3Qkc7SUFDSSxpQ0FBaUMsQ0FBQyxNQUFjLEVBQUUsS0FBd0IsRUFBRSxRQUE0QjtRQUM3RyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0NBQXFDLE1BQU8sRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDbEcsQ0FBQztJQUVEOzs7T0FHRztJQUNILFlBQVksS0FBZ0M7UUFDMUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBL1hSLGtCQUFhLEdBQUcsaUJBQWlCLENBQUM7UUFvTy9CLG9CQUFlLEdBQW9CO1lBQzNDLEtBQUssRUFBRTtnQkFDTCxtQkFBbUI7Z0JBQ25CLGtCQUFrQjtnQkFDbEIsZ0NBQWdDO2dCQUNoQyxnQ0FBZ0M7Z0JBQ2hDLGtCQUFrQjtnQkFDbEIsaUJBQWlCO2dCQUNqQiwyQ0FBMkM7Z0JBQzNDLGdDQUFnQztnQkFDaEMsbUNBQW1DO2dCQUNuQyxrQkFBa0I7YUFDbkI7WUFDRCxJQUFJLEVBQUU7Z0JBQ0osd0NBQXdDO2dCQUN4QyxvQkFBb0I7YUFDckI7WUFDRCxJQUFJLEVBQUU7Z0JBQ0osZ0JBQWdCO2dCQUNoQixzQkFBc0I7Z0JBQ3RCLGlCQUFpQjthQUNsQjtTQUNGLENBQUM7SUFzSUYsQ0FBQztDQUNGO0FBbFlELHNDQWtZQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEFjY2Vzc0xldmVsTGlzdCB9IGZyb20gJy4uLy4uL3NoYXJlZC9hY2Nlc3MtbGV2ZWwnO1xuaW1wb3J0IHsgUG9saWN5U3RhdGVtZW50LCBPcGVyYXRvciB9IGZyb20gJy4uLy4uL3NoYXJlZCc7XG5pbXBvcnQgeyBhd3NfaWFtIGFzIGlhbSB9IGZyb20gXCJhd3MtY2RrLWxpYlwiO1xuXG4vKipcbiAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbYXJjLXpvbmFsLXNoaWZ0XShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hbWF6b25hcHBsaWNhdGlvbnJlY292ZXJ5Y29udHJvbGxlci16b25hbHNoaWZ0Lmh0bWwpLlxuICpcbiAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICovXG5leHBvcnQgY2xhc3MgQXJjWm9uYWxTaGlmdCBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ2FyYy16b25hbC1zaGlmdCc7XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNhbmNlbCBhbiBhY3RpdmUgcHJhY3RpY2UgcnVuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZSZXNvdXJjZUlkZW50aWZpZXIoKVxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICogLSAuaWZFbGFzdGljbG9hZGJhbGFuY2luZ1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FyYy16b25hbC1zaGlmdC9sYXRlc3QvYXBpL0FQSV9DYW5jZWxQcmFjdGljZVJ1bi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DYW5jZWxQcmFjdGljZVJ1bigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ2FuY2VsUHJhY3RpY2VSdW4nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjYW5jZWwgYW4gYWN0aXZlIHpvbmFsIHNoaWZ0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZSZXNvdXJjZUlkZW50aWZpZXIoKVxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICogLSAuaWZFbGFzdGljbG9hZGJhbGFuY2luZ1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FyYy16b25hbC1zaGlmdC9sYXRlc3QvYXBpL0FQSV9DYW5jZWxab25hbFNoaWZ0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NhbmNlbFpvbmFsU2hpZnQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NhbmNlbFpvbmFsU2hpZnQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBwcmFjdGljZSBydW4gY29uZmlndXJhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmUmVzb3VyY2VJZGVudGlmaWVyKClcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqIC0gLmlmRWxhc3RpY2xvYWRiYWxhbmNpbmdSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGNsb3Vkd2F0Y2g6RGVzY3JpYmVBbGFybXNcbiAgICogLSBpYW06Q3JlYXRlU2VydmljZUxpbmtlZFJvbGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FyYy16b25hbC1zaGlmdC9sYXRlc3QvYXBpL0FQSV9DcmVhdGVQcmFjdGljZVJ1bkNvbmZpZ3VyYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlUHJhY3RpY2VSdW5Db25maWd1cmF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVQcmFjdGljZVJ1bkNvbmZpZ3VyYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBwcmFjdGljZSBydW4gY29uZmlndXJhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmUmVzb3VyY2VJZGVudGlmaWVyKClcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqIC0gLmlmRWxhc3RpY2xvYWRiYWxhbmNpbmdSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcmMtem9uYWwtc2hpZnQvbGF0ZXN0L2FwaS9BUElfRGVsZXRlUHJhY3RpY2VSdW5Db25maWd1cmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVByYWN0aWNlUnVuQ29uZmlndXJhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlUHJhY3RpY2VSdW5Db25maWd1cmF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGF1dG9zaGlmdCBvYnNlcnZlciBub3RpZmljYXRpb24gc3RhdHVzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXJjLXpvbmFsLXNoaWZ0L2xhdGVzdC9hcGkvQVBJX0dldEF1dG9zaGlmdE9ic2VydmVyTm90aWZpY2F0aW9uU3RhdHVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEF1dG9zaGlmdE9ic2VydmVyTm90aWZpY2F0aW9uU3RhdHVzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRBdXRvc2hpZnRPYnNlcnZlck5vdGlmaWNhdGlvblN0YXR1cycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBpbmZvcm1hdGlvbiBhYm91dCBhIG1hbmFnZWQgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmUmVzb3VyY2VJZGVudGlmaWVyKClcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqIC0gLmlmRWxhc3RpY2xvYWRiYWxhbmNpbmdSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcmMtem9uYWwtc2hpZnQvbGF0ZXN0L2FwaS9BUElfR2V0TWFuYWdlZFJlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldE1hbmFnZWRSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0TWFuYWdlZFJlc291cmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBhY3RpdmUgYW5kIGNvbXBsZXRlZCBhdXRvc2hpZnRzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXJjLXpvbmFsLXNoaWZ0L2xhdGVzdC9hcGkvQVBJX0xpc3RBdXRvc2hpZnRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RBdXRvc2hpZnRzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0QXV0b3NoaWZ0cycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgbWFuYWdlZCByZXNvdXJjZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcmMtem9uYWwtc2hpZnQvbGF0ZXN0L2FwaS9BUElfTGlzdE1hbmFnZWRSZXNvdXJjZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdE1hbmFnZWRSZXNvdXJjZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RNYW5hZ2VkUmVzb3VyY2VzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCB6b25hbCBzaGlmdHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcmMtem9uYWwtc2hpZnQvbGF0ZXN0L2FwaS9BUElfTGlzdFpvbmFsU2hpZnRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3Rab25hbFNoaWZ0cygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFpvbmFsU2hpZnRzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3RhcnQgYSBwcmFjdGljZSBydW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZlJlc291cmNlSWRlbnRpZmllcigpXG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKiAtIC5pZkVsYXN0aWNsb2FkYmFsYW5jaW5nUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXJjLXpvbmFsLXNoaWZ0L2xhdGVzdC9hcGkvQVBJX1N0YXJ0UHJhY3RpY2VSdW4uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU3RhcnRQcmFjdGljZVJ1bigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU3RhcnRQcmFjdGljZVJ1bicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHN0YXJ0IGEgem9uYWwgc2hpZnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZlJlc291cmNlSWRlbnRpZmllcigpXG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKiAtIC5pZkVsYXN0aWNsb2FkYmFsYW5jaW5nUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXJjLXpvbmFsLXNoaWZ0L2xhdGVzdC9hcGkvQVBJX1N0YXJ0Wm9uYWxTaGlmdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdGFydFpvbmFsU2hpZnQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1N0YXJ0Wm9uYWxTaGlmdCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhdXRvc2hpZnQgb2JzZXJ2ZXIgbm90aWZpY2F0aW9uIHN0YXR1c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcmMtem9uYWwtc2hpZnQvbGF0ZXN0L2FwaS9BUElfVXBkYXRlQXV0b3NoaWZ0T2JzZXJ2ZXJOb3RpZmljYXRpb25TdGF0dXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlQXV0b3NoaWZ0T2JzZXJ2ZXJOb3RpZmljYXRpb25TdGF0dXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUF1dG9zaGlmdE9ic2VydmVyTm90aWZpY2F0aW9uU3RhdHVzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGEgcHJhY3RpY2UgcnVuIGNvbmZpZ3VyYXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZlJlc291cmNlSWRlbnRpZmllcigpXG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKiAtIC5pZkVsYXN0aWNsb2FkYmFsYW5jaW5nUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBjbG91ZHdhdGNoOkRlc2NyaWJlQWxhcm1zXG4gICAqIC0gaWFtOkNyZWF0ZVNlcnZpY2VMaW5rZWRSb2xlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcmMtem9uYWwtc2hpZnQvbGF0ZXN0L2FwaS9BUElfVXBkYXRlUHJhY3RpY2VSdW5Db25maWd1cmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZVByYWN0aWNlUnVuQ29uZmlndXJhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlUHJhY3RpY2VSdW5Db25maWd1cmF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGEgem9uYWwgYXV0b3NoaWZ0IHN0YXR1c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmUmVzb3VyY2VJZGVudGlmaWVyKClcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqIC0gLmlmRWxhc3RpY2xvYWRiYWxhbmNpbmdSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGlhbTpDcmVhdGVTZXJ2aWNlTGlua2VkUm9sZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXJjLXpvbmFsLXNoaWZ0L2xhdGVzdC9hcGkvQVBJX1VwZGF0ZVpvbmFsQXV0b3NoaWZ0Q29uZmlndXJhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVab25hbEF1dG9zaGlmdENvbmZpZ3VyYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZVpvbmFsQXV0b3NoaWZ0Q29uZmlndXJhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhbiBleGlzdGluZyB6b25hbCBzaGlmdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmUmVzb3VyY2VJZGVudGlmaWVyKClcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqIC0gLmlmRWxhc3RpY2xvYWRiYWxhbmNpbmdSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcmMtem9uYWwtc2hpZnQvbGF0ZXN0L2FwaS9BUElfVXBkYXRlWm9uYWxTaGlmdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVab25hbFNoaWZ0KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVab25hbFNoaWZ0Jyk7XG4gIH1cblxuICBwcm90ZWN0ZWQgYWNjZXNzTGV2ZWxMaXN0OiBBY2Nlc3NMZXZlbExpc3QgPSB7XG4gICAgV3JpdGU6IFtcbiAgICAgICdDYW5jZWxQcmFjdGljZVJ1bicsXG4gICAgICAnQ2FuY2VsWm9uYWxTaGlmdCcsXG4gICAgICAnQ3JlYXRlUHJhY3RpY2VSdW5Db25maWd1cmF0aW9uJyxcbiAgICAgICdEZWxldGVQcmFjdGljZVJ1bkNvbmZpZ3VyYXRpb24nLFxuICAgICAgJ1N0YXJ0UHJhY3RpY2VSdW4nLFxuICAgICAgJ1N0YXJ0Wm9uYWxTaGlmdCcsXG4gICAgICAnVXBkYXRlQXV0b3NoaWZ0T2JzZXJ2ZXJOb3RpZmljYXRpb25TdGF0dXMnLFxuICAgICAgJ1VwZGF0ZVByYWN0aWNlUnVuQ29uZmlndXJhdGlvbicsXG4gICAgICAnVXBkYXRlWm9uYWxBdXRvc2hpZnRDb25maWd1cmF0aW9uJyxcbiAgICAgICdVcGRhdGVab25hbFNoaWZ0J1xuICAgIF0sXG4gICAgUmVhZDogW1xuICAgICAgJ0dldEF1dG9zaGlmdE9ic2VydmVyTm90aWZpY2F0aW9uU3RhdHVzJyxcbiAgICAgICdHZXRNYW5hZ2VkUmVzb3VyY2UnXG4gICAgXSxcbiAgICBMaXN0OiBbXG4gICAgICAnTGlzdEF1dG9zaGlmdHMnLFxuICAgICAgJ0xpc3RNYW5hZ2VkUmVzb3VyY2VzJyxcbiAgICAgICdMaXN0Wm9uYWxTaGlmdHMnXG4gICAgXVxuICB9O1xuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBBTEIgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcjUzcmVjb3ZlcnkvbGF0ZXN0L2RnL2FyYy16b25hbC1zaGlmdC5yZXNvdXJjZS10eXBlcy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBsb2FkQmFsYW5jZXJOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGxvYWRCYWxhbmNlck5hbWUuXG4gICAqIEBwYXJhbSBsb2FkQmFsYW5jZXJJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBsb2FkQmFsYW5jZXJJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmUmVzb3VyY2VJZGVudGlmaWVyKClcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqIC0gLmlmRWxhc3RpY2xvYWRiYWxhbmNpbmdSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25BTEIobG9hZEJhbGFuY2VyTmFtZTogc3RyaW5nLCBsb2FkQmFsYW5jZXJJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmVsYXN0aWNsb2FkYmFsYW5jaW5nOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpsb2FkYmFsYW5jZXIvYXBwLyR7IGxvYWRCYWxhbmNlck5hbWUgfS8keyBsb2FkQmFsYW5jZXJJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgTkxCIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3I1M3JlY292ZXJ5L2xhdGVzdC9kZy9hcmMtem9uYWwtc2hpZnQucmVzb3VyY2UtdHlwZXMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gbG9hZEJhbGFuY2VyTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBsb2FkQmFsYW5jZXJOYW1lLlxuICAgKiBAcGFyYW0gbG9hZEJhbGFuY2VySWQgLSBJZGVudGlmaWVyIGZvciB0aGUgbG9hZEJhbGFuY2VySWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZlJlc291cmNlSWRlbnRpZmllcigpXG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKiAtIC5pZkVsYXN0aWNsb2FkYmFsYW5jaW5nUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uTkxCKGxvYWRCYWxhbmNlck5hbWU6IHN0cmluZywgbG9hZEJhbGFuY2VySWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTplbGFzdGljbG9hZGJhbGFuY2luZzokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06bG9hZGJhbGFuY2VyL25ldC8keyBsb2FkQmFsYW5jZXJOYW1lIH0vJHsgbG9hZEJhbGFuY2VySWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSByZXNvdXJjZSBpZGVudGlmaWVyIG9mIHRoZSBtYW5hZ2VkIHJlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlLWF1dGhvcml6YXRpb24vbGF0ZXN0L3JlZmVyZW5jZS9saXN0X2FtYXpvbnJvdXRlNTNhcHBsaWNhdGlvbnJlY292ZXJ5Y29udHJvbGxlci16b25hbHNoaWZ0Lmh0bWwjYW1hem9ucm91dGU1M2FwcGxpY2F0aW9ucmVjb3Zlcnljb250cm9sbGVyLXpvbmFsc2hpZnQtcG9saWN5LWtleXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NhbmNlbFByYWN0aWNlUnVuKClcbiAgICogLSAudG9DYW5jZWxab25hbFNoaWZ0KClcbiAgICogLSAudG9DcmVhdGVQcmFjdGljZVJ1bkNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b0RlbGV0ZVByYWN0aWNlUnVuQ29uZmlndXJhdGlvbigpXG4gICAqIC0gLnRvR2V0TWFuYWdlZFJlc291cmNlKClcbiAgICogLSAudG9TdGFydFByYWN0aWNlUnVuKClcbiAgICogLSAudG9TdGFydFpvbmFsU2hpZnQoKVxuICAgKiAtIC50b1VwZGF0ZVByYWN0aWNlUnVuQ29uZmlndXJhdGlvbigpXG4gICAqIC0gLnRvVXBkYXRlWm9uYWxBdXRvc2hpZnRDb25maWd1cmF0aW9uKClcbiAgICogLSAudG9VcGRhdGVab25hbFNoaWZ0KClcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBBTEJcbiAgICogLSBOTEJcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZSZXNvdXJjZUlkZW50aWZpZXIodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFJlc291cmNlSWRlbnRpZmllcmAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSB0YWdzIGFzc29jaWF0ZWQgd2l0aCB0aGUgbWFuYWdlZCByZXNvdXJjZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2xvYWRiYWxhbmNpbmcvbGF0ZXN0L3VzZXJndWlkZS9sb2FkLWJhbGFuY2VyLWF1dGhlbnRpY2F0aW9uLWFjY2Vzcy1jb250cm9sLmh0bWwjZWxiLWNvbmRpdGlvbi1rZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9DYW5jZWxQcmFjdGljZVJ1bigpXG4gICAqIC0gLnRvQ2FuY2VsWm9uYWxTaGlmdCgpXG4gICAqIC0gLnRvQ3JlYXRlUHJhY3RpY2VSdW5Db25maWd1cmF0aW9uKClcbiAgICogLSAudG9EZWxldGVQcmFjdGljZVJ1bkNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b0dldE1hbmFnZWRSZXNvdXJjZSgpXG4gICAqIC0gLnRvU3RhcnRQcmFjdGljZVJ1bigpXG4gICAqIC0gLnRvU3RhcnRab25hbFNoaWZ0KClcbiAgICogLSAudG9VcGRhdGVQcmFjdGljZVJ1bkNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b1VwZGF0ZVpvbmFsQXV0b3NoaWZ0Q29uZmlndXJhdGlvbigpXG4gICAqIC0gLnRvVXBkYXRlWm9uYWxTaGlmdCgpXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gQUxCXG4gICAqIC0gTkxCXG4gICAqXG4gICAqIEBwYXJhbSB0YWdLZXkgVGhlIHRhZyBrZXkgdG8gY2hlY2tcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NSZXNvdXJjZVRhZyh0YWdLZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpSZXNvdXJjZVRhZy8keyB0YWdLZXkgfWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSB0YWdzIGFzc29jaWF0ZWQgd2l0aCB0aGUgbWFuYWdlZCByZXNvdXJjZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2xvYWRiYWxhbmNpbmcvbGF0ZXN0L3VzZXJndWlkZS9sb2FkLWJhbGFuY2VyLWF1dGhlbnRpY2F0aW9uLWFjY2Vzcy1jb250cm9sLmh0bWwjZWxiLWNvbmRpdGlvbi1rZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9DYW5jZWxQcmFjdGljZVJ1bigpXG4gICAqIC0gLnRvQ2FuY2VsWm9uYWxTaGlmdCgpXG4gICAqIC0gLnRvQ3JlYXRlUHJhY3RpY2VSdW5Db25maWd1cmF0aW9uKClcbiAgICogLSAudG9EZWxldGVQcmFjdGljZVJ1bkNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b0dldE1hbmFnZWRSZXNvdXJjZSgpXG4gICAqIC0gLnRvU3RhcnRQcmFjdGljZVJ1bigpXG4gICAqIC0gLnRvU3RhcnRab25hbFNoaWZ0KClcbiAgICogLSAudG9VcGRhdGVQcmFjdGljZVJ1bkNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b1VwZGF0ZVpvbmFsQXV0b3NoaWZ0Q29uZmlndXJhdGlvbigpXG4gICAqIC0gLnRvVXBkYXRlWm9uYWxTaGlmdCgpXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gQUxCXG4gICAqIC0gTkxCXG4gICAqXG4gICAqIEBwYXJhbSB0YWdLZXkgVGhlIHRhZyBrZXkgdG8gY2hlY2tcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZFbGFzdGljbG9hZGJhbGFuY2luZ1Jlc291cmNlVGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgZWxhc3RpY2xvYWRiYWxhbmNpbmc6UmVzb3VyY2VUYWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW2FyYy16b25hbC1zaGlmdF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYW1hem9uYXBwbGljYXRpb25yZWNvdmVyeWNvbnRyb2xsZXItem9uYWxzaGlmdC5odG1sKS5cbiAgICpcbiAgICovXG4gIGNvbnN0cnVjdG9yKHByb3BzPzogaWFtLlBvbGljeVN0YXRlbWVudFByb3BzKSB7XG4gICAgc3VwZXIocHJvcHMpO1xuICB9XG59XG4iXX0=