"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Appstream = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [appstream](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonappstream2.0.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Appstream extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate the specified app block builder with the app block
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_AssociateAppBlockBuilderAppBlock.html
     */
    toAssociateAppBlockBuilderAppBlock() {
        return this.to('AssociateAppBlockBuilderAppBlock');
    }
    /**
     * Grants permission to associate the specified application with the fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_AssociateApplicationFleet.html
     */
    toAssociateApplicationFleet() {
        return this.to('AssociateApplicationFleet');
    }
    /**
     * Grants permission to associate the specified application to the specified entitlement
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_AssociateApplicationToEntitlement.html
     */
    toAssociateApplicationToEntitlement() {
        return this.to('AssociateApplicationToEntitlement');
    }
    /**
     * Grants permission to associate the specified fleet with the specified stack
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_AssociateFleet.html
     */
    toAssociateFleet() {
        return this.to('AssociateFleet');
    }
    /**
     * Grants permission to associate the specified users with the specified stacks. Users in a user pool cannot be assigned to stacks with fleets that are joined to an Active Directory domain
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_BatchAssociateUserStack.html
     */
    toBatchAssociateUserStack() {
        return this.to('BatchAssociateUserStack');
    }
    /**
     * Grants permission to disassociate the specified users from the specified stacks
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_BatchDisassociateUserStack.html
     */
    toBatchDisassociateUserStack() {
        return this.to('BatchDisassociateUserStack');
    }
    /**
     * Grants permission to copy the specified image within the same Region or to a new Region within the same AWS account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CopyImage.html
     */
    toCopyImage() {
        return this.to('CopyImage');
    }
    /**
     * Grants permission to create an app block. App blocks store details about the virtual hard disk that contains the files for the application in an S3 bucket. It also stores the setup script with details about how to mount the virtual hard disk. App blocks are only supported for Elastic fleets
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateAppBlock.html
     */
    toCreateAppBlock() {
        return this.to('CreateAppBlock');
    }
    /**
     * Grants permission to create an app block builder. An app block builder is a virtual machine that is used to create an app block
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateAppBlockBuilder.html
     */
    toCreateAppBlockBuilder() {
        return this.to('CreateAppBlockBuilder');
    }
    /**
     * Grants permission to create a URL to start an app block builder streaming session
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateAppBlockBuilderStreamingURL.html
     */
    toCreateAppBlockBuilderStreamingURL() {
        return this.to('CreateAppBlockBuilderStreamingURL');
    }
    /**
     * Grants permission to create an application within customer account. Applications store the details about how to launch applications on streaming instances. This is only supported for Elastic fleets
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateApplication.html
     */
    toCreateApplication() {
        return this.to('CreateApplication');
    }
    /**
     * Grants permission to create a Directory Config object in AppStream 2.0. This object includes the configuration information required to join fleets and image builders to Microsoft Active Directory domains
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateDirectoryConfig.html
     */
    toCreateDirectoryConfig() {
        return this.to('CreateDirectoryConfig');
    }
    /**
     * Grants permission to create an entitlement to control access to applications based on user attributes
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateEntitlement.html
     */
    toCreateEntitlement() {
        return this.to('CreateEntitlement');
    }
    /**
     * Grants permission to create a fleet. A fleet is a group of streaming instances from which applications are launched and streamed to users
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateFleet.html
     */
    toCreateFleet() {
        return this.to('CreateFleet');
    }
    /**
     * Grants permission to create an image builder. An image builder is a virtual machine that is used to create an image
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateImageBuilder.html
     */
    toCreateImageBuilder() {
        return this.to('CreateImageBuilder');
    }
    /**
     * Grants permission to create a URL to start an image builder streaming session
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateImageBuilderStreamingURL.html
     */
    toCreateImageBuilderStreamingURL() {
        return this.to('CreateImageBuilderStreamingURL');
    }
    /**
     * Grants permission to create a stack to start streaming applications to users. A stack consists of an associated fleet, user access policies, and storage configurations
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateStack.html
     */
    toCreateStack() {
        return this.to('CreateStack');
    }
    /**
     * Grants permission to create a temporary URL to start an AppStream 2.0 streaming session for the specified user. A streaming URL enables application streaming to be tested without user setup
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateStreamingURL.html
     */
    toCreateStreamingURL() {
        return this.to('CreateStreamingURL');
    }
    /**
     * Grants permission to create a custom branding theme, which might includes a custom logo, website links, and other branding to display to your users
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateThemeForStack.html
     */
    toCreateThemeForStack() {
        return this.to('CreateThemeForStack');
    }
    /**
     * Grants permission to update an existing image within customer account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateUpdatedImage.html
     */
    toCreateUpdatedImage() {
        return this.to('CreateUpdatedImage');
    }
    /**
     * Grants permission to create a usage report subscription. Usage reports are generated daily
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateUsageReportSubscription.html
     */
    toCreateUsageReportSubscription() {
        return this.to('CreateUsageReportSubscription');
    }
    /**
     * Grants permission to create a new user in the user pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateUser.html
     */
    toCreateUser() {
        return this.to('CreateUser');
    }
    /**
     * Grants permission to delete the specified app block
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteAppBlock.html
     */
    toDeleteAppBlock() {
        return this.to('DeleteAppBlock');
    }
    /**
     * Grants permission to delete the specified app block builder and release capacity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteAppBlockBuilder.html
     */
    toDeleteAppBlockBuilder() {
        return this.to('DeleteAppBlockBuilder');
    }
    /**
     * Grants permission to delete the specified application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteApplication.html
     */
    toDeleteApplication() {
        return this.to('DeleteApplication');
    }
    /**
     * Grants permission to delete the specified Directory Config object from AppStream 2.0. This object includes the configuration information required to join fleets and image builders to Microsoft Active Directory domains
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteDirectoryConfig.html
     */
    toDeleteDirectoryConfig() {
        return this.to('DeleteDirectoryConfig');
    }
    /**
     * Grants permission to delete the specified entitlement
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteEntitlement.html
     */
    toDeleteEntitlement() {
        return this.to('DeleteEntitlement');
    }
    /**
     * Grants permission to delete the specified fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteFleet.html
     */
    toDeleteFleet() {
        return this.to('DeleteFleet');
    }
    /**
     * Grants permission to delete the specified image. An image cannot be deleted when it is in use
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteImage.html
     */
    toDeleteImage() {
        return this.to('DeleteImage');
    }
    /**
     * Grants permission to delete the specified image builder and release capacity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteImageBuilder.html
     */
    toDeleteImageBuilder() {
        return this.to('DeleteImageBuilder');
    }
    /**
     * Grants permission to delete permissions for the specified private image
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteImagePermissions.html
     */
    toDeleteImagePermissions() {
        return this.to('DeleteImagePermissions');
    }
    /**
     * Grants permission to delete the specified stack. After the stack is deleted, the application streaming environment provided by the stack is no longer available to users. Also, any reservations made for application streaming sessions for the stack are released
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteStack.html
     */
    toDeleteStack() {
        return this.to('DeleteStack');
    }
    /**
     * Grants permission to delete a custom branding theme, which might includes a custom logo, website links, and other branding to display to your users
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteThemeForStack.html
     */
    toDeleteThemeForStack() {
        return this.to('DeleteThemeForStack');
    }
    /**
     * Grants permission to disable usage report generation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteUsageReportSubscription.html
     */
    toDeleteUsageReportSubscription() {
        return this.to('DeleteUsageReportSubscription');
    }
    /**
     * Grants permission to delete a user from the user pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteUser.html
     */
    toDeleteUser() {
        return this.to('DeleteUser');
    }
    /**
     * Grants permission to retrieve the associations that are associated with the specified app block builder or app block
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeAppBlockBuilderAppBlockAssociations.html
     */
    toDescribeAppBlockBuilderAppBlockAssociations() {
        return this.to('DescribeAppBlockBuilderAppBlockAssociations');
    }
    /**
     * Grants permission to retrieve a list that describes one or more specified app block builders, if the app block builder names are provided. Otherwise, all app block builders in the account are described
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeAppBlockBuilders.html
     */
    toDescribeAppBlockBuilders() {
        return this.to('DescribeAppBlockBuilders');
    }
    /**
     * Grants permission to retrieve a list that describes one or more specified app blocks, if the app block arns are provided. Otherwise, all app blocks in the account are described
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeAppBlocks.html
     */
    toDescribeAppBlocks() {
        return this.to('DescribeAppBlocks');
    }
    /**
     * Grants permission to retrieve the associations that are associated with the specified application or fleet
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeApplicationFleetAssociations.html
     */
    toDescribeApplicationFleetAssociations() {
        return this.to('DescribeApplicationFleetAssociations');
    }
    /**
     * Grants permission to retrieve a list that describes one or more specified applications, if the application arns are provided. Otherwise, all applications in the account are described
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeApplications.html
     */
    toDescribeApplications() {
        return this.to('DescribeApplications');
    }
    /**
     * Grants permission to retrieve a list that describes one or more specified Directory Config objects for AppStream 2.0, if the names for these objects are provided. Otherwise, all Directory Config objects in the account are described. This object includes the configuration information required to join fleets and image builders to Microsoft Active Directory domains
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeDirectoryConfigs.html
     */
    toDescribeDirectoryConfigs() {
        return this.to('DescribeDirectoryConfigs');
    }
    /**
     * Grants permission to retrieve one or all entitlements for the specified stack
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeEntitlements.html
     */
    toDescribeEntitlements() {
        return this.to('DescribeEntitlements');
    }
    /**
     * Grants permission to retrieve a list that describes one or more specified fleets, if the fleet names are provided. Otherwise, all fleets in the account are described
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeFleets.html
     */
    toDescribeFleets() {
        return this.to('DescribeFleets');
    }
    /**
     * Grants permission to retrieve a list that describes one or more specified image builders, if the image builder names are provided. Otherwise, all image builders in the account are described
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeImageBuilders.html
     */
    toDescribeImageBuilders() {
        return this.to('DescribeImageBuilders');
    }
    /**
     * Grants permission to retrieve a list that describes the permissions for shared AWS account IDs on a private image that you own
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeImagePermissions.html
     */
    toDescribeImagePermissions() {
        return this.to('DescribeImagePermissions');
    }
    /**
     * Grants permission to retrieve a list that describes one or more specified images, if the image names or image ARNs are provided. Otherwise, all images in the account are described
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeImages.html
     */
    toDescribeImages() {
        return this.to('DescribeImages');
    }
    /**
     * Grants permission to retrieve a list that describes the streaming sessions for the specified stack and fleet. If a user ID is provided for the stack and fleet, only the streaming sessions for that user are described
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeSessions.html
     */
    toDescribeSessions() {
        return this.to('DescribeSessions');
    }
    /**
     * Grants permission to retrieve a list that describes one or more specified stacks, if the stack names are provided. Otherwise, all stacks in the account are described
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeStacks.html
     */
    toDescribeStacks() {
        return this.to('DescribeStacks');
    }
    /**
     * Grants permission to get the custom branding theme information, which might includes a custom logo, website links, and other branding to display to your users
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeThemeForStack.html
     */
    toDescribeThemeForStack() {
        return this.to('DescribeThemeForStack');
    }
    /**
     * Grants permission to retrieve a list that describes one or more usage report subscriptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeUsageReportSubscriptions.html
     */
    toDescribeUsageReportSubscriptions() {
        return this.to('DescribeUsageReportSubscriptions');
    }
    /**
     * Grants permission to retrieve a list that describes the UserStackAssociation objects
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeUserStackAssociations.html
     */
    toDescribeUserStackAssociations() {
        return this.to('DescribeUserStackAssociations');
    }
    /**
     * Grants permission to retrieve a list that describes users in the user pool
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeUsers.html
     */
    toDescribeUsers() {
        return this.to('DescribeUsers');
    }
    /**
     * Grants permission to disable the specified user in the user pool. This action does not delete the user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DisableUser.html
     */
    toDisableUser() {
        return this.to('DisableUser');
    }
    /**
     * Grants permission to disassociate the specified app block builder with the app block
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DisassociateAppBlockBuilderAppBlock.html
     */
    toDisassociateAppBlockBuilderAppBlock() {
        return this.to('DisassociateAppBlockBuilderAppBlock');
    }
    /**
     * Grants permission to disassociate the specified application from the specified fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DisassociateApplicationFleet.html
     */
    toDisassociateApplicationFleet() {
        return this.to('DisassociateApplicationFleet');
    }
    /**
     * Grants permission to disassociate the specified application from the specified entitlement
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DisassociateApplicationFromEntitlement.html
     */
    toDisassociateApplicationFromEntitlement() {
        return this.to('DisassociateApplicationFromEntitlement');
    }
    /**
     * Grants permission to disassociate the specified fleet from the specified stack
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DisassociateFleet.html
     */
    toDisassociateFleet() {
        return this.to('DisassociateFleet');
    }
    /**
     * Grants permission to enable a user in the user pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_EnableUser.html
     */
    toEnableUser() {
        return this.to('EnableUser');
    }
    /**
     * Grants permission to immediately stop the specified streaming session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_ExpireSession.html
     */
    toExpireSession() {
        return this.to('ExpireSession');
    }
    /**
     * Grants permission to retrieve the name of the fleet that is associated with the specified stack
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_ListAssociatedFleets.html
     */
    toListAssociatedFleets() {
        return this.to('ListAssociatedFleets');
    }
    /**
     * Grants permission to retrieve the name of the stack with which the specified fleet is associated
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_ListAssociatedStacks.html
     */
    toListAssociatedStacks() {
        return this.to('ListAssociatedStacks');
    }
    /**
     * Grants permission to retrieve the applications that are associated with the specified entitlement
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_ListEntitledApplications.html
     */
    toListEntitledApplications() {
        return this.to('ListEntitledApplications');
    }
    /**
     * Grants permission to retrieve a list of all tags for the specified AppStream 2.0 resource. The following resources can be tagged: Image builders, images, fleets, and stacks
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to start the specified app block builder
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_StartAppBlockBuilder.html
     */
    toStartAppBlockBuilder() {
        return this.to('StartAppBlockBuilder');
    }
    /**
     * Grants permission to start the specified fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_StartFleet.html
     */
    toStartFleet() {
        return this.to('StartFleet');
    }
    /**
     * Grants permission to start the specified image builder
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_StartImageBuilder.html
     */
    toStartImageBuilder() {
        return this.to('StartImageBuilder');
    }
    /**
     * Grants permission to stop the specified app block builder
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_StopAppBlockBuilder.html
     */
    toStopAppBlockBuilder() {
        return this.to('StopAppBlockBuilder');
    }
    /**
     * Grants permission to stop the specified fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_StopFleet.html
     */
    toStopFleet() {
        return this.to('StopFleet');
    }
    /**
     * Grants permission to stop the specified image builder
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_StopImageBuilder.html
     */
    toStopImageBuilder() {
        return this.to('StopImageBuilder');
    }
    /**
     * Grants permission to federated users to sign in by using their existing credentials and stream applications from the specified stack
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserId()
     *
     * https://docs.aws.amazon.com/appstream2/latest/developerguide/external-identity-providers-setting-up-saml.html#external-identity-providers-embed-inline-policy-for-IAM-role
     */
    toStream() {
        return this.to('Stream');
    }
    /**
     * Grants permission to add or overwrite one or more tags for the specified AppStream 2.0 resource. The following resources can be tagged: Image builders, images, fleets, stacks, app blocks and applications
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to disassociate one or more tags from the specified AppStream 2.0 resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a specific app block builder. An app block builder is a virtual machine that is used to create an app block
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_UpdateAppBlockBuilder.html
     */
    toUpdateAppBlockBuilder() {
        return this.to('UpdateAppBlockBuilder');
    }
    /**
     * Grants permission to update the specified fields for the specified application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_UpdateApplication.html
     */
    toUpdateApplication() {
        return this.to('UpdateApplication');
    }
    /**
     * Grants permission to update the specified Directory Config object in AppStream 2.0. This object includes the configuration information required to join fleets and image builders to Microsoft Active Directory domains
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_UpdateDirectoryConfig.html
     */
    toUpdateDirectoryConfig() {
        return this.to('UpdateDirectoryConfig');
    }
    /**
     * Grants permission to update the specified fields for the specified entitlement
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_UpdateEntitlement.html
     */
    toUpdateEntitlement() {
        return this.to('UpdateEntitlement');
    }
    /**
     * Grants permission to update the specified fleet. All attributes except the fleet name can be updated when the fleet is in the STOPPED state
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_UpdateFleet.html
     */
    toUpdateFleet() {
        return this.to('UpdateFleet');
    }
    /**
     * Grants permission to add or update permissions for the specified private image
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_UpdateImagePermissions.html
     */
    toUpdateImagePermissions() {
        return this.to('UpdateImagePermissions');
    }
    /**
     * Grants permission to update the specified fields for the specified stack
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_UpdateStack.html
     */
    toUpdateStack() {
        return this.to('UpdateStack');
    }
    /**
     * Grants permission to update the custom branding theme information, which might includes a custom logo, website links, and other branding to display to your users
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_UpdateThemeForStack.html
     */
    toUpdateThemeForStack() {
        return this.to('UpdateThemeForStack');
    }
    /**
     * Adds a resource of type fleet to the statement
     *
     * https://docs.aws.amazon.com/appstream2/latest/developerguide/what-is-appstream.html#what-is-concepts
     *
     * @param fleetName - Identifier for the fleetName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFleet(fleetName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appstream:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:fleet/${fleetName}`);
    }
    /**
     * Adds a resource of type image to the statement
     *
     * https://docs.aws.amazon.com/appstream2/latest/developerguide/what-is-appstream.html#what-is-concepts
     *
     * @param imageName - Identifier for the imageName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onImage(imageName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appstream:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:image/${imageName}`);
    }
    /**
     * Adds a resource of type image-builder to the statement
     *
     * https://docs.aws.amazon.com/appstream2/latest/developerguide/what-is-appstream.html#what-is-concepts
     *
     * @param imageBuilderName - Identifier for the imageBuilderName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onImageBuilder(imageBuilderName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appstream:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:image-builder/${imageBuilderName}`);
    }
    /**
     * Adds a resource of type stack to the statement
     *
     * https://docs.aws.amazon.com/appstream2/latest/developerguide/what-is-appstream.html#what-is-concepts
     *
     * @param stackName - Identifier for the stackName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStack(stackName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appstream:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:stack/${stackName}`);
    }
    /**
     * Adds a resource of type app-block to the statement
     *
     * https://docs.aws.amazon.com/appstream2/latest/developerguide/what-is-appstream.html#what-is-concepts
     *
     * @param appBlockName - Identifier for the appBlockName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAppBlock(appBlockName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appstream:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:app-block/${appBlockName}`);
    }
    /**
     * Adds a resource of type application to the statement
     *
     * https://docs.aws.amazon.com/appstream2/latest/developerguide/what-is-appstream.html#what-is-concepts
     *
     * @param applicationName - Identifier for the applicationName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApplication(applicationName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appstream:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${applicationName}`);
    }
    /**
     * Adds a resource of type app-block-builder to the statement
     *
     * https://docs.aws.amazon.com/appstream2/latest/developerguide/what-is-appstream.html#what-is-concepts
     *
     * @param appBlockBuilderName - Identifier for the appBlockBuilderName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAppBlockBuilder(appBlockBuilderName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appstream:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:app-block-builder/${appBlockBuilderName}`);
    }
    /**
     * Filters access by the ID of the AppStream 2.0 user
     *
     * https://docs.aws.amazon.com/appstream2/latest/developerguide/external-identity-providers-setting-up-saml.html#external-identity-providers-embed-inline-policy-for-IAM-role
     *
     * Applies to actions:
     * - .toStream()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUserId(value, operator) {
        return this.if(`userId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAppBlock()
     * - .toCreateAppBlockBuilder()
     * - .toCreateApplication()
     * - .toCreateFleet()
     * - .toCreateImageBuilder()
     * - .toCreateStack()
     * - .toCreateUpdatedImage()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toAssociateAppBlockBuilderAppBlock()
     * - .toAssociateApplicationFleet()
     * - .toAssociateFleet()
     * - .toBatchAssociateUserStack()
     * - .toBatchDisassociateUserStack()
     * - .toCopyImage()
     * - .toCreateAppBlock()
     * - .toCreateAppBlockBuilderStreamingURL()
     * - .toCreateApplication()
     * - .toCreateImageBuilderStreamingURL()
     * - .toCreateStreamingURL()
     * - .toCreateUpdatedImage()
     * - .toDeleteAppBlock()
     * - .toDeleteAppBlockBuilder()
     * - .toDeleteApplication()
     * - .toDeleteFleet()
     * - .toDeleteImage()
     * - .toDeleteImageBuilder()
     * - .toDeleteImagePermissions()
     * - .toDeleteStack()
     * - .toDisassociateAppBlockBuilderAppBlock()
     * - .toDisassociateApplicationFleet()
     * - .toDisassociateFleet()
     * - .toStartAppBlockBuilder()
     * - .toStartFleet()
     * - .toStartImageBuilder()
     * - .toStopAppBlockBuilder()
     * - .toStopFleet()
     * - .toStopImageBuilder()
     * - .toTagResource()
     * - .toUpdateAppBlockBuilder()
     * - .toUpdateApplication()
     * - .toUpdateFleet()
     * - .toUpdateImagePermissions()
     * - .toUpdateStack()
     *
     * Applies to resource types:
     * - fleet
     * - image
     * - image-builder
     * - stack
     * - app-block
     * - application
     * - app-block-builder
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAppBlock()
     * - .toCreateAppBlockBuilder()
     * - .toCreateApplication()
     * - .toCreateFleet()
     * - .toCreateImageBuilder()
     * - .toCreateStack()
     * - .toCreateUpdatedImage()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [appstream](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonappstream2.0.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'appstream';
        this.accessLevelList = {
            Write: [
                'AssociateAppBlockBuilderAppBlock',
                'AssociateApplicationFleet',
                'AssociateApplicationToEntitlement',
                'AssociateFleet',
                'BatchAssociateUserStack',
                'BatchDisassociateUserStack',
                'CopyImage',
                'CreateAppBlock',
                'CreateAppBlockBuilder',
                'CreateAppBlockBuilderStreamingURL',
                'CreateApplication',
                'CreateDirectoryConfig',
                'CreateEntitlement',
                'CreateFleet',
                'CreateImageBuilder',
                'CreateImageBuilderStreamingURL',
                'CreateStack',
                'CreateStreamingURL',
                'CreateThemeForStack',
                'CreateUpdatedImage',
                'CreateUsageReportSubscription',
                'CreateUser',
                'DeleteAppBlock',
                'DeleteAppBlockBuilder',
                'DeleteApplication',
                'DeleteDirectoryConfig',
                'DeleteEntitlement',
                'DeleteFleet',
                'DeleteImage',
                'DeleteImageBuilder',
                'DeleteImagePermissions',
                'DeleteStack',
                'DeleteThemeForStack',
                'DeleteUsageReportSubscription',
                'DeleteUser',
                'DisableUser',
                'DisassociateAppBlockBuilderAppBlock',
                'DisassociateApplicationFleet',
                'DisassociateApplicationFromEntitlement',
                'DisassociateFleet',
                'EnableUser',
                'ExpireSession',
                'StartAppBlockBuilder',
                'StartFleet',
                'StartImageBuilder',
                'StopAppBlockBuilder',
                'StopFleet',
                'StopImageBuilder',
                'Stream',
                'UpdateAppBlockBuilder',
                'UpdateApplication',
                'UpdateDirectoryConfig',
                'UpdateEntitlement',
                'UpdateFleet',
                'UpdateImagePermissions',
                'UpdateStack',
                'UpdateThemeForStack'
            ],
            List: [
                'DescribeAppBlockBuilderAppBlockAssociations',
                'DescribeAppBlockBuilders',
                'DescribeAppBlocks',
                'DescribeApplicationFleetAssociations',
                'DescribeApplications',
                'DescribeDirectoryConfigs',
                'DescribeEntitlements',
                'DescribeFleets',
                'DescribeImageBuilders',
                'DescribeImages',
                'DescribeSessions',
                'DescribeStacks',
                'DescribeUsageReportSubscriptions',
                'DescribeUserStackAssociations',
                'DescribeUsers',
                'ListEntitledApplications'
            ],
            Read: [
                'DescribeImagePermissions',
                'DescribeThemeForStack',
                'ListAssociatedFleets',
                'ListAssociatedStacks',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Appstream = Appstream;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBwc3RyZWFtMi0wLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiYXBwc3RyZWFtMi0wLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUNBLHlDQUF5RDtBQUd6RDs7OztHQUlHO0FBQ0gsTUFBYSxTQUFVLFNBQVEsd0JBQWU7SUFHNUM7Ozs7Ozs7OztPQVNHO0lBQ0ksa0NBQWtDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO0lBQ3JELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSwyQkFBMkI7UUFDaEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7SUFDOUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1DQUFtQztRQUN4QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUNBQW1DLENBQUMsQ0FBQztJQUN0RCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksZ0JBQWdCO1FBQ3JCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO0lBQ25DLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSx5QkFBeUI7UUFDOUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7SUFDNUMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLDRCQUE0QjtRQUNqQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQztJQUMvQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksV0FBVztRQUNoQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsV0FBVyxDQUFDLENBQUM7SUFDOUIsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksZ0JBQWdCO1FBQ3JCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO0lBQ25DLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksdUJBQXVCO1FBQzVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxtQ0FBbUM7UUFDeEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1DQUFtQyxDQUFDLENBQUM7SUFDdEQsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksbUJBQW1CO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGFBQWE7UUFDbEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksb0JBQW9CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxnQ0FBZ0M7UUFDckMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdDQUFnQyxDQUFDLENBQUM7SUFDbkQsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxhQUFhO1FBQ2xCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUNoQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksb0JBQW9CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksb0JBQW9CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwrQkFBK0I7UUFDcEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLCtCQUErQixDQUFDLENBQUM7SUFDbEQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFlBQVksQ0FBQyxDQUFDO0lBQy9CLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHVCQUF1QjtRQUM1QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUMsQ0FBQztJQUMxQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksbUJBQW1CO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksYUFBYTtRQUNsQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGFBQWE7UUFDbEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxvQkFBb0I7UUFDekIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksYUFBYTtRQUNsQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksK0JBQStCO1FBQ3BDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxZQUFZO1FBQ2pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxZQUFZLENBQUMsQ0FBQztJQUMvQixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNkNBQTZDO1FBQ2xELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw2Q0FBNkMsQ0FBQyxDQUFDO0lBQ2hFLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0NBQXNDO1FBQzNDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDO0lBQ3pELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBCQUEwQjtRQUMvQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUMsQ0FBQztJQUMxQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQWtCLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO0lBQ25DLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtDQUFrQztRQUN2QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0NBQWtDLENBQUMsQ0FBQztJQUNyRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksK0JBQStCO1FBQ3BDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHFDQUFxQztRQUMxQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUNBQXFDLENBQUMsQ0FBQztJQUN4RCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksOEJBQThCO1FBQ25DLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3Q0FBd0M7UUFDN0MsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHdDQUF3QyxDQUFDLENBQUM7SUFDM0QsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksWUFBWTtRQUNqQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsWUFBWSxDQUFDLENBQUM7SUFDL0IsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksWUFBWTtRQUNqQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsWUFBWSxDQUFDLENBQUM7SUFDL0IsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxXQUFXO1FBQ2hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxXQUFXLENBQUMsQ0FBQztJQUM5QixDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxRQUFRO1FBQ2IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQzNCLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLGFBQWE7UUFDbEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksdUJBQXVCO1FBQzVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUMsQ0FBQztJQUMxQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxhQUFhO1FBQ2xCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUNoQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksd0JBQXdCO1FBQzdCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxhQUFhO1FBQ2xCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUNoQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUE2RkQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksT0FBTyxDQUFDLFNBQWlCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDckYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsY0FBZSxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsVUFBVyxTQUFVLEVBQUUsQ0FBQyxDQUFDO0lBQ3JLLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxPQUFPLENBQUMsU0FBaUIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNyRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixjQUFlLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxVQUFXLFNBQVUsRUFBRSxDQUFDLENBQUM7SUFDckssQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGNBQWMsQ0FBQyxnQkFBd0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNuRyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixjQUFlLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxrQkFBbUIsZ0JBQWlCLEVBQUUsQ0FBQyxDQUFDO0lBQ3BMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxPQUFPLENBQUMsU0FBaUIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNyRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixjQUFlLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxVQUFXLFNBQVUsRUFBRSxDQUFDLENBQUM7SUFDckssQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLFVBQVUsQ0FBQyxZQUFvQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzNGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGNBQWUsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLGNBQWUsWUFBYSxFQUFFLENBQUMsQ0FBQztJQUM1SyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksYUFBYSxDQUFDLGVBQXVCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDakcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsY0FBZSxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsZ0JBQWlCLGVBQWdCLEVBQUUsQ0FBQyxDQUFDO0lBQ2pMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxpQkFBaUIsQ0FBQyxtQkFBMkIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN6RyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixjQUFlLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxzQkFBdUIsbUJBQW9CLEVBQUUsQ0FBQyxDQUFDO0lBQzNMLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksUUFBUSxDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDcEUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFFBQVEsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQzVELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7O09Ba0JHO0lBQ0ksZUFBZSxDQUFDLE1BQWMsRUFBRSxLQUF3QixFQUFFLFFBQTRCO1FBQzNGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBbUIsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNoRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQXNERztJQUNJLGdCQUFnQixDQUFDLE1BQWMsRUFBRSxLQUF3QixFQUFFLFFBQTRCO1FBQzVGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBb0IsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQWtCRztJQUNJLFlBQVksQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3hFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRSxDQUFDO0lBRUQ7OztPQUdHO0lBQ0gsWUFBWSxLQUFnQztRQUMxQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7UUF4MENSLGtCQUFhLEdBQUcsV0FBVyxDQUFDO1FBeS9CekIsb0JBQWUsR0FBb0I7WUFDM0MsS0FBSyxFQUFFO2dCQUNMLGtDQUFrQztnQkFDbEMsMkJBQTJCO2dCQUMzQixtQ0FBbUM7Z0JBQ25DLGdCQUFnQjtnQkFDaEIseUJBQXlCO2dCQUN6Qiw0QkFBNEI7Z0JBQzVCLFdBQVc7Z0JBQ1gsZ0JBQWdCO2dCQUNoQix1QkFBdUI7Z0JBQ3ZCLG1DQUFtQztnQkFDbkMsbUJBQW1CO2dCQUNuQix1QkFBdUI7Z0JBQ3ZCLG1CQUFtQjtnQkFDbkIsYUFBYTtnQkFDYixvQkFBb0I7Z0JBQ3BCLGdDQUFnQztnQkFDaEMsYUFBYTtnQkFDYixvQkFBb0I7Z0JBQ3BCLHFCQUFxQjtnQkFDckIsb0JBQW9CO2dCQUNwQiwrQkFBK0I7Z0JBQy9CLFlBQVk7Z0JBQ1osZ0JBQWdCO2dCQUNoQix1QkFBdUI7Z0JBQ3ZCLG1CQUFtQjtnQkFDbkIsdUJBQXVCO2dCQUN2QixtQkFBbUI7Z0JBQ25CLGFBQWE7Z0JBQ2IsYUFBYTtnQkFDYixvQkFBb0I7Z0JBQ3BCLHdCQUF3QjtnQkFDeEIsYUFBYTtnQkFDYixxQkFBcUI7Z0JBQ3JCLCtCQUErQjtnQkFDL0IsWUFBWTtnQkFDWixhQUFhO2dCQUNiLHFDQUFxQztnQkFDckMsOEJBQThCO2dCQUM5Qix3Q0FBd0M7Z0JBQ3hDLG1CQUFtQjtnQkFDbkIsWUFBWTtnQkFDWixlQUFlO2dCQUNmLHNCQUFzQjtnQkFDdEIsWUFBWTtnQkFDWixtQkFBbUI7Z0JBQ25CLHFCQUFxQjtnQkFDckIsV0FBVztnQkFDWCxrQkFBa0I7Z0JBQ2xCLFFBQVE7Z0JBQ1IsdUJBQXVCO2dCQUN2QixtQkFBbUI7Z0JBQ25CLHVCQUF1QjtnQkFDdkIsbUJBQW1CO2dCQUNuQixhQUFhO2dCQUNiLHdCQUF3QjtnQkFDeEIsYUFBYTtnQkFDYixxQkFBcUI7YUFDdEI7WUFDRCxJQUFJLEVBQUU7Z0JBQ0osNkNBQTZDO2dCQUM3QywwQkFBMEI7Z0JBQzFCLG1CQUFtQjtnQkFDbkIsc0NBQXNDO2dCQUN0QyxzQkFBc0I7Z0JBQ3RCLDBCQUEwQjtnQkFDMUIsc0JBQXNCO2dCQUN0QixnQkFBZ0I7Z0JBQ2hCLHVCQUF1QjtnQkFDdkIsZ0JBQWdCO2dCQUNoQixrQkFBa0I7Z0JBQ2xCLGdCQUFnQjtnQkFDaEIsa0NBQWtDO2dCQUNsQywrQkFBK0I7Z0JBQy9CLGVBQWU7Z0JBQ2YsMEJBQTBCO2FBQzNCO1lBQ0QsSUFBSSxFQUFFO2dCQUNKLDBCQUEwQjtnQkFDMUIsdUJBQXVCO2dCQUN2QixzQkFBc0I7Z0JBQ3RCLHNCQUFzQjtnQkFDdEIscUJBQXFCO2FBQ3RCO1lBQ0QsT0FBTyxFQUFFO2dCQUNQLGFBQWE7Z0JBQ2IsZUFBZTthQUNoQjtTQUNGLENBQUM7SUF1UEYsQ0FBQztDQUNGO0FBMzBDRCw4QkEyMENDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQWNjZXNzTGV2ZWxMaXN0IH0gZnJvbSAnLi4vLi4vc2hhcmVkL2FjY2Vzcy1sZXZlbCc7XG5pbXBvcnQgeyBQb2xpY3lTdGF0ZW1lbnQsIE9wZXJhdG9yIH0gZnJvbSAnLi4vLi4vc2hhcmVkJztcbmltcG9ydCB7IGF3c19pYW0gYXMgaWFtIH0gZnJvbSBcImF3cy1jZGstbGliXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFthcHBzdHJlYW1dKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlLWF1dGhvcml6YXRpb24vbGF0ZXN0L3JlZmVyZW5jZS9saXN0X2FtYXpvbmFwcHN0cmVhbTIuMC5odG1sKS5cbiAqXG4gKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAqL1xuZXhwb3J0IGNsYXNzIEFwcHN0cmVhbSBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ2FwcHN0cmVhbSc7XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFzc29jaWF0ZSB0aGUgc3BlY2lmaWVkIGFwcCBibG9jayBidWlsZGVyIHdpdGggdGhlIGFwcCBibG9ja1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3RyZWFtMi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9Bc3NvY2lhdGVBcHBCbG9ja0J1aWxkZXJBcHBCbG9jay5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Bc3NvY2lhdGVBcHBCbG9ja0J1aWxkZXJBcHBCbG9jaygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQXNzb2NpYXRlQXBwQmxvY2tCdWlsZGVyQXBwQmxvY2snKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBhc3NvY2lhdGUgdGhlIHNwZWNpZmllZCBhcHBsaWNhdGlvbiB3aXRoIHRoZSBmbGVldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3RyZWFtMi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9Bc3NvY2lhdGVBcHBsaWNhdGlvbkZsZWV0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Fzc29jaWF0ZUFwcGxpY2F0aW9uRmxlZXQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Fzc29jaWF0ZUFwcGxpY2F0aW9uRmxlZXQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBhc3NvY2lhdGUgdGhlIHNwZWNpZmllZCBhcHBsaWNhdGlvbiB0byB0aGUgc3BlY2lmaWVkIGVudGl0bGVtZW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQXNzb2NpYXRlQXBwbGljYXRpb25Ub0VudGl0bGVtZW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Fzc29jaWF0ZUFwcGxpY2F0aW9uVG9FbnRpdGxlbWVudCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQXNzb2NpYXRlQXBwbGljYXRpb25Ub0VudGl0bGVtZW50Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYXNzb2NpYXRlIHRoZSBzcGVjaWZpZWQgZmxlZXQgd2l0aCB0aGUgc3BlY2lmaWVkIHN0YWNrXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzdHJlYW0yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Fzc29jaWF0ZUZsZWV0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Fzc29jaWF0ZUZsZWV0KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdBc3NvY2lhdGVGbGVldCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFzc29jaWF0ZSB0aGUgc3BlY2lmaWVkIHVzZXJzIHdpdGggdGhlIHNwZWNpZmllZCBzdGFja3MuIFVzZXJzIGluIGEgdXNlciBwb29sIGNhbm5vdCBiZSBhc3NpZ25lZCB0byBzdGFja3Mgd2l0aCBmbGVldHMgdGhhdCBhcmUgam9pbmVkIHRvIGFuIEFjdGl2ZSBEaXJlY3RvcnkgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzdHJlYW0yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0JhdGNoQXNzb2NpYXRlVXNlclN0YWNrLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0JhdGNoQXNzb2NpYXRlVXNlclN0YWNrKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdCYXRjaEFzc29jaWF0ZVVzZXJTdGFjaycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRpc2Fzc29jaWF0ZSB0aGUgc3BlY2lmaWVkIHVzZXJzIGZyb20gdGhlIHNwZWNpZmllZCBzdGFja3NcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQmF0Y2hEaXNhc3NvY2lhdGVVc2VyU3RhY2suaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQmF0Y2hEaXNhc3NvY2lhdGVVc2VyU3RhY2soKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0JhdGNoRGlzYXNzb2NpYXRlVXNlclN0YWNrJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY29weSB0aGUgc3BlY2lmaWVkIGltYWdlIHdpdGhpbiB0aGUgc2FtZSBSZWdpb24gb3IgdG8gYSBuZXcgUmVnaW9uIHdpdGhpbiB0aGUgc2FtZSBBV1MgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3RyZWFtMi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9Db3B5SW1hZ2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ29weUltYWdlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDb3B5SW1hZ2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYW4gYXBwIGJsb2NrLiBBcHAgYmxvY2tzIHN0b3JlIGRldGFpbHMgYWJvdXQgdGhlIHZpcnR1YWwgaGFyZCBkaXNrIHRoYXQgY29udGFpbnMgdGhlIGZpbGVzIGZvciB0aGUgYXBwbGljYXRpb24gaW4gYW4gUzMgYnVja2V0LiBJdCBhbHNvIHN0b3JlcyB0aGUgc2V0dXAgc2NyaXB0IHdpdGggZGV0YWlscyBhYm91dCBob3cgdG8gbW91bnQgdGhlIHZpcnR1YWwgaGFyZCBkaXNrLiBBcHAgYmxvY2tzIGFyZSBvbmx5IHN1cHBvcnRlZCBmb3IgRWxhc3RpYyBmbGVldHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlQXBwQmxvY2suaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlQXBwQmxvY2soKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZUFwcEJsb2NrJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGFuIGFwcCBibG9jayBidWlsZGVyLiBBbiBhcHAgYmxvY2sgYnVpbGRlciBpcyBhIHZpcnR1YWwgbWFjaGluZSB0aGF0IGlzIHVzZWQgdG8gY3JlYXRlIGFuIGFwcCBibG9ja1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzdHJlYW0yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZUFwcEJsb2NrQnVpbGRlci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVBcHBCbG9ja0J1aWxkZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZUFwcEJsb2NrQnVpbGRlcicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIFVSTCB0byBzdGFydCBhbiBhcHAgYmxvY2sgYnVpbGRlciBzdHJlYW1pbmcgc2Vzc2lvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3RyZWFtMi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVBcHBCbG9ja0J1aWxkZXJTdHJlYW1pbmdVUkwuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlQXBwQmxvY2tCdWlsZGVyU3RyZWFtaW5nVVJMKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVBcHBCbG9ja0J1aWxkZXJTdHJlYW1pbmdVUkwnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYW4gYXBwbGljYXRpb24gd2l0aGluIGN1c3RvbWVyIGFjY291bnQuIEFwcGxpY2F0aW9ucyBzdG9yZSB0aGUgZGV0YWlscyBhYm91dCBob3cgdG8gbGF1bmNoIGFwcGxpY2F0aW9ucyBvbiBzdHJlYW1pbmcgaW5zdGFuY2VzLiBUaGlzIGlzIG9ubHkgc3VwcG9ydGVkIGZvciBFbGFzdGljIGZsZWV0c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3RyZWFtMi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVBcHBsaWNhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVBcHBsaWNhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlQXBwbGljYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBEaXJlY3RvcnkgQ29uZmlnIG9iamVjdCBpbiBBcHBTdHJlYW0gMi4wLiBUaGlzIG9iamVjdCBpbmNsdWRlcyB0aGUgY29uZmlndXJhdGlvbiBpbmZvcm1hdGlvbiByZXF1aXJlZCB0byBqb2luIGZsZWV0cyBhbmQgaW1hZ2UgYnVpbGRlcnMgdG8gTWljcm9zb2Z0IEFjdGl2ZSBEaXJlY3RvcnkgZG9tYWluc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzdHJlYW0yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZURpcmVjdG9yeUNvbmZpZy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVEaXJlY3RvcnlDb25maWcoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZURpcmVjdG9yeUNvbmZpZycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhbiBlbnRpdGxlbWVudCB0byBjb250cm9sIGFjY2VzcyB0byBhcHBsaWNhdGlvbnMgYmFzZWQgb24gdXNlciBhdHRyaWJ1dGVzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlRW50aXRsZW1lbnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlRW50aXRsZW1lbnQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZUVudGl0bGVtZW50Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgZmxlZXQuIEEgZmxlZXQgaXMgYSBncm91cCBvZiBzdHJlYW1pbmcgaW5zdGFuY2VzIGZyb20gd2hpY2ggYXBwbGljYXRpb25zIGFyZSBsYXVuY2hlZCBhbmQgc3RyZWFtZWQgdG8gdXNlcnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3RyZWFtMi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVGbGVldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVGbGVldCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlRmxlZXQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYW4gaW1hZ2UgYnVpbGRlci4gQW4gaW1hZ2UgYnVpbGRlciBpcyBhIHZpcnR1YWwgbWFjaGluZSB0aGF0IGlzIHVzZWQgdG8gY3JlYXRlIGFuIGltYWdlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlSW1hZ2VCdWlsZGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUltYWdlQnVpbGRlcigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlSW1hZ2VCdWlsZGVyJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgVVJMIHRvIHN0YXJ0IGFuIGltYWdlIGJ1aWxkZXIgc3RyZWFtaW5nIHNlc3Npb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlSW1hZ2VCdWlsZGVyU3RyZWFtaW5nVVJMLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUltYWdlQnVpbGRlclN0cmVhbWluZ1VSTCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlSW1hZ2VCdWlsZGVyU3RyZWFtaW5nVVJMJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgc3RhY2sgdG8gc3RhcnQgc3RyZWFtaW5nIGFwcGxpY2F0aW9ucyB0byB1c2Vycy4gQSBzdGFjayBjb25zaXN0cyBvZiBhbiBhc3NvY2lhdGVkIGZsZWV0LCB1c2VyIGFjY2VzcyBwb2xpY2llcywgYW5kIHN0b3JhZ2UgY29uZmlndXJhdGlvbnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3RyZWFtMi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVTdGFjay5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVTdGFjaygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlU3RhY2snKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSB0ZW1wb3JhcnkgVVJMIHRvIHN0YXJ0IGFuIEFwcFN0cmVhbSAyLjAgc3RyZWFtaW5nIHNlc3Npb24gZm9yIHRoZSBzcGVjaWZpZWQgdXNlci4gQSBzdHJlYW1pbmcgVVJMIGVuYWJsZXMgYXBwbGljYXRpb24gc3RyZWFtaW5nIHRvIGJlIHRlc3RlZCB3aXRob3V0IHVzZXIgc2V0dXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlU3RyZWFtaW5nVVJMLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVN0cmVhbWluZ1VSTCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlU3RyZWFtaW5nVVJMJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgY3VzdG9tIGJyYW5kaW5nIHRoZW1lLCB3aGljaCBtaWdodCBpbmNsdWRlcyBhIGN1c3RvbSBsb2dvLCB3ZWJzaXRlIGxpbmtzLCBhbmQgb3RoZXIgYnJhbmRpbmcgdG8gZGlzcGxheSB0byB5b3VyIHVzZXJzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlVGhlbWVGb3JTdGFjay5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVUaGVtZUZvclN0YWNrKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVUaGVtZUZvclN0YWNrJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGFuIGV4aXN0aW5nIGltYWdlIHdpdGhpbiBjdXN0b21lciBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzdHJlYW0yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZVVwZGF0ZWRJbWFnZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVVcGRhdGVkSW1hZ2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZVVwZGF0ZWRJbWFnZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIHVzYWdlIHJlcG9ydCBzdWJzY3JpcHRpb24uIFVzYWdlIHJlcG9ydHMgYXJlIGdlbmVyYXRlZCBkYWlseVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzdHJlYW0yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZVVzYWdlUmVwb3J0U3Vic2NyaXB0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVVzYWdlUmVwb3J0U3Vic2NyaXB0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVVc2FnZVJlcG9ydFN1YnNjcmlwdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIG5ldyB1c2VyIGluIHRoZSB1c2VyIHBvb2xcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3RyZWFtMi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVVc2VyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVVzZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZVVzZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgdGhlIHNwZWNpZmllZCBhcHAgYmxvY2tcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlQXBwQmxvY2suaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlQXBwQmxvY2soKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZUFwcEJsb2NrJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIHRoZSBzcGVjaWZpZWQgYXBwIGJsb2NrIGJ1aWxkZXIgYW5kIHJlbGVhc2UgY2FwYWNpdHlcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlQXBwQmxvY2tCdWlsZGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUFwcEJsb2NrQnVpbGRlcigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlQXBwQmxvY2tCdWlsZGVyJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIHRoZSBzcGVjaWZpZWQgYXBwbGljYXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlQXBwbGljYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlQXBwbGljYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZUFwcGxpY2F0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIHRoZSBzcGVjaWZpZWQgRGlyZWN0b3J5IENvbmZpZyBvYmplY3QgZnJvbSBBcHBTdHJlYW0gMi4wLiBUaGlzIG9iamVjdCBpbmNsdWRlcyB0aGUgY29uZmlndXJhdGlvbiBpbmZvcm1hdGlvbiByZXF1aXJlZCB0byBqb2luIGZsZWV0cyBhbmQgaW1hZ2UgYnVpbGRlcnMgdG8gTWljcm9zb2Z0IEFjdGl2ZSBEaXJlY3RvcnkgZG9tYWluc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzdHJlYW0yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZURpcmVjdG9yeUNvbmZpZy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVEaXJlY3RvcnlDb25maWcoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZURpcmVjdG9yeUNvbmZpZycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSB0aGUgc3BlY2lmaWVkIGVudGl0bGVtZW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlRW50aXRsZW1lbnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlRW50aXRsZW1lbnQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZUVudGl0bGVtZW50Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIHRoZSBzcGVjaWZpZWQgZmxlZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlRmxlZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlRmxlZXQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZUZsZWV0Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIHRoZSBzcGVjaWZpZWQgaW1hZ2UuIEFuIGltYWdlIGNhbm5vdCBiZSBkZWxldGVkIHdoZW4gaXQgaXMgaW4gdXNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzdHJlYW0yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUltYWdlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUltYWdlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVJbWFnZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSB0aGUgc3BlY2lmaWVkIGltYWdlIGJ1aWxkZXIgYW5kIHJlbGVhc2UgY2FwYWNpdHlcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlSW1hZ2VCdWlsZGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUltYWdlQnVpbGRlcigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlSW1hZ2VCdWlsZGVyJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIHBlcm1pc3Npb25zIGZvciB0aGUgc3BlY2lmaWVkIHByaXZhdGUgaW1hZ2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlSW1hZ2VQZXJtaXNzaW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVJbWFnZVBlcm1pc3Npb25zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVJbWFnZVBlcm1pc3Npb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIHRoZSBzcGVjaWZpZWQgc3RhY2suIEFmdGVyIHRoZSBzdGFjayBpcyBkZWxldGVkLCB0aGUgYXBwbGljYXRpb24gc3RyZWFtaW5nIGVudmlyb25tZW50IHByb3ZpZGVkIGJ5IHRoZSBzdGFjayBpcyBubyBsb25nZXIgYXZhaWxhYmxlIHRvIHVzZXJzLiBBbHNvLCBhbnkgcmVzZXJ2YXRpb25zIG1hZGUgZm9yIGFwcGxpY2F0aW9uIHN0cmVhbWluZyBzZXNzaW9ucyBmb3IgdGhlIHN0YWNrIGFyZSByZWxlYXNlZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3RyZWFtMi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVTdGFjay5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVTdGFjaygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlU3RhY2snKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBjdXN0b20gYnJhbmRpbmcgdGhlbWUsIHdoaWNoIG1pZ2h0IGluY2x1ZGVzIGEgY3VzdG9tIGxvZ28sIHdlYnNpdGUgbGlua3MsIGFuZCBvdGhlciBicmFuZGluZyB0byBkaXNwbGF5IHRvIHlvdXIgdXNlcnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3RyZWFtMi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVUaGVtZUZvclN0YWNrLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVRoZW1lRm9yU3RhY2soKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZVRoZW1lRm9yU3RhY2snKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkaXNhYmxlIHVzYWdlIHJlcG9ydCBnZW5lcmF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlVXNhZ2VSZXBvcnRTdWJzY3JpcHRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlVXNhZ2VSZXBvcnRTdWJzY3JpcHRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZVVzYWdlUmVwb3J0U3Vic2NyaXB0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgdXNlciBmcm9tIHRoZSB1c2VyIHBvb2xcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3RyZWFtMi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVVc2VyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVVzZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZVVzZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSB0aGUgYXNzb2NpYXRpb25zIHRoYXQgYXJlIGFzc29jaWF0ZWQgd2l0aCB0aGUgc3BlY2lmaWVkIGFwcCBibG9jayBidWlsZGVyIG9yIGFwcCBibG9ja1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVBcHBCbG9ja0J1aWxkZXJBcHBCbG9ja0Fzc29jaWF0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUFwcEJsb2NrQnVpbGRlckFwcEJsb2NrQXNzb2NpYXRpb25zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZUFwcEJsb2NrQnVpbGRlckFwcEJsb2NrQXNzb2NpYXRpb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBsaXN0IHRoYXQgZGVzY3JpYmVzIG9uZSBvciBtb3JlIHNwZWNpZmllZCBhcHAgYmxvY2sgYnVpbGRlcnMsIGlmIHRoZSBhcHAgYmxvY2sgYnVpbGRlciBuYW1lcyBhcmUgcHJvdmlkZWQuIE90aGVyd2lzZSwgYWxsIGFwcCBibG9jayBidWlsZGVycyBpbiB0aGUgYWNjb3VudCBhcmUgZGVzY3JpYmVkXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3RyZWFtMi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUFwcEJsb2NrQnVpbGRlcnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVBcHBCbG9ja0J1aWxkZXJzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZUFwcEJsb2NrQnVpbGRlcnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBhIGxpc3QgdGhhdCBkZXNjcmliZXMgb25lIG9yIG1vcmUgc3BlY2lmaWVkIGFwcCBibG9ja3MsIGlmIHRoZSBhcHAgYmxvY2sgYXJucyBhcmUgcHJvdmlkZWQuIE90aGVyd2lzZSwgYWxsIGFwcCBibG9ja3MgaW4gdGhlIGFjY291bnQgYXJlIGRlc2NyaWJlZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVBcHBCbG9ja3MuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVBcHBCbG9ja3MoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlQXBwQmxvY2tzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgdGhlIGFzc29jaWF0aW9ucyB0aGF0IGFyZSBhc3NvY2lhdGVkIHdpdGggdGhlIHNwZWNpZmllZCBhcHBsaWNhdGlvbiBvciBmbGVldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVBcHBsaWNhdGlvbkZsZWV0QXNzb2NpYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlQXBwbGljYXRpb25GbGVldEFzc29jaWF0aW9ucygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVBcHBsaWNhdGlvbkZsZWV0QXNzb2NpYXRpb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBsaXN0IHRoYXQgZGVzY3JpYmVzIG9uZSBvciBtb3JlIHNwZWNpZmllZCBhcHBsaWNhdGlvbnMsIGlmIHRoZSBhcHBsaWNhdGlvbiBhcm5zIGFyZSBwcm92aWRlZC4gT3RoZXJ3aXNlLCBhbGwgYXBwbGljYXRpb25zIGluIHRoZSBhY2NvdW50IGFyZSBkZXNjcmliZWRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzdHJlYW0yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlQXBwbGljYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlQXBwbGljYXRpb25zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZUFwcGxpY2F0aW9ucycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGEgbGlzdCB0aGF0IGRlc2NyaWJlcyBvbmUgb3IgbW9yZSBzcGVjaWZpZWQgRGlyZWN0b3J5IENvbmZpZyBvYmplY3RzIGZvciBBcHBTdHJlYW0gMi4wLCBpZiB0aGUgbmFtZXMgZm9yIHRoZXNlIG9iamVjdHMgYXJlIHByb3ZpZGVkLiBPdGhlcndpc2UsIGFsbCBEaXJlY3RvcnkgQ29uZmlnIG9iamVjdHMgaW4gdGhlIGFjY291bnQgYXJlIGRlc2NyaWJlZC4gVGhpcyBvYmplY3QgaW5jbHVkZXMgdGhlIGNvbmZpZ3VyYXRpb24gaW5mb3JtYXRpb24gcmVxdWlyZWQgdG8gam9pbiBmbGVldHMgYW5kIGltYWdlIGJ1aWxkZXJzIHRvIE1pY3Jvc29mdCBBY3RpdmUgRGlyZWN0b3J5IGRvbWFpbnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzdHJlYW0yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlRGlyZWN0b3J5Q29uZmlncy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZURpcmVjdG9yeUNvbmZpZ3MoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlRGlyZWN0b3J5Q29uZmlncycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIG9uZSBvciBhbGwgZW50aXRsZW1lbnRzIGZvciB0aGUgc3BlY2lmaWVkIHN0YWNrXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3RyZWFtMi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUVudGl0bGVtZW50cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUVudGl0bGVtZW50cygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVFbnRpdGxlbWVudHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBhIGxpc3QgdGhhdCBkZXNjcmliZXMgb25lIG9yIG1vcmUgc3BlY2lmaWVkIGZsZWV0cywgaWYgdGhlIGZsZWV0IG5hbWVzIGFyZSBwcm92aWRlZC4gT3RoZXJ3aXNlLCBhbGwgZmxlZXRzIGluIHRoZSBhY2NvdW50IGFyZSBkZXNjcmliZWRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzdHJlYW0yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlRmxlZXRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlRmxlZXRzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZUZsZWV0cycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGEgbGlzdCB0aGF0IGRlc2NyaWJlcyBvbmUgb3IgbW9yZSBzcGVjaWZpZWQgaW1hZ2UgYnVpbGRlcnMsIGlmIHRoZSBpbWFnZSBidWlsZGVyIG5hbWVzIGFyZSBwcm92aWRlZC4gT3RoZXJ3aXNlLCBhbGwgaW1hZ2UgYnVpbGRlcnMgaW4gdGhlIGFjY291bnQgYXJlIGRlc2NyaWJlZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVJbWFnZUJ1aWxkZXJzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlSW1hZ2VCdWlsZGVycygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVJbWFnZUJ1aWxkZXJzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBsaXN0IHRoYXQgZGVzY3JpYmVzIHRoZSBwZXJtaXNzaW9ucyBmb3Igc2hhcmVkIEFXUyBhY2NvdW50IElEcyBvbiBhIHByaXZhdGUgaW1hZ2UgdGhhdCB5b3Ugb3duXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3RyZWFtMi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUltYWdlUGVybWlzc2lvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVJbWFnZVBlcm1pc3Npb25zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZUltYWdlUGVybWlzc2lvbnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBhIGxpc3QgdGhhdCBkZXNjcmliZXMgb25lIG9yIG1vcmUgc3BlY2lmaWVkIGltYWdlcywgaWYgdGhlIGltYWdlIG5hbWVzIG9yIGltYWdlIEFSTnMgYXJlIHByb3ZpZGVkLiBPdGhlcndpc2UsIGFsbCBpbWFnZXMgaW4gdGhlIGFjY291bnQgYXJlIGRlc2NyaWJlZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVJbWFnZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVJbWFnZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlSW1hZ2VzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBsaXN0IHRoYXQgZGVzY3JpYmVzIHRoZSBzdHJlYW1pbmcgc2Vzc2lvbnMgZm9yIHRoZSBzcGVjaWZpZWQgc3RhY2sgYW5kIGZsZWV0LiBJZiBhIHVzZXIgSUQgaXMgcHJvdmlkZWQgZm9yIHRoZSBzdGFjayBhbmQgZmxlZXQsIG9ubHkgdGhlIHN0cmVhbWluZyBzZXNzaW9ucyBmb3IgdGhhdCB1c2VyIGFyZSBkZXNjcmliZWRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzdHJlYW0yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlU2Vzc2lvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVTZXNzaW9ucygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVTZXNzaW9ucycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGEgbGlzdCB0aGF0IGRlc2NyaWJlcyBvbmUgb3IgbW9yZSBzcGVjaWZpZWQgc3RhY2tzLCBpZiB0aGUgc3RhY2sgbmFtZXMgYXJlIHByb3ZpZGVkLiBPdGhlcndpc2UsIGFsbCBzdGFja3MgaW4gdGhlIGFjY291bnQgYXJlIGRlc2NyaWJlZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVTdGFja3MuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVTdGFja3MoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlU3RhY2tzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IHRoZSBjdXN0b20gYnJhbmRpbmcgdGhlbWUgaW5mb3JtYXRpb24sIHdoaWNoIG1pZ2h0IGluY2x1ZGVzIGEgY3VzdG9tIGxvZ28sIHdlYnNpdGUgbGlua3MsIGFuZCBvdGhlciBicmFuZGluZyB0byBkaXNwbGF5IHRvIHlvdXIgdXNlcnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzdHJlYW0yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlVGhlbWVGb3JTdGFjay5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZVRoZW1lRm9yU3RhY2soKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlVGhlbWVGb3JTdGFjaycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGEgbGlzdCB0aGF0IGRlc2NyaWJlcyBvbmUgb3IgbW9yZSB1c2FnZSByZXBvcnQgc3Vic2NyaXB0aW9uc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVVc2FnZVJlcG9ydFN1YnNjcmlwdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVVc2FnZVJlcG9ydFN1YnNjcmlwdGlvbnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlVXNhZ2VSZXBvcnRTdWJzY3JpcHRpb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBsaXN0IHRoYXQgZGVzY3JpYmVzIHRoZSBVc2VyU3RhY2tBc3NvY2lhdGlvbiBvYmplY3RzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3RyZWFtMi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVVzZXJTdGFja0Fzc29jaWF0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZVVzZXJTdGFja0Fzc29jaWF0aW9ucygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVVc2VyU3RhY2tBc3NvY2lhdGlvbnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBhIGxpc3QgdGhhdCBkZXNjcmliZXMgdXNlcnMgaW4gdGhlIHVzZXIgcG9vbFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVVc2Vycy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZVVzZXJzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZVVzZXJzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGlzYWJsZSB0aGUgc3BlY2lmaWVkIHVzZXIgaW4gdGhlIHVzZXIgcG9vbC4gVGhpcyBhY3Rpb24gZG9lcyBub3QgZGVsZXRlIHRoZSB1c2VyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGlzYWJsZVVzZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGlzYWJsZVVzZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rpc2FibGVVc2VyJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGlzYXNzb2NpYXRlIHRoZSBzcGVjaWZpZWQgYXBwIGJsb2NrIGJ1aWxkZXIgd2l0aCB0aGUgYXBwIGJsb2NrXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzdHJlYW0yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rpc2Fzc29jaWF0ZUFwcEJsb2NrQnVpbGRlckFwcEJsb2NrLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rpc2Fzc29jaWF0ZUFwcEJsb2NrQnVpbGRlckFwcEJsb2NrKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEaXNhc3NvY2lhdGVBcHBCbG9ja0J1aWxkZXJBcHBCbG9jaycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRpc2Fzc29jaWF0ZSB0aGUgc3BlY2lmaWVkIGFwcGxpY2F0aW9uIGZyb20gdGhlIHNwZWNpZmllZCBmbGVldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3RyZWFtMi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EaXNhc3NvY2lhdGVBcHBsaWNhdGlvbkZsZWV0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rpc2Fzc29jaWF0ZUFwcGxpY2F0aW9uRmxlZXQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rpc2Fzc29jaWF0ZUFwcGxpY2F0aW9uRmxlZXQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkaXNhc3NvY2lhdGUgdGhlIHNwZWNpZmllZCBhcHBsaWNhdGlvbiBmcm9tIHRoZSBzcGVjaWZpZWQgZW50aXRsZW1lbnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3RyZWFtMi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EaXNhc3NvY2lhdGVBcHBsaWNhdGlvbkZyb21FbnRpdGxlbWVudC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EaXNhc3NvY2lhdGVBcHBsaWNhdGlvbkZyb21FbnRpdGxlbWVudCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGlzYXNzb2NpYXRlQXBwbGljYXRpb25Gcm9tRW50aXRsZW1lbnQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkaXNhc3NvY2lhdGUgdGhlIHNwZWNpZmllZCBmbGVldCBmcm9tIHRoZSBzcGVjaWZpZWQgc3RhY2tcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGlzYXNzb2NpYXRlRmxlZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGlzYXNzb2NpYXRlRmxlZXQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rpc2Fzc29jaWF0ZUZsZWV0Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZW5hYmxlIGEgdXNlciBpbiB0aGUgdXNlciBwb29sXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRW5hYmxlVXNlci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9FbmFibGVVc2VyKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdFbmFibGVVc2VyJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gaW1tZWRpYXRlbHkgc3RvcCB0aGUgc3BlY2lmaWVkIHN0cmVhbWluZyBzZXNzaW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRXhwaXJlU2Vzc2lvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9FeHBpcmVTZXNzaW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdFeHBpcmVTZXNzaW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgdGhlIG5hbWUgb2YgdGhlIGZsZWV0IHRoYXQgaXMgYXNzb2NpYXRlZCB3aXRoIHRoZSBzcGVjaWZpZWQgc3RhY2tcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzdHJlYW0yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RBc3NvY2lhdGVkRmxlZXRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RBc3NvY2lhdGVkRmxlZXRzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0QXNzb2NpYXRlZEZsZWV0cycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIHRoZSBuYW1lIG9mIHRoZSBzdGFjayB3aXRoIHdoaWNoIHRoZSBzcGVjaWZpZWQgZmxlZXQgaXMgYXNzb2NpYXRlZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdEFzc29jaWF0ZWRTdGFja3MuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEFzc29jaWF0ZWRTdGFja3MoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RBc3NvY2lhdGVkU3RhY2tzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgdGhlIGFwcGxpY2F0aW9ucyB0aGF0IGFyZSBhc3NvY2lhdGVkIHdpdGggdGhlIHNwZWNpZmllZCBlbnRpdGxlbWVudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdEVudGl0bGVkQXBwbGljYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RFbnRpdGxlZEFwcGxpY2F0aW9ucygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdEVudGl0bGVkQXBwbGljYXRpb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBsaXN0IG9mIGFsbCB0YWdzIGZvciB0aGUgc3BlY2lmaWVkIEFwcFN0cmVhbSAyLjAgcmVzb3VyY2UuIFRoZSBmb2xsb3dpbmcgcmVzb3VyY2VzIGNhbiBiZSB0YWdnZWQ6IEltYWdlIGJ1aWxkZXJzLCBpbWFnZXMsIGZsZWV0cywgYW5kIHN0YWNrc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdFRhZ3NGb3JSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0VGFnc0ZvclJlc291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0VGFnc0ZvclJlc291cmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3RhcnQgdGhlIHNwZWNpZmllZCBhcHAgYmxvY2sgYnVpbGRlclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3RyZWFtMi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TdGFydEFwcEJsb2NrQnVpbGRlci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdGFydEFwcEJsb2NrQnVpbGRlcigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU3RhcnRBcHBCbG9ja0J1aWxkZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzdGFydCB0aGUgc3BlY2lmaWVkIGZsZWV0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzdHJlYW0yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0YXJ0RmxlZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU3RhcnRGbGVldCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU3RhcnRGbGVldCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHN0YXJ0IHRoZSBzcGVjaWZpZWQgaW1hZ2UgYnVpbGRlclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3RyZWFtMi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TdGFydEltYWdlQnVpbGRlci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdGFydEltYWdlQnVpbGRlcigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU3RhcnRJbWFnZUJ1aWxkZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzdG9wIHRoZSBzcGVjaWZpZWQgYXBwIGJsb2NrIGJ1aWxkZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3RvcEFwcEJsb2NrQnVpbGRlci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdG9wQXBwQmxvY2tCdWlsZGVyKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdG9wQXBwQmxvY2tCdWlsZGVyJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3RvcCB0aGUgc3BlY2lmaWVkIGZsZWV0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzdHJlYW0yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0b3BGbGVldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdG9wRmxlZXQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1N0b3BGbGVldCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHN0b3AgdGhlIHNwZWNpZmllZCBpbWFnZSBidWlsZGVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzdHJlYW0yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0b3BJbWFnZUJ1aWxkZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU3RvcEltYWdlQnVpbGRlcigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU3RvcEltYWdlQnVpbGRlcicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGZlZGVyYXRlZCB1c2VycyB0byBzaWduIGluIGJ5IHVzaW5nIHRoZWlyIGV4aXN0aW5nIGNyZWRlbnRpYWxzIGFuZCBzdHJlYW0gYXBwbGljYXRpb25zIGZyb20gdGhlIHNwZWNpZmllZCBzdGFja1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmVXNlcklkKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2V4dGVybmFsLWlkZW50aXR5LXByb3ZpZGVycy1zZXR0aW5nLXVwLXNhbWwuaHRtbCNleHRlcm5hbC1pZGVudGl0eS1wcm92aWRlcnMtZW1iZWQtaW5saW5lLXBvbGljeS1mb3ItSUFNLXJvbGVcbiAgICovXG4gIHB1YmxpYyB0b1N0cmVhbSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU3RyZWFtJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYWRkIG9yIG92ZXJ3cml0ZSBvbmUgb3IgbW9yZSB0YWdzIGZvciB0aGUgc3BlY2lmaWVkIEFwcFN0cmVhbSAyLjAgcmVzb3VyY2UuIFRoZSBmb2xsb3dpbmcgcmVzb3VyY2VzIGNhbiBiZSB0YWdnZWQ6IEltYWdlIGJ1aWxkZXJzLCBpbWFnZXMsIGZsZWV0cywgc3RhY2tzLCBhcHAgYmxvY2tzIGFuZCBhcHBsaWNhdGlvbnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3RyZWFtMi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9UYWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9UYWdSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVGFnUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkaXNhc3NvY2lhdGUgb25lIG9yIG1vcmUgdGFncyBmcm9tIHRoZSBzcGVjaWZpZWQgQXBwU3RyZWFtIDIuMCByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVW50YWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VbnRhZ1Jlc291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVbnRhZ1Jlc291cmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGEgc3BlY2lmaWMgYXBwIGJsb2NrIGJ1aWxkZXIuIEFuIGFwcCBibG9jayBidWlsZGVyIGlzIGEgdmlydHVhbCBtYWNoaW5lIHRoYXQgaXMgdXNlZCB0byBjcmVhdGUgYW4gYXBwIGJsb2NrXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzdHJlYW0yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUFwcEJsb2NrQnVpbGRlci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVBcHBCbG9ja0J1aWxkZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUFwcEJsb2NrQnVpbGRlcicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSB0aGUgc3BlY2lmaWVkIGZpZWxkcyBmb3IgdGhlIHNwZWNpZmllZCBhcHBsaWNhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3RyZWFtMi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVBcHBsaWNhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVBcHBsaWNhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlQXBwbGljYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgdGhlIHNwZWNpZmllZCBEaXJlY3RvcnkgQ29uZmlnIG9iamVjdCBpbiBBcHBTdHJlYW0gMi4wLiBUaGlzIG9iamVjdCBpbmNsdWRlcyB0aGUgY29uZmlndXJhdGlvbiBpbmZvcm1hdGlvbiByZXF1aXJlZCB0byBqb2luIGZsZWV0cyBhbmQgaW1hZ2UgYnVpbGRlcnMgdG8gTWljcm9zb2Z0IEFjdGl2ZSBEaXJlY3RvcnkgZG9tYWluc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzdHJlYW0yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZURpcmVjdG9yeUNvbmZpZy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVEaXJlY3RvcnlDb25maWcoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZURpcmVjdG9yeUNvbmZpZycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSB0aGUgc3BlY2lmaWVkIGZpZWxkcyBmb3IgdGhlIHNwZWNpZmllZCBlbnRpdGxlbWVudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzdHJlYW0yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUVudGl0bGVtZW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUVudGl0bGVtZW50KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVFbnRpdGxlbWVudCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSB0aGUgc3BlY2lmaWVkIGZsZWV0LiBBbGwgYXR0cmlidXRlcyBleGNlcHQgdGhlIGZsZWV0IG5hbWUgY2FuIGJlIHVwZGF0ZWQgd2hlbiB0aGUgZmxlZXQgaXMgaW4gdGhlIFNUT1BQRUQgc3RhdGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlRmxlZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlRmxlZXQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUZsZWV0Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYWRkIG9yIHVwZGF0ZSBwZXJtaXNzaW9ucyBmb3IgdGhlIHNwZWNpZmllZCBwcml2YXRlIGltYWdlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzdHJlYW0yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUltYWdlUGVybWlzc2lvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlSW1hZ2VQZXJtaXNzaW9ucygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlSW1hZ2VQZXJtaXNzaW9ucycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSB0aGUgc3BlY2lmaWVkIGZpZWxkcyBmb3IgdGhlIHNwZWNpZmllZCBzdGFja1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3RyZWFtMi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVTdGFjay5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVTdGFjaygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlU3RhY2snKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgdGhlIGN1c3RvbSBicmFuZGluZyB0aGVtZSBpbmZvcm1hdGlvbiwgd2hpY2ggbWlnaHQgaW5jbHVkZXMgYSBjdXN0b20gbG9nbywgd2Vic2l0ZSBsaW5rcywgYW5kIG90aGVyIGJyYW5kaW5nIHRvIGRpc3BsYXkgdG8geW91ciB1c2Vyc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzdHJlYW0yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZVRoZW1lRm9yU3RhY2suaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlVGhlbWVGb3JTdGFjaygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlVGhlbWVGb3JTdGFjaycpO1xuICB9XG5cbiAgcHJvdGVjdGVkIGFjY2Vzc0xldmVsTGlzdDogQWNjZXNzTGV2ZWxMaXN0ID0ge1xuICAgIFdyaXRlOiBbXG4gICAgICAnQXNzb2NpYXRlQXBwQmxvY2tCdWlsZGVyQXBwQmxvY2snLFxuICAgICAgJ0Fzc29jaWF0ZUFwcGxpY2F0aW9uRmxlZXQnLFxuICAgICAgJ0Fzc29jaWF0ZUFwcGxpY2F0aW9uVG9FbnRpdGxlbWVudCcsXG4gICAgICAnQXNzb2NpYXRlRmxlZXQnLFxuICAgICAgJ0JhdGNoQXNzb2NpYXRlVXNlclN0YWNrJyxcbiAgICAgICdCYXRjaERpc2Fzc29jaWF0ZVVzZXJTdGFjaycsXG4gICAgICAnQ29weUltYWdlJyxcbiAgICAgICdDcmVhdGVBcHBCbG9jaycsXG4gICAgICAnQ3JlYXRlQXBwQmxvY2tCdWlsZGVyJyxcbiAgICAgICdDcmVhdGVBcHBCbG9ja0J1aWxkZXJTdHJlYW1pbmdVUkwnLFxuICAgICAgJ0NyZWF0ZUFwcGxpY2F0aW9uJyxcbiAgICAgICdDcmVhdGVEaXJlY3RvcnlDb25maWcnLFxuICAgICAgJ0NyZWF0ZUVudGl0bGVtZW50JyxcbiAgICAgICdDcmVhdGVGbGVldCcsXG4gICAgICAnQ3JlYXRlSW1hZ2VCdWlsZGVyJyxcbiAgICAgICdDcmVhdGVJbWFnZUJ1aWxkZXJTdHJlYW1pbmdVUkwnLFxuICAgICAgJ0NyZWF0ZVN0YWNrJyxcbiAgICAgICdDcmVhdGVTdHJlYW1pbmdVUkwnLFxuICAgICAgJ0NyZWF0ZVRoZW1lRm9yU3RhY2snLFxuICAgICAgJ0NyZWF0ZVVwZGF0ZWRJbWFnZScsXG4gICAgICAnQ3JlYXRlVXNhZ2VSZXBvcnRTdWJzY3JpcHRpb24nLFxuICAgICAgJ0NyZWF0ZVVzZXInLFxuICAgICAgJ0RlbGV0ZUFwcEJsb2NrJyxcbiAgICAgICdEZWxldGVBcHBCbG9ja0J1aWxkZXInLFxuICAgICAgJ0RlbGV0ZUFwcGxpY2F0aW9uJyxcbiAgICAgICdEZWxldGVEaXJlY3RvcnlDb25maWcnLFxuICAgICAgJ0RlbGV0ZUVudGl0bGVtZW50JyxcbiAgICAgICdEZWxldGVGbGVldCcsXG4gICAgICAnRGVsZXRlSW1hZ2UnLFxuICAgICAgJ0RlbGV0ZUltYWdlQnVpbGRlcicsXG4gICAgICAnRGVsZXRlSW1hZ2VQZXJtaXNzaW9ucycsXG4gICAgICAnRGVsZXRlU3RhY2snLFxuICAgICAgJ0RlbGV0ZVRoZW1lRm9yU3RhY2snLFxuICAgICAgJ0RlbGV0ZVVzYWdlUmVwb3J0U3Vic2NyaXB0aW9uJyxcbiAgICAgICdEZWxldGVVc2VyJyxcbiAgICAgICdEaXNhYmxlVXNlcicsXG4gICAgICAnRGlzYXNzb2NpYXRlQXBwQmxvY2tCdWlsZGVyQXBwQmxvY2snLFxuICAgICAgJ0Rpc2Fzc29jaWF0ZUFwcGxpY2F0aW9uRmxlZXQnLFxuICAgICAgJ0Rpc2Fzc29jaWF0ZUFwcGxpY2F0aW9uRnJvbUVudGl0bGVtZW50JyxcbiAgICAgICdEaXNhc3NvY2lhdGVGbGVldCcsXG4gICAgICAnRW5hYmxlVXNlcicsXG4gICAgICAnRXhwaXJlU2Vzc2lvbicsXG4gICAgICAnU3RhcnRBcHBCbG9ja0J1aWxkZXInLFxuICAgICAgJ1N0YXJ0RmxlZXQnLFxuICAgICAgJ1N0YXJ0SW1hZ2VCdWlsZGVyJyxcbiAgICAgICdTdG9wQXBwQmxvY2tCdWlsZGVyJyxcbiAgICAgICdTdG9wRmxlZXQnLFxuICAgICAgJ1N0b3BJbWFnZUJ1aWxkZXInLFxuICAgICAgJ1N0cmVhbScsXG4gICAgICAnVXBkYXRlQXBwQmxvY2tCdWlsZGVyJyxcbiAgICAgICdVcGRhdGVBcHBsaWNhdGlvbicsXG4gICAgICAnVXBkYXRlRGlyZWN0b3J5Q29uZmlnJyxcbiAgICAgICdVcGRhdGVFbnRpdGxlbWVudCcsXG4gICAgICAnVXBkYXRlRmxlZXQnLFxuICAgICAgJ1VwZGF0ZUltYWdlUGVybWlzc2lvbnMnLFxuICAgICAgJ1VwZGF0ZVN0YWNrJyxcbiAgICAgICdVcGRhdGVUaGVtZUZvclN0YWNrJ1xuICAgIF0sXG4gICAgTGlzdDogW1xuICAgICAgJ0Rlc2NyaWJlQXBwQmxvY2tCdWlsZGVyQXBwQmxvY2tBc3NvY2lhdGlvbnMnLFxuICAgICAgJ0Rlc2NyaWJlQXBwQmxvY2tCdWlsZGVycycsXG4gICAgICAnRGVzY3JpYmVBcHBCbG9ja3MnLFxuICAgICAgJ0Rlc2NyaWJlQXBwbGljYXRpb25GbGVldEFzc29jaWF0aW9ucycsXG4gICAgICAnRGVzY3JpYmVBcHBsaWNhdGlvbnMnLFxuICAgICAgJ0Rlc2NyaWJlRGlyZWN0b3J5Q29uZmlncycsXG4gICAgICAnRGVzY3JpYmVFbnRpdGxlbWVudHMnLFxuICAgICAgJ0Rlc2NyaWJlRmxlZXRzJyxcbiAgICAgICdEZXNjcmliZUltYWdlQnVpbGRlcnMnLFxuICAgICAgJ0Rlc2NyaWJlSW1hZ2VzJyxcbiAgICAgICdEZXNjcmliZVNlc3Npb25zJyxcbiAgICAgICdEZXNjcmliZVN0YWNrcycsXG4gICAgICAnRGVzY3JpYmVVc2FnZVJlcG9ydFN1YnNjcmlwdGlvbnMnLFxuICAgICAgJ0Rlc2NyaWJlVXNlclN0YWNrQXNzb2NpYXRpb25zJyxcbiAgICAgICdEZXNjcmliZVVzZXJzJyxcbiAgICAgICdMaXN0RW50aXRsZWRBcHBsaWNhdGlvbnMnXG4gICAgXSxcbiAgICBSZWFkOiBbXG4gICAgICAnRGVzY3JpYmVJbWFnZVBlcm1pc3Npb25zJyxcbiAgICAgICdEZXNjcmliZVRoZW1lRm9yU3RhY2snLFxuICAgICAgJ0xpc3RBc3NvY2lhdGVkRmxlZXRzJyxcbiAgICAgICdMaXN0QXNzb2NpYXRlZFN0YWNrcycsXG4gICAgICAnTGlzdFRhZ3NGb3JSZXNvdXJjZSdcbiAgICBdLFxuICAgIFRhZ2dpbmc6IFtcbiAgICAgICdUYWdSZXNvdXJjZScsXG4gICAgICAnVW50YWdSZXNvdXJjZSdcbiAgICBdXG4gIH07XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGZsZWV0IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3doYXQtaXMtYXBwc3RyZWFtLmh0bWwjd2hhdC1pcy1jb25jZXB0c1xuICAgKlxuICAgKiBAcGFyYW0gZmxlZXROYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGZsZWV0TmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRmxlZXQoZmxlZXROYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBwc3RyZWFtOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpmbGVldC8keyBmbGVldE5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGltYWdlIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3doYXQtaXMtYXBwc3RyZWFtLmh0bWwjd2hhdC1pcy1jb25jZXB0c1xuICAgKlxuICAgKiBAcGFyYW0gaW1hZ2VOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGltYWdlTmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uSW1hZ2UoaW1hZ2VOYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBwc3RyZWFtOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTppbWFnZS8keyBpbWFnZU5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGltYWdlLWJ1aWxkZXIgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3RyZWFtMi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvd2hhdC1pcy1hcHBzdHJlYW0uaHRtbCN3aGF0LWlzLWNvbmNlcHRzXG4gICAqXG4gICAqIEBwYXJhbSBpbWFnZUJ1aWxkZXJOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGltYWdlQnVpbGRlck5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkltYWdlQnVpbGRlcihpbWFnZUJ1aWxkZXJOYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBwc3RyZWFtOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTppbWFnZS1idWlsZGVyLyR7IGltYWdlQnVpbGRlck5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHN0YWNrIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN0cmVhbTIvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3doYXQtaXMtYXBwc3RyZWFtLmh0bWwjd2hhdC1pcy1jb25jZXB0c1xuICAgKlxuICAgKiBAcGFyYW0gc3RhY2tOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIHN0YWNrTmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uU3RhY2soc3RhY2tOYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBwc3RyZWFtOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpzdGFjay8keyBzdGFja05hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGFwcC1ibG9jayB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzdHJlYW0yL2xhdGVzdC9kZXZlbG9wZXJndWlkZS93aGF0LWlzLWFwcHN0cmVhbS5odG1sI3doYXQtaXMtY29uY2VwdHNcbiAgICpcbiAgICogQHBhcmFtIGFwcEJsb2NrTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBhcHBCbG9ja05hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkFwcEJsb2NrKGFwcEJsb2NrTmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwcHN0cmVhbTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06YXBwLWJsb2NrLyR7IGFwcEJsb2NrTmFtZSB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgYXBwbGljYXRpb24gdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3RyZWFtMi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvd2hhdC1pcy1hcHBzdHJlYW0uaHRtbCN3aGF0LWlzLWNvbmNlcHRzXG4gICAqXG4gICAqIEBwYXJhbSBhcHBsaWNhdGlvbk5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgYXBwbGljYXRpb25OYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25BcHBsaWNhdGlvbihhcHBsaWNhdGlvbk5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcHBzdHJlYW06JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmFwcGxpY2F0aW9uLyR7IGFwcGxpY2F0aW9uTmFtZSB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgYXBwLWJsb2NrLWJ1aWxkZXIgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3RyZWFtMi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvd2hhdC1pcy1hcHBzdHJlYW0uaHRtbCN3aGF0LWlzLWNvbmNlcHRzXG4gICAqXG4gICAqIEBwYXJhbSBhcHBCbG9ja0J1aWxkZXJOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGFwcEJsb2NrQnVpbGRlck5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkFwcEJsb2NrQnVpbGRlcihhcHBCbG9ja0J1aWxkZXJOYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBwc3RyZWFtOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTphcHAtYmxvY2stYnVpbGRlci8keyBhcHBCbG9ja0J1aWxkZXJOYW1lIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgSUQgb2YgdGhlIEFwcFN0cmVhbSAyLjAgdXNlclxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3RyZWFtMi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvZXh0ZXJuYWwtaWRlbnRpdHktcHJvdmlkZXJzLXNldHRpbmctdXAtc2FtbC5odG1sI2V4dGVybmFsLWlkZW50aXR5LXByb3ZpZGVycy1lbWJlZC1pbmxpbmUtcG9saWN5LWZvci1JQU0tcm9sZVxuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvU3RyZWFtKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZVc2VySWQodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYHVzZXJJZGAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSBwcmVzZW5jZSBvZiB0YWcga2V5LXZhbHVlIHBhaXJzIGluIHRoZSByZXF1ZXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1yZXF1ZXN0dGFnXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9DcmVhdGVBcHBCbG9jaygpXG4gICAqIC0gLnRvQ3JlYXRlQXBwQmxvY2tCdWlsZGVyKClcbiAgICogLSAudG9DcmVhdGVBcHBsaWNhdGlvbigpXG4gICAqIC0gLnRvQ3JlYXRlRmxlZXQoKVxuICAgKiAtIC50b0NyZWF0ZUltYWdlQnVpbGRlcigpXG4gICAqIC0gLnRvQ3JlYXRlU3RhY2soKVxuICAgKiAtIC50b0NyZWF0ZVVwZGF0ZWRJbWFnZSgpXG4gICAqIC0gLnRvVGFnUmVzb3VyY2UoKVxuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzUmVxdWVzdFRhZyh0YWdLZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpSZXF1ZXN0VGFnLyR7IHRhZ0tleSB9YCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIHRhZyBrZXktdmFsdWUgcGFpcnMgYXR0YWNoZWQgdG8gdGhlIHJlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1yZXNvdXJjZXRhZ1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQXNzb2NpYXRlQXBwQmxvY2tCdWlsZGVyQXBwQmxvY2soKVxuICAgKiAtIC50b0Fzc29jaWF0ZUFwcGxpY2F0aW9uRmxlZXQoKVxuICAgKiAtIC50b0Fzc29jaWF0ZUZsZWV0KClcbiAgICogLSAudG9CYXRjaEFzc29jaWF0ZVVzZXJTdGFjaygpXG4gICAqIC0gLnRvQmF0Y2hEaXNhc3NvY2lhdGVVc2VyU3RhY2soKVxuICAgKiAtIC50b0NvcHlJbWFnZSgpXG4gICAqIC0gLnRvQ3JlYXRlQXBwQmxvY2soKVxuICAgKiAtIC50b0NyZWF0ZUFwcEJsb2NrQnVpbGRlclN0cmVhbWluZ1VSTCgpXG4gICAqIC0gLnRvQ3JlYXRlQXBwbGljYXRpb24oKVxuICAgKiAtIC50b0NyZWF0ZUltYWdlQnVpbGRlclN0cmVhbWluZ1VSTCgpXG4gICAqIC0gLnRvQ3JlYXRlU3RyZWFtaW5nVVJMKClcbiAgICogLSAudG9DcmVhdGVVcGRhdGVkSW1hZ2UoKVxuICAgKiAtIC50b0RlbGV0ZUFwcEJsb2NrKClcbiAgICogLSAudG9EZWxldGVBcHBCbG9ja0J1aWxkZXIoKVxuICAgKiAtIC50b0RlbGV0ZUFwcGxpY2F0aW9uKClcbiAgICogLSAudG9EZWxldGVGbGVldCgpXG4gICAqIC0gLnRvRGVsZXRlSW1hZ2UoKVxuICAgKiAtIC50b0RlbGV0ZUltYWdlQnVpbGRlcigpXG4gICAqIC0gLnRvRGVsZXRlSW1hZ2VQZXJtaXNzaW9ucygpXG4gICAqIC0gLnRvRGVsZXRlU3RhY2soKVxuICAgKiAtIC50b0Rpc2Fzc29jaWF0ZUFwcEJsb2NrQnVpbGRlckFwcEJsb2NrKClcbiAgICogLSAudG9EaXNhc3NvY2lhdGVBcHBsaWNhdGlvbkZsZWV0KClcbiAgICogLSAudG9EaXNhc3NvY2lhdGVGbGVldCgpXG4gICAqIC0gLnRvU3RhcnRBcHBCbG9ja0J1aWxkZXIoKVxuICAgKiAtIC50b1N0YXJ0RmxlZXQoKVxuICAgKiAtIC50b1N0YXJ0SW1hZ2VCdWlsZGVyKClcbiAgICogLSAudG9TdG9wQXBwQmxvY2tCdWlsZGVyKClcbiAgICogLSAudG9TdG9wRmxlZXQoKVxuICAgKiAtIC50b1N0b3BJbWFnZUJ1aWxkZXIoKVxuICAgKiAtIC50b1RhZ1Jlc291cmNlKClcbiAgICogLSAudG9VcGRhdGVBcHBCbG9ja0J1aWxkZXIoKVxuICAgKiAtIC50b1VwZGF0ZUFwcGxpY2F0aW9uKClcbiAgICogLSAudG9VcGRhdGVGbGVldCgpXG4gICAqIC0gLnRvVXBkYXRlSW1hZ2VQZXJtaXNzaW9ucygpXG4gICAqIC0gLnRvVXBkYXRlU3RhY2soKVxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIGZsZWV0XG4gICAqIC0gaW1hZ2VcbiAgICogLSBpbWFnZS1idWlsZGVyXG4gICAqIC0gc3RhY2tcbiAgICogLSBhcHAtYmxvY2tcbiAgICogLSBhcHBsaWNhdGlvblxuICAgKiAtIGFwcC1ibG9jay1idWlsZGVyXG4gICAqXG4gICAqIEBwYXJhbSB0YWdLZXkgVGhlIHRhZyBrZXkgdG8gY2hlY2tcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NSZXNvdXJjZVRhZyh0YWdLZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpSZXNvdXJjZVRhZy8keyB0YWdLZXkgfWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSBwcmVzZW5jZSBvZiB0YWcga2V5cyBpbiB0aGUgcmVxdWVzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtdGFna2V5c1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQ3JlYXRlQXBwQmxvY2soKVxuICAgKiAtIC50b0NyZWF0ZUFwcEJsb2NrQnVpbGRlcigpXG4gICAqIC0gLnRvQ3JlYXRlQXBwbGljYXRpb24oKVxuICAgKiAtIC50b0NyZWF0ZUZsZWV0KClcbiAgICogLSAudG9DcmVhdGVJbWFnZUJ1aWxkZXIoKVxuICAgKiAtIC50b0NyZWF0ZVN0YWNrKClcbiAgICogLSAudG9DcmVhdGVVcGRhdGVkSW1hZ2UoKVxuICAgKiAtIC50b1RhZ1Jlc291cmNlKClcbiAgICogLSAudG9VbnRhZ1Jlc291cmNlKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NUYWdLZXlzKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6VGFnS2V5c2AsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbYXBwc3RyZWFtXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hbWF6b25hcHBzdHJlYW0yLjAuaHRtbCkuXG4gICAqXG4gICAqL1xuICBjb25zdHJ1Y3Rvcihwcm9wcz86IGlhbS5Qb2xpY3lTdGF0ZW1lbnRQcm9wcykge1xuICAgIHN1cGVyKHByb3BzKTtcbiAgfVxufVxuIl19