"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ArcRegionSwitch = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [arc-region-switch](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonarcregionswitch.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class ArcRegionSwitch extends shared_1.PolicyStatement {
    /**
     * Grants permission to approve a plan execution step
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/arc-region-switch/latest/api/API_ApprovePlanExecutionStep.html
     */
    toApprovePlanExecutionStep() {
        return this.to('ApprovePlanExecutionStep');
    }
    /**
     * Grants permission to cancel a plan execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/arc-region-switch/latest/api/API_CancelPlanExecution.html
     */
    toCancelPlanExecution() {
        return this.to('CancelPlanExecution');
    }
    /**
     * Grants permission to create a plan
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/arc-region-switch/latest/api/API_CreatePlan.html
     */
    toCreatePlan() {
        return this.to('CreatePlan');
    }
    /**
     * Grants permission to delete a plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/arc-region-switch/latest/api/API_DeletePlan.html
     */
    toDeletePlan() {
        return this.to('DeletePlan');
    }
    /**
     * Grants permission to delete the RAM access control policy for a plan
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/r53recovery/latest/dg/arc-region-switch.region-switch-different-accounts.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to get information about plans in all AWS Regions using a control plane
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/arc-region-switch/latest/api/API_GetPlan.html
     */
    toGetPlan() {
        return this.to('GetPlan');
    }
    /**
     * Grants permission to get a plan's evaluation status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/arc-region-switch/latest/api/API_GetPlanEvaluationStatus.html
     */
    toGetPlanEvaluationStatus() {
        return this.to('GetPlanEvaluationStatus');
    }
    /**
     * Grants permission to get plan execution details and setup information
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/arc-region-switch/latest/api/API_GetPlanExecution.html
     */
    toGetPlanExecution() {
        return this.to('GetPlanExecution');
    }
    /**
     * Grants permission to get information about a plan in a specific AWS Region using the Region switch Regional data plane
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/arc-region-switch/latest/api/API_GetPlanInRegion.html
     */
    toGetPlanInRegion() {
        return this.to('GetPlanInRegion');
    }
    /**
     * Grants permission to get the resource policy of a plan
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/r53recovery/latest/dg/arc-region-switch.region-switch-different-accounts.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to list plan execution events
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/arc-region-switch/latest/api/API_ListPlanExecutionEvents.html
     */
    toListPlanExecutionEvents() {
        return this.to('ListPlanExecutionEvents');
    }
    /**
     * Grants permission to list plan executions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/arc-region-switch/latest/api/API_ListPlanExecutions.html
     */
    toListPlanExecutions() {
        return this.to('ListPlanExecutions');
    }
    /**
     * Grants permission to list plans in all AWS Regions using a control plane
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/arc-region-switch/latest/api/API_ListPlans.html
     */
    toListPlans() {
        return this.to('ListPlans');
    }
    /**
     * Grants permission to list plans in a specific AWS Region using the Region switch Regional data plane
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/arc-region-switch/latest/api/API_ListPlansInRegion.html
     */
    toListPlansInRegion() {
        return this.to('ListPlansInRegion');
    }
    /**
     * Grants permission to list Route 53 health checks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/arc-region-switch/latest/api/API_ListRoute53HealthChecks.html
     */
    toListRoute53HealthChecks() {
        return this.to('ListRoute53HealthChecks');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/arc-region-switch/latest/api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to define the RAM access control policy for a plan
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/r53recovery/latest/dg/arc-region-switch.region-switch-different-accounts.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to start a plan execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/arc-region-switch/latest/api/API_StartPlanExecution.html
     */
    toStartPlanExecution() {
        return this.to('StartPlanExecution');
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/arc-region-switch/latest/api/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/arc-region-switch/latest/api/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/arc-region-switch/latest/api/API_UpdatePlan.html
     */
    toUpdatePlan() {
        return this.to('UpdatePlan');
    }
    /**
     * Grants permission to update a plan execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/arc-region-switch/latest/api/API_UpdatePlanExecution.html
     */
    toUpdatePlanExecution() {
        return this.to('UpdatePlanExecution');
    }
    /**
     * Grants permission to update a plan execution step
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/arc-region-switch/latest/api/API_UpdatePlanExecutionStep.html
     */
    toUpdatePlanExecutionStep() {
        return this.to('UpdatePlanExecutionStep');
    }
    /**
     * Adds a resource of type plan to the statement
     *
     * https://docs.aws.amazon.com/r53recovery/latest/dg/plan.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPlan(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:arc-region-switch:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:plan/${resourceId}`);
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreatePlan()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a tag's key and value in a request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toListTagsForResource()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - plan
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreatePlan()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [arc-region-switch](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonarcregionswitch.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'arc-region-switch';
        this.accessLevelList = {
            Write: [
                'ApprovePlanExecutionStep',
                'CancelPlanExecution',
                'CreatePlan',
                'DeletePlan',
                'StartPlanExecution',
                'UpdatePlan',
                'UpdatePlanExecution',
                'UpdatePlanExecutionStep'
            ],
            'Permissions management': [
                'DeleteResourcePolicy',
                'GetResourcePolicy',
                'PutResourcePolicy'
            ],
            Read: [
                'GetPlan',
                'GetPlanEvaluationStatus',
                'GetPlanExecution',
                'GetPlanInRegion',
                'ListTagsForResource'
            ],
            List: [
                'ListPlanExecutionEvents',
                'ListPlanExecutions',
                'ListPlans',
                'ListPlansInRegion',
                'ListRoute53HealthChecks'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.ArcRegionSwitch = ArcRegionSwitch;
//# sourceMappingURL=data:application/json;base64,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