"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Artifact = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [artifact](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsartifact.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Artifact extends shared_1.PolicyStatement {
    /**
     * Grants permission to accept an AWS agreement that has not yet been accepted by the customer account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_AcceptAgreement.html
     */
    toAcceptAgreement() {
        return this.to('AcceptAgreement');
    }
    /**
     * Grants permission to accept the terms of an NDA Document for a given agreement resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_AcceptNdaForAgreement.html
     */
    toAcceptNdaForAgreement() {
        return this.to('AcceptNdaForAgreement');
    }
    /**
     * Grants permission to get the account settings for Artifact
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_GetAccountSettings.html
     */
    toGetAccountSettings() {
        return this.to('GetAccountSettings');
    }
    /**
     * Grants permission to get an AWS agreement that has not yet been accepted by the customer account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_GetAgreement.html
     */
    toGetAgreement() {
        return this.to('GetAgreement');
    }
    /**
     * Grants permission to get an AWS agreement that has been accepted by the customer account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_GetCustomerAgreement.html
     */
    toGetCustomerAgreement() {
        return this.to('GetCustomerAgreement');
    }
    /**
     * Grants permission to retrieve the NDA Document for a given agreement resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_GetNdaForAgreement.html
     */
    toGetNdaForAgreement() {
        return this.to('GetNdaForAgreement');
    }
    /**
     * Grants permission to download a report
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_GetReport.html
     */
    toGetReport() {
        return this.to('GetReport');
    }
    /**
     * Grants permission to download metadata associated with a report
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_GetReportMetadata.html
     */
    toGetReportMetadata() {
        return this.to('GetReportMetadata');
    }
    /**
     * Grants permission to download a term associated with a report
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_GetTermForReport.html
     */
    toGetTermForReport() {
        return this.to('GetTermForReport');
    }
    /**
     * Grants permission to list AWS agreements
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_ListAgreements.html
     */
    toListAgreements() {
        return this.to('ListAgreements');
    }
    /**
     * Grants permission to list customer-agreement resources that have been accepted by the customer account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_ListCustomerAgreements.html
     */
    toListCustomerAgreements() {
        return this.to('ListCustomerAgreements');
    }
    /**
     * Grants permission to list reports in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_ListReports.html
     */
    toListReports() {
        return this.to('ListReports');
    }
    /**
     * Grants permission to put account settings for Artifact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_PutAccountSettings.html
     */
    toPutAccountSettings() {
        return this.to('PutAccountSettings');
    }
    /**
     * Grants permission to terminate a customer agreement that was previously accepted by the customer account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_TerminateAgreement.html
     */
    toTerminateAgreement() {
        return this.to('TerminateAgreement');
    }
    /**
     * Adds a resource of type customer-agreement to the statement
     *
     * https://docs.aws.amazon.com/artifact/latest/ug/managing-agreements.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onCustomerAgreement(resourceName, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:artifact::${account ?? this.defaultAccount}:customer-agreement/${resourceName}`);
    }
    /**
     * Adds a resource of type agreement to the statement
     *
     * https://docs.aws.amazon.com/artifact/latest/ug/managing-agreements.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAgreement(resourceName, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:artifact:::agreement/${resourceName}`);
    }
    /**
     * Adds a resource of type report to the statement
     *
     * https://docs.aws.amazon.com/artifact/latest/ug/what-is-aws-artifact.html
     *
     * @param reportId - Identifier for the reportId.
     * @param version - Identifier for the version.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifReportCategory()
     * - .ifReportSeries()
     */
    onReport(reportId, version, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:artifact:${region ?? this.defaultRegion}::report/${reportId}:${version}`);
    }
    /**
     * Filters access by which category reports are associated with
     *
     * https://docs.aws.amazon.com/artifact/latest/ug/using-condition-keys.html
     *
     * Applies to resource types:
     * - report
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReportCategory(value, operator) {
        return this.if(`ReportCategory`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by which series reports are associated with
     *
     * https://docs.aws.amazon.com/artifact/latest/ug/using-condition-keys.html
     *
     * Applies to resource types:
     * - report
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReportSeries(value, operator) {
        return this.if(`ReportSeries`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [artifact](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsartifact.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'artifact';
        this.accessLevelList = {
            Write: [
                'AcceptAgreement',
                'AcceptNdaForAgreement',
                'PutAccountSettings',
                'TerminateAgreement'
            ],
            Read: [
                'GetAccountSettings',
                'GetAgreement',
                'GetCustomerAgreement',
                'GetNdaForAgreement',
                'GetReport',
                'GetReportMetadata',
                'GetTermForReport'
            ],
            List: [
                'ListAgreements',
                'ListCustomerAgreements',
                'ListReports'
            ]
        };
    }
}
exports.Artifact = Artifact;
//# sourceMappingURL=data:application/json;base64,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