"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Athena = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [athena](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonathena.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Athena extends shared_1.PolicyStatement {
    /**
     * Grants permission to get information about one or more named queries
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_BatchGetNamedQuery.html
     */
    toBatchGetNamedQuery() {
        return this.to('BatchGetNamedQuery');
    }
    /**
     * Grants permission to get information about one or more prepared statements
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_BatchGetPreparedStatement.html
     */
    toBatchGetPreparedStatement() {
        return this.to('BatchGetPreparedStatement');
    }
    /**
     * Grants permission to get information about one or more query executions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_BatchGetQueryExecution.html
     */
    toBatchGetQueryExecution() {
        return this.to('BatchGetQueryExecution');
    }
    /**
     * Grants permission to cancel a capacity reservation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_CancelCapacityReservation.html
     */
    toCancelCapacityReservation() {
        return this.to('CancelCapacityReservation');
    }
    /**
     * Grants permission to cancel query execution. Deprecated. Applies only to AWS services and principals that use Athena JDBC driver earlier than 1.1.0. Use StopQueryExecution otherwise
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_StopQueryExecution.html
     */
    toCancelQueryExecution() {
        return this.to('CancelQueryExecution');
    }
    /**
     * Grants permission to create a capacity reservation
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_CreateCapacityReservation.html
     */
    toCreateCapacityReservation() {
        return this.to('CreateCapacityReservation');
    }
    /**
     * Grants permission to create a datacatalog
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_CreateDataCatalog.html
     */
    toCreateDataCatalog() {
        return this.to('CreateDataCatalog');
    }
    /**
     * Grants permission to create a named query
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_CreateNamedQuery.html
     */
    toCreateNamedQuery() {
        return this.to('CreateNamedQuery');
    }
    /**
     * Grants permission to create a notebook
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_CreateNotebook.html
     */
    toCreateNotebook() {
        return this.to('CreateNotebook');
    }
    /**
     * Grants permission to create a prepared statement
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_CreatePreparedStatement.html
     */
    toCreatePreparedStatement() {
        return this.to('CreatePreparedStatement');
    }
    /**
     * Grants permission to create a presigned notebook url
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_CreatePresignedNotebookUrl.html
     */
    toCreatePresignedNotebookUrl() {
        return this.to('CreatePresignedNotebookUrl');
    }
    /**
     * Grants permission to create a workgroup
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_CreateWorkGroup.html
     */
    toCreateWorkGroup() {
        return this.to('CreateWorkGroup');
    }
    /**
     * Grants permission to delete a capacity reservation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_DeleteCapacityReservation.html
     */
    toDeleteCapacityReservation() {
        return this.to('DeleteCapacityReservation');
    }
    /**
     * Grants permission to delete a datacatalog
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_DeleteDataCatalog.html
     */
    toDeleteDataCatalog() {
        return this.to('DeleteDataCatalog');
    }
    /**
     * Grants permission to delete a named query specified
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_DeleteNamedQuery.html
     */
    toDeleteNamedQuery() {
        return this.to('DeleteNamedQuery');
    }
    /**
     * Grants permission to delete a notebook
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_DeleteNotebook.html
     */
    toDeleteNotebook() {
        return this.to('DeleteNotebook');
    }
    /**
     * Grants permission to delete a prepared statement specified
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_DeletePreparedStatement.html
     */
    toDeletePreparedStatement() {
        return this.to('DeletePreparedStatement');
    }
    /**
     * Grants permission to delete a workgroup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_DeleteWorkGroup.html
     */
    toDeleteWorkGroup() {
        return this.to('DeleteWorkGroup');
    }
    /**
     * Grants permission to export a notebook
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ExportNotebook.html
     */
    toExportNotebook() {
        return this.to('ExportNotebook');
    }
    /**
     * Grants permission to get a calculation execution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetCalculationExecution.html
     */
    toGetCalculationExecution() {
        return this.to('GetCalculationExecution');
    }
    /**
     * Grants permission to get a calculation execution code
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetCalculationExecutionCode.html
     */
    toGetCalculationExecutionCode() {
        return this.to('GetCalculationExecutionCode');
    }
    /**
     * Grants permission to get a calculation execution status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetCalculationExecutionStatus.html
     */
    toGetCalculationExecutionStatus() {
        return this.to('GetCalculationExecutionStatus');
    }
    /**
     * Grants permission to get capacity assignment information for a capacity reservation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetCapacityAssignmentConfiguration.html
     */
    toGetCapacityAssignmentConfiguration() {
        return this.to('GetCapacityAssignmentConfiguration');
    }
    /**
     * Grants permission to get a capacity reservation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetCapacityReservation.html
     */
    toGetCapacityReservation() {
        return this.to('GetCapacityReservation');
    }
    /**
     * Grants permission to enable access to databases and tables. Applies only to AWS services managed policy and principals that use an Athena JDBC driver version 1.1.0
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/ug/connect-with-previous-jdbc.html#jdbc-prev-version-policies
     */
    toGetCatalogs() {
        return this.to('GetCatalogs');
    }
    /**
     * Grants permission to get a datacatalog
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetDataCatalog.html
     */
    toGetDataCatalog() {
        return this.to('GetDataCatalog');
    }
    /**
     * Grants permission to get a database for a given datacatalog
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetDatabase.html
     */
    toGetDatabase() {
        return this.to('GetDatabase');
    }
    /**
     * Grants permission to enable access to the specified database and table. Applies only to AWS services managed policy and principals that use an Athena JDBC driver version 1.1.0
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/ug/connect-with-previous-jdbc.html#jdbc-prev-version-policies
     */
    toGetExecutionEngine() {
        return this.to('GetExecutionEngine');
    }
    /**
     * Grants permission to enable access to databases and tables. Applies only to AWS services managed policy and principals that use an Athena JDBC driver version 1.1.0
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/ug/connect-with-previous-jdbc.html#jdbc-prev-version-policies
     */
    toGetExecutionEngines() {
        return this.to('GetExecutionEngines');
    }
    /**
     * Grants permission to get information about the specified named query
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetNamedQuery.html
     */
    toGetNamedQuery() {
        return this.to('GetNamedQuery');
    }
    /**
     * Grants permission to enable access to the specified database and table. Applies only to AWS services managed policy and principals that use an Athena JDBC driver version 1.1.0
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/ug/connect-with-previous-jdbc.html#jdbc-prev-version-policies
     */
    toGetNamespace() {
        return this.to('GetNamespace');
    }
    /**
     * Grants permission to enable access to databases and tables. Applies only to AWS services managed policy and principals that use an Athena JDBC driver version 1.1.0
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/ug/connect-with-previous-jdbc.html#jdbc-prev-version-policies
     */
    toGetNamespaces() {
        return this.to('GetNamespaces');
    }
    /**
     * Grants permission to get notebook metadata
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetNotebookMetadata.html
     */
    toGetNotebookMetadata() {
        return this.to('GetNotebookMetadata');
    }
    /**
     * Grants permission to get information about the specified prepared statement
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetPreparedStatement.html
     */
    toGetPreparedStatement() {
        return this.to('GetPreparedStatement');
    }
    /**
     * Grants permission to get information about the specified query execution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetQueryExecution.html
     */
    toGetQueryExecution() {
        return this.to('GetQueryExecution');
    }
    /**
     * Grants permission to get query executions. Deprecated. Applies only to AWS services and principals that use Athena JDBC driver earlier than 1.1.0. Use ListQueryExecutions otherwise
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListQueryExecutions.html
     */
    toGetQueryExecutions() {
        return this.to('GetQueryExecutions');
    }
    /**
     * Grants permission to get the query results
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetQueryResults.html
     */
    toGetQueryResults() {
        return this.to('GetQueryResults');
    }
    /**
     * Grants permission to get the query results stream
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/ug/jdbc-v3-driver.html#jdbc-v3-driver-download
     */
    toGetQueryResultsStream() {
        return this.to('GetQueryResultsStream');
    }
    /**
     * Grants permission to get runtime statistics for the specified query execution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetQueryRuntimeStatistics.html
     */
    toGetQueryRuntimeStatistics() {
        return this.to('GetQueryRuntimeStatistics');
    }
    /**
     * Grants permission to get a session
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetSession.html
     */
    toGetSession() {
        return this.to('GetSession');
    }
    /**
     * Grants permission to get a session status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetSessionStatus.html
     */
    toGetSessionStatus() {
        return this.to('GetSessionStatus');
    }
    /**
     * Grants permission to enable access to the specified table. Applies only to AWS services managed policy and principals that use an Athena JDBC driver version 1.1.0
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/ug/connect-with-previous-jdbc.html#jdbc-prev-version-policies
     */
    toGetTable() {
        return this.to('GetTable');
    }
    /**
     * Grants permission to get a metadata about a table for a given datacatalog
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetTableMetadata.html
     */
    toGetTableMetadata() {
        return this.to('GetTableMetadata');
    }
    /**
     * Grants permission to enable access to tables. Applies only to AWS services managed policy and principals that use an Athena JDBC driver version 1.1.0
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/ug/connect-with-previous-jdbc.html#jdbc-prev-version-policies
     */
    toGetTables() {
        return this.to('GetTables');
    }
    /**
     * Grants permission to get a workgroup
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetWorkGroup.html
     */
    toGetWorkGroup() {
        return this.to('GetWorkGroup');
    }
    /**
     * Grants permission to import a notebook
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ImportNotebook.html
     */
    toImportNotebook() {
        return this.to('ImportNotebook');
    }
    /**
     * Grants permission to return a list of ApplicationRuntimeIds
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListApplicationDPUSizes.html
     */
    toListApplicationDPUSizes() {
        return this.to('ListApplicationDPUSizes');
    }
    /**
     * Grants permission to return a list of calculation executions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListCalculationExecutions.html
     */
    toListCalculationExecutions() {
        return this.to('ListCalculationExecutions');
    }
    /**
     * Grants permission to return a list of capacity reservations for the specified AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListCapacityReservations.html
     */
    toListCapacityReservations() {
        return this.to('ListCapacityReservations');
    }
    /**
     * Grants permission to return a list of datacatalogs for the specified AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListDataCatalogs.html
     */
    toListDataCatalogs() {
        return this.to('ListDataCatalogs');
    }
    /**
     * Grants permission to return a list of databases for a given datacatalog
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListDatabases.html
     */
    toListDatabases() {
        return this.to('ListDatabases');
    }
    /**
     * Grants permission to return a list of athena engine versions for the specified AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListEngineVersions.html
     */
    toListEngineVersions() {
        return this.to('ListEngineVersions');
    }
    /**
     * Grants permission to return a list of executors
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListExecutors.html
     */
    toListExecutors() {
        return this.to('ListExecutors');
    }
    /**
     * Grants permission to return a list of named queries in Amazon Athena for the specified AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListNamedQueries.html
     */
    toListNamedQueries() {
        return this.to('ListNamedQueries');
    }
    /**
     * Grants permission to return a list of notebooks for a given workgroup
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListNotebookMetadata.html
     */
    toListNotebookMetadata() {
        return this.to('ListNotebookMetadata');
    }
    /**
     * Grants permission to return a list of sessions for a given notebook
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListNotebookSessions.html
     */
    toListNotebookSessions() {
        return this.to('ListNotebookSessions');
    }
    /**
     * Grants permission to return a list of prepared statements for the specified workgroup
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListPreparedStatements.html
     */
    toListPreparedStatements() {
        return this.to('ListPreparedStatements');
    }
    /**
     * Grants permission to return a list of query executions for the specified AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListQueryExecutions.html
     */
    toListQueryExecutions() {
        return this.to('ListQueryExecutions');
    }
    /**
     * Grants permission to return a list of sessions for a given workgroup
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListSessions.html
     */
    toListSessions() {
        return this.to('ListSessions');
    }
    /**
     * Grants permission to return a list of table metadata in a database for a given datacatalog
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListTableMetadata.html
     */
    toListTableMetadata() {
        return this.to('ListTableMetadata');
    }
    /**
     * Grants permission to return a list of tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to return a list of workgroups for the specified AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListWorkGroups.html
     */
    toListWorkGroups() {
        return this.to('ListWorkGroups');
    }
    /**
     * Grants permission to assign capacity from a capacity reservation to queries
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_PutCapacityAssignmentConfiguration.html
     */
    toPutCapacityAssignmentConfiguration() {
        return this.to('PutCapacityAssignmentConfiguration');
    }
    /**
     * Grants permission to run a query. Deprecated. Applies only to AWS services and principals that use Athena JDBC driver earlier than 1.1.0. Use StartQueryExecution otherwise
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_StartQueryExecution.html
     */
    toRunQuery() {
        return this.to('RunQuery');
    }
    /**
     * Grants permission to start a calculation execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_StartCalculationExecution.html
     */
    toStartCalculationExecution() {
        return this.to('StartCalculationExecution');
    }
    /**
     * Grants permission to start a query execution using an SQL query provided as a string
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_StartQueryExecution.html
     */
    toStartQueryExecution() {
        return this.to('StartQueryExecution');
    }
    /**
     * Grants permission to start a session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_StartSession.html
     */
    toStartSession() {
        return this.to('StartSession');
    }
    /**
     * Grants permission to stop a calculation execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_StopCalculationExecution.html
     */
    toStopCalculationExecution() {
        return this.to('StopCalculationExecution');
    }
    /**
     * Grants permission to stop the specified query execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_StopQueryExecution.html
     */
    toStopQueryExecution() {
        return this.to('StopQueryExecution');
    }
    /**
     * Grants permission to add a tag to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to terminate a session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_TerminateSession.html
     */
    toTerminateSession() {
        return this.to('TerminateSession');
    }
    /**
     * Grants permission to remove a tag from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a capacity reservation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_UpdateCapacityReservation.html
     */
    toUpdateCapacityReservation() {
        return this.to('UpdateCapacityReservation');
    }
    /**
     * Grants permission to update a datacatalog
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_UpdateDataCatalog.html
     */
    toUpdateDataCatalog() {
        return this.to('UpdateDataCatalog');
    }
    /**
     * Grants permission to update a named query specified
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_UpdateNamedQuery.html
     */
    toUpdateNamedQuery() {
        return this.to('UpdateNamedQuery');
    }
    /**
     * Grants permission to update a notebook
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_UpdateNotebook.html
     */
    toUpdateNotebook() {
        return this.to('UpdateNotebook');
    }
    /**
     * Grants permission to update notebook metadata
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_UpdateNotebookMetadata.html
     */
    toUpdateNotebookMetadata() {
        return this.to('UpdateNotebookMetadata');
    }
    /**
     * Grants permission to update a prepared statement
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_UpdatePreparedStatement.html
     */
    toUpdatePreparedStatement() {
        return this.to('UpdatePreparedStatement');
    }
    /**
     * Grants permission to update a workgroup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_UpdateWorkGroup.html
     */
    toUpdateWorkGroup() {
        return this.to('UpdateWorkGroup');
    }
    /**
     * Adds a resource of type datacatalog to the statement
     *
     * https://docs.aws.amazon.com/athena/latest/ug/datacatalogs-example-policies.html
     *
     * @param dataCatalogName - Identifier for the dataCatalogName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDatacatalog(dataCatalogName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:athena:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:datacatalog/${dataCatalogName}`);
    }
    /**
     * Adds a resource of type workgroup to the statement
     *
     * https://docs.aws.amazon.com/athena/latest/ug/example-policies-workgroup.html
     *
     * @param workGroupName - Identifier for the workGroupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkgroup(workGroupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:athena:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workgroup/${workGroupName}`);
    }
    /**
     * Adds a resource of type capacity-reservation to the statement
     *
     * https://docs.aws.amazon.com/athena/latest/ug/example-policies-capacity-reservations.html
     *
     * @param capacityReservationName - Identifier for the capacityReservationName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCapacityReservation(capacityReservationName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:athena:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:capacity-reservation/${capacityReservationName}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateCapacityReservation()
     * - .toCreateDataCatalog()
     * - .toCreateWorkGroup()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - datacatalog
     * - workgroup
     * - capacity-reservation
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateCapacityReservation()
     * - .toCreateDataCatalog()
     * - .toCreateWorkGroup()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [athena](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonathena.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'athena';
        this.accessLevelList = {
            Read: [
                'BatchGetNamedQuery',
                'BatchGetPreparedStatement',
                'BatchGetQueryExecution',
                'GetCalculationExecution',
                'GetCalculationExecutionCode',
                'GetCalculationExecutionStatus',
                'GetCapacityAssignmentConfiguration',
                'GetCapacityReservation',
                'GetCatalogs',
                'GetDataCatalog',
                'GetDatabase',
                'GetExecutionEngine',
                'GetExecutionEngines',
                'GetNamedQuery',
                'GetNamespace',
                'GetNamespaces',
                'GetNotebookMetadata',
                'GetPreparedStatement',
                'GetQueryExecution',
                'GetQueryExecutions',
                'GetQueryResults',
                'GetQueryResultsStream',
                'GetQueryRuntimeStatistics',
                'GetSession',
                'GetSessionStatus',
                'GetTable',
                'GetTableMetadata',
                'GetTables',
                'GetWorkGroup',
                'ListEngineVersions',
                'ListQueryExecutions',
                'ListTableMetadata',
                'ListTagsForResource'
            ],
            Write: [
                'CancelCapacityReservation',
                'CancelQueryExecution',
                'CreateCapacityReservation',
                'CreateDataCatalog',
                'CreateNamedQuery',
                'CreateNotebook',
                'CreatePreparedStatement',
                'CreatePresignedNotebookUrl',
                'CreateWorkGroup',
                'DeleteCapacityReservation',
                'DeleteDataCatalog',
                'DeleteNamedQuery',
                'DeleteNotebook',
                'DeletePreparedStatement',
                'DeleteWorkGroup',
                'ExportNotebook',
                'ImportNotebook',
                'PutCapacityAssignmentConfiguration',
                'RunQuery',
                'StartCalculationExecution',
                'StartQueryExecution',
                'StartSession',
                'StopCalculationExecution',
                'StopQueryExecution',
                'TerminateSession',
                'UpdateCapacityReservation',
                'UpdateDataCatalog',
                'UpdateNamedQuery',
                'UpdateNotebook',
                'UpdateNotebookMetadata',
                'UpdatePreparedStatement',
                'UpdateWorkGroup'
            ],
            List: [
                'ListApplicationDPUSizes',
                'ListCalculationExecutions',
                'ListCapacityReservations',
                'ListDataCatalogs',
                'ListDatabases',
                'ListExecutors',
                'ListNamedQueries',
                'ListNotebookMetadata',
                'ListNotebookSessions',
                'ListPreparedStatements',
                'ListSessions',
                'ListWorkGroups'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Athena = Athena;
//# sourceMappingURL=data:application/json;base64,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