"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Auditmanager = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [auditmanager](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsauditmanager.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Auditmanager extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate an evidence folder with an assessment report in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_AssociateAssessmentReportEvidenceFolder.html
     */
    toAssociateAssessmentReportEvidenceFolder() {
        return this.to('AssociateAssessmentReportEvidenceFolder');
    }
    /**
     * Grants permission to associate a list of evidence to an assessment report in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_BatchAssociateAssessmentReportEvidence.html
     */
    toBatchAssociateAssessmentReportEvidence() {
        return this.to('BatchAssociateAssessmentReportEvidence');
    }
    /**
     * Grants permission to create delegations for an assessment in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_BatchCreateDelegationByAssessment.html
     */
    toBatchCreateDelegationByAssessment() {
        return this.to('BatchCreateDelegationByAssessment');
    }
    /**
     * Grants permission to delete delegations for an assessment in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_BatchDeleteDelegationByAssessment.html
     */
    toBatchDeleteDelegationByAssessment() {
        return this.to('BatchDeleteDelegationByAssessment');
    }
    /**
     * Grants permission to disassociate a list of evidence from an assessment report in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_BatchDisassociateAssessmentReportEvidence.html
     */
    toBatchDisassociateAssessmentReportEvidence() {
        return this.to('BatchDisassociateAssessmentReportEvidence');
    }
    /**
     * Grants permission to import a list of evidence to an assessment control in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_BatchImportEvidenceToAssessmentControl.html
     */
    toBatchImportEvidenceToAssessmentControl() {
        return this.to('BatchImportEvidenceToAssessmentControl');
    }
    /**
     * Grants permission to create an assessment to be used with AWS Audit Manager
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_CreateAssessment.html
     */
    toCreateAssessment() {
        return this.to('CreateAssessment');
    }
    /**
     * Grants permission to create a framework for use in AWS Audit Manager
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_CreateAssessmentFramework.html
     */
    toCreateAssessmentFramework() {
        return this.to('CreateAssessmentFramework');
    }
    /**
     * Grants permission to create an assessment report in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_CreateAssessmentReport.html
     */
    toCreateAssessmentReport() {
        return this.to('CreateAssessmentReport');
    }
    /**
     * Grants permission to create a control to be used in AWS Audit Manager
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_CreateControl.html
     */
    toCreateControl() {
        return this.to('CreateControl');
    }
    /**
     * Grants permission to delete an assessment in AWS Audit Manager
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_DeleteAssessment.html
     */
    toDeleteAssessment() {
        return this.to('DeleteAssessment');
    }
    /**
     * Grants permission to delete an assessment framework in AWS Audit Manager
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_DeleteAssessmentFramework.html
     */
    toDeleteAssessmentFramework() {
        return this.to('DeleteAssessmentFramework');
    }
    /**
     * Grants permission to delete a share request for a custom framework in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_DeleteAssessmentFrameworkShare.html
     */
    toDeleteAssessmentFrameworkShare() {
        return this.to('DeleteAssessmentFrameworkShare');
    }
    /**
     * Grants permission to delete an assessment report in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_DeleteAssessmentReport.html
     */
    toDeleteAssessmentReport() {
        return this.to('DeleteAssessmentReport');
    }
    /**
     * Grants permission to delete a control in AWS Audit Manager
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_DeleteControl.html
     */
    toDeleteControl() {
        return this.to('DeleteControl');
    }
    /**
     * Grants permission to deregister an account in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_DeregisterAccount.html
     */
    toDeregisterAccount() {
        return this.to('DeregisterAccount');
    }
    /**
     * Grants permission to deregister the delegated administrator account for AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_DeregisterOrganizationAdminAccount.html
     */
    toDeregisterOrganizationAdminAccount() {
        return this.to('DeregisterOrganizationAdminAccount');
    }
    /**
     * Grants permission to disassociate an evidence folder from an assessment report in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_DisassociateAssessmentReportEvidenceFolder.html
     */
    toDisassociateAssessmentReportEvidenceFolder() {
        return this.to('DisassociateAssessmentReportEvidenceFolder');
    }
    /**
     * Grants permission to get the status of an account in AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetAccountStatus.html
     */
    toGetAccountStatus() {
        return this.to('GetAccountStatus');
    }
    /**
     * Grants permission to get an assessment created in AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetAssessment.html
     */
    toGetAssessment() {
        return this.to('GetAssessment');
    }
    /**
     * Grants permission to get an assessment framework in AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetAssessmentFramework.html
     */
    toGetAssessmentFramework() {
        return this.to('GetAssessmentFramework');
    }
    /**
     * Grants permission to get the URL for an assessment report in AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetAssessmentReportUrl.html
     */
    toGetAssessmentReportUrl() {
        return this.to('GetAssessmentReportUrl');
    }
    /**
     * Grants permission to get changelogs for an assessment in AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetChangeLogs.html
     */
    toGetChangeLogs() {
        return this.to('GetChangeLogs');
    }
    /**
     * Grants permission to get a control in AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetControl.html
     */
    toGetControl() {
        return this.to('GetControl');
    }
    /**
     * Grants permission to get all delegations in AWS Audit Manager
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetDelegations.html
     */
    toGetDelegations() {
        return this.to('GetDelegations');
    }
    /**
     * Grants permission to get evidence from AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetEvidence.html
     */
    toGetEvidence() {
        return this.to('GetEvidence');
    }
    /**
     * Grants permission to get all the evidence from an evidence folder in AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetEvidenceByEvidenceFolder.html
     */
    toGetEvidenceByEvidenceFolder() {
        return this.to('GetEvidenceByEvidenceFolder');
    }
    /**
     * Grants permission to get a presigned Amazon S3 URL that can be used to upload a file as manual evidence
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetEvidenceFileUploadUrl.html
     */
    toGetEvidenceFileUploadUrl() {
        return this.to('GetEvidenceFileUploadUrl');
    }
    /**
     * Grants permission to get the evidence folder from AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetEvidenceFolder.html
     */
    toGetEvidenceFolder() {
        return this.to('GetEvidenceFolder');
    }
    /**
     * Grants permission to get the evidence folders from an assessment in AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetEvidenceFoldersByAssessment.html
     */
    toGetEvidenceFoldersByAssessment() {
        return this.to('GetEvidenceFoldersByAssessment');
    }
    /**
     * Grants permission to get the evidence folders from an assessment control in AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetEvidenceFoldersByAssessmentControl.html
     */
    toGetEvidenceFoldersByAssessmentControl() {
        return this.to('GetEvidenceFoldersByAssessmentControl');
    }
    /**
     * Grants permission to get analytics data for all active assessments
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetInsights.html
     */
    toGetInsights() {
        return this.to('GetInsights');
    }
    /**
     * Grants permission to get analytics data for a specific active assessment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetInsightsByAssessment.html
     */
    toGetInsightsByAssessment() {
        return this.to('GetInsightsByAssessment');
    }
    /**
     * Grants permission to get the delegated administrator account in AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetOrganizationAdminAccount.html
     */
    toGetOrganizationAdminAccount() {
        return this.to('GetOrganizationAdminAccount');
    }
    /**
     * Grants permission to get the services in scope for an assessment in AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetServicesInScope.html
     */
    toGetServicesInScope() {
        return this.to('GetServicesInScope');
    }
    /**
     * Grants permission to get all settings configured in AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetSettings.html
     */
    toGetSettings() {
        return this.to('GetSettings');
    }
    /**
     * Grants permission to list analytics data for controls in a specific control domain and active assessment
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_ListAssessmentControlInsightsByControlDomain.html
     */
    toListAssessmentControlInsightsByControlDomain() {
        return this.to('ListAssessmentControlInsightsByControlDomain');
    }
    /**
     * Grants permission to list all sent or received share requests for custom frameworks in AWS Audit Manager
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_ListAssessmentFrameworkShareRequests.html
     */
    toListAssessmentFrameworkShareRequests() {
        return this.to('ListAssessmentFrameworkShareRequests');
    }
    /**
     * Grants permission to list all assessment frameworks in AWS Audit Manager
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_ListAssessmentFrameworks.html
     */
    toListAssessmentFrameworks() {
        return this.to('ListAssessmentFrameworks');
    }
    /**
     * Grants permission to list all assessment reports in AWS Audit Manager
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_ListAssessmentReports.html
     */
    toListAssessmentReports() {
        return this.to('ListAssessmentReports');
    }
    /**
     * Grants permission to list all assessments in AWS Audit Manager
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_ListAssessments.html
     */
    toListAssessments() {
        return this.to('ListAssessments');
    }
    /**
     * Grants permission to list analytics data for control domains across all active assessments
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_ListControlDomainInsights.html
     */
    toListControlDomainInsights() {
        return this.to('ListControlDomainInsights');
    }
    /**
     * Grants permission to list analytics data for control domains in a specific active assessment
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_ListControlDomainInsightsByAssessment.html
     */
    toListControlDomainInsightsByAssessment() {
        return this.to('ListControlDomainInsightsByAssessment');
    }
    /**
     * Grants permission to list analytics data for controls in a specific control domain across all active assessments
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_ListControlInsightsByControlDomain.html
     */
    toListControlInsightsByControlDomain() {
        return this.to('ListControlInsightsByControlDomain');
    }
    /**
     * Grants permission to list all controls in AWS Audit Manager
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_ListControls.html
     */
    toListControls() {
        return this.to('ListControls');
    }
    /**
     * Grants permission to list all the data source keywords in AWS Audit Manager
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_ListKeywordsForDataSource.html
     */
    toListKeywordsForDataSource() {
        return this.to('ListKeywordsForDataSource');
    }
    /**
     * Grants permission to list all notifications in AWS Audit Manager
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_ListNotifications.html
     */
    toListNotifications() {
        return this.to('ListNotifications');
    }
    /**
     * Grants permission to list tags for an AWS Audit Manager resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to register an account in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_RegisterAccount.html
     */
    toRegisterAccount() {
        return this.to('RegisterAccount');
    }
    /**
     * Grants permission to register an account within the organization as the delegated administrator for AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_RegisterOrganizationAdminAccount.html
     */
    toRegisterOrganizationAdminAccount() {
        return this.to('RegisterOrganizationAdminAccount');
    }
    /**
     * Grants permission to create a share request for a custom framework in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_StartAssessmentFrameworkShare.html
     */
    toStartAssessmentFrameworkShare() {
        return this.to('StartAssessmentFrameworkShare');
    }
    /**
     * Grants permission to tag an AWS Audit Manager resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag an AWS Audit Manager resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an assessment in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_UpdateAssessment.html
     */
    toUpdateAssessment() {
        return this.to('UpdateAssessment');
    }
    /**
     * Grants permission to update an assessment control in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_UpdateAssessmentControl.html
     */
    toUpdateAssessmentControl() {
        return this.to('UpdateAssessmentControl');
    }
    /**
     * Grants permission to update the status of an assessment control set in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_UpdateAssessmentControlSetStatus.html
     */
    toUpdateAssessmentControlSetStatus() {
        return this.to('UpdateAssessmentControlSetStatus');
    }
    /**
     * Grants permission to update an assessment framework in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_UpdateAssessmentFramework.html
     */
    toUpdateAssessmentFramework() {
        return this.to('UpdateAssessmentFramework');
    }
    /**
     * Grants permission to update a share request for a custom framework in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_UpdateAssessmentFrameworkShare.html
     */
    toUpdateAssessmentFrameworkShare() {
        return this.to('UpdateAssessmentFrameworkShare');
    }
    /**
     * Grants permission to update the status of an assessment in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_UpdateAssessmentStatus.html
     */
    toUpdateAssessmentStatus() {
        return this.to('UpdateAssessmentStatus');
    }
    /**
     * Grants permission to update a control in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_UpdateControl.html
     */
    toUpdateControl() {
        return this.to('UpdateControl');
    }
    /**
     * Grants permission to update settings in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_UpdateSettings.html
     */
    toUpdateSettings() {
        return this.to('UpdateSettings');
    }
    /**
     * Grants permission to validate the integrity of an assessment report in AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_ValidateAssessmentReportIntegrity.html
     */
    toValidateAssessmentReportIntegrity() {
        return this.to('ValidateAssessmentReportIntegrity');
    }
    /**
     * Adds a resource of type assessment to the statement
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_Assessment.html
     *
     * @param assessmentId - Identifier for the assessmentId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAssessment(assessmentId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:auditmanager:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:assessment/${assessmentId}`);
    }
    /**
     * Adds a resource of type assessmentFramework to the statement
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_AssessmentFramework.html
     *
     * @param assessmentFrameworkId - Identifier for the assessmentFrameworkId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAssessmentFramework(assessmentFrameworkId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:auditmanager:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:assessmentFramework/${assessmentFrameworkId}`);
    }
    /**
     * Adds a resource of type assessmentControlSet to the statement
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_AssessmentControlSet.html
     *
     * @param assessmentId - Identifier for the assessmentId.
     * @param controlSetId - Identifier for the controlSetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAssessmentControlSet(assessmentId, controlSetId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:auditmanager:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:assessment/${assessmentId}/controlSet/${controlSetId}`);
    }
    /**
     * Adds a resource of type control to the statement
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_Control.html
     *
     * @param controlId - Identifier for the controlId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onControl(controlId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:auditmanager:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:control/${controlId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAssessment()
     * - .toCreateAssessmentFramework()
     * - .toCreateControl()
     * - .toDeleteAssessment()
     * - .toDeleteAssessmentFramework()
     * - .toDeleteControl()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCreateAssessment()
     * - .toCreateAssessmentFramework()
     * - .toCreateControl()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - assessment
     * - assessmentFramework
     * - control
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAssessment()
     * - .toCreateAssessmentFramework()
     * - .toCreateControl()
     * - .toDeleteAssessment()
     * - .toDeleteAssessmentFramework()
     * - .toDeleteControl()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [auditmanager](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsauditmanager.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'auditmanager';
        this.accessLevelList = {
            Write: [
                'AssociateAssessmentReportEvidenceFolder',
                'BatchAssociateAssessmentReportEvidence',
                'BatchCreateDelegationByAssessment',
                'BatchDeleteDelegationByAssessment',
                'BatchDisassociateAssessmentReportEvidence',
                'BatchImportEvidenceToAssessmentControl',
                'CreateAssessment',
                'CreateAssessmentFramework',
                'CreateAssessmentReport',
                'CreateControl',
                'DeleteAssessment',
                'DeleteAssessmentFramework',
                'DeleteAssessmentFrameworkShare',
                'DeleteAssessmentReport',
                'DeleteControl',
                'DeregisterAccount',
                'DeregisterOrganizationAdminAccount',
                'DisassociateAssessmentReportEvidenceFolder',
                'RegisterAccount',
                'RegisterOrganizationAdminAccount',
                'StartAssessmentFrameworkShare',
                'UpdateAssessment',
                'UpdateAssessmentControl',
                'UpdateAssessmentControlSetStatus',
                'UpdateAssessmentFramework',
                'UpdateAssessmentFrameworkShare',
                'UpdateAssessmentStatus',
                'UpdateControl',
                'UpdateSettings'
            ],
            Read: [
                'GetAccountStatus',
                'GetAssessment',
                'GetAssessmentFramework',
                'GetAssessmentReportUrl',
                'GetChangeLogs',
                'GetControl',
                'GetEvidence',
                'GetEvidenceByEvidenceFolder',
                'GetEvidenceFileUploadUrl',
                'GetEvidenceFolder',
                'GetEvidenceFoldersByAssessment',
                'GetEvidenceFoldersByAssessmentControl',
                'GetInsights',
                'GetInsightsByAssessment',
                'GetOrganizationAdminAccount',
                'GetServicesInScope',
                'GetSettings',
                'ListTagsForResource',
                'ValidateAssessmentReportIntegrity'
            ],
            List: [
                'GetDelegations',
                'ListAssessmentControlInsightsByControlDomain',
                'ListAssessmentFrameworkShareRequests',
                'ListAssessmentFrameworks',
                'ListAssessmentReports',
                'ListAssessments',
                'ListControlDomainInsights',
                'ListControlDomainInsightsByAssessment',
                'ListControlInsightsByControlDomain',
                'ListControls',
                'ListKeywordsForDataSource',
                'ListNotifications'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Auditmanager = Auditmanager;
//# sourceMappingURL=data:application/json;base64,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