"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Dsql = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [dsql](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonauroradsql.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Dsql extends shared_1.PolicyStatement {
    /**
     * Grants permission to add a peer cluster to a multi-Region cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - dsql:PutMultiRegionProperties
     *
     * https://docs.aws.amazon.com/aurora-dsql/latest/APIReference/API_CreateCluster.html
     */
    toAddPeerCluster() {
        return this.to('AddPeerCluster');
    }
    /**
     * Grants permission to create new clusters
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifWitnessRegion()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/aurora-dsql/latest/APIReference/API_CreateCluster.html
     */
    toCreateCluster() {
        return this.to('CreateCluster');
    }
    /**
     * Grants permission to connect to the database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aurora-dsql/latest/userguide/accessing-sql-clients.html
     */
    toDbConnect() {
        return this.to('DbConnect');
    }
    /**
     * Grants permission to connect to the database with admin role. Connecting with any other role requires DbConnect permission
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aurora-dsql/latest/userguide/accessing-sql-clients.html
     */
    toDbConnectAdmin() {
        return this.to('DbConnectAdmin');
    }
    /**
     * Grants permission to delete a cluster and all of its data
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aurora-dsql/latest/APIReference/API_DeleteCluster.html
     */
    toDeleteCluster() {
        return this.to('DeleteCluster');
    }
    /**
     * Grants permission to get the status of an Aurora DSQL cluster backup job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aurora-dsql/latest/userguide/backup-aurora-dsql.html
     */
    toGetBackupJob() {
        return this.to('GetBackupJob');
    }
    /**
     * Grants permission to get information about a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aurora-dsql/latest/APIReference/API_GetCluster.html
     */
    toGetCluster() {
        return this.to('GetCluster');
    }
    /**
     * Grants permission to get the status of an Aurora DSQL cluster restore job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aurora-dsql/latest/userguide/backup-aurora-dsql.html
     */
    toGetRestoreJob() {
        return this.to('GetRestoreJob');
    }
    /**
     * Grants permission to retrieve the VPC endpoint service name for a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aurora-dsql/latest/APIReference/API_GetVpcEndpointServiceName.html
     */
    toGetVpcEndpointServiceName() {
        return this.to('GetVpcEndpointServiceName');
    }
    /**
     * Grants permission to inject errors in targeted clusters
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFisActionId()
     * - .ifFisTargetArns()
     *
     * https://docs.aws.amazon.com/fis/latest/userguide/fis-actions-reference.html
     */
    toInjectError() {
        return this.to('InjectError');
    }
    /**
     * Grants permission to retrieve a list of clusters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aurora-dsql/latest/APIReference/API_ListClusters.html
     */
    toListClusters() {
        return this.to('ListClusters');
    }
    /**
     * Grants permission to list all tags on an Aurora DSQL resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aurora-dsql/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to update multi-Region properties of a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aurora-dsql/latest/APIReference/API_CreateCluster.html
     */
    toPutMultiRegionProperties() {
        return this.to('PutMultiRegionProperties');
    }
    /**
     * Grants permission to configure and update the witness Region of a multi-Region cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifWitnessRegion()
     *
     * Dependent actions:
     * - dsql:PutMultiRegionProperties
     *
     * https://docs.aws.amazon.com/aurora-dsql/latest/APIReference/API_CreateCluster.html
     */
    toPutWitnessRegion() {
        return this.to('PutWitnessRegion');
    }
    /**
     * Grants permission to remove a peer cluster from a multi-Region cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - dsql:PutMultiRegionProperties
     *
     * https://docs.aws.amazon.com/aurora-dsql/latest/APIReference/API_UpdateCluster.html
     */
    toRemovePeerCluster() {
        return this.to('RemovePeerCluster');
    }
    /**
     * Grants permission to start a backup job for an Aurora DSQL cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aurora-dsql/latest/userguide/backup-aurora-dsql.html
     */
    toStartBackupJob() {
        return this.to('StartBackupJob');
    }
    /**
     * Grants permission to start a restore job for an Aurora DSQL cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - dsql:CreateCluster
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/aurora-dsql/latest/userguide/backup-aurora-dsql.html
     */
    toStartRestoreJob() {
        return this.to('StartRestoreJob');
    }
    /**
     * Grants permission to stop a backup job for an Aurora DSQL cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aurora-dsql/latest/userguide/backup-aurora-dsql.html
     */
    toStopBackupJob() {
        return this.to('StopBackupJob');
    }
    /**
     * Grants permission to stop a restore job for an Aurora DSQL Cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aurora-dsql/latest/userguide/backup-aurora-dsql.html
     */
    toStopRestoreJob() {
        return this.to('StopRestoreJob');
    }
    /**
     * Grants permission to add tags to Aurora DSQL resources
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/aurora-dsql/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from Aurora DSQL resources
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/aurora-dsql/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to modify cluster attributes
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifWitnessRegion()
     *
     * https://docs.aws.amazon.com/aurora-dsql/latest/APIReference/API_UpdateCluster.html
     */
    toUpdateCluster() {
        return this.to('UpdateCluster');
    }
    /**
     * Adds a resource of type Cluster to the statement
     *
     * https://docs.aws.amazon.com/aurora-dsql/latest/userguide/what-is-aurora-dsql.html
     *
     * @param identifier - Identifier for the identifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCluster(identifier, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dsql:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:cluster/${identifier}`);
    }
    /**
     * Filters access by a tag key and value pair that is allowed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateCluster()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - Cluster
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a list of tag keys that are allowed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateCluster()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ID of an AWS FIS action
     *
     * https://docs.aws.amazon.com/fis/latest/userguide/fis-actions-reference.html
     *
     * Applies to actions:
     * - .toInjectError()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFisActionId(value, operator) {
        return this.if(`FisActionId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ARN of an AWS FIS target
     *
     * https://docs.aws.amazon.com/fis/latest/userguide/fis-actions-reference.html
     *
     * Applies to actions:
     * - .toInjectError()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifFisTargetArns(value, operator) {
        return this.if(`FisTargetArns`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the witness region of multi-Region clusters
     *
     * https://docs.aws.amazon.com/aurora-dsql/latest/userguide/using-iam-condition-keys.html#using-iam-condition-keys-create-mr-cluster-witness
     *
     * Applies to actions:
     * - .toCreateCluster()
     * - .toPutWitnessRegion()
     * - .toUpdateCluster()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifWitnessRegion(value, operator) {
        return this.if(`WitnessRegion`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [dsql](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonauroradsql.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'dsql';
        this.accessLevelList = {
            Write: [
                'AddPeerCluster',
                'CreateCluster',
                'DbConnect',
                'DbConnectAdmin',
                'DeleteCluster',
                'InjectError',
                'PutMultiRegionProperties',
                'PutWitnessRegion',
                'RemovePeerCluster',
                'StartBackupJob',
                'StartRestoreJob',
                'StopBackupJob',
                'StopRestoreJob',
                'UpdateCluster'
            ],
            Read: [
                'GetBackupJob',
                'GetCluster',
                'GetRestoreJob',
                'GetVpcEndpointServiceName',
                'ListTagsForResource'
            ],
            List: [
                'ListClusters'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Dsql = Dsql;
//# sourceMappingURL=data:application/json;base64,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