"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Backup = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [backup](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbackup.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Backup extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate an MPA approval team with a backup vault
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifMpaApprovalTeamArn()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_AssociateBackupVaultMpaApprovalTeam.html
     */
    toAssociateBackupVaultMpaApprovalTeam() {
        return this.to('AssociateBackupVaultMpaApprovalTeam');
    }
    /**
     * Grants permission to cancel a legal hold
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CancelLegalHold.html
     */
    toCancelLegalHold() {
        return this.to('CancelLegalHold');
    }
    /**
     * Grants permission to copy from a backup vault
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCopyTargets()
     * - .ifCopyTargetOrgPaths()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/vaults.html.html
     */
    toCopyFromBackupVault() {
        return this.to('CopyFromBackupVault');
    }
    /**
     * Grants permission to copy into a backup vault
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/vaults.html
     */
    toCopyIntoBackupVault() {
        return this.to('CopyIntoBackupVault');
    }
    /**
     * Grants permission to create a new backup plan
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateBackupPlan.html
     */
    toCreateBackupPlan() {
        return this.to('CreateBackupPlan');
    }
    /**
     * Grants permission to create a new resource assignment in a backup plan
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateBackupSelection.html
     */
    toCreateBackupSelection() {
        return this.to('CreateBackupSelection');
    }
    /**
     * Grants permission to create a new backup vault
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateBackupVault.html
     */
    toCreateBackupVault() {
        return this.to('CreateBackupVault');
    }
    /**
     * Grants permission to create a new framework
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateFramework.html
     */
    toCreateFramework() {
        return this.to('CreateFramework');
    }
    /**
     * Grants permission to create a new legal hold
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateLegalHold.html
     */
    toCreateLegalHold() {
        return this.to('CreateLegalHold');
    }
    /**
     * Grants permission to create a new logically air-gapped backup vault, a logical container where backups are stored
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifMinRetentionDays()
     * - .ifMaxRetentionDays()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateLogicallyAirGappedBackupVault.html
     */
    toCreateLogicallyAirGappedBackupVault() {
        return this.to('CreateLogicallyAirGappedBackupVault');
    }
    /**
     * Grants permission to create a new report plan
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifFrameworkArns()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateReportPlan.html
     */
    toCreateReportPlan() {
        return this.to('CreateReportPlan');
    }
    /**
     * Grants permission to create a restore access backup vault
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateRestoreAccessBackupVault.html
     */
    toCreateRestoreAccessBackupVault() {
        return this.to('CreateRestoreAccessBackupVault');
    }
    /**
     * Grants permission to create a new restore testing plan
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateRestoreTestingPlan.html
     */
    toCreateRestoreTestingPlan() {
        return this.to('CreateRestoreTestingPlan');
    }
    /**
     * Grants permission to create a new resource assignment in a restore testing plan
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateRestoreTestingSelection.html
     */
    toCreateRestoreTestingSelection() {
        return this.to('CreateRestoreTestingSelection');
    }
    /**
     * Grants permission to delete a backup plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupPlan.html
     */
    toDeleteBackupPlan() {
        return this.to('DeleteBackupPlan');
    }
    /**
     * Grants permission to delete a resource assignment from a backup plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupSelection.html
     */
    toDeleteBackupSelection() {
        return this.to('DeleteBackupSelection');
    }
    /**
     * Grants permission to delete a backup vault
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupVault.html
     */
    toDeleteBackupVault() {
        return this.to('DeleteBackupVault');
    }
    /**
     * Grants permission to delete backup vault access policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupVaultAccessPolicy.html
     */
    toDeleteBackupVaultAccessPolicy() {
        return this.to('DeleteBackupVaultAccessPolicy');
    }
    /**
     * Grants permission to remove the lock configuration from a backup vault
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupVaultLockConfiguration.html
     */
    toDeleteBackupVaultLockConfiguration() {
        return this.to('DeleteBackupVaultLockConfiguration');
    }
    /**
     * Grants permission to remove the notifications from a backup vault
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupVaultNotifications.html
     */
    toDeleteBackupVaultNotifications() {
        return this.to('DeleteBackupVaultNotifications');
    }
    /**
     * Grants permission to delete backup vault sharing policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/logicallyairgappedvault.html
     */
    toDeleteBackupVaultSharingPolicy() {
        return this.to('DeleteBackupVaultSharingPolicy');
    }
    /**
     * Grants permission to delete a framework
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteFramework.html
     */
    toDeleteFramework() {
        return this.to('DeleteFramework');
    }
    /**
     * Grants permission to delete a recovery point from a backup vault
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteRecoveryPoint.html
     */
    toDeleteRecoveryPoint() {
        return this.to('DeleteRecoveryPoint');
    }
    /**
     * Grants permission to delete a report plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteReportPlan.html
     */
    toDeleteReportPlan() {
        return this.to('DeleteReportPlan');
    }
    /**
     * Grants permission to delete a restore testing plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteRestoreTestingPlan.html
     */
    toDeleteRestoreTestingPlan() {
        return this.to('DeleteRestoreTestingPlan');
    }
    /**
     * Grants permission to delete a resource assignment from a restore testing plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteRestoreTestingSelection.html
     */
    toDeleteRestoreTestingSelection() {
        return this.to('DeleteRestoreTestingSelection');
    }
    /**
     * Grants permission to describe a backup job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeBackupJob.html
     */
    toDescribeBackupJob() {
        return this.to('DescribeBackupJob');
    }
    /**
     * Grants permission to describe a new backup vault with the specified name
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeBackupVault.html
     */
    toDescribeBackupVault() {
        return this.to('DescribeBackupVault');
    }
    /**
     * Grants permission to describe a copy job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeCopyJob.html
     */
    toDescribeCopyJob() {
        return this.to('DescribeCopyJob');
    }
    /**
     * Grants permission to describe a framework with the specified name
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeFramework.html
     */
    toDescribeFramework() {
        return this.to('DescribeFramework');
    }
    /**
     * Grants permission to describe global settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeGlobalSettings.html
     */
    toDescribeGlobalSettings() {
        return this.to('DescribeGlobalSettings');
    }
    /**
     * Grants permission to describe a protected resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeProtectedResource.html
     */
    toDescribeProtectedResource() {
        return this.to('DescribeProtectedResource');
    }
    /**
     * Grants permission to describe a recovery point
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeRecoveryPoint.html
     */
    toDescribeRecoveryPoint() {
        return this.to('DescribeRecoveryPoint');
    }
    /**
     * Grants permission to describe region settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeRegionSettings.html
     */
    toDescribeRegionSettings() {
        return this.to('DescribeRegionSettings');
    }
    /**
     * Grants permission to describe a report job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeReportJob.html
     */
    toDescribeReportJob() {
        return this.to('DescribeReportJob');
    }
    /**
     * Grants permission to describe a report plan with the specified name
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeReportPlan.html
     */
    toDescribeReportPlan() {
        return this.to('DescribeReportPlan');
    }
    /**
     * Grants permission to describe a restore job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeRestoreJob.html
     */
    toDescribeRestoreJob() {
        return this.to('DescribeRestoreJob');
    }
    /**
     * Grants permission to disassociate an MPA approval team from a backup vault
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DisassociateBackupVaultMpaApprovalTeam.html
     */
    toDisassociateBackupVaultMpaApprovalTeam() {
        return this.to('DisassociateBackupVaultMpaApprovalTeam');
    }
    /**
     * Grants permission to disassociate a recovery point from a backup vault
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DisassociateRecoveryPoint.html
     */
    toDisassociateRecoveryPoint() {
        return this.to('DisassociateRecoveryPoint');
    }
    /**
     * Grants permission to disassociate a recovery point from its parent
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DisassociateRecoveryPointFromParent.html
     */
    toDisassociateRecoveryPointFromParent() {
        return this.to('DisassociateRecoveryPointFromParent');
    }
    /**
     * Grants permission to export a backup plan as a JSON
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ExportBackupPlanTemplate.html
     */
    toExportBackupPlanTemplate() {
        return this.to('ExportBackupPlanTemplate');
    }
    /**
     * Grants permission to get a backup plan
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupPlan.html
     */
    toGetBackupPlan() {
        return this.to('GetBackupPlan');
    }
    /**
     * Grants permission to transform a JSON to a backup plan
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupPlanFromJSON.html
     */
    toGetBackupPlanFromJSON() {
        return this.to('GetBackupPlanFromJSON');
    }
    /**
     * Grants permission to transform a template to a backup plan
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupPlanFromTemplate.html
     */
    toGetBackupPlanFromTemplate() {
        return this.to('GetBackupPlanFromTemplate');
    }
    /**
     * Grants permission to get a backup plan resource assignment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupSelection.html
     */
    toGetBackupSelection() {
        return this.to('GetBackupSelection');
    }
    /**
     * Grants permission to get backup vault access policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupVaultAccessPolicy.html
     */
    toGetBackupVaultAccessPolicy() {
        return this.to('GetBackupVaultAccessPolicy');
    }
    /**
     * Grants permission to get backup vault notifications
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupVaultNotifications.html
     */
    toGetBackupVaultNotifications() {
        return this.to('GetBackupVaultNotifications');
    }
    /**
     * Grants permission to get backup vault sharing policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/logicallyairgappedvault.html
     */
    toGetBackupVaultSharingPolicy() {
        return this.to('GetBackupVaultSharingPolicy');
    }
    /**
     * Grants permission to get a legal hold
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetLegalHold.html
     */
    toGetLegalHold() {
        return this.to('GetLegalHold');
    }
    /**
     * Grants permission to get indexing details for a recovery point
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetRecoveryPointIndexDetails.html
     */
    toGetRecoveryPointIndexDetails() {
        return this.to('GetRecoveryPointIndexDetails');
    }
    /**
     * Grants permission to get recovery point restore metadata
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetRecoveryPointRestoreMetadata.html
     */
    toGetRecoveryPointRestoreMetadata() {
        return this.to('GetRecoveryPointRestoreMetadata');
    }
    /**
     * Grants permission to get the restore metadata associated with a restore job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetRestoreJobMetadata.html
     */
    toGetRestoreJobMetadata() {
        return this.to('GetRestoreJobMetadata');
    }
    /**
     * Grants permission to get inferred metadata generated by restore testing
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetRestoreTestingInferredMetadata.html
     */
    toGetRestoreTestingInferredMetadata() {
        return this.to('GetRestoreTestingInferredMetadata');
    }
    /**
     * Grants permission to get a restore testing plan
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetRestoreTestingPlan.html
     */
    toGetRestoreTestingPlan() {
        return this.to('GetRestoreTestingPlan');
    }
    /**
     * Grants permission to get a restore testing plan resource assignment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetRestoreTestingSelection.html
     */
    toGetRestoreTestingSelection() {
        return this.to('GetRestoreTestingSelection');
    }
    /**
     * Grants permission to get supported resource types
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetSupportedResourceTypes.html
     */
    toGetSupportedResourceTypes() {
        return this.to('GetSupportedResourceTypes');
    }
    /**
     * Grants permission to list backup job summaries
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupJobSummaries.html
     */
    toListBackupJobSummaries() {
        return this.to('ListBackupJobSummaries');
    }
    /**
     * Grants permission to list backup jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupJobs.html
     */
    toListBackupJobs() {
        return this.to('ListBackupJobs');
    }
    /**
     * Grants permission to list backup plan templates provided by AWS Backup
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupPlanTemplates.html
     */
    toListBackupPlanTemplates() {
        return this.to('ListBackupPlanTemplates');
    }
    /**
     * Grants permission to list backup plan versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupPlanVersions.html
     */
    toListBackupPlanVersions() {
        return this.to('ListBackupPlanVersions');
    }
    /**
     * Grants permission to list backup plans
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupPlans.html
     */
    toListBackupPlans() {
        return this.to('ListBackupPlans');
    }
    /**
     * Grants permission to list resource assignments for a specific backup plan
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupSelections.html
     */
    toListBackupSelections() {
        return this.to('ListBackupSelections');
    }
    /**
     * Grants permission to list backup vaults
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupVaults.html
     */
    toListBackupVaults() {
        return this.to('ListBackupVaults');
    }
    /**
     * Grants permission to list copy job summaries
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListCopyJobSummaries.html
     */
    toListCopyJobSummaries() {
        return this.to('ListCopyJobSummaries');
    }
    /**
     * Grants permission to list copy jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListCopyJobs.html
     */
    toListCopyJobs() {
        return this.to('ListCopyJobs');
    }
    /**
     * Grants permission to list frameworks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListFrameworks.html
     */
    toListFrameworks() {
        return this.to('ListFrameworks');
    }
    /**
     * Grants permission to get list indexed recovery points
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListIndexedRecoveryPoints.html
     */
    toListIndexedRecoveryPoints() {
        return this.to('ListIndexedRecoveryPoints');
    }
    /**
     * Grants permission to list indexed recovery points to search
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListIndexedRecoveryPointsForSearch.html
     */
    toListIndexedRecoveryPointsForSearch() {
        return this.to('ListIndexedRecoveryPointsForSearch');
    }
    /**
     * Grants permission to list legal holds
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListLegalHolds.html
     */
    toListLegalHolds() {
        return this.to('ListLegalHolds');
    }
    /**
     * Grants permission to list protected resources by AWS Backup
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListProtectedResources.html
     */
    toListProtectedResources() {
        return this.to('ListProtectedResources');
    }
    /**
     * Grants permission to list protected resources inside a backup vault
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListProtectedResourcesByBackupVault.html
     */
    toListProtectedResourcesByBackupVault() {
        return this.to('ListProtectedResourcesByBackupVault');
    }
    /**
     * Grants permission to list recovery points inside a backup vault
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRecoveryPointsByBackupVault.html
     */
    toListRecoveryPointsByBackupVault() {
        return this.to('ListRecoveryPointsByBackupVault');
    }
    /**
     * Grants permission to list recovery points by legal hold
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRecoveryPointsByLegalHold.html
     */
    toListRecoveryPointsByLegalHold() {
        return this.to('ListRecoveryPointsByLegalHold');
    }
    /**
     * Grants permission to list recovery points for a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRecoveryPointsByResource.html
     */
    toListRecoveryPointsByResource() {
        return this.to('ListRecoveryPointsByResource');
    }
    /**
     * Grants permission to list report jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListReportJobs.html
     */
    toListReportJobs() {
        return this.to('ListReportJobs');
    }
    /**
     * Grants permission to list report plans
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListReportPlans.html
     */
    toListReportPlans() {
        return this.to('ListReportPlans');
    }
    /**
     * Grants permission to list a restore access backup vaults associated with a backup vault
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRestoreAccessBackupVaults.html
     */
    toListRestoreAccessBackupVaults() {
        return this.to('ListRestoreAccessBackupVaults');
    }
    /**
     * Grants permission to list restore job summaries
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRestoreJobSummaries.html
     */
    toListRestoreJobSummaries() {
        return this.to('ListRestoreJobSummaries');
    }
    /**
     * Grants permission to list restore jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRestoreJobs.html
     */
    toListRestoreJobs() {
        return this.to('ListRestoreJobs');
    }
    /**
     * Grants permission to list restore jobs for a protected resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRestoreJobsByProtectedResource.html
     */
    toListRestoreJobsByProtectedResource() {
        return this.to('ListRestoreJobsByProtectedResource');
    }
    /**
     * Grants permission to list restore testing plans
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRestoreTestingPlans.html
     */
    toListRestoreTestingPlans() {
        return this.to('ListRestoreTestingPlans');
    }
    /**
     * Grants permission to list resource assignments for a specific restore testing plan
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRestoreTestingSelections.html
     */
    toListRestoreTestingSelections() {
        return this.to('ListRestoreTestingSelections');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListTags.html
     */
    toListTags() {
        return this.to('ListTags');
    }
    /**
     * Grants permission to add an access policy to the backup vault
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_PutBackupVaultAccessPolicy.html
     */
    toPutBackupVaultAccessPolicy() {
        return this.to('PutBackupVaultAccessPolicy');
    }
    /**
     * Grants permission to add a lock configuration to the backup vault
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifChangeableForDays()
     * - .ifMinRetentionDays()
     * - .ifMaxRetentionDays()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_PutBackupVaultLockConfiguration.html
     */
    toPutBackupVaultLockConfiguration() {
        return this.to('PutBackupVaultLockConfiguration');
    }
    /**
     * Grants permission to add an SNS topic to the backup vault
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_PutBackupVaultNotifications.html
     */
    toPutBackupVaultNotifications() {
        return this.to('PutBackupVaultNotifications');
    }
    /**
     * Grants permission to add a sharing policy to the backup vault
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/logicallyairgappedvault.html
     */
    toPutBackupVaultSharingPolicy() {
        return this.to('PutBackupVaultSharingPolicy');
    }
    /**
     * Grants permission to put a restore validation result
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_PutRestoreValidationResult.html
     */
    toPutRestoreValidationResult() {
        return this.to('PutRestoreValidationResult');
    }
    /**
     * Grants permission to revoke a restore access backup vault
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_RevokeRestoreAccessBackupVault.html
     */
    toRevokeRestoreAccessBackupVault() {
        return this.to('RevokeRestoreAccessBackupVault');
    }
    /**
     * Grants permission to search a recovery point
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_SearchRecoveryPoint.html
     */
    toSearchRecoveryPoint() {
        return this.to('SearchRecoveryPoint');
    }
    /**
     * Grants permission to start a new backup job
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StartBackupJob.html
     */
    toStartBackupJob() {
        return this.to('StartBackupJob');
    }
    /**
     * Grants permission to copy a backup from a source backup vault to a destination backup vault
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StartCopyJob.html
     */
    toStartCopyJob() {
        return this.to('StartCopyJob');
    }
    /**
     * Grants permission to start a new report job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StartReportJob.html
     */
    toStartReportJob() {
        return this.to('StartReportJob');
    }
    /**
     * Grants permission to start a new restore job
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StartRestoreJob.html
     */
    toStartRestoreJob() {
        return this.to('StartRestoreJob');
    }
    /**
     * Grants permission to stop a backup job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StopBackupJob.html
     */
    toStopBackupJob() {
        return this.to('StopBackupJob');
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a backup plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UpdateBackupPlan.html
     */
    toUpdateBackupPlan() {
        return this.to('UpdateBackupPlan');
    }
    /**
     * Grants permission to update a framework
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UpdateFramework.html
     */
    toUpdateFramework() {
        return this.to('UpdateFramework');
    }
    /**
     * Grants permission to update the current global settings for the AWS Account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UpdateGlobalSettings.html
     */
    toUpdateGlobalSettings() {
        return this.to('UpdateGlobalSettings');
    }
    /**
     * Grants permission to update recovery point index settings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifIndex()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UpdateRecoveryPointIndexSettings.html
     */
    toUpdateRecoveryPointIndexSettings() {
        return this.to('UpdateRecoveryPointIndexSettings');
    }
    /**
     * Grants permission to update the lifecycle of the recovery point
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UpdateRecoveryPointLifecycle.html
     */
    toUpdateRecoveryPointLifecycle() {
        return this.to('UpdateRecoveryPointLifecycle');
    }
    /**
     * Grants permission to update the current service opt-in settings for the Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UpdateRegionSettings.html
     */
    toUpdateRegionSettings() {
        return this.to('UpdateRegionSettings');
    }
    /**
     * Grants permission to update a report plan
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFrameworkArns()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UpdateReportPlan.html
     */
    toUpdateReportPlan() {
        return this.to('UpdateReportPlan');
    }
    /**
     * Grants permission to update a restore testing plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UpdateRestoreTestingPlan.html
     */
    toUpdateRestoreTestingPlan() {
        return this.to('UpdateRestoreTestingPlan');
    }
    /**
     * Grants permission to update a resource assignment in a restore testing plan
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UpdateRestoreTestingSelection.html
     */
    toUpdateRestoreTestingSelection() {
        return this.to('UpdateRestoreTestingSelection');
    }
    /**
     * Adds a resource of type backupVault to the statement
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/vaults.html
     *
     * @param backupVaultName - Identifier for the backupVaultName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBackupVault(backupVaultName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:backup:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:backup-vault:${backupVaultName}`);
    }
    /**
     * Adds a resource of type backupPlan to the statement
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/about-backup-plans.html
     *
     * @param backupPlanId - Identifier for the backupPlanId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBackupPlan(backupPlanId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:backup:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:backup-plan:${backupPlanId}`);
    }
    /**
     * Adds a resource of type recoveryPoint to the statement
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/recovery-points.html
     *
     * @param recoveryPointId - Identifier for the recoveryPointId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRecoveryPoint(recoveryPointId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:backup:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:recovery-point:${recoveryPointId}`);
    }
    /**
     * Adds a resource of type framework to the statement
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/working-with-audit-frameworks.html
     *
     * @param frameworkName - Identifier for the frameworkName.
     * @param frameworkId - Identifier for the frameworkId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFramework(frameworkName, frameworkId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:backup:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:framework:${frameworkName}-${frameworkId}`);
    }
    /**
     * Adds a resource of type reportPlan to the statement
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/create-report-plan-api.html
     *
     * @param reportPlanName - Identifier for the reportPlanName.
     * @param reportPlanId - Identifier for the reportPlanId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onReportPlan(reportPlanName, reportPlanId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:backup:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:report-plan:${reportPlanName}-${reportPlanId}`);
    }
    /**
     * Adds a resource of type legalHold to the statement
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/legalhold.html
     *
     * @param legalHoldId - Identifier for the legalHoldId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLegalHold(legalHoldId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:backup:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:legal-hold:${legalHoldId}`);
    }
    /**
     * Adds a resource of type restoreTestingPlan to the statement
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/restore-testing.html
     *
     * @param restoreTestingPlanName - Identifier for the restoreTestingPlanName.
     * @param restoreTestingPlanId - Identifier for the restoreTestingPlanId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRestoreTestingPlan(restoreTestingPlanName, restoreTestingPlanId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:backup:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:restore-testing-plan:${restoreTestingPlanName}-${restoreTestingPlanId}`);
    }
    /**
     * Filters access by the allowed set of values for each of the tags
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCopyIntoBackupVault()
     * - .toCreateBackupPlan()
     * - .toCreateBackupVault()
     * - .toCreateFramework()
     * - .toCreateLegalHold()
     * - .toCreateLogicallyAirGappedBackupVault()
     * - .toCreateReportPlan()
     * - .toCreateRestoreAccessBackupVault()
     * - .toCreateRestoreTestingPlan()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - backupVault
     * - backupPlan
     * - recoveryPoint
     * - framework
     * - reportPlan
     * - legalHold
     * - restoreTestingPlan
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of mandatory tags in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateBackupPlan()
     * - .toCreateBackupVault()
     * - .toCreateFramework()
     * - .toCreateLegalHold()
     * - .toCreateLogicallyAirGappedBackupVault()
     * - .toCreateReportPlan()
     * - .toCreateRestoreAccessBackupVault()
     * - .toCreateRestoreTestingPlan()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the value of the ChangeableForDays parameter
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/access-control.html#amazon-backup-keys
     *
     * Applies to actions:
     * - .toPutBackupVaultLockConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifChangeableForDays(value, operator) {
        return this.if(`ChangeableForDays`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the organization unit
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/access-control.html#amazon-backup-keys
     *
     * Applies to actions:
     * - .toCopyFromBackupVault()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCopyTargetOrgPaths(value, operator) {
        return this.if(`CopyTargetOrgPaths`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ARN of a backup vault
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/access-control.html#amazon-backup-keys
     *
     * Applies to actions:
     * - .toCopyFromBackupVault()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifCopyTargets(value, operator) {
        return this.if(`CopyTargets`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the Framework ARNs
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/access-control.html#amazon-backup-keys
     *
     * Applies to actions:
     * - .toCreateReportPlan()
     * - .toUpdateReportPlan()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifFrameworkArns(value, operator) {
        return this.if(`FrameworkArns`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the value of Index parameter
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/access-control.html#amazon-backup-keys
     *
     * Applies to actions:
     * - .toUpdateRecoveryPointIndexSettings()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifIndex(value, operator) {
        return this.if(`Index`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the value of the MaxRetentionDays parameter
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/access-control.html#amazon-backup-keys
     *
     * Applies to actions:
     * - .toCreateLogicallyAirGappedBackupVault()
     * - .toPutBackupVaultLockConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMaxRetentionDays(value, operator) {
        return this.if(`MaxRetentionDays`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the value of the MinRetentionDays parameter
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/access-control.html#amazon-backup-keys
     *
     * Applies to actions:
     * - .toCreateLogicallyAirGappedBackupVault()
     * - .toPutBackupVaultLockConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMinRetentionDays(value, operator) {
        return this.if(`MinRetentionDays`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the MPA Approval Team ARN of a backup vault
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/access-control.html#amazon-backup-keys
     *
     * Applies to actions:
     * - .toAssociateBackupVaultMpaApprovalTeam()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifMpaApprovalTeamArn(value, operator) {
        return this.if(`MpaApprovalTeamArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Statement provider for service [backup](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbackup.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'backup';
        this.accessLevelList = {
            Write: [
                'AssociateBackupVaultMpaApprovalTeam',
                'CancelLegalHold',
                'CopyFromBackupVault',
                'CopyIntoBackupVault',
                'CreateBackupPlan',
                'CreateBackupSelection',
                'CreateBackupVault',
                'CreateFramework',
                'CreateLegalHold',
                'CreateLogicallyAirGappedBackupVault',
                'CreateReportPlan',
                'CreateRestoreAccessBackupVault',
                'CreateRestoreTestingPlan',
                'CreateRestoreTestingSelection',
                'DeleteBackupPlan',
                'DeleteBackupSelection',
                'DeleteBackupVault',
                'DeleteBackupVaultLockConfiguration',
                'DeleteBackupVaultNotifications',
                'DeleteFramework',
                'DeleteRecoveryPoint',
                'DeleteReportPlan',
                'DeleteRestoreTestingPlan',
                'DeleteRestoreTestingSelection',
                'DisassociateBackupVaultMpaApprovalTeam',
                'DisassociateRecoveryPoint',
                'DisassociateRecoveryPointFromParent',
                'PutBackupVaultLockConfiguration',
                'PutBackupVaultNotifications',
                'PutRestoreValidationResult',
                'RevokeRestoreAccessBackupVault',
                'StartBackupJob',
                'StartCopyJob',
                'StartReportJob',
                'StartRestoreJob',
                'StopBackupJob',
                'UpdateBackupPlan',
                'UpdateFramework',
                'UpdateGlobalSettings',
                'UpdateRecoveryPointIndexSettings',
                'UpdateRecoveryPointLifecycle',
                'UpdateRegionSettings',
                'UpdateReportPlan',
                'UpdateRestoreTestingPlan',
                'UpdateRestoreTestingSelection'
            ],
            'Permissions management': [
                'DeleteBackupVaultAccessPolicy',
                'DeleteBackupVaultSharingPolicy',
                'ListIndexedRecoveryPointsForSearch',
                'PutBackupVaultAccessPolicy',
                'PutBackupVaultSharingPolicy',
                'SearchRecoveryPoint'
            ],
            Read: [
                'DescribeBackupJob',
                'DescribeBackupVault',
                'DescribeCopyJob',
                'DescribeFramework',
                'DescribeGlobalSettings',
                'DescribeProtectedResource',
                'DescribeRecoveryPoint',
                'DescribeRegionSettings',
                'DescribeReportJob',
                'DescribeReportPlan',
                'DescribeRestoreJob',
                'ExportBackupPlanTemplate',
                'GetBackupPlan',
                'GetBackupPlanFromJSON',
                'GetBackupPlanFromTemplate',
                'GetBackupSelection',
                'GetBackupVaultAccessPolicy',
                'GetBackupVaultNotifications',
                'GetBackupVaultSharingPolicy',
                'GetLegalHold',
                'GetRecoveryPointIndexDetails',
                'GetRecoveryPointRestoreMetadata',
                'GetRestoreJobMetadata',
                'GetRestoreTestingInferredMetadata',
                'GetRestoreTestingPlan',
                'GetRestoreTestingSelection',
                'GetSupportedResourceTypes',
                'ListTags'
            ],
            List: [
                'ListBackupJobSummaries',
                'ListBackupJobs',
                'ListBackupPlanTemplates',
                'ListBackupPlanVersions',
                'ListBackupPlans',
                'ListBackupSelections',
                'ListBackupVaults',
                'ListCopyJobSummaries',
                'ListCopyJobs',
                'ListFrameworks',
                'ListIndexedRecoveryPoints',
                'ListLegalHolds',
                'ListProtectedResources',
                'ListProtectedResourcesByBackupVault',
                'ListRecoveryPointsByBackupVault',
                'ListRecoveryPointsByLegalHold',
                'ListRecoveryPointsByResource',
                'ListReportJobs',
                'ListReportPlans',
                'ListRestoreAccessBackupVaults',
                'ListRestoreJobSummaries',
                'ListRestoreJobs',
                'ListRestoreJobsByProtectedResource',
                'ListRestoreTestingPlans',
                'ListRestoreTestingSelections'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Backup = Backup;
//# sourceMappingURL=data:application/json;base64,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