import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [bedrock](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonbedrock.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class Bedrock extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to configure vended log delivery for a knowledge base
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toAllowVendedLogDeliveryForResource(): this;
    /**
     * Grants permission to apply a guardrail
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toApplyGuardrail(): this;
    /**
     * Grants permission to associate another existing agent as a collaborator to an existing agent
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_AssociateAgentCollaborator.html
     */
    toAssociateAgentCollaborator(): this;
    /**
     * Grants permission to associate a knowledge base with an agent
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_AssociateAgentKnowledgeBase.html
     */
    toAssociateAgentKnowledgeBase(): this;
    /**
     * Grants permission to use 3rd party platform to store knowledge data
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifThirdPartyKnowledgeBaseCredentialsSecretArn()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toAssociateThirdPartyKnowledgeBase(): this;
    /**
     * Grants permission to batch delete list of bedrock evaluation jobs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_BatchDeleteEvaluationJob.html
     */
    toBatchDeleteEvaluationJob(): this;
    /**
     * Grants permission to use bearer token
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toCallWithBearerToken(): this;
    /**
     * Grants permission to cancel a build workflow for an automated reasoning policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toCancelAutomatedReasoningPolicyBuildWorkflow(): this;
    /**
     * Grants permission to create a new agent and a test agent alias pointing to the DRAFT agent version
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_CreateAgent.html
     */
    toCreateAgent(): this;
    /**
     * Grants permission to create a new action group in an existing agent
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_CreateAgentActionGroup.html
     */
    toCreateAgentActionGroup(): this;
    /**
     * Grants permission to create a new alias for an agent
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_CreateAgentAlias.html
     */
    toCreateAgentAlias(): this;
    /**
     * Grants permission to create a new automated reasoning policy
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifBearerTokenType()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toCreateAutomatedReasoningPolicy(): this;
    /**
     * Grants permission to create a test case for an automated reasoning policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toCreateAutomatedReasoningPolicyTestCase(): this;
    /**
     * Grants permission to create a new automated reasoning policy version
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifBearerTokenType()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toCreateAutomatedReasoningPolicyVersion(): this;
    /**
     * Grants permission to create a blueprint for custom output from data automation
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_data-automation_CreateBlueprint.html
     */
    toCreateBlueprint(): this;
    /**
     * Grants permission to create a new version for an existing blueprint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_data-automation_CreateBlueprintVersion.html
     */
    toCreateBlueprintVersion(): this;
    /**
     * Grants permission to create a custom model into Bedrock
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifBearerTokenType()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_CreateCustomModel.html
     */
    toCreateCustomModel(): this;
    /**
     * Grants permission to create a custom model deployment for custom model
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifBearerTokenType()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_CreateCustomModelDeployment.html
     */
    toCreateCustomModelDeployment(): this;
    /**
     * Grants permission to create a data automation project
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_data-automation_CreateDataAutomationProject.html
     */
    toCreateDataAutomationProject(): this;
    /**
     * Grants permission to create a data source
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_CreateDataSource.html
     */
    toCreateDataSource(): this;
    /**
     * Grants permission to create a job for evaluation foundation models or custom models
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifBearerTokenType()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_CreateEvaluationJob.html
     */
    toCreateEvaluationJob(): this;
    /**
     * Grants permission to create a prompt flow
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_CreateFlow.html
     */
    toCreateFlow(): this;
    /**
     * Grants permission to create an alias of a prompt flow
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_CreateFlowAlias.html
     */
    toCreateFlowAlias(): this;
    /**
     * Grants permission to create an immutable version of a prompt flow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_CreateFlowVersion.html
     */
    toCreateFlowVersion(): this;
    /**
     * Grants permission to create a new foundation model agreement
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonbedrock.html
     */
    toCreateFoundationModelAgreement(): this;
    /**
     * Grants permission to create a new guardrail
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifBearerTokenType()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toCreateGuardrail(): this;
    /**
     * Grants permission to create a new guardrail version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toCreateGuardrailVersion(): this;
    /**
     * Grants permission to create inference profiles
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_CreateInferenceProfile.html
     */
    toCreateInferenceProfile(): this;
    /**
     * Grants permission to create a new invocation in an existing session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_CreateInvocation.html
     */
    toCreateInvocation(): this;
    /**
     * Grants permission to create a knowledge base
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_CreateKnowledgeBase.html
     */
    toCreateKnowledgeBase(): this;
    /**
     * Grants permission to create a marketplace model endpoint
     *
     * Access Level: Write
     */
    toCreateMarketplaceModelEndpoint(): this;
    /**
     * Grants permission to create a job for copying a custom model across region or across account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifBearerTokenType()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_CreateModelCopyJob.html
     */
    toCreateModelCopyJob(): this;
    /**
     * Grants permission to create a job for customizing the model with your custom training data
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifBearerTokenType()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_CreateModelCustomizationJob.html
     */
    toCreateModelCustomizationJob(): this;
    /**
     * Grants permission to create a job for evaluation foundation models or custom models
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifBearerTokenType()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_CreateModelEvaluationJob.html
     */
    toCreateModelEvaluationJob(): this;
    /**
     * Grants permission to create a job for importing model into Bedrock
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifBearerTokenType()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_CreateModelImportJob.html
     */
    toCreateModelImportJob(): this;
    /**
     * Grants permission to create a new model invocation job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifBearerTokenType()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_CreateModelInvocationJob.html
     */
    toCreateModelInvocationJob(): this;
    /**
     * Grants permission to create a prompt
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_CreatePrompt.html
     */
    toCreatePrompt(): this;
    /**
     * Grants permission to create a custom prompt router
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifBearerTokenType()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_CreatePromptRouter.html
     */
    toCreatePromptRouter(): this;
    /**
     * Grants permission to create a version of a prompt
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_CreatePromptVersion.html
     */
    toCreatePromptVersion(): this;
    /**
     * Grants permission to create a new provisioned model throughput
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifBearerTokenType()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_CreateProvisionedModelThroughput.html
     */
    toCreateProvisionedModelThroughput(): this;
    /**
     * Grants permission to create a new session
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_CreateSession.html
     */
    toCreateSession(): this;
    /**
     * Grants permission to delete an Agent that you created earlier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_DeleteAgent.html
     */
    toDeleteAgent(): this;
    /**
     * Grants permission to delete an actionGroup that you created earlier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_DeleteAgentActionGroup.html
     */
    toDeleteAgentActionGroup(): this;
    /**
     * Grants permission to delete an AgentAlias that you created earlier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_DeleteAgentAlias.html
     */
    toDeleteAgentAlias(): this;
    /**
     * Grants permission to delete existing memory for an alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_DeleteAgentMemory.html
     */
    toDeleteAgentMemory(): this;
    /**
     * Grants permission to delete an Agent Version that you created earlier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_DeleteAgentVersion.html
     */
    toDeleteAgentVersion(): this;
    /**
     * Grants permission to delete an automated reasoning policy or its version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toDeleteAutomatedReasoningPolicy(): this;
    /**
     * Grants permission to delete a build workflow for an automated reasoning policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toDeleteAutomatedReasoningPolicyBuildWorkflow(): this;
    /**
     * Grants permission to delete a test case for an automated reasoning policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toDeleteAutomatedReasoningPolicyTestCase(): this;
    /**
     * Grants permission to delete a blueprint for data automation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_data-automation_DeleteBlueprint.html
     */
    toDeleteBlueprint(): this;
    /**
     * Grants permission to delete a custom model that you created earlier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_DeleteCustomModel.html
     */
    toDeleteCustomModel(): this;
    /**
     * Grants permission to delete a custom model deployment that you created earlier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_DeleteCustomModelDeployment.html
     */
    toDeleteCustomModelDeployment(): this;
    /**
     * Grants permission to delete a data automation project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_data-automation_DeleteDataAutomationProject.html
     */
    toDeleteDataAutomationProject(): this;
    /**
     * Grants permission to delete a data source
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_DeleteDataSource.html
     */
    toDeleteDataSource(): this;
    /**
     * Grants permission to delete a prompt flow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_DeleteFlow.html
     */
    toDeleteFlow(): this;
    /**
     * Grants permission to delete an alias of a prompt flow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_DeleteFlowAlias.html
     */
    toDeleteFlowAlias(): this;
    /**
     * Grants permission to delete a version of a prompt flow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_DeleteFlowVersion.html
     */
    toDeleteFlowVersion(): this;
    /**
     * Grants permission to delete a foundation model agreement that you created earlier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonbedrock.html
     */
    toDeleteFoundationModelAgreement(): this;
    /**
     * Grants permission to delete a guardrail or its version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toDeleteGuardrail(): this;
    /**
     * Grants permission to delete previously created Bedrock imported model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_DeleteImportedModel.html
     */
    toDeleteImportedModel(): this;
    /**
     * Grants permission to delete inference profiles
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_DeleteInferenceProfile.html
     */
    toDeleteInferenceProfile(): this;
    /**
     * Grants permission to delete a knowledge base
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_DeleteKnowledgeBase.html
     */
    toDeleteKnowledgeBase(): this;
    /**
     * Grants permission to delete documents from a knowledge base
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_DeleteKnowledgeBaseDocuments.html
     */
    toDeleteKnowledgeBaseDocuments(): this;
    /**
     * Grants permission to unsubscribe from a bedrock marketplace enabled AWS marketplace model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonbedrock.html
     */
    toDeleteMarketplaceModelAgreement(): this;
    /**
     * Grants permission to delete a marketplace model endpoint
     *
     * Access Level: Write
     */
    toDeleteMarketplaceModelEndpoint(): this;
    /**
     * Grants permission to delete an existing Invocation logging configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_DeleteModelInvocationLoggingConfiguration.html
     */
    toDeleteModelInvocationLoggingConfiguration(): this;
    /**
     * Grants permission to delete a prompt or its version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_DeletePrompt.html
     */
    toDeletePrompt(): this;
    /**
     * Grants permission to delete a custom prompt router
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_DeletePromptRouter.html
     */
    toDeletePromptRouter(): this;
    /**
     * Grants permission to delete a provisioned model throughput that you created earlier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_DeleteProvisionedModelThroughput.html
     */
    toDeleteProvisionedModelThroughput(): this;
    /**
     * Deletes a previously created Bedrock resource policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy(): this;
    /**
     * Grants permission to delete a Session that you created earlier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_DeleteSession.html
     */
    toDeleteSession(): this;
    /**
     * Grants permission to deregister a marketplace model endpoint to make it unusable in Bedrock Marketplace
     *
     * Access Level: Write
     */
    toDeregisterMarketplaceModelEndpoint(): this;
    /**
     * Grants permission to detect if the provided content is generated using Amazon Bedrock
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toDetectGeneratedContent(): this;
    /**
     * Grants permission to diassociate a collaborator that you associated earlier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_DisassociateAgentCollaborator.html
     */
    toDisassociateAgentCollaborator(): this;
    /**
     * Grants permission to disassociate a knowledge base from the agent
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_DisassociateAgentKnowledgeBase.html
     */
    toDisassociateAgentKnowledgeBase(): this;
    /**
     * Grants permission to end a Session that you created earlier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_EndSession.html
     */
    toEndSession(): this;
    /**
     * Grants permission to retrieve an automated reasoning policy version artifact
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toExportAutomatedReasoningPolicyVersion(): this;
    /**
     * Grants permission to generate queries associated with user input
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toGenerateQuery(): this;
    /**
     * Grants permission to retrieve an existing agent
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_GetAgent.html
     */
    toGetAgent(): this;
    /**
     * Grants permission to retrieve an existing action group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_GetAgentActionGroup.html
     */
    toGetAgentActionGroup(): this;
    /**
     * Grants permission to retrieve an existing alias
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_GetAgentAlias.html
     */
    toGetAgentAlias(): this;
    /**
     * Grants permission to retrieve an existing collaborator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_GetAgentCollaborator.html
     */
    toGetAgentCollaborator(): this;
    /**
     * Grants permission to describe a knowledge base associated with an agent
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_GetAgentKnowledgeBase.html
     */
    toGetAgentKnowledgeBase(): this;
    /**
     * Grants permission to retrieve existing memory for an alias
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_GetAgentMemory.html
     */
    toGetAgentMemory(): this;
    /**
     * Grants permission to retrieve an existing version of an agent
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_GetAgentVersion.html
     */
    toGetAgentVersion(): this;
    /**
     * Grants permission to get the properties associated with an asynchronous invocation that you have submitted
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_GetAsyncInvoke.html
     */
    toGetAsyncInvoke(): this;
    /**
     * Grants permission to retrieve an automated reasoning policy or its version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toGetAutomatedReasoningPolicy(): this;
    /**
     * Grants permission to retrieve annotations for a build workflow for an automated reasoning policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toGetAutomatedReasoningPolicyAnnotations(): this;
    /**
     * Grants permission to retrieve a build workflow for an automated reasoning policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toGetAutomatedReasoningPolicyBuildWorkflow(): this;
    /**
     * Grants permission to retrieve assets for a build workflow for an automated reasoning policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toGetAutomatedReasoningPolicyBuildWorkflowResultAssets(): this;
    /**
     * Grants permission to retrieve the next unreviewed generated scenario for a build workflow for an automated reasoning policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toGetAutomatedReasoningPolicyNextScenario(): this;
    /**
     * Grants permission to retrieve a test case for an automated reasoning policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toGetAutomatedReasoningPolicyTestCase(): this;
    /**
     * Grants permission to retrieve result for a test case for an automated reasoning policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toGetAutomatedReasoningPolicyTestResult(): this;
    /**
     * Grants permission to retrieve an existing blueprint for data automation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_data-automation_GetBlueprint.html
     */
    toGetBlueprint(): this;
    /**
     * Grants permission to retrieve blueprint recommendation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toGetBlueprintRecommendation(): this;
    /**
     * Grants permission to get the properties associated with a Bedrock custom model that you have created
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_GetCustomModel.html
     */
    toGetCustomModel(): this;
    /**
     * Grants permission to get the properties associated with a custom model deployment. Use this operation to get the status of a custom model deployment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_GetCustomModelDeployment.html
     */
    toGetCustomModelDeployment(): this;
    /**
     * Grants permission to retrieve an existing data automation project
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_data-automation_GetDataAutomationProject.html
     */
    toGetDataAutomationProject(): this;
    /**
     * Grants permission to retrieve the status of a data automation invocation job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_data-automation-runtime_GetDataAutomationStatus.html
     */
    toGetDataAutomationStatus(): this;
    /**
     * Grants permission to retrieve an existing data source
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_GetDataSource.html
     */
    toGetDataSource(): this;
    /**
     * Grants permission to get the properties associated with a evaluation job. Use this operation to get the status of a evaluation job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_GetEvaluationJob.html
     */
    toGetEvaluationJob(): this;
    /**
     * Grants permission to retrieve the flow definition for a flow execution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_GetExecutionFlowSnapshot.html
     */
    toGetExecutionFlowSnapshot(): this;
    /**
     * Grants permission to retrieve an existing prompt flow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_GetFlow.html
     */
    toGetFlow(): this;
    /**
     * Grants permission to retrieve an existing alias of a prompt flow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_GetFlowAlias.html
     */
    toGetFlowAlias(): this;
    /**
     * Grants permission to retrieve an existing execution of a flow alias
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_GetFlowExecution.html
     */
    toGetFlowExecution(): this;
    /**
     * Grants permission to retrieve an existing version of a prompt flow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_GetFlowVersion.html
     */
    toGetFlowVersion(): this;
    /**
     * Grants permission to get the properties associated with a Bedrock foundation model
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_GetFoundationModel.html
     */
    toGetFoundationModel(): this;
    /**
     * Grants permission to get the availability of a foundation model
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonbedrock.html
     */
    toGetFoundationModelAvailability(): this;
    /**
     * Grants permission to retrieve a guardrail or its version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toGetGuardrail(): this;
    /**
     * Grants permission to get the properties associated with Bedrock imported model
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_GetImportedModel.html
     */
    toGetImportedModel(): this;
    /**
     * Grants permission to get the properties associated with an inference profile
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_GetInferenceProfile.html
     */
    toGetInferenceProfile(): this;
    /**
     * Grants permission to retrieve an existing ingestion job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_GetIngestionJob.html
     */
    toGetIngestionJob(): this;
    /**
     * Grants permission to get an invocation step from a session
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_GetInvocationStep.html
     */
    toGetInvocationStep(): this;
    /**
     * Grants permission to retrieve an existing knowledge base
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_GetKnowledgeBase.html
     */
    toGetKnowledgeBase(): this;
    /**
     * Grants permission to get details for documents in a knowledge base
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_GetKnowledgeBaseDocuments.html
     */
    toGetKnowledgeBaseDocuments(): this;
    /**
     * Grants permission to get the properties of a marketplace model endpoint
     *
     * Access Level: Read
     */
    toGetMarketplaceModelEndpoint(): this;
    /**
     * Grants permission to get the properties associated with a model-copy job. Use this operation to get the status of a model-copy job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_GetModelCopyJob.html
     */
    toGetModelCopyJob(): this;
    /**
     * Grants permission to get the properties associated with a model-customization job. Use this operation to get the status of a model-customization job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_GetModelCustomizationJob.html
     */
    toGetModelCustomizationJob(): this;
    /**
     * Grants permission to get the properties associated with a model-evaluation job. Use this operation to get the status of a model-evaluation job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_GetModelEvaluationJob.html
     */
    toGetModelEvaluationJob(): this;
    /**
     * Grants permission to get the properties associated with a model import job and is used to get the status of a model import job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_GetModelImportJob.html
     */
    toGetModelImportJob(): this;
    /**
     * Grants permission to retrieve a model invocation job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_GetModelInvocationJob.html
     */
    toGetModelInvocationJob(): this;
    /**
     * Grants permission to retrieve an existing Invocation logging configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_GetModelInvocationLoggingConfiguration.html
     */
    toGetModelInvocationLoggingConfiguration(): this;
    /**
     * Grants permission to retrieve an existing prompt or its version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_GetPrompt.html
     */
    toGetPrompt(): this;
    /**
     * Grants permission to get the properties associated with a prompt router
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_GetPromptRouter.html
     */
    toGetPromptRouter(): this;
    /**
     * Grants permission to retrieve a provisioned model throughput
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_GetProvisionedModelThroughput.html
     */
    toGetProvisionedModelThroughput(): this;
    /**
     * Gets the resource policy document for a Bedrock resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_GetResourePolicy.html
     */
    toGetResourcePolicy(): this;
    /**
     * Grants permission to retrieve an existing session
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_GetSession.html
     */
    toGetSession(): this;
    /**
     * Grants permission to retrieve a use case for model access
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonbedrock.html
     */
    toGetUseCaseForModelAccess(): this;
    /**
     * Grants permission to directly ingest documents into a knowledge base
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_IngestKnowledgeBaseDocuments.html
     */
    toIngestKnowledgeBaseDocuments(): this;
    /**
     * Grants permission to send user input (text-only) to the alias of an agent for Bedrock
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_InvokeAgent.html
     */
    toInvokeAgent(): this;
    /**
     * Grants permission to invoke an Automated Reasoning policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toInvokeAutomatedReasoningPolicy(): this;
    /**
     * Grants permission to invoke blueprint recommendations asynchronously
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toInvokeBlueprintRecommendationAsync(): this;
    /**
     * Grants permission to use the conversational builder which aids in building supported bedrock resources
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/userguide/agents-create-cb.html
     */
    toInvokeBuilder(): this;
    /**
     * Grants permission to invoke a Bedrock data automation job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_data-automation-runtime_InvokeDataAutomationAsync.html
     */
    toInvokeDataAutomationAsync(): this;
    /**
     * Grants permission to invoke a prompt flow with user input
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_InvokeFlow.html
     */
    toInvokeFlow(): this;
    /**
     * Grants permission to send user input (text-only) to the inline agent for Bedrock
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInlineAgentName()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_InvokeInlineAgent.html
     */
    toInvokeInlineAgent(): this;
    /**
     * Grants permission to invoke the specified Bedrock model to run inference using the input provided in the request body
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInferenceProfileArn()
     * - .ifPromptRouterArn()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifGuardrailIdentifier()
     * - .ifBearerTokenType()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_InvokeModel.html
     */
    toInvokeModel(): this;
    /**
     * Grants permission to invoke the specified Bedrock model to run inference using the input provided in the request body with streaming response
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInferenceProfileArn()
     * - .ifPromptRouterArn()
     * - .ifGuardrailIdentifier()
     * - .ifBearerTokenType()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_InvokeModelWithResponseStream.html
     */
    toInvokeModelWithResponseStream(): this;
    /**
     * Grants permission to list action groups in an agent
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_ListAgentActionGroups.html
     */
    toListAgentActionGroups(): this;
    /**
     * Grants permission to list aliases for an agent
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_ListAgentAliases.html
     */
    toListAgentAliases(): this;
    /**
     * Grants permission to list collaborators for an agent
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_ListAgentCollaborators.html
     */
    toListAgentCollaborators(): this;
    /**
     * Grants permission to list knowledge bases associated with an agent
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_ListAgentKnowledgeBases.html
     */
    toListAgentKnowledgeBases(): this;
    /**
     * Grants permission to list existing versions of an agent
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_ListAgentVersions.html
     */
    toListAgentVersions(): this;
    /**
     * Grants permission to list existing agents
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_ListAgents.html
     */
    toListAgents(): this;
    /**
     * Grants permission to get a list of asynchronous invocations that you have submitted
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_ListAsyncInvokes.html
     */
    toListAsyncInvokes(): this;
    /**
     * Grants permission to list automated reasoning policies or its versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toListAutomatedReasoningPolicies(): this;
    /**
     * Grants permission to list build workflows for an automated reasoning policy
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toListAutomatedReasoningPolicyBuildWorkflows(): this;
    /**
     * Grants permission to list test cases for an automated reasoning policy
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toListAutomatedReasoningPolicyTestCases(): this;
    /**
     * Grants permission to list test result for an automated reasoning policy
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toListAutomatedReasoningPolicyTestResults(): this;
    /**
     * Grants permission to list existing blueprints for data automation
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_data-automation_ListBlueprints.html
     */
    toListBlueprints(): this;
    /**
     * Grants permission to get the list of custom model deployments that you have submitted
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_ListCustomModelDeployments.html
     */
    toListCustomModelDeployments(): this;
    /**
     * Grants permission to get a list of Bedrock custom models that you have created
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_ListCustomModels.html
     */
    toListCustomModels(): this;
    /**
     * Grants permission to list existing data automation projects
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_data-automation_ListDataAutomationProjects.html
     */
    toListDataAutomationProjects(): this;
    /**
     * Grants permission to list existing data sources in an knowledge base
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_ListDataSources.html
     */
    toListDataSources(): this;
    /**
     * Grants permission to get the list of evaluation jobs that you have submitted
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_ListEvaluationJobs.html
     */
    toListEvaluationJobs(): this;
    /**
     * Grants permission to list existing aliases of a prompt flow
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_ListFlowAliases.html
     */
    toListFlowAliases(): this;
    /**
     * Grants permission to retrieve events for a flow execution
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_ListFlowExecutionEvents.html
     */
    toListFlowExecutionEvents(): this;
    /**
     * Grants permission to list executions of a flow or a flow alias
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_ListFlowExecutions.html
     */
    toListFlowExecutions(): this;
    /**
     * Grants permission to list existing versions of a prompt flow
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_ListFlowVersions.html
     */
    toListFlowVersions(): this;
    /**
     * Grants permission to list existing prompt flows
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_ListFlows.html
     */
    toListFlows(): this;
    /**
     * Grants permission to get a list of foundation model agreement offers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonbedrock.html
     */
    toListFoundationModelAgreementOffers(): this;
    /**
     * Grants permission to list Bedrock foundation models that you can use
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_ListFoundationModels.html
     */
    toListFoundationModels(): this;
    /**
     * Grants permission to list guardrails or its versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toListGuardrails(): this;
    /**
     * Grants permission to get list of Bedrock imported models
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_ListImportedModels.html
     */
    toListImportedModels(): this;
    /**
     * Grants permission to list inference profiles that you can use
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_ListInferenceProfiles.html
     */
    toListInferenceProfiles(): this;
    /**
     * Grants permission to list ingestion jobs in a data source
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_ListIngestionJobs.html
     */
    toListIngestionJobs(): this;
    /**
     * Grants permission to get list of invocation step from a session
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_ListInvocationSteps.html
     */
    toListInvocationSteps(): this;
    /**
     * Grants permission to list invocations in a session
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_ListInvocations.html
     */
    toListInvocations(): this;
    /**
     * Grants permission to list documents in a knowledge base
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_ListKnowledgeBaseDocuments.html
     */
    toListKnowledgeBaseDocuments(): this;
    /**
     * Grants permission to list existing knowledge bases
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_ListKnowledgeBases.html
     */
    toListKnowledgeBases(): this;
    /**
     * Grants permission to list marketplace model endpoints that you can use
     *
     * Access Level: Read
     */
    toListMarketplaceModelEndpoints(): this;
    /**
     * Grants permission to get the list of model copy jobs that you have submitted
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_ListModelCopyJobs.html
     */
    toListModelCopyJobs(): this;
    /**
     * Grants permission to get the list of model customization jobs that you have submitted
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_ListModelCustomizationJobs.html
     */
    toListModelCustomizationJobs(): this;
    /**
     * Grants permission to get the list of model evaluation jobs that you have submitted
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_ListModelEvaluationJobs.html
     */
    toListModelEvaluationJobs(): this;
    /**
     * Grants permission to get list of model import jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_ListModelImportJobs.html
     */
    toListModelImportJobs(): this;
    /**
     * Grants permission to list model invocation jobs that you created earlier
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_ListModelInvocationJobs.html
     */
    toListModelInvocationJobs(): this;
    /**
     * Grants permission to list prompt routers that you can use
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_ListPromptRouters.html
     */
    toListPromptRouters(): this;
    /**
     * Grants permission to list existing prompts
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_ListPrompts.html
     */
    toListPrompts(): this;
    /**
     * Grants permission to list provisioned model throughputs that you created earlier
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_ListProvisionedModelThroughputs.html
     */
    toListProvisionedModelThroughputs(): this;
    /**
     * Grants permission to list existing sessions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_ListSessions.html
     */
    toListSessions(): this;
    /**
     * Grants permission to list tags for a Bedrock resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to optimize a prompt with user input
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_OptimizePrompt.html
     */
    toOptimizePrompt(): this;
    /**
     * Grants permission to prepare an existing agent to receive runtime requests
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_PrepareAgent.html
     */
    toPrepareAgent(): this;
    /**
     * Grants permission to apply the latest changes made to a prompt flow, so that they are reflected at runtime
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_PrepareFlow.html
     */
    toPrepareFlow(): this;
    /**
     * Grants permission to put entitlement to access a foundation model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonbedrock.html
     */
    toPutFoundationModelEntitlement(): this;
    /**
     * Grants permission to put an invocation step into an invocation in session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_PutInvocationStep.html
     */
    toPutInvocationStep(): this;
    /**
     * Grants permission to create an existing Invocation logging configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_PutModelInvocationLoggingConfiguration.html
     */
    toPutModelInvocationLoggingConfiguration(): this;
    /**
     * Adds a resource policy for a Bedrock resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy(): this;
    /**
     * Grants permission to put a use case for model access
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonbedrock.html
     */
    toPutUseCaseForModelAccess(): this;
    /**
     * Grants permission to register a sagemaker endpoint as a marketplace model endpoint
     *
     * Access Level: Write
     */
    toRegisterMarketplaceModelEndpoint(): this;
    /**
     * Grants permission to render an existing prompt or its version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toRenderPrompt(): this;
    /**
     * Grants permission to rank documents based on user input
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toRerank(): this;
    /**
     * Grants permission to retrieve ingested data from a knowledge base
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toRetrieve(): this;
    /**
     * Grants permission to send user input to perform retrieval and generation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toRetrieveAndGenerate(): this;
    /**
     * Grants permission to start a build workflow for an automated reasoning policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toStartAutomatedReasoningPolicyBuildWorkflow(): this;
    /**
     * Grants permission to start a test workflow for an automated reasoning policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toStartAutomatedReasoningPolicyTestWorkflow(): this;
    /**
     * Grants permission to start an execution of a flow alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_StartFlowExecution.html
     */
    toStartFlowExecution(): this;
    /**
     * Grants permission to start an ingestion job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_StartIngestionJob.html
     */
    toStartIngestionJob(): this;
    /**
     * Grants permission to stop a evaluation job while in progress
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_StopEvaluationJob.html
     */
    toStopEvaluationJob(): this;
    /**
     * Grants permission to stop an execution of a flow alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_StopFlowExecution.html
     */
    toStopFlowExecution(): this;
    /**
     * Grants permission to stop an ingestion job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_StopIngestionJob.html
     */
    toStopIngestionJob(): this;
    /**
     * Grants permission to stop a Bedrock model customization job while in progress
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_StopModelCustomizationJob.html
     */
    toStopModelCustomizationJob(): this;
    /**
     * Grants permission to stop a model invocation job that you started earlier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_StopModelInvocationJob.html
     */
    toStopModelInvocationJob(): this;
    /**
     * Grants permission to Tag a Bedrock resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifBearerTokenType()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_TagResource.html
     */
    toTagResource(): this;
    /**
     * Grants permission to Untag a Bedrock resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifBearerTokenType()
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_UntagResource.html
     */
    toUntagResource(): this;
    /**
     * Grants permission to update an existing agent
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_UpdateAgent.html
     */
    toUpdateAgent(): this;
    /**
     * Grants permission to update an existing action group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_UpdateAgentActionGroup.html
     */
    toUpdateAgentActionGroup(): this;
    /**
     * Grants permission to update an existing alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_UpdateAgentAlias.html
     */
    toUpdateAgentAlias(): this;
    /**
     * Grants permission to update an existing collaborator
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_UpdateAgentCollaborator.html
     */
    toUpdateAgentCollaborator(): this;
    /**
     * Grants permission to update a knowledge base associated with an agent
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_UpdateAgentKnowledgeBase.html
     */
    toUpdateAgentKnowledgeBase(): this;
    /**
     * Grants permission to update an automated reasoning policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toUpdateAutomatedReasoningPolicy(): this;
    /**
     * Grants permission to update annotations for a build workflow for an automated reasoning policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toUpdateAutomatedReasoningPolicyAnnotations(): this;
    /**
     * Grants permission to update a test case for automated reasoning policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toUpdateAutomatedReasoningPolicyTestCase(): this;
    /**
     * Grants permission to update a blueprint for data automation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_data-automation_UpdateBlueprint.html
     */
    toUpdateBlueprint(): this;
    /**
     * Grants permission to update a data automation project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_data-automation_UpdateDataAutomationProject.html
     */
    toUpdateDataAutomationProject(): this;
    /**
     * Grants permission to update a data source
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_UpdateDataSource.html
     */
    toUpdateDataSource(): this;
    /**
     * Grants permission to update a prompt flow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_UpdateFlow.html
     */
    toUpdateFlow(): this;
    /**
     * Grants permission to update the configuration of an alias of a prompt flow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_UpdateFlowAlias.html
     */
    toUpdateFlowAlias(): this;
    /**
     * Grants permission to update a guardrail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     */
    toUpdateGuardrail(): this;
    /**
     * Grants permission to update a knowledge base
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_UpdateKnowledgeBase.html
     */
    toUpdateKnowledgeBase(): this;
    /**
     * Grants permission to update a marketplace model endpoint
     *
     * Access Level: Write
     */
    toUpdateMarketplaceModelEndpoint(): this;
    /**
     * Grants permission to update a prompt
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_UpdatePrompt.html
     */
    toUpdatePrompt(): this;
    /**
     * Grants permission to update a provisioned model throughput that you created earlier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_UpdateProvisionedModelThroughput.html
     */
    toUpdateProvisionedModelThroughput(): this;
    /**
     * Grants permission to update an existing session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_UpdateSession.html
     */
    toUpdateSession(): this;
    /**
     * Grants permission to validate prompt flow definitions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_ValidateFlowDefinition.html
     */
    toValidateFlowDefinition(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type foundation-model to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onFoundationModel(resourceId: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type async-invoke to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAsyncInvoke(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type inference-profile to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onInferenceProfile(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type default-prompt-router to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onDefaultPromptRouter(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type prompt-router to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onPromptRouter(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type application-inference-profile to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApplicationInferenceProfile(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type custom-model to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCustomModel(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type provisioned-model to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProvisionedModel(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type model-customization-job to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onModelCustomizationJob(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type agent to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param agentId - Identifier for the agentId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAgent(agentId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type agent-alias to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param agentId - Identifier for the agentId.
     * @param agentAliasId - Identifier for the agentAliasId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAgentAlias(agentId: string, agentAliasId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type knowledge-base to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param knowledgeBaseId - Identifier for the knowledgeBaseId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onKnowledgeBase(knowledgeBaseId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type model-evaluation-job to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onModelEvaluationJob(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type evaluation-job to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEvaluationJob(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type model-invocation-job to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param jobIdentifier - Identifier for the jobIdentifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onModelInvocationJob(jobIdentifier: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type guardrail to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param guardrailId - Identifier for the guardrailId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGuardrail(guardrailId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type guardrail-profile to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/userguide/guardrail-profiles-permissions.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onGuardrailProfile(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type automated-reasoning-policy to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param automatedReasoningPolicyId - Identifier for the automatedReasoningPolicyId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAutomatedReasoningPolicy(automatedReasoningPolicyId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type automated-reasoning-policy-version to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param automatedReasoningPolicyId - Identifier for the automatedReasoningPolicyId.
     * @param automatedReasoningPolicyVersion - Identifier for the automatedReasoningPolicyVersion.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAutomatedReasoningPolicyVersion(automatedReasoningPolicyId: string, automatedReasoningPolicyVersion: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type flow to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_FlowSummary.html
     *
     * @param flowId - Identifier for the flowId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFlow(flowId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type flow-alias to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_FlowAliasSummary.html
     *
     * @param flowId - Identifier for the flowId.
     * @param flowAliasId - Identifier for the flowAliasId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFlowAlias(flowId: string, flowAliasId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type flow-execution to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_FlowExecutionSummary.html
     *
     * @param flowId - Identifier for the flowId.
     * @param flowAliasId - Identifier for the flowAliasId.
     * @param flowExecutionId - Identifier for the flowExecutionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onFlowExecution(flowId: string, flowAliasId: string, flowExecutionId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type model-copy-job to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onModelCopyJob(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type prompt to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_PromptSummary.html
     *
     * @param promptId - Identifier for the promptId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPrompt(promptId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type prompt-version to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_PromptSummary.html
     *
     * @param promptId - Identifier for the promptId.
     * @param promptVersion - Identifier for the promptVersion.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPromptVersion(promptId: string, promptVersion: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type model-import-job to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onModelImportJob(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type imported-model to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onImportedModel(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type bedrock-marketplace-model-endpoint to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onBedrockMarketplaceModelEndpoint(account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type data-automation-project to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param projectId - Identifier for the projectId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDataAutomationProject(projectId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type blueprint to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param blueprintId - Identifier for the blueprintId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBlueprint(blueprintId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type data-automation-invocation-job to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDataAutomationInvocationJob(jobId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type data-automation-profile to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param profileId - Identifier for the profileId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onDataAutomationProfile(profileId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type session to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param sessionId - Identifier for the sessionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSession(sessionId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type custom-model-deployment to the statement
     *
     * https://docs.aws.amazon.com/bedrock/latest/APIReference/welcome.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCustomModelDeployment(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by creating requests based on the allowed set of values for each of the mandatory tags
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * Applies to actions:
     * - .toCreateAgent()
     * - .toCreateAgentActionGroup()
     * - .toCreateAgentAlias()
     * - .toCreateAutomatedReasoningPolicy()
     * - .toCreateAutomatedReasoningPolicyVersion()
     * - .toCreateBlueprint()
     * - .toCreateCustomModel()
     * - .toCreateCustomModelDeployment()
     * - .toCreateDataAutomationProject()
     * - .toCreateEvaluationJob()
     * - .toCreateFlow()
     * - .toCreateFlowAlias()
     * - .toCreateGuardrail()
     * - .toCreateInferenceProfile()
     * - .toCreateKnowledgeBase()
     * - .toCreateModelCopyJob()
     * - .toCreateModelCustomizationJob()
     * - .toCreateModelEvaluationJob()
     * - .toCreateModelImportJob()
     * - .toCreateModelInvocationJob()
     * - .toCreatePrompt()
     * - .toCreatePromptRouter()
     * - .toCreatePromptVersion()
     * - .toCreateProvisionedModelThroughput()
     * - .toCreateSession()
     * - .toInvokeDataAutomationAsync()
     * - .toInvokeModel()
     * - .toPutResourcePolicy()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by having actions based on the tag value associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * Applies to resource types:
     * - async-invoke
     * - application-inference-profile
     * - custom-model
     * - provisioned-model
     * - model-customization-job
     * - agent
     * - agent-alias
     * - knowledge-base
     * - model-evaluation-job
     * - evaluation-job
     * - model-invocation-job
     * - guardrail
     * - automated-reasoning-policy
     * - automated-reasoning-policy-version
     * - flow
     * - flow-alias
     * - model-copy-job
     * - prompt
     * - prompt-version
     * - model-import-job
     * - imported-model
     * - data-automation-project
     * - blueprint
     * - data-automation-invocation-job
     * - session
     * - custom-model-deployment
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by creating requests based on the presence of mandatory tags in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * Applies to actions:
     * - .toCreateAgent()
     * - .toCreateAgentActionGroup()
     * - .toCreateAgentAlias()
     * - .toCreateAutomatedReasoningPolicy()
     * - .toCreateAutomatedReasoningPolicyVersion()
     * - .toCreateBlueprint()
     * - .toCreateCustomModel()
     * - .toCreateCustomModelDeployment()
     * - .toCreateDataAutomationProject()
     * - .toCreateEvaluationJob()
     * - .toCreateFlow()
     * - .toCreateFlowAlias()
     * - .toCreateGuardrail()
     * - .toCreateInferenceProfile()
     * - .toCreateKnowledgeBase()
     * - .toCreateModelCopyJob()
     * - .toCreateModelCustomizationJob()
     * - .toCreateModelEvaluationJob()
     * - .toCreateModelImportJob()
     * - .toCreateModelInvocationJob()
     * - .toCreatePrompt()
     * - .toCreatePromptRouter()
     * - .toCreatePromptVersion()
     * - .toCreateProvisionedModelThroughput()
     * - .toCreateSession()
     * - .toInvokeDataAutomationAsync()
     * - .toInvokeModel()
     * - .toPutResourcePolicy()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the Short-term or Long-term bearer tokens
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonbedrock.html#amazonbedrock-policy-keys
     *
     * Applies to actions:
     * - .toCreateAutomatedReasoningPolicy()
     * - .toCreateAutomatedReasoningPolicyVersion()
     * - .toCreateCustomModel()
     * - .toCreateCustomModelDeployment()
     * - .toCreateEvaluationJob()
     * - .toCreateGuardrail()
     * - .toCreateModelCopyJob()
     * - .toCreateModelCustomizationJob()
     * - .toCreateModelEvaluationJob()
     * - .toCreateModelImportJob()
     * - .toCreateModelInvocationJob()
     * - .toCreatePromptRouter()
     * - .toCreateProvisionedModelThroughput()
     * - .toInvokeModel()
     * - .toInvokeModelWithResponseStream()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifBearerTokenType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the GuardrailIdentifier containing the GuardrailArn or the GuardrailArn:NumericVersion
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonbedrock.html#amazonbedrock-policy-keys
     *
     * Applies to actions:
     * - .toInvokeModel()
     * - .toInvokeModelWithResponseStream()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifGuardrailIdentifier(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the specified inference profile
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * Applies to actions:
     * - .toInvokeModel()
     * - .toInvokeModelWithResponseStream()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifInferenceProfileArn(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the Inline Agent Names, this will be used in InvokeInlineAgent API names
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonbedrock.html#amazonbedrock-policy-keys
     *
     * Applies to actions:
     * - .toInvokeInlineAgent()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInlineAgentName(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the specified prompt router
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * Applies to actions:
     * - .toInvokeModel()
     * - .toInvokeModelWithResponseStream()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifPromptRouterArn(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the secretArn containing the credentials of the third party platform
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * Applies to actions:
     * - .toAssociateThirdPartyKnowledgeBase()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifThirdPartyKnowledgeBaseCredentialsSecretArn(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [bedrock](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonbedrock.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
