"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BedrockAgentcore = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [bedrock-agentcore](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonbedrockagentcore.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class BedrockAgentcore extends shared_1.PolicyStatement {
    /**
     * Grants permission to configure vended telemetry for a resource
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toAllowVendedLogDeliveryForResource() {
        return this.to('AllowVendedLogDeliveryForResource');
    }
    /**
     * Grants permission to connect to a browser automation stream
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toConnectBrowserAutomationStream() {
        return this.to('ConnectBrowserAutomationStream');
    }
    /**
     * Grants permission to connect to a browser live view stream
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toConnectBrowserLiveViewStream() {
        return this.to('ConnectBrowserLiveViewStream');
    }
    /**
     * Grants permission to create a new agent runtime
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toCreateAgentRuntime() {
        return this.to('CreateAgentRuntime');
    }
    /**
     * Grants permission to create a new agent runtime endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toCreateAgentRuntimeEndpoint() {
        return this.to('CreateAgentRuntimeEndpoint');
    }
    /**
     * Grants permission to create a new API Key Credential Provider
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_CreateApiKeyCredentialProvider.html
     */
    toCreateApiKeyCredentialProvider() {
        return this.to('CreateApiKeyCredentialProvider');
    }
    /**
     * Grants permission to create a new custom browser
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toCreateBrowser() {
        return this.to('CreateBrowser');
    }
    /**
     * Grants permission to create a new custom code interpreter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toCreateCodeInterpreter() {
        return this.to('CreateCodeInterpreter');
    }
    /**
     * Grants permission to create an Event
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSessionId()
     * - .ifActorId()
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_CreateEvent.html
     */
    toCreateEvent() {
        return this.to('CreateEvent');
    }
    /**
     * Grants permission to create a new gateway
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_CreateGateway.html
     */
    toCreateGateway() {
        return this.to('CreateGateway');
    }
    /**
     * Grants permission to create a new target in an existing gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_CreateGatewayTarget.html
     */
    toCreateGatewayTarget() {
        return this.to('CreateGatewayTarget');
    }
    /**
     * Grants permission to create a Memory resource
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_CreateMemory.html
     */
    toCreateMemory() {
        return this.to('CreateMemory');
    }
    /**
     * Grants permission to create a new Credential Provider to access external resources with OAuth2 protocol
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_CreateOauth2CredentialProvider.html
     */
    toCreateOauth2CredentialProvider() {
        return this.to('CreateOauth2CredentialProvider');
    }
    /**
     * Grants permission to create a new Workload Identity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_CreateWorkloadIdentity.html
     */
    toCreateWorkloadIdentity() {
        return this.to('CreateWorkloadIdentity');
    }
    /**
     * Grants permission to delete an agent runtime
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toDeleteAgentRuntime() {
        return this.to('DeleteAgentRuntime');
    }
    /**
     * Grants permission to delete an agent runtime endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toDeleteAgentRuntimeEndpoint() {
        return this.to('DeleteAgentRuntimeEndpoint');
    }
    /**
     * Grants permission to delete a registered API Key Credential Provider
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_DeleteApiKeyCredentialProvider.html
     */
    toDeleteApiKeyCredentialProvider() {
        return this.to('DeleteApiKeyCredentialProvider');
    }
    /**
     * Grants permission to delete a custom browser
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toDeleteBrowser() {
        return this.to('DeleteBrowser');
    }
    /**
     * Grants permission to delete a custom code interpreter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toDeleteCodeInterpreter() {
        return this.to('DeleteCodeInterpreter');
    }
    /**
     * Grants permission to delete an Event
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSessionId()
     * - .ifActorId()
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_DeleteEvent.html
     */
    toDeleteEvent() {
        return this.to('DeleteEvent');
    }
    /**
     * Grants permission to delete an existing gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_DeleteGateway.html
     */
    toDeleteGateway() {
        return this.to('DeleteGateway');
    }
    /**
     * Grants permission to delete an existing gateway target
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_DeleteGatewayTarget.html
     */
    toDeleteGatewayTarget() {
        return this.to('DeleteGatewayTarget');
    }
    /**
     * Grants permission to delete a Memory resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_DeleteMemory.html
     */
    toDeleteMemory() {
        return this.to('DeleteMemory');
    }
    /**
     * Grants permission to delete a Memory Record
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_DeleteMemoryRecord.html
     */
    toDeleteMemoryRecord() {
        return this.to('DeleteMemoryRecord');
    }
    /**
     * Grants permission to delete a registered OAuth2 Credential Provider
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_DeleteOauth2CredentialProvider.html
     */
    toDeleteOauth2CredentialProvider() {
        return this.to('DeleteOauth2CredentialProvider');
    }
    /**
     * Grants permission to delete a registered Workload Identity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_DeleteWorkloadIdentity.html
     */
    toDeleteWorkloadIdentity() {
        return this.to('DeleteWorkloadIdentity');
    }
    /**
     * Grants permission to get details of an agent runtime
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toGetAgentRuntime() {
        return this.to('GetAgentRuntime');
    }
    /**
     * Grants permission to get details of an agent runtime endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toGetAgentRuntimeEndpoint() {
        return this.to('GetAgentRuntimeEndpoint');
    }
    /**
     * Grants permission to fetch a registered API Key Credential Provider by its name
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_GetApiKeyCredentialProvider.html
     */
    toGetApiKeyCredentialProvider() {
        return this.to('GetApiKeyCredentialProvider');
    }
    /**
     * Grants permission to get details of a browser
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toGetBrowser() {
        return this.to('GetBrowser');
    }
    /**
     * Grants permission to get details of a browser session
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toGetBrowserSession() {
        return this.to('GetBrowserSession');
    }
    /**
     * Grants permission to get details of a code interpreter
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toGetCodeInterpreter() {
        return this.to('GetCodeInterpreter');
    }
    /**
     * Grants permission to get details of a code interpreter session
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toGetCodeInterpreterSession() {
        return this.to('GetCodeInterpreterSession');
    }
    /**
     * Grants permission to fetch an Event
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifSessionId()
     * - .ifActorId()
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_GetEvent.html
     */
    toGetEvent() {
        return this.to('GetEvent');
    }
    /**
     * Grants permission to retrieve an existing gateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_GetGateway.html
     */
    toGetGateway() {
        return this.to('GetGateway');
    }
    /**
     * Grants permission to retrieve an existing gateway target
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_GetGatewayTarget.html
     */
    toGetGatewayTarget() {
        return this.to('GetGatewayTarget');
    }
    /**
     * Grants permission to fetch details for a Memory resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_GetMemory.html
     */
    toGetMemory() {
        return this.to('GetMemory');
    }
    /**
     * Grants permission to fetch a Memory Record
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_GetMemoryRecord.html
     */
    toGetMemoryRecord() {
        return this.to('GetMemoryRecord');
    }
    /**
     * Grants permission to fetch a registered OAuth2 Credential Provider by its name
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_GetOauth2CredentialProvider.html
     */
    toGetOauth2CredentialProvider() {
        return this.to('GetOauth2CredentialProvider');
    }
    /**
     * Grants permission to retrieve an API Key associated with an Api Key Credential Provider
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_GetResourceApiKey.html
     */
    toGetResourceApiKey() {
        return this.to('GetResourceApiKey');
    }
    /**
     * Grants permission to retrieve access token with OAuth2 2LO or 3LO flow to access external resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_GetResourceOauth2Token.html
     */
    toGetResourceOauth2Token() {
        return this.to('GetResourceOauth2Token');
    }
    /**
     * Grants permission to fetch the current configuration of the TokenVault, including encryption settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_GetTokenVault.html
     */
    toGetTokenVault() {
        return this.to('GetTokenVault');
    }
    /**
     * Grants permission to retrieve an Workload access token for agentic workloads not acting on behalf of a user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_GetWorkloadAccessToken.html
     */
    toGetWorkloadAccessToken() {
        return this.to('GetWorkloadAccessToken');
    }
    /**
     * Grants permission to retrieve an Workload access token for agentic workloads acting on behalf of user with JWT token
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_GetWorkloadAccessTokenForJWT.html
     */
    toGetWorkloadAccessTokenForJWT() {
        return this.to('GetWorkloadAccessTokenForJWT');
    }
    /**
     * Grants permission to retrieve an Workload access token for agentic workloads acting on behalf of user with User Id
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_GetWorkloadAccessTokenForUserId.html
     */
    toGetWorkloadAccessTokenForUserId() {
        return this.to('GetWorkloadAccessTokenForUserId');
    }
    /**
     * Grants permission to fetch details for a specific Workload identity, including its name and allowed OAuth2 return URLs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_GetWorkloadIdentity.html
     */
    toGetWorkloadIdentity() {
        return this.to('GetWorkloadIdentity');
    }
    /**
     * Grants permission to invoke an agent runtime endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toInvokeAgentRuntime() {
        return this.to('InvokeAgentRuntime');
    }
    /**
     * Grants permission to invoke a code interpreter session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toInvokeCodeInterpreter() {
        return this.to('InvokeCodeInterpreter');
    }
    /**
     * Grants permission to list Actors
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_ListActors.html
     */
    toListActors() {
        return this.to('ListActors');
    }
    /**
     * Grants permission to list agent runtime endpoints
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toListAgentRuntimeEndpoints() {
        return this.to('ListAgentRuntimeEndpoints');
    }
    /**
     * Grants permission to list agent runtime versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toListAgentRuntimeVersions() {
        return this.to('ListAgentRuntimeVersions');
    }
    /**
     * Grants permission to list agent runtimes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toListAgentRuntimes() {
        return this.to('ListAgentRuntimes');
    }
    /**
     * Grants permission to list all API Key Credential Providers in the Token Vault
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_ListApiKeyCredentialProviders.html
     */
    toListApiKeyCredentialProviders() {
        return this.to('ListApiKeyCredentialProviders');
    }
    /**
     * Grants permission to list browser sessions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toListBrowserSessions() {
        return this.to('ListBrowserSessions');
    }
    /**
     * Grants permission to list browsers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toListBrowsers() {
        return this.to('ListBrowsers');
    }
    /**
     * Grants permission to list code interpreter sessions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toListCodeInterpreterSessions() {
        return this.to('ListCodeInterpreterSessions');
    }
    /**
     * Grants permission to list code interpreters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toListCodeInterpreters() {
        return this.to('ListCodeInterpreters');
    }
    /**
     * Grants permission to list events
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifSessionId()
     * - .ifActorId()
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_ListEvents.html
     */
    toListEvents() {
        return this.to('ListEvents');
    }
    /**
     * Grants permission to list existing gateway targets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_ListGatewayTargets.html
     */
    toListGatewayTargets() {
        return this.to('ListGatewayTargets');
    }
    /**
     * Grants permission to list existing gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_ListGateways.html
     */
    toListGateways() {
        return this.to('ListGateways');
    }
    /**
     * Grants permission to list memory resources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_ListMemories.html
     */
    toListMemories() {
        return this.to('ListMemories');
    }
    /**
     * Grants permission to list memory records
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifNamespace()
     * - .ifStrategyId()
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_ListMemoryRecords.html
     */
    toListMemoryRecords() {
        return this.to('ListMemoryRecords');
    }
    /**
     * Grants permission to list all OAuth2 Credential Providers in the Token Vault
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_ListOauth2CredentialProviders.html
     */
    toListOauth2CredentialProviders() {
        return this.to('ListOauth2CredentialProviders');
    }
    /**
     * Grants permission to list sessions
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifActorId()
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_ListSessions.html
     */
    toListSessions() {
        return this.to('ListSessions');
    }
    /**
     * Grants permission to list all Workload Identities in the caller's AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_ListWorkloadIdentities.html
     */
    toListWorkloadIdentities() {
        return this.to('ListWorkloadIdentities');
    }
    /**
     * Grants permission to retrieve memory records through sematic query
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifNamespace()
     * - .ifStrategyId()
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_RetrieveMemoryRecords.html
     */
    toRetrieveMemoryRecords() {
        return this.to('RetrieveMemoryRecords');
    }
    /**
     * Grants permission to associate a Customer Managed Key (CMK) or a Service Managed Key with a specific TokenVault
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_SetTokenVaultCMK.html
     */
    toSetTokenVaultCMK() {
        return this.to('SetTokenVaultCMK');
    }
    /**
     * Grants permission to starts a new browser session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toStartBrowserSession() {
        return this.to('StartBrowserSession');
    }
    /**
     * Grants permission to start a new code interpreter session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toStartCodeInterpreterSession() {
        return this.to('StartCodeInterpreterSession');
    }
    /**
     * Grants permission to stop a browser session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toStopBrowserSession() {
        return this.to('StopBrowserSession');
    }
    /**
     * Grants permission to stop a code interpreter session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toStopCodeInterpreterSession() {
        return this.to('StopCodeInterpreterSession');
    }
    /**
     * Grants permission to enable search on gateways
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/welcome.html
     */
    toSynchronizeGatewayTargets() {
        return this.to('SynchronizeGatewayTargets');
    }
    /**
     * Grants permission to update an agent runtime
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toUpdateAgentRuntime() {
        return this.to('UpdateAgentRuntime');
    }
    /**
     * Grants permission to update an agent runtime endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toUpdateAgentRuntimeEndpoint() {
        return this.to('UpdateAgentRuntimeEndpoint');
    }
    /**
     * Grants permission to update an existing API Key Credential Provider
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_UpdateApiKeyCredentialProvider.html
     */
    toUpdateApiKeyCredentialProvider() {
        return this.to('UpdateApiKeyCredentialProvider');
    }
    /**
     * Grants permission to update the status of browser session stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/
     */
    toUpdateBrowserStream() {
        return this.to('UpdateBrowserStream');
    }
    /**
     * Grants permission to update an existing gateway
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_UpdateGateway.html
     */
    toUpdateGateway() {
        return this.to('UpdateGateway');
    }
    /**
     * Grants permission to update an existing gateway target
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_UpdateGatewayTarget.html
     */
    toUpdateGatewayTarget() {
        return this.to('UpdateGatewayTarget');
    }
    /**
     * Grants permission to update a Memory resource
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_UpdateMemory.html
     */
    toUpdateMemory() {
        return this.to('UpdateMemory');
    }
    /**
     * Grants permission to update an existing OAuth2 Credential Provider
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_UpdateOauth2CredentialProvider.html
     */
    toUpdateOauth2CredentialProvider() {
        return this.to('UpdateOauth2CredentialProvider');
    }
    /**
     * Grants permission to update the metadata of an existing Workload Identity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/API_UpdateWorkloadIdentity.html
     */
    toUpdateWorkloadIdentity() {
        return this.to('UpdateWorkloadIdentity');
    }
    /**
     * Adds a resource of type memory to the statement
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/memory.html
     *
     * @param memoryId - Identifier for the memoryId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onMemory(memoryId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:bedrock-agentcore:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:memory/${memoryId}`);
    }
    /**
     * Adds a resource of type gateway to the statement
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/gateway.html
     *
     * @param gatewayId - Identifier for the gatewayId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onGateway(gatewayId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:bedrock-agentcore:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:gateway/${gatewayId}`);
    }
    /**
     * Adds a resource of type workload-identity to the statement
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/workloadIdentity.html
     *
     * @param directoryId - Identifier for the directoryId.
     * @param workloadIdentityName - Identifier for the workloadIdentityName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onWorkloadIdentity(directoryId, workloadIdentityName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:bedrock-agentcore:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workload-identity-directory/${directoryId}/workload-identity/${workloadIdentityName}`);
    }
    /**
     * Adds a resource of type oauth2credentialprovider to the statement
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/oauth2credentialprovider.html
     *
     * @param tokenVaultId - Identifier for the tokenVaultId.
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onOauth2credentialprovider(tokenVaultId, name, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:bedrock-agentcore:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:token-vault/${tokenVaultId}/oauth2credentialprovider/${name}`);
    }
    /**
     * Adds a resource of type apikeycredentialprovider to the statement
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/apikeycredentialprovider.html
     *
     * @param tokenVaultId - Identifier for the tokenVaultId.
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onApikeycredentialprovider(tokenVaultId, name, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:bedrock-agentcore:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:token-vault/${tokenVaultId}/apikeycredentialprovider/${name}`);
    }
    /**
     * Adds a resource of type runtime to the statement
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/runtime.html
     *
     * @param runtimeId - Identifier for the runtimeId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onRuntime(runtimeId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:bedrock-agentcore:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:runtime/${runtimeId}`);
    }
    /**
     * Adds a resource of type runtime-endpoint to the statement
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/runtimeEndpoint.html
     *
     * @param runtimeId - Identifier for the runtimeId.
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onRuntimeEndpoint(runtimeId, name, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:bedrock-agentcore:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:runtime/${runtimeId}/runtime-endpoint/${name}`);
    }
    /**
     * Adds a resource of type code-interpreter-custom to the statement
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/codeInterpreter.html
     *
     * @param codeInterpreterId - Identifier for the codeInterpreterId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onCodeInterpreterCustom(codeInterpreterId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:bedrock-agentcore:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:code-interpreter-custom/${codeInterpreterId}`);
    }
    /**
     * Adds a resource of type code-interpreter to the statement
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/codeInterpreter.html
     *
     * @param codeInterpreterId - Identifier for the codeInterpreterId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onCodeInterpreter(codeInterpreterId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:bedrock-agentcore:${region ?? this.defaultRegion}:aws:code-interpreter/${codeInterpreterId}`);
    }
    /**
     * Adds a resource of type browser-custom to the statement
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/browser.html
     *
     * @param browserId - Identifier for the browserId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onBrowserCustom(browserId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:bedrock-agentcore:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:browser-custom/${browserId}`);
    }
    /**
     * Adds a resource of type browser to the statement
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/browser.html
     *
     * @param browserId - Identifier for the browserId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onBrowser(browserId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:bedrock-agentcore:${region ?? this.defaultRegion}:aws:browser/${browserId}`);
    }
    /**
     * Adds a resource of type workload-identity-directory to the statement
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/workloadIdentityDirectory.html
     *
     * @param directoryId - Identifier for the directoryId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onWorkloadIdentityDirectory(directoryId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:bedrock-agentcore:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workload-identity-directory/${directoryId}`);
    }
    /**
     * Adds a resource of type token-vault to the statement
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/tokenVault.html
     *
     * @param tokenVaultId - Identifier for the tokenVaultId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onTokenVault(tokenVaultId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:bedrock-agentcore:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:token-vault/${tokenVaultId}`);
    }
    /**
     * Filters access by Actor Id
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/#condition-keys-actorId
     *
     * Applies to actions:
     * - .toCreateEvent()
     * - .toDeleteEvent()
     * - .toGetEvent()
     * - .toListEvents()
     * - .toListSessions()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifActorId(value, operator) {
        return this.if(`actorId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by namespace
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/#condition-keys-namespace
     *
     * Applies to actions:
     * - .toListMemoryRecords()
     * - .toRetrieveMemoryRecords()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifNamespace(value, operator) {
        return this.if(`namespace`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by Session Id
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/#condition-keys-sessionId
     *
     * Applies to actions:
     * - .toCreateEvent()
     * - .toDeleteEvent()
     * - .toGetEvent()
     * - .toListEvents()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSessionId(value, operator) {
        return this.if(`sessionId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by Memory Strategy Id
     *
     * https://docs.aws.amazon.com/bedrock-agentcore/latest/APIReference/#condition-keys-strategyId
     *
     * Applies to actions:
     * - .toListMemoryRecords()
     * - .toRetrieveMemoryRecords()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifStrategyId(value, operator) {
        return this.if(`strategyId`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [bedrock-agentcore](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonbedrockagentcore.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'bedrock-agentcore';
        this.accessLevelList = {
            'Permissions management': [
                'AllowVendedLogDeliveryForResource',
                'SynchronizeGatewayTargets'
            ],
            Read: [
                'ConnectBrowserAutomationStream',
                'ConnectBrowserLiveViewStream',
                'GetAgentRuntime',
                'GetAgentRuntimeEndpoint',
                'GetApiKeyCredentialProvider',
                'GetBrowser',
                'GetBrowserSession',
                'GetCodeInterpreter',
                'GetCodeInterpreterSession',
                'GetEvent',
                'GetGateway',
                'GetGatewayTarget',
                'GetMemory',
                'GetMemoryRecord',
                'GetOauth2CredentialProvider',
                'GetResourceApiKey',
                'GetResourceOauth2Token',
                'GetTokenVault',
                'GetWorkloadIdentity',
                'ListApiKeyCredentialProviders',
                'ListOauth2CredentialProviders',
                'ListWorkloadIdentities',
                'SetTokenVaultCMK'
            ],
            Write: [
                'CreateAgentRuntime',
                'CreateAgentRuntimeEndpoint',
                'CreateApiKeyCredentialProvider',
                'CreateBrowser',
                'CreateCodeInterpreter',
                'CreateEvent',
                'CreateGateway',
                'CreateGatewayTarget',
                'CreateMemory',
                'CreateOauth2CredentialProvider',
                'CreateWorkloadIdentity',
                'DeleteAgentRuntime',
                'DeleteAgentRuntimeEndpoint',
                'DeleteApiKeyCredentialProvider',
                'DeleteBrowser',
                'DeleteCodeInterpreter',
                'DeleteEvent',
                'DeleteGateway',
                'DeleteGatewayTarget',
                'DeleteMemory',
                'DeleteMemoryRecord',
                'DeleteOauth2CredentialProvider',
                'DeleteWorkloadIdentity',
                'GetWorkloadAccessToken',
                'GetWorkloadAccessTokenForJWT',
                'GetWorkloadAccessTokenForUserId',
                'InvokeAgentRuntime',
                'InvokeCodeInterpreter',
                'StartBrowserSession',
                'StartCodeInterpreterSession',
                'StopBrowserSession',
                'StopCodeInterpreterSession',
                'UpdateAgentRuntime',
                'UpdateAgentRuntimeEndpoint',
                'UpdateApiKeyCredentialProvider',
                'UpdateBrowserStream',
                'UpdateGateway',
                'UpdateGatewayTarget',
                'UpdateMemory',
                'UpdateOauth2CredentialProvider',
                'UpdateWorkloadIdentity'
            ],
            List: [
                'ListActors',
                'ListAgentRuntimeEndpoints',
                'ListAgentRuntimeVersions',
                'ListAgentRuntimes',
                'ListBrowserSessions',
                'ListBrowsers',
                'ListCodeInterpreterSessions',
                'ListCodeInterpreters',
                'ListEvents',
                'ListGatewayTargets',
                'ListGateways',
                'ListMemories',
                'ListMemoryRecords',
                'ListSessions',
                'RetrieveMemoryRecords'
            ]
        };
    }
}
exports.BedrockAgentcore = BedrockAgentcore;
//# sourceMappingURL=data:application/json;base64,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