"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Chatbot = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [chatbot](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awschatbot.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Chatbot extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate a resource with a configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_AssociateToConfiguration.html
     */
    toAssociateToConfiguration() {
        return this.to('AssociateToConfiguration');
    }
    /**
     * Grants permission to create an AWS Chatbot Chime Webhook Configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_CreateChimeWebhookConfiguration.html
     */
    toCreateChimeWebhookConfiguration() {
        return this.to('CreateChimeWebhookConfiguration');
    }
    /**
     * Grants permission to create a custom action
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_CreateCustomAction.html
     */
    toCreateCustomAction() {
        return this.to('CreateCustomAction');
    }
    /**
     * Grants permission to create an AWS Chatbot Microsoft Teams Channel Configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_CreateMicrosoftTeamsChannelConfiguration.html
     */
    toCreateMicrosoftTeamsChannelConfiguration() {
        return this.to('CreateMicrosoftTeamsChannelConfiguration');
    }
    /**
     * Grants permission to create an AWS Chatbot Slack Channel Configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_CreateSlackChannelConfiguration.html
     */
    toCreateSlackChannelConfiguration() {
        return this.to('CreateSlackChannelConfiguration');
    }
    /**
     * Grants permission to delete an AWS Chatbot Chime Webhook Configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_DeleteChimeWebhookConfiguration.html
     */
    toDeleteChimeWebhookConfiguration() {
        return this.to('DeleteChimeWebhookConfiguration');
    }
    /**
     * Grants permission to delete a custom action
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_DeleteCustomAction.html
     */
    toDeleteCustomAction() {
        return this.to('DeleteCustomAction');
    }
    /**
     * Grants permission to delete an AWS Chatbot Microsoft Teams Channel Configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_DeleteMicrosoftTeamsChannelConfiguration.html
     */
    toDeleteMicrosoftTeamsChannelConfiguration() {
        return this.to('DeleteMicrosoftTeamsChannelConfiguration');
    }
    /**
     * Grants permission to delete the Microsoft Teams configured with AWS Chatbot in an AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_DeleteMicrosoftTeamsConfiguredTeam.html
     */
    toDeleteMicrosoftTeamsConfiguredTeam() {
        return this.to('DeleteMicrosoftTeamsConfiguredTeam');
    }
    /**
     * Grants permission to delete an AWS Chatbot Microsoft Teams User Identity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_DeleteMicrosoftTeamsUserIdentity.html
     */
    toDeleteMicrosoftTeamsUserIdentity() {
        return this.to('DeleteMicrosoftTeamsUserIdentity');
    }
    /**
     * Grants permission to delete an AWS Chatbot Slack Channel Configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_DeleteSlackChannelConfiguration.html
     */
    toDeleteSlackChannelConfiguration() {
        return this.to('DeleteSlackChannelConfiguration');
    }
    /**
     * Grants permission to delete an AWS Chatbot Slack User Identity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_DeleteSlackUserIdentity.html
     */
    toDeleteSlackUserIdentity() {
        return this.to('DeleteSlackUserIdentity');
    }
    /**
     * Grants permission to delete the Slack workspace authorization with AWS Chatbot, associated with an AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_DeleteSlackWorkspaceAuthorization.html
     */
    toDeleteSlackWorkspaceAuthorization() {
        return this.to('DeleteSlackWorkspaceAuthorization');
    }
    /**
     * Grants permission to list all AWS Chatbot Chime Webhook Configurations in an AWS Account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_DescribeChimeWebhookConfigurations.html
     */
    toDescribeChimeWebhookConfigurations() {
        return this.to('DescribeChimeWebhookConfigurations');
    }
    /**
     * Grants permission to list all AWS Chatbot Slack Channel Configurations in an AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_DescribeSlackChannelConfigurations.html
     */
    toDescribeSlackChannelConfigurations() {
        return this.to('DescribeSlackChannelConfigurations');
    }
    /**
     * Grants permission to list all public Slack channels in the Slack workspace connected to the AWS Account onboarded with AWS Chatbot service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_Operations.html
     */
    toDescribeSlackChannels() {
        return this.to('DescribeSlackChannels');
    }
    /**
     * Grants permission to describe AWS Chatbot Slack User Identities
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_DescribeSlackUserIdentities.html
     */
    toDescribeSlackUserIdentities() {
        return this.to('DescribeSlackUserIdentities');
    }
    /**
     * Grants permission to list all authorized Slack workspaces connected to the AWS Account onboarded with AWS Chatbot service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_DescribeSlackWorkspaces.html
     */
    toDescribeSlackWorkspaces() {
        return this.to('DescribeSlackWorkspaces');
    }
    /**
     * Grants permission to disassociate a resource from a configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_DisassociateFromConfiguration.html
     */
    toDisassociateFromConfiguration() {
        return this.to('DisassociateFromConfiguration');
    }
    /**
     * Grants permission to retrieve AWS Chatbot account preferences
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_GetAccountPreferences.html
     */
    toGetAccountPreferences() {
        return this.to('GetAccountPreferences');
    }
    /**
     * Grants permission to get a custom action
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_GetCustomAction.html
     */
    toGetCustomAction() {
        return this.to('GetCustomAction');
    }
    /**
     * Grants permission to get a single AWS Chatbot Microsoft Teams Channel Configurations in an AWS account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_GetMicrosoftTeamsChannelConfiguration.html
     */
    toGetMicrosoftTeamsChannelConfiguration() {
        return this.to('GetMicrosoftTeamsChannelConfiguration');
    }
    /**
     * Grants permission to generate OAuth parameters to request Microsoft Teams OAuth code to be used by the AWS Chatbot service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_Operations.html
     */
    toGetMicrosoftTeamsOauthParameters() {
        return this.to('GetMicrosoftTeamsOauthParameters');
    }
    /**
     * Grants permission to generate OAuth parameters to request Slack OAuth code to be used by the AWS Chatbot service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_Operations.html
     */
    toGetSlackOauthParameters() {
        return this.to('GetSlackOauthParameters');
    }
    /**
     * Grants permission to list resources associated with a configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_ListAssociations.html
     */
    toListAssociations() {
        return this.to('ListAssociations');
    }
    /**
     * Grants permission to list custom actions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_ListCustomActions.html
     */
    toListCustomActions() {
        return this.to('ListCustomActions');
    }
    /**
     * Grants permission to list all AWS Chatbot Microsoft Teams Channel Configurations in an AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_ListMicrosoftTeamsChannelConfigurations.html
     */
    toListMicrosoftTeamsChannelConfigurations() {
        return this.to('ListMicrosoftTeamsChannelConfigurations');
    }
    /**
     * Grants permission to list all Microsoft Teams connected to the AWS Account onboarded with AWS Chatbot service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_ListMicrosoftTeamsConfiguredTeams.html
     */
    toListMicrosoftTeamsConfiguredTeams() {
        return this.to('ListMicrosoftTeamsConfiguredTeams');
    }
    /**
     * Grants permission to describe AWS Chatbot Microsoft Teams User Identities
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_ListMicrosoftTeamsUserIdentities.html
     */
    toListMicrosoftTeamsUserIdentities() {
        return this.to('ListMicrosoftTeamsUserIdentities');
    }
    /**
     * Grants permission to List all tags associated with the AWS Chatbot Channel Configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to redeem previously generated parameters with Microsoft APIs, to acquire OAuth tokens to be used by the AWS Chatbot service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_Operations.html
     */
    toRedeemMicrosoftTeamsOauthCode() {
        return this.to('RedeemMicrosoftTeamsOauthCode');
    }
    /**
     * Grants permission to redeem previously generated parameters with Slack API, to acquire OAuth tokens to be used by the AWS Chatbot service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_Operations.html
     */
    toRedeemSlackOauthCode() {
        return this.to('RedeemSlackOauthCode');
    }
    /**
     * Grants permission to create tags on AWS Chatbot Channel Configuration
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags on AWS Chatbot Channel Configuration
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update AWS Chatbot account preferences
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_UpdateAccountPreferences.html
     */
    toUpdateAccountPreferences() {
        return this.to('UpdateAccountPreferences');
    }
    /**
     * Grants permission to update an AWS Chatbot Chime Webhook Configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_UpdateChimeWebhookConfiguration.html
     */
    toUpdateChimeWebhookConfiguration() {
        return this.to('UpdateChimeWebhookConfiguration');
    }
    /**
     * Grants permission to update a custom action
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_UpdateCustomAction.html
     */
    toUpdateCustomAction() {
        return this.to('UpdateCustomAction');
    }
    /**
     * Grants permission to update an AWS Chatbot Microsoft Teams Channel Configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_UpdateMicrosoftTeamsChannelConfiguration.html
     */
    toUpdateMicrosoftTeamsChannelConfiguration() {
        return this.to('UpdateMicrosoftTeamsChannelConfiguration');
    }
    /**
     * Grants permission to update an AWS Chatbot Slack Channel Configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_UpdateSlackChannelConfiguration.html
     */
    toUpdateSlackChannelConfiguration() {
        return this.to('UpdateSlackChannelConfiguration');
    }
    /**
     * Adds a resource of type ChatbotConfiguration to the statement
     *
     * https://docs.aws.amazon.com/chatbot/latest/adminguide/what-is.html
     *
     * @param configurationType - Identifier for the configurationType.
     * @param chatbotConfigurationName - Identifier for the chatbotConfigurationName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onChatbotConfiguration(configurationType, chatbotConfigurationName, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:chatbot::${account ?? this.defaultAccount}:chat-configuration/${configurationType}/${chatbotConfigurationName}`);
    }
    /**
     * Adds a resource of type custom-action to the statement
     *
     * https://docs.aws.amazon.com/chatbot/latest/adminguide/what-is.html
     *
     * @param actionName - Identifier for the actionName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCustomAction(actionName, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:chatbot::${account ?? this.defaultAccount}:custom-action/${actionName}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateChimeWebhookConfiguration()
     * - .toCreateCustomAction()
     * - .toCreateMicrosoftTeamsChannelConfiguration()
     * - .toCreateSlackChannelConfiguration()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toAssociateToConfiguration()
     * - .toDeleteChimeWebhookConfiguration()
     * - .toDeleteCustomAction()
     * - .toDeleteMicrosoftTeamsChannelConfiguration()
     * - .toDeleteSlackChannelConfiguration()
     * - .toDisassociateFromConfiguration()
     * - .toGetCustomAction()
     * - .toGetMicrosoftTeamsChannelConfiguration()
     * - .toUpdateChimeWebhookConfiguration()
     * - .toUpdateCustomAction()
     * - .toUpdateMicrosoftTeamsChannelConfiguration()
     * - .toUpdateSlackChannelConfiguration()
     *
     * Applies to resource types:
     * - ChatbotConfiguration
     * - custom-action
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateChimeWebhookConfiguration()
     * - .toCreateCustomAction()
     * - .toCreateMicrosoftTeamsChannelConfiguration()
     * - .toCreateSlackChannelConfiguration()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [chatbot](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awschatbot.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'chatbot';
        this.accessLevelList = {
            Write: [
                'AssociateToConfiguration',
                'CreateChimeWebhookConfiguration',
                'CreateCustomAction',
                'CreateMicrosoftTeamsChannelConfiguration',
                'CreateSlackChannelConfiguration',
                'DeleteChimeWebhookConfiguration',
                'DeleteCustomAction',
                'DeleteMicrosoftTeamsChannelConfiguration',
                'DeleteMicrosoftTeamsConfiguredTeam',
                'DeleteMicrosoftTeamsUserIdentity',
                'DeleteSlackChannelConfiguration',
                'DeleteSlackUserIdentity',
                'DeleteSlackWorkspaceAuthorization',
                'DisassociateFromConfiguration',
                'RedeemMicrosoftTeamsOauthCode',
                'RedeemSlackOauthCode',
                'UpdateAccountPreferences',
                'UpdateChimeWebhookConfiguration',
                'UpdateCustomAction',
                'UpdateMicrosoftTeamsChannelConfiguration',
                'UpdateSlackChannelConfiguration'
            ],
            Read: [
                'DescribeChimeWebhookConfigurations',
                'DescribeSlackChannelConfigurations',
                'DescribeSlackChannels',
                'DescribeSlackUserIdentities',
                'DescribeSlackWorkspaces',
                'GetAccountPreferences',
                'GetCustomAction',
                'GetMicrosoftTeamsChannelConfiguration',
                'GetMicrosoftTeamsOauthParameters',
                'GetSlackOauthParameters',
                'ListAssociations',
                'ListMicrosoftTeamsChannelConfigurations',
                'ListMicrosoftTeamsConfiguredTeams',
                'ListMicrosoftTeamsUserIdentities',
                'ListTagsForResource'
            ],
            List: [
                'ListCustomActions'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Chatbot = Chatbot;
//# sourceMappingURL=data:application/json;base64,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