"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cleanrooms = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [cleanrooms](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscleanrooms.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Cleanrooms extends shared_1.PolicyStatement {
    /**
     * Grants permission to view details of analysisTemplates associated to the collaboration
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - cleanrooms:GetCollaborationAnalysisTemplate
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_BatchGetCollaborationAnalysisTemplate.html
     */
    toBatchGetCollaborationAnalysisTemplate() {
        return this.to('BatchGetCollaborationAnalysisTemplate');
    }
    /**
     * Grants permission to view details for schemas
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - cleanrooms:GetSchema
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_BatchGetSchema.html
     */
    toBatchGetSchema() {
        return this.to('BatchGetSchema');
    }
    /**
     * Grants permission to view analysis rules associated with schemas
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - cleanrooms:GetSchema
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_BatchGetSchemaAnalysisRule.html
     */
    toBatchGetSchemaAnalysisRule() {
        return this.to('BatchGetSchemaAnalysisRule');
    }
    /**
     * Grants permission to create a new analysis template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_CreateAnalysisTemplate.html
     */
    toCreateAnalysisTemplate() {
        return this.to('CreateAnalysisTemplate');
    }
    /**
     * Grants permission to create a new collaboration, a shared data collaboration environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_CreateCollaboration.html
     */
    toCreateCollaboration() {
        return this.to('CreateCollaboration');
    }
    /**
     * Grants permission to create a change request in a collaboration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_CreateCollaborationChangeRequest.html
     */
    toCreateCollaborationChangeRequest() {
        return this.to('CreateCollaborationChangeRequest');
    }
    /**
     * Grants permission to link a Cleanrooms ML configured audience model with a collaboration by creating a new association
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - cleanrooms-ml:GetConfiguredAudienceModel
     * - cleanrooms-ml:GetConfiguredAudienceModelPolicy
     * - cleanrooms-ml:PutConfiguredAudienceModelPolicy
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_CreateConfiguredAudienceModelAssociation.html
     */
    toCreateConfiguredAudienceModelAssociation() {
        return this.to('CreateConfiguredAudienceModelAssociation');
    }
    /**
     * Grants permission to create a new configured table
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - athena:GetTableMetadata
     * - glue:BatchGetPartition
     * - glue:GetDatabase
     * - glue:GetDatabases
     * - glue:GetPartition
     * - glue:GetPartitions
     * - glue:GetSchemaVersion
     * - glue:GetTable
     * - glue:GetTables
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_CreateConfiguredTable.html
     */
    toCreateConfiguredTable() {
        return this.to('CreateConfiguredTable');
    }
    /**
     * Grants permission to create a analysis rule for a configured table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_CreateConfiguredTableAnalysisRule.html
     */
    toCreateConfiguredTableAnalysisRule() {
        return this.to('CreateConfiguredTableAnalysisRule');
    }
    /**
     * Grants permission to link a configured table with a collaboration by creating a new association
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_CreateConfiguredTableAssociation.html
     */
    toCreateConfiguredTableAssociation() {
        return this.to('CreateConfiguredTableAssociation');
    }
    /**
     * Grants permission to create an analysis rule for a configured table association
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_CreateConfiguredTableAssociationAnalysisRule.html
     */
    toCreateConfiguredTableAssociationAnalysisRule() {
        return this.to('CreateConfiguredTableAssociationAnalysisRule');
    }
    /**
     * Grants permission to link an id mapping workflow with a collaboration by creating a new id mapping table
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - entityresolution:AddPolicyStatement
     * - entityresolution:GetIdMappingWorkflow
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_CreateIdMappingTable.html
     */
    toCreateIdMappingTable() {
        return this.to('CreateIdMappingTable');
    }
    /**
     * Grants permission to link an AWS Entity Resolution Id Namespace with a collaboration by creating a new association
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - entityresolution:AddPolicyStatement
     * - entityresolution:GetIdNamespace
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_CreateIdNamespaceAssociation.html
     */
    toCreateIdNamespaceAssociation() {
        return this.to('CreateIdNamespaceAssociation');
    }
    /**
     * Grants permission to join collaborations by creating a membership
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     * - logs:CreateLogDelivery
     * - logs:CreateLogGroup
     * - logs:DeleteLogDelivery
     * - logs:DescribeLogGroups
     * - logs:DescribeResourcePolicies
     * - logs:GetLogDelivery
     * - logs:ListLogDeliveries
     * - logs:PutResourcePolicy
     * - logs:UpdateLogDelivery
     * - s3:GetBucketLocation
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_CreateMembership.html
     */
    toCreateMembership() {
        return this.to('CreateMembership');
    }
    /**
     * Grants permission to create a new privacy budget template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_CreatePrivacyBudgetTemplate.html
     */
    toCreatePrivacyBudgetTemplate() {
        return this.to('CreatePrivacyBudgetTemplate');
    }
    /**
     * Grants permission to delete an existing analysis template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_DeleteAnalysisTemplate.html
     */
    toDeleteAnalysisTemplate() {
        return this.to('DeleteAnalysisTemplate');
    }
    /**
     * Grants permission to delete an existing collaboration
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - cleanrooms-ml:DeleteConfiguredAudienceModelPolicy
     * - cleanrooms-ml:GetConfiguredAudienceModelPolicy
     * - cleanrooms-ml:PutConfiguredAudienceModelPolicy
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_DeleteCollaboration.html
     */
    toDeleteCollaboration() {
        return this.to('DeleteCollaboration');
    }
    /**
     * Grants permission to delete an existing configured audience model association
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - cleanrooms-ml:DeleteConfiguredAudienceModelPolicy
     * - cleanrooms-ml:GetConfiguredAudienceModelPolicy
     * - cleanrooms-ml:PutConfiguredAudienceModelPolicy
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_DeleteConfiguredAudienceModelAssociation.html
     */
    toDeleteConfiguredAudienceModelAssociation() {
        return this.to('DeleteConfiguredAudienceModelAssociation');
    }
    /**
     * Grants permission to delete a configured table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_DeleteConfiguredTable.html
     */
    toDeleteConfiguredTable() {
        return this.to('DeleteConfiguredTable');
    }
    /**
     * Grants permission to delete an existing analysis rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_DeleteConfiguredTableAnalysisRule.html
     */
    toDeleteConfiguredTableAnalysisRule() {
        return this.to('DeleteConfiguredTableAnalysisRule');
    }
    /**
     * Grants permission to remove a configured table association from a collaboration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_DeleteConfiguredTableAssociation.html
     */
    toDeleteConfiguredTableAssociation() {
        return this.to('DeleteConfiguredTableAssociation');
    }
    /**
     * Grants permission to delete an existing configured table association analysis rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_DeleteConfiguredTableAssociationAnalysisRule.html
     */
    toDeleteConfiguredTableAssociationAnalysisRule() {
        return this.to('DeleteConfiguredTableAssociationAnalysisRule');
    }
    /**
     * Grants permission to remove an id mapping table from a collaboration
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - entityresolution:DeletePolicyStatement
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_DeleteIdMappingTable.html
     */
    toDeleteIdMappingTable() {
        return this.to('DeleteIdMappingTable');
    }
    /**
     * Grants permission to remove an Id Namespace Association from a collaboration
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - entityresolution:DeletePolicyStatement
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_DeleteIdNamespaceAssociation.html
     */
    toDeleteIdNamespaceAssociation() {
        return this.to('DeleteIdNamespaceAssociation');
    }
    /**
     * Grants permission to delete members from a collaboration
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - cleanrooms-ml:DeleteConfiguredAudienceModelPolicy
     * - cleanrooms-ml:GetConfiguredAudienceModelPolicy
     * - cleanrooms-ml:PutConfiguredAudienceModelPolicy
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_DeleteMember.html
     */
    toDeleteMember() {
        return this.to('DeleteMember');
    }
    /**
     * Grants permission to leave collaborations by deleting a membership
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_DeleteMembership.html
     */
    toDeleteMembership() {
        return this.to('DeleteMembership');
    }
    /**
     * Grants permission to delete an existing privacy budget template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_DeletePrivacyBudgetTemplate.html
     */
    toDeletePrivacyBudgetTemplate() {
        return this.to('DeletePrivacyBudgetTemplate');
    }
    /**
     * Grants permission to view details for an analysis template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_GetAnalysisTemplate.html
     */
    toGetAnalysisTemplate() {
        return this.to('GetAnalysisTemplate');
    }
    /**
     * Grants permission to view details for a collaboration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_GetCollaboration.html
     */
    toGetCollaboration() {
        return this.to('GetCollaboration');
    }
    /**
     * Grants permission to view details for an analysis template within a collaboration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_GetCollaborationAnalysisTemplate.html
     */
    toGetCollaborationAnalysisTemplate() {
        return this.to('GetCollaborationAnalysisTemplate');
    }
    /**
     * Grants permission to get a change request in a collaboration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_GetCollaborationChangeRequest.html
     */
    toGetCollaborationChangeRequest() {
        return this.to('GetCollaborationChangeRequest');
    }
    /**
     * Grants permission to view details for a configured audience model association within a collaboration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_GetCollaborationConfiguredAudienceModelAssociation.html
     */
    toGetCollaborationConfiguredAudienceModelAssociation() {
        return this.to('GetCollaborationConfiguredAudienceModelAssociation');
    }
    /**
     * Grants permission to get id namespace association within a collaboration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_GetCollaborationIdNamespaceAssociation.html
     */
    toGetCollaborationIdNamespaceAssociation() {
        return this.to('GetCollaborationIdNamespaceAssociation');
    }
    /**
     * Grants permission to view details for a privacy budget template within a collaboration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_GetCollaborationPrivacyBudgetTemplate.html
     */
    toGetCollaborationPrivacyBudgetTemplate() {
        return this.to('GetCollaborationPrivacyBudgetTemplate');
    }
    /**
     * Grants permission to view details for a configured audience model association
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_GetConfiguredAudienceModelAssociation.html
     */
    toGetConfiguredAudienceModelAssociation() {
        return this.to('GetConfiguredAudienceModelAssociation');
    }
    /**
     * Grants permission to view details for a configured table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_GetConfiguredTable.html
     */
    toGetConfiguredTable() {
        return this.to('GetConfiguredTable');
    }
    /**
     * Grants permission to view analysis rules for a configured table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_GetConfiguredTableAnalysisRule.html
     */
    toGetConfiguredTableAnalysisRule() {
        return this.to('GetConfiguredTableAnalysisRule');
    }
    /**
     * Grants permission to view details for a configured table association
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_GetConfiguredTableAssociation.html
     */
    toGetConfiguredTableAssociation() {
        return this.to('GetConfiguredTableAssociation');
    }
    /**
     * Grants permission to view analysis rules for a configured table association
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_GetConfiguredTableAssociationAnalysisRule.html
     */
    toGetConfiguredTableAssociationAnalysisRule() {
        return this.to('GetConfiguredTableAssociationAnalysisRule');
    }
    /**
     * Grants permission to view details of an id mapping table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_GetIdMappingTable.html
     */
    toGetIdMappingTable() {
        return this.to('GetIdMappingTable');
    }
    /**
     * Grants permission to view details of an id namespace association
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - entityresolution:GetIdNamespace
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_GetIdNamespaceAssociation.html
     */
    toGetIdNamespaceAssociation() {
        return this.to('GetIdNamespaceAssociation');
    }
    /**
     * Grants permission to view details about a membership
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_GetMembership.html
     */
    toGetMembership() {
        return this.to('GetMembership');
    }
    /**
     * Grants permission to view details for a privacy budget template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_GetPrivacyBudgetTemplate.html
     */
    toGetPrivacyBudgetTemplate() {
        return this.to('GetPrivacyBudgetTemplate');
    }
    /**
     * Grants permission to view a protected job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_GetProtectedJob.html
     */
    toGetProtectedJob() {
        return this.to('GetProtectedJob');
    }
    /**
     * Grants permission to view a protected query
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_GetProtectedQuery.html
     */
    toGetProtectedQuery() {
        return this.to('GetProtectedQuery');
    }
    /**
     * Grants permission to view details for a schema
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_GetSchema.html
     */
    toGetSchema() {
        return this.to('GetSchema');
    }
    /**
     * Grants permission to view analysis rules associated with a schema
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - cleanrooms:GetSchema
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_GetSchemaAnalysisRule.html
     */
    toGetSchemaAnalysisRule() {
        return this.to('GetSchemaAnalysisRule');
    }
    /**
     * Grants permission to list available analysis templates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_ListAnalysisTemplates.html
     */
    toListAnalysisTemplates() {
        return this.to('ListAnalysisTemplates');
    }
    /**
     * Grants permission to list available analysis templates within a collaboration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_ListCollaborationAnalysisTemplates.html
     */
    toListCollaborationAnalysisTemplates() {
        return this.to('ListCollaborationAnalysisTemplates');
    }
    /**
     * Grants permission to list change requests in a collaboration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_ListCollaborationChangeRequests.html
     */
    toListCollaborationChangeRequests() {
        return this.to('ListCollaborationChangeRequests');
    }
    /**
     * Grants permission to list available configured audience model association within a collaboration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_ListCollaborationConfiguredAudienceModelAssociations.html
     */
    toListCollaborationConfiguredAudienceModelAssociations() {
        return this.to('ListCollaborationConfiguredAudienceModelAssociations');
    }
    /**
     * Grants permission to list id namespace within a collaboration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_ListCollaborationIdNamespaceAssociations.html
     */
    toListCollaborationIdNamespaceAssociations() {
        return this.to('ListCollaborationIdNamespaceAssociations');
    }
    /**
     * Grants permission to list available privacy budget templates within a collaboration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_ListCollaborationPrivacyBudgetTemplates.html
     */
    toListCollaborationPrivacyBudgetTemplates() {
        return this.to('ListCollaborationPrivacyBudgetTemplates');
    }
    /**
     * Grants permission to list privacy budgets within a collaboration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_ListCollaborationPrivacyBudgets.html
     */
    toListCollaborationPrivacyBudgets() {
        return this.to('ListCollaborationPrivacyBudgets');
    }
    /**
     * Grants permission to list available collaborations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_ListCollaborations.html
     */
    toListCollaborations() {
        return this.to('ListCollaborations');
    }
    /**
     * Grants permission to list available configured audience model associations for a membership
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_ListConfiguredAudienceModelAssociations.html
     */
    toListConfiguredAudienceModelAssociations() {
        return this.to('ListConfiguredAudienceModelAssociations');
    }
    /**
     * Grants permission to list available configured table associations for a membership
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_ListConfiguredTableAssociations.html
     */
    toListConfiguredTableAssociations() {
        return this.to('ListConfiguredTableAssociations');
    }
    /**
     * Grants permission to list available configured tables
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_ListConfiguredTables.html
     */
    toListConfiguredTables() {
        return this.to('ListConfiguredTables');
    }
    /**
     * Grants permission to list available id mapping tables for a membership
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_ListIdMappingTables.html
     */
    toListIdMappingTables() {
        return this.to('ListIdMappingTables');
    }
    /**
     * Grants permission to list entity resolution data associations for a membership
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_ListIdNamespaceAssociations.html
     */
    toListIdNamespaceAssociations() {
        return this.to('ListIdNamespaceAssociations');
    }
    /**
     * Grants permission to list the members of a collaboration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_ListMembers.html
     */
    toListMembers() {
        return this.to('ListMembers');
    }
    /**
     * Grants permission to list available memberships
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_ListMemberships.html
     */
    toListMemberships() {
        return this.to('ListMemberships');
    }
    /**
     * Grants permission to list available privacy budget templates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_ListPrivacyBudgetTemplates.html
     */
    toListPrivacyBudgetTemplates() {
        return this.to('ListPrivacyBudgetTemplates');
    }
    /**
     * Grants permission to list available privacy budgets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_ListPrivacyBudgets.html
     */
    toListPrivacyBudgets() {
        return this.to('ListPrivacyBudgets');
    }
    /**
     * Grants permission to list protected jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_ListProtectedJobs.html
     */
    toListProtectedJobs() {
        return this.to('ListProtectedJobs');
    }
    /**
     * Grants permission to list protected queries
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_ListProtectedQueries.html
     */
    toListProtectedQueries() {
        return this.to('ListProtectedQueries');
    }
    /**
     * Grants permission to view available schemas for a collaboration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_ListSchemas.html
     */
    toListSchemas() {
        return this.to('ListSchemas');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to access a collaboration in the context of Clean Rooms ML custom models
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/userguide/ml-behaviors-byom.html#ml-behaviors-byom-membership-collaboration-access
     */
    toPassCollaboration() {
        return this.to('PassCollaboration');
    }
    /**
     * Grants permission to access a membership in the context of Clean Rooms ML custom models
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/userguide/ml-behaviors-byom.html#ml-behaviors-byom-membership-collaboration-access
     */
    toPassMembership() {
        return this.to('PassMembership');
    }
    /**
     * Grants permission to start an Id Mapping Job in AWS Entity Resolution to generate id mapping results in cleanrooms collaboration.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - entityresolution:GetIdMappingWorkflow
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_PopulateIdMappingTable.html
     */
    toPopulateIdMappingTable() {
        return this.to('PopulateIdMappingTable');
    }
    /**
     * Grants permission to preview privacy budget template settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_PreviewPrivacyImpact.html
     */
    toPreviewPrivacyImpact() {
        return this.to('PreviewPrivacyImpact');
    }
    /**
     * Grants permission to start protected jobs
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - cleanrooms:GetCollaborationAnalysisTemplate
     * - cleanrooms:GetSchema
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_StartProtectedJob.html
     */
    toStartProtectedJob() {
        return this.to('StartProtectedJob');
    }
    /**
     * Grants permission to start protected queries
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - cleanrooms:GetCollaborationAnalysisTemplate
     * - cleanrooms:GetSchema
     * - s3:GetBucketLocation
     * - s3:ListBucket
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_StartProtectedQuery.html
     */
    toStartProtectedQuery() {
        return this.to('StartProtectedQuery');
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update details of the analysis template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_UpdateAnalysisTemplate.html
     */
    toUpdateAnalysisTemplate() {
        return this.to('UpdateAnalysisTemplate');
    }
    /**
     * Grants permission to update details of the collaboration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_UpdateCollaboration.html
     */
    toUpdateCollaboration() {
        return this.to('UpdateCollaboration');
    }
    /**
     * Grants permission to update a configured audience model association
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_UpdateConfiguredAudienceModelAssociation.html
     */
    toUpdateConfiguredAudienceModelAssociation() {
        return this.to('UpdateConfiguredAudienceModelAssociation');
    }
    /**
     * Grants permission to update an existing configured table
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - athena:GetTableMetadata
     * - cleanrooms:UpdateConfiguredTableAllowedColumns
     * - cleanrooms:UpdateConfiguredTableReference
     * - glue:BatchGetPartition
     * - glue:GetDatabase
     * - glue:GetDatabases
     * - glue:GetPartition
     * - glue:GetPartitions
     * - glue:GetSchemaVersion
     * - glue:GetTable
     * - glue:GetTables
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_UpdateConfiguredTable.html
     */
    toUpdateConfiguredTable() {
        return this.to('UpdateConfiguredTable');
    }
    /**
     * Grants permission to update the allowed columns of an existing configured table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_UpdateConfiguredTable.html
     */
    toUpdateConfiguredTableAllowedColumns() {
        return this.to('UpdateConfiguredTableAllowedColumns');
    }
    /**
     * Grants permission to update analysis rules for a configured table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_UpdateConfiguredTableAnalysisRule.html
     */
    toUpdateConfiguredTableAnalysisRule() {
        return this.to('UpdateConfiguredTableAnalysisRule');
    }
    /**
     * Grants permission to update a configured table association
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_UpdateConfiguredTableAssociation.html
     */
    toUpdateConfiguredTableAssociation() {
        return this.to('UpdateConfiguredTableAssociation');
    }
    /**
     * Grants permission to update analysis rules for a configured table association
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_UpdateConfiguredTableAssociationAnalysisRule.html
     */
    toUpdateConfiguredTableAssociationAnalysisRule() {
        return this.to('UpdateConfiguredTableAssociationAnalysisRule');
    }
    /**
     * Grants permission to update the table reference of an existing configured table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_UpdateConfiguredTable.html
     */
    toUpdateConfiguredTableReference() {
        return this.to('UpdateConfiguredTableReference');
    }
    /**
     * Grants permission to update an id mapping table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_UpdateIdMappingTable.html
     */
    toUpdateIdMappingTable() {
        return this.to('UpdateIdMappingTable');
    }
    /**
     * Grants permission to update a entity resolution input association
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - entityresolution:GetIdNamespace
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_UpdateIdNamespaceAssociation.html
     */
    toUpdateIdNamespaceAssociation() {
        return this.to('UpdateIdNamespaceAssociation');
    }
    /**
     * Grants permission to update details of a membership
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     * - logs:CreateLogDelivery
     * - logs:CreateLogGroup
     * - logs:DeleteLogDelivery
     * - logs:DescribeLogGroups
     * - logs:DescribeResourcePolicies
     * - logs:GetLogDelivery
     * - logs:ListLogDeliveries
     * - logs:PutResourcePolicy
     * - logs:UpdateLogDelivery
     * - s3:GetBucketLocation
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_UpdateMembership.html
     */
    toUpdateMembership() {
        return this.to('UpdateMembership');
    }
    /**
     * Grants permission to update details of the privacy budget template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_UpdatePrivacyBudgetTemplate.html
     */
    toUpdatePrivacyBudgetTemplate() {
        return this.to('UpdatePrivacyBudgetTemplate');
    }
    /**
     * Grants permission to update protected jobs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_UpdateProtectedJob.html
     */
    toUpdateProtectedJob() {
        return this.to('UpdateProtectedJob');
    }
    /**
     * Grants permission to update protected queries
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/apireference/API_UpdateProtectedQuery.html
     */
    toUpdateProtectedQuery() {
        return this.to('UpdateProtectedQuery');
    }
    /**
     * Adds a resource of type analysistemplate to the statement
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/userguide/security-iam.html
     *
     * @param membershipId - Identifier for the membershipId.
     * @param analysisTemplateId - Identifier for the analysisTemplateId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAnalysistemplate(membershipId, analysisTemplateId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cleanrooms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:membership/${membershipId}/analysistemplate/${analysisTemplateId}`);
    }
    /**
     * Adds a resource of type collaboration to the statement
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/userguide/security-iam.html
     *
     * @param collaborationId - Identifier for the collaborationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCollaboration(collaborationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cleanrooms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:collaboration/${collaborationId}`);
    }
    /**
     * Adds a resource of type configuredaudiencemodelassociation to the statement
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/userguide/security-iam.html
     *
     * @param membershipId - Identifier for the membershipId.
     * @param configuredAudienceModelAssociationId - Identifier for the configuredAudienceModelAssociationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfiguredaudiencemodelassociation(membershipId, configuredAudienceModelAssociationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cleanrooms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:membership/${membershipId}/configuredaudiencemodelassociation/${configuredAudienceModelAssociationId}`);
    }
    /**
     * Adds a resource of type configuredtable to the statement
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/userguide/security-iam.html
     *
     * @param configuredTableId - Identifier for the configuredTableId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfiguredtable(configuredTableId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cleanrooms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:configuredtable/${configuredTableId}`);
    }
    /**
     * Adds a resource of type configuredtableassociation to the statement
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/userguide/security-iam.html
     *
     * @param membershipId - Identifier for the membershipId.
     * @param configuredTableAssociationId - Identifier for the configuredTableAssociationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfiguredtableassociation(membershipId, configuredTableAssociationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cleanrooms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:membership/${membershipId}/configuredtableassociation/${configuredTableAssociationId}`);
    }
    /**
     * Adds a resource of type idmappingtable to the statement
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/userguide/security-iam.html
     *
     * @param membershipId - Identifier for the membershipId.
     * @param idMappingTableId - Identifier for the idMappingTableId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIdmappingtable(membershipId, idMappingTableId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cleanrooms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:membership/${membershipId}/idmappingtable/${idMappingTableId}`);
    }
    /**
     * Adds a resource of type idnamespaceassociation to the statement
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/userguide/security-iam.html
     *
     * @param membershipId - Identifier for the membershipId.
     * @param idNamespaceAssociationId - Identifier for the idNamespaceAssociationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIdnamespaceassociation(membershipId, idNamespaceAssociationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cleanrooms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:membership/${membershipId}/idnamespaceassociation/${idNamespaceAssociationId}`);
    }
    /**
     * Adds a resource of type membership to the statement
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/userguide/security-iam.html
     *
     * @param membershipId - Identifier for the membershipId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMembership(membershipId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cleanrooms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:membership/${membershipId}`);
    }
    /**
     * Adds a resource of type privacybudgettemplate to the statement
     *
     * https://docs.aws.amazon.com/clean-rooms/latest/userguide/security-iam.html
     *
     * @param membershipId - Identifier for the membershipId.
     * @param privacyBudgetTemplateId - Identifier for the privacyBudgetTemplateId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPrivacybudgettemplate(membershipId, privacyBudgetTemplateId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cleanrooms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:membership/${membershipId}/privacybudgettemplate/${privacyBudgetTemplateId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAnalysisTemplate()
     * - .toCreateCollaboration()
     * - .toCreateCollaborationChangeRequest()
     * - .toCreateConfiguredAudienceModelAssociation()
     * - .toCreateConfiguredTable()
     * - .toCreateConfiguredTableAssociation()
     * - .toCreateConfiguredTableAssociationAnalysisRule()
     * - .toCreateIdMappingTable()
     * - .toCreateIdNamespaceAssociation()
     * - .toCreateMembership()
     * - .toCreatePrivacyBudgetTemplate()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCreateAnalysisTemplate()
     * - .toCreateCollaboration()
     * - .toCreateCollaborationChangeRequest()
     * - .toCreateConfiguredAudienceModelAssociation()
     * - .toCreateConfiguredTable()
     * - .toCreateConfiguredTableAssociation()
     * - .toCreateConfiguredTableAssociationAnalysisRule()
     * - .toCreateIdMappingTable()
     * - .toCreateIdNamespaceAssociation()
     * - .toCreateMembership()
     * - .toCreatePrivacyBudgetTemplate()
     *
     * Applies to resource types:
     * - analysistemplate
     * - collaboration
     * - configuredaudiencemodelassociation
     * - configuredtable
     * - configuredtableassociation
     * - idmappingtable
     * - idnamespaceassociation
     * - membership
     * - privacybudgettemplate
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAnalysisTemplate()
     * - .toCreateCollaboration()
     * - .toCreateCollaborationChangeRequest()
     * - .toCreateConfiguredAudienceModelAssociation()
     * - .toCreateConfiguredTable()
     * - .toCreateConfiguredTableAssociation()
     * - .toCreateConfiguredTableAssociationAnalysisRule()
     * - .toCreateIdMappingTable()
     * - .toCreateIdNamespaceAssociation()
     * - .toCreateMembership()
     * - .toCreatePrivacyBudgetTemplate()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [cleanrooms](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscleanrooms.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'cleanrooms';
        this.accessLevelList = {
            Read: [
                'BatchGetCollaborationAnalysisTemplate',
                'BatchGetSchema',
                'BatchGetSchemaAnalysisRule',
                'GetAnalysisTemplate',
                'GetCollaboration',
                'GetCollaborationAnalysisTemplate',
                'GetCollaborationChangeRequest',
                'GetCollaborationConfiguredAudienceModelAssociation',
                'GetCollaborationIdNamespaceAssociation',
                'GetCollaborationPrivacyBudgetTemplate',
                'GetConfiguredAudienceModelAssociation',
                'GetConfiguredTable',
                'GetConfiguredTableAnalysisRule',
                'GetConfiguredTableAssociation',
                'GetConfiguredTableAssociationAnalysisRule',
                'GetIdMappingTable',
                'GetIdNamespaceAssociation',
                'GetMembership',
                'GetPrivacyBudgetTemplate',
                'GetProtectedJob',
                'GetProtectedQuery',
                'GetSchema',
                'GetSchemaAnalysisRule',
                'PassCollaboration',
                'PassMembership',
                'PreviewPrivacyImpact'
            ],
            Write: [
                'CreateAnalysisTemplate',
                'CreateCollaboration',
                'CreateCollaborationChangeRequest',
                'CreateConfiguredAudienceModelAssociation',
                'CreateConfiguredTable',
                'CreateConfiguredTableAnalysisRule',
                'CreateConfiguredTableAssociation',
                'CreateConfiguredTableAssociationAnalysisRule',
                'CreateIdMappingTable',
                'CreateIdNamespaceAssociation',
                'CreateMembership',
                'CreatePrivacyBudgetTemplate',
                'DeleteAnalysisTemplate',
                'DeleteCollaboration',
                'DeleteConfiguredAudienceModelAssociation',
                'DeleteConfiguredTable',
                'DeleteConfiguredTableAnalysisRule',
                'DeleteConfiguredTableAssociation',
                'DeleteConfiguredTableAssociationAnalysisRule',
                'DeleteIdMappingTable',
                'DeleteIdNamespaceAssociation',
                'DeleteMember',
                'DeleteMembership',
                'DeletePrivacyBudgetTemplate',
                'PopulateIdMappingTable',
                'StartProtectedJob',
                'StartProtectedQuery',
                'UpdateAnalysisTemplate',
                'UpdateCollaboration',
                'UpdateConfiguredAudienceModelAssociation',
                'UpdateConfiguredTable',
                'UpdateConfiguredTableAllowedColumns',
                'UpdateConfiguredTableAnalysisRule',
                'UpdateConfiguredTableAssociation',
                'UpdateConfiguredTableAssociationAnalysisRule',
                'UpdateConfiguredTableReference',
                'UpdateIdMappingTable',
                'UpdateIdNamespaceAssociation',
                'UpdateMembership',
                'UpdatePrivacyBudgetTemplate',
                'UpdateProtectedJob',
                'UpdateProtectedQuery'
            ],
            List: [
                'ListAnalysisTemplates',
                'ListCollaborationAnalysisTemplates',
                'ListCollaborationChangeRequests',
                'ListCollaborationConfiguredAudienceModelAssociations',
                'ListCollaborationIdNamespaceAssociations',
                'ListCollaborationPrivacyBudgetTemplates',
                'ListCollaborationPrivacyBudgets',
                'ListCollaborations',
                'ListConfiguredAudienceModelAssociations',
                'ListConfiguredTableAssociations',
                'ListConfiguredTables',
                'ListIdMappingTables',
                'ListIdNamespaceAssociations',
                'ListMembers',
                'ListMemberships',
                'ListPrivacyBudgetTemplates',
                'ListPrivacyBudgets',
                'ListProtectedJobs',
                'ListProtectedQueries',
                'ListSchemas',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Cleanrooms = Cleanrooms;
//# sourceMappingURL=data:application/json;base64,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