"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cloud9 = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [cloud9](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscloud9.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Cloud9 extends shared_1.PolicyStatement {
    /**
     * Grants permission to start the Amazon EC2 instance that your AWS Cloud9 IDE connects to
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toActivateEC2Remote() {
        return this.to('ActivateEC2Remote');
    }
    /**
     * Grants permission to create an AWS Cloud9 development environment, launches an Amazon Elastic Compute Cloud (Amazon EC2) instance, and then hosts the environment on the instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEnvironmentName()
     * - .ifInstanceType()
     * - .ifSubnetId()
     * - .ifUserArn()
     * - .ifOwnerArn()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_CreateEnvironmentEC2.html
     */
    toCreateEnvironmentEC2() {
        return this.to('CreateEnvironmentEC2');
    }
    /**
     * Grants permission to add an environment member to an AWS Cloud9 development environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserArn()
     * - .ifEnvironmentId()
     * - .ifPermissions()
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_CreateEnvironmentMembership.html
     */
    toCreateEnvironmentMembership() {
        return this.to('CreateEnvironmentMembership');
    }
    /**
     * Grants permission to create an AWS Cloud9 SSH development environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEnvironmentName()
     * - .ifOwnerArn()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toCreateEnvironmentSSH() {
        return this.to('CreateEnvironmentSSH');
    }
    /**
     * Grants permission to create an authentication token that allows a connection between the AWS Cloud9 IDE and the user's environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toCreateEnvironmentToken() {
        return this.to('CreateEnvironmentToken');
    }
    /**
     * Grants permission to delete an AWS Cloud9 development environment. If the environment is hosted on an Amazon Elastic Compute Cloud (Amazon EC2) instance, also terminates the instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DeleteEnvironment.html
     */
    toDeleteEnvironment() {
        return this.to('DeleteEnvironment');
    }
    /**
     * Grants permission to delete an environment member from an AWS Cloud9 development environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserArn()
     * - .ifEnvironmentId()
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DeleteEnvironmentMembership.html
     */
    toDeleteEnvironmentMembership() {
        return this.to('DeleteEnvironmentMembership');
    }
    /**
     * Grants permission to get details about the connection to the EC2 development environment, including host, user, and port
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toDescribeEC2Remote() {
        return this.to('DescribeEC2Remote');
    }
    /**
     * Grants permission to get information about environment members for an AWS Cloud9 development environment
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserArn()
     * - .ifEnvironmentId()
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DescribeEnvironmentMemberships.html
     */
    toDescribeEnvironmentMemberships() {
        return this.to('DescribeEnvironmentMemberships');
    }
    /**
     * Grants permission to get status information for an AWS Cloud9 development environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DescribeEnvironmentStatus.html
     */
    toDescribeEnvironmentStatus() {
        return this.to('DescribeEnvironmentStatus');
    }
    /**
     * Grants permission to get information about AWS Cloud9 development environments
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DescribeEnvironments.html
     */
    toDescribeEnvironments() {
        return this.to('DescribeEnvironments');
    }
    /**
     * Grants permission to get details about the connection to the SSH development environment, including host, user, and port
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toDescribeSSHRemote() {
        return this.to('DescribeSSHRemote');
    }
    /**
     * Grants permission to get configuration information that's used to initialize the AWS Cloud9 IDE
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toGetEnvironmentConfig() {
        return this.to('GetEnvironmentConfig');
    }
    /**
     * Grants permission to get the AWS Cloud9 IDE settings for a specified development environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toGetEnvironmentSettings() {
        return this.to('GetEnvironmentSettings');
    }
    /**
     * Grants permission to get the AWS Cloud9 IDE settings for a specified environment member
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toGetMembershipSettings() {
        return this.to('GetMembershipSettings');
    }
    /**
     * Grants permission to get the migration experience for a cloud9 user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toGetMigrationExperiences() {
        return this.to('GetMigrationExperiences');
    }
    /**
     * Grants permission to get the user's public SSH key, which is used by AWS Cloud9 to connect to SSH development environments
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserArn()
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toGetUserPublicKey() {
        return this.to('GetUserPublicKey');
    }
    /**
     * Grants permission to get the AWS Cloud9 IDE settings for a specified user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toGetUserSettings() {
        return this.to('GetUserSettings');
    }
    /**
     * Grants permission to get a list of AWS Cloud9 development environment identifiers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_ListEnvironments.html
     */
    toListEnvironments() {
        return this.to('ListEnvironments');
    }
    /**
     * Grants permission to list tags for a cloud9 environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to set AWS managed temporary credentials on the Amazon EC2 instance that's used by the AWS Cloud9 integrated development environment (IDE)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toModifyTemporaryCredentialsOnEnvironmentEC2() {
        return this.to('ModifyTemporaryCredentialsOnEnvironmentEC2');
    }
    /**
     * Grants permission to add tags to a cloud9 environment
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a cloud9 environment
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to change the settings of an existing AWS Cloud9 development environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_UpdateEnvironment.html
     */
    toUpdateEnvironment() {
        return this.to('UpdateEnvironment');
    }
    /**
     * Grants permission to change the settings of an existing environment member for an AWS Cloud9 development environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserArn()
     * - .ifEnvironmentId()
     * - .ifPermissions()
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_UpdateEnvironmentMembership.html
     */
    toUpdateEnvironmentMembership() {
        return this.to('UpdateEnvironmentMembership');
    }
    /**
     * Grants permission to update the AWS Cloud9 IDE settings for a specified development environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toUpdateEnvironmentSettings() {
        return this.to('UpdateEnvironmentSettings');
    }
    /**
     * Grants permission to update the AWS Cloud9 IDE settings for a specified environment member
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toUpdateMembershipSettings() {
        return this.to('UpdateMembershipSettings');
    }
    /**
     * Grants permission to update details about the connection to the SSH development environment, including host, user, and port
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toUpdateSSHRemote() {
        return this.to('UpdateSSHRemote');
    }
    /**
     * Grants permission to update IDE-specific settings of an AWS Cloud9 user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toUpdateUserSettings() {
        return this.to('UpdateUserSettings');
    }
    /**
     * Adds a resource of type environment to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html##awscloud9-environment
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEnvironment(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloud9:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:environment:${resourceId}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateEnvironmentEC2()
     * - .toCreateEnvironmentSSH()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - environment
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateEnvironmentEC2()
     * - .toCreateEnvironmentSSH()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the AWS Cloud9 environment ID
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html##awscloud9-cloud9_EnvironmentId
     *
     * Applies to actions:
     * - .toCreateEnvironmentMembership()
     * - .toDeleteEnvironmentMembership()
     * - .toDescribeEnvironmentMemberships()
     * - .toUpdateEnvironmentMembership()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentId(value, operator) {
        return this.if(`EnvironmentId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the AWS Cloud9 environment name
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html##awscloud9-cloud9_EnvironmentName
     *
     * Applies to actions:
     * - .toCreateEnvironmentEC2()
     * - .toCreateEnvironmentSSH()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentName(value, operator) {
        return this.if(`EnvironmentName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the instance type of the AWS Cloud9 environment's Amazon EC2 instance
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html##awscloud9-cloud9_InstanceType
     *
     * Applies to actions:
     * - .toCreateEnvironmentEC2()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstanceType(value, operator) {
        return this.if(`InstanceType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the owner ARN specified
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html##awscloud9-cloud9_OwnerArn
     *
     * Applies to actions:
     * - .toCreateEnvironmentEC2()
     * - .toCreateEnvironmentSSH()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifOwnerArn(value, operator) {
        return this.if(`OwnerArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the type of AWS Cloud9 permissions
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html##awscloud9-cloud9_Permissions
     *
     * Applies to actions:
     * - .toCreateEnvironmentMembership()
     * - .toUpdateEnvironmentMembership()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPermissions(value, operator) {
        return this.if(`Permissions`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the subnet ID that the AWS Cloud9 environment will be created in
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html##awscloud9-cloud9_SubnetId
     *
     * Applies to actions:
     * - .toCreateEnvironmentEC2()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSubnetId(value, operator) {
        return this.if(`SubnetId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the user ARN specified
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html##awscloud9-cloud9_UserArn
     *
     * Applies to actions:
     * - .toCreateEnvironmentEC2()
     * - .toCreateEnvironmentMembership()
     * - .toDeleteEnvironmentMembership()
     * - .toDescribeEnvironmentMemberships()
     * - .toGetUserPublicKey()
     * - .toUpdateEnvironmentMembership()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifUserArn(value, operator) {
        return this.if(`UserArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Statement provider for service [cloud9](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscloud9.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'cloud9';
        this.accessLevelList = {
            Write: [
                'ActivateEC2Remote',
                'CreateEnvironmentEC2',
                'CreateEnvironmentMembership',
                'CreateEnvironmentSSH',
                'DeleteEnvironment',
                'DeleteEnvironmentMembership',
                'ModifyTemporaryCredentialsOnEnvironmentEC2',
                'UpdateEnvironment',
                'UpdateEnvironmentMembership',
                'UpdateEnvironmentSettings',
                'UpdateMembershipSettings',
                'UpdateSSHRemote',
                'UpdateUserSettings'
            ],
            Read: [
                'CreateEnvironmentToken',
                'DescribeEC2Remote',
                'DescribeEnvironmentMemberships',
                'DescribeEnvironmentStatus',
                'DescribeEnvironments',
                'DescribeSSHRemote',
                'GetEnvironmentConfig',
                'GetEnvironmentSettings',
                'GetMembershipSettings',
                'GetMigrationExperiences',
                'GetUserPublicKey',
                'GetUserSettings',
                'ListEnvironments',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Cloud9 = Cloud9;
//# sourceMappingURL=data:application/json;base64,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