"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cloudformation = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [cloudformation](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscloudformation.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Cloudformation extends shared_1.PolicyStatement {
    /**
     * Grants permission to activate trusted access between StackSets and Organizations. With trusted access between StackSets and Organizations activated, the management account has permissions to create and manage StackSets for your organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ActivateOrganizationsAccess.html
     */
    toActivateOrganizationsAccess() {
        return this.to('ActivateOrganizationsAccess');
    }
    /**
     * Grants permission to activate a public third-party extension, making it available for use in stack templates
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ActivateType.html
     */
    toActivateType() {
        return this.to('ActivateType');
    }
    /**
     * Grants permission to return configuration data for the specified CloudFormation extensions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_BatchDescribeTypeConfigurations.html
     */
    toBatchDescribeTypeConfigurations() {
        return this.to('BatchDescribeTypeConfigurations');
    }
    /**
     * Grants permission to cancel an update on the specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CancelUpdateStack.html
     */
    toCancelUpdateStack() {
        return this.to('CancelUpdateStack');
    }
    /**
     * Grants permission to continue rolling back a stack that is in the UPDATE_ROLLBACK_FAILED state to the UPDATE_ROLLBACK_COMPLETE state
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifRoleArn()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ContinueUpdateRollback.html
     */
    toContinueUpdateRollback() {
        return this.to('ContinueUpdateRollback');
    }
    /**
     * Grants permission to create a list of changes for a stack
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifChangeSetName()
     * - .ifResourceTypes()
     * - .ifImportResourceTypes()
     * - .ifRoleArn()
     * - .ifStackPolicyUrl()
     * - .ifTemplateUrl()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CreateChangeSet.html
     */
    toCreateChangeSet() {
        return this.to('CreateChangeSet');
    }
    /**
     * Grants permission to create a template from existing resources that are not already managed with CloudFormation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CreateGeneratedTemplate.html
     */
    toCreateGeneratedTemplate() {
        return this.to('CreateGeneratedTemplate');
    }
    /**
     * Grants permission to create a stack as specified in the template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceTypes()
     * - .ifRoleArn()
     * - .ifStackPolicyUrl()
     * - .ifTemplateUrl()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CreateStack.html
     */
    toCreateStack() {
        return this.to('CreateStack');
    }
    /**
     * Grants permission to create stack instances for the specified accounts, within the specified regions
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifTargetRegion()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CreateStackInstances.html
     */
    toCreateStackInstances() {
        return this.to('CreateStackInstances');
    }
    /**
     * Grants permission to create a stack refactor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CreateStackRefactor.html
     */
    toCreateStackRefactor() {
        return this.to('CreateStackRefactor');
    }
    /**
     * Grants permission to create a stackset as specified in the template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifRoleArn()
     * - .ifTemplateUrl()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CreateStackSet.html
     */
    toCreateStackSet() {
        return this.to('CreateStackSet');
    }
    /**
     * Grants permission to upload templates to Amazon S3 buckets. Used only by the AWS CloudFormation console and is not documented in the API reference
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html
     */
    toCreateUploadBucket() {
        return this.to('CreateUploadBucket');
    }
    /**
     * Grants permission to deactivate trusted access between StackSets and Organizations. If trusted access is deactivated, the management account does not have permissions to create and manage service-managed StackSets for your organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeactivateOrganizationsAccess.html
     */
    toDeactivateOrganizationsAccess() {
        return this.to('DeactivateOrganizationsAccess');
    }
    /**
     * Grants permission to deactivate a public extension that was previously activated in this account and region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeactivateType.html
     */
    toDeactivateType() {
        return this.to('DeactivateType');
    }
    /**
     * Grants permission to delete the specified change set. Deleting change sets ensures that no one executes the wrong change set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifChangeSetName()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeleteChangeSet.html
     */
    toDeleteChangeSet() {
        return this.to('DeleteChangeSet');
    }
    /**
     * Grants permission to delete a generated template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeleteGeneratedTemplate.html
     */
    toDeleteGeneratedTemplate() {
        return this.to('DeleteGeneratedTemplate');
    }
    /**
     * Grants permission to delete a specified stack
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifRoleArn()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeleteStack.html
     */
    toDeleteStack() {
        return this.to('DeleteStack');
    }
    /**
     * Grants permission to delete stack instances for the specified accounts, in the specified regions
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTargetRegion()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeleteStackInstances.html
     */
    toDeleteStackInstances() {
        return this.to('DeleteStackInstances');
    }
    /**
     * Grants permission to delete a specified stackset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeleteStackSet.html
     */
    toDeleteStackSet() {
        return this.to('DeleteStackSet');
    }
    /**
     * Grants permission to deregister an existing CloudFormation type or type version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeregisterType.html
     */
    toDeregisterType() {
        return this.to('DeregisterType');
    }
    /**
     * Grants permission to retrieve your account's AWS CloudFormation limits
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeAccountLimits.html
     */
    toDescribeAccountLimits() {
        return this.to('DescribeAccountLimits');
    }
    /**
     * Grants permission to return the description for the specified change set
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifChangeSetName()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeChangeSet.html
     */
    toDescribeChangeSet() {
        return this.to('DescribeChangeSet');
    }
    /**
     * Grants permission to return the Hook invocation information for the specified change set
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifChangeSetName()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeChangeSetHooks.html
     */
    toDescribeChangeSetHooks() {
        return this.to('DescribeChangeSetHooks');
    }
    /**
     * Grants permission to describe a generated template. The output includes details about the progress of the creation of a generated template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeGeneratedTemplate.html
     */
    toDescribeGeneratedTemplate() {
        return this.to('DescribeGeneratedTemplate');
    }
    /**
     * Grants permission to return information about the account's OrganizationAccess status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeOrganizationsAccess.html
     */
    toDescribeOrganizationsAccess() {
        return this.to('DescribeOrganizationsAccess');
    }
    /**
     * Grants permission to return information about a CloudFormation extension publisher
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribePublisher.html
     */
    toDescribePublisher() {
        return this.to('DescribePublisher');
    }
    /**
     * Grants permission to describe details of a resource scan
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeResourceScan.html
     */
    toDescribeResourceScan() {
        return this.to('DescribeResourceScan');
    }
    /**
     * Grants permission to return information about a stack drift detection operation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackDriftDetectionStatus.html
     */
    toDescribeStackDriftDetectionStatus() {
        return this.to('DescribeStackDriftDetectionStatus');
    }
    /**
     * Grants permission to return all stack related events for a specified stack
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackEvents.html
     */
    toDescribeStackEvents() {
        return this.to('DescribeStackEvents');
    }
    /**
     * Grants permission to return the stack instance that's associated with the specified stack set, AWS account, and region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackInstance.html
     */
    toDescribeStackInstance() {
        return this.to('DescribeStackInstance');
    }
    /**
     * Grants permission to return the description for the specified stack refactor
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackRefactor.html
     */
    toDescribeStackRefactor() {
        return this.to('DescribeStackRefactor');
    }
    /**
     * Grants permission to return a description of the specified resource in the specified stack
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackResource.html
     */
    toDescribeStackResource() {
        return this.to('DescribeStackResource');
    }
    /**
     * Grants permission to return drift information for the resources that have been checked for drift in the specified stack
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackResourceDrifts.html
     */
    toDescribeStackResourceDrifts() {
        return this.to('DescribeStackResourceDrifts');
    }
    /**
     * Grants permission to return AWS resource descriptions for running and deleted stacks
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackResources.html
     */
    toDescribeStackResources() {
        return this.to('DescribeStackResources');
    }
    /**
     * Grants permission to return the description of the specified stack set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackSet.html
     */
    toDescribeStackSet() {
        return this.to('DescribeStackSet');
    }
    /**
     * Grants permission to return the description of the specified stack set operation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackSetOperation.html
     */
    toDescribeStackSetOperation() {
        return this.to('DescribeStackSetOperation');
    }
    /**
     * Grants permission to return the description for the specified stack, and to all stacks when used in combination with the ListStacks action
     *
     * Access Level: List
     *
     * Dependent actions:
     * - cloudformation:ListStacks
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStacks.html
     */
    toDescribeStacks() {
        return this.to('DescribeStacks');
    }
    /**
     * Grants permission to return information about the CloudFormation type requested
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeType.html
     */
    toDescribeType() {
        return this.to('DescribeType');
    }
    /**
     * Grants permission to return information about the registration process for a CloudFormation type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeTypeRegistration.html
     */
    toDescribeTypeRegistration() {
        return this.to('DescribeTypeRegistration');
    }
    /**
     * Grants permission to detects whether a stack's actual configuration differs, or has drifted, from it's expected configuration, as defined in the stack template and any values specified as template parameters
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DetectStackDrift.html
     */
    toDetectStackDrift() {
        return this.to('DetectStackDrift');
    }
    /**
     * Grants permission to return information about whether a resource's actual configuration differs, or has drifted, from it's expected configuration, as defined in the stack template and any values specified as template parameters
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DetectStackResourceDrift.html
     */
    toDetectStackResourceDrift() {
        return this.to('DetectStackResourceDrift');
    }
    /**
     * Grants permission to enable users to detect drift on a stack set and the stack instances that belong to that stack set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DetectStackSetDrift.html
     */
    toDetectStackSetDrift() {
        return this.to('DetectStackSetDrift');
    }
    /**
     * Grants permission to return the estimated monthly cost of a template
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifTemplateUrl()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_EstimateTemplateCost.html
     */
    toEstimateTemplateCost() {
        return this.to('EstimateTemplateCost');
    }
    /**
     * Grants permission to update a stack using the input information that was provided when the specified change set was created
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifChangeSetName()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ExecuteChangeSet.html
     */
    toExecuteChangeSet() {
        return this.to('ExecuteChangeSet');
    }
    /**
     * Grants permission to execute a stack refactor using the input information that was provided when the specified stack refactor was created
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ExecuteStackRefactor.html
     */
    toExecuteStackRefactor() {
        return this.to('ExecuteStackRefactor');
    }
    /**
     * Grants permission to retrieve a generated template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_GetGeneratedTemplate.html
     */
    toGetGeneratedTemplate() {
        return this.to('GetGeneratedTemplate');
    }
    /**
     * Grants permission to return the stack policy for a specified stack
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_GetStackPolicy.html
     */
    toGetStackPolicy() {
        return this.to('GetStackPolicy');
    }
    /**
     * Grants permission to return the template body for a specified stack
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_GetTemplate.html
     */
    toGetTemplate() {
        return this.to('GetTemplate');
    }
    /**
     * Grants permission to return information about a new or existing template
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifTemplateUrl()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_GetTemplateSummary.html
     */
    toGetTemplateSummary() {
        return this.to('GetTemplateSummary');
    }
    /**
     * Grants permission to enable users to import existing stacks to a new or existing stackset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ImportStacksToStackSet.html
     */
    toImportStacksToStackSet() {
        return this.to('ImportStacksToStackSet');
    }
    /**
     * Grants permission to return Hook invocations result information for a specified Hook, a combination of Hook and status, or all Hooks
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifTypeArn()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListHookResults.html
     */
    toListAllHookResults() {
        return this.to('ListAllHookResults');
    }
    /**
     * Grants permission to return the ID and status of each active change set for a stack. For example, AWS CloudFormation lists change sets that are in the CREATE_IN_PROGRESS or CREATE_PENDING state
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListChangeSets.html
     */
    toListChangeSets() {
        return this.to('ListChangeSets');
    }
    /**
     * Grants permission to list all exported output values in the account and region in which you call this action
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListExports.html
     */
    toListExports() {
        return this.to('ListExports');
    }
    /**
     * Grants permission to list your generated templates in this Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListGeneratedTemplates.html
     */
    toListGeneratedTemplates() {
        return this.to('ListGeneratedTemplates');
    }
    /**
     * Grants permission to return Hook invocations result information for the specified target
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifChangeSetName()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListHookResults.html
     */
    toListHookResults() {
        return this.to('ListHookResults');
    }
    /**
     * Grants permission to list all stacks that are importing an exported output value
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListImports.html
     */
    toListImports() {
        return this.to('ListImports');
    }
    /**
     * Grants permission to list the related resources for a list of resources from a resource scan. The response indicates whether each returned resource is already managed by CloudFormation
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListResourceScanRelatedResources.html
     */
    toListResourceScanRelatedResources() {
        return this.to('ListResourceScanRelatedResources');
    }
    /**
     * Grants permission to list the resources from a resource scan. The results can be filtered by resource identifier, resource type prefix, tag key, and tag value
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListResourceScanResources.html
     */
    toListResourceScanResources() {
        return this.to('ListResourceScanResources');
    }
    /**
     * Grants permission to list the resource scans from newest to oldest. By default it will return up to 10 resource scans
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListResourceScans.html
     */
    toListResourceScans() {
        return this.to('ListResourceScans');
    }
    /**
     * Grants permission to return drift information for the resources that have been checked for drift in the specified stack instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackInstanceResourceDrifts.html
     */
    toListStackInstanceResourceDrifts() {
        return this.to('ListStackInstanceResourceDrifts');
    }
    /**
     * Grants permission to return summary information about stack instances that are associated with the specified stack set
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackSets.html
     */
    toListStackInstances() {
        return this.to('ListStackInstances');
    }
    /**
     * Grants permission to return the list of actions of the specified stack refactor
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackRefactorActions.html
     */
    toListStackRefactorActions() {
        return this.to('ListStackRefactorActions');
    }
    /**
     * Grants permission to return the ID and status of each active stack refactor
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackRefactors.html
     */
    toListStackRefactors() {
        return this.to('ListStackRefactors');
    }
    /**
     * Grants permission to return descriptions of all resources of the specified stack
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackResources.html
     */
    toListStackResources() {
        return this.to('ListStackResources');
    }
    /**
     * Grants permission to return summary information about StackSet Auto Deployment Targets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackSetAutoDeploymentTargets.html
     */
    toListStackSetAutoDeploymentTargets() {
        return this.to('ListStackSetAutoDeploymentTargets');
    }
    /**
     * Grants permission to return summary information about the results of a stack set operation
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackSetOperationResults.html
     */
    toListStackSetOperationResults() {
        return this.to('ListStackSetOperationResults');
    }
    /**
     * Grants permission to return summary information about operations performed on a stack set
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackSetOperations.html
     */
    toListStackSetOperations() {
        return this.to('ListStackSetOperations');
    }
    /**
     * Grants permission to return summary information about stack sets that are associated with the user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackSets.html
     */
    toListStackSets() {
        return this.to('ListStackSets');
    }
    /**
     * Grants permission to return the summary information for stacks whose status matches the specified StackStatusFilter. In combination with the DescribeStacks action, grants permission to list descriptions for stacks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStacks.html
     */
    toListStacks() {
        return this.to('ListStacks');
    }
    /**
     * Grants permission to list CloudFormation type registration attempts
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListTypeRegistrations.html
     */
    toListTypeRegistrations() {
        return this.to('ListTypeRegistrations');
    }
    /**
     * Grants permission to list versions of a particular CloudFormation type
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListTypeVersions.html
     */
    toListTypeVersions() {
        return this.to('ListTypeVersions');
    }
    /**
     * Grants permission to list available CloudFormation types
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListTypes.html
     */
    toListTypes() {
        return this.to('ListTypes');
    }
    /**
     * Grants permission to publish the specified extension to the CloudFormation registry as a public extension in this region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_PublishType.html
     */
    toPublishType() {
        return this.to('PublishType');
    }
    /**
     * Grants permission to record the handler progress
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_RecordHandlerProgress.html
     */
    toRecordHandlerProgress() {
        return this.to('RecordHandlerProgress');
    }
    /**
     * Grants permission to register account as a publisher of public extensions in the CloudFormation registry
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_RegisterPublisher.html
     */
    toRegisterPublisher() {
        return this.to('RegisterPublisher');
    }
    /**
     * Grants permission to register a new CloudFormation type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_RegisterType.html
     */
    toRegisterType() {
        return this.to('RegisterType');
    }
    /**
     * Grants permission to rollback the stack to the last stable state
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifRoleArn()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_RollbackStack.html
     */
    toRollbackStack() {
        return this.to('RollbackStack');
    }
    /**
     * Grants permission to set a stack policy for a specified stack
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifStackPolicyUrl()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_SetStackPolicy.html
     */
    toSetStackPolicy() {
        return this.to('SetStackPolicy');
    }
    /**
     * Grants permission to set the configuration data for a registered CloudFormation extension, in the given account and region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_SetTypeConfiguration.html
     */
    toSetTypeConfiguration() {
        return this.to('SetTypeConfiguration');
    }
    /**
     * Grants permission to set which version of a CloudFormation type applies to CloudFormation operations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_SetTypeDefaultVersion.html
     */
    toSetTypeDefaultVersion() {
        return this.to('SetTypeDefaultVersion');
    }
    /**
     * Grants permission to send a signal to the specified resource with a success or failure status
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_SignalResource.html
     */
    toSignalResource() {
        return this.to('SignalResource');
    }
    /**
     * Grants permission to start a scan of the resources in this account in this Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_StartResourceScan.html
     */
    toStartResourceScan() {
        return this.to('StartResourceScan');
    }
    /**
     * Grants permission to stop an in-progress operation on a stack set and its associated stack instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_StopStackSetOperation.html
     */
    toStopStackSetOperation() {
        return this.to('StopStackSetOperation');
    }
    /**
     * Grants permission to tag cloudformation resources
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifCreateAction()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to test a registered extension to make sure it meets all necessary requirements for being published in the CloudFormation registry
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_TestType.html
     */
    toTestType() {
        return this.to('TestType');
    }
    /**
     * Grants permission to untag cloudformation resources
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifCreateAction()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a generated template. This can be used to change the name, add and remove resources, refresh resources, and change the DeletionPolicy and UpdateReplacePolicy settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateGeneratedTemplate.html
     */
    toUpdateGeneratedTemplate() {
        return this.to('UpdateGeneratedTemplate');
    }
    /**
     * Grants permission to update a stack as specified in the template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceTypes()
     * - .ifRoleArn()
     * - .ifStackPolicyUrl()
     * - .ifTemplateUrl()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateStack.html
     */
    toUpdateStack() {
        return this.to('UpdateStack');
    }
    /**
     * Grants permission to update the parameter values for stack instances for the specified accounts, within the specified regions
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTargetRegion()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateStackInstances.html
     */
    toUpdateStackInstances() {
        return this.to('UpdateStackInstances');
    }
    /**
     * Grants permission to update a stackset as specified in the template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifRoleArn()
     * - .ifTemplateUrl()
     * - .ifTargetRegion()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateStackSet.html
     */
    toUpdateStackSet() {
        return this.to('UpdateStackSet');
    }
    /**
     * Grants permission to update termination protection for the specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateTerminationProtection.html
     */
    toUpdateTerminationProtection() {
        return this.to('UpdateTerminationProtection');
    }
    /**
     * Grants permission to validate a specified template
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifTemplateUrl()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ValidateTemplate.html
     */
    toValidateTemplate() {
        return this.to('ValidateTemplate');
    }
    /**
     * Adds a resource of type changeset to the statement
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-whatis-concepts.html#w2ab1b5c15c11
     *
     * @param changeSetName - Identifier for the changeSetName.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onChangeset(changeSetName, id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudformation:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:changeSet/${changeSetName}/${id}`);
    }
    /**
     * Adds a resource of type stack to the statement
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-whatis-concepts.html#w2ab1b5c15b9
     *
     * @param stackName - Identifier for the stackName.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStack(stackName, id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudformation:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:stack/${stackName}/${id}`);
    }
    /**
     * Adds a resource of type stackset to the statement
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-concepts.html#stacksets-concepts-stackset
     *
     * @param stackSetName - Identifier for the stackSetName.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStackset(stackSetName, id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudformation:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:stackset/${stackSetName}:${id}`);
    }
    /**
     * Adds a resource of type stackset-target to the statement
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html
     *
     * @param stackSetTarget - Identifier for the stackSetTarget.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onStacksetTarget(stackSetTarget, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudformation:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:stackset-target/${stackSetTarget}`);
    }
    /**
     * Adds a resource of type type to the statement
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html
     *
     * @param type - Identifier for the type.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onType(type, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudformation:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:type/resource/${type}`);
    }
    /**
     * Adds a resource of type generatedtemplate to the statement
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/generate-IaC.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onGeneratedtemplate(id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudformation:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:generatedTemplate/${id}`);
    }
    /**
     * Adds a resource of type resourcescan to the statement
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/generate-IaC.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onResourcescan(id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudformation:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:resourceScan/${id}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#using-iam-template-conditions
     *
     * Applies to actions:
     * - .toCreateChangeSet()
     * - .toCreateStack()
     * - .toCreateStackSet()
     * - .toTagResource()
     * - .toUpdateStack()
     * - .toUpdateStackSet()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#using-iam-template-conditions
     *
     * Applies to resource types:
     * - changeset
     * - stack
     * - stackset
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#using-iam-template-conditions
     *
     * Applies to actions:
     * - .toCreateChangeSet()
     * - .toCreateStack()
     * - .toCreateStackInstances()
     * - .toCreateStackSet()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateStack()
     * - .toUpdateStackSet()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by an AWS CloudFormation change set name. Use to control which change sets IAM users can execute or delete
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#using-iam-template-conditions
     *
     * Applies to actions:
     * - .toCreateChangeSet()
     * - .toDeleteChangeSet()
     * - .toDescribeChangeSet()
     * - .toDescribeChangeSetHooks()
     * - .toExecuteChangeSet()
     * - .toListHookResults()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifChangeSetName(value, operator) {
        return this.if(`ChangeSetName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the name of a resource-mutating API action. Use to control which APIs IAM users can use to add or remove tags on a stack or stack set
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#using-iam-template-conditions
     *
     * Applies to actions:
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCreateAction(value, operator) {
        return this.if(`CreateAction`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the template resource types, such as AWS::EC2::Instance. Use to control which resource types IAM users can work with when they want to import a resource into a stack
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#using-iam-template-conditions
     *
     * Applies to actions:
     * - .toCreateChangeSet()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifImportResourceTypes(value, operator) {
        return this.if(`ImportResourceTypes`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the template resource types, such as AWS::EC2::Instance. Use to control which resource types IAM users can work with when they create or update a stack
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#using-iam-template-conditions
     *
     * Applies to actions:
     * - .toCreateChangeSet()
     * - .toCreateStack()
     * - .toUpdateStack()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTypes(value, operator) {
        return this.if(`ResourceTypes`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ARN of an IAM service role. Use to control which service role IAM users can use to work with stacks or change sets
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#using-iam-template-conditions
     *
     * Applies to actions:
     * - .toContinueUpdateRollback()
     * - .toCreateChangeSet()
     * - .toCreateStack()
     * - .toCreateStackSet()
     * - .toDeleteStack()
     * - .toRollbackStack()
     * - .toUpdateStack()
     * - .toUpdateStackSet()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifRoleArn(value, operator) {
        return this.if(`RoleArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by an Amazon S3 stack policy URL. Use to control which stack policies IAM users can associate with a stack during a create or update stack action
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#using-iam-template-conditions
     *
     * Applies to actions:
     * - .toCreateChangeSet()
     * - .toCreateStack()
     * - .toSetStackPolicy()
     * - .toUpdateStack()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifStackPolicyUrl(value, operator) {
        return this.if(`StackPolicyUrl`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by stack set target region. Use to control which regions IAM users can use when they create or update stack sets
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#using-iam-template-conditions
     *
     * Applies to actions:
     * - .toCreateStackInstances()
     * - .toDeleteStackInstances()
     * - .toUpdateStackInstances()
     * - .toUpdateStackSet()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTargetRegion(value, operator) {
        return this.if(`TargetRegion`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by an Amazon S3 template URL. Use to control which templates IAM users can use when they create or update stacks
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#using-iam-template-conditions
     *
     * Applies to actions:
     * - .toCreateChangeSet()
     * - .toCreateStack()
     * - .toCreateStackSet()
     * - .toEstimateTemplateCost()
     * - .toGetTemplateSummary()
     * - .toUpdateStack()
     * - .toUpdateStackSet()
     * - .toValidateTemplate()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTemplateUrl(value, operator) {
        return this.if(`TemplateUrl`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ARN of a CloudFormation extension
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#using-iam-template-conditions
     *
     * Applies to actions:
     * - .toListAllHookResults()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifTypeArn(value, operator) {
        return this.if(`TypeArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Statement provider for service [cloudformation](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscloudformation.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'cloudformation';
        this.accessLevelList = {
            Write: [
                'ActivateOrganizationsAccess',
                'ActivateType',
                'CancelUpdateStack',
                'ContinueUpdateRollback',
                'CreateChangeSet',
                'CreateGeneratedTemplate',
                'CreateStack',
                'CreateStackInstances',
                'CreateStackRefactor',
                'CreateStackSet',
                'CreateUploadBucket',
                'DeactivateOrganizationsAccess',
                'DeactivateType',
                'DeleteChangeSet',
                'DeleteGeneratedTemplate',
                'DeleteStack',
                'DeleteStackInstances',
                'DeleteStackSet',
                'DeregisterType',
                'ExecuteChangeSet',
                'ExecuteStackRefactor',
                'ImportStacksToStackSet',
                'PublishType',
                'RecordHandlerProgress',
                'RegisterPublisher',
                'RegisterType',
                'RollbackStack',
                'SetTypeConfiguration',
                'SetTypeDefaultVersion',
                'SignalResource',
                'StartResourceScan',
                'StopStackSetOperation',
                'TestType',
                'UpdateGeneratedTemplate',
                'UpdateStack',
                'UpdateStackInstances',
                'UpdateStackSet',
                'UpdateTerminationProtection'
            ],
            Read: [
                'BatchDescribeTypeConfigurations',
                'DescribeAccountLimits',
                'DescribeChangeSet',
                'DescribeChangeSetHooks',
                'DescribeGeneratedTemplate',
                'DescribeOrganizationsAccess',
                'DescribePublisher',
                'DescribeResourceScan',
                'DescribeStackDriftDetectionStatus',
                'DescribeStackEvents',
                'DescribeStackInstance',
                'DescribeStackRefactor',
                'DescribeStackResource',
                'DescribeStackResourceDrifts',
                'DescribeStackResources',
                'DescribeStackSet',
                'DescribeStackSetOperation',
                'DescribeType',
                'DescribeTypeRegistration',
                'DetectStackDrift',
                'DetectStackResourceDrift',
                'DetectStackSetDrift',
                'EstimateTemplateCost',
                'GetGeneratedTemplate',
                'GetStackPolicy',
                'GetTemplate',
                'GetTemplateSummary',
                'ValidateTemplate'
            ],
            List: [
                'DescribeStacks',
                'ListAllHookResults',
                'ListChangeSets',
                'ListExports',
                'ListGeneratedTemplates',
                'ListHookResults',
                'ListImports',
                'ListResourceScanRelatedResources',
                'ListResourceScanResources',
                'ListResourceScans',
                'ListStackInstanceResourceDrifts',
                'ListStackInstances',
                'ListStackRefactorActions',
                'ListStackRefactors',
                'ListStackResources',
                'ListStackSetAutoDeploymentTargets',
                'ListStackSetOperationResults',
                'ListStackSetOperations',
                'ListStackSets',
                'ListStacks',
                'ListTypeRegistrations',
                'ListTypeVersions',
                'ListTypes'
            ],
            'Permissions management': [
                'SetStackPolicy'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Cloudformation = Cloudformation;
//# sourceMappingURL=data:application/json;base64,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