"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Servicediscovery = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [servicediscovery](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscloudmap.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Servicediscovery extends shared_1.PolicyStatement {
    /**
     * Grants permission to create an HTTP namespace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateHttpNamespace.html
     */
    toCreateHttpNamespace() {
        return this.to('CreateHttpNamespace');
    }
    /**
     * Grants permission to create a private namespace based on DNS, which will be visible only inside a specified Amazon VPC
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_CreatePrivateDnsNamespace.html
     */
    toCreatePrivateDnsNamespace() {
        return this.to('CreatePrivateDnsNamespace');
    }
    /**
     * Grants permission to create a public namespace based on DNS, which will be visible on the internet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_CreatePublicDnsNamespace.html
     */
    toCreatePublicDnsNamespace() {
        return this.to('CreatePublicDnsNamespace');
    }
    /**
     * Grants permission to create a service
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifNamespaceArn()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateService.html
     */
    toCreateService() {
        return this.to('CreateService');
    }
    /**
     * Grants permission to delete a specified namespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_DeleteNamespace.html
     */
    toDeleteNamespace() {
        return this.to('DeleteNamespace');
    }
    /**
     * Grants permission to delete the RAM access control policy for a namespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/sharing-namespaces.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete a specified service
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServiceCreatedByAccount()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_DeleteService.html
     */
    toDeleteService() {
        return this.to('DeleteService');
    }
    /**
     * Grants permission to delete specified attributes from a service
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServiceCreatedByAccount()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_DeleteServiceAttributes.html
     */
    toDeleteServiceAttributes() {
        return this.to('DeleteServiceAttributes');
    }
    /**
     * Grants permission to delete the records and the health check, if any, that Amazon Route 53 created for the specified instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServiceArn()
     * - .ifServiceCreatedByAccount()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_DeregisterInstance.html
     */
    toDeregisterInstance() {
        return this.to('DeregisterInstance');
    }
    /**
     * Grants permission to discover registered instances for a specified namespace and service
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifNamespaceName()
     * - .ifServiceName()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_DiscoverInstances.html
     */
    toDiscoverInstances() {
        return this.to('DiscoverInstances');
    }
    /**
     * Grants permission to discover the revision of the instances for a specified namespace and service
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifNamespaceName()
     * - .ifServiceName()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_DiscoverInstancesRevision.html
     */
    toDiscoverInstancesRevision() {
        return this.to('DiscoverInstancesRevision');
    }
    /**
     * Grants permission to get information about a specified instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifServiceArn()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_GetInstance.html
     */
    toGetInstance() {
        return this.to('GetInstance');
    }
    /**
     * Grants permission to get the current health status (Healthy, Unhealthy, or Unknown) of one or more instances
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifServiceArn()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_GetInstancesHealthStatus.html
     */
    toGetInstancesHealthStatus() {
        return this.to('GetInstancesHealthStatus');
    }
    /**
     * Grants permission to get information about a namespace
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_GetNamespace.html
     */
    toGetNamespace() {
        return this.to('GetNamespace');
    }
    /**
     * Grants permission to get information about a specific operation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_GetOperation.html
     */
    toGetOperation() {
        return this.to('GetOperation');
    }
    /**
     * Grants permission to read the RAM access control policy for a namespace
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/sharing-namespaces.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to get the settings for a specified service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_GetService.html
     */
    toGetService() {
        return this.to('GetService');
    }
    /**
     * Grants permission to get the attributes for a specified service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_GetServiceAttributes.html
     */
    toGetServiceAttributes() {
        return this.to('GetServiceAttributes');
    }
    /**
     * Grants permission to get summary information about the instances that were registered with a specified service
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifServiceArn()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_ListInstances.html
     */
    toListInstances() {
        return this.to('ListInstances');
    }
    /**
     * Grants permission to get information about the namespaces
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_ListNamespaces.html
     */
    toListNamespaces() {
        return this.to('ListNamespaces');
    }
    /**
     * Grants permission to list operations that match the criteria that you specify
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_ListOperations.html
     */
    toListOperations() {
        return this.to('ListOperations');
    }
    /**
     * Grants permission to get settings for all the services that match specified filters
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_ListServices.html
     */
    toListServices() {
        return this.to('ListServices');
    }
    /**
     * Grants permission to lists tags for the specified resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to define the RAM access control policy for a namespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/sharing-namespaces.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to register an instance based on the settings in a specified service
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServiceArn()
     * - .ifServiceCreatedByAccount()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html
     */
    toRegisterInstance() {
        return this.to('RegisterInstance');
    }
    /**
     * Grants permission to add one or more tags to the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove one or more tags from the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the settings for a HTTP namespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_UpdateHttpNamespace.html
     */
    toUpdateHttpNamespace() {
        return this.to('UpdateHttpNamespace');
    }
    /**
     * Grants permission to update the current health status for an instance that has a custom health check
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServiceArn()
     * - .ifServiceCreatedByAccount()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_UpdateInstanceCustomHealthStatus.html
     */
    toUpdateInstanceCustomHealthStatus() {
        return this.to('UpdateInstanceCustomHealthStatus');
    }
    /**
     * Grants permission to update the settings for a private DNS namespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_UpdatePrivateDnsNamespace.html
     */
    toUpdatePrivateDnsNamespace() {
        return this.to('UpdatePrivateDnsNamespace');
    }
    /**
     * Grants permission to update the settings for a public DNS namespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_UpdatePublicDnsNamespace.html
     */
    toUpdatePublicDnsNamespace() {
        return this.to('UpdatePublicDnsNamespace');
    }
    /**
     * Grants permission to update the settings in a specified service
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServiceCreatedByAccount()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_UpdateService.html
     */
    toUpdateService() {
        return this.to('UpdateService');
    }
    /**
     * Grants permission to update the attributes in a specified service
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServiceCreatedByAccount()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_UpdateServiceAttributes.html
     */
    toUpdateServiceAttributes() {
        return this.to('UpdateServiceAttributes');
    }
    /**
     * Adds a resource of type namespace to the statement
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/API_Namespace.html
     *
     * @param namespaceId - Identifier for the namespaceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onNamespace(namespaceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:servicediscovery:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:namespace/${namespaceId}`);
    }
    /**
     * Adds a resource of type service to the statement
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/API_Service.html
     *
     * @param serviceId - Identifier for the serviceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onService(serviceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:servicediscovery:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:service/${serviceId}`);
    }
    /**
     * Filters actions based on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateHttpNamespace()
     * - .toCreatePrivateDnsNamespace()
     * - .toCreatePublicDnsNamespace()
     * - .toCreateService()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - namespace
     * - service
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateHttpNamespace()
     * - .toCreatePrivateDnsNamespace()
     * - .toCreatePublicDnsNamespace()
     * - .toCreateService()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by specifying the Amazon Resource Name (ARN) for the related namespace
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/access-control-overview.html#specifying-conditions
     *
     * Applies to actions:
     * - .toCreateService()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifNamespaceArn(value, operator) {
        return this.if(`NamespaceArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by specifying the name of the related namespace
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/access-control-overview.html#specifying-conditions
     *
     * Applies to actions:
     * - .toDiscoverInstances()
     * - .toDiscoverInstancesRevision()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifNamespaceName(value, operator) {
        return this.if(`NamespaceName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by specifying the Amazon Resource Name (ARN) for the related service
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/access-control-overview.html#specifying-conditions
     *
     * Applies to actions:
     * - .toDeregisterInstance()
     * - .toGetInstance()
     * - .toGetInstancesHealthStatus()
     * - .toListInstances()
     * - .toRegisterInstance()
     * - .toUpdateInstanceCustomHealthStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifServiceArn(value, operator) {
        return this.if(`ServiceArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by specifying the account id of the related service creator
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/access-control-overview.html#specifying-conditions
     *
     * Applies to actions:
     * - .toDeleteService()
     * - .toDeleteServiceAttributes()
     * - .toDeregisterInstance()
     * - .toRegisterInstance()
     * - .toUpdateInstanceCustomHealthStatus()
     * - .toUpdateService()
     * - .toUpdateServiceAttributes()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifServiceCreatedByAccount(value, operator) {
        return this.if(`ServiceCreatedByAccount`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by specifying the name of the related service
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/access-control-overview.html#specifying-conditions
     *
     * Applies to actions:
     * - .toDiscoverInstances()
     * - .toDiscoverInstancesRevision()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifServiceName(value, operator) {
        return this.if(`ServiceName`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [servicediscovery](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscloudmap.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'servicediscovery';
        this.accessLevelList = {
            Write: [
                'CreateHttpNamespace',
                'CreatePrivateDnsNamespace',
                'CreatePublicDnsNamespace',
                'CreateService',
                'DeleteNamespace',
                'DeleteResourcePolicy',
                'DeleteService',
                'DeleteServiceAttributes',
                'DeregisterInstance',
                'PutResourcePolicy',
                'RegisterInstance',
                'UpdateHttpNamespace',
                'UpdateInstanceCustomHealthStatus',
                'UpdatePrivateDnsNamespace',
                'UpdatePublicDnsNamespace',
                'UpdateService',
                'UpdateServiceAttributes'
            ],
            Read: [
                'DiscoverInstances',
                'DiscoverInstancesRevision',
                'GetInstance',
                'GetInstancesHealthStatus',
                'GetNamespace',
                'GetOperation',
                'GetResourcePolicy',
                'GetService',
                'GetServiceAttributes',
                'ListInstances',
                'ListNamespaces',
                'ListServices',
                'ListTagsForResource'
            ],
            List: [
                'ListOperations'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Servicediscovery = Servicediscovery;
//# sourceMappingURL=data:application/json;base64,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