"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cloudtrail = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [cloudtrail](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscloudtrail.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Cloudtrail extends shared_1.PolicyStatement {
    /**
     * Grants permission to add one or more tags to a trail, event data store, channel or dashboard, up to a limit of 50
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_AddTags.html
     */
    toAddTags() {
        return this.to('AddTags');
    }
    /**
     * Grants permission to cancel a running query
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_CancelQuery.html
     */
    toCancelQuery() {
        return this.to('CancelQuery');
    }
    /**
     * Grants permission to create a channel
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - cloudtrail:AddTags
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_CreateChannel.html
     */
    toCreateChannel() {
        return this.to('CreateChannel');
    }
    /**
     * Grants permission to create a dashboard
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - cloudtrail:AddTags
     * - cloudtrail:StartDashboardRefresh
     * - cloudtrail:StartQuery
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_CreateDashboard.html
     */
    toCreateDashboard() {
        return this.to('CreateDashboard');
    }
    /**
     * Grants permission to create an event data store
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - cloudtrail:AddTags
     * - iam:CreateServiceLinkedRole
     * - iam:GetRole
     * - kms:Decrypt
     * - kms:GenerateDataKey
     * - organizations:ListAWSServiceAccessForOrganization
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_CreateEventDataStore.html
     */
    toCreateEventDataStore() {
        return this.to('CreateEventDataStore');
    }
    /**
     * Grants permission to create a service-linked channel that specifies the settings for delivery of log data to an AWS service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/viewing-service-linked-channels.html#slc-service-events
     */
    toCreateServiceLinkedChannel() {
        return this.to('CreateServiceLinkedChannel');
    }
    /**
     * Grants permission to create a trail that specifies the settings for delivery of log data to an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - cloudtrail:AddTags
     * - iam:CreateServiceLinkedRole
     * - iam:GetRole
     * - organizations:ListAWSServiceAccessForOrganization
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_CreateTrail.html
     */
    toCreateTrail() {
        return this.to('CreateTrail');
    }
    /**
     * Grants permission to delete a channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_DeleteChannel.html
     */
    toDeleteChannel() {
        return this.to('DeleteChannel');
    }
    /**
     * Grants permission to delete a dashboard
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_DeleteDashboard.html
     */
    toDeleteDashboard() {
        return this.to('DeleteDashboard');
    }
    /**
     * Grants permission to delete an event data store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_DeleteEventDataStore.html
     */
    toDeleteEventDataStore() {
        return this.to('DeleteEventDataStore');
    }
    /**
     * Grants permission to delete a resource policy from the provided resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete a service-linked channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/viewing-service-linked-channels.html#slc-service-events
     */
    toDeleteServiceLinkedChannel() {
        return this.to('DeleteServiceLinkedChannel');
    }
    /**
     * Grants permission to delete a trail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_DeleteTrail.html
     */
    toDeleteTrail() {
        return this.to('DeleteTrail');
    }
    /**
     * Grants permission to deregister an AWS Organizations member account as a delegated administrator
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - organizations:DeregisterDelegatedAdministrator
     * - organizations:ListAWSServiceAccessForOrganization
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_DeregisterOrganizationDelegatedAdmin.html
     */
    toDeregisterOrganizationDelegatedAdmin() {
        return this.to('DeregisterOrganizationDelegatedAdmin');
    }
    /**
     * Grants permission to list details for the query
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_DescribeQuery.html
     */
    toDescribeQuery() {
        return this.to('DescribeQuery');
    }
    /**
     * Grants permission to list settings for the trails associated with the current region for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_DescribeTrails.html
     */
    toDescribeTrails() {
        return this.to('DescribeTrails');
    }
    /**
     * Grants permission to disable federation of event data store data by using the AWS Glue Data Catalog
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - glue:DeleteDatabase
     * - glue:DeleteTable
     * - glue:PassConnection
     * - lakeformation:DeregisterResource
     * - lakeformation:RegisterResource
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_DisableFederation.html
     */
    toDisableFederation() {
        return this.to('DisableFederation');
    }
    /**
     * Grants permission to enable federation of event data store data by using the AWS Glue Data Catalog
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - glue:CreateDatabase
     * - glue:CreateTable
     * - iam:GetRole
     * - iam:PassRole
     * - lakeformation:DeregisterResource
     * - lakeformation:RegisterResource
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_EnableFederation.html
     */
    toEnableFederation() {
        return this.to('EnableFederation');
    }
    /**
     * Grants permission to generate a query for a specified event data store using the CloudTrail Lake query generator
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/lake-query-generator.html
     */
    toGenerateQuery() {
        return this.to('GenerateQuery');
    }
    /**
     * Grants permission to generate a results summary for specified queries using the CloudTrail natural language generator
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - cloudtrail:GetQueryResults
     * - kms:Decrypt
     * - kms:GenerateDataKey
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/query-results-summary.html
     */
    toGenerateQueryResultsSummary() {
        return this.to('GenerateQueryResultsSummary');
    }
    /**
     * Grants permission to return information about a specific channel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetChannel.html
     */
    toGetChannel() {
        return this.to('GetChannel');
    }
    /**
     * Grants permission to list settings for the dashboard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetDashboard.html
     */
    toGetDashboard() {
        return this.to('GetDashboard');
    }
    /**
     * Grants permission to list event configurations that are configured for an event data store
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetEventConfiguration.html
     */
    toGetEventConfiguration() {
        return this.to('GetEventConfiguration');
    }
    /**
     * Grants permission to list settings for the event data store
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetEventDataStore.html
     */
    toGetEventDataStore() {
        return this.to('GetEventDataStore');
    }
    /**
     * Grants permission to get data from an event data store by using the AWS Glue Data Catalog
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     * - kms:GenerateDataKey
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/query-federation.html#query-federation-permissions
     */
    toGetEventDataStoreData() {
        return this.to('GetEventDataStoreData');
    }
    /**
     * Grants permission to list settings for event selectors configured for a trail
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetEventSelectors.html
     */
    toGetEventSelectors() {
        return this.to('GetEventSelectors');
    }
    /**
     * Grants permission to return information about a specific import
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetImport.html
     */
    toGetImport() {
        return this.to('GetImport');
    }
    /**
     * Grants permission to list CloudTrail Insights selectors that are configured for a trail or event data store
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetInsightSelectors.html
     */
    toGetInsightSelectors() {
        return this.to('GetInsightSelectors');
    }
    /**
     * Grants permission to fetch results of a complete query
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     * - kms:GenerateDataKey
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetQueryResults.html
     */
    toGetQueryResults() {
        return this.to('GetQueryResults');
    }
    /**
     * Grants permission to get the resource policy attached to the provided resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetResourcePolicy.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to list settings for the service-linked channel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/viewing-service-linked-channels.html#slc-service-events
     */
    toGetServiceLinkedChannel() {
        return this.to('GetServiceLinkedChannel');
    }
    /**
     * Grants permission to list settings for the trail
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetTrail.html
     */
    toGetTrail() {
        return this.to('GetTrail');
    }
    /**
     * Grants permission to retrieve a JSON-formatted list of information about the specified trail
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetTrailStatus.html
     */
    toGetTrailStatus() {
        return this.to('GetTrailStatus');
    }
    /**
     * Grants permission to list the channels in the current account, and their source names
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_ListChannels.html
     */
    toListChannels() {
        return this.to('ListChannels');
    }
    /**
     * Grants permission to list dashboards associated with the current region for your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_ListDashboards.html
     */
    toListDashboards() {
        return this.to('ListDashboards');
    }
    /**
     * Grants permission to list event data stores associated with the current region for your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_ListEventDataStores.html
     */
    toListEventDataStores() {
        return this.to('ListEventDataStores');
    }
    /**
     * Grants permission to return a list of failures for the specified import
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_ListImportFailures.html
     */
    toListImportFailures() {
        return this.to('ListImportFailures');
    }
    /**
     * Grants permission to return information on all imports, or a select set of imports by ImportStatus or Destination
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_ListImports.html
     */
    toListImports() {
        return this.to('ListImports');
    }
    /**
     * Grants permission to list the public keys whose private keys were used to sign trail digest files within a specified time range
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_ListPublicKeys.html
     */
    toListPublicKeys() {
        return this.to('ListPublicKeys');
    }
    /**
     * Grants permission to list queries associated with an event data store
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_ListQueries.html
     */
    toListQueries() {
        return this.to('ListQueries');
    }
    /**
     * Grants permission to list service-linked channels associated with the current region for a specified account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/viewing-service-linked-channels.html#slc-service-events
     */
    toListServiceLinkedChannels() {
        return this.to('ListServiceLinkedChannels');
    }
    /**
     * Grants permission to list the tags for trails, event data stores, channels or dashboards in the current region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_ListTags.html
     */
    toListTags() {
        return this.to('ListTags');
    }
    /**
     * Grants permission to list trails associated with the current region for your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_ListTrails.html
     */
    toListTrails() {
        return this.to('ListTrails');
    }
    /**
     * Grants permission to look up and retrieve metric data for API activity events captured by CloudTrail that create, update, or delete resources in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_LookupEvents.html
     */
    toLookupEvents() {
        return this.to('LookupEvents');
    }
    /**
     * Grants permission to create and update event configurations for an event data store
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - iam:GetRole
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_PutEventConfiguration.html
     */
    toPutEventConfiguration() {
        return this.to('PutEventConfiguration');
    }
    /**
     * Grants permission to create and update event selectors for a trail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_PutEventSelectors.html
     */
    toPutEventSelectors() {
        return this.to('PutEventSelectors');
    }
    /**
     * Grants permission to create and update CloudTrail Insights selectors for a trail or event data store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_PutInsightSelectors.html
     */
    toPutInsightSelectors() {
        return this.to('PutInsightSelectors');
    }
    /**
     * Grants permission to attach a resource policy to the provided resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to register an AWS Organizations member account as a delegated administrator
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - iam:GetRole
     * - organizations:ListAWSServiceAccessForOrganization
     * - organizations:RegisterDelegatedAdministrator
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_RegisterOrganizationDelegatedAdmin.html
     */
    toRegisterOrganizationDelegatedAdmin() {
        return this.to('RegisterOrganizationDelegatedAdmin');
    }
    /**
     * Grants permission to remove tags from a trail, event data store, channel or dashboard
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_RemoveTags.html
     */
    toRemoveTags() {
        return this.to('RemoveTags');
    }
    /**
     * Grants permission to restore an event data store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_RestoreEventDataStore.html
     */
    toRestoreEventDataStore() {
        return this.to('RestoreEventDataStore');
    }
    /**
     * Grants permission to perform semantic search for CloudTrail Lake sample queries
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/lake-console-queries.html
     */
    toSearchSampleQueries() {
        return this.to('SearchSampleQueries');
    }
    /**
     * Grants permission to start a refresh on the specified dashboard
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - cloudtrail:StartQuery
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_StartDashboardRefresh.html
     */
    toStartDashboardRefresh() {
        return this.to('StartDashboardRefresh');
    }
    /**
     * Grants permission to start ingestion on an event data store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_StartEventDataStoreIngestion.html
     */
    toStartEventDataStoreIngestion() {
        return this.to('StartEventDataStoreIngestion');
    }
    /**
     * Grants permission to start an import of logged trail events from a source S3 bucket to a destination event data store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_StartImport.html
     */
    toStartImport() {
        return this.to('StartImport');
    }
    /**
     * Grants permission to start the recording of AWS API calls and log file delivery for a trail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_StartLogging.html
     */
    toStartLogging() {
        return this.to('StartLogging');
    }
    /**
     * Grants permission to start a new query on a specified event data store
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     * - kms:GenerateDataKey
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_StartQuery.html
     */
    toStartQuery() {
        return this.to('StartQuery');
    }
    /**
     * Grants permission to stop ingestion on an event data store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_StopEventDataStoreIngestion.html
     */
    toStopEventDataStoreIngestion() {
        return this.to('StopEventDataStoreIngestion');
    }
    /**
     * Grants permission to stop a specified import
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_StopImport.html
     */
    toStopImport() {
        return this.to('StopImport');
    }
    /**
     * Grants permission to stop the recording of AWS API calls and log file delivery for a trail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_StopLogging.html
     */
    toStopLogging() {
        return this.to('StopLogging');
    }
    /**
     * Grants permission to update a channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_UpdateChannel.html
     */
    toUpdateChannel() {
        return this.to('UpdateChannel');
    }
    /**
     * Grants permission to update a dashboard
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - cloudtrail:StartDashboardRefresh
     * - cloudtrail:StartQuery
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_UpdateDashboard.html
     */
    toUpdateDashboard() {
        return this.to('UpdateDashboard');
    }
    /**
     * Grants permission to update an event data store
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - iam:GetRole
     * - kms:Decrypt
     * - kms:GenerateDataKey
     * - organizations:ListAWSServiceAccessForOrganization
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_UpdateEventDataStore.html
     */
    toUpdateEventDataStore() {
        return this.to('UpdateEventDataStore');
    }
    /**
     * Grants permission to update the service-linked channel settings for delivery of log data to an AWS service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/viewing-service-linked-channels.html#slc-service-events
     */
    toUpdateServiceLinkedChannel() {
        return this.to('UpdateServiceLinkedChannel');
    }
    /**
     * Grants permission to update the settings that specify delivery of log files
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - iam:GetRole
     * - organizations:ListAWSServiceAccessForOrganization
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_UpdateTrail.html
     */
    toUpdateTrail() {
        return this.to('UpdateTrail');
    }
    /**
     * Adds a resource of type trail to the statement
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/how-cloudtrail-works.html#how-cloudtrail-works-trails
     *
     * @param trailName - Identifier for the trailName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTrail(trailName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudtrail:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:trail/${trailName}`);
    }
    /**
     * Adds a resource of type eventdatastore to the statement
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/how-cloudtrail-works.html#how-cloudtrail-works-lake
     *
     * @param eventDataStoreId - Identifier for the eventDataStoreId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEventdatastore(eventDataStoreId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudtrail:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:eventdatastore/${eventDataStoreId}`);
    }
    /**
     * Adds a resource of type channel to the statement
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/how-cloudtrail-works.html#how-cloudtrail-works-channels
     *
     * @param channelId - Identifier for the channelId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onChannel(channelId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudtrail:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:channel/${channelId}`);
    }
    /**
     * Adds a resource of type dashboard to the statement
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/lake-dashboard.html
     *
     * @param dashboardName - Identifier for the dashboardName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDashboard(dashboardName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudtrail:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:dashboard/${dashboardName}`);
    }
    /**
     * Filters access by the tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toAddTags()
     * - .toCreateChannel()
     * - .toCreateDashboard()
     * - .toCreateEventDataStore()
     * - .toCreateTrail()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - trail
     * - eventdatastore
     * - channel
     * - dashboard
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys in a request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toAddTags()
     * - .toCreateChannel()
     * - .toCreateDashboard()
     * - .toCreateEventDataStore()
     * - .toCreateTrail()
     * - .toRemoveTags()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [cloudtrail](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscloudtrail.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'cloudtrail';
        this.accessLevelList = {
            Tagging: [
                'AddTags',
                'RemoveTags'
            ],
            Write: [
                'CancelQuery',
                'CreateChannel',
                'CreateDashboard',
                'CreateEventDataStore',
                'CreateServiceLinkedChannel',
                'CreateTrail',
                'DeleteChannel',
                'DeleteDashboard',
                'DeleteEventDataStore',
                'DeleteResourcePolicy',
                'DeleteServiceLinkedChannel',
                'DeleteTrail',
                'DeregisterOrganizationDelegatedAdmin',
                'DisableFederation',
                'EnableFederation',
                'GenerateQuery',
                'PutEventConfiguration',
                'PutEventSelectors',
                'PutInsightSelectors',
                'PutResourcePolicy',
                'RegisterOrganizationDelegatedAdmin',
                'RestoreEventDataStore',
                'StartDashboardRefresh',
                'StartEventDataStoreIngestion',
                'StartImport',
                'StartLogging',
                'StartQuery',
                'StopEventDataStoreIngestion',
                'StopImport',
                'StopLogging',
                'UpdateChannel',
                'UpdateDashboard',
                'UpdateEventDataStore',
                'UpdateServiceLinkedChannel',
                'UpdateTrail'
            ],
            Read: [
                'DescribeQuery',
                'DescribeTrails',
                'GenerateQueryResultsSummary',
                'GetChannel',
                'GetDashboard',
                'GetEventConfiguration',
                'GetEventDataStore',
                'GetEventDataStoreData',
                'GetEventSelectors',
                'GetImport',
                'GetInsightSelectors',
                'GetQueryResults',
                'GetResourcePolicy',
                'GetServiceLinkedChannel',
                'GetTrail',
                'GetTrailStatus',
                'ListImportFailures',
                'ListPublicKeys',
                'ListTags',
                'LookupEvents',
                'SearchSampleQueries'
            ],
            List: [
                'ListChannels',
                'ListDashboards',
                'ListEventDataStores',
                'ListImports',
                'ListQueries',
                'ListServiceLinkedChannels',
                'ListTrails'
            ]
        };
    }
}
exports.Cloudtrail = Cloudtrail;
//# sourceMappingURL=data:application/json;base64,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