"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cloudwatch = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [cloudwatch](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncloudwatch.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Cloudwatch extends shared_1.PolicyStatement {
    /**
     * Grants permission to batch get service level indicator report
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Application-Monitoring-Sections.html#ApplicationSignals-PreviewSDK
     */
    toBatchGetServiceLevelIndicatorReport() {
        return this.to('BatchGetServiceLevelIndicatorReport');
    }
    /**
     * Grants permission to batch retrieve a service level objective budget report
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Application-Monitoring-Sections.html#ApplicationSignals-PreviewSDK
     */
    toBatchGetServiceLevelObjectiveBudgetReport() {
        return this.to('BatchGetServiceLevelObjectiveBudgetReport');
    }
    /**
     * Grants permission to create a service level objective
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Application-Monitoring-Sections.html#ApplicationSignals-PreviewSDK
     */
    toCreateServiceLevelObjective() {
        return this.to('CreateServiceLevelObjective');
    }
    /**
     * Grants permission to delete a collection of alarms
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DeleteAlarms.html
     */
    toDeleteAlarms() {
        return this.to('DeleteAlarms');
    }
    /**
     * Grants permission to delete the specified anomaly detection model from your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DeleteAnomalyDetector.html
     */
    toDeleteAnomalyDetector() {
        return this.to('DeleteAnomalyDetector');
    }
    /**
     * Grants permission to delete all CloudWatch dashboards that you specify
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DeleteDashboards.html
     */
    toDeleteDashboards() {
        return this.to('DeleteDashboards');
    }
    /**
     * Grants permission to delete a collection of insight rules
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DeleteInsightRules.html
     */
    toDeleteInsightRules() {
        return this.to('DeleteInsightRules');
    }
    /**
     * Grants permission to delete the CloudWatch metric stream that you specify
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DeleteMetricStream.html
     */
    toDeleteMetricStream() {
        return this.to('DeleteMetricStream');
    }
    /**
     * Grants permission to delete a service level objective
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Application-Monitoring-Sections.html#ApplicationSignals-PreviewSDK
     */
    toDeleteServiceLevelObjective() {
        return this.to('DeleteServiceLevelObjective');
    }
    /**
     * Grants permission to retrieve the history for the specified alarm
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeAlarmHistory.html
     */
    toDescribeAlarmHistory() {
        return this.to('DescribeAlarmHistory');
    }
    /**
     * Grants permission to describe all alarms, currently owned by the user's account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeAlarms.html
     */
    toDescribeAlarms() {
        return this.to('DescribeAlarms');
    }
    /**
     * Grants permission to describe all alarms configured on the specified metric, currently owned by the user's account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeAlarmsForMetric.html
     */
    toDescribeAlarmsForMetric() {
        return this.to('DescribeAlarmsForMetric');
    }
    /**
     * Grants permission to list the anomaly detection models that you have created in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeAnomalyDetectors.html
     */
    toDescribeAnomalyDetectors() {
        return this.to('DescribeAnomalyDetectors');
    }
    /**
     * Grants permission to describe all insight rules, currently owned by the user's account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeInsightRules.html
     */
    toDescribeInsightRules() {
        return this.to('DescribeInsightRules');
    }
    /**
     * Grants permission to disable actions for a collection of alarms
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DisableAlarmActions.html
     */
    toDisableAlarmActions() {
        return this.to('DisableAlarmActions');
    }
    /**
     * Grants permission to disable a collection of insight rules
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DisableInsightRules.html
     */
    toDisableInsightRules() {
        return this.to('DisableInsightRules');
    }
    /**
     * Grants permission to enable actions for a collection of alarms
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_EnableAlarmActions.html
     */
    toEnableAlarmActions() {
        return this.to('EnableAlarmActions');
    }
    /**
     * Grants permission to enable a collection of insight rules
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_EnableInsightRules.html
     */
    toEnableInsightRules() {
        return this.to('EnableInsightRules');
    }
    /**
     * Grants permission to enable a CloudWatch topology discovery
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Application-Monitoring-Sections.html#ApplicationSignals-PreviewSDK
     */
    toEnableTopologyDiscovery() {
        return this.to('EnableTopologyDiscovery');
    }
    /**
     * Grants permission to generate a Metrics Insights or Logs Insights query string from a natural language prompt
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch-metrics-insights-query-assist.html
     */
    toGenerateQuery() {
        return this.to('GenerateQuery');
    }
    /**
     * Grants permission to generate a summary of CloudWatch LogInsights query results in natural language using generative AI
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/CloudWatchLogs-Insights-Query-Results-Summary.html
     */
    toGenerateQueryResultsSummary() {
        return this.to('GenerateQueryResultsSummary');
    }
    /**
     * Grants permission to display the details of the CloudWatch dashboard you specify
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetDashboard.html
     */
    toGetDashboard() {
        return this.to('GetDashboard');
    }
    /**
     * Grants permission to return the top-N report of unique contributors over a time range for a given insight rule
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetInsightRuleReport.html
     */
    toGetInsightRuleReport() {
        return this.to('GetInsightRuleReport');
    }
    /**
     * Grants permission to retrieve batch amounts of CloudWatch metric data and perform metric math on retrieved data
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricData.html
     */
    toGetMetricData() {
        return this.to('GetMetricData');
    }
    /**
     * Grants permission to retrieve statistics for the specified metric
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricStatistics.html
     */
    toGetMetricStatistics() {
        return this.to('GetMetricStatistics');
    }
    /**
     * Grants permission to return the details of a CloudWatch metric stream
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricStream.html
     */
    toGetMetricStream() {
        return this.to('GetMetricStream');
    }
    /**
     * Grants permission to retrieve snapshots of metric widgets
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricWidgetImage.html
     */
    toGetMetricWidgetImage() {
        return this.to('GetMetricWidgetImage');
    }
    /**
     * Grants permission to retrieve information about a service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Application-Monitoring-Sections.html#ApplicationSignals-PreviewSDK
     */
    toGetService() {
        return this.to('GetService');
    }
    /**
     * Grants permission to retrieve service data
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/permissions-reference-cw.html
     */
    toGetServiceData() {
        return this.to('GetServiceData');
    }
    /**
     * Grants permission to retrieve information about service level objective
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Application-Monitoring-Sections.html#ApplicationSignals-PreviewSDK
     */
    toGetServiceLevelObjective() {
        return this.to('GetServiceLevelObjective');
    }
    /**
     * Grants permission to retrieve a CloudWatch topology discovery status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/permissions-reference-cw.html
     */
    toGetTopologyDiscoveryStatus() {
        return this.to('GetTopologyDiscoveryStatus');
    }
    /**
     * Grants permission to retrieve a CloudWatch topology map
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Application-Monitoring-Sections.html#ApplicationSignals-PreviewSDK
     */
    toGetTopologyMap() {
        return this.to('GetTopologyMap');
    }
    /**
     * Grants permission to share CloudWatch resources with a monitoring account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Unified-Cross-Account-Setup.html#CloudWatch-Unified-Cross-Account-Setup-permissions
     */
    toLink() {
        return this.to('Link');
    }
    /**
     * Grants permission to return a list of all CloudWatch dashboards in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_ListDashboards.html
     */
    toListDashboards() {
        return this.to('ListDashboards');
    }
    /**
     * Grants permission to retrieve all the entities that are emitting a given metric
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/permissions-reference-cw.html
     */
    toListEntitiesForMetric() {
        return this.to('ListEntitiesForMetric');
    }
    /**
     * Grants permission to list available managed Insight Rules for a given Resource ARN
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifRequestManagedResourceARNs()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_ListManagedInsightRules.html
     */
    toListManagedInsightRules() {
        return this.to('ListManagedInsightRules');
    }
    /**
     * Grants permission to return a list of all CloudWatch metric streams in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_ListMetricStreams.html
     */
    toListMetricStreams() {
        return this.to('ListMetricStreams');
    }
    /**
     * Grants permission to retrieve a list of valid metrics stored for the AWS account owner
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_ListMetrics.html
     */
    toListMetrics() {
        return this.to('ListMetrics');
    }
    /**
     * Grants permission to list service level objectives
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Application-Monitoring-Sections.html#ApplicationSignals-PreviewSDK
     */
    toListServiceLevelObjectives() {
        return this.to('ListServiceLevelObjectives');
    }
    /**
     * Grants permission to list services
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Application-Monitoring-Sections.html#ApplicationSignals-PreviewSDK
     */
    toListServices() {
        return this.to('ListServices');
    }
    /**
     * Grants permission to list tags for an Amazon CloudWatch resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to create or update an anomaly detection model for a CloudWatch metric
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_PutAnomalyDetector.html
     */
    toPutAnomalyDetector() {
        return this.to('PutAnomalyDetector');
    }
    /**
     * Grants permission to create or update a composite alarm
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAlarmActions()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_PutCompositeAlarm.html
     */
    toPutCompositeAlarm() {
        return this.to('PutCompositeAlarm');
    }
    /**
     * Grants permission to create a CloudWatch dashboard, or update an existing dashboard if it already exists
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_PutDashboard.html
     */
    toPutDashboard() {
        return this.to('PutDashboard');
    }
    /**
     * Grants permission to create a new insight rule or replace an existing insight rule
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifRequestInsightRuleLogGroups()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_PutInsightRule.html
     */
    toPutInsightRule() {
        return this.to('PutInsightRule');
    }
    /**
     * Grants permission to create managed Insight Rules
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifRequestManagedResourceARNs()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_PutManagedInsightRules.html
     */
    toPutManagedInsightRules() {
        return this.to('PutManagedInsightRules');
    }
    /**
     * Grants permission to create or update an alarm and associates it with the specified Amazon CloudWatch metric
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAlarmActions()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_PutMetricAlarm.html
     */
    toPutMetricAlarm() {
        return this.to('PutMetricAlarm');
    }
    /**
     * Grants permission to publish metric data points to Amazon CloudWatch
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifNamespace()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_PutMetricData.html
     */
    toPutMetricData() {
        return this.to('PutMetricData');
    }
    /**
     * Grants permission to create a CloudWatch metric stream, or update an existing metric stream if it already exists
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_PutMetricStream.html
     */
    toPutMetricStream() {
        return this.to('PutMetricStream');
    }
    /**
     * Grants permission to temporarily set the state of an alarm for testing purposes
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_SetAlarmState.html
     */
    toSetAlarmState() {
        return this.to('SetAlarmState');
    }
    /**
     * Grants permission to start all CloudWatch metric streams that you specify
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_StartMetricStreams.html
     */
    toStartMetricStreams() {
        return this.to('StartMetricStreams');
    }
    /**
     * Grants permission to stop all CloudWatch metric streams that you specify
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_StopMetricStreams.html
     */
    toStopMetricStreams() {
        return this.to('StopMetricStreams');
    }
    /**
     * Grants permission to add tags to an Amazon CloudWatch resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove a tag from an Amazon CloudWatch resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a service level objective
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Application-Monitoring-Sections.html#ApplicationSignals-PreviewSDK
     */
    toUpdateServiceLevelObjective() {
        return this.to('UpdateServiceLevelObjective');
    }
    /**
     * Adds a resource of type alarm to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/auth-and-access-control-cw.html
     *
     * @param alarmName - Identifier for the alarmName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAlarm(alarmName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudwatch:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:alarm:${alarmName}`);
    }
    /**
     * Adds a resource of type dashboard to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/auth-and-access-control-cw.html
     *
     * @param dashboardName - Identifier for the dashboardName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onDashboard(dashboardName, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudwatch::${account ?? this.defaultAccount}:dashboard/${dashboardName}`);
    }
    /**
     * Adds a resource of type insight-rule to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/auth-and-access-control-cw.html
     *
     * @param insightRuleName - Identifier for the insightRuleName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onInsightRule(insightRuleName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudwatch:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:insight-rule/${insightRuleName}`);
    }
    /**
     * Adds a resource of type metric-stream to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/auth-and-access-control-cw.html
     *
     * @param metricStreamName - Identifier for the metricStreamName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMetricStream(metricStreamName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudwatch:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:metric-stream/${metricStreamName}`);
    }
    /**
     * Adds a resource of type slo to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/auth-and-access-control-cw.html
     *
     * @param sloName - Identifier for the sloName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSlo(sloName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudwatch:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:slo/${sloName}`);
    }
    /**
     * Adds a resource of type service to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/auth-and-access-control-cw.html
     *
     * @param serviceName - Identifier for the serviceName.
     * @param uniqueAttributesHex - Identifier for the uniqueAttributesHex.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onService(serviceName, uniqueAttributesHex, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudwatch:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:service/${serviceName}-${uniqueAttributesHex}`);
    }
    /**
     * Filters actions based on the allowed set of values for each of the tags
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateServiceLevelObjective()
     * - .toListManagedInsightRules()
     * - .toPutCompositeAlarm()
     * - .toPutInsightRule()
     * - .toPutManagedInsightRules()
     * - .toPutMetricAlarm()
     * - .toPutMetricStream()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on tag-value associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - alarm
     * - insight-rule
     * - metric-stream
     * - slo
     * - service
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the presence of mandatory tags in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateServiceLevelObjective()
     * - .toListManagedInsightRules()
     * - .toPutCompositeAlarm()
     * - .toPutInsightRule()
     * - .toPutManagedInsightRules()
     * - .toPutMetricAlarm()
     * - .toPutMetricStream()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on defined alarm actions
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/iam-cw-condition-keys-alarm-actions.html
     *
     * Applies to actions:
     * - .toPutCompositeAlarm()
     * - .toPutMetricAlarm()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAlarmActions(value, operator) {
        return this.if(`AlarmActions`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the presence of optional namespace values
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/iam-cw-condition-keys-namespace.html
     *
     * Applies to actions:
     * - .toPutMetricData()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifNamespace(value, operator) {
        return this.if(`namespace`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the Log Groups specified in an Insight Rule
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/iam-cw-condition-keys-contributor.html
     *
     * Applies to actions:
     * - .toPutInsightRule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestInsightRuleLogGroups(value, operator) {
        return this.if(`requestInsightRuleLogGroups`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the Resource ARNs specified in a managed Insight Rule
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/iam-cw-condition-keys-contributor.html
     *
     * Applies to actions:
     * - .toListManagedInsightRules()
     * - .toPutManagedInsightRules()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifRequestManagedResourceARNs(value, operator) {
        return this.if(`requestManagedResourceARNs`, value, operator ?? 'ArnLike');
    }
    /**
     * Statement provider for service [cloudwatch](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncloudwatch.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'cloudwatch';
        this.accessLevelList = {
            Read: [
                'BatchGetServiceLevelIndicatorReport',
                'BatchGetServiceLevelObjectiveBudgetReport',
                'DescribeAlarmHistory',
                'DescribeAlarms',
                'DescribeAlarmsForMetric',
                'DescribeAnomalyDetectors',
                'DescribeInsightRules',
                'GenerateQuery',
                'GenerateQueryResultsSummary',
                'GetDashboard',
                'GetInsightRuleReport',
                'GetMetricData',
                'GetMetricStatistics',
                'GetMetricStream',
                'GetMetricWidgetImage',
                'GetService',
                'GetServiceData',
                'GetServiceLevelObjective',
                'GetTopologyDiscoveryStatus',
                'GetTopologyMap',
                'ListManagedInsightRules'
            ],
            Write: [
                'CreateServiceLevelObjective',
                'DeleteAlarms',
                'DeleteAnomalyDetector',
                'DeleteDashboards',
                'DeleteInsightRules',
                'DeleteMetricStream',
                'DeleteServiceLevelObjective',
                'DisableAlarmActions',
                'DisableInsightRules',
                'EnableAlarmActions',
                'EnableInsightRules',
                'EnableTopologyDiscovery',
                'Link',
                'PutAnomalyDetector',
                'PutCompositeAlarm',
                'PutDashboard',
                'PutInsightRule',
                'PutManagedInsightRules',
                'PutMetricAlarm',
                'PutMetricData',
                'PutMetricStream',
                'SetAlarmState',
                'StartMetricStreams',
                'StopMetricStreams',
                'UpdateServiceLevelObjective'
            ],
            List: [
                'ListDashboards',
                'ListEntitiesForMetric',
                'ListMetricStreams',
                'ListMetrics',
                'ListServiceLevelObjectives',
                'ListServices',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Cloudwatch = Cloudwatch;
//# sourceMappingURL=data:application/json;base64,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