"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Codeartifact = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [codeartifact](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodeartifact.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Codeartifact extends shared_1.PolicyStatement {
    /**
     * Grants permission to add an external connection to a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_AssociateExternalConnection.html
     */
    toAssociateExternalConnection() {
        return this.to('AssociateExternalConnection');
    }
    /**
     * Grants permission to associate an existing repository as an upstream repository to another repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/ug/repos-upstream.html
     */
    toAssociateWithDownstreamRepository() {
        return this.to('AssociateWithDownstreamRepository');
    }
    /**
     * Grants permission to copy package versions from one repository to another repository in the same domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_CopyPackageVersions.html
     */
    toCopyPackageVersions() {
        return this.to('CopyPackageVersions');
    }
    /**
     * Grants permission to create a new domain
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_CreateDomain.html
     */
    toCreateDomain() {
        return this.to('CreateDomain');
    }
    /**
     * Grants permission to create a package group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_CreatePackageGroup.html
     */
    toCreatePackageGroup() {
        return this.to('CreatePackageGroup');
    }
    /**
     * Grants permission to create a new repository
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_CreateRepository.html
     */
    toCreateRepository() {
        return this.to('CreateRepository');
    }
    /**
     * Grants permission to delete a domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeleteDomain.html
     */
    toDeleteDomain() {
        return this.to('DeleteDomain');
    }
    /**
     * Grants permission to delete the resource policy set on a domain
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeleteDomainPermissionsPolicy.html
     */
    toDeleteDomainPermissionsPolicy() {
        return this.to('DeleteDomainPermissionsPolicy');
    }
    /**
     * Grants permission to delete a package
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeletePackage.html
     */
    toDeletePackage() {
        return this.to('DeletePackage');
    }
    /**
     * Grants permission to delete a package group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeletePackageGroup.html
     */
    toDeletePackageGroup() {
        return this.to('DeletePackageGroup');
    }
    /**
     * Grants permission to delete package versions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeletePackageVersions.html
     */
    toDeletePackageVersions() {
        return this.to('DeletePackageVersions');
    }
    /**
     * Grants permission to delete a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeleteRepository.html
     */
    toDeleteRepository() {
        return this.to('DeleteRepository');
    }
    /**
     * Grants permission to delete the resource policy set on a repository
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeleteRepositoryPermissionsPolicy.html
     */
    toDeleteRepositoryPermissionsPolicy() {
        return this.to('DeleteRepositoryPermissionsPolicy');
    }
    /**
     * Grants permission to return information about a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DescribeDomain.html
     */
    toDescribeDomain() {
        return this.to('DescribeDomain');
    }
    /**
     * Grants permission to retrieve information about a package
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DescribePackage.html
     */
    toDescribePackage() {
        return this.to('DescribePackage');
    }
    /**
     * Grants permission to return detailed information about a package group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DescribePackageGroup.html
     */
    toDescribePackageGroup() {
        return this.to('DescribePackageGroup');
    }
    /**
     * Grants permission to return information about a package version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DescribePackageVersion.html
     */
    toDescribePackageVersion() {
        return this.to('DescribePackageVersion');
    }
    /**
     * Grants permission to return detailed information about a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DescribeRepository.html
     */
    toDescribeRepository() {
        return this.to('DescribeRepository');
    }
    /**
     * Grants permission to disassociate an external connection from a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DisassociateExternalConnection.html
     */
    toDisassociateExternalConnection() {
        return this.to('DisassociateExternalConnection');
    }
    /**
     * Grants permission to set the status of package versions to Disposed and delete their assets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DisposePackageVersions.html
     */
    toDisposePackageVersions() {
        return this.to('DisposePackageVersions');
    }
    /**
     * Grants permission to return a package's associated package group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetAssociatedPackageGroup.html
     */
    toGetAssociatedPackageGroup() {
        return this.to('GetAssociatedPackageGroup');
    }
    /**
     * Grants permission to generate a temporary authentication token for accessing repositories in a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetAuthorizationToken.html
     */
    toGetAuthorizationToken() {
        return this.to('GetAuthorizationToken');
    }
    /**
     * Grants permission to return a domain's resource policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetDomainPermissionsPolicy.html
     */
    toGetDomainPermissionsPolicy() {
        return this.to('GetDomainPermissionsPolicy');
    }
    /**
     * Grants permission to return an asset (or file) that is part of a package version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetPackageVersionAsset.html
     */
    toGetPackageVersionAsset() {
        return this.to('GetPackageVersionAsset');
    }
    /**
     * Grants permission to return a package version's readme file
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetPackageVersionReadme.html
     */
    toGetPackageVersionReadme() {
        return this.to('GetPackageVersionReadme');
    }
    /**
     * Grants permission to return an endpoint for a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetRepositoryEndpoint.html
     */
    toGetRepositoryEndpoint() {
        return this.to('GetRepositoryEndpoint');
    }
    /**
     * Grants permission to return a repository's resource policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetRepositoryPermissionsPolicy.html
     */
    toGetRepositoryPermissionsPolicy() {
        return this.to('GetRepositoryPermissionsPolicy');
    }
    /**
     * Grants permission to list the allowed repositories for a package group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListRepositoriesInDomain.html
     */
    toListAllowedRepositoriesForGroup() {
        return this.to('ListAllowedRepositoriesForGroup');
    }
    /**
     * Grants permission to list the packages associated to a package group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListAssociatedPackages.html
     */
    toListAssociatedPackages() {
        return this.to('ListAssociatedPackages');
    }
    /**
     * Grants permission to list the domains in the current user's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListDomains.html
     */
    toListDomains() {
        return this.to('ListDomains');
    }
    /**
     * Grants permission to list the package groups in a domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackageGroups.html
     */
    toListPackageGroups() {
        return this.to('ListPackageGroups');
    }
    /**
     * Grants permission to list a package version's assets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackageVersionAssets.html
     */
    toListPackageVersionAssets() {
        return this.to('ListPackageVersionAssets');
    }
    /**
     * Grants permission to list the direct dependencies of a package version
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackageVersionDependencies.html
     */
    toListPackageVersionDependencies() {
        return this.to('ListPackageVersionDependencies');
    }
    /**
     * Grants permission to list a package's versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackageVersions.html
     */
    toListPackageVersions() {
        return this.to('ListPackageVersions');
    }
    /**
     * Grants permission to list the packages in a repository
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackages.html
     */
    toListPackages() {
        return this.to('ListPackages');
    }
    /**
     * Grants permission to list the repositories administered by the calling account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListRepositories.html
     */
    toListRepositories() {
        return this.to('ListRepositories');
    }
    /**
     * Grants permission to list the repositories in a domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListRepositoriesInDomain.html
     */
    toListRepositoriesInDomain() {
        return this.to('ListRepositoriesInDomain');
    }
    /**
     * Grants permission to list the sub package groups for a parent package group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListSubPackageGroups.html
     */
    toListSubPackageGroups() {
        return this.to('ListSubPackageGroups');
    }
    /**
     * Grants permission to list tags for a CodeArtifact resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to publish assets and metadata to a repository endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/ug/repo-policies.html
     */
    toPublishPackageVersion() {
        return this.to('PublishPackageVersion');
    }
    /**
     * Grants permission to attach a resource policy to a domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PutDomainPermissionsPolicy.html
     */
    toPutDomainPermissionsPolicy() {
        return this.to('PutDomainPermissionsPolicy');
    }
    /**
     * Grants permission to add, modify or remove package metadata using a repository endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/ug/repo-policies.html
     */
    toPutPackageMetadata() {
        return this.to('PutPackageMetadata');
    }
    /**
     * Grants permission to set origin configuration for a package
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PutPackageOriginConfiguration.html
     */
    toPutPackageOriginConfiguration() {
        return this.to('PutPackageOriginConfiguration');
    }
    /**
     * Grants permission to attach a resource policy to a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PutRepositoryPermissionsPolicy.html
     */
    toPutRepositoryPermissionsPolicy() {
        return this.to('PutRepositoryPermissionsPolicy');
    }
    /**
     * Grants permission to return package assets and metadata from a repository endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/ug/repo-policies.html
     */
    toReadFromRepository() {
        return this.to('ReadFromRepository');
    }
    /**
     * Grants permission to tag a CodeArtifact resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove a tag from a CodeArtifact resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to modify the properties of a package group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_UpdatePackageGroup.html
     */
    toUpdatePackageGroup() {
        return this.to('UpdatePackageGroup');
    }
    /**
     * Grants permission to modify the package origin configuration of a package group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_UpdatePackageGroupOriginConfiguration.html
     */
    toUpdatePackageGroupOriginConfiguration() {
        return this.to('UpdatePackageGroupOriginConfiguration');
    }
    /**
     * Grants permission to modify the status of one or more versions of a package
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_UpdatePackageVersionsStatus.html
     */
    toUpdatePackageVersionsStatus() {
        return this.to('UpdatePackageVersionsStatus');
    }
    /**
     * Grants permission to modify the properties of a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_UpdateRepository.html
     */
    toUpdateRepository() {
        return this.to('UpdateRepository');
    }
    /**
     * Adds a resource of type domain to the statement
     *
     * https://docs.aws.amazon.com/codeartifact/latest/ug/domains.html
     *
     * @param domainName - Identifier for the domainName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDomain(domainName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codeartifact:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:domain/${domainName}`);
    }
    /**
     * Adds a resource of type repository to the statement
     *
     * https://docs.aws.amazon.com/codeartifact/latest/ug/repos.html
     *
     * @param domainName - Identifier for the domainName.
     * @param repositoryName - Identifier for the repositoryName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRepository(domainName, repositoryName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codeartifact:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:repository/${domainName}/${repositoryName}`);
    }
    /**
     * Adds a resource of type package-group to the statement
     *
     * https://docs.aws.amazon.com/codeartifact/latest/ug/package-groups.html
     *
     * @param domainName - Identifier for the domainName.
     * @param encodedPackageGroupPattern - Identifier for the encodedPackageGroupPattern.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPackageGroup(domainName, encodedPackageGroupPattern, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codeartifact:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:package-group/${domainName}${encodedPackageGroupPattern}`);
    }
    /**
     * Adds a resource of type package to the statement
     *
     * https://docs.aws.amazon.com/codeartifact/latest/ug/packages.html
     *
     * @param domainName - Identifier for the domainName.
     * @param repositoryName - Identifier for the repositoryName.
     * @param packageFormat - Identifier for the packageFormat.
     * @param packageNamespace - Identifier for the packageNamespace.
     * @param packageName - Identifier for the packageName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onPackage(domainName, repositoryName, packageFormat, packageNamespace, packageName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codeartifact:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:package/${domainName}/${repositoryName}/${packageFormat}/${packageNamespace}/${packageName}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateDomain()
     * - .toCreatePackageGroup()
     * - .toCreateRepository()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - domain
     * - repository
     * - package-group
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateDomain()
     * - .toCreatePackageGroup()
     * - .toCreateRepository()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [codeartifact](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodeartifact.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'codeartifact';
        this.accessLevelList = {
            Write: [
                'AssociateExternalConnection',
                'AssociateWithDownstreamRepository',
                'CopyPackageVersions',
                'CreateDomain',
                'CreatePackageGroup',
                'CreateRepository',
                'DeleteDomain',
                'DeletePackage',
                'DeletePackageGroup',
                'DeletePackageVersions',
                'DeleteRepository',
                'DisassociateExternalConnection',
                'DisposePackageVersions',
                'PublishPackageVersion',
                'PutDomainPermissionsPolicy',
                'PutPackageMetadata',
                'PutPackageOriginConfiguration',
                'PutRepositoryPermissionsPolicy',
                'UpdatePackageGroup',
                'UpdatePackageGroupOriginConfiguration',
                'UpdatePackageVersionsStatus',
                'UpdateRepository'
            ],
            'Permissions management': [
                'DeleteDomainPermissionsPolicy',
                'DeleteRepositoryPermissionsPolicy'
            ],
            Read: [
                'DescribeDomain',
                'DescribePackage',
                'DescribePackageGroup',
                'DescribePackageVersion',
                'DescribeRepository',
                'GetAssociatedPackageGroup',
                'GetAuthorizationToken',
                'GetDomainPermissionsPolicy',
                'GetPackageVersionAsset',
                'GetPackageVersionReadme',
                'GetRepositoryEndpoint',
                'GetRepositoryPermissionsPolicy',
                'ReadFromRepository'
            ],
            List: [
                'ListAllowedRepositoriesForGroup',
                'ListAssociatedPackages',
                'ListDomains',
                'ListPackageGroups',
                'ListPackageVersionAssets',
                'ListPackageVersionDependencies',
                'ListPackageVersions',
                'ListPackages',
                'ListRepositories',
                'ListRepositoriesInDomain',
                'ListSubPackageGroups',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Codeartifact = Codeartifact;
//# sourceMappingURL=data:application/json;base64,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