import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [codebuild](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodebuild.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class Codebuild extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to delete one or more builds
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchDeleteBuilds.html
     */
    toBatchDeleteBuilds(): this;
    /**
     * Grants permission to get information about one or more build batches
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetBuildBatches.html
     */
    toBatchGetBuildBatches(): this;
    /**
     * Grants permission to get information about one or more builds
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetBuilds.html
     */
    toBatchGetBuilds(): this;
    /**
     * Grants permission to get information about one or more command executions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetCommandExecutions.html
     */
    toBatchGetCommandExecutions(): this;
    /**
     * Grants permission to return an array of the Fleet objects specified by the input parameter
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetFleets.html
     */
    toBatchGetFleets(): this;
    /**
     * Grants permission to get information about one or more build projects
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetProjects.html
     */
    toBatchGetProjects(): this;
    /**
     * Grants permission to return an array of ReportGroup objects that are specified by the input reportGroupArns parameter
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetReportGroups.html
     */
    toBatchGetReportGroups(): this;
    /**
     * Grants permission to return an array of the Report objects specified by the input reportArns parameter
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetReports.html
     */
    toBatchGetReports(): this;
    /**
     * Grants permission to get information about one or more sandboxes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetSandboxes.html
     */
    toBatchGetSandboxes(): this;
    /**
     * Grants permission to add or update information about a report
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    toBatchPutCodeCoverages(): this;
    /**
     * Grants permission to add or update information about a report
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    toBatchPutTestCases(): this;
    /**
     * Grants permission to create a compute fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifImageId()
     * - .ifComputeType()
     * - .ifVpcConfig()
     * - .ifVpcConfigVpcId()
     * - .ifVpcConfigSecurityGroupIds()
     * - .ifVpcConfigSubnets()
     * - .ifComputeConfiguration()
     * - .ifComputeConfigurationDisk()
     * - .ifComputeConfigurationInstanceType()
     * - .ifComputeConfigurationMachineType()
     * - .ifComputeConfigurationMemory()
     * - .ifComputeConfigurationVCpu()
     * - .ifEnvironmentType()
     * - .ifFleetServiceRole()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_CreateFleet.html
     */
    toCreateFleet(): this;
    /**
     * Grants permission to create a build project
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAutoRetryLimit()
     * - .ifConcurrentBuildLimit()
     * - .ifArtifacts()
     * - .ifArtifactsBucketOwnerAccess()
     * - .ifArtifactsEncryptionDisabled()
     * - .ifArtifactsLocation()
     * - .ifSecondaryArtifacts()
     * - .ifSecondaryArtifactsArtifactIdentifier()
     * - .ifSecondaryArtifactsBucketOwnerAccess()
     * - .ifSecondaryArtifactsEncryptionDisabled()
     * - .ifSecondaryArtifactsLocation()
     * - .ifSecondaryArtifacts()
     * - .ifSecondaryArtifacts()
     * - .ifSecondaryArtifacts()
     * - .ifSource()
     * - .ifSourceBuildStatusConfigTargetUrl()
     * - .ifSourceBuildStatusConfigContext()
     * - .ifSourceLocation()
     * - .ifSourceInsecureSsl()
     * - .ifSourceBuildspec()
     * - .ifSourceAuthResource()
     * - .ifSourceAuthType()
     * - .ifSecondarySources()
     * - .ifSecondarySourcesSourceIdentifier()
     * - .ifSecondarySourcesBuildStatusConfigTargetUrl()
     * - .ifSecondarySourcesBuildStatusConfigContext()
     * - .ifSecondarySourcesLocation()
     * - .ifSecondarySourcesAuthResource()
     * - .ifSecondarySourcesAuthType()
     * - .ifSecondarySourcesBuildspec()
     * - .ifSecondarySourcesInsecureSsl()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifLogsConfig()
     * - .ifLogsConfigS3Logs()
     * - .ifLogsConfigS3LogsBucketOwnerAccess()
     * - .ifLogsConfigS3LogsEncryptionDisabled()
     * - .ifLogsConfigS3LogsLocation()
     * - .ifLogsConfigS3LogsStatus()
     * - .ifFileSystemLocationsIdentifier()
     * - .ifFileSystemLocationsType()
     * - .ifFileSystemLocationsLocation()
     * - .ifFileSystemLocations()
     * - .ifFileSystemLocations()
     * - .ifBuildBatchConfig()
     * - .ifBuildBatchConfigServiceRole()
     * - .ifBuildBatchConfigRestrictionsComputeTypesAllowed()
     * - .ifBuildBatchConfigRestrictionsFleetsAllowed()
     * - .ifVpcConfig()
     * - .ifVpcConfigSubnets()
     * - .ifVpcConfigVpcId()
     * - .ifVpcConfigSecurityGroupIds()
     * - .ifEnvironment()
     * - .ifEnvironmentType()
     * - .ifEnvironmentFleetFleetArn()
     * - .ifEnvironmentComputeType()
     * - .ifEnvironmentImage()
     * - .ifEnvironmentImagePullCredentialsType()
     * - .ifEnvironmentPrivilegedMode()
     * - .ifEnvironmentCertificate()
     * - .ifEnvironmentComputeConfiguration()
     * - .ifEnvironmentComputeConfigurationDisk()
     * - .ifEnvironmentComputeConfigurationInstanceType()
     * - .ifEnvironmentComputeConfigurationMachineType()
     * - .ifEnvironmentComputeConfigurationMemory()
     * - .ifEnvironmentComputeConfigurationVCpu()
     * - .ifEnvironmentEnvironmentVariables()
     * - .ifEnvironmentEnvironmentVariablesName()
     * - .ifEnvironmentEnvironmentVariablesValue()
     * - .ifEnvironmentEnvironmentVariables()
     * - .ifEnvironmentRegistryCredential()
     * - .ifEnvironmentRegistryCredentialCredential()
     * - .ifEnvironmentRegistryCredentialCredentialProvider()
     * - .ifEncryptionKey()
     * - .ifCache()
     * - .ifCacheType()
     * - .ifCacheLocation()
     * - .ifCacheModes()
     * - .ifServiceRole()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_CreateProject.html
     */
    toCreateProject(): this;
    /**
     * Grants permission to create a report. A report is created when tests specified in the buildspec file for a report groups run during the build of a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    toCreateReport(): this;
    /**
     * Grants permission to create a report group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifExportConfigS3DestinationBucket()
     * - .ifExportConfigS3DestinationBucketOwner()
     * - .ifExportConfigS3DestinationEncryptionKey()
     * - .ifExportConfigS3DestinationEncryptionDisabled()
     * - .ifExportConfigS3DestinationPath()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_CreateReportGroup.html
     */
    toCreateReportGroup(): this;
    /**
     * Grants permission to create webhook. For an existing AWS CodeBuild build project that has its source code stored in a GitHub or Bitbucket repository, enables AWS CodeBuild to start rebuilding the source code every time a code change is pushed to the repository
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifBuildType()
     * - .ifManualCreation()
     * - .ifScopeConfigurationDomain()
     * - .ifScopeConfigurationName()
     * - .ifScopeConfigurationScope()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_CreateWebhook.html
     */
    toCreateWebhook(): this;
    /**
     * Grants permission to delete a build batch
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteBuildBatch.html
     */
    toDeleteBuildBatch(): this;
    /**
     * Grants permission to delete a compute fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteFleet.html
     */
    toDeleteFleet(): this;
    /**
     * Grants permission to delete an OAuth token from a connected third-party OAuth provider. Only used in the AWS CodeBuild console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    toDeleteOAuthToken(): this;
    /**
     * Grants permission to delete a build project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteProject.html
     */
    toDeleteProject(): this;
    /**
     * Grants permission to delete a report
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteReport.html
     */
    toDeleteReport(): this;
    /**
     * Grants permission to delete a report group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteReportGroup.html
     */
    toDeleteReportGroup(): this;
    /**
     * Grants permission to delete a resource policy for the associated project or report group
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy(): this;
    /**
     * Grants permission to delete a set of GitHub, GitHub Enterprise, or Bitbucket source credentials
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteSourceCredentials.html
     */
    toDeleteSourceCredentials(): this;
    /**
     * Grants permission to delete webhook. For an existing AWS CodeBuild build project that has its source code stored in a GitHub or Bitbucket repository, stops AWS CodeBuild from rebuilding the source code every time a code change is pushed to the repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteWebhook.html
     */
    toDeleteWebhook(): this;
    /**
     * Grants permission to return an array of CodeCoverage objects
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DescribeCodeCoverages.html
     */
    toDescribeCodeCoverages(): this;
    /**
     * Grants permission to return an array of TestCase objects
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DescribeTestCases.html
     */
    toDescribeTestCases(): this;
    /**
     * Grants permission to analyze and accumulate test report values for the test reports in the specified report group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_GetReportGroupTrend.html
     */
    toGetReportGroupTrend(): this;
    /**
     * Grants permission to return a resource policy for the specified project or report group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_GetResourcePolicy.html
     */
    toGetResourcePolicy(): this;
    /**
     * Grants permission to import the source repository credentials for an AWS CodeBuild project that has its source code stored in a GitHub, GitHub Enterprise, or Bitbucket repository
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifServerType()
     * - .ifShouldOverwrite()
     * - .ifToken()
     * - .ifUsername()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ImportSourceCredentials.html
     */
    toImportSourceCredentials(): this;
    /**
     * Grants permission to reset the cache for a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_InvalidateProjectCache.html
     */
    toInvalidateProjectCache(): this;
    /**
     * Grants permission to get a list of build batch IDs, with each build batch ID representing a single build batch
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListBuildBatches.html
     */
    toListBuildBatches(): this;
    /**
     * Grants permission to get a list of build batch IDs for the specified build project, with each build batch ID representing a single build batch
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListBuildBatchesForProject.html
     */
    toListBuildBatchesForProject(): this;
    /**
     * Grants permission to get a list of build IDs, with each build ID representing a single build
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListBuilds.html
     */
    toListBuilds(): this;
    /**
     * Grants permission to get a list of build IDs for the specified build project, with each build ID representing a single build
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListBuildsForProject.html
     */
    toListBuildsForProject(): this;
    /**
     * Grants permission to get a list of command execution IDs for the specified sandbox, with each command execution ID representing a single command execution
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListCommandExecutionsForSandbox.html
     */
    toListCommandExecutionsForSandbox(): this;
    /**
     * Grants permission to list connected third-party OAuth providers. Only used in the AWS CodeBuild console
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    toListConnectedOAuthAccounts(): this;
    /**
     * Grants permission to get information about Docker images that are managed by AWS CodeBuild
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListCuratedEnvironmentImages.html
     */
    toListCuratedEnvironmentImages(): this;
    /**
     * Grants permission to get a list of compute fleet ARNs, with each compute fleet ARN representing a single fleet
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListFleets.html
     */
    toListFleets(): this;
    /**
     * Grants permission to get a list of build project names, with each build project name representing a single build project
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListProjects.html
     */
    toListProjects(): this;
    /**
     * Grants permission to return a list of report group ARNs. Each report group ARN represents one report group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListReportGroups.html
     */
    toListReportGroups(): this;
    /**
     * Grants permission to return a list of report ARNs. Each report ARN representing one report
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListReports.html
     */
    toListReports(): this;
    /**
     * Grants permission to return a list of report ARNs that belong to the specified report group. Each report ARN represents one report
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListReportsForReportGroup.html
     */
    toListReportsForReportGroup(): this;
    /**
     * Grants permission to list source code repositories from a connected third-party OAuth provider. Only used in the AWS CodeBuild console
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    toListRepositories(): this;
    /**
     * Grants permission to get a list of sandbox IDs, with each sandbox ID representing a single sandbox
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListSandboxes.html
     */
    toListSandboxes(): this;
    /**
     * Grants permission to get a list of sandbox IDs for the specified sandbox project, with each sandbox ID representing a single sandbox
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListSandboxesForProject.html
     */
    toListSandboxesForProject(): this;
    /**
     * Grants permission to return a list of project ARNs that have been shared with the requester. Each project ARN represents one project
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListSharedProjects.html
     */
    toListSharedProjects(): this;
    /**
     * Grants permission to return a list of report group ARNs that have been shared with the requester. Each report group ARN represents one report group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListSharedReportGroups.html
     */
    toListSharedReportGroups(): this;
    /**
     * Grants permission to return a list of SourceCredentialsInfo objects
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListSourceCredentials.html
     */
    toListSourceCredentials(): this;
    /**
     * Grants permission to save an OAuth token from a connected third-party OAuth provider. Only used in the AWS CodeBuild console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    toPersistOAuthToken(): this;
    /**
     * Grants permission to create a resource policy for the associated project or report group
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy(): this;
    /**
     * Grants permission to retry a build
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_RetryBuild.html
     */
    toRetryBuild(): this;
    /**
     * Grants permission to retry a build batch
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_RetryBuildBatch.html
     */
    toRetryBuildBatch(): this;
    /**
     * Grants permission to start running a build
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAutoRetryLimit()
     * - .ifArtifacts()
     * - .ifArtifactsBucketOwnerAccess()
     * - .ifArtifactsEncryptionDisabled()
     * - .ifArtifactsLocation()
     * - .ifSecondaryArtifacts()
     * - .ifSecondaryArtifactsArtifactIdentifier()
     * - .ifSecondaryArtifactsBucketOwnerAccess()
     * - .ifSecondaryArtifactsEncryptionDisabled()
     * - .ifSecondaryArtifactsLocation()
     * - .ifSecondaryArtifacts()
     * - .ifSecondaryArtifacts()
     * - .ifSecondaryArtifacts()
     * - .ifSource()
     * - .ifSourceBuildStatusConfigTargetUrl()
     * - .ifSourceBuildStatusConfigContext()
     * - .ifSourceLocation()
     * - .ifSourceInsecureSsl()
     * - .ifSourceBuildspec()
     * - .ifSourceAuthResource()
     * - .ifSourceAuthType()
     * - .ifSecondarySources()
     * - .ifSecondarySourcesSourceIdentifier()
     * - .ifSecondarySourcesBuildStatusConfigTargetUrl()
     * - .ifSecondarySourcesBuildStatusConfigContext()
     * - .ifSecondarySourcesLocation()
     * - .ifSecondarySourcesAuthResource()
     * - .ifSecondarySourcesAuthType()
     * - .ifSecondarySourcesBuildspec()
     * - .ifSecondarySourcesInsecureSsl()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifLogsConfig()
     * - .ifLogsConfigS3Logs()
     * - .ifLogsConfigS3LogsBucketOwnerAccess()
     * - .ifLogsConfigS3LogsEncryptionDisabled()
     * - .ifLogsConfigS3LogsLocation()
     * - .ifLogsConfigS3LogsStatus()
     * - .ifEnvironment()
     * - .ifEnvironmentType()
     * - .ifEnvironmentFleetFleetArn()
     * - .ifEnvironmentComputeType()
     * - .ifEnvironmentImage()
     * - .ifEnvironmentImagePullCredentialsType()
     * - .ifEnvironmentPrivilegedMode()
     * - .ifEnvironmentCertificate()
     * - .ifEnvironmentEnvironmentVariables()
     * - .ifEnvironmentEnvironmentVariablesName()
     * - .ifEnvironmentEnvironmentVariablesValue()
     * - .ifEnvironmentEnvironmentVariables()
     * - .ifEnvironmentRegistryCredential()
     * - .ifEnvironmentRegistryCredentialCredential()
     * - .ifEnvironmentRegistryCredentialCredentialProvider()
     * - .ifEncryptionKey()
     * - .ifCache()
     * - .ifCacheType()
     * - .ifCacheLocation()
     * - .ifCacheModes()
     * - .ifServiceRole()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartBuild.html
     */
    toStartBuild(): this;
    /**
     * Grants permission to start running a build batch
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifArtifacts()
     * - .ifArtifactsBucketOwnerAccess()
     * - .ifArtifactsEncryptionDisabled()
     * - .ifArtifactsLocation()
     * - .ifSecondaryArtifacts()
     * - .ifSecondaryArtifactsArtifactIdentifier()
     * - .ifSecondaryArtifactsBucketOwnerAccess()
     * - .ifSecondaryArtifactsEncryptionDisabled()
     * - .ifSecondaryArtifactsLocation()
     * - .ifSecondaryArtifacts()
     * - .ifSecondaryArtifacts()
     * - .ifSecondaryArtifacts()
     * - .ifSource()
     * - .ifSourceLocation()
     * - .ifSourceInsecureSsl()
     * - .ifSourceBuildspec()
     * - .ifSourceAuthResource()
     * - .ifSourceAuthType()
     * - .ifSecondarySources()
     * - .ifSecondarySourcesSourceIdentifier()
     * - .ifSecondarySourcesBuildStatusConfigTargetUrl()
     * - .ifSecondarySourcesBuildStatusConfigContext()
     * - .ifSecondarySourcesLocation()
     * - .ifSecondarySourcesAuthResource()
     * - .ifSecondarySourcesAuthType()
     * - .ifSecondarySourcesBuildspec()
     * - .ifSecondarySourcesInsecureSsl()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifLogsConfig()
     * - .ifLogsConfigS3Logs()
     * - .ifLogsConfigS3LogsBucketOwnerAccess()
     * - .ifLogsConfigS3LogsEncryptionDisabled()
     * - .ifLogsConfigS3LogsLocation()
     * - .ifLogsConfigS3LogsStatus()
     * - .ifBuildBatchConfig()
     * - .ifBuildBatchConfigServiceRole()
     * - .ifBuildBatchConfigRestrictionsComputeTypesAllowed()
     * - .ifBuildBatchConfigRestrictionsFleetsAllowed()
     * - .ifEnvironment()
     * - .ifEnvironmentType()
     * - .ifEnvironmentComputeType()
     * - .ifEnvironmentImage()
     * - .ifEnvironmentImagePullCredentialsType()
     * - .ifEnvironmentPrivilegedMode()
     * - .ifEnvironmentCertificate()
     * - .ifEnvironmentEnvironmentVariables()
     * - .ifEnvironmentEnvironmentVariablesName()
     * - .ifEnvironmentEnvironmentVariablesValue()
     * - .ifEnvironmentEnvironmentVariables()
     * - .ifEnvironmentRegistryCredential()
     * - .ifEnvironmentRegistryCredentialCredential()
     * - .ifEnvironmentRegistryCredentialCredentialProvider()
     * - .ifEncryptionKey()
     * - .ifCache()
     * - .ifCacheType()
     * - .ifCacheLocation()
     * - .ifCacheModes()
     * - .ifServiceRole()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartBuildBatch.html
     */
    toStartBuildBatch(): this;
    /**
     * Grants permission to start running a command execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartCommandExecution.html
     */
    toStartCommandExecution(): this;
    /**
     * Grants permission to start running a sandbox
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartSandbox.html
     */
    toStartSandbox(): this;
    /**
     * Grants permission to establish a connection to the sandbox
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartSandboxConnection.html
     */
    toStartSandboxConnection(): this;
    /**
     * Grants permission to attempt to stop running a build
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StopBuild.html
     */
    toStopBuild(): this;
    /**
     * Grants permission to attempt to stop running a build batch
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StopBuildBatch.html
     */
    toStopBuildBatch(): this;
    /**
     * Grants permission to attempt to stop running a sandbox
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StopSandbox.html
     */
    toStopSandbox(): this;
    /**
     * Grants permission to change the settings of an existing compute fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifImageId()
     * - .ifComputeType()
     * - .ifVpcConfig()
     * - .ifVpcConfigVpcId()
     * - .ifVpcConfigSecurityGroupIds()
     * - .ifVpcConfigSubnets()
     * - .ifComputeConfiguration()
     * - .ifComputeConfigurationDisk()
     * - .ifComputeConfigurationInstanceType()
     * - .ifComputeConfigurationMachineType()
     * - .ifComputeConfigurationMemory()
     * - .ifComputeConfigurationVCpu()
     * - .ifEnvironmentType()
     * - .ifFleetServiceRole()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_UpdateFleet.html
     */
    toUpdateFleet(): this;
    /**
     * Grants permission to change the settings of an existing build project
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAutoRetryLimit()
     * - .ifConcurrentBuildLimit()
     * - .ifArtifacts()
     * - .ifArtifactsBucketOwnerAccess()
     * - .ifArtifactsEncryptionDisabled()
     * - .ifArtifactsLocation()
     * - .ifSecondaryArtifacts()
     * - .ifSecondaryArtifactsArtifactIdentifier()
     * - .ifSecondaryArtifactsBucketOwnerAccess()
     * - .ifSecondaryArtifactsEncryptionDisabled()
     * - .ifSecondaryArtifactsLocation()
     * - .ifSecondaryArtifacts()
     * - .ifSecondaryArtifacts()
     * - .ifSecondaryArtifacts()
     * - .ifSource()
     * - .ifSourceBuildStatusConfigTargetUrl()
     * - .ifSourceBuildStatusConfigContext()
     * - .ifSourceLocation()
     * - .ifSourceInsecureSsl()
     * - .ifSourceBuildspec()
     * - .ifSourceAuthResource()
     * - .ifSourceAuthType()
     * - .ifSecondarySources()
     * - .ifSecondarySourcesSourceIdentifier()
     * - .ifSecondarySourcesBuildStatusConfigTargetUrl()
     * - .ifSecondarySourcesBuildStatusConfigContext()
     * - .ifSecondarySourcesLocation()
     * - .ifSecondarySourcesAuthResource()
     * - .ifSecondarySourcesAuthType()
     * - .ifSecondarySourcesBuildspec()
     * - .ifSecondarySourcesInsecureSsl()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifLogsConfig()
     * - .ifLogsConfigS3Logs()
     * - .ifLogsConfigS3LogsBucketOwnerAccess()
     * - .ifLogsConfigS3LogsEncryptionDisabled()
     * - .ifLogsConfigS3LogsLocation()
     * - .ifLogsConfigS3LogsStatus()
     * - .ifFileSystemLocationsIdentifier()
     * - .ifFileSystemLocationsType()
     * - .ifFileSystemLocationsLocation()
     * - .ifFileSystemLocations()
     * - .ifFileSystemLocations()
     * - .ifBuildBatchConfig()
     * - .ifBuildBatchConfigServiceRole()
     * - .ifBuildBatchConfigRestrictionsComputeTypesAllowed()
     * - .ifBuildBatchConfigRestrictionsFleetsAllowed()
     * - .ifVpcConfig()
     * - .ifVpcConfigSubnets()
     * - .ifVpcConfigVpcId()
     * - .ifVpcConfigSecurityGroupIds()
     * - .ifEnvironment()
     * - .ifEnvironmentType()
     * - .ifEnvironmentFleetFleetArn()
     * - .ifEnvironmentComputeType()
     * - .ifEnvironmentImage()
     * - .ifEnvironmentImagePullCredentialsType()
     * - .ifEnvironmentPrivilegedMode()
     * - .ifEnvironmentCertificate()
     * - .ifEnvironmentComputeConfiguration()
     * - .ifEnvironmentComputeConfigurationDisk()
     * - .ifEnvironmentComputeConfigurationInstanceType()
     * - .ifEnvironmentComputeConfigurationMachineType()
     * - .ifEnvironmentComputeConfigurationMemory()
     * - .ifEnvironmentComputeConfigurationVCpu()
     * - .ifEnvironmentEnvironmentVariables()
     * - .ifEnvironmentEnvironmentVariablesName()
     * - .ifEnvironmentEnvironmentVariablesValue()
     * - .ifEnvironmentEnvironmentVariables()
     * - .ifEnvironmentRegistryCredential()
     * - .ifEnvironmentRegistryCredentialCredential()
     * - .ifEnvironmentRegistryCredentialCredentialProvider()
     * - .ifEncryptionKey()
     * - .ifCache()
     * - .ifCacheType()
     * - .ifCacheLocation()
     * - .ifCacheModes()
     * - .ifServiceRole()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_UpdateProject.html
     */
    toUpdateProject(): this;
    /**
     * Grants permission to change the public visibility of a project and its builds
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifProjectVisibility()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_UpdateProjectVisibility.html
     */
    toUpdateProjectVisibility(): this;
    /**
     * Grants permission to update information about a report
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    toUpdateReport(): this;
    /**
     * Grants permission to change the settings of an existing report group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifExportConfigS3DestinationBucket()
     * - .ifExportConfigS3DestinationBucketOwner()
     * - .ifExportConfigS3DestinationEncryptionKey()
     * - .ifExportConfigS3DestinationEncryptionDisabled()
     * - .ifExportConfigS3DestinationPath()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_UpdateReportGroup.html
     */
    toUpdateReportGroup(): this;
    /**
     * Grants permission to update the webhook associated with an AWS CodeBuild build project
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifBuildType()
     * - .ifManualCreation()
     * - .ifScopeConfigurationDomain()
     * - .ifScopeConfigurationName()
     * - .ifScopeConfigurationScope()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_UpdateWebhook.html
     */
    toUpdateWebhook(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type build to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param buildId - Identifier for the buildId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onBuild(buildId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type build-batch to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param buildBatchId - Identifier for the buildBatchId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onBuildBatch(buildBatchId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type project to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param projectName - Identifier for the projectName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProject(projectName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type report-group to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param reportGroupName - Identifier for the reportGroupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onReportGroup(reportGroupName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type report to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param reportGroupName - Identifier for the reportGroupName.
     * @param reportId - Identifier for the reportId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onReport(reportGroupName: string, reportId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type fleet to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param fleetName - Identifier for the fleetName.
     * @param fleetId - Identifier for the fleetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onFleet(fleetName: string, fleetId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type sandbox to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param sandboxId - Identifier for the sandboxId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onSandbox(sandboxId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toCreateProject()
     * - .toCreateReportGroup()
     * - .toUpdateFleet()
     * - .toUpdateProject()
     * - .toUpdateProjectVisibility()
     * - .toUpdateReportGroup()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - project
     * - report-group
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toCreateProject()
     * - .toCreateReportGroup()
     * - .toUpdateFleet()
     * - .toUpdateProject()
     * - .toUpdateProjectVisibility()
     * - .toUpdateReportGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifArtifacts(value?: boolean): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifArtifactsBucketOwnerAccess(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifArtifactsEncryptionDisabled(value?: boolean): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifArtifactsLocation(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toImportSourceCredentials()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifAutoRetryLimit(value: number | number[], operator?: Operator | string): this;
    /**
     * Filters access by the ARN of the AWS CodeBuild build from which the request originated
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-permissions-reference.html
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifBuildArn(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifBuildBatchConfig(value?: boolean): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifBuildBatchConfigRestrictionsComputeTypesAllowed(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifBuildBatchConfigRestrictionsFleetsAllowed(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifBuildBatchConfigServiceRole(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateWebhook()
     * - .toUpdateWebhook()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifBuildType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifCache(value?: boolean): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCacheLocation(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCacheModes(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCacheType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toUpdateFleet()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifComputeConfiguration(value?: boolean): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toUpdateFleet()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifComputeConfigurationDisk(value: number | number[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toUpdateFleet()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifComputeConfigurationInstanceType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toUpdateFleet()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifComputeConfigurationMachineType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toUpdateFleet()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifComputeConfigurationMemory(value: number | number[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toUpdateFleet()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifComputeConfigurationVCpu(value: number | number[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toUpdateFleet()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifComputeType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifConcurrentBuildLimit(value: number | number[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEncryptionKey(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifEnvironment(value?: boolean): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentCertificate(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifEnvironmentComputeConfiguration(value?: boolean): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifEnvironmentComputeConfigurationDisk(value: number | number[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentComputeConfigurationInstanceType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentComputeConfigurationMachineType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifEnvironmentComputeConfigurationMemory(value: number | number[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifEnvironmentComputeConfigurationVCpu(value: number | number[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentComputeType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifEnvironmentEnvironmentVariables(value?: boolean): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentEnvironmentVariablesName(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentEnvironmentVariablesValue(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param namevalue The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentEnvironmentVariablesNamevalue(namevalue: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifEnvironmentFleetFleetArn(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentImage(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentImagePullCredentialsType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifEnvironmentPrivilegedMode(value?: boolean): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifEnvironmentRegistryCredential(value?: boolean): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentRegistryCredentialCredential(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentRegistryCredentialCredentialProvider(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toUpdateFleet()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFleetEnvironmentType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateReportGroup()
     * - .toUpdateReportGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExportConfigS3DestinationBucket(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateReportGroup()
     * - .toUpdateReportGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExportConfigS3DestinationBucketOwner(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateReportGroup()
     * - .toUpdateReportGroup()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifExportConfigS3DestinationEncryptionDisabled(value?: boolean): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateReportGroup()
     * - .toUpdateReportGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExportConfigS3DestinationEncryptionKey(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateReportGroup()
     * - .toUpdateReportGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExportConfigS3DestinationPath(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFileSystemLocationsIdentifier(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFileSystemLocationsLocation(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFileSystemLocationsType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toUpdateProject()
     *
     * @param identifierlocation The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFileSystemLocationsIdentifierlocation(identifierlocation: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toUpdateProject()
     *
     * @param identifiertype The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFileSystemLocationsIdentifiertype(identifiertype: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toUpdateFleet()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFleetServiceRole(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toUpdateFleet()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifImageId(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifLogsConfig(value?: boolean): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifLogsConfigS3Logs(value?: boolean): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLogsConfigS3LogsBucketOwnerAccess(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifLogsConfigS3LogsEncryptionDisabled(value?: boolean): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLogsConfigS3LogsLocation(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLogsConfigS3LogsStatus(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateWebhook()
     * - .toUpdateWebhook()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifManualCreation(value?: boolean): this;
    /**
     * Filters access by the ARN of the AWS CodeBuild project from which the request originated
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-permissions-reference.html
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifProjectArn(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toUpdateProjectVisibility()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProjectVisibility(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateWebhook()
     * - .toUpdateWebhook()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifScopeConfigurationDomain(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateWebhook()
     * - .toUpdateWebhook()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifScopeConfigurationName(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateWebhook()
     * - .toUpdateWebhook()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifScopeConfigurationScope(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifSecondaryArtifacts(value?: boolean): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondaryArtifactsArtifactIdentifier(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondaryArtifactsBucketOwnerAccess(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifSecondaryArtifactsEncryptionDisabled(value?: boolean): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondaryArtifactsLocation(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param artifactIdentifierbucketOwnerAccess The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondaryArtifactsArtifactIdentifierbucketOwnerAccess(artifactIdentifierbucketOwnerAccess: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param artifactIdentifierencryptionDisabled The tag key to check
     * @param value `true` or `false`. **Default:** `true`
     */
    ifSecondaryArtifactsArtifactIdentifierencryptionDisabled(artifactIdentifierencryptionDisabled: string, value?: boolean): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param artifactIdentifierlocation The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondaryArtifactsArtifactIdentifierlocation(artifactIdentifierlocation: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifSecondarySources(value?: boolean): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondarySourcesAuthResource(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondarySourcesAuthType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondarySourcesBuildStatusConfigContext(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondarySourcesBuildStatusConfigTargetUrl(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifSecondarySourcesBuildspec(value?: boolean): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifSecondarySourcesInsecureSsl(value?: boolean): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondarySourcesLocation(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondarySourcesSourceIdentifier(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param sourceIdentifierauthresource The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondarySourcesSourceIdentifierauthresource(sourceIdentifierauthresource: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param sourceIdentifierauthtype The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondarySourcesSourceIdentifierauthtype(sourceIdentifierauthtype: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param sourceIdentifierbuildStatusConfigcontext The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondarySourcesSourceIdentifierbuildStatusConfigcontext(sourceIdentifierbuildStatusConfigcontext: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param sourceIdentifierbuildStatusConfigtargetUrl The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondarySourcesSourceIdentifierbuildStatusConfigtargetUrl(sourceIdentifierbuildStatusConfigtargetUrl: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param sourceIdentifierbuildspec The tag key to check
     * @param value `true` or `false`. **Default:** `true`
     */
    ifSecondarySourcesSourceIdentifierbuildspec(sourceIdentifierbuildspec: string, value?: boolean): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param sourceIdentifierinsecureSsl The tag key to check
     * @param value `true` or `false`. **Default:** `true`
     */
    ifSecondarySourcesSourceIdentifierinsecureSsl(sourceIdentifierinsecureSsl: string, value?: boolean): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param sourceIdentifierlocation The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondarySourcesSourceIdentifierlocation(sourceIdentifierlocation: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toImportSourceCredentials()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifServerType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifServiceRole(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toImportSourceCredentials()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifShouldOverwrite(value?: boolean): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifSource(value?: boolean): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSourceAuthResource(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSourceAuthType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSourceBuildStatusConfigContext(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSourceBuildStatusConfigTargetUrl(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifSourceBuildspec(value?: boolean): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifSourceInsecureSsl(value?: boolean): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSourceLocation(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toImportSourceCredentials()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifToken(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toImportSourceCredentials()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUsername(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toCreateProject()
     * - .toUpdateFleet()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifVpcConfig(value?: boolean): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toCreateProject()
     * - .toUpdateFleet()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpcConfigSecurityGroupIds(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toCreateProject()
     * - .toUpdateFleet()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpcConfigSubnets(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toCreateProject()
     * - .toUpdateFleet()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpcConfigVpcId(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [codebuild](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodebuild.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
