"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Codebuild = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [codebuild](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodebuild.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Codebuild extends shared_1.PolicyStatement {
    /**
     * Grants permission to delete one or more builds
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchDeleteBuilds.html
     */
    toBatchDeleteBuilds() {
        return this.to('BatchDeleteBuilds');
    }
    /**
     * Grants permission to get information about one or more build batches
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetBuildBatches.html
     */
    toBatchGetBuildBatches() {
        return this.to('BatchGetBuildBatches');
    }
    /**
     * Grants permission to get information about one or more builds
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetBuilds.html
     */
    toBatchGetBuilds() {
        return this.to('BatchGetBuilds');
    }
    /**
     * Grants permission to get information about one or more command executions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetCommandExecutions.html
     */
    toBatchGetCommandExecutions() {
        return this.to('BatchGetCommandExecutions');
    }
    /**
     * Grants permission to return an array of the Fleet objects specified by the input parameter
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetFleets.html
     */
    toBatchGetFleets() {
        return this.to('BatchGetFleets');
    }
    /**
     * Grants permission to get information about one or more build projects
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetProjects.html
     */
    toBatchGetProjects() {
        return this.to('BatchGetProjects');
    }
    /**
     * Grants permission to return an array of ReportGroup objects that are specified by the input reportGroupArns parameter
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetReportGroups.html
     */
    toBatchGetReportGroups() {
        return this.to('BatchGetReportGroups');
    }
    /**
     * Grants permission to return an array of the Report objects specified by the input reportArns parameter
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetReports.html
     */
    toBatchGetReports() {
        return this.to('BatchGetReports');
    }
    /**
     * Grants permission to get information about one or more sandboxes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetSandboxes.html
     */
    toBatchGetSandboxes() {
        return this.to('BatchGetSandboxes');
    }
    /**
     * Grants permission to add or update information about a report
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    toBatchPutCodeCoverages() {
        return this.to('BatchPutCodeCoverages');
    }
    /**
     * Grants permission to add or update information about a report
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    toBatchPutTestCases() {
        return this.to('BatchPutTestCases');
    }
    /**
     * Grants permission to create a compute fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifImageId()
     * - .ifComputeType()
     * - .ifVpcConfig()
     * - .ifVpcConfigVpcId()
     * - .ifVpcConfigSecurityGroupIds()
     * - .ifVpcConfigSubnets()
     * - .ifComputeConfiguration()
     * - .ifComputeConfigurationDisk()
     * - .ifComputeConfigurationInstanceType()
     * - .ifComputeConfigurationMachineType()
     * - .ifComputeConfigurationMemory()
     * - .ifComputeConfigurationVCpu()
     * - .ifEnvironmentType()
     * - .ifFleetServiceRole()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_CreateFleet.html
     */
    toCreateFleet() {
        return this.to('CreateFleet');
    }
    /**
     * Grants permission to create a build project
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAutoRetryLimit()
     * - .ifConcurrentBuildLimit()
     * - .ifArtifacts()
     * - .ifArtifactsBucketOwnerAccess()
     * - .ifArtifactsEncryptionDisabled()
     * - .ifArtifactsLocation()
     * - .ifSecondaryArtifacts()
     * - .ifSecondaryArtifactsArtifactIdentifier()
     * - .ifSecondaryArtifactsBucketOwnerAccess()
     * - .ifSecondaryArtifactsEncryptionDisabled()
     * - .ifSecondaryArtifactsLocation()
     * - .ifSecondaryArtifacts()
     * - .ifSecondaryArtifacts()
     * - .ifSecondaryArtifacts()
     * - .ifSource()
     * - .ifSourceBuildStatusConfigTargetUrl()
     * - .ifSourceBuildStatusConfigContext()
     * - .ifSourceLocation()
     * - .ifSourceInsecureSsl()
     * - .ifSourceBuildspec()
     * - .ifSourceAuthResource()
     * - .ifSourceAuthType()
     * - .ifSecondarySources()
     * - .ifSecondarySourcesSourceIdentifier()
     * - .ifSecondarySourcesBuildStatusConfigTargetUrl()
     * - .ifSecondarySourcesBuildStatusConfigContext()
     * - .ifSecondarySourcesLocation()
     * - .ifSecondarySourcesAuthResource()
     * - .ifSecondarySourcesAuthType()
     * - .ifSecondarySourcesBuildspec()
     * - .ifSecondarySourcesInsecureSsl()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifLogsConfig()
     * - .ifLogsConfigS3Logs()
     * - .ifLogsConfigS3LogsBucketOwnerAccess()
     * - .ifLogsConfigS3LogsEncryptionDisabled()
     * - .ifLogsConfigS3LogsLocation()
     * - .ifLogsConfigS3LogsStatus()
     * - .ifFileSystemLocationsIdentifier()
     * - .ifFileSystemLocationsType()
     * - .ifFileSystemLocationsLocation()
     * - .ifFileSystemLocations()
     * - .ifFileSystemLocations()
     * - .ifBuildBatchConfig()
     * - .ifBuildBatchConfigServiceRole()
     * - .ifBuildBatchConfigRestrictionsComputeTypesAllowed()
     * - .ifBuildBatchConfigRestrictionsFleetsAllowed()
     * - .ifVpcConfig()
     * - .ifVpcConfigSubnets()
     * - .ifVpcConfigVpcId()
     * - .ifVpcConfigSecurityGroupIds()
     * - .ifEnvironment()
     * - .ifEnvironmentType()
     * - .ifEnvironmentFleetFleetArn()
     * - .ifEnvironmentComputeType()
     * - .ifEnvironmentImage()
     * - .ifEnvironmentImagePullCredentialsType()
     * - .ifEnvironmentPrivilegedMode()
     * - .ifEnvironmentCertificate()
     * - .ifEnvironmentComputeConfiguration()
     * - .ifEnvironmentComputeConfigurationDisk()
     * - .ifEnvironmentComputeConfigurationInstanceType()
     * - .ifEnvironmentComputeConfigurationMachineType()
     * - .ifEnvironmentComputeConfigurationMemory()
     * - .ifEnvironmentComputeConfigurationVCpu()
     * - .ifEnvironmentEnvironmentVariables()
     * - .ifEnvironmentEnvironmentVariablesName()
     * - .ifEnvironmentEnvironmentVariablesValue()
     * - .ifEnvironmentEnvironmentVariables()
     * - .ifEnvironmentRegistryCredential()
     * - .ifEnvironmentRegistryCredentialCredential()
     * - .ifEnvironmentRegistryCredentialCredentialProvider()
     * - .ifEncryptionKey()
     * - .ifCache()
     * - .ifCacheType()
     * - .ifCacheLocation()
     * - .ifCacheModes()
     * - .ifServiceRole()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_CreateProject.html
     */
    toCreateProject() {
        return this.to('CreateProject');
    }
    /**
     * Grants permission to create a report. A report is created when tests specified in the buildspec file for a report groups run during the build of a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    toCreateReport() {
        return this.to('CreateReport');
    }
    /**
     * Grants permission to create a report group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifExportConfigS3DestinationBucket()
     * - .ifExportConfigS3DestinationBucketOwner()
     * - .ifExportConfigS3DestinationEncryptionKey()
     * - .ifExportConfigS3DestinationEncryptionDisabled()
     * - .ifExportConfigS3DestinationPath()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_CreateReportGroup.html
     */
    toCreateReportGroup() {
        return this.to('CreateReportGroup');
    }
    /**
     * Grants permission to create webhook. For an existing AWS CodeBuild build project that has its source code stored in a GitHub or Bitbucket repository, enables AWS CodeBuild to start rebuilding the source code every time a code change is pushed to the repository
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifBuildType()
     * - .ifManualCreation()
     * - .ifScopeConfigurationDomain()
     * - .ifScopeConfigurationName()
     * - .ifScopeConfigurationScope()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_CreateWebhook.html
     */
    toCreateWebhook() {
        return this.to('CreateWebhook');
    }
    /**
     * Grants permission to delete a build batch
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteBuildBatch.html
     */
    toDeleteBuildBatch() {
        return this.to('DeleteBuildBatch');
    }
    /**
     * Grants permission to delete a compute fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteFleet.html
     */
    toDeleteFleet() {
        return this.to('DeleteFleet');
    }
    /**
     * Grants permission to delete an OAuth token from a connected third-party OAuth provider. Only used in the AWS CodeBuild console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    toDeleteOAuthToken() {
        return this.to('DeleteOAuthToken');
    }
    /**
     * Grants permission to delete a build project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteProject.html
     */
    toDeleteProject() {
        return this.to('DeleteProject');
    }
    /**
     * Grants permission to delete a report
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteReport.html
     */
    toDeleteReport() {
        return this.to('DeleteReport');
    }
    /**
     * Grants permission to delete a report group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteReportGroup.html
     */
    toDeleteReportGroup() {
        return this.to('DeleteReportGroup');
    }
    /**
     * Grants permission to delete a resource policy for the associated project or report group
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete a set of GitHub, GitHub Enterprise, or Bitbucket source credentials
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteSourceCredentials.html
     */
    toDeleteSourceCredentials() {
        return this.to('DeleteSourceCredentials');
    }
    /**
     * Grants permission to delete webhook. For an existing AWS CodeBuild build project that has its source code stored in a GitHub or Bitbucket repository, stops AWS CodeBuild from rebuilding the source code every time a code change is pushed to the repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteWebhook.html
     */
    toDeleteWebhook() {
        return this.to('DeleteWebhook');
    }
    /**
     * Grants permission to return an array of CodeCoverage objects
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DescribeCodeCoverages.html
     */
    toDescribeCodeCoverages() {
        return this.to('DescribeCodeCoverages');
    }
    /**
     * Grants permission to return an array of TestCase objects
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DescribeTestCases.html
     */
    toDescribeTestCases() {
        return this.to('DescribeTestCases');
    }
    /**
     * Grants permission to analyze and accumulate test report values for the test reports in the specified report group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_GetReportGroupTrend.html
     */
    toGetReportGroupTrend() {
        return this.to('GetReportGroupTrend');
    }
    /**
     * Grants permission to return a resource policy for the specified project or report group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_GetResourcePolicy.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to import the source repository credentials for an AWS CodeBuild project that has its source code stored in a GitHub, GitHub Enterprise, or Bitbucket repository
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifServerType()
     * - .ifShouldOverwrite()
     * - .ifToken()
     * - .ifUsername()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ImportSourceCredentials.html
     */
    toImportSourceCredentials() {
        return this.to('ImportSourceCredentials');
    }
    /**
     * Grants permission to reset the cache for a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_InvalidateProjectCache.html
     */
    toInvalidateProjectCache() {
        return this.to('InvalidateProjectCache');
    }
    /**
     * Grants permission to get a list of build batch IDs, with each build batch ID representing a single build batch
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListBuildBatches.html
     */
    toListBuildBatches() {
        return this.to('ListBuildBatches');
    }
    /**
     * Grants permission to get a list of build batch IDs for the specified build project, with each build batch ID representing a single build batch
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListBuildBatchesForProject.html
     */
    toListBuildBatchesForProject() {
        return this.to('ListBuildBatchesForProject');
    }
    /**
     * Grants permission to get a list of build IDs, with each build ID representing a single build
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListBuilds.html
     */
    toListBuilds() {
        return this.to('ListBuilds');
    }
    /**
     * Grants permission to get a list of build IDs for the specified build project, with each build ID representing a single build
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListBuildsForProject.html
     */
    toListBuildsForProject() {
        return this.to('ListBuildsForProject');
    }
    /**
     * Grants permission to get a list of command execution IDs for the specified sandbox, with each command execution ID representing a single command execution
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListCommandExecutionsForSandbox.html
     */
    toListCommandExecutionsForSandbox() {
        return this.to('ListCommandExecutionsForSandbox');
    }
    /**
     * Grants permission to list connected third-party OAuth providers. Only used in the AWS CodeBuild console
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    toListConnectedOAuthAccounts() {
        return this.to('ListConnectedOAuthAccounts');
    }
    /**
     * Grants permission to get information about Docker images that are managed by AWS CodeBuild
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListCuratedEnvironmentImages.html
     */
    toListCuratedEnvironmentImages() {
        return this.to('ListCuratedEnvironmentImages');
    }
    /**
     * Grants permission to get a list of compute fleet ARNs, with each compute fleet ARN representing a single fleet
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListFleets.html
     */
    toListFleets() {
        return this.to('ListFleets');
    }
    /**
     * Grants permission to get a list of build project names, with each build project name representing a single build project
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListProjects.html
     */
    toListProjects() {
        return this.to('ListProjects');
    }
    /**
     * Grants permission to return a list of report group ARNs. Each report group ARN represents one report group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListReportGroups.html
     */
    toListReportGroups() {
        return this.to('ListReportGroups');
    }
    /**
     * Grants permission to return a list of report ARNs. Each report ARN representing one report
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListReports.html
     */
    toListReports() {
        return this.to('ListReports');
    }
    /**
     * Grants permission to return a list of report ARNs that belong to the specified report group. Each report ARN represents one report
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListReportsForReportGroup.html
     */
    toListReportsForReportGroup() {
        return this.to('ListReportsForReportGroup');
    }
    /**
     * Grants permission to list source code repositories from a connected third-party OAuth provider. Only used in the AWS CodeBuild console
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    toListRepositories() {
        return this.to('ListRepositories');
    }
    /**
     * Grants permission to get a list of sandbox IDs, with each sandbox ID representing a single sandbox
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListSandboxes.html
     */
    toListSandboxes() {
        return this.to('ListSandboxes');
    }
    /**
     * Grants permission to get a list of sandbox IDs for the specified sandbox project, with each sandbox ID representing a single sandbox
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListSandboxesForProject.html
     */
    toListSandboxesForProject() {
        return this.to('ListSandboxesForProject');
    }
    /**
     * Grants permission to return a list of project ARNs that have been shared with the requester. Each project ARN represents one project
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListSharedProjects.html
     */
    toListSharedProjects() {
        return this.to('ListSharedProjects');
    }
    /**
     * Grants permission to return a list of report group ARNs that have been shared with the requester. Each report group ARN represents one report group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListSharedReportGroups.html
     */
    toListSharedReportGroups() {
        return this.to('ListSharedReportGroups');
    }
    /**
     * Grants permission to return a list of SourceCredentialsInfo objects
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListSourceCredentials.html
     */
    toListSourceCredentials() {
        return this.to('ListSourceCredentials');
    }
    /**
     * Grants permission to save an OAuth token from a connected third-party OAuth provider. Only used in the AWS CodeBuild console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    toPersistOAuthToken() {
        return this.to('PersistOAuthToken');
    }
    /**
     * Grants permission to create a resource policy for the associated project or report group
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to retry a build
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_RetryBuild.html
     */
    toRetryBuild() {
        return this.to('RetryBuild');
    }
    /**
     * Grants permission to retry a build batch
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_RetryBuildBatch.html
     */
    toRetryBuildBatch() {
        return this.to('RetryBuildBatch');
    }
    /**
     * Grants permission to start running a build
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAutoRetryLimit()
     * - .ifArtifacts()
     * - .ifArtifactsBucketOwnerAccess()
     * - .ifArtifactsEncryptionDisabled()
     * - .ifArtifactsLocation()
     * - .ifSecondaryArtifacts()
     * - .ifSecondaryArtifactsArtifactIdentifier()
     * - .ifSecondaryArtifactsBucketOwnerAccess()
     * - .ifSecondaryArtifactsEncryptionDisabled()
     * - .ifSecondaryArtifactsLocation()
     * - .ifSecondaryArtifacts()
     * - .ifSecondaryArtifacts()
     * - .ifSecondaryArtifacts()
     * - .ifSource()
     * - .ifSourceBuildStatusConfigTargetUrl()
     * - .ifSourceBuildStatusConfigContext()
     * - .ifSourceLocation()
     * - .ifSourceInsecureSsl()
     * - .ifSourceBuildspec()
     * - .ifSourceAuthResource()
     * - .ifSourceAuthType()
     * - .ifSecondarySources()
     * - .ifSecondarySourcesSourceIdentifier()
     * - .ifSecondarySourcesBuildStatusConfigTargetUrl()
     * - .ifSecondarySourcesBuildStatusConfigContext()
     * - .ifSecondarySourcesLocation()
     * - .ifSecondarySourcesAuthResource()
     * - .ifSecondarySourcesAuthType()
     * - .ifSecondarySourcesBuildspec()
     * - .ifSecondarySourcesInsecureSsl()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifLogsConfig()
     * - .ifLogsConfigS3Logs()
     * - .ifLogsConfigS3LogsBucketOwnerAccess()
     * - .ifLogsConfigS3LogsEncryptionDisabled()
     * - .ifLogsConfigS3LogsLocation()
     * - .ifLogsConfigS3LogsStatus()
     * - .ifEnvironment()
     * - .ifEnvironmentType()
     * - .ifEnvironmentFleetFleetArn()
     * - .ifEnvironmentComputeType()
     * - .ifEnvironmentImage()
     * - .ifEnvironmentImagePullCredentialsType()
     * - .ifEnvironmentPrivilegedMode()
     * - .ifEnvironmentCertificate()
     * - .ifEnvironmentEnvironmentVariables()
     * - .ifEnvironmentEnvironmentVariablesName()
     * - .ifEnvironmentEnvironmentVariablesValue()
     * - .ifEnvironmentEnvironmentVariables()
     * - .ifEnvironmentRegistryCredential()
     * - .ifEnvironmentRegistryCredentialCredential()
     * - .ifEnvironmentRegistryCredentialCredentialProvider()
     * - .ifEncryptionKey()
     * - .ifCache()
     * - .ifCacheType()
     * - .ifCacheLocation()
     * - .ifCacheModes()
     * - .ifServiceRole()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartBuild.html
     */
    toStartBuild() {
        return this.to('StartBuild');
    }
    /**
     * Grants permission to start running a build batch
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifArtifacts()
     * - .ifArtifactsBucketOwnerAccess()
     * - .ifArtifactsEncryptionDisabled()
     * - .ifArtifactsLocation()
     * - .ifSecondaryArtifacts()
     * - .ifSecondaryArtifactsArtifactIdentifier()
     * - .ifSecondaryArtifactsBucketOwnerAccess()
     * - .ifSecondaryArtifactsEncryptionDisabled()
     * - .ifSecondaryArtifactsLocation()
     * - .ifSecondaryArtifacts()
     * - .ifSecondaryArtifacts()
     * - .ifSecondaryArtifacts()
     * - .ifSource()
     * - .ifSourceLocation()
     * - .ifSourceInsecureSsl()
     * - .ifSourceBuildspec()
     * - .ifSourceAuthResource()
     * - .ifSourceAuthType()
     * - .ifSecondarySources()
     * - .ifSecondarySourcesSourceIdentifier()
     * - .ifSecondarySourcesBuildStatusConfigTargetUrl()
     * - .ifSecondarySourcesBuildStatusConfigContext()
     * - .ifSecondarySourcesLocation()
     * - .ifSecondarySourcesAuthResource()
     * - .ifSecondarySourcesAuthType()
     * - .ifSecondarySourcesBuildspec()
     * - .ifSecondarySourcesInsecureSsl()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifLogsConfig()
     * - .ifLogsConfigS3Logs()
     * - .ifLogsConfigS3LogsBucketOwnerAccess()
     * - .ifLogsConfigS3LogsEncryptionDisabled()
     * - .ifLogsConfigS3LogsLocation()
     * - .ifLogsConfigS3LogsStatus()
     * - .ifBuildBatchConfig()
     * - .ifBuildBatchConfigServiceRole()
     * - .ifBuildBatchConfigRestrictionsComputeTypesAllowed()
     * - .ifBuildBatchConfigRestrictionsFleetsAllowed()
     * - .ifEnvironment()
     * - .ifEnvironmentType()
     * - .ifEnvironmentComputeType()
     * - .ifEnvironmentImage()
     * - .ifEnvironmentImagePullCredentialsType()
     * - .ifEnvironmentPrivilegedMode()
     * - .ifEnvironmentCertificate()
     * - .ifEnvironmentEnvironmentVariables()
     * - .ifEnvironmentEnvironmentVariablesName()
     * - .ifEnvironmentEnvironmentVariablesValue()
     * - .ifEnvironmentEnvironmentVariables()
     * - .ifEnvironmentRegistryCredential()
     * - .ifEnvironmentRegistryCredentialCredential()
     * - .ifEnvironmentRegistryCredentialCredentialProvider()
     * - .ifEncryptionKey()
     * - .ifCache()
     * - .ifCacheType()
     * - .ifCacheLocation()
     * - .ifCacheModes()
     * - .ifServiceRole()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartBuildBatch.html
     */
    toStartBuildBatch() {
        return this.to('StartBuildBatch');
    }
    /**
     * Grants permission to start running a command execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartCommandExecution.html
     */
    toStartCommandExecution() {
        return this.to('StartCommandExecution');
    }
    /**
     * Grants permission to start running a sandbox
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartSandbox.html
     */
    toStartSandbox() {
        return this.to('StartSandbox');
    }
    /**
     * Grants permission to establish a connection to the sandbox
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartSandboxConnection.html
     */
    toStartSandboxConnection() {
        return this.to('StartSandboxConnection');
    }
    /**
     * Grants permission to attempt to stop running a build
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StopBuild.html
     */
    toStopBuild() {
        return this.to('StopBuild');
    }
    /**
     * Grants permission to attempt to stop running a build batch
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StopBuildBatch.html
     */
    toStopBuildBatch() {
        return this.to('StopBuildBatch');
    }
    /**
     * Grants permission to attempt to stop running a sandbox
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StopSandbox.html
     */
    toStopSandbox() {
        return this.to('StopSandbox');
    }
    /**
     * Grants permission to change the settings of an existing compute fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifImageId()
     * - .ifComputeType()
     * - .ifVpcConfig()
     * - .ifVpcConfigVpcId()
     * - .ifVpcConfigSecurityGroupIds()
     * - .ifVpcConfigSubnets()
     * - .ifComputeConfiguration()
     * - .ifComputeConfigurationDisk()
     * - .ifComputeConfigurationInstanceType()
     * - .ifComputeConfigurationMachineType()
     * - .ifComputeConfigurationMemory()
     * - .ifComputeConfigurationVCpu()
     * - .ifEnvironmentType()
     * - .ifFleetServiceRole()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_UpdateFleet.html
     */
    toUpdateFleet() {
        return this.to('UpdateFleet');
    }
    /**
     * Grants permission to change the settings of an existing build project
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAutoRetryLimit()
     * - .ifConcurrentBuildLimit()
     * - .ifArtifacts()
     * - .ifArtifactsBucketOwnerAccess()
     * - .ifArtifactsEncryptionDisabled()
     * - .ifArtifactsLocation()
     * - .ifSecondaryArtifacts()
     * - .ifSecondaryArtifactsArtifactIdentifier()
     * - .ifSecondaryArtifactsBucketOwnerAccess()
     * - .ifSecondaryArtifactsEncryptionDisabled()
     * - .ifSecondaryArtifactsLocation()
     * - .ifSecondaryArtifacts()
     * - .ifSecondaryArtifacts()
     * - .ifSecondaryArtifacts()
     * - .ifSource()
     * - .ifSourceBuildStatusConfigTargetUrl()
     * - .ifSourceBuildStatusConfigContext()
     * - .ifSourceLocation()
     * - .ifSourceInsecureSsl()
     * - .ifSourceBuildspec()
     * - .ifSourceAuthResource()
     * - .ifSourceAuthType()
     * - .ifSecondarySources()
     * - .ifSecondarySourcesSourceIdentifier()
     * - .ifSecondarySourcesBuildStatusConfigTargetUrl()
     * - .ifSecondarySourcesBuildStatusConfigContext()
     * - .ifSecondarySourcesLocation()
     * - .ifSecondarySourcesAuthResource()
     * - .ifSecondarySourcesAuthType()
     * - .ifSecondarySourcesBuildspec()
     * - .ifSecondarySourcesInsecureSsl()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifSecondarySources()
     * - .ifLogsConfig()
     * - .ifLogsConfigS3Logs()
     * - .ifLogsConfigS3LogsBucketOwnerAccess()
     * - .ifLogsConfigS3LogsEncryptionDisabled()
     * - .ifLogsConfigS3LogsLocation()
     * - .ifLogsConfigS3LogsStatus()
     * - .ifFileSystemLocationsIdentifier()
     * - .ifFileSystemLocationsType()
     * - .ifFileSystemLocationsLocation()
     * - .ifFileSystemLocations()
     * - .ifFileSystemLocations()
     * - .ifBuildBatchConfig()
     * - .ifBuildBatchConfigServiceRole()
     * - .ifBuildBatchConfigRestrictionsComputeTypesAllowed()
     * - .ifBuildBatchConfigRestrictionsFleetsAllowed()
     * - .ifVpcConfig()
     * - .ifVpcConfigSubnets()
     * - .ifVpcConfigVpcId()
     * - .ifVpcConfigSecurityGroupIds()
     * - .ifEnvironment()
     * - .ifEnvironmentType()
     * - .ifEnvironmentFleetFleetArn()
     * - .ifEnvironmentComputeType()
     * - .ifEnvironmentImage()
     * - .ifEnvironmentImagePullCredentialsType()
     * - .ifEnvironmentPrivilegedMode()
     * - .ifEnvironmentCertificate()
     * - .ifEnvironmentComputeConfiguration()
     * - .ifEnvironmentComputeConfigurationDisk()
     * - .ifEnvironmentComputeConfigurationInstanceType()
     * - .ifEnvironmentComputeConfigurationMachineType()
     * - .ifEnvironmentComputeConfigurationMemory()
     * - .ifEnvironmentComputeConfigurationVCpu()
     * - .ifEnvironmentEnvironmentVariables()
     * - .ifEnvironmentEnvironmentVariablesName()
     * - .ifEnvironmentEnvironmentVariablesValue()
     * - .ifEnvironmentEnvironmentVariables()
     * - .ifEnvironmentRegistryCredential()
     * - .ifEnvironmentRegistryCredentialCredential()
     * - .ifEnvironmentRegistryCredentialCredentialProvider()
     * - .ifEncryptionKey()
     * - .ifCache()
     * - .ifCacheType()
     * - .ifCacheLocation()
     * - .ifCacheModes()
     * - .ifServiceRole()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_UpdateProject.html
     */
    toUpdateProject() {
        return this.to('UpdateProject');
    }
    /**
     * Grants permission to change the public visibility of a project and its builds
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifProjectVisibility()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_UpdateProjectVisibility.html
     */
    toUpdateProjectVisibility() {
        return this.to('UpdateProjectVisibility');
    }
    /**
     * Grants permission to update information about a report
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    toUpdateReport() {
        return this.to('UpdateReport');
    }
    /**
     * Grants permission to change the settings of an existing report group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifExportConfigS3DestinationBucket()
     * - .ifExportConfigS3DestinationBucketOwner()
     * - .ifExportConfigS3DestinationEncryptionKey()
     * - .ifExportConfigS3DestinationEncryptionDisabled()
     * - .ifExportConfigS3DestinationPath()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_UpdateReportGroup.html
     */
    toUpdateReportGroup() {
        return this.to('UpdateReportGroup');
    }
    /**
     * Grants permission to update the webhook associated with an AWS CodeBuild build project
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifBuildType()
     * - .ifManualCreation()
     * - .ifScopeConfigurationDomain()
     * - .ifScopeConfigurationName()
     * - .ifScopeConfigurationScope()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_UpdateWebhook.html
     */
    toUpdateWebhook() {
        return this.to('UpdateWebhook');
    }
    /**
     * Adds a resource of type build to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param buildId - Identifier for the buildId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onBuild(buildId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codebuild:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:build/${buildId}`);
    }
    /**
     * Adds a resource of type build-batch to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param buildBatchId - Identifier for the buildBatchId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onBuildBatch(buildBatchId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codebuild:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:build-batch/${buildBatchId}`);
    }
    /**
     * Adds a resource of type project to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param projectName - Identifier for the projectName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProject(projectName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codebuild:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:project/${projectName}`);
    }
    /**
     * Adds a resource of type report-group to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param reportGroupName - Identifier for the reportGroupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onReportGroup(reportGroupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codebuild:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:report-group/${reportGroupName}`);
    }
    /**
     * Adds a resource of type report to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param reportGroupName - Identifier for the reportGroupName.
     * @param reportId - Identifier for the reportId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onReport(reportGroupName, reportId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codebuild:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:report/${reportGroupName}:${reportId}`);
    }
    /**
     * Adds a resource of type fleet to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param fleetName - Identifier for the fleetName.
     * @param fleetId - Identifier for the fleetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onFleet(fleetName, fleetId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codebuild:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:fleet/${fleetName}:${fleetId}`);
    }
    /**
     * Adds a resource of type sandbox to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param sandboxId - Identifier for the sandboxId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onSandbox(sandboxId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codebuild:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:sandbox/${sandboxId}`);
    }
    /**
     * Filters access by actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toCreateProject()
     * - .toCreateReportGroup()
     * - .toUpdateFleet()
     * - .toUpdateProject()
     * - .toUpdateProjectVisibility()
     * - .toUpdateReportGroup()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - project
     * - report-group
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toCreateProject()
     * - .toCreateReportGroup()
     * - .toUpdateFleet()
     * - .toUpdateProject()
     * - .toUpdateProjectVisibility()
     * - .toUpdateReportGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifArtifacts(value) {
        return this.if(`artifacts`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifArtifactsBucketOwnerAccess(value, operator) {
        return this.if(`artifacts.bucketOwnerAccess`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifArtifactsEncryptionDisabled(value) {
        return this.if(`artifacts.encryptionDisabled`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifArtifactsLocation(value, operator) {
        return this.if(`artifacts.location`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toImportSourceCredentials()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthType(value, operator) {
        return this.if(`authType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifAutoRetryLimit(value, operator) {
        return this.if(`autoRetryLimit`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the ARN of the AWS CodeBuild build from which the request originated
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-permissions-reference.html
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifBuildArn(value, operator) {
        return this.if(`buildArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifBuildBatchConfig(value) {
        return this.if(`buildBatchConfig`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifBuildBatchConfigRestrictionsComputeTypesAllowed(value, operator) {
        return this.if(`buildBatchConfig.restrictions.computeTypesAllowed`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifBuildBatchConfigRestrictionsFleetsAllowed(value, operator) {
        return this.if(`buildBatchConfig.restrictions.fleetsAllowed`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifBuildBatchConfigServiceRole(value, operator) {
        return this.if(`buildBatchConfig.serviceRole`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateWebhook()
     * - .toUpdateWebhook()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifBuildType(value, operator) {
        return this.if(`buildType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifCache(value) {
        return this.if(`cache`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCacheLocation(value, operator) {
        return this.if(`cache.location`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCacheModes(value, operator) {
        return this.if(`cache.modes`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCacheType(value, operator) {
        return this.if(`cache.type`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toUpdateFleet()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifComputeConfiguration(value) {
        return this.if(`computeConfiguration`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toUpdateFleet()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifComputeConfigurationDisk(value, operator) {
        return this.if(`computeConfiguration.disk`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toUpdateFleet()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifComputeConfigurationInstanceType(value, operator) {
        return this.if(`computeConfiguration.instanceType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toUpdateFleet()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifComputeConfigurationMachineType(value, operator) {
        return this.if(`computeConfiguration.machineType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toUpdateFleet()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifComputeConfigurationMemory(value, operator) {
        return this.if(`computeConfiguration.memory`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toUpdateFleet()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifComputeConfigurationVCpu(value, operator) {
        return this.if(`computeConfiguration.vCpu`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toUpdateFleet()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifComputeType(value, operator) {
        return this.if(`computeType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifConcurrentBuildLimit(value, operator) {
        return this.if(`concurrentBuildLimit`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEncryptionKey(value, operator) {
        return this.if(`encryptionKey`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifEnvironment(value) {
        return this.if(`environment`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentCertificate(value, operator) {
        return this.if(`environment.certificate`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifEnvironmentComputeConfiguration(value) {
        return this.if(`environment.computeConfiguration`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifEnvironmentComputeConfigurationDisk(value, operator) {
        return this.if(`environment.computeConfiguration.disk`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentComputeConfigurationInstanceType(value, operator) {
        return this.if(`environment.computeConfiguration.instanceType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentComputeConfigurationMachineType(value, operator) {
        return this.if(`environment.computeConfiguration.machineType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifEnvironmentComputeConfigurationMemory(value, operator) {
        return this.if(`environment.computeConfiguration.memory`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifEnvironmentComputeConfigurationVCpu(value, operator) {
        return this.if(`environment.computeConfiguration.vCpu`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentComputeType(value, operator) {
        return this.if(`environment.computeType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifEnvironmentEnvironmentVariables(value) {
        return this.if(`environment.environmentVariables`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentEnvironmentVariablesName(value, operator) {
        return this.if(`environment.environmentVariables.name`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentEnvironmentVariablesValue(value, operator) {
        return this.if(`environment.environmentVariables.value`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param namevalue The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentEnvironmentVariablesNamevalue(namevalue, value, operator) {
        return this.if(`environment.environmentVariables/${namevalue}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifEnvironmentFleetFleetArn(value, operator) {
        return this.if(`environment.fleet.fleetArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentImage(value, operator) {
        return this.if(`environment.image`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentImagePullCredentialsType(value, operator) {
        return this.if(`environment.imagePullCredentialsType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifEnvironmentPrivilegedMode(value) {
        return this.if(`environment.privilegedMode`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifEnvironmentRegistryCredential(value) {
        return this.if(`environment.registryCredential`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentRegistryCredentialCredential(value, operator) {
        return this.if(`environment.registryCredential.credential`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentRegistryCredentialCredentialProvider(value, operator) {
        return this.if(`environment.registryCredential.credentialProvider`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentType(value, operator) {
        return this.if(`environment.type`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toUpdateFleet()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFleetEnvironmentType(value, operator) {
        return this.if(`environmentType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateReportGroup()
     * - .toUpdateReportGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExportConfigS3DestinationBucket(value, operator) {
        return this.if(`exportConfig.s3Destination.bucket`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateReportGroup()
     * - .toUpdateReportGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExportConfigS3DestinationBucketOwner(value, operator) {
        return this.if(`exportConfig.s3Destination.bucketOwner`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateReportGroup()
     * - .toUpdateReportGroup()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifExportConfigS3DestinationEncryptionDisabled(value) {
        return this.if(`exportConfig.s3Destination.encryptionDisabled`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateReportGroup()
     * - .toUpdateReportGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExportConfigS3DestinationEncryptionKey(value, operator) {
        return this.if(`exportConfig.s3Destination.encryptionKey`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateReportGroup()
     * - .toUpdateReportGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExportConfigS3DestinationPath(value, operator) {
        return this.if(`exportConfig.s3Destination.path`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFileSystemLocationsIdentifier(value, operator) {
        return this.if(`fileSystemLocations.identifier`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFileSystemLocationsLocation(value, operator) {
        return this.if(`fileSystemLocations.location`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFileSystemLocationsType(value, operator) {
        return this.if(`fileSystemLocations.type`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toUpdateProject()
     *
     * @param identifierlocation The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFileSystemLocationsIdentifierlocation(identifierlocation, value, operator) {
        return this.if(`fileSystemLocations/${identifierlocation}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toUpdateProject()
     *
     * @param identifiertype The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFileSystemLocationsIdentifiertype(identifiertype, value, operator) {
        return this.if(`fileSystemLocations/${identifiertype}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toUpdateFleet()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFleetServiceRole(value, operator) {
        return this.if(`fleetServiceRole`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toUpdateFleet()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifImageId(value, operator) {
        return this.if(`imageId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifLogsConfig(value) {
        return this.if(`logsConfig`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifLogsConfigS3Logs(value) {
        return this.if(`logsConfig.s3Logs`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLogsConfigS3LogsBucketOwnerAccess(value, operator) {
        return this.if(`logsConfig.s3Logs.bucketOwnerAccess`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifLogsConfigS3LogsEncryptionDisabled(value) {
        return this.if(`logsConfig.s3Logs.encryptionDisabled`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLogsConfigS3LogsLocation(value, operator) {
        return this.if(`logsConfig.s3Logs.location`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLogsConfigS3LogsStatus(value, operator) {
        return this.if(`logsConfig.s3Logs.status`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateWebhook()
     * - .toUpdateWebhook()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifManualCreation(value) {
        return this.if(`manualCreation`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the ARN of the AWS CodeBuild project from which the request originated
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-permissions-reference.html
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifProjectArn(value, operator) {
        return this.if(`projectArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toUpdateProjectVisibility()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProjectVisibility(value, operator) {
        return this.if(`projectVisibility`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateWebhook()
     * - .toUpdateWebhook()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifScopeConfigurationDomain(value, operator) {
        return this.if(`scopeConfiguration.domain`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateWebhook()
     * - .toUpdateWebhook()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifScopeConfigurationName(value, operator) {
        return this.if(`scopeConfiguration.name`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateWebhook()
     * - .toUpdateWebhook()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifScopeConfigurationScope(value, operator) {
        return this.if(`scopeConfiguration.scope`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifSecondaryArtifacts(value) {
        return this.if(`secondaryArtifacts`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondaryArtifactsArtifactIdentifier(value, operator) {
        return this.if(`secondaryArtifacts.artifactIdentifier`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondaryArtifactsBucketOwnerAccess(value, operator) {
        return this.if(`secondaryArtifacts.bucketOwnerAccess`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifSecondaryArtifactsEncryptionDisabled(value) {
        return this.if(`secondaryArtifacts.encryptionDisabled`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondaryArtifactsLocation(value, operator) {
        return this.if(`secondaryArtifacts.location`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param artifactIdentifierbucketOwnerAccess The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondaryArtifactsArtifactIdentifierbucketOwnerAccess(artifactIdentifierbucketOwnerAccess, value, operator) {
        return this.if(`secondaryArtifacts/${artifactIdentifierbucketOwnerAccess}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param artifactIdentifierencryptionDisabled The tag key to check
     * @param value `true` or `false`. **Default:** `true`
     */
    ifSecondaryArtifactsArtifactIdentifierencryptionDisabled(artifactIdentifierencryptionDisabled, value) {
        return this.if(`secondaryArtifacts/${artifactIdentifierencryptionDisabled}`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param artifactIdentifierlocation The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondaryArtifactsArtifactIdentifierlocation(artifactIdentifierlocation, value, operator) {
        return this.if(`secondaryArtifacts/${artifactIdentifierlocation}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifSecondarySources(value) {
        return this.if(`secondarySources`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondarySourcesAuthResource(value, operator) {
        return this.if(`secondarySources.auth.resource`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondarySourcesAuthType(value, operator) {
        return this.if(`secondarySources.auth.type`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondarySourcesBuildStatusConfigContext(value, operator) {
        return this.if(`secondarySources.buildStatusConfig.context`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondarySourcesBuildStatusConfigTargetUrl(value, operator) {
        return this.if(`secondarySources.buildStatusConfig.targetUrl`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifSecondarySourcesBuildspec(value) {
        return this.if(`secondarySources.buildspec`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifSecondarySourcesInsecureSsl(value) {
        return this.if(`secondarySources.insecureSsl`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondarySourcesLocation(value, operator) {
        return this.if(`secondarySources.location`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondarySourcesSourceIdentifier(value, operator) {
        return this.if(`secondarySources.sourceIdentifier`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param sourceIdentifierauthresource The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondarySourcesSourceIdentifierauthresource(sourceIdentifierauthresource, value, operator) {
        return this.if(`secondarySources/${sourceIdentifierauthresource}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param sourceIdentifierauthtype The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondarySourcesSourceIdentifierauthtype(sourceIdentifierauthtype, value, operator) {
        return this.if(`secondarySources/${sourceIdentifierauthtype}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param sourceIdentifierbuildStatusConfigcontext The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondarySourcesSourceIdentifierbuildStatusConfigcontext(sourceIdentifierbuildStatusConfigcontext, value, operator) {
        return this.if(`secondarySources/${sourceIdentifierbuildStatusConfigcontext}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param sourceIdentifierbuildStatusConfigtargetUrl The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondarySourcesSourceIdentifierbuildStatusConfigtargetUrl(sourceIdentifierbuildStatusConfigtargetUrl, value, operator) {
        return this.if(`secondarySources/${sourceIdentifierbuildStatusConfigtargetUrl}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param sourceIdentifierbuildspec The tag key to check
     * @param value `true` or `false`. **Default:** `true`
     */
    ifSecondarySourcesSourceIdentifierbuildspec(sourceIdentifierbuildspec, value) {
        return this.if(`secondarySources/${sourceIdentifierbuildspec}`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param sourceIdentifierinsecureSsl The tag key to check
     * @param value `true` or `false`. **Default:** `true`
     */
    ifSecondarySourcesSourceIdentifierinsecureSsl(sourceIdentifierinsecureSsl, value) {
        return this.if(`secondarySources/${sourceIdentifierinsecureSsl}`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param sourceIdentifierlocation The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecondarySourcesSourceIdentifierlocation(sourceIdentifierlocation, value, operator) {
        return this.if(`secondarySources/${sourceIdentifierlocation}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toImportSourceCredentials()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifServerType(value, operator) {
        return this.if(`serverType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifServiceRole(value, operator) {
        return this.if(`serviceRole`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toImportSourceCredentials()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifShouldOverwrite(value) {
        return this.if(`shouldOverwrite`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifSource(value) {
        return this.if(`source`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSourceAuthResource(value, operator) {
        return this.if(`source.auth.resource`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSourceAuthType(value, operator) {
        return this.if(`source.auth.type`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSourceBuildStatusConfigContext(value, operator) {
        return this.if(`source.buildStatusConfig.context`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSourceBuildStatusConfigTargetUrl(value, operator) {
        return this.if(`source.buildStatusConfig.targetUrl`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifSourceBuildspec(value) {
        return this.if(`source.buildspec`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifSourceInsecureSsl(value) {
        return this.if(`source.insecureSsl`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toStartBuild()
     * - .toStartBuildBatch()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSourceLocation(value, operator) {
        return this.if(`source.location`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toImportSourceCredentials()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifToken(value, operator) {
        return this.if(`token`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toImportSourceCredentials()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUsername(value, operator) {
        return this.if(`username`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toCreateProject()
     * - .toUpdateFleet()
     * - .toUpdateProject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifVpcConfig(value) {
        return this.if(`vpcConfig`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toCreateProject()
     * - .toUpdateFleet()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpcConfigSecurityGroupIds(value, operator) {
        return this.if(`vpcConfig.securityGroupIds`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toCreateProject()
     * - .toUpdateFleet()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpcConfigSubnets(value, operator) {
        return this.if(`vpcConfig.subnets`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API corresponding argument value
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/action-context-keys.html
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toCreateProject()
     * - .toUpdateFleet()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpcConfigVpcId(value, operator) {
        return this.if(`vpcConfig.vpcId`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [codebuild](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodebuild.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'codebuild';
        this.accessLevelList = {
            Write: [
                'BatchDeleteBuilds',
                'BatchPutCodeCoverages',
                'BatchPutTestCases',
                'CreateFleet',
                'CreateProject',
                'CreateReport',
                'CreateReportGroup',
                'CreateWebhook',
                'DeleteBuildBatch',
                'DeleteFleet',
                'DeleteOAuthToken',
                'DeleteProject',
                'DeleteReport',
                'DeleteReportGroup',
                'DeleteSourceCredentials',
                'DeleteWebhook',
                'ImportSourceCredentials',
                'InvalidateProjectCache',
                'PersistOAuthToken',
                'RetryBuild',
                'RetryBuildBatch',
                'StartBuild',
                'StartBuildBatch',
                'StartCommandExecution',
                'StartSandbox',
                'StartSandboxConnection',
                'StopBuild',
                'StopBuildBatch',
                'StopSandbox',
                'UpdateFleet',
                'UpdateProject',
                'UpdateProjectVisibility',
                'UpdateReport',
                'UpdateReportGroup',
                'UpdateWebhook'
            ],
            Read: [
                'BatchGetBuildBatches',
                'BatchGetBuilds',
                'BatchGetCommandExecutions',
                'BatchGetFleets',
                'BatchGetProjects',
                'BatchGetReportGroups',
                'BatchGetReports',
                'BatchGetSandboxes',
                'DescribeCodeCoverages',
                'DescribeTestCases',
                'GetReportGroupTrend',
                'GetResourcePolicy'
            ],
            'Permissions management': [
                'DeleteResourcePolicy',
                'PutResourcePolicy'
            ],
            List: [
                'ListBuildBatches',
                'ListBuildBatchesForProject',
                'ListBuilds',
                'ListBuildsForProject',
                'ListCommandExecutionsForSandbox',
                'ListConnectedOAuthAccounts',
                'ListCuratedEnvironmentImages',
                'ListFleets',
                'ListProjects',
                'ListReportGroups',
                'ListReports',
                'ListReportsForReportGroup',
                'ListRepositories',
                'ListSandboxes',
                'ListSandboxesForProject',
                'ListSharedProjects',
                'ListSharedReportGroups',
                'ListSourceCredentials'
            ]
        };
    }
}
exports.Codebuild = Codebuild;
//# sourceMappingURL=data:application/json;base64,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